/**********************************************************************
 * asym_method.c                                            August 2005
 *
 * KSSLD(key_tool): An implementation of SSL/TLS in the Linux Kernel
 * Copyright (C) 2005  NTT COMWARE Corporation.
 *
 * This file based in part on code from LVS www.linuxvirtualserver.org
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 *
 **********************************************************************/

#include "asym_method.h"

#include "kssl.h"

#include <vanessa_adt.h>

#include <stdio.h>

static const char * asym_method_name(kssl_asym_method_t am);

static const kssl_asym_method_t asym_method_byname(const char *name);


int
asym_method_available_printf(void)
{
	return asym_method_available_fprintf(stdout);
}


int
asym_method_available_fprintf(FILE *stream)
{
	if (fprintf(stream, KSSL_ASYM_METHOD_SOFTWARE "\n"
				KSSL_ASYM_METHOD_AEP_USER "\n"
				KSSL_ASYM_METHOD_AEP_KERNEL "\n") < 0)
		return -1;

	return 0;
}       


int
asym_method_list_printf(kssl_asym_method_t *am_list, size_t noam)
{
	return asym_method_list_fprintf(stdout, am_list, noam);
}


int
asym_method_list_fprintf(FILE *stream, kssl_asym_method_t *am_list, 
		size_t noam)
{
	while (noam--) {
		if (am_list == kssl_asym_method_none)
			break;
		if (fprintf(stream, "%s\n", asym_method_name(*am_list)) < 0)
			return -1;
		am_list++;
	}

	return 0;
}       


static const char *
asym_method_name(kssl_asym_method_t am)
{
	switch (am) {
		case kssl_asym_method_software:
			return KSSL_ASYM_METHOD_SOFTWARE;
		case kssl_asym_method_aep_user:
			return KSSL_ASYM_METHOD_AEP_USER;
		case kssl_asym_method_aep_kernel:
			return KSSL_ASYM_METHOD_AEP_KERNEL;
		case kssl_asym_method_none:
			return "none";
		case kssl_asym_method_unknown:
			break;
	}

	return "unknown";
}       


static const kssl_asym_method_t
asym_method_byname(const char *name)
{
	if (!strcasecmp(KSSL_ASYM_METHOD_SOFTWARE, name))
		return kssl_asym_method_software;
	else if (!strcasecmp(KSSL_ASYM_METHOD_AEP_USER, name))
		return kssl_asym_method_aep_user;
	else if (!strcasecmp(KSSL_ASYM_METHOD_AEP_KERNEL, name))
		return kssl_asym_method_aep_kernel;

	return kssl_asym_method_unknown;
}


int
asym_method_byname_list_alloc(const char *asym_method_str, const char delimiter,
		kssl_asym_method_t **am_list_ret, size_t *noam_ret)
{
	int status = -1;
	size_t i;
	size_t noam;
	char *am_str;
	char *asym_method_str_cpy = NULL;
	kssl_asym_method_t *am_list = NULL;
	vanessa_dynamic_array_t *da = NULL;

	if(!asym_method_str)
		return -1;

	asym_method_str_cpy = strdup(asym_method_str);
	if (!asym_method_str_cpy)
		goto leave;

	da = vanessa_dynamic_array_split_str(asym_method_str_cpy, delimiter);
	if (!da)
		goto leave;

	noam = vanessa_dynamic_array_get_count(da);

	am_list = (kssl_asym_method_t *)malloc(sizeof(kssl_asym_method_t) * 
			(noam + 1));
	if (!am_list)
		goto leave;

	*am_list_ret = am_list;
	*noam_ret = noam;

	for (i = 0; i < noam; i++) {
		am_str = (char *)vanessa_dynamic_array_get_element(da, i);
		if (!am_str)
			goto leave;
		*am_list = asym_method_byname(am_str);
		if (*am_list == kssl_asym_method_unknown)
			goto leave;
		am_list++;
	}
	*am_list = kssl_asym_method_none;

	status = 0;
leave:
	if (asym_method_str_cpy)
		free(asym_method_str_cpy);
	if (da)
		vanessa_dynamic_array_destroy(da);
	if (status < 0 && *am_list_ret)
		free(*am_list_ret);
	return status;
}
