/**********************************************************************
 * cipher.c                                                 August 2005
 *
 * KSSLD(key_tool): An implementation of SSL/TLS in the Linux Kernel
 * Copyright (C) 2005  NTT COMWARE Corporation.
 *
 * This file based in part on code from LVS www.linuxvirtualserver.org
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 *
 **********************************************************************/

#include "types/cipher_suite_t.h"

#include "cipher.h"

#include <vanessa_adt.h>

#include <stdio.h>

#define CIPHER_STATUS_IMPLEMENTED   0x01
#define CIPHER_STATUS_UNIMPLEMENTED 0x02
#define CIPHER_STATUS_NEGOTIABLE    0x04
#define CIPHER_STATUS_NONNEGOTIABLE 0x08

typedef struct {
	const cipher_suite_t cs;
	const char *name;
	int status;
} cipher_suite_info_t;

static const char * cipher_name(cipher_suite_t *cs);


/* This should list all the cipher suites which are known.
 * Their #defines can be found in types/cipher_suite_t.h
 * They are in no particular order, other than that the last entry must be
 * CIPHER_SUITE_END_MARKER 
 */

static const cipher_suite_info_t all_cs[] =
{
	{
		.cs     = TLS_NULL_WITH_NULL_NULL,
		.name   = TLS_NULL_WITH_NULL_NULL_S,
		.status = CIPHER_STATUS_IMPLEMENTED|CIPHER_STATUS_NONNEGOTIABLE
	},

	{
		.cs     = TLS_RSA_WITH_NULL_MD5,
		.name   = TLS_RSA_WITH_NULL_MD5_S,
		.status = CIPHER_STATUS_IMPLEMENTED|CIPHER_STATUS_NEGOTIABLE
	},
	{
		.cs     = TLS_RSA_WITH_NULL_SHA,
		.name   = TLS_RSA_WITH_NULL_SHA_S,
		.status = CIPHER_STATUS_IMPLEMENTED|CIPHER_STATUS_NEGOTIABLE
	},
	{
		.cs     = TLS_RSA_EXPORT_WITH_RC4_40_MD5,
		.name   = TLS_RSA_EXPORT_WITH_RC4_40_MD5_S,
		.status = CIPHER_STATUS_UNIMPLEMENTED|CIPHER_STATUS_NEGOTIABLE
	},
	{
		.cs     = TLS_RSA_WITH_RC4_128_MD5,
		.name   = TLS_RSA_WITH_RC4_128_MD5_S,
		.status = CIPHER_STATUS_UNIMPLEMENTED|CIPHER_STATUS_NEGOTIABLE
	},
	{
		.cs     = TLS_RSA_WITH_RC4_128_SHA,
		.name   = TLS_RSA_WITH_RC4_128_SHA_S,
		.status = CIPHER_STATUS_UNIMPLEMENTED|CIPHER_STATUS_NEGOTIABLE
	},
	{
		.cs     = TLS_RSA_EXPORT_WITH_RC2_CBC_40_MD5,
		.name   = TLS_RSA_EXPORT_WITH_RC2_CBC_40_MD5_S,
		.status = CIPHER_STATUS_UNIMPLEMENTED|CIPHER_STATUS_NEGOTIABLE
	},
	{
		.cs     = TLS_RSA_WITH_IDEA_CBC_SHA,
		.name   = TLS_RSA_WITH_IDEA_CBC_SHA_S,
		.status = CIPHER_STATUS_UNIMPLEMENTED|CIPHER_STATUS_NEGOTIABLE
	},
	{
		.cs     = TLS_RSA_EXPORT_WITH_DES40_CBC_SHA,
		.name   = TLS_RSA_EXPORT_WITH_DES40_CBC_SHA_S,
		.status = CIPHER_STATUS_IMPLEMENTED|CIPHER_STATUS_NEGOTIABLE
	},
	{
		.cs     = TLS_RSA_WITH_DES_CBC_SHA,
		.name   = TLS_RSA_WITH_DES_CBC_SHA_S,
		.status = CIPHER_STATUS_IMPLEMENTED|CIPHER_STATUS_NEGOTIABLE
	},
	{
		.cs     = TLS_RSA_WITH_3DES_EDE_CBC_SHA,
		.name   = TLS_RSA_WITH_3DES_EDE_CBC_SHA_S,
		.status = CIPHER_STATUS_IMPLEMENTED|CIPHER_STATUS_NEGOTIABLE
	},
	{
		.cs     = TLS_DH_DSS_EXPORT_WITH_DES40_CBC_SHA,
		.name   = TLS_DH_DSS_EXPORT_WITH_DES40_CBC_SHA_S,
		.status = CIPHER_STATUS_UNIMPLEMENTED|CIPHER_STATUS_NEGOTIABLE
	},
	{
		.cs     = TLS_DH_DSS_WITH_DES_CBC_SHA,
		.name   = TLS_DH_DSS_WITH_DES_CBC_SHA_S,
		.status = CIPHER_STATUS_UNIMPLEMENTED|CIPHER_STATUS_NEGOTIABLE
	},
	{
		.cs     = TLS_DH_DSS_WITH_3DES_EDE_CBC_SHA,
		.name   = TLS_DH_DSS_WITH_3DES_EDE_CBC_SHA_S,
		.status = CIPHER_STATUS_UNIMPLEMENTED|CIPHER_STATUS_NEGOTIABLE
	},
	{
		.cs     = TLS_DH_RSA_EXPORT_WITH_DES40_CBC_SHA,
		.name   = TLS_DH_RSA_EXPORT_WITH_DES40_CBC_SHA_S,
		.status = CIPHER_STATUS_UNIMPLEMENTED|CIPHER_STATUS_NEGOTIABLE
	},
	{
		.cs     = TLS_DH_RSA_WITH_DES_CBC_SHA,
		.name   = TLS_DH_RSA_WITH_DES_CBC_SHA_S,
		.status = CIPHER_STATUS_UNIMPLEMENTED|CIPHER_STATUS_NEGOTIABLE
	},
	{
		.cs     = TLS_DH_RSA_WITH_3DES_EDE_CBC_SHA,
		.name   = TLS_DH_RSA_WITH_3DES_EDE_CBC_SHA_S,
		.status = CIPHER_STATUS_UNIMPLEMENTED|CIPHER_STATUS_NEGOTIABLE
	},
	{
		.cs     = TLS_DHE_DSS_EXPORT_WITH_DES40_CBC_SHA,
		.name   = TLS_DHE_DSS_EXPORT_WITH_DES40_CBC_SHA_S,
		.status = CIPHER_STATUS_UNIMPLEMENTED|CIPHER_STATUS_NEGOTIABLE
	},
	{
		.cs     = TLS_DHE_DSS_WITH_DES_CBC_SHA,
		.name   = TLS_DHE_DSS_WITH_DES_CBC_SHA_S,
		.status = CIPHER_STATUS_UNIMPLEMENTED|CIPHER_STATUS_NEGOTIABLE
	},
	{
		.cs     = TLS_DHE_DSS_WITH_3DES_EDE_CBC_SHA,
		.name   = TLS_DHE_DSS_WITH_3DES_EDE_CBC_SHA_S,
		.status = CIPHER_STATUS_UNIMPLEMENTED|CIPHER_STATUS_NEGOTIABLE
	},
	{
		.cs     = TLS_DHE_RSA_EXPORT_WITH_DES40_CBC_SHA,
		.name   = TLS_DHE_RSA_EXPORT_WITH_DES40_CBC_SHA_S,
		.status = CIPHER_STATUS_UNIMPLEMENTED|CIPHER_STATUS_NEGOTIABLE
	},
	{
		.cs     = TLS_DHE_RSA_WITH_DES_CBC_SHA,
		.name   = TLS_DHE_RSA_WITH_DES_CBC_SHA_S,
		.status = CIPHER_STATUS_UNIMPLEMENTED|CIPHER_STATUS_NEGOTIABLE
	},
	{
		.cs     = TLS_DHE_RSA_WITH_3DES_EDE_CBC_SHA,
		.name   = TLS_DHE_RSA_WITH_3DES_EDE_CBC_SHA_S,
		.status = CIPHER_STATUS_UNIMPLEMENTED|CIPHER_STATUS_NEGOTIABLE
	},
	{
		.cs     = TLS_DH_anon_EXPORT_WITH_RC4_40_MD5,
		.name   = TLS_DH_anon_EXPORT_WITH_RC4_40_MD5_S,
		.status = CIPHER_STATUS_UNIMPLEMENTED|CIPHER_STATUS_NEGOTIABLE
	},
	{
		.cs     = TLS_DH_anon_WITH_RC4_128_MD5,
		.name   = TLS_DH_anon_WITH_RC4_128_MD5_S,
		.status = CIPHER_STATUS_UNIMPLEMENTED|CIPHER_STATUS_NEGOTIABLE
	},
	{
		.cs     = TLS_DH_anon_EXPORT_WITH_DES40_CBC_SHA,
		.name   = TLS_DH_anon_EXPORT_WITH_DES40_CBC_SHA_S,
		.status = CIPHER_STATUS_UNIMPLEMENTED|CIPHER_STATUS_NEGOTIABLE
	},
	{
		.cs     = TLS_DH_anon_WITH_DES_CBC_SHA,
		.name   = TLS_DH_anon_WITH_DES_CBC_SHA_S,
		.status = CIPHER_STATUS_UNIMPLEMENTED|CIPHER_STATUS_NEGOTIABLE
	},
	{
		.cs     = TLS_DH_anon_WITH_3DES_EDE_CBC_SHA,
		.name   = TLS_DH_anon_WITH_3DES_EDE_CBC_SHA_S,
		.status = CIPHER_STATUS_UNIMPLEMENTED|CIPHER_STATUS_NEGOTIABLE
	},
	{
		.cs     = TLS_RSA_EXPORT1024_WITH_DES_CBC_SHA,
		.name   = TLS_RSA_EXPORT1024_WITH_DES_CBC_SHA_S,
		.status = CIPHER_STATUS_IMPLEMENTED|CIPHER_STATUS_NEGOTIABLE
	},
	{
		.cs     = TLS_DHE_DSS_EXPORT1024_WITH_DES_CBC_SHA,
		.name   = TLS_DHE_DSS_EXPORT1024_WITH_DES_CBC_SHA_S,
		.status = CIPHER_STATUS_UNIMPLEMENTED|CIPHER_STATUS_NEGOTIABLE
	},
	{
		.cs     = TLS_RSA_EXPORT1024_WITH_RC4_56_SHA,
		.name   = TLS_RSA_EXPORT1024_WITH_RC4_56_SHA_S,
		.status = CIPHER_STATUS_UNIMPLEMENTED|CIPHER_STATUS_NEGOTIABLE
	},
	{
		.cs     = TLS_DHE_DSS_EXPORT1024_WITH_RC4_56_SHA,
		.name   = TLS_DHE_DSS_EXPORT1024_WITH_RC4_56_SHA_S,
		.status = CIPHER_STATUS_UNIMPLEMENTED|CIPHER_STATUS_NEGOTIABLE
	},
	{
		.cs     = TLS_DHE_DSS_WITH_RC4_128_SHA,
		.name   = TLS_DHE_DSS_WITH_RC4_128_SHA_S,
		.status = CIPHER_STATUS_UNIMPLEMENTED|CIPHER_STATUS_NEGOTIABLE
	},
	{
		.cs     = TLS_RSA_WITH_AES_128_CBC_SHA,
		.name   = TLS_RSA_WITH_AES_128_CBC_SHA_S,
		.status = CIPHER_STATUS_IMPLEMENTED|CIPHER_STATUS_NEGOTIABLE
	},
	{
		.cs     = TLS_DH_DSS_WITH_AES_128_CBC_SHA,
		.name   = TLS_DH_DSS_WITH_AES_128_CBC_SHA_S,
		.status = CIPHER_STATUS_UNIMPLEMENTED|CIPHER_STATUS_NEGOTIABLE
	},
	{
		.cs     = TLS_DH_RSA_WITH_AES_128_CBC_SHA,
		.name   = TLS_DH_RSA_WITH_AES_128_CBC_SHA_S,
		.status = CIPHER_STATUS_UNIMPLEMENTED|CIPHER_STATUS_NEGOTIABLE
	},
	{
		.cs     = TLS_DHE_DSS_WITH_AES_128_CBC_SHA,
		.name   = TLS_DHE_DSS_WITH_AES_128_CBC_SHA_S,
		.status = CIPHER_STATUS_UNIMPLEMENTED|CIPHER_STATUS_NEGOTIABLE
	},
	{
		.cs     = TLS_DHE_RSA_WITH_AES_128_CBC_SHA,
		.name   = TLS_DHE_RSA_WITH_AES_128_CBC_SHA_S,
		.status = CIPHER_STATUS_UNIMPLEMENTED|CIPHER_STATUS_NEGOTIABLE
	},
	{
		.cs     = TLS_DH_anon_WITH_AES_128_CBC_SHA,
		.name   = TLS_DH_anon_WITH_AES_128_CBC_SHA_S,
		.status = CIPHER_STATUS_UNIMPLEMENTED|CIPHER_STATUS_NEGOTIABLE
	},
	{
		.cs     = TLS_RSA_WITH_AES_256_CBC_SHA,
		.name   = TLS_RSA_WITH_AES_256_CBC_SHA_S,
		.status = CIPHER_STATUS_IMPLEMENTED|CIPHER_STATUS_NEGOTIABLE
	},
	{
		.cs     = TLS_DH_DSS_WITH_AES_256_CBC_SHA,
		.name   = TLS_DH_DSS_WITH_AES_256_CBC_SHA_S,
		.status = CIPHER_STATUS_UNIMPLEMENTED|CIPHER_STATUS_NEGOTIABLE
	},
	{
		.cs     = TLS_DH_RSA_WITH_AES_256_CBC_SHA,
		.name   = TLS_DH_RSA_WITH_AES_256_CBC_SHA_S,
		.status = CIPHER_STATUS_UNIMPLEMENTED|CIPHER_STATUS_NEGOTIABLE
	},
	{
		.cs     = TLS_DHE_DSS_WITH_AES_256_CBC_SHA,
		.name   = TLS_DHE_DSS_WITH_AES_256_CBC_SHA_S,
		.status = CIPHER_STATUS_UNIMPLEMENTED|CIPHER_STATUS_NEGOTIABLE
	},
	{
		.cs     = TLS_DHE_RSA_WITH_AES_256_CBC_SHA,
		.name   = TLS_DHE_RSA_WITH_AES_256_CBC_SHA_S,
		.status = CIPHER_STATUS_UNIMPLEMENTED|CIPHER_STATUS_NEGOTIABLE
	},
	{
		.cs     = TLS_DH_anon_WITH_AES_256_CBC_SHA,
		.name   = TLS_DH_anon_WITH_AES_256_CBC_SHA_S,
		.status = CIPHER_STATUS_UNIMPLEMENTED|CIPHER_STATUS_NEGOTIABLE
	},
	{
		.cs     = CIPHER_SUITE_END_MARKER,
		.name   = NULL,
		.status = 0
	},
};


static int
cipher_available(const cipher_suite_t *cs)
{
	cipher_suite_info_t *a;

	for (a = (cipher_suite_info_t *)all_cs; a->status; a++) {
		if (!cipher_suite_cmp(&(a->cs), cs)) {
			if (a->status & CIPHER_STATUS_IMPLEMENTED &&
					a->status & CIPHER_STATUS_NEGOTIABLE)
				return 1;
			return 0;
		}
	}

	return 0;
}       

int
cipher_available_printf(void)
{
	return cipher_available_fprintf(stdout);
}


int
cipher_available_fprintf(FILE *stream)
{
	cipher_suite_info_t *a;

	for (a = (cipher_suite_info_t *)all_cs; a->status; a++) {
		if (!(a->status & CIPHER_STATUS_IMPLEMENTED) ||
				!(a->status & CIPHER_STATUS_NEGOTIABLE))
			continue;
		if (fprintf(stream, "%s\n", a->name) < 0)
			return -1;
	}

	return 0;
}       


int
cipher_list_printf(cipher_suite_t *cs_list, size_t nocs)
{
	return cipher_list_fprintf(stdout, cs_list, nocs);
}


int
cipher_list_fprintf(FILE *stream, cipher_suite_t *cs_list, size_t nocs)
{
	cipher_suite_t *a;
	cipher_suite_t cs_end = CIPHER_SUITE_END_MARKER;

	a = cs_list;
	while (nocs--) {
		if (!cipher_suite_cmp(a, &cs_end))
			break;
		if (fprintf(stream, "%s\n", cipher_name(a)) < 0)
			return -1;
		a++;
	}

	return 0;
}       


static char cipher_name_buf[8];

static const char *
cipher_name(cipher_suite_t *cs)
{
	cipher_suite_info_t *a;

	for (a = (cipher_suite_info_t *)all_cs; a->status; a++) {
		if (!cipher_suite_cmp(&(a->cs), cs))
			return a->name;
	}

	/* Not in list, just return a string represtentation of the
	 * numeric value */

	if(snprintf(cipher_name_buf, sizeof(cipher_name_buf), "0x%02x%02x",
			cs->cs[0] & 0xff, 
			cs->cs[1] & 0xff) >= sizeof(cipher_name_buf))
		return("Error");

	return cipher_name_buf;
}       


const cipher_suite_t *
cipher_byname(const char *name)
{
	cipher_suite_info_t *a;

	for(a = (cipher_suite_info_t *)all_cs; a->status; a++) {
		if (!strcasecmp(a->name, name)) {
			return &(a->cs);
		}
	}

	return NULL;
}


int
cipher_byname_list_alloc(const char *cipher_str, const char delimiter,
		cipher_suite_t **cs_list_ret, size_t *nocs_ret)
{
	int status = -1;
	size_t i;
	size_t nocs;
	cipher_suite_t cs_end = CIPHER_SUITE_END_MARKER;
	char *cs_str;
	char *cipher_str_cpy = NULL;
	cipher_suite_t *cs_list = NULL;
	const cipher_suite_t *cs;
	vanessa_dynamic_array_t *da = NULL;

	if(!cipher_str)
		return -1;

	cipher_str_cpy = strdup(cipher_str);
	if (!cipher_str_cpy)
		goto leave;

	da = vanessa_dynamic_array_split_str(cipher_str_cpy, delimiter);
	if (!da)
		goto leave;

	nocs = vanessa_dynamic_array_get_count(da);

	cs_list = (cipher_suite_t *)malloc(sizeof(cipher_suite_t) * 
			(nocs + 1));
	if (!cs_list)
		goto leave;

	*cs_list_ret = cs_list;
	*nocs_ret = nocs;

	for (i = 0; i < nocs; i++) {
		cs_str = (char *)vanessa_dynamic_array_get_element(da, i);
		if (!cs_str)
			goto leave;
		cs = cipher_byname(cs_str);
		if (!cs)
			goto leave;
		if (!cipher_available(cs))
			goto leave;
		memcpy(cs_list, cs, sizeof(cipher_suite_t));	
		cs_list++;
	}
	memcpy(cs_list, &cs_end, sizeof(cipher_suite_t));	

	status = 0;
leave:
	if (cipher_str_cpy)
		free(cipher_str_cpy);
	if (da)
		vanessa_dynamic_array_destroy(da);
	if (status < 0 && *cs_list_ret)
		free(*cs_list_ret);
	return status;
}

