/**********************************************************************
 * conn.c                                                   August 2005
 *
 * KSSLD: An implementation of SSL/TLS in the Linux Kernel
 * Copyright (C) 2005  NTT COMWARE Corporation.
 *
 * This file based in part on code from LVS www.linuxvirtualserver.org
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 *
 **********************************************************************/

#include <linux/kernel.h>
#include <linux/net.h>
#include <linux/version.h>
#include <linux/crypto.h>
#include <linux/random.h>

#include <net/tcp.h>

#include "conn.h"
#include "record.h"
#include "socket.h"
#include "message_mask.h"
#include "prf.h"
#include "kssl_alloc.h"
#include "log.h"
#include "session.h"

#ifndef MIN
#define MIN(a,b) ((a) < (b) ? (a) : (b))
#endif

/*
 * The maximum number of records associated with a connection.
 * 2 are needed to recieve messages from the ssl and plaintext
 * sockets respectivey. Putting a bound on the rest limits
 * the number of messsages that will be queued for further
 * output, thus bounding the memory used. When the limit
 * is reached no additional records will be read, this
 * effectively throttles the connection 
 *
 * 32 seems to be big enough to prevent the connection
 * from using an unbounded ammount of memory without effecting
 * performance.
 * */


/* XXX: This should be set via proc or sysctl */
#define KSSL_CONN_USERS_MAX 32

static void __kssl_conn_finish_close(kssl_conn_t *conn);

static inline security_parameters_t *
__kssl_conn_new_pend_sec_param(kssl_conn_t *conn, connection_end_t entity)
{
	security_parameters_t *sec_param;

	sec_param = (security_parameters_t *)kssl_kmalloc(
			sizeof(security_parameters_t), GFP_KERNEL);
       	if (!sec_param) {
		KSSL_DEBUG(6, "__kssl_conn_new_pend_sec_param: kssl_kmalloc\n");
		return NULL;
	}

	security_parameters_init(sec_param, entity);

	return sec_param;
}

kssl_conn_t *
kssl_conn_create(kssl_daemon_t *daemon, connection_end_t entity)
{
	kssl_conn_t *conn;

	conn = (kssl_conn_t *)kssl_kmalloc(sizeof(kssl_conn_t), GFP_KERNEL);
	if (!conn)
		goto error;
	memset(conn, 0, sizeof(kssl_conn_t));

	conn->sec_param_in_pend = conn->sec_param_out_pend =
		__kssl_conn_new_pend_sec_param(conn, entity);
	if (!conn->sec_param_in_pend)
		goto error;

	conn->conn_state.hs_digest_md5 = crypto_alloc_tfm("md5", 0);
	if (!conn->conn_state.hs_digest_md5)
		goto error;
	crypto_digest_init(conn->conn_state.hs_digest_md5);

	conn->conn_state.hs_digest_sha1 = crypto_alloc_tfm("sha1", 0);
	if (!conn->conn_state.hs_digest_sha1)
		goto error;
	crypto_digest_init(conn->conn_state.hs_digest_sha1);

	conn->msg_mask = KSSL_CONN_M_ALERT | KSSL_CONN_M_H_CLIENT_HELLO;

	kssl_daemon_get(daemon);
	conn->daemon = daemon;


	INIT_LIST_HEAD(&(conn->list));
	kssl_conn_get(conn);

	spin_lock_init(&(conn->sock_lock));

	return conn;

error:
	if (conn) {
		if (conn->sec_param_in_pend)
			kssl_kfree(conn->sec_param_in_pend);
		if (conn->conn_state.hs_digest_md5)
			kssl_kfree(conn->conn_state.hs_digest_md5);
		if (conn->conn_state.hs_digest_sha1)
			kssl_kfree(conn->conn_state.hs_digest_sha1);
		kssl_kfree(conn);
	}
	return NULL;
}

static int 
__kssl_conn_encrypt_init(kssl_conn_t *conn, security_parameters_t *sec_param) 

{
	int status = 0;
	char *name = "unknown";
	u32 flags = 0;

	KSSL_DEBUG(12, "__kssl_conn_encrypt_init: enter\n");

	if (conn->conn_state.enc) {
		crypto_free_tfm(conn->conn_state.enc);
		conn->conn_state.enc = NULL;
	}

	if (conn->conn_state.dec) {
		crypto_free_tfm(conn->conn_state.dec);
		conn->conn_state.dec = NULL;
	}

	switch(sec_param->bulk_cipher_algorithm) {
		case bca_3des:
			name = "des3_ede";
			flags =  CRYPTO_TFM_MODE_CBC;
			break;
		case bca_des:
		case bca_des40:
			name = "des";
			flags =  CRYPTO_TFM_MODE_CBC;
			break;
		case bca_aes_128:
		case bca_aes_256:
			name = "aes";
			flags =  CRYPTO_TFM_MODE_CBC;
			break;
		case bca_null:
			return 0;
	        case bca_rc4:
	        case bca_rc2:
		case bca_idea:
		default:
			/* Unknown/Unsuported */
			KSSL_DEBUG(3, "__kssl_conn_encrypt_init: "
				"unknown/unsupported algorithm (%d)\n",
				sec_param->bulk_cipher_algorithm);
			return -EINVAL;
	}

	conn->conn_state.enc = crypto_alloc_tfm(name, flags);
	conn->conn_state.dec = crypto_alloc_tfm(name, flags);

	if (!conn->conn_state.enc || !conn->conn_state.dec) {
		KSSL_DEBUG(6, "__kssl_conn_encrypt_init: "
				"error initialising %s\n", name);
		status = -ENOMEM;
		goto error;
	}

	status = crypto_cipher_setkey(conn->conn_state.enc, 
			conn->conn_state.server_key, sec_param->key_size);
	if (status < 0) {
		KSSL_DEBUG(6, "__kssl_conn_encrypt_init: "
				"error setting key (enc)\n");
		goto error;
	}
	status = crypto_cipher_setkey(conn->conn_state.dec, 
			conn->conn_state.client_key, sec_param->key_size);
	if (status < 0) {
		KSSL_DEBUG(6, "__kssl_conn_encrypt_init: "
				"error setting key (dec)\n");
		goto error;
	}

	if (sec_param->cipher_type == ct_block) {
		crypto_cipher_set_iv(conn->conn_state.enc, 
				conn->conn_state.server_iv, 
				sec_param->block_size);
		crypto_cipher_set_iv(conn->conn_state.dec, 
				conn->conn_state.client_iv,
				sec_param->block_size);
	}

	return 0;

error:
	if (conn->conn_state.enc) {
		kssl_kfree(conn->conn_state.enc);
		conn->conn_state.enc = NULL;
	}
	if (conn->conn_state.dec) {
		kssl_kfree(conn->conn_state.dec);
		conn->conn_state.dec = NULL;
	}
	return status;
}


static int 
__kssl_conn_activate_sec_param(kssl_conn_t *conn,
		security_parameters_t **a_pend, security_parameters_t **a_act,
		security_parameters_t **b_pend, security_parameters_t **b_act)
{
	int status;

	if (*a_act)
		kssl_kfree(*a_act);
	*a_act = *a_pend;

	if (*a_act == *b_act) {
		*a_pend = *b_pend;
		return 0;
	}

	status = __kssl_conn_encrypt_init(conn, *a_act);
	if (status < 0) {
		KSSL_DEBUG(6, "__kssl_conn_activate_sec_param: "
				"__kssl_conn_encrypt_init: failed\n");
		return status;
	}

	*a_pend = __kssl_conn_new_pend_sec_param(conn, (*a_act)->entity);
	if (!*a_pend) {
		KSSL_DEBUG(6, "__kssl_conn_activate_sec_param: "
				"__kssl_conn_new_pend_sec_param: failed\n");
		return -ENOMEM;
	}

	return 0;
}


int kssl_conn_activate_sec_param_in(kssl_conn_t *conn)
{
	KSSL_DEBUG(12, "kssl_conn_activate_sec_param_in: enter\n");
	return __kssl_conn_activate_sec_param(conn, 
			&conn->sec_param_in_pend, &conn->sec_param_in_act, 
			&conn->sec_param_out_pend, &conn->sec_param_out_act);
}


int kssl_conn_activate_sec_param_out(kssl_conn_t *conn)
{
	KSSL_DEBUG(12, "kssl_conn_activate_sec_param_out: enter\n");
	return __kssl_conn_activate_sec_param(conn,
			&conn->sec_param_out_pend, &conn->sec_param_out_act, 
			&conn->sec_param_in_pend, &conn->sec_param_in_act);
}


static void __kssl_clear_keys_sec_param(security_parameters_t *sp)
{
	memset(sp->master_secret, 0, MASTER_SECRET_LEN);
}


static void __kssl_conn_clear_keys(kssl_conn_t *conn) {
	security_parameters_t *sec_param;

	if (conn->sec_param_in_act)
		sec_param = conn->sec_param_in_act;
	if (conn->sec_param_in_pend)
		sec_param = conn->sec_param_in_pend;
	else
		return;

	KSSL_DEBUG(12, "__kssl_clear_keys_sec_param: enter: %p %p\n", 
			conn, sec_param);

	if (conn->conn_state.client_mac_secret) {
		memset(conn->conn_state.client_mac_secret, 0, 
				sec_param->hash_size);
		kssl_kfree(conn->conn_state.client_mac_secret);
		conn->conn_state.client_mac_secret = NULL;
	}
  
	if (conn->conn_state.server_mac_secret) {
		memset(conn->conn_state.server_mac_secret, 0, 
				sec_param->hash_size);
		kssl_kfree(conn->conn_state.server_mac_secret);
		conn->conn_state.server_mac_secret = NULL;
	}
 
	if (conn->conn_state.client_key) {
		memset(conn->conn_state.client_key, 0, sec_param->key_size);
		kssl_kfree(conn->conn_state.client_key);
		conn->conn_state.client_key = NULL;
	}
 
	if (conn->conn_state.server_key) {
		memset(conn->conn_state.server_key, 0, sec_param->key_size);
		kssl_kfree(conn->conn_state.server_key);
		conn->conn_state.server_key = NULL;
	}

	if (conn->conn_state.client_iv) {
		memset(conn->conn_state.client_iv, 0, sec_param->block_size);
		kssl_kfree(conn->conn_state.client_iv);
		conn->conn_state.client_iv = NULL;
	}
 
	if (conn->conn_state.server_iv) {
		memset(conn->conn_state.server_iv, 0, sec_param->block_size);
		kssl_kfree(conn->conn_state.server_iv);
		conn->conn_state.server_iv = NULL;
	}
}


static void 
__kssl_conn_clear_crypto(kssl_conn_t *conn) {
	if (conn->conn_state.hs_digest_md5) {
		crypto_free_tfm(conn->conn_state.hs_digest_md5);
		conn->conn_state.hs_digest_md5 = NULL;
	}

	if (conn->conn_state.hs_digest_sha1) {
		crypto_free_tfm(conn->conn_state.hs_digest_sha1);
		conn->conn_state.hs_digest_sha1 = NULL;
	}

	if (conn->conn_state.enc) {
		crypto_free_tfm(conn->conn_state.enc);
		conn->conn_state.enc = NULL;
	}

	if (conn->conn_state.dec) {
		crypto_free_tfm(conn->conn_state.dec);
		conn->conn_state.dec = NULL;
	}

	if (conn->conn_state.mac) {
		crypto_free_tfm(conn->conn_state.mac);
		conn->conn_state.mac = NULL;
	}
}

int 
kssl_conn_cpy_digest(kssl_conn_t *conn, struct crypto_tfm **hs_digest_md5,
		struct crypto_tfm **hs_digest_sha1)
{
	*hs_digest_md5 = crypto_cpy_tfm(conn->conn_state.hs_digest_md5);
	if (!*hs_digest_md5) {
		return -ENOMEM;
	}
	*hs_digest_sha1 = crypto_cpy_tfm(conn->conn_state.hs_digest_sha1);
	if (!*hs_digest_sha1) {
		crypto_free_tfm(*hs_digest_md5);
		return -ENOMEM;
	}

	return 0;
}


static void __kssl_conn_finish_close(kssl_conn_t *conn)
{
	spin_lock(&(conn->sock_lock));

	if (conn->ssl_sock) {
		KSSL_DEBUG(3, "Close (ssl): %p: "
				"%08x:%04x->%08x:%04x in=%08x out=%08x\n", 
				conn, ntohl(conn->ssl_sock->sk->daddr), 
				ntohs(conn->ssl_sock->sk->dport),
				ntohl(conn->ssl_sock->sk->saddr), 
				ntohs(conn->ssl_sock->sk->sport),
				conn->ssl_in_bytes, conn->ssl_out_bytes);
		kssl_socket_close(conn->ssl_sock);
		conn->ssl_sock = NULL;
	}

	if (conn->pt_sock) {
		KSSL_DEBUG(3, "Close (pt):  %p: "
				"%08x:%04x->%08x:%04x in=%08x out=%08x\n", 
				conn, ntohl(conn->pt_sock->sk->saddr), 
				ntohs(conn->pt_sock->sk->sport),
				ntohl(conn->pt_sock->sk->daddr),
				ntohs(conn->pt_sock->sk->dport),
				conn->pt_in_bytes, conn->pt_out_bytes);
		kssl_socket_close(conn->pt_sock);
		conn->pt_sock = NULL;
	}

	spin_unlock(&(conn->sock_lock));
}


void __kssl_conn_start_close(kssl_conn_t *conn, unsigned int flag) 
{
	conn->flag |= (flag & KSSL_CONN_ALL_CLOSE);
	if ((flag & KSSL_CONN_SSL_CLOSE) && conn->ssl_record) {
		kssl_record_destroy(conn->ssl_record);
		conn->ssl_record = NULL;
	}
	
	if ((flag & KSSL_CONN_PT_CLOSE) && conn->pt_record) {
		kssl_record_destroy(conn->pt_record);
		conn->pt_record = NULL;
	}

	if ((conn->flag & KSSL_CONN_ALL_CLOSE) == KSSL_CONN_ALL_CLOSE ||
			((conn->flag & KSSL_CONN_SSL_CLOSE) && 
			 !conn->pt_sock) ||
			((conn->flag & KSSL_CONN_PT_CLOSE) && 
			 !conn->ssl_sock)) {
		conn->flag |= KSSL_CONN_ALL_CLOSE;
		list_move(&(conn->list), &kssl_conn_close_list);
	}
}


void 
__kssl_conn_destroy(kssl_conn_t *conn)
{
        kssl_conn_close(conn, KSSL_CONN_ALL_CLOSE);
        if (conn->users != 1)
		return;

	list_del(&(conn->list));
	__kssl_conn_finish_close(conn);
	__kssl_conn_clear_keys(conn);
	__kssl_conn_clear_crypto(conn);
	if (conn->sec_param_in_pend) {
		__kssl_clear_keys_sec_param(conn->sec_param_in_pend);
		kssl_kfree(conn->sec_param_in_pend);
	}
	if (conn->sec_param_in_act) {
		__kssl_clear_keys_sec_param(conn->sec_param_in_act);
		kssl_kfree(conn->sec_param_in_act);
	}
	if (conn->sec_param_out_pend && 
			conn->sec_param_out_pend != conn->sec_param_in_pend &&
			conn->sec_param_out_pend != conn->sec_param_in_act) {
		__kssl_clear_keys_sec_param(conn->sec_param_out_pend);
		kssl_kfree(conn->sec_param_out_pend);
	}
	if (conn->sec_param_out_act && 
			conn->sec_param_out_act != conn->sec_param_in_act &&
			conn->sec_param_out_act != conn->sec_param_in_pend) {
		__kssl_clear_keys_sec_param(conn->sec_param_out_act);
		kssl_kfree(conn->sec_param_out_act);
	}
	kssl_daemon_put(conn->daemon);
	kssl_kfree(conn);
}


int kssl_conn_accept(kssl_daemon_t *daemon, kssl_conn_t **conn)
{
	int status;
	struct socket *sock;

	kssl_daemon_get_read(daemon);

	sock = daemon->sock;
	kssl_daemon_put_read(daemon);

	if (!sock->sk->tp_pinfo.af_tcp.accept_queue)
		return 0;

	*conn = kssl_conn_create(daemon, ce_server);
	if (!*conn)
		return -ENOMEM;

	spin_lock(&((*conn)->sock_lock));
	status = kssl_socket_accept(sock, &((*conn)->ssl_sock));
	if (status <= 0) {
		kssl_conn_destroy(*conn);
		return status;
	}
	spin_unlock(&((*conn)->sock_lock));

	KSSL_DEBUG(3, "Open (ssl):  %p: %08x:%04x->%08x:%04x\n", *conn,
			ntohl((*conn)->ssl_sock->sk->daddr), 
			ntohs((*conn)->ssl_sock->sk->dport),
			ntohl((*conn)->ssl_sock->sk->saddr), 
			ntohs((*conn)->ssl_sock->sk->sport));

	KSSL_NOTICE(3, "ISSL001: Info(socket): Client connection established\n");

	return status;
}


int
kssl_conn_open_pt(kssl_conn_t *conn) 
{
	u32 raddr;
	u16 rport;
	int status;

	if (conn->flag & KSSL_CONN_PT_CLOSE) {
		KSSL_DEBUG(6, "kssl_record_process_out: conn is closed\n");
		return -EBADF;
	}

	kssl_daemon_get_read(conn->daemon);
	raddr = conn->daemon->raddr;
	rport = conn->daemon->rport;
	kssl_daemon_put_read(conn->daemon);

	if (raddr == 0 || rport == 0) {
		KSSL_DEBUG(6, "kssl_record_process_out: "
				"null real server %08x:%04x\n", raddr, rport);
		kssl_conn_close(conn, KSSL_CONN_ALL_CLOSE);
		return -EBADF;
	}

	spin_lock(&(conn->sock_lock));
	status = kssl_socket_inet_open_connect(raddr, rport, &(conn->pt_sock));
	if (status < 0) {
		KSSL_DEBUG(6, "kssl_record_process_out: "
				"kssl_socket_open_connect\n");
		return status;
	}
	spin_unlock(&(conn->sock_lock));

	KSSL_DEBUG(3, "Open (pt):   %p: %08x:%04x->%08x:%04x\n", conn,
			ntohl(conn->pt_sock->sk->saddr), 
			ntohs(conn->pt_sock->sk->sport),
			ntohl(conn->pt_sock->sk->daddr), 
			ntohs(conn->pt_sock->sk->dport));

	return 0;
}


typedef struct {
	kssl_conn_t *conn;
	int error;
	int conn_flag;
} kssl_read_descriptor_t;


static int kssl_conn_recv_skb(read_descriptor_t *rd_desc, 
		struct sk_buff *skb, size_t offset, size_t len)
{
	int status;
	kssl_conn_t *conn;
	kssl_record_t **record;
	kssl_read_descriptor_t *kssl_rd_desc;

	KSSL_DEBUG(12, "kssl_conn_recv_skb: enter\n");

	if (!len)
		return 0;

	kssl_rd_desc = (kssl_read_descriptor_t *)(rd_desc->buf);
	conn = kssl_rd_desc->conn;

	/*
	if (conn->flag & KSSL_CONN_ALL_CLOSE)
		return 0; */

	if (kssl_rd_desc->conn_flag == KSSL_CONN_PT)
		record = &(conn->pt_record);
	else
		record = &(conn->ssl_record);

	if (!*record) {
		*record = kssl_record_create(conn, kssl_rd_desc->conn_flag);
		if (!*record) {
			KSSL_DEBUG(6, "kssl_conn_recv_skb: "
					"kssl_record_create\n");
			kssl_rd_desc->error = -ENOMEM;
			return len-offset;
		}
	}

	status = kssl_record_add_skb(*record, skb, offset, len);
	if (status < 0) {
		KSSL_DEBUG(6, "kssl_conn_recv_skb: kssl_record_add_skb\n");
		kssl_record_destroy(*record);
		*record = NULL;
		kssl_rd_desc->error = status;
		return len-offset;
	}

	while(1) {
		status = kssl_record_update(record);
		if (status < 0) {
			KSSL_DEBUG(6, "kssl_conn_recv_skb: "
					"kssl_record_update\n");
			if(*record) {
				kssl_record_destroy(*record);
				*record = NULL;
			}
			kssl_rd_desc->error = status;
			break;
		}
		if (!*record || !status)
			break;
	}

	return len-offset;
}


int 
__kssl_conn_recv(kssl_conn_t *conn, u32 conn_flag)
{       
	int status = 0;
	struct sock *sk;
	read_descriptor_t rd_desc;
	kssl_read_descriptor_t kssl_rd_desc;

	if (conn->users >= KSSL_CONN_USERS_MAX)
		return -EAGAIN;

	sk = conn_flag == KSSL_CONN_PT ? conn->pt_sock->sk : 
		conn->ssl_sock->sk;

	if (sk->state != TCP_ESTABLISHED && sk->state != TCP_CLOSE_WAIT
			&& sk->state != TCP_CLOSE) {
		KSSL_DEBUG(6, "__kssl_conn_recv: %p: Connection is not in "
				"ESTABLISHED, CLOSE_WAIT or TCP_CLOSE state: "
				"%u\n", conn, sk->state);
		return (-ECONNABORTED);
	}
	
	if (skb_queue_empty(&(sk->receive_queue))) {
		if (sk->state == TCP_CLOSE_WAIT || sk->state == TCP_CLOSE)
			return 0;
		return -EAGAIN;
	}

	kssl_rd_desc.error = 0;
	kssl_rd_desc.conn_flag = conn_flag;
	kssl_rd_desc.conn = conn;

	rd_desc.buf = (u8 *)&kssl_rd_desc;
	rd_desc.count = 65535;

	lock_sock(sk);
	status = tcp_read_sock(sk, &rd_desc, kssl_conn_recv_skb);
	release_sock(sk);
	if (status < 0) {
		KSSL_DEBUG(6, "__kssl_conn_recv: tcp_read_sock\n");
		return status;
	}
	if (kssl_rd_desc.error < 0) {
		KSSL_DEBUG(6, "__kssl_conn_recv: tcp_read_sock\n");
		return kssl_rd_desc.error;
	}

	if (conn_flag == KSSL_CONN_PT)
		conn->pt_in_bytes += status;
	else
		conn->ssl_in_bytes += status;

	if (!status) {
		KSSL_DEBUG(6, "__kssl_conn_recv: 0 bytes: "
				"conn=%p conn_flag=%d\n", conn, conn_flag);
		return -EAGAIN;
	}

	return status;
}


int 
kssl_conn_recv(kssl_conn_t *conn)
{       
	int status = 0;

	if (conn->ssl_sock && ! (conn->flag & KSSL_CONN_SSL_CLOSE)) {
		status = __kssl_conn_recv(conn, KSSL_CONN_SSL);
		if (status != -EAGAIN)
			KSSL_DEBUG(4, "Recv (ssl):  %p: %d\n", 
					conn, status);
		if (status <= 0 && status != -EAGAIN) {
			if (conn->users == 1)
				kssl_conn_close(conn, KSSL_CONN_ALL_CLOSE);
			else
				kssl_conn_close(conn, KSSL_CONN_SSL_CLOSE);
			return status;
		}
	}

	if (conn->pt_sock && ! (conn->flag & KSSL_CONN_PT_CLOSE)) {
		status = __kssl_conn_recv(conn, KSSL_CONN_PT);
		if (status != -EAGAIN)
			KSSL_DEBUG(4, "Recv (pt):   %p: %d\n", 
					conn, status);
		if (status <= 0 && status != -EAGAIN) {
			if (conn->users == 1)
				kssl_conn_close(conn, KSSL_CONN_ALL_CLOSE);
			else
				kssl_conn_close(conn, KSSL_CONN_PT_CLOSE);
			return status;
		}
	}

	return status;
}


static int 
__kssl_conn_set_cipher_suite_pending(kssl_conn_t *conn,
		cipher_suite_t *cs) 
{
	security_parameters_t *sp;

	KSSL_DEBUG(12, "__kssl_conn_set_cipher_suite_pending: enter\n");
	sp = conn->sec_param_in_pend;

	/* Set Mac Algorithm */
	switch(cipher_suite_toi(cs)) {
		case TLS_NULL_WITH_NULL_NULL_I:
			sp->mac_algorithm = ma_null;
			break;

		case TLS_RSA_WITH_NULL_MD5_I:
		case TLS_RSA_EXPORT_WITH_RC4_40_MD5_I:
		case TLS_RSA_EXPORT_WITH_RC2_CBC_40_MD5_I:
		case TLS_DH_anon_EXPORT_WITH_RC4_40_MD5_I:
		case TLS_RSA_WITH_RC4_128_MD5_I:
		case TLS_DH_anon_WITH_RC4_128_MD5_I:
			sp->mac_algorithm = ma_md5;
			sp->hash_size = 16;
			break;

		case TLS_RSA_WITH_NULL_SHA_I:
		case TLS_RSA_WITH_IDEA_CBC_SHA_I:
		case TLS_RSA_WITH_RC4_128_SHA_I:
		case TLS_DHE_DSS_WITH_RC4_128_SHA_I:
		case TLS_DH_DSS_EXPORT_WITH_DES40_CBC_SHA_I:
		case TLS_RSA_EXPORT_WITH_DES40_CBC_SHA_I:
		case TLS_DH_RSA_EXPORT_WITH_DES40_CBC_SHA_I:
		case TLS_DHE_DSS_EXPORT_WITH_DES40_CBC_SHA_I:
		case TLS_DHE_RSA_EXPORT_WITH_DES40_CBC_SHA_I:
		case TLS_DH_anon_EXPORT_WITH_DES40_CBC_SHA_I:
		case TLS_RSA_WITH_DES_CBC_SHA_I:
		case TLS_DH_DSS_WITH_DES_CBC_SHA_I:
		case TLS_DH_RSA_WITH_DES_CBC_SHA_I:
		case TLS_DHE_DSS_WITH_DES_CBC_SHA_I:
		case TLS_DHE_RSA_WITH_DES_CBC_SHA_I:
		case TLS_DH_anon_WITH_DES_CBC_SHA_I:
		case TLS_RSA_EXPORT1024_WITH_DES_CBC_SHA_I:
		case TLS_DHE_DSS_EXPORT1024_WITH_DES_CBC_SHA_I:
		case TLS_RSA_WITH_3DES_EDE_CBC_SHA_I:
		case TLS_DH_DSS_WITH_3DES_EDE_CBC_SHA_I:
		case TLS_DH_RSA_WITH_3DES_EDE_CBC_SHA_I:
		case TLS_DHE_DSS_WITH_3DES_EDE_CBC_SHA_I:
		case TLS_DHE_RSA_WITH_3DES_EDE_CBC_SHA_I:
		case TLS_DH_anon_WITH_3DES_EDE_CBC_SHA_I:
		case TLS_RSA_WITH_AES_128_CBC_SHA_I:
		case TLS_DH_DSS_WITH_AES_128_CBC_SHA_I:
		case TLS_DH_RSA_WITH_AES_128_CBC_SHA_I:
		case TLS_DHE_DSS_WITH_AES_128_CBC_SHA_I:
		case TLS_DHE_RSA_WITH_AES_128_CBC_SHA_I:
		case TLS_DH_anon_WITH_AES_128_CBC_SHA_I:
		case TLS_RSA_WITH_AES_256_CBC_SHA_I:
		case TLS_DH_DSS_WITH_AES_256_CBC_SHA_I:
		case TLS_DH_RSA_WITH_AES_256_CBC_SHA_I:
		case TLS_DHE_DSS_WITH_AES_256_CBC_SHA_I:
		case TLS_DHE_RSA_WITH_AES_256_CBC_SHA_I:
		case TLS_DH_anon_WITH_AES_256_CBC_SHA_I:
			sp->mac_algorithm = ma_sha;
			sp->hash_size = 20;
			break;

		/* case CIPHER_SUITE_END_MARKER_I: */
		default:
			KSSL_DEBUG(3, "__kssl_conn_set_cipher_suite_pending: "
					"unknown cipher suite 1: %04x\n",
					cipher_suite_toi(cs));
			return -EINVAL;
	}

	/* Set Encryption Algorithm */
	switch(cipher_suite_toi(cs)) {
		case TLS_NULL_WITH_NULL_NULL_I:
		case TLS_RSA_WITH_NULL_MD5_I:
		case TLS_RSA_WITH_NULL_SHA_I:
			/* NULL */
			sp->bulk_cipher_algorithm = bca_null;
			sp->cipher_type =           ct_stream;
			sp->key_size =              0;
			sp->key_material_length =   0;
			sp->is_exportable =         1;
			break;

		case TLS_RSA_WITH_IDEA_CBC_SHA_I:
			/* IDEA_CBC */
			sp->bulk_cipher_algorithm = bca_idea;
			sp->cipher_type =           ct_block;
			sp->block_size =	    8;
			sp->key_size =              16;
			sp->key_material_length =   16;
			sp->is_exportable =         0;
			break;

		case TLS_RSA_EXPORT_WITH_RC2_CBC_40_MD5_I:
			/* RC2_CBC_40 */
			sp->bulk_cipher_algorithm = bca_rc2;
			sp->cipher_type =           ct_block;
			sp->block_size =	    8;
			sp->key_size =              16;
			sp->key_material_length =   5;
			sp->is_exportable =         1;
			break;

		case TLS_RSA_EXPORT_WITH_RC4_40_MD5_I:
		case TLS_DH_anon_EXPORT_WITH_RC4_40_MD5_I:
			/* RC4_40 */
			sp->bulk_cipher_algorithm = bca_rc4;
			sp->cipher_type =           ct_stream;
			sp->key_size =              16;
			sp->key_material_length =   5;
			sp->is_exportable =         1;
			break;

		case TLS_RSA_WITH_RC4_128_MD5_I:
		case TLS_RSA_WITH_RC4_128_SHA_I:
		case TLS_DH_anon_WITH_RC4_128_MD5_I:
		case TLS_DHE_DSS_WITH_RC4_128_SHA_I:
			/* RC4_128 */
			sp->bulk_cipher_algorithm = bca_rc4;
			sp->cipher_type =           ct_stream;
			sp->key_size =              16;
			sp->key_material_length =   16;
			sp->is_exportable =         0;
			break;

		case TLS_DH_DSS_EXPORT_WITH_DES40_CBC_SHA_I:
		case TLS_RSA_EXPORT_WITH_DES40_CBC_SHA_I:
		case TLS_DH_RSA_EXPORT_WITH_DES40_CBC_SHA_I:
		case TLS_DHE_DSS_EXPORT_WITH_DES40_CBC_SHA_I:
		case TLS_DHE_RSA_EXPORT_WITH_DES40_CBC_SHA_I:
		case TLS_DH_anon_EXPORT_WITH_DES40_CBC_SHA_I:
			/* DES40_CBC */
			sp->bulk_cipher_algorithm = bca_des40;
			sp->cipher_type =           ct_block;
			sp->block_size =	    8;
			sp->key_size =              8;
			sp->key_material_length =   5;
			sp->is_exportable =         1;
			break;

		case TLS_RSA_EXPORT1024_WITH_DES_CBC_SHA_I:
		case TLS_DHE_DSS_EXPORT1024_WITH_DES_CBC_SHA_I:
			/* DES_CBC EXPORT_1024 */
			sp->bulk_cipher_algorithm = bca_des;
			sp->cipher_type =           ct_block;
			sp->block_size =	    8;
			sp->key_size =              8;
			sp->key_material_length =   8;
			sp->is_exportable =         1;
			break;

		case TLS_RSA_WITH_DES_CBC_SHA_I:
		case TLS_DH_DSS_WITH_DES_CBC_SHA_I:
		case TLS_DH_RSA_WITH_DES_CBC_SHA_I:
		case TLS_DHE_DSS_WITH_DES_CBC_SHA_I:
		case TLS_DHE_RSA_WITH_DES_CBC_SHA_I:
		case TLS_DH_anon_WITH_DES_CBC_SHA_I:
			/* DES_CBC */
			sp->bulk_cipher_algorithm = bca_des;
			sp->cipher_type =           ct_block;
			sp->block_size =	    8;
			sp->key_size =              8;
			sp->key_material_length =   8;
			sp->is_exportable =         0;
			break;

		case TLS_RSA_WITH_3DES_EDE_CBC_SHA_I:
		case TLS_DH_DSS_WITH_3DES_EDE_CBC_SHA_I:
		case TLS_DH_RSA_WITH_3DES_EDE_CBC_SHA_I:
		case TLS_DHE_DSS_WITH_3DES_EDE_CBC_SHA_I:
		case TLS_DHE_RSA_WITH_3DES_EDE_CBC_SHA_I:
		case TLS_DH_anon_WITH_3DES_EDE_CBC_SHA_I:
			/* 3DES_EDE_CBC */
			sp->bulk_cipher_algorithm = bca_3des;
			sp->cipher_type =           ct_block;
			sp->block_size =	    8;
			sp->key_size =              24;
			sp->key_material_length =   24;
			sp->is_exportable =         0;
			break;

		case TLS_RSA_WITH_AES_128_CBC_SHA_I:
		case TLS_DH_DSS_WITH_AES_128_CBC_SHA_I:
		case TLS_DH_RSA_WITH_AES_128_CBC_SHA_I:
		case TLS_DHE_DSS_WITH_AES_128_CBC_SHA_I:
		case TLS_DHE_RSA_WITH_AES_128_CBC_SHA_I:
		case TLS_DH_anon_WITH_AES_128_CBC_SHA_I:
			/* AES 128 */
			sp->bulk_cipher_algorithm = bca_aes_128;
			sp->cipher_type =           ct_block;
			sp->block_size =	    16;
			sp->key_size =              16;
			sp->key_material_length =   16;
			sp->is_exportable =         0;
			break;

		case TLS_RSA_WITH_AES_256_CBC_SHA_I:
		case TLS_DH_DSS_WITH_AES_256_CBC_SHA_I:
		case TLS_DH_RSA_WITH_AES_256_CBC_SHA_I:
		case TLS_DHE_DSS_WITH_AES_256_CBC_SHA_I:
		case TLS_DHE_RSA_WITH_AES_256_CBC_SHA_I:
		case TLS_DH_anon_WITH_AES_256_CBC_SHA_I:
			/* AES 256 */
			sp->bulk_cipher_algorithm = bca_aes_256;
			sp->cipher_type =           ct_block;
			sp->block_size =	    16;
			sp->key_size =              32;
			sp->key_material_length =   32;
			sp->is_exportable =         0;
			break;

		case TLS_RSA_EXPORT1024_WITH_RC4_56_SHA_I:
		case TLS_DHE_DSS_EXPORT1024_WITH_RC4_56_SHA_I:
			/* RC4_56 */
			sp->bulk_cipher_algorithm = bca_rc4;
			sp->cipher_type =           ct_stream;
			sp->key_size =              16;
			sp->key_material_length =   7;
			sp->is_exportable =         1;
			break;

		/* case CIPHER_SUITE_END_MARKER_I: */
		default:
			KSSL_DEBUG(3, "__kssl_conn_set_cipher_suite_pending: "
					"unknown cipher suite 2: %04x\n",
					cipher_suite_toi(cs));
			return -EINVAL;
	}

	/* Set  Authorisation Algorithm */
	switch(cipher_suite_toi(cs)) {
		case TLS_NULL_WITH_NULL_NULL_I:
		case TLS_DH_anon_EXPORT_WITH_RC4_40_MD5_I:
		case TLS_DH_anon_WITH_RC4_128_MD5_I:
		case TLS_DH_anon_EXPORT_WITH_DES40_CBC_SHA_I:
		case TLS_DH_anon_WITH_DES_CBC_SHA_I:
		case TLS_DH_anon_WITH_3DES_EDE_CBC_SHA_I:
			conn->conn_state.kes.aa = kssl_aa_null;
			break;

		case TLS_RSA_WITH_NULL_MD5_I:
		case TLS_RSA_WITH_RC4_128_MD5_I:
		case TLS_RSA_WITH_NULL_SHA_I:
		case TLS_RSA_WITH_IDEA_CBC_SHA_I:
		case TLS_RSA_EXPORT_WITH_RC2_CBC_40_MD5_I:
		case TLS_RSA_EXPORT_WITH_RC4_40_MD5_I:
		case TLS_RSA_WITH_RC4_128_SHA_I:
		case TLS_RSA_EXPORT_WITH_DES40_CBC_SHA_I:
		case TLS_RSA_WITH_DES_CBC_SHA_I:
		case TLS_RSA_EXPORT1024_WITH_DES_CBC_SHA_I:
		case TLS_RSA_WITH_3DES_EDE_CBC_SHA_I:
		case TLS_DH_RSA_EXPORT_WITH_DES40_CBC_SHA_I:
		case TLS_DHE_RSA_EXPORT_WITH_DES40_CBC_SHA_I:
		case TLS_DH_RSA_WITH_DES_CBC_SHA_I:
		case TLS_DHE_RSA_WITH_DES_CBC_SHA_I:
		case TLS_DH_RSA_WITH_3DES_EDE_CBC_SHA_I:
		case TLS_DHE_RSA_WITH_3DES_EDE_CBC_SHA_I:
		case TLS_RSA_WITH_AES_128_CBC_SHA_I:
		case TLS_RSA_WITH_AES_256_CBC_SHA_I:
		case TLS_DH_RSA_WITH_AES_128_CBC_SHA_I:
		case TLS_DHE_RSA_WITH_AES_128_CBC_SHA_I:
		case TLS_DH_RSA_WITH_AES_256_CBC_SHA_I:
		case TLS_DHE_RSA_WITH_AES_256_CBC_SHA_I:
			conn->conn_state.kes.aa = kssl_aa_rsa;
			break;

		case TLS_DHE_DSS_EXPORT_WITH_DES40_CBC_SHA_I:
		case TLS_DHE_DSS_WITH_DES_CBC_SHA_I:
		case TLS_DHE_DSS_WITH_3DES_EDE_CBC_SHA_I:
		case TLS_DH_DSS_WITH_DES_CBC_SHA_I:
		case TLS_DH_DSS_WITH_3DES_EDE_CBC_SHA_I:
		case TLS_DHE_DSS_EXPORT1024_WITH_DES_CBC_SHA_I:
		case TLS_DHE_DSS_WITH_RC4_128_SHA_I:
		case TLS_DH_DSS_EXPORT_WITH_DES40_CBC_SHA_I:
		case TLS_DH_DSS_WITH_AES_128_CBC_SHA_I:
		case TLS_DHE_DSS_WITH_AES_128_CBC_SHA_I:
		case TLS_DH_anon_WITH_AES_128_CBC_SHA_I:
		case TLS_DH_DSS_WITH_AES_256_CBC_SHA_I:
		case TLS_DHE_DSS_WITH_AES_256_CBC_SHA_I:
		case TLS_DH_anon_WITH_AES_256_CBC_SHA_I:
			conn->conn_state.kes.aa = kssl_aa_dss;
			break;

		/* case CIPHER_SUITE_END_MARKER_I: */
		default:
			KSSL_DEBUG(3, "__kssl_conn_set_cipher_suite_pending: "
					"unknown cipher suite 3: %04x\n",
					cipher_suite_toi(cs));
			return -EINVAL;
	}

	/* Set  Key Exchange Algorithm */
	conn->conn_state.kes.key_limit = 0;
	switch(cipher_suite_toi(cs)) {
		case TLS_NULL_WITH_NULL_NULL_I:
			conn->conn_state.kes.kea = kssl_kea_anon;
			conn->conn_state.kes.sa = kssl_sa_null;
			break;

		case TLS_DHE_DSS_EXPORT1024_WITH_DES_CBC_SHA_I:
		case TLS_DHE_DSS_EXPORT_WITH_DES40_CBC_SHA_I:
			/* DHE_DSS_EXPORT */
			conn->conn_state.kes.key_limit = 512;
			/* fall-through to DHE_DSS */

		case TLS_DHE_DSS_WITH_RC4_128_SHA_I:
		case TLS_DHE_DSS_WITH_DES_CBC_SHA_I:
		case TLS_DHE_DSS_WITH_3DES_EDE_CBC_SHA_I:
		case TLS_DHE_DSS_WITH_AES_128_CBC_SHA_I:
		case TLS_DHE_DSS_WITH_AES_256_CBC_SHA_I:
			/* DHE_DSS */
			conn->conn_state.kes.kea = kssl_kea_dhe;
			conn->conn_state.kes.sa = kssl_sa_dss;
			break;

		case TLS_DHE_RSA_EXPORT_WITH_DES40_CBC_SHA_I:
			/* DHE_RSA_EXPORT */
			conn->conn_state.kes.key_limit = 512;
			/* fall-through to DHE_RSA */

		case TLS_DHE_RSA_WITH_DES_CBC_SHA_I:
		case TLS_DHE_RSA_WITH_3DES_EDE_CBC_SHA_I:
		case TLS_DHE_RSA_WITH_AES_128_CBC_SHA_I:
		case TLS_DHE_RSA_WITH_AES_256_CBC_SHA_I:
			/* DHE_RSA */
			conn->conn_state.kes.kea = kssl_kea_dhe;
			conn->conn_state.kes.sa = kssl_sa_rsa;
			break;

		case TLS_DH_anon_EXPORT_WITH_RC4_40_MD5_I:
		case TLS_DH_anon_EXPORT_WITH_DES40_CBC_SHA_I:
			/* DH_anon_EXPORT */
			conn->conn_state.kes.key_limit = 512;
			/* fall-through  to DH_anon_EXPORT */

		case TLS_DH_anon_WITH_RC4_128_MD5_I:
		case TLS_DH_anon_WITH_DES_CBC_SHA_I:
		case TLS_DH_anon_WITH_3DES_EDE_CBC_SHA_I:
		case TLS_DH_anon_WITH_AES_128_CBC_SHA_I:
		case TLS_DH_anon_WITH_AES_256_CBC_SHA_I:
			/* DH_anon */
			conn->conn_state.kes.kea = kssl_kea_dha;
			conn->conn_state.kes.sa = kssl_sa_null;
			break;

		case TLS_DH_DSS_EXPORT_WITH_DES40_CBC_SHA_I:
			/* DH_DSS_EXPORT */
			conn->conn_state.kes.key_limit = 512;
			/* fall throuth to DH_DSS */

		case TLS_DH_DSS_WITH_3DES_EDE_CBC_SHA_I:
		case TLS_DH_DSS_WITH_DES_CBC_SHA_I:
		case TLS_DH_DSS_WITH_AES_128_CBC_SHA_I:
		case TLS_DH_DSS_WITH_AES_256_CBC_SHA_I:
			/* DH_DSS  */
			conn->conn_state.kes.kea = kssl_kea_dh;
			conn->conn_state.kes.sa = kssl_sa_dss;
			break;

		case TLS_DH_RSA_EXPORT_WITH_DES40_CBC_SHA_I:
			/* DH_RSA_EXPORT */
			conn->conn_state.kes.key_limit = 512;
			/* fall throuth to DH_RSA */

		case TLS_DH_RSA_WITH_DES_CBC_SHA_I:
		case TLS_DH_RSA_WITH_3DES_EDE_CBC_SHA_I:
		case TLS_DH_RSA_WITH_AES_128_CBC_SHA_I:
		case TLS_DH_RSA_WITH_AES_256_CBC_SHA_I:
			/* DH_RSA */
			conn->conn_state.kes.kea = kssl_kea_dh;
			conn->conn_state.kes.sa = kssl_sa_rsa;
			break;

		case TLS_RSA_EXPORT_WITH_RC2_CBC_40_MD5_I:
		case TLS_RSA_EXPORT_WITH_RC4_40_MD5_I:
		case TLS_RSA_EXPORT_WITH_DES40_CBC_SHA_I:
		case TLS_RSA_EXPORT1024_WITH_DES_CBC_SHA_I:
			/* DH_RSA_EXPORT */
			conn->conn_state.kes.key_limit = 512;
			/* fall throuth to DH_RSA */

		case TLS_RSA_WITH_NULL_MD5_I:
		case TLS_RSA_WITH_RC4_128_MD5_I:
		case TLS_RSA_WITH_NULL_SHA_I:
		case TLS_RSA_WITH_IDEA_CBC_SHA_I:
		case TLS_RSA_WITH_RC4_128_SHA_I:
		case TLS_RSA_WITH_DES_CBC_SHA_I:
		case TLS_RSA_WITH_3DES_EDE_CBC_SHA_I:
		case TLS_RSA_WITH_AES_128_CBC_SHA_I:
		case TLS_RSA_WITH_AES_256_CBC_SHA_I:
			/* DH_RSA */
			conn->conn_state.kes.kea = kssl_kea_rsa;
			conn->conn_state.kes.sa = kssl_sa_rsa;
			break;

		/* case CIPHER_SUITE_END_MARKER_I: */
		default:
			KSSL_DEBUG(3, "__kssl_conn_set_cipher_suite_pending: "
					"unknown cipher suite 4: %04x\n",
					cipher_suite_toi(cs));
			return -EINVAL;
	}

	cipher_suite_cpy(&(conn->conn_state.cs), cs);

	return 0;
}

int kssl_conn_set_cipher_pending(kssl_conn_t *conn, random_t *client_random,
		cipher_suite_t *cs, compression_method_t cm) 
{
	KSSL_DEBUG(12, "kssl_conn_set_cipher_pending: enter\n");

	if (!conn->sec_param_in_pend) {
		if (!conn->sec_param_in_act) {
			KSSL_DEBUG(6, "kssl_conn_set_cipher_pending: "
					"no sec_param_act\n");
			return -EINVAL;
		}
		conn->sec_param_in_pend = (security_parameters_t *)kssl_kmalloc(
				sizeof(security_parameters_t), GFP_KERNEL);
       		if (!conn->sec_param_in_pend) {
			KSSL_DEBUG(6, "kssl_conn_set_cipher_pending: "
					"kssl_kmalloc\n");
			return -ENOMEM;
		}
		security_parameters_init(conn->sec_param_in_pend, 
				conn->sec_param_in_act->entity);
		conn->sec_param_out_pend = conn->sec_param_in_pend;
	}


	if (__kssl_conn_set_cipher_suite_pending(conn, cs) < 0) {
		KSSL_DEBUG(6, "kssl_conn_set_cipher_pending: "
				"__kssl_conn_set_cipher_suite_pending\n");
		return -EINVAL;
	}

	memcpy(conn->sec_param_in_pend->client_random, 
			client_random->random_bytes, RANDOM_NBYTES);
	get_random_bytes(conn->sec_param_in_pend->server_random,
			RANDOM_NLEN);

	conn->sec_param_in_pend->compression_algorithm = cm;

	return 0;
}


/* The following need to be set
 *      conn->sec_param_in_pend->cipher_type,
 *      conn->sec_param_in_pend->is_exportable,
 *      conn->sec_param_in_pend->block_size,
 *      conn->sec_param_in_pend->key_material_length,
 *      conn->sec_param_in_pend->key_size and
 *      conn->sec_param_in_pend->hash_size need to be set.
 * I.e. kssl_conn_set_cipher_pending() should have been called */

static int kssl_conn_keying_material_init_from_master_secret_tls(
		kssl_conn_t *conn, u8 *master_secret)
{
	u32 key_material_len;
	u8 *key_material = NULL;
	u8 *p;
	int status = 0; 

	KSSL_DEBUG(12, "kssl_conn_keying_material_init_from_master_secret_tls: "
			"enter\n");

	key_material_len = (conn->sec_param_in_pend->key_material_length * 2) +
		(conn->sec_param_in_pend->hash_size *2);
	if (conn->sec_param_in_pend->cipher_type == ct_block &&
			!conn->sec_param_in_pend->is_exportable) {
		key_material_len += conn->sec_param_in_pend->block_size * 2;
	}

	memcpy(conn->sec_param_in_pend->master_secret, master_secret,
			MASTER_SECRET_LEN);

	key_material = kssl_prf_tls(conn->sec_param_in_pend->master_secret,
			MASTER_SECRET_LEN,
			KEY_EXPANSION_LABEL, strlen(KEY_EXPANSION_LABEL),
			conn->sec_param_in_pend->server_random, RANDOM_NBYTES,
			conn->sec_param_in_pend->client_random, RANDOM_NBYTES,
			key_material_len);
	if (!key_material) {
		KSSL_DEBUG(6, "conn_keying_material_init_tls: "
				"key material generation failed\n");
		goto error;
	}
	p = key_material;

	conn->conn_state.client_mac_secret = (opaque_t *)kssl_kmalloc(
			conn->sec_param_in_pend->hash_size, GFP_KERNEL);
	if (!conn->conn_state.client_mac_secret) {
		KSSL_DEBUG(6, "conn_keying_material_init_tls: "
				"client secret allocation failed\n");
		goto error;
	}
	memcpy(conn->conn_state.client_mac_secret, p,
			conn->sec_param_in_pend->hash_size);
	p += conn->sec_param_in_pend->hash_size;

	conn->conn_state.server_mac_secret = (opaque_t *)kssl_kmalloc(
			conn->sec_param_in_pend->hash_size, GFP_KERNEL);
	if (!conn->conn_state.server_mac_secret) {
		KSSL_DEBUG(6, "conn_keying_material_init_tls: "
				"server secret allocation failed\n");
		goto error;
	}
	memcpy(conn->conn_state.server_mac_secret, p,
			conn->sec_param_in_pend->hash_size);
	p += conn->sec_param_in_pend->hash_size;

	/* If there is no encryption algorithm then we are done */
	if (conn->sec_param_in_pend->bulk_cipher_algorithm == bca_null)
		goto leave;

	/* Non-Exportable Cipher Suites have their keys and initialisation 
	 * vector set from the key_material */
	if (!conn->sec_param_in_pend->is_exportable) {
		conn->conn_state.client_key = (opaque_t *)kssl_kmalloc(
				conn->sec_param_in_pend->key_size,
				GFP_KERNEL);
		if (!conn->conn_state.client_key) {
			KSSL_DEBUG(6, "conn_keying_material_init_tls: "
					"client key (exp) allocation failed\n");
			goto error;
		}
		memcpy(conn->conn_state.client_key, p,
				conn->sec_param_in_pend->key_size);
		p += conn->sec_param_in_pend->key_material_length;
	
		conn->conn_state.server_key = (opaque_t *)kssl_kmalloc(
				conn->sec_param_in_pend->key_material_length, 
				GFP_KERNEL);
		if (!conn->conn_state.server_key) {
			KSSL_DEBUG(6, "conn_keying_material_init_tls: "
					"server key (exp) allocation failed\n");
			goto error;
		}
		memcpy(conn->conn_state.server_key, p,
				conn->sec_param_in_pend->key_size);
		p += conn->sec_param_in_pend->key_material_length;

		/* Stream Cyphers don't have initialisation vectors
	 	 * so there is nothing more to do */
		if (conn->sec_param_in_pend->cipher_type != ct_block) {
			goto leave;
		}

		conn->conn_state.client_iv = (opaque_t *)kssl_kmalloc(
				conn->sec_param_in_pend->block_size, 
				GFP_KERNEL);
		if (!conn->conn_state.client_iv) {
			KSSL_DEBUG(6, "conn_keying_material_init_tls: "
					"client iv (exp) allocation failed\n");
			goto error;
		}
		memcpy(conn->conn_state.client_iv, p, 
				conn->sec_param_in_pend->block_size);
		p += conn->sec_param_in_pend->block_size;
	
		conn->conn_state.server_iv = (opaque_t *)kssl_kmalloc(
				conn->sec_param_in_pend->block_size, 
				GFP_KERNEL);
		if (!conn->conn_state.server_iv) {
			KSSL_DEBUG(6, "conn_keying_material_init_tls: "
					"server iv (exp) allocation failed\n");
			goto error;
		}
		memcpy(conn->conn_state.server_iv, p, 
				conn->sec_param_in_pend->block_size);
		p += conn->sec_param_in_pend->block_size;

		goto leave;
	}

	/* Exportable Cipher Suites need extra work */

	conn->conn_state.client_key = kssl_prf_tls(p,
			conn->sec_param_in_pend->key_material_length,
			CLIENT_WRITE_LABEL, strlen(CLIENT_WRITE_LABEL),
			conn->sec_param_in_pend->client_random, RANDOM_NBYTES,
			conn->sec_param_in_pend->server_random, RANDOM_NBYTES,
			conn->sec_param_in_pend->key_size);
	if (!conn->conn_state.client_key) {
		KSSL_DEBUG(6, "conn_keying_material_init_tls: "
				"client key generation failed\n");
		goto error;
	}
	p += conn->sec_param_in_pend->key_material_length;

	conn->conn_state.server_key = kssl_prf_tls(p,
			conn->sec_param_in_pend->key_material_length,
			SERVER_WRITE_LABEL, strlen(SERVER_WRITE_LABEL),
			conn->sec_param_in_pend->client_random, RANDOM_NBYTES,
			conn->sec_param_in_pend->server_random, RANDOM_NBYTES,
			conn->sec_param_in_pend->key_size);
	if (!conn->conn_state.server_key) {
		KSSL_DEBUG(6, "conn_keying_material_init_tls: "
				"server key generation failed\n");
		goto error;
	}
	p += conn->sec_param_in_pend->key_material_length;

	memset(key_material, 0, key_material_len);
	kssl_kfree(key_material);
	key_material_len = conn->sec_param_in_pend->block_size * 2;

	/* Stream Cyphers don't have initialisation vectors;
	 * so there is nothing more to do */
	if (conn->sec_param_in_pend->cipher_type != ct_block) {
		goto leave;
	}

	key_material = kssl_prf_tls("", 0,
			IV_BLOCK_LABEL, strlen(IV_BLOCK_LABEL),
			conn->sec_param_in_pend->client_random, RANDOM_NBYTES,
			conn->sec_param_in_pend->server_random, RANDOM_NBYTES,
			key_material_len);
	if (!key_material) {
		KSSL_DEBUG(6, "conn_keying_material_init_tls: "
				"key material generation 2 failed\n");
		goto error;
	}

	p = key_material;
	conn->conn_state.client_iv = (opaque_t *)kssl_kmalloc(
			conn->sec_param_in_pend->block_size, GFP_KERNEL);
	if (!conn->conn_state.client_iv) {
		KSSL_DEBUG(6, "conn_keying_material_init_tls: "
				"client iv allocation failed\n");
		goto error;
	}
	memcpy(conn->conn_state.client_iv, p,
			conn->sec_param_in_pend->block_size);
	p += conn->sec_param_in_pend->block_size;

	conn->conn_state.server_iv = (opaque_t *)kssl_kmalloc(
			conn->sec_param_in_pend->block_size, GFP_KERNEL);
	if (!conn->conn_state.server_iv) {
		KSSL_DEBUG(6, "conn_keying_material_init_tls: "
				"server iv allocation failed\n");
		goto error;
	}
	memcpy(conn->conn_state.server_iv, p,
			conn->sec_param_in_pend->block_size);
	p += conn->sec_param_in_pend->block_size;

	goto leave;

error:
	status = -ENOMEM;
	__kssl_clear_keys_sec_param(conn->sec_param_in_pend);
	__kssl_conn_clear_keys(conn);

leave:
	
	if (key_material) {
		memset(key_material, 0, key_material_len);
		kssl_kfree(key_material);
	} 

	return status;
}



/* N.B The folowing need to be set
 *      conn->sec_param_in_pend->cipher_type,
 *      conn->sec_param_in_pend->is_exportable,
 *      conn->sec_param_in_pend->block_size,
 *      conn->sec_param_in_pend->key_material_length,
 *      conn->sec_param_in_pend->key_size and
 *      conn->sec_param_in_pend->hash_size need to be set.
 * I.e. kssl_conn_set_cipher_pending() should have been called */

static int kssl_conn_keying_material_init_tls(kssl_conn_t *conn, u8 *secret, 
		u32 secret_len)
{
	u8 *master_secret_tmp = NULL;
	int status = 0;

	KSSL_DEBUG(12, "kssl_conn_keying_material_init_tls: enter\n");

	master_secret_tmp = kssl_prf_tls(secret, secret_len,
			MASTER_SECRET_LABEL, strlen(MASTER_SECRET_LABEL),
			conn->sec_param_in_pend->client_random, RANDOM_NBYTES,
			conn->sec_param_in_pend->server_random, RANDOM_NBYTES,
			MASTER_SECRET_LEN);
	if (!master_secret_tmp) {
		KSSL_DEBUG(6, "conn_keying_material_init_tls: "
				"master secret generation failed\n");
		return -ENOMEM;
	}

	status = kssl_conn_keying_material_init_from_master_secret(conn, 
			master_secret_tmp);
	memset(master_secret_tmp, 0, MASTER_SECRET_LEN);
	kssl_kfree(master_secret_tmp);
	if (status < 0) {
		KSSL_DEBUG(6, "conn_keying_material_init_tls: "
		"kssl_conn_keying_material_init_from_master_secret\n");
		return status;
	}

	return status;
}


/* N.B conn->sec_param_in_pend->client_random and
 *     conn->sec_param_in_pend->server_random neet to be set.
 *
 * Also conn->sec_param_in_pend->cipher_type,
 *      conn->sec_param_in_pend->is_exportable,
 *      conn->sec_param_in_pend->block_size,
 *      conn->sec_param_in_pend->key_material_length,
 *      conn->sec_param_in_pend->key_size and
 *      conn->sec_param_in_pend->hash_size need to be set.
 * I.e. kssl_conn_set_cipher_pending() should have been called */

static int kssl_conn_keying_material_init_from_master_secret_ssl3(
		kssl_conn_t *conn, u8 *master_secret)
{
	u32 key_material_len;
	u32 tmp_len;
	u8 *key_material = NULL;
	u8 *p;
	int status = 0;

	KSSL_DEBUG(12, "kssl_conn_keying_material_init_from_master_secret_ssl3:"
			" enter\n");

	key_material_len = (conn->sec_param_in_pend->key_material_length * 2) +
		(conn->sec_param_in_pend->hash_size *2);
	if (conn->sec_param_in_pend->cipher_type == ct_block &&
			!conn->sec_param_in_pend->is_exportable) {
		key_material_len += conn->sec_param_in_pend->block_size * 2;
	}

	memcpy(conn->sec_param_in_pend->master_secret, master_secret,
			MASTER_SECRET_LEN);

	tmp_len = key_material_len;
	key_material = kssl_prf_ssl3(conn->sec_param_in_pend->master_secret,
			MASTER_SECRET_LEN,
			conn->sec_param_in_pend->server_random, RANDOM_NBYTES,
			conn->sec_param_in_pend->client_random, RANDOM_NBYTES,
			&tmp_len);
	if (!key_material) {
		KSSL_DEBUG(6, "conn_keying_material_init_ssl3: "
				"error generating key material\n");
		goto error;
	}
	p = key_material;

	conn->conn_state.client_mac_secret = (opaque_t *)kssl_kmalloc(
			conn->sec_param_in_pend->hash_size, GFP_KERNEL);
	if (!conn->conn_state.client_mac_secret) {
		KSSL_DEBUG(6, "conn_keying_material_init_ssl3: "
				"error allocating client mac secret\n");
		goto error;
	}
	memcpy(conn->conn_state.client_mac_secret, p,
			conn->sec_param_in_pend->hash_size);
	p += conn->sec_param_in_pend->hash_size;

	conn->conn_state.server_mac_secret = (opaque_t *)kssl_kmalloc(
			conn->sec_param_in_pend->hash_size, GFP_KERNEL);
	if (!conn->conn_state.server_mac_secret) {
		KSSL_DEBUG(6, "conn_keying_material_init_ssl3: "
				"error allocating server mac secret\n");
		goto error;
	}
	memcpy(conn->conn_state.server_mac_secret, p,
			conn->sec_param_in_pend->hash_size);
	p += conn->sec_param_in_pend->hash_size;

	/* If there is no encryption algorithm then we are done */
	if (conn->sec_param_in_pend->bulk_cipher_algorithm == bca_null)
		goto leave;

	/* Non-Exportable Cipher Suites have their keys and initialisation 
	 * vector set from the key_material */
	if (!conn->sec_param_in_pend->is_exportable) {
		conn->conn_state.client_key = (opaque_t *)kssl_kmalloc(
				conn->sec_param_in_pend->key_size,
				GFP_KERNEL);
		if (!conn->conn_state.client_key) {
			KSSL_DEBUG(6, "conn_keying_material_init_ssl3: "
					"error allocating client key\n");
			goto error;
		}
		memcpy(conn->conn_state.client_key, p,
				conn->sec_param_in_pend->key_size);
		p += conn->sec_param_in_pend->key_size;
	
		conn->conn_state.server_key = (opaque_t *)kssl_kmalloc(
				conn->sec_param_in_pend->key_size, 
				GFP_KERNEL);
		if (!conn->conn_state.server_key) {
			KSSL_DEBUG(6, "conn_keying_material_init_ssl3: "
					"error allocating server key\n");
			goto error;
		}
		memcpy(conn->conn_state.server_key, p,
				conn->sec_param_in_pend->key_size);
		p += conn->sec_param_in_pend->key_size;

		/* Stream Cyphers don't have initialisation vectors
	 	 * so there is nothing more to do */
		if (conn->sec_param_in_pend->cipher_type != ct_block) {
			goto leave;
		}

		conn->conn_state.client_iv = (opaque_t *)kssl_kmalloc(
				conn->sec_param_in_pend->block_size, 
				GFP_KERNEL);
		if (!conn->conn_state.client_iv) {
			KSSL_DEBUG(6, "conn_keying_material_init_ssl3: "
					"error allocating client iv\n");
			goto error;
		}
		memcpy(conn->conn_state.client_iv, p,
				conn->sec_param_in_pend->block_size);
		p += conn->sec_param_in_pend->block_size;
	
		conn->conn_state.server_iv = (opaque_t *)kssl_kmalloc(
				conn->sec_param_in_pend->block_size, 
				GFP_KERNEL);
		if (!conn->conn_state.server_iv) {
			KSSL_DEBUG(6, "conn_keying_material_init_ssl3: "
					"error allocating server iv\n");
			goto error;
		}
		memcpy(conn->conn_state.server_iv, p,
				conn->sec_param_in_pend->block_size);
		p += conn->sec_param_in_pend->block_size;

		goto leave;
	}

	/* Exportable Cipher Suites need extra work */

	tmp_len = conn->sec_param_in_pend->key_size;
	conn->conn_state.client_key = kssl_prf_ssl3_export_md5(p,
			conn->sec_param_in_pend->key_material_length,
			conn->sec_param_in_pend->client_random, RANDOM_NBYTES,
			conn->sec_param_in_pend->server_random, RANDOM_NBYTES,
			&tmp_len);
	if (!conn->conn_state.client_key) {
		KSSL_DEBUG(6, "conn_keying_material_init_ssl3: "
				"error generating client key\n");
		goto error;
	}
	p += conn->sec_param_in_pend->key_material_length;

	tmp_len = conn->sec_param_in_pend->key_size;
	conn->conn_state.server_key = kssl_prf_ssl3_export_md5(p,
			conn->sec_param_in_pend->key_material_length,
			conn->sec_param_in_pend->server_random, RANDOM_NBYTES,
			conn->sec_param_in_pend->client_random, RANDOM_NBYTES,
			&tmp_len);
	if (!conn->conn_state.server_key) {
		KSSL_DEBUG(6, "conn_keying_material_init_ssl3: "
				"error generating server key\n");
		goto error;
	}
	p += conn->sec_param_in_pend->key_material_length;


	/* Stream Cyphers don't have initialisation vectors
	 * so there is nothing more to do */
	if (conn->sec_param_in_pend->cipher_type != ct_block) {
		goto leave;
	}

	tmp_len = conn->sec_param_in_pend->block_size;
	conn->conn_state.client_iv = kssl_prf_ssl3_export_md5(NULL, 0,
			conn->sec_param_in_pend->client_random, RANDOM_NBYTES,
			conn->sec_param_in_pend->server_random, RANDOM_NBYTES,
			&tmp_len);
	if (!conn->conn_state.client_iv) {
		KSSL_DEBUG(6, "conn_keying_material_init_ssl3: "
				"error generating client iv\n");
		goto error;
	}

	tmp_len = conn->sec_param_in_pend->block_size;
	conn->conn_state.server_iv = kssl_prf_ssl3_export_md5("", 0,
			conn->sec_param_in_pend->server_random, RANDOM_NBYTES,
			conn->sec_param_in_pend->client_random, RANDOM_NBYTES,
			&tmp_len);
	if (!conn->conn_state.server_iv) {
		KSSL_DEBUG(6, "conn_keying_material_init_ssl3: "
				"error generating server iv\n");
		goto error;
	}

	goto leave;
error:
	status = -ENOMEM;
	__kssl_clear_keys_sec_param(conn->sec_param_in_pend);
	__kssl_conn_clear_keys(conn);

leave:
#if 0 /* DEBUG */
	if (!status) {
		asym_print_char(KERN_DEBUG "client_mac_secret",  
				conn->conn_state.client_mac_secret, 
				conn->sec_param_in_pend->hash_size);
		asym_print_char(KERN_DEBUG "server_mac_secret",  
				conn->conn_state.server_mac_secret, 
				conn->sec_param_in_pend->hash_size);
		asym_print_char(KERN_DEBUG "client_key",  
				conn->conn_state.client_key, 
				conn->sec_param_in_pend->key_size);
		asym_print_char(KERN_DEBUG "server_key",  
				conn->conn_state.server_key, 
				conn->sec_param_in_pend->key_size);
		if (conn->sec_param_in_pend->cipher_type == ct_block) {
			asym_print_char(KERN_DEBUG "client_iv",  
					conn->conn_state.client_iv,
					conn->sec_param_in_pend->block_size);
			asym_print_char(KERN_DEBUG "server_iv",  
					conn->conn_state.server_iv,
					conn->sec_param_in_pend->block_size);
		}
	}
#endif

	if (key_material) {
		memset(key_material, 0, key_material_len);
		kssl_kfree(key_material);
	} 

	return status;
}


/* N.B conn->sec_param_in_pend->client_random and
 *     conn->sec_param_in_pend->server_random neet to be set.
 *
 * Also conn->sec_param_in_pend->cipher_type,
 *      conn->sec_param_in_pend->is_exportable,
 *      conn->sec_param_in_pend->block_size,
 *      conn->sec_param_in_pend->key_material_length,
 *      conn->sec_param_in_pend->key_size and
 *      conn->sec_param_in_pend->hash_size need to be set.
 * I.e. kssl_conn_set_cipher_pending() should have been called */

static int kssl_conn_keying_material_init_ssl3(kssl_conn_t *conn, u8 *secret, 
		u32 secret_len)
{
	u32 tmp_len;
	u8 *master_secret_tmp = NULL;
	int status = 0;

	KSSL_DEBUG(12, "conn_keying_material_init_ssl3 enter\n");

	tmp_len = MASTER_SECRET_LEN;
	master_secret_tmp = kssl_prf_ssl3(secret, secret_len,
			conn->sec_param_in_pend->client_random, RANDOM_NBYTES,
			conn->sec_param_in_pend->server_random, RANDOM_NBYTES,
			&tmp_len);
	if (!master_secret_tmp) {
		KSSL_DEBUG(6, "conn_keying_material_init_ssl3: "
				"error generating master secret\n");
		return -ENOMEM;
	}

	status = kssl_conn_keying_material_init_from_master_secret(conn, 
			master_secret_tmp);
	memset(master_secret_tmp, 0, MASTER_SECRET_LEN);
	kssl_kfree(master_secret_tmp);
	if (status < 0) {
		KSSL_DEBUG(6, "kssl_conn_keying_material_init_ssl3: "
			"kssl_conn_keying_material_init_from_master_secret\n");
		return status;
	}

	return status;
}


/* The following need to be set
 *      conn->sec_param_in_pend->cipher_type,
 *      conn->sec_param_in_pend->is_exportable,
 *      conn->sec_param_in_pend->block_size,
 *      conn->sec_param_in_pend->key_material_length,
 *      conn->sec_param_in_pend->key_size and
 *      conn->sec_param_in_pend->hash_size need to be set.
 * I.e. kssl_conn_set_cipher_pending() should have been called */

int kssl_conn_keying_material_init_from_master_secret(kssl_conn_t *conn, 
		u8 *master_secret)
{
	int status;

	KSSL_DEBUG(12, "kssl_conn_keying_material_init_from_master_secret: "
			"enter\n");

	/* Major version 3 and minor version of 0 or 1 
	 * should already have been checked */
	if (conn->conn_state.version.minor)
		status = kssl_conn_keying_material_init_from_master_secret_tls(
				conn, master_secret);
	else 
		status = kssl_conn_keying_material_init_from_master_secret_ssl3(
				conn, master_secret);
	if (status < 0)
		return status;

	/* We don't actually care if this fails */
	kssl_session_set_master_secret(conn->sess_state.id, master_secret);

	return 0;
}


/* N.B conn->sec_param_in_pend->client_random and
 *     conn->sec_param_in_pend->server_random neet to be set.
 *
 * Also conn->sec_param_in_pend->cipher_type,
 *      conn->sec_param_in_pend->is_exportable,
 *      conn->sec_param_in_pend->block_size,
 *      conn->sec_param_in_pend->key_material_length,
 *      conn->sec_param_in_pend->key_size and
 *      conn->sec_param_in_pend->hash_size need to be set.
 * I.e. kssl_conn_set_cipher_pending() should have been called */

int kssl_conn_keying_material_init(kssl_conn_t *conn, u8 *secret, 
		u32 secret_len)
{
	int status;

	KSSL_DEBUG(12, "kssl_conn_keying_material_init: enter\n");

	/* Major version 3 and minor version of 0 or 1 
	 * should already have been checked */
	if (conn->conn_state.version.minor)
		status = kssl_conn_keying_material_init_tls(conn, secret, 
					secret_len);
	else 
		status = kssl_conn_keying_material_init_ssl3(conn, secret, 
				secret_len);
	if (status < 0)
		return status;

	return 0;
}

