/**********************************************************************
 * service.c                                                August 2005
 *
 * L7VSD: Linux Virtual Server for Layer7 Load Balancing
 * Copyright (C) 2005  NTT COMWARE Corporation.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 **********************************************************************/

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <unistd.h>
#include <stdlib.h>
#include <glib.h>
#include "vanessa_logger.h"
#include "l7vs.h"

static GList *l7vs_service_list;

static struct l7vs_service_arg *l7vs_service_get_service_arg(
                struct l7vs_service *srv);
static void l7vs_service_put_service_arg(struct l7vs_service_arg *sa);

static struct l7vs_service_arg *
l7vs_service_get_service_arg(struct l7vs_service *srv)
{
        struct l7vs_service_arg *sa;

        sa = srv->pm->service_arg(srv);
        if (sa == NULL) {
                VANESSA_LOGGER_ERR("Could not allocate memory");
                return NULL;
        }

        sa->addr = srv->lsock->addr;
        sa->proto = srv->lsock->proto;
        sa->persist = 0 /* XXX not yet */;
        sa->backlog = 5 /* XXX not yet */;
        strcpy(sa->protomod, srv->pm->modname);
        strcpy(sa->schedmod, srv->scheduler->modname);

        return sa;
}

static void
l7vs_service_put_service_arg(struct l7vs_service_arg *sa)
{
        free(sa);
}



struct l7vs_dest *
l7vs_service_lookup_dest(struct l7vs_service *srv,
                         struct sockaddr_in *sin)
{
        GList *l;
        struct l7vs_dest *d, *found;

        found = NULL;
        for (l = g_list_first(srv->dest_list); l != NULL; l = g_list_next(l)) {
                d = (struct l7vs_dest *)l->data;
                if ((d->addr.sin_addr.s_addr == sin->sin_addr.s_addr) &&
                    (d->addr.sin_port == sin->sin_port)) {
                        found = d;
                        break;
                }
        }

        return found;
}

int
l7vs_service_list_service_arg(struct l7vs_service_arg **sas, int *num)
{
        GList *l;
        struct l7vs_service *srv;
        struct l7vs_service_arg *r, *s;
        struct l7vs_service_arg **sa;
        int i, n, len;

        n = g_list_length(l7vs_service_list);
        if (n == 0) {
                *sas = NULL;
                *num = 0;
                return 0;
        }

        sa = (struct l7vs_service_arg **)calloc(n, sizeof(*sa));
        len = 0;
        l = g_list_first(l7vs_service_list);
        for (i = 0; i < n; i++) {
                srv = (struct l7vs_service *)l->data;
                sa[i] = l7vs_service_get_service_arg(srv);
                if (sa[i] == NULL) {
                        for (i-- ; i >= 0; i--) {
                                l7vs_service_put_service_arg(sa[i]);
                        }
                        free(sa);
                        return -1;
                }
                len += sa[i]->len;
                l = g_list_next(l);
        }

        r = (struct l7vs_service_arg *)malloc(len);
        if (r == NULL) {
                VANESSA_LOGGER_ERR("Could not allocate memory");
                return -1;
        }

        s = r;
        for (i = 0; i < n; i++) {
                memcpy(s, sa[i], sa[i]->len);
                s = (struct l7vs_service_arg *)((uint8_t *)s + sa[i]->len);
                l7vs_service_put_service_arg(sa[i]);
        }
        free(sa);
        *sas = r;
        *num = n;

        return len;
}

int
l7vs_service_list_dest_arg(struct l7vs_service *srv,
                           struct l7vs_dest_arg **das)
{
        GList *l;
        struct l7vs_dest *d;
        struct l7vs_dest_arg *darg;
        int i, num;

        num = g_list_length(srv->dest_list);
        if (num == 0) {
                *das = NULL;
                return num;
        }

        darg = (struct l7vs_dest_arg *)malloc(num * sizeof(*darg));
        if (darg == NULL) {
                VANESSA_LOGGER_ERR("Could not allocate memory");
                return -1;
        }

        i = 0;
        for (l = g_list_first(srv->dest_list); l != NULL; l = g_list_next(l)) {
                d = (struct l7vs_dest *)l->data;
                l7vs_dest_to_arg(d, &darg[i]);
                i++;
        }

        *das = darg;
        return num;
}

struct l7vs_service *
l7vs_service_create(struct l7vs_service_arg *arg, int *err)
{
        struct l7vs_protomod *pmod;
        struct l7vs_scheduler *sched;
        struct l7vs_service *srv, *sref;
        struct l7vs_lsock *lsock;
        GList *l;

        *err = 0;
        lsock = l7vs_lsock_get(&arg->addr, arg->proto, arg->backlog);
        if (lsock == NULL) {
                VANESSA_LOGGER_ERR("Could not create listen socket");
                *err = L7VS_CONFIG_ERR_NOSOCK;
                return NULL;
        }

        pmod = l7vs_protomod_get(arg->protomod);
        if (pmod == NULL) {
                l7vs_lsock_put(lsock);
                *err = L7VS_CONFIG_ERR_NOMEM;
                return NULL;
        }

        sched = l7vs_sched_get(arg->schedmod);
        if (sched == NULL) {
                l7vs_protomod_put(pmod);
                l7vs_lsock_put(lsock);
                *err = L7VS_CONFIG_ERR_NOMEM;
                return NULL;
        }

        srv = pmod->create(arg);
        if (srv == NULL) {
                l7vs_sched_put(sched);
                l7vs_protomod_put(pmod);
                l7vs_lsock_put(lsock);
                *err = L7VS_CONFIG_ERR_NOMEM;
                return NULL;
        }

        for (l = g_list_first(l7vs_service_list); l != NULL;
             l = g_list_next(l)) {
                sref = (struct l7vs_service *)l->data;
                if (lsock != sref->lsock) {
                        continue;
                }

                if (pmod != sref->pm) {
                        continue;
                }

                if (pmod->compare(srv, sref) != 0) {
                        continue;
                }

                VANESSA_LOGGER_ERR("Virtual service already exists");
                l7vs_sched_put(sched);
                l7vs_protomod_put(pmod);
                l7vs_lsock_put(lsock);
                pmod->destroy(srv);
                *err = L7VS_CONFIG_ERR_VS_EXISTS;
                return NULL;
        }

        srv->lsock = lsock;
        srv->pm = pmod;
        l7vs_sched_bind(sched, srv);

        l7vs_lsock_add_service(lsock, srv);
        l7vs_service_list = g_list_append(l7vs_service_list, srv);

        return srv;
}

void
l7vs_service_destroy(struct l7vs_service *srv)
{
        struct l7vs_scheduler *sched;
        GList *l;
        struct l7vs_conn *conn;

        /* unbind associated conns from this service. */
        for (l = srv->conn_list; l != NULL; l = g_list_next(l)) {
                conn = (struct l7vs_conn *)l->data;
                conn->srv = NULL;
        }
        g_list_free(srv->conn_list);

        sched = srv->scheduler;
        l7vs_service_list = g_list_remove(l7vs_service_list, srv);
        l7vs_sched_unbind(sched, srv);
        l7vs_sched_put(sched);
        l7vs_lsock_remove_service(srv->lsock, srv);
        l7vs_lsock_put(srv->lsock);
        srv->pm->destroy(srv);
}

struct l7vs_service *
l7vs_service_lookup(struct l7vs_service_arg *arg) //checks if the virtual service to be added already exists--Anshu
{
        GList *l;
        struct l7vs_lsock *lsock;
        struct l7vs_protomod *pmod;
        struct l7vs_service *s, *sref;

        lsock = l7vs_lsock_table_lookup(&arg->addr, arg->proto);
        if (lsock == NULL) {
                return NULL;
        }

        pmod = l7vs_protomod_lookup(arg->protomod);
        if (pmod == NULL) {
                return NULL;
        }

        s = pmod->create(arg);
        for (l = g_list_first(l7vs_service_list);
             l != NULL; l = g_list_next(l)) {
                sref = (struct l7vs_service *)l->data;
                if (lsock != sref->lsock) {
                        continue;
                }

                if (pmod != sref->pm) {
                        continue;
                }

                if (pmod->compare(s, sref) != 0) {
                        continue;
                }

                pmod->destroy(s);
                return sref;
        }
        if (s != NULL) {
                pmod->destroy(s);
        } 
        return NULL;
}

int
l7vs_service_add_dest(struct l7vs_service *srv,
                      struct l7vs_dest_arg *darg)
{
        struct l7vs_dest *d;

        d = l7vs_service_lookup_dest(srv, &darg->addr);
        if (d != NULL) {
                VANESSA_LOGGER_INFO("Cannot add duplicate real service");
                return -1;
        }

        d = l7vs_dest_create(&darg->addr, darg->weight);
        if (d == NULL) {
                VANESSA_LOGGER_ERR("Could not allocate memory");
                return -1;
	}
		else {
			VANESSA_LOGGER_INFO("ADD_RS: IFRM004: added real server");
		}
        
        srv->dest_list = g_list_append(srv->dest_list, d);

        return 0;
}

int
l7vs_service_remove_dest(struct l7vs_service *srv,
                         struct l7vs_dest_arg *darg)
{
        struct l7vs_dest *d;

        d = l7vs_service_lookup_dest(srv, &darg->addr);
        if (d == NULL) {
                VANESSA_LOGGER_ERR("No such real service");
                return -1;
        }
		else {
			VANESSA_LOGGER_INFO("DEL_RS: IFRM005: removed real server");
		}
        srv->dest_list = g_list_remove(srv->dest_list, d);
        l7vs_dest_destroy(d);

        return 0;
}

int
l7vs_service_schedule(struct l7vs_service *srv, 
                      struct l7vs_conn *conn)
{
        struct l7vs_dest *dest;

        dest = srv->scheduler->schedule(srv, conn);
        if (dest == NULL) {
                VANESSA_LOGGER_ERR("no real server defined");
                return -1;
        }

        return l7vs_conn_connect_rs(conn, dest);
}

int
l7vs_service_establish(struct l7vs_service *srv, 
                       struct l7vs_conn *conn)
{
        return 0;
}

void
l7vs_service_register_conn(struct l7vs_service *srv,
                           struct l7vs_conn *conn)
{
        srv->conn_list = g_list_append(srv->conn_list, conn);
}

void l7vs_service_remove_conn(struct l7vs_service *srv,
                              struct l7vs_conn *conn)
{
        srv->conn_list = g_list_remove(srv->conn_list, conn);
}

void
l7vs_service_flush_all(void)
{
        GList *l, *m;
        struct l7vs_service *srv;
        struct l7vs_dest *d;

        while ((l = g_list_first(l7vs_service_list)) != NULL) {
                srv = (struct l7vs_service *)l->data;
                l7vs_service_list = g_list_remove(l7vs_service_list, srv);

                while ((m = g_list_first(srv->dest_list)) != NULL) {
                        d = (struct l7vs_dest *)m->data;
                        srv->dest_list = g_list_remove(srv->dest_list, d);
                        l7vs_dest_destroy(d);
                }

                l7vs_service_destroy(srv);
        }
}
