#include <stdlib.h>
#include <time.h>
#include <getopt.h>
#include "vanessa_logger.h"
#include "l7vs_service.h"
#include "l7vs_conn.h"
#include "l7vs_dest.h"
#include "l7vs_module.h"
#include "module_http.h"

#define SERVICE_ARG_MAXSIZE    (512)
#define SESSIONLESS_SERVICE_NUMBER (16)

struct l7vs_sessionless_service {
	uint32_t service_handle;
	int reschedule;
};

struct  l7vs_sessionless_service_arg {
	int reschedule;
};

static void  fini(void);
static int   create(void *sessionless_arg, uint32_t service_handle);
static void* create_sa(struct l7vs_service_arg *srv_arg);
static int   compare(uint32_t srv_handle1, uint32_t srv_handle2);
static int   match_cldata(struct l7vs_service *srv, struct l7vs_conn *conn,
                        char *buf, size_t *len, struct l7vs_dest **dest, int *tcps);
static int   analyze_rsdata(struct l7vs_service *srv, struct l7vs_conn *conn,
                        char *response, size_t *response_length);
static int   destroy(uint32_t srv_handle);
static void  destroy_sa(void **sessionless_arg);
static int   service_arg(struct l7vs_service_arg_multi *srv_arg_mt, uint32_t srv_handle);
static int   parse(void *sessionless_arg, int argc, char *argv[]);

static struct l7vs_sessionless_service *l7vs_protomod_sessionless_search_service(uint32_t service_handle);
static struct l7vs_sessionless_service *l7vs_protomod_sessionless_create_service();
static struct l7vs_sessionless_service *l7vs_protomod_sessionless_create_temp_service();

struct l7vs_sessionless_service *sessionless_service_list[SESSIONLESS_SERVICE_NUMBER];

static struct l7vs_protomod sessionless_protomod = {
	NULL,           /* handle */
	"sessionless",      /* modname */
	0,              /* refcnt */
	create,         /* create function */
	compare,        /* compare function */
	match_cldata,   /* match_cldata function */
	analyze_rsdata, /* analyze_rsdata function */
	destroy,        /* destroy function */
	fini,           /* fini function */
	create_sa,      /* create_sa function */
	service_arg,    /* service_arg function */
	parse,          /* parse function */
	destroy_sa,     /* destroy_sa function */
};

/*!
 * Protocol module initialize function. This function run when dlopen and dlsym at first time.
 * @param  handle dlopen's handle
 * @return l7vs_protomod struct
 */
struct l7vs_protomod *
init(void *handle)
{
	/* initialize sessionless service list */
	memset(sessionless_service_list, 0, sizeof(struct l7vs_sessionless_service *) * SESSIONLESS_SERVICE_NUMBER);
	/* set dlopen's handle */
	sessionless_protomod.handle = handle;

	return &sessionless_protomod;
}

/*!
 * Protocol module finalize function. free all sessionless service list just in case.
 * @param   void
 * @return  void
 */
static void
fini(void)
{
	/* sessionless service list counter */
	int service_number = 0;

	/* check all sessionless service list */
	for (service_number = 0; service_number < SESSIONLESS_SERVICE_NUMBER; ++service_number) {
		/* if pointer that does not point NULL exists ... */
		if (sessionless_service_list[service_number] != NULL) {
			/* free and points NULL */
			free(sessionless_service_list[service_number]);
			sessionless_service_list[service_number] = NULL;
		}
	}
}

/*!
 * Create sessionless service struct.
 * @param  sessionless_arg    sessionless service argument struct
 * @param  service_handle a unique service ID
 * @retval 0  successfully create sessionless service.
 * @retval -1 some errors occur.
 */
static int
create(void *sessionless_arg, uint32_t service_handle)
{
	struct l7vs_sessionless_service *sessionless_service;
	struct l7vs_sessionless_service_arg *sessionless_service_arg;

	/* check null */
	if (sessionless_arg == NULL) {
		VANESSA_LOGGER_ERR("sessionless_arg is null");
		return -1;
	}

	if (service_handle != TEMP_SERVICEHANDLE) {
		/* search empty sessionless service list and create sessionless service */
		sessionless_service = l7vs_protomod_sessionless_create_service();
	} else {
		/* create temporary sessionless service */
		sessionless_service = l7vs_protomod_sessionless_create_temp_service();
	}
	if (sessionless_service == NULL) {
		VANESSA_LOGGER_ERR("Could not make temporary sessionless service");
		return -1;
	}

	sessionless_service_arg = (struct l7vs_sessionless_service_arg *) sessionless_arg;

	/* set service handle */
	sessionless_service->service_handle = service_handle;
	/* set reschedule  */
	sessionless_service->reschedule = sessionless_service_arg->reschedule;

	return 0;
}

/*!
 * Create sessionless service argument struct.
 * @param  srv_arg service argument struct
 * @return void*   sessionless service argument struct
 */
static void *
create_sa(struct l7vs_service_arg *srv_arg)
{
	struct l7vs_sessionless_service_arg *sessionless_service_arg;

	if (srv_arg == NULL) {
		return NULL;
	}

	/* create sessionless service argument struct */
	sessionless_service_arg = (struct l7vs_sessionless_service_arg *) calloc(1, sizeof(struct l7vs_sessionless_service_arg));
	if (sessionless_service_arg == NULL) {
		VANESSA_LOGGER_ERR("Could not allocate memory");
		return (void *) sessionless_service_arg;
	}

	/* set sessionless service argument size and protomod name "sessionless" */
	srv_arg->len = sizeof(struct l7vs_sessionless_service_arg);
	strcpy(srv_arg->protomod, sessionless_protomod.modname);

	return (void *) sessionless_service_arg;
}

/*!
 * Compare two service.
 * @param  srv_handle1 one of a unique service ID
 * @param  srv_handle2 one of a unique service ID
 * @retval 0  they matched perfectly.
 * @retval -1 they are different.
 */
static int
compare(uint32_t srv_handle1, uint32_t srv_handle2)
{
	struct l7vs_sessionless_service *sessionless_srv1, *sessionless_srv2;

	/* search service that has such a service ID(1) */
	sessionless_srv1 = l7vs_protomod_sessionless_search_service(srv_handle1);
	if (sessionless_srv1 == NULL) {
		VANESSA_LOGGER_ERR("Could not find such service handle's sessionless service");
		return -1;
	}

	/* search service that has such a service ID(2) */
	sessionless_srv2 = l7vs_protomod_sessionless_search_service(srv_handle2);
	if (sessionless_srv2 == NULL) {
		VANESSA_LOGGER_ERR("Could not find such service handle's sessionless service");
		return -1;
	}

	return 0;
}

/*!
 * Do not check the request packet.
 * @param  srv service struct include service handle, protocol module and schedule module.
 * @param  conn connection data.
 * @param  request packet data from client
 * @param  len length of packet data
 * @param  dest destination (real server) list
 * @param  tcps TCP Splicer flag
 * @retval 0  successfully check packet data
 * @retval -1 some errors occur.
 */
static int
match_cldata(struct l7vs_service *srv, struct l7vs_conn *conn,
      char *request, size_t *len, struct l7vs_dest **dest, int *tcps)
{
	int ret;

	/* check null */
	if (srv == NULL) {
		VANESSA_LOGGER_ERR("srv is null");
		return -1;
	}
	if (srv->pm == NULL) {
		VANESSA_LOGGER_ERR("srv->pm is null");
		return -1;
	}
	if (dest == NULL) {
		VANESSA_LOGGER_ERR("dest is null");
		return -1;
	}

	/* initialize protocol module ... clear destination list */
	ret = srv->pm->initialize(srv, conn, request, *len, dest);
	if (ret != 0){
		VANESSA_LOGGER_ERR("Could not initialize protomod");
		return -1;
	}

	*tcps = 0;

	/* finalize, always set reschedule flag */
	ret = srv->pm->finalize(srv, conn, request, *len, dest, 1);

	if (ret != 0){
		VANESSA_LOGGER_ERR("Could not finalize protomod");
		return -1;
	}

	return 0;
}

/*!
 * Do nothing.
 * @param  srv service struct include service handle, protocol module and schedule module.
 * @param  conn connection data.
 * @param  response packet data from real server
 * @param  len length of packet data. it will be lengthened.
 * @retval 0  successfully check packet data.
 * @retval -1 some errors occur.
 */
static int
analyze_rsdata(struct l7vs_service *srv, struct l7vs_conn *conn,
	char *response, size_t *response_length)
{
	/* check null */
	if (srv == NULL) {
		VANESSA_LOGGER_ERR("srv is null");
		return -1;
	}
	if (conn == NULL) {
		VANESSA_LOGGER_ERR("conn is null");
		return -1;
	}
	if (conn->dest == NULL) {
		VANESSA_LOGGER_ERR("conn->dest is null");
		return -1;
	}
	if (response == NULL) {
		VANESSA_LOGGER_ERR("response is null");
		return -1;
	}
	if (response_length == NULL) {
		VANESSA_LOGGER_ERR("response_length is null");
		return -1;
	}

	return 0;
}

/*!
 * Destroy sessionless service
 * @param  srv_handle a unique service ID
 * @retval 0  successfully check packet data.
 * @retval -1 some errors occur.
 */
static int
destroy(uint32_t srv_handle)
{
	/* sessionless service list counter */
	int service_number = 0;
	int free_flag = 0;

	/* check all sessionless service list */
	for (service_number = 0; service_number < SESSIONLESS_SERVICE_NUMBER; ++service_number) {
		/* found sessionless service that has srv_handle */
		if (sessionless_service_list[service_number] != NULL && 
		    sessionless_service_list[service_number]->service_handle == srv_handle) {

			/* free and NULL */
			free(sessionless_service_list[service_number]);
			sessionless_service_list[service_number] = NULL;

			free_flag = 1;
			break;
		}
	}
	
	/* sessionless service was not found */
	if (free_flag == 0) {
		VANESSA_LOGGER_ERR("Could not find such service handle's sessionless service");
		return -1;
	}

	return 0;
}

/*!
 * Destroy sessionless service argument
 * @param  sessionless_arg sessionless service argument
 * @return void
 */
static void
destroy_sa(void **sessionless_arg)
{
	/* check null */
	if (sessionless_arg != NULL) {
		/* free and NULL */
		free((struct l7vs_sessionless_service_arg *) *sessionless_arg);
		*sessionless_arg = NULL;
	}
}

/*!
 * Create strings for service list of l7vsadm
 * @param  srv_arg service argument struct
 * @param  srv_handle a unique service ID
 * @retval 0  successfully create strings
 * @retval -1 some errors occur.
 */
static int
service_arg(struct l7vs_service_arg_multi *srv_arg_mt, uint32_t srv_handle)
{
	struct l7vs_sessionless_service *sessionless_service;
	struct l7vs_sessionless_service_arg c_sarg;
	char sessionless_argument[SERVICE_ARG_MAXSIZE];

	/* check null */
	if (srv_arg_mt == NULL) {
		VANESSA_LOGGER_ERR("srv_arg_mt is null");
		return -1;
	}

	/* search service that has such a service ID */
	sessionless_service = l7vs_protomod_sessionless_search_service(srv_handle);
	if (sessionless_service == NULL) {
		VANESSA_LOGGER_ERR("Could not find such service handle's sessionless service");
		return -1;
	}

	/* initialize argument strings */
	memset(sessionless_argument, 0, SERVICE_ARG_MAXSIZE);

	/* set sessionless args to service argument struct */
	srv_arg_mt->srv_arg.reschedule = sessionless_service->reschedule;

	/* create long argument (l7vsadm option -L/-l) */
	//strcpy(srv_arg_mt->srv_arg.protomod_key_string, sessionless_argument);

	/* create verbose argument (l7vsadm option -V/-v) */
	//strcpy(srv_arg_mt->srv_arg.protomod_opt_string, sessionless_argument);

	c_sarg.reschedule = sessionless_service->reschedule;

	memcpy(srv_arg_mt->protomod_arg, &c_sarg, sizeof(struct l7vs_sessionless_service_arg));

	return 0;
}

/*!
 * Parse l7vsadm options to sessionless argument
 * @param  sessionless_arg sessionless service argument struct
 * @param  argc number of l7vsadm argument
 * @param  argv l7vsadm argument list
 * @retval 0  successfully parse argument
 * @retval -1 some errors occur.
 */
static int
parse(void *sessionless_arg, int argc, char *argv[])
{
	struct l7vs_sessionless_service_arg *sessionless_service_arg;
	static struct option opt[] = {
		{NULL,            0,                 NULL, 0  }
	};
	int c;

	/* check null */
	if (sessionless_arg == NULL) {
		VANESSA_LOGGER_ERR("sessionless_arg is null");
		return -1;
	}

	sessionless_service_arg = (struct l7vs_sessionless_service_arg *) sessionless_arg;
	optind = 0;

	/* check all argument */
	while ((c = getopt_long(argc, argv, "", opt, NULL)) != -1) {
		/* not allow any argument*/
		return -1;
	}

	/* set default reschedule */
	sessionless_service_arg->reschedule = 1;

	return 0;
}

/*!
 * Search sessionless service from sessionless service list using service handle
 * @param  service_handle a unique service ID
 * @return sessionless service struct when service was found. NULL when service was not found.
 */
static struct l7vs_sessionless_service *
l7vs_protomod_sessionless_search_service(uint32_t service_handle)
{
	/* sessionless service list counter */
	int service_number = 0;

	/* check all sessionless service list */
	for (service_number = 0; service_number < SESSIONLESS_SERVICE_NUMBER; ++service_number) {
		/* found the service has same service handle */
		if (sessionless_service_list[service_number] != NULL && 
		    sessionless_service_list[service_number]->service_handle == service_handle) {
			return sessionless_service_list[service_number];
		}
	}
	
	/* not found */
	return NULL;
}

/*!
 * Create sessionless service.
 * @param  void
 * @return sessionless service struct when create a service. NULL when cannot create service.
 */
static struct l7vs_sessionless_service *
l7vs_protomod_sessionless_create_service()
{
	/* sessionless service list counter */
	int service_number = 0;

	/* check all sessionless service list */
	for (service_number = 0; service_number < SESSIONLESS_SERVICE_NUMBER - 1; ++service_number) {
		/* if pointer that does not point NULL exists ... */
		if (sessionless_service_list[service_number] == NULL) {
			/* create a service at empty pointer */
			sessionless_service_list[service_number] = (struct l7vs_sessionless_service *) calloc(1, sizeof(struct l7vs_sessionless_service));
			if (sessionless_service_list[service_number] == NULL) {
				VANESSA_LOGGER_ERR("Could not allocate memory");
				break;
			}
			return sessionless_service_list[service_number];
		}
	}
	
	/* all service list is full */
	return NULL;
}

/*!
 * Create temporary sessionless service.
 * @param  void
 * @return sessionless service struct when create a service. NULL when cannot create service.
 */
static struct l7vs_sessionless_service *
l7vs_protomod_sessionless_create_temp_service()
{
	/* if pointer that does not point NULL exists ... */
	if (sessionless_service_list[SESSIONLESS_SERVICE_NUMBER - 1] != NULL) {
		VANESSA_LOGGER_ERR("temporary sessionless service is being used by other process");
		return NULL;
	}
	sessionless_service_list[SESSIONLESS_SERVICE_NUMBER - 1] = (struct l7vs_sessionless_service *) calloc(1, sizeof(struct l7vs_sessionless_service));
	if (sessionless_service_list[SESSIONLESS_SERVICE_NUMBER - 1] == NULL) {
		VANESSA_LOGGER_ERR("Could not allocate memory");
		return NULL;
	}

	return sessionless_service_list[SESSIONLESS_SERVICE_NUMBER - 1];
}
