/*
 * @file  module.c
 * @brief the framework module of manage shared object 
 * @brief it enables dynamic loading of shared object 
 *
 * L7VSD: Linux Virtual Server for Layer7 Load Balancing
 * Copyright (C) 2005  NTT COMWARE Corporation.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 **********************************************************************/

#ifndef _GNU_SOURCE
#define _GNU_SOURCE
#endif /* _GNU_SOURCE */

#include <sys/param.h>
#include <sys/types.h>
#include <dlfcn.h>
#include <stdio.h>
#include <stdlib.h>
#include <glib.h>
#include "vanessa_logger.h"
#include "l7vs_module.h"
#include "l7vs.h"

#define L7VS_MODULE_INITFN	      "init"

char l7vs_module_path[MAXPATHLEN];

int
l7vs_module_init(char *modpath)
{
	if (modpath == NULL) {
		strcpy(l7vs_module_path, L7VS_MODULE_PATH);
	} else {
		if (strlen(modpath) > sizeof(l7vs_module_path) - 1) {
			return -1;
		}
		strcpy(l7vs_module_path, modpath);
	}

	return 0;
}

void
l7vs_module_fini(void)
{
	return;
}

void *
l7vs_module_load(char *modname, char *type)
{
	void *h;
	void *modhandle;
	char *modpath;
	void *(*initf)(void *);

	if (modname == NULL || type == NULL) {
		return NULL;
	}

	modpath = (char *) malloc(strlen(l7vs_module_path) + strlen(type) + strlen(modname) + 6);
	if (modpath == NULL) {
		VANESSA_LOGGER_ERR("Could not allocate memory");
		return NULL;
	}
	sprintf(modpath, "%s/%s_%s.so", l7vs_module_path, type, modname);

	h = dlopen(modpath, RTLD_LAZY);
	if (h == NULL) {
		VANESSA_LOGGER_ERR_UNSAFE("Could not open %s module: %s",
					  modpath, dlerror());
		free(modpath);
		modpath = NULL;
		return NULL;
	}
	free(modpath);
	modpath = NULL;

	*(void **) (&initf) = dlsym(h, L7VS_MODULE_INITFN);
	if (initf == NULL) {
		VANESSA_LOGGER_ERR_UNSAFE("Could not find symbol %s: %s",
					  L7VS_MODULE_INITFN, dlerror());
		dlclose(h);
		return NULL;
	}

	modhandle = (*initf)(h);
	if (modhandle == NULL) {
		VANESSA_LOGGER_ERR("Module initialization failed");
		dlclose(h);
		return NULL;
	}

	return modhandle;
}

void
l7vs_module_unload(void *handle)
{
	if (handle != NULL) {
		dlclose(handle);
	}
}

void
l7vs_module_register(GList **modlist, void *mod)
{
	if (modlist != NULL && mod != NULL) {
		*modlist = g_list_append(*modlist, (gpointer)mod);
	}
}

void
l7vs_module_remove(GList **modlist, void *mod)
{
	if (modlist != NULL && mod != NULL) {
		*modlist = g_list_remove(*modlist, mod);
	}
}

void *
l7vs_module_lookup(GList *modlist, void *key, GCompareFunc cmp)
{
	GList *l;

	if (modlist == NULL || key == NULL || cmp == NULL) {
		return NULL;
	}

	l = g_list_find_custom(modlist, (gpointer)key, cmp);
	if (l == NULL) {
		return NULL;
	}

	return (void *)l->data;
}
