#include <stdlib.h>
#include <time.h>
#include <getopt.h>
#include "vanessa_logger.h"
#include "l7vs_service.h"
#include "l7vs_conn.h"
#include "l7vs_dest.h"
#include "l7vs_module.h"
#include "module_http.h"

#define EXPIRE_MAXSIZE         (64)
#define COOKIE_NAME_MAXSIZE    (128)
#define SERVICE_ARG_MAXSIZE    (512)
#define CINSERT_SERVICE_NUMBER (16)

struct l7vs_cinsert_xf_service {
	uint32_t service_handle;
	char cookie_name[COOKIE_NAME_MAXSIZE];
	int cookie_expire;
	int reschedule;
};

struct  l7vs_cinsert_xf_service_arg {
	char cookie_name[COOKIE_NAME_MAXSIZE];
	int cookie_expire;
	int reschedule;
};

static void  fini(void);
static int   create(void *cinsert_xf_arg, uint32_t service_handle);
static void* create_sa(struct l7vs_service_arg *srv_arg);
static int   compare(uint32_t srv_handle1, uint32_t srv_handle2);
static int   match_cldata(struct l7vs_service *srv, struct l7vs_conn *conn,
                        char *buf, size_t *len, struct l7vs_dest **dest, int *tcps);
static int   analyze_rsdata(struct l7vs_service *srv, struct l7vs_conn *conn,
                        char *response, size_t *response_length);
static int   destroy(uint32_t srv_handle);
static void  destroy_sa(void **cinsert_xf_arg);
static int   service_arg(struct l7vs_service_arg_multi *srv_arg_mt, uint32_t srv_handle);
static int   parse(void *cinsert_xf_arg, int argc, char *argv[]);

static time_t l7vs_protomod_cinsert_xf_timeout(int timeout);
static int    l7vs_protomod_cinsert_xf_set_cookie(char *response, int offset_length, char *set_cookie_field, int response_length);
static struct l7vs_cinsert_xf_service *l7vs_protomod_cinsert_xf_search_service(uint32_t service_handle);
static struct l7vs_cinsert_xf_service *l7vs_protomod_cinsert_xf_create_service();
static struct l7vs_cinsert_xf_service *l7vs_protomod_cinsert_xf_create_temp_service();

struct l7vs_cinsert_xf_service *cinsert_xf_service_list[CINSERT_SERVICE_NUMBER];

static struct l7vs_protomod cinsert_xf_protomod = {
	NULL,           /* handle */
	"cinsert_xf",      /* modname */
	0,              /* refcnt */
	create,         /* create function */
	compare,        /* compare function */
	match_cldata,   /* match_cldata function */
	analyze_rsdata, /* analyze_rsdata function */
	destroy,        /* destroy function */
	fini,           /* fini function */
	create_sa,      /* create_sa function */
	service_arg,    /* service_arg function */
	parse,          /* parse function */
	destroy_sa,     /* destroy_sa function */
};

/*!
 * Protocol module initialize function. This function run when dlopen and dlsym at first time.
 * @param  handle dlopen's handle
 * @return l7vs_protomod struct
 */
struct l7vs_protomod *
init(void *handle)
{
	/* initialize cinsert_xf service list */
	memset(cinsert_xf_service_list, 0, sizeof(struct l7vs_cinsert_xf_service *) * CINSERT_SERVICE_NUMBER);
	/* set dlopen's handle */
	cinsert_xf_protomod.handle = handle;

	return &cinsert_xf_protomod;
}

/*!
 * Protocol module finalize function. free all cinsert_xf service list just in case.
 * @param   void
 * @return  void
 */
static void
fini(void)
{
	/* cinsert_xf service list counter */
	int service_number = 0;

	/* check all cinsert_xf service list */
	for (service_number = 0; service_number < CINSERT_SERVICE_NUMBER; ++service_number) {
		/* if pointer that does not point NULL exists ... */
		if (cinsert_xf_service_list[service_number] != NULL) {
			/* free and points NULL */
			free(cinsert_xf_service_list[service_number]);
			cinsert_xf_service_list[service_number] = NULL;
		}
	}
}

/*!
 * Create cinsert_xf service struct.
 * @param  cinsert_xf_arg    cinsert_xf service argument struct
 * @param  service_handle a unique service ID
 * @retval 0  successfully create cinsert_xf service.
 * @retval -1 some errors occur.
 */
static int
create(void *cinsert_xf_arg, uint32_t service_handle)
{
	struct l7vs_cinsert_xf_service *cinsert_xf_service;
	struct l7vs_cinsert_xf_service_arg *cinsert_xf_service_arg;

	/* check null */
	if (cinsert_xf_arg == NULL) {
		VANESSA_LOGGER_ERR("cinsert_xf_arg is null");
		return -1;
	}

	if (service_handle != TEMP_SERVICEHANDLE) {
		/* search empty cinsert_xf service list and create cinsert_xf service */
		cinsert_xf_service = l7vs_protomod_cinsert_xf_create_service();
	} else {
		/* create temporary cinsert_xf service */
		cinsert_xf_service = l7vs_protomod_cinsert_xf_create_temp_service();
	}
	if (cinsert_xf_service == NULL) {
		VANESSA_LOGGER_ERR("Could not make temporary cinsert_xf service");
		return -1;
	}

	cinsert_xf_service_arg = (struct l7vs_cinsert_xf_service_arg *) cinsert_xf_arg;

	/* set service handle, cookie name, cookie expire, and reschedule flag */
	cinsert_xf_service->service_handle = service_handle;
	strncpy(cinsert_xf_service->cookie_name, cinsert_xf_service_arg->cookie_name, COOKIE_NAME_MAXSIZE);
	cinsert_xf_service->cookie_expire = cinsert_xf_service_arg->cookie_expire;
	cinsert_xf_service->reschedule = cinsert_xf_service_arg->reschedule;

	return 0;
}

/*!
 * Create cinsert_xf service argument struct.
 * @param  srv_arg service argument struct
 * @return void*   cinsert_xf service argument struct
 */
static void *
create_sa(struct l7vs_service_arg *srv_arg)
{
	struct l7vs_cinsert_xf_service_arg *cinsert_xf_service_arg;

	if (srv_arg == NULL) {
		return NULL;
	}

	/* create cinsert_xf service argument struct */
	cinsert_xf_service_arg = (struct l7vs_cinsert_xf_service_arg *) calloc(1, sizeof(struct l7vs_cinsert_xf_service_arg));
	if (cinsert_xf_service_arg == NULL) {
		VANESSA_LOGGER_ERR("Could not allocate memory");
		return (void *) cinsert_xf_service_arg;
	}

	/* set cinsert_xf service argument size and protomod name "cinsert_xf" */
	srv_arg->len = sizeof(struct l7vs_cinsert_xf_service_arg);
	strcpy(srv_arg->protomod, cinsert_xf_protomod.modname);

	return (void *) cinsert_xf_service_arg;
}

/*!
 * Compare two service.
 * @param  srv_handle1 one of a unique service ID
 * @param  srv_handle2 one of a unique service ID
 * @retval 0  they matched perfectly.
 * @retval -1 they are different.
 */
static int
compare(uint32_t srv_handle1, uint32_t srv_handle2)
{
	struct l7vs_cinsert_xf_service *cinsert_xf_srv1, *cinsert_xf_srv2;

	/* search service that has such a service ID(1) */
	cinsert_xf_srv1 = l7vs_protomod_cinsert_xf_search_service(srv_handle1);
	if (cinsert_xf_srv1 == NULL) {
		VANESSA_LOGGER_ERR("Could not find such service handle's cinsert_xf service");
		return -1;
	}

	/* search service that has such a service ID(2) */
	cinsert_xf_srv2 = l7vs_protomod_cinsert_xf_search_service(srv_handle2);
	if (cinsert_xf_srv2 == NULL) {
		VANESSA_LOGGER_ERR("Could not find such service handle's cinsert_xf service");
		return -1;
	}

	/* compare two cookie name */
	if (strcmp(cinsert_xf_srv1->cookie_name, cinsert_xf_srv2->cookie_name) != 0) {
		return -1;
	}

	return 0;
}

/*!
 * Check the client packet and determine a real server.
 * @param  srv service struct include service handle, protocol module and schedule module.
 * @param  conn connection data.
 * @param  request packet data from client
 * @param  len length of packet data
 * @param  dest destination (real server) list
 * @param  tcps TCP Splicer flag
 * @retval 0  successfully check packet data
 * @retval -1 some errors occur.
 */
static int
match_cldata(struct l7vs_service *srv, struct l7vs_conn *conn,
      char *request, size_t *len, struct l7vs_dest **dest, int *tcps)
{
	struct l7vs_cinsert_xf_service *cinsert_xf_service;
	struct l7vs_dest destination;
	int ret;
	int offset_length;
	char *cookie_value;
	char *next_line;
	char encoded_address[16];
	char x_forwarded_for_header[48];

	/* check null */
	if (srv == NULL) {
		VANESSA_LOGGER_ERR("srv is null");
		return -1;
	}
	if (srv->pm == NULL) {
		VANESSA_LOGGER_ERR("srv->pm is null");
		return -1;
	}
	if (dest == NULL) {
		VANESSA_LOGGER_ERR("dest is null");
		return -1;
	}

	/* search service that has such a service ID */
	cinsert_xf_service = l7vs_protomod_cinsert_xf_search_service(srv->handle);
	if (cinsert_xf_service == NULL) {
		VANESSA_LOGGER_ERR("Could not find such service handle's cinsert_xf service");
		return -1;
	}

	/* initialize protocol module ... clear destination list */
	ret = srv->pm->initialize(srv, conn, request, *len, dest);
	if (ret != 0){
		VANESSA_LOGGER_ERR("Could not initialize protomod");
		return -1;
	}

	/* check the HTTP method in HTTP request header */
	if (http_check_request_method(request, len) == NULL) {
		goto OUT;
	}

	/* construct X-Forwarded-For header item */
	sprintf(x_forwarded_for_header, "X-Forwarded-For: %s\r\n", inet_ntoa(conn->caddr.sin_addr));
	
	/* get insert position (end of request excludes last CRLF) */
	next_line = request + strlen(request) - 2;
	offset_length = (int) (next_line - request);

	/* insert X-Forwarded-For header field */
	l7vs_protomod_cinsert_xf_set_cookie(request, offset_length, x_forwarded_for_header, *len);

	/* add header length */
	*len += strlen(x_forwarded_for_header);

	/* search Cookie field in HTTP request header */
	cookie_value = http_search_header_cookie_value(request, cinsert_xf_service->cookie_name);
	if (cookie_value == NULL) {
		goto OUT;
	}

	/* memory copy encoded strings(15 byte) of IP address and port number */
	memcpy(encoded_address, cookie_value, 15);
	encoded_address[15] = '\0';

	/* decode IP address and port number and set to destination struct */
	http_decode_address(encoded_address, (u_long *) &destination.addr.sin_addr.s_addr, &destination.addr.sin_port);

	/* set return value */
	*dest = &destination;

OUT:
	*tcps = 0;

	/* finalize */
	ret = srv->pm->finalize(srv, conn, request, *len, dest, cinsert_xf_service->reschedule);

	if (ret != 0){
		VANESSA_LOGGER_ERR("Could not finalize protomod");
		return -1;
	}

	return 0;
}

/*!
 * Check the real server packet and insert a Set-Cookie field.
 * @param  srv service struct include service handle, protocol module and schedule module.
 * @param  conn connection data.
 * @param  response packet data from real server
 * @param  len length of packet data. it will be lengthened.
 * @retval 0  successfully check packet data.
 * @retval -1 some errors occur.
 */
static int
analyze_rsdata(struct l7vs_service *srv, struct l7vs_conn *conn,
	char *response, size_t *response_length)
{
	struct l7vs_cinsert_xf_service *cinsert_xf_service;
	char *next_line;
	char expire_strings[EXPIRE_MAXSIZE];
	char encoded_address[16];
	char set_cookie_field[L7VS_PROTOMOD_MAX_ADD_BUFSIZE];
	int  offset_length;
	int  ret;
	time_t expire_time;

	/* check null */
	if (srv == NULL) {
		VANESSA_LOGGER_ERR("srv is null");
		return -1;
	}
	if (conn == NULL) {
		VANESSA_LOGGER_ERR("conn is null");
		return -1;
	}
	if (conn->dest == NULL) {
		VANESSA_LOGGER_ERR("conn->dest is null");
		return -1;
	}
	if (response == NULL) {
		VANESSA_LOGGER_ERR("response is null");
		return -1;
	}
	if (response_length == NULL) {
		VANESSA_LOGGER_ERR("response_length is null");
		return -1;
	}

	/* sorry flag check */
	if (conn->sorry_conn_flag == 1) {
		return 0;
	}

	/* search service that has such a service ID */
	cinsert_xf_service = l7vs_protomod_cinsert_xf_search_service(srv->handle);
	if (cinsert_xf_service == NULL) {
		VANESSA_LOGGER_ERR("Could not find such service handle's cinsert_xf service");
		return -1;
	}

	/* check the HTTP status in HTTP response header */
	ret = http_check_response_status(response);
	/* when status is 4xx or 5xx, do not insert Set-Cookie field */
	if (ret != 0) {
		return 0;
	}

	/* get top of HTTP header field line */
	next_line = http_skip_header_line(response);
	if (next_line == NULL) {
		return 0;
	}

	/* create encode strings of IP address and port number */
	http_encode_address(encoded_address, conn->dest->addr.sin_addr.s_addr, conn->dest->addr.sin_port);

	/* calculate expire time */
	expire_time = l7vs_protomod_cinsert_xf_timeout(cinsert_xf_service->cookie_expire);

	/* create expire strings */
	http_cookie_expire(expire_time, expire_strings);

	/* create Set-Cookie header field */
	sprintf(set_cookie_field, "Set-Cookie: %s=%s; expires=%s; path=/\r\n",
	        cinsert_xf_service->cookie_name, encoded_address, expire_strings);

	/* offset (HTTP status line length) */
	offset_length = (int) (next_line - response);

	/* insert Set-Cookie header field */
	l7vs_protomod_cinsert_xf_set_cookie(response, offset_length, set_cookie_field, *response_length);

	/* add response length */
	*response_length += strlen(set_cookie_field);

	return 0;
}

/*!
 * Destroy cinsert_xf service
 * @param  srv_handle a unique service ID
 * @retval 0  successfully check packet data.
 * @retval -1 some errors occur.
 */
static int
destroy(uint32_t srv_handle)
{
	/* cinsert_xf service list counter */
	int service_number = 0;
	int free_flag = 0;

	/* check all cinsert_xf service list */
	for (service_number = 0; service_number < CINSERT_SERVICE_NUMBER; ++service_number) {
		/* found cinsert_xf service that has srv_handle */
		if (cinsert_xf_service_list[service_number] != NULL && 
		    cinsert_xf_service_list[service_number]->service_handle == srv_handle) {

			/* free and NULL */
			free(cinsert_xf_service_list[service_number]);
			cinsert_xf_service_list[service_number] = NULL;

			free_flag = 1;
			break;
		}
	}
	
	/* cinsert_xf service was not found */
	if (free_flag == 0) {
		VANESSA_LOGGER_ERR("Could not find such service handle's cinsert_xf service");
		return -1;
	}

	return 0;
}

/*!
 * Destroy cinsert_xf service argument
 * @param  cinsert_xf_arg cinsert_xf service argument
 * @return void
 */
static void
destroy_sa(void **cinsert_xf_arg)
{
	/* check null */
	if (cinsert_xf_arg != NULL) {
		/* free and NULL */
		free((struct l7vs_cinsert_xf_service_arg *) *cinsert_xf_arg);
		*cinsert_xf_arg = NULL;
	}
}

/*!
 * Create strings for service list of l7vsadm
 * @param  srv_arg service argument struct
 * @param  srv_handle a unique service ID
 * @retval 0  successfully create strings
 * @retval -1 some errors occur.
 */
static int
service_arg(struct l7vs_service_arg_multi *srv_arg_mt, uint32_t srv_handle)
{
	struct l7vs_cinsert_xf_service *cinsert_xf_service;
	struct l7vs_cinsert_xf_service_arg c_sarg;
	char cinsert_xf_argument[SERVICE_ARG_MAXSIZE];

	/* check null */
	if (srv_arg_mt == NULL) {
		VANESSA_LOGGER_ERR("srv_arg_mt is null");
		return -1;
	}

	/* search service that has such a service ID */
	cinsert_xf_service = l7vs_protomod_cinsert_xf_search_service(srv_handle);
	if (cinsert_xf_service == NULL) {
		VANESSA_LOGGER_ERR("Could not find such service handle's cinsert_xf service");
		return -1;
	}

	/* initialize argument strings */
	memset(cinsert_xf_argument, 0, SERVICE_ARG_MAXSIZE);

	/* set cinsert_xf args to service argument struct */
	srv_arg_mt->srv_arg.reschedule = cinsert_xf_service->reschedule;

	/* create long argument (l7vsadm option -L/-l) */
	sprintf(cinsert_xf_argument, "--cookie-name %s", cinsert_xf_service->cookie_name);
	strcpy(srv_arg_mt->srv_arg.protomod_key_string, cinsert_xf_argument);

	/* create verbose argument (l7vsadm option -V/-v) */
	sprintf(cinsert_xf_argument + strlen(cinsert_xf_argument), " --cookie-expire %d", cinsert_xf_service->cookie_expire);
	strcpy(srv_arg_mt->srv_arg.protomod_opt_string, cinsert_xf_argument);

	strncpy(c_sarg.cookie_name, cinsert_xf_service->cookie_name, COOKIE_NAME_MAXSIZE);
	c_sarg.cookie_expire = cinsert_xf_service->cookie_expire;
	c_sarg.reschedule = cinsert_xf_service->reschedule;

	memcpy(srv_arg_mt->protomod_arg, &c_sarg, sizeof(struct l7vs_cinsert_xf_service_arg));

	return 0;
}

/*!
 * Parse l7vsadm options to cinsert_xf argument
 * @param  cinsert_xf_arg cinsert_xf service argument struct
 * @param  argc number of l7vsadm argument
 * @param  argv l7vsadm argument list
 * @retval 0  successfully parse argument
 * @retval -1 some errors occur.
 */
static int
parse(void *cinsert_xf_arg, int argc, char *argv[])
{
	struct l7vs_cinsert_xf_service_arg *cinsert_xf_service_arg;
	static struct option opt[] = {
		{"cookie-name",   required_argument, NULL, 'C'},
		{"cookie-expire", required_argument, NULL, 'E'},
		{"reschedule",    no_argument,       NULL, 'F'},
		{"no-reschedule", no_argument,       NULL, 'N'},
		{NULL,            0,                 NULL, 0  }
	};
	int c;
	int cookie_name_flag = 0;
	int cookie_expire_flag = 0;
	int reschedule_flag = 0;

	/* check null */
	if (cinsert_xf_arg == NULL) {
		VANESSA_LOGGER_ERR("cinsert_xf_arg is null");
		return -1;
	}

	cinsert_xf_service_arg = (struct l7vs_cinsert_xf_service_arg *) cinsert_xf_arg;
	optind = 0;

	/* check all argument */
	while ((c = getopt_long(argc, argv, "C:E:FN", opt, NULL)) != -1) {
		switch (c) {
		/* --cookie-name / -C */
		case 'C':
			/* check maximum length */
			if (strlen(optarg) >= sizeof(cinsert_xf_service_arg->cookie_name)) {
				VANESSA_LOGGER_ERR_UNSAFE("%s: path too long", optarg);
				return -1;
			}
			strcpy(cinsert_xf_service_arg->cookie_name, optarg);
			cookie_name_flag = 1;
			break;

		/* --cookie-expire / -E */
		case 'E':
			/* reject minus */
			if (atoi(optarg) < 0) {
				VANESSA_LOGGER_ERR("You can't specify less than 0");
				return -1;
			}
			cinsert_xf_service_arg->cookie_expire = atoi(optarg);
			cookie_expire_flag = 1;
			break;

		/* --reschedule / -F */
		case 'F':
			/* reschedule on */
			cinsert_xf_service_arg->reschedule = 1;
			reschedule_flag++;
			break;

		/* --no-reschedule / -N */
		case 'N':
			/* reschedule off */
			cinsert_xf_service_arg->reschedule = 0;
			reschedule_flag++;
			break;

		/* else error */
		default:
			return -1;
		}
	}

	/* when set both -F and -N at the same time */
	if (reschedule_flag == 2) {
		VANESSA_LOGGER_ERR("You should choose either of reschdule or no-reschedule");
		return -1;
	}

	/* set default no reschedule */
	if (reschedule_flag == 0) {
		cinsert_xf_service_arg->reschedule = 0;
	}

	/* set default cookie name */
	if (cookie_name_flag == 0) {
		strcpy(cinsert_xf_service_arg->cookie_name, "CookieName");
	}

	/* set default cookie expire (1 day) */
	if (cookie_expire_flag == 0) {
		cinsert_xf_service_arg->cookie_expire = 86400;
	}

	return 0;
}

/*!
 * Calculate future time
 * @param  timeout plus second
 * @return time_t future time
 */
static time_t
l7vs_protomod_cinsert_xf_timeout(int timeout)
{
	time_t timer;

	/* now */
	timer = time((time_t *) 0);

	/* future */
	timer += timeout;

	return timer;
}

/*!
 * Insert Set-Cookie header field to second line of response header
 * @param  response HTTP response header strings
 * @param  offset_length HTTP status line length
 * @param  set_cookie_field Set-Cookie header field strings
 * @param  response_length all of HTTP response length
 * @retval 0  successfully insert Set-Cookie field
 * @retval -1 some errors occur.
 */
static int
l7vs_protomod_cinsert_xf_set_cookie(char *response, int offset_length, char *set_cookie_field, int response_length)
{
	char *copy_buffer;

	/* check args */
	if (response == NULL) {
		VANESSA_LOGGER_ERR("response is null");
		return -1;
	}
	if (set_cookie_field == NULL) {
		VANESSA_LOGGER_ERR("set_cookie_field is null");
		return -1;
	}
	if (offset_length < 0) {
		VANESSA_LOGGER_ERR("offset_length is negative number");
		return -1;
	}
	if (response_length < 0) {
		VANESSA_LOGGER_ERR("response_length is negative number");
		return -1;
	}
	if (response_length < offset_length) {
		VANESSA_LOGGER_ERR("offset_length is longer than response_length");
		return -1;
	}

	/* calloc buffer */
	copy_buffer = (char *) calloc(1, response_length - offset_length);
	if (copy_buffer == NULL) {
		VANESSA_LOGGER_ERR("Could not allocate memory");
		return -1;
	}

	/* backup strings */
	memcpy(copy_buffer, response + offset_length, response_length - offset_length);

	/* insert Set-Cookie field */
	memcpy(response + offset_length, set_cookie_field, strlen(set_cookie_field));

	/* append backup strings and terminate null*/
	memcpy(response + offset_length + strlen(set_cookie_field), copy_buffer, response_length - offset_length);
	response[offset_length + strlen(set_cookie_field) + response_length - offset_length] = '\0';

	/* free */
	free(copy_buffer);
	copy_buffer = NULL;

	return 0;
}

/*!
 * Search cinsert_xf service from cinsert_xf service list using service handle
 * @param  service_handle a unique service ID
 * @return cinsert_xf service struct when service was found. NULL when service was not found.
 */
static struct l7vs_cinsert_xf_service *
l7vs_protomod_cinsert_xf_search_service(uint32_t service_handle)
{
	/* cinsert_xf service list counter */
	int service_number = 0;

	/* check all cinsert_xf service list */
	for (service_number = 0; service_number < CINSERT_SERVICE_NUMBER; ++service_number) {
		/* found the service has same service handle */
		if (cinsert_xf_service_list[service_number] != NULL && 
		    cinsert_xf_service_list[service_number]->service_handle == service_handle) {
			return cinsert_xf_service_list[service_number];
		}
	}
	
	/* not found */
	return NULL;
}

/*!
 * Create cinsert_xf service.
 * @param  void
 * @return cinsert_xf service struct when create a service. NULL when cannot create service.
 */
static struct l7vs_cinsert_xf_service *
l7vs_protomod_cinsert_xf_create_service()
{
	/* cinsert_xf service list counter */
	int service_number = 0;

	/* check all cinsert_xf service list */
	for (service_number = 0; service_number < CINSERT_SERVICE_NUMBER - 1; ++service_number) {
		/* if pointer that does not point NULL exists ... */
		if (cinsert_xf_service_list[service_number] == NULL) {
			/* create a service at empty pointer */
			cinsert_xf_service_list[service_number] = (struct l7vs_cinsert_xf_service *) calloc(1, sizeof(struct l7vs_cinsert_xf_service));
			if (cinsert_xf_service_list[service_number] == NULL) {
				VANESSA_LOGGER_ERR("Could not allocate memory");
				break;
			}
			return cinsert_xf_service_list[service_number];
		}
	}
	
	/* all service list is full */
	return NULL;
}

/*!
 * Create temporary cinsert_xf service.
 * @param  void
 * @return cinsert_xf service struct when create a service. NULL when cannot create service.
 */
static struct l7vs_cinsert_xf_service *
l7vs_protomod_cinsert_xf_create_temp_service()
{
	/* if pointer that does not point NULL exists ... */
	if (cinsert_xf_service_list[CINSERT_SERVICE_NUMBER - 1] != NULL) {
		VANESSA_LOGGER_ERR("temporary cinsert_xf service is being used by other process");
		return NULL;
	}
	cinsert_xf_service_list[CINSERT_SERVICE_NUMBER - 1] = (struct l7vs_cinsert_xf_service *) calloc(1, sizeof(struct l7vs_cinsert_xf_service));
	if (cinsert_xf_service_list[CINSERT_SERVICE_NUMBER - 1] == NULL) {
		VANESSA_LOGGER_ERR("Could not allocate memory");
		return NULL;
	}

	return cinsert_xf_service_list[CINSERT_SERVICE_NUMBER - 1];
}
