/*
 * @file  l7vsd.c
 * @brief the main module of l7vsd 
 * @brief it drives main event loop 
 *
 * L7VSD: Linux Virtual Server for Layer7 Load Balancing
 * Copyright (C) 2005  NTT COMWARE Corporation.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 **********************************************************************/

#ifndef _GNU_SOURCE
#define _GNU_SOURCE     /* for getopt_long() */
#endif /* _GNU_SOURCE */
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <signal.h>
#include <unistd.h>
#include <stdlib.h>
#include <getopt.h>
#include <glib.h>
#include "vanessa_logger.h"
#include "l7vs.h"
#include "l7vs_module.h"
#include "l7vs_config.h"
#include "l7vs_iomuxlist.h"
#include "l7vs_config.h"
#include "l7vs_lsock.h"

static void sig_exit_handler(int sig);
static int set_sighandler(int sig, void (*handler)(int));
static int set_sighandlers(void);
static void usage(FILE *fp, char *ident);

static int exit_requested = 0;
static int received_sig = 0;

static void
sig_exit_handler(int sig)
{
	received_sig = sig;
        exit_requested++;
}

static int
set_sighandler(int sig, void (*handler)(int))
{
        struct sigaction act;
        int ret;

        ret = sigaction(sig, NULL, &act);
        if (ret < 0) {
                VANESSA_LOGGER_ERR_UNSAFE("sigaction on signal %d failed",
                                          sig);
                return ret;
        }
        act.sa_flags &= ~SA_RESETHAND;
        act.sa_handler = handler;

        ret = sigaction(sig, &act, NULL);
        if (ret < 0) {
                VANESSA_LOGGER_ERR_UNSAFE("sigaction on signal %d failed",
                                          sig);
                return ret;
        }

        return 0;
}

static int
set_sighandlers(void)
{
        int ret;

#define SET_SIGHANDLER(sig, handler)                    \
        do {                                            \
                ret = set_sighandler((sig), (handler)); \
                if (ret < 0) {                          \
                        return ret;                     \
                }                                       \
        } while (0)

        SET_SIGHANDLER(SIGHUP,  sig_exit_handler);
        SET_SIGHANDLER(SIGINT,  sig_exit_handler);
        SET_SIGHANDLER(SIGQUIT, sig_exit_handler);
        SET_SIGHANDLER(SIGTERM, sig_exit_handler);
        SET_SIGHANDLER(SIGUSR1, SIG_IGN);
        SET_SIGHANDLER(SIGUSR2, SIG_IGN);
        SET_SIGHANDLER(SIGALRM, SIG_IGN);
        SET_SIGHANDLER(SIGCHLD, SIG_IGN);

#undef SET_SIGHANDLER

        return 0;
}

static void
usage(FILE *fp, char *ident)
{
	if (!fp) {
		VANESSA_LOGGER_ERR("fp is NULL.");
		return;
	}
	if (!ident) {
		VANESSA_LOGGER_ERR("ident is NULL.");
		return;
	}
        fprintf(fp,
                "Usage: %s [-d] [-b] [-h]\n"
                "   -d    --debug        run in debug mode (in foreground)\n"
                "   -h    --help         print this help messages and exit\n"
		"   -b    --blocking	 blocking mode enable(default non-blocking)",
                ident);
}

int
main(int argc, char *argv[])
{
        vanessa_logger_t *vl;
        int c;
        int ret;
        int debug = 0;
	int blocking = 0;

        static struct option options[] = {
                {"debug",               no_argument,            NULL, 'd'},
                {"help",                no_argument,            NULL, 'h'},
		{"blocking",		no_argument,		NULL, 'b'},
                {NULL,                  0,                      NULL, 0}
        };

        while ((c = getopt_long(argc, argv, "dhb", options, NULL)) != -1) {
                switch (c) {
                case 'd':
                        debug = 1;
                        break;
		case 'b':
			blocking = -1;
			break;
                case '?':
                default:
                        fprintf(stderr, "%s: unknown option: %s\n",
                                argv[0], argv[optind - 1]);
                        usage(stderr, argv[0]);
                        exit(1);
                        break;
                case 'h':
                        usage(stdout, argv[0]);
                        exit(0);
                        break;
                }
        }

        if (debug) {
                vl = vanessa_logger_openlog_filehandle(stderr, argv[0], 
                                                       LOG_DEBUG, 0);
        } else {
                ret = daemon(0, 0);
                if (ret < 0) {
                        perror("daemon() failed");
                        exit(1);
                }

                /* XXX TBD: make log faculty configurable. */
                vl = vanessa_logger_openlog_syslog(LOG_LOCAL0, argv[0], 
                                                       LOG_INFO, 0);
        }
        vanessa_logger_set(vl);

        VANESSA_LOGGER_INFO("Starting.");

        set_sighandlers();
        sprintf(l7vs_module_path, L7VS_MODULE_PATH);
	ret = l7vs_iomux_init();
        if (ret < 0) {
                exit(1);
        }
        ret = l7vs_config_init();
        if (ret < 0) {
                exit(1);
        }

        l7vs_module_init(NULL);
        l7vs_lsock_init();

        while (!exit_requested) {
		ret = l7vs_iomux_poll(NULL, blocking);
                if (ret < 0) {
                        if (errno == EINTR) {
                                /* stop_requested will be checked... */
                                continue;
                        }
                        VANESSA_LOGGER_ERR_UNSAFE("poll: %s",
                                                  strerror(errno));
                        break;
                }
        }

	if (received_sig) {
	        VANESSA_LOGGER_INFO_UNSAFE("signal %d received.", received_sig);
	}

        l7vs_lsock_fini();
        l7vs_module_fini();
        l7vs_config_fini();
	l7vs_iomux_fini();

        VANESSA_LOGGER_INFO("Exiting.");

        return 0;
}
