/*! @file
 *  @brief Sync module program
 *
 *  @author NTT COMWARE
 *  @date 2007-07-24
 *  @version 0.1
 */


#include <stdlib.h>
#include <unistd.h>
#include "l7vs_sync.h"
#include "l7vs_iom.h"
#include "l7vs_iomuxlist.h"
#include "l7vs.h"
#include "vanessa_logger.h"
#include "sync.h"

//! iomux structure
static struct l7vs_iomux *l7vs_sync_iomux;

//! Initial value of argument structure

struct l7vs_sync_iom sync_iom ={
		SYNC_SEND,
		NULL,
		0,
		0,
};


/*! Initialize sync
 *
 * @param block_num Memory block number.
 * @param mode Operation mode. (0 : send, 1: receive)
 * @param fd File descriptor.
 * 
 * @return Initialize results. (0 , 1)
 * @retval 0 Everything is ok
 * @retval 1 Error
 */
int l7vs_sync_init(struct  l7vs_sync_init_data *init_arg)
{
	size_t block_num;
	enum sync_operation mode;
	int fd;
	int ret;

	// Repeat check
	if (sync_iom.memory != NULL)
	{
		VANESSA_LOGGER_ERR(" Initialization has already been done. ");
		return 1;
	}

	// Argument check
	if (!init_arg)
	{
		VANESSA_LOGGER_ERR(" Init argument is NULL.");
		return 1;
	}
	// sync socket initialize
	ret = sync_sock_init(init_arg->ip_addr, init_arg->servname,
				init_arg->nic, init_arg->mode, &init_arg->fd);
	if (ret != SYNC_OK)
	{
		VANESSA_LOGGER_ERR("sync_sock_init error");
		return 1;
	}
	block_num = init_arg->size;
	mode = init_arg->mode;
	fd = init_arg->fd;

	// Argument check
	// block_num error : 0
	if (block_num == 0)
	{
		VANESSA_LOGGER_ERR(" The memory block is 0.  ");
		return 1;
	}
	// block_num error : max	L7VS_PROTOMOD_MAX_BLOCK=10
	if (block_num > L7VS_PROTOMOD_MAX_BLOCK )
	{
		VANESSA_LOGGER_ERR(" The memory block is over max value. ");
		return 1;
	}

	// get iomux 
	l7vs_sync_iomux = l7vs_iomux_get_from_avail_list();
	if (!l7vs_sync_iomux) {
		VANESSA_LOGGER_ERR("can not get sync_iomux");
		return 1;
	}

	// mode error
	if (mode != SYNC_SEND && mode != SYNC_RECV)
	{
		l7vs_sync_iomux->status = iomux_sync_modeerr;
		VANESSA_LOGGER_ERR(" Unknown operation mode. ");
		return 1;
	}
	// fd error
	if (!(3 < fd && fd < (int)sizeof(int)+1))
	{
		VANESSA_LOGGER_ERR(" Unsuitable value of fd. ");
		return 1;
	}


	// set : struct l7vs_sync_iom sync_iom
	// number of memory block
	sync_iom.block_num = block_num;
	// memory area
	sync_iom.memory = l7vs_sync_get_memory(block_num);
	// memory get error
	if ( sync_iom.memory == NULL )
	{
		VANESSA_LOGGER_ERR("Could not allocate memory in l7vs_sync_init()");
		return 1;
	}
	// operation mode
	sync_iom.mode = mode;
	// initialize of turn number
	sync_iom.number = 0;

	// set : struct l7vs_iomux l7vs_sync_iomux
	// fd
	l7vs_sync_iomux->fd = fd;
	// callback
	l7vs_sync_iomux->callback = l7vs_sync_callback;
	// data
	l7vs_sync_iomux->data = &sync_iom;
	// status
	switch(mode)
	{
		case SYNC_SEND:
			// send mode
			l7vs_sync_iomux->status = iomux_sync_sending;
			break;
		case SYNC_RECV:
			// recv mode
			l7vs_sync_iomux->status = iomux_sync_receving;
			break;
	}

	// add : struct l7vs_iomux l7vs_sync_iomux
	l7vs_iomux_add(l7vs_sync_iomux, iom_read);

	return 0;
}

/*! Finalize sync
 *
 *  @return Finalize results. (0 , 1)
 *  @retval 0 Everything is ok.
 *  @retval 1 Error.
 */
int l7vs_sync_fini()
{
	int ret;

	// memory is null (initialize yet)
	if (sync_iom.memory == (void *)NULL)
	{
		VANESSA_LOGGER_ERR(" Initialization has not been done yet ");
		return 1;
	}

	// sync socket finalize
	ret = sync_sock_fini();
	if (ret != SYNC_OK)
	{
		VANESSA_LOGGER_ERR("sync_fini error");
		return 1;
	}

	// status
	// sync fini
	l7vs_sync_iomux->status = iomux_sync_destroy;
	// free : struct l7vs_iomux l7vs_iomux.data
	free(sync_iom.memory);

	// sync_iom initialize
	sync_iom.mode = SYNC_SEND;
	sync_iom.memory = NULL;
	sync_iom.block_num = 0;
	sync_iom.number = 0;

	// delete l7vs_iomux
	l7vs_iomux_remove(l7vs_sync_iomux);
	if (l7vs_sync_iomux->fd != -1) {
		close(l7vs_sync_iomux->fd);
	}
	l7vs_iomux_put_to_avail_list(l7vs_sync_iomux);
	l7vs_sync_iomux = NULL;

	return 0;
}

/*! Sync CallBack 
 *
 *  @return CallBack results.(0 , 1)
 *  @retval 0 Everything is ok.
 *  @retval 1 Error.
 */
int l7vs_sync_callback(struct l7vs_iomux *iom)
{
	// iom is NULL pointer
	if (iom == (void *)NULL)
	{
		VANESSA_LOGGER_ERR(" Argument is NULL pointer ");
		return 1;
	}

	// memory is null (initialize yet)
	if (sync_iom.memory == (void *)NULL)
	{
		VANESSA_LOGGER_ERR(" Initialization has not been done yet ");
		return 1;
	}

	// operation mode 
	switch(sync_iom.mode)
	{
		case SYNC_SEND:
			// sending
			l7vs_sync_send();
			break;

		case SYNC_RECV:
			// recving
			l7vs_sync_recv();
			break;

		default:
			// error
			VANESSA_LOGGER_ERR(" Unknown operation mode ");
			return 1;
	}
	return 0;
}

/*! Aquire the memory.
 *
 *  @return sync_iom Memory. (void *memory, NULL)
 *  @retval nonnull Memory.
 *  @retval NULL Error.
 */
void *l7vs_sync_get_memory(size_t block_num)
{
	void	*memory ;
	size_t	memory_size ;

	// Repeat check
	if (sync_iom.memory != NULL)
	{
		VANESSA_LOGGER_ERR(" malloc() has already been done. ");
		return NULL;
	}

	// Argument check
	// block_num error : 0
	if (block_num == 0)
	{
		VANESSA_LOGGER_ERR(" The memory block is 0.  ");
		return NULL;
	}
	// block_num error : max	L7VS_PROTOMOD_MAX_BLOCK=10
	if (block_num > L7VS_PROTOMOD_MAX_BLOCK)
	{
		VANESSA_LOGGER_ERR(" The memory block is over max value.  ");
		return NULL;
	}

	// size of memory
	memory_size = block_num * DATA_SIZE;


	// get memory area
	memory = malloc(memory_size);

	// malloc error
	if( memory == (void *)NULL )
	{
		VANESSA_LOGGER_ERR("Could not allocate memory");
		return NULL;
	}

	return memory;
}

/*! Return the memory size
 *
 *  @return Memory size (memory_size, NULL)
 *  @retval nonnull Memory size.
 *  @retval NULL Error.
 */
size_t l7vs_sync_memory_size()
{
	size_t memory_size ;

	// memory is null (initialize yet)
	if (sync_iom.memory == (void *)NULL)
	{
		VANESSA_LOGGER_ERR(" Initialization has not been done yet ");
		return 1;
	}

	// error
	// memory size is 0
	if (sync_iom.block_num == 0)
	{
		return 0;
	}
	// memory size is over
	if (sync_iom.block_num > L7VS_PROTOMOD_MAX_BLOCK)
	{
		VANESSA_LOGGER_ERR("Could not get memory over");
		return 1;
	}

        // size
	memory_size = sync_iom.block_num * DATA_SIZE ;

	return memory_size;
}

/*! Return the top of memory address.
 *
 *  @return Top of memory address.
 *  @retval sync_iom.memory Everything is ok.
 *  @retval NULL Error.
 */
void *l7vs_sync_memory_addr()
{
	if (sync_iom.memory == (void *)NULL)
	{
		VANESSA_LOGGER_ERR(" Initialization has not been done yet ");
	}
		
	return sync_iom.memory;
}

/*! Send sync data (1 block)
 *
 *  @result Send results.(0 , 1)
 *  @retval 0 Everything is ok.
 *  @retval 1 Error.
 */
int l7vs_sync_send()
{
	struct	sync_data send_data;
	void	*addr;
	enum sync_err_t	ret;

	// mode is different.
	if (sync_iom.mode != SYNC_SEND )
	{
		VANESSA_LOGGER_ERR("Mede is different.");
		return 1;
	}

	// memory is null (initialize yet)
	if (sync_iom.memory == (void *)NULL)
	{
		VANESSA_LOGGER_ERR("Initialization has not been done yet");
		return 1;
	}

	// block_num is 0
	if (sync_iom.block_num == 0)
	{
		VANESSA_LOGGER_ERR("Initialization has not been done yet");
		return 1;
	}

	// turn number is over
	if ( sync_iom.number > L7VS_PROTOMOD_MAX_BLOCK-1)
	{
		VANESSA_LOGGER_ERR("Turn number is over to max number");
		return 1;
	}

	// turn number of memory block
	send_data.type = (unsigned char)sync_iom.number;

	// top of sending memory
	addr = (char *)sync_iom.memory + (sync_iom.number * DATA_SIZE);

	// addr check
	if(!((char *)sync_iom.memory <= (char *)addr && (char *)addr <= (char *)sync_iom.memory + (sync_iom.block_num - 1) * DATA_SIZE))
	{
		VANESSA_LOGGER_ERR("Memory has exceeded get memory.");
		return 1;
	}

	// send data
	memcpy(&send_data.word, addr, DATA_SIZE);
	
	// size of send data
	send_data.size = sizeof(send_data.word);

	// send data
	ret = sync_send_to(&send_data);
	// error
	if ( ret != SYNC_OK )
	{
		VANESSA_LOGGER_ERR("sync_send failed");
		return 1;
	}

	// next turn number
	if ( sync_iom.number < sync_iom.block_num - 1)
	{
		++sync_iom.number;
	}
	else if ( sync_iom.number == sync_iom.block_num - 1 )
	{
		sync_iom.number = 0 ;
	}
	else
	{
		VANESSA_LOGGER_ERR("memory number is inadequate");
		return 1;
	}
	return 0;

}

/*! Receive sync data (1 block)
 *
 *  @return Recive results.(0 , 1)
 *  @retval 0 Everything is ok.
 *  @retval 1 Error.
 */

int l7vs_sync_recv()
{
	struct	sync_data recv_data;
	void	*addr;
	enum sync_err_t	ret;

	// mode is different.
	if (sync_iom.mode != SYNC_RECV )
	{
		VANESSA_LOGGER_ERR("Mede is different.");
		return 1;
	}

	// memory is null (initialize yet)
	if (sync_iom.memory == (void *)NULL)
	{
		VANESSA_LOGGER_ERR("Initialization has not been done yet");
		return 1;
	}

	// receive data
	ret = sync_recv_from(&recv_data);
	// error
	if (ret != SYNC_OK )
	{
		VANESSA_LOGGER_ERR("sync_recv_from failed");
		return 1;
	}

	// number limits check
	if (recv_data.type > sync_iom.block_num-1)
	{
		VANESSA_LOGGER_ERR("memory number is inadequate");
		return 1;
	}

	// top of received memory
	addr = (char *)sync_iom.memory + (recv_data.type * DATA_SIZE);

	// addr check
	if(!((char *)sync_iom.memory <= (char *)addr && (char *)addr <= (char *)sync_iom.memory + (sync_iom.block_num - 1) * DATA_SIZE))
	{
		VANESSA_LOGGER_ERR("Memory has exceeded get memory.");
		return 1;
	}

	// write data to iomux
	memcpy(addr, recv_data.word, DATA_SIZE);

	return 0;
}
