/*
 *  The MIT License
 *
 *  Copyright 2011-2012 Masahiko, SAWAI <masahiko.sawai@gmail.com>.
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a copy
 *  of this software and associated documentation files (the "Software"), to deal
 *  in the Software without restriction, including without limitation the rights
 *  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 *  copies of the Software, and to permit persons to whom the Software is
 *  furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in
 *  all copies or substantial portions of the Software.
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 *  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 *  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 *  THE SOFTWARE.
 */
package org.routine_work.simple_battery_logger;

import android.app.ListActivity;
import android.content.ActivityNotFoundException;
import android.content.Intent;
import android.net.Uri;
import android.os.Bundle;
import android.os.Environment;
import android.view.ContextMenu;
import android.view.ContextMenu.ContextMenuInfo;
import android.view.Menu;
import android.view.MenuInflater;
import android.view.MenuItem;
import android.view.View;
import android.widget.AdapterView;
import android.widget.AdapterView.AdapterContextMenuInfo;
import android.widget.AdapterView.OnItemClickListener;
import android.widget.ArrayAdapter;
import android.widget.ListView;
import android.widget.TextView;
import android.widget.Toast;
import java.io.File;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import org.routine_work.simple_battery_logger.db.DBConstants;
import org.routine_work.simple_battery_logger.util.CsvFilenameFilter;
import org.routine_work.utils.Log;
import org.routine_work.simple_battery_logger.util.PreferenceUtils;
import org.routine_work.simple_battery_logger.util.SystemResourceUtils;

public class CsvFileListActivity extends ListActivity
	implements OnItemClickListener, DBConstants
{

	private static final String LOG_TAG = "simple-battery-logger";
	private List<String> csvFilenameList;
	ArrayAdapter<String> listAdapter;

	/**
	 *
	 * @param savedInstanceState
	 */
	@Override
	protected void onCreate(Bundle savedInstanceState)
	{
		super.onCreate(savedInstanceState);
		setContentView(R.layout.simple_list_activity);

		ListView listView = (ListView) findViewById(android.R.id.list);
		listView.setOnItemClickListener(this);
		registerForContextMenu(listView);

		initializeListData();

	}

	@Override
	protected void onDestroy()
	{
		Log.v(LOG_TAG, "Hello");
		super.onDestroy();
		Log.v(LOG_TAG, "Bye");
	}

	@Override
	protected void onResume()
	{
		Log.v(LOG_TAG, "Hello");

		super.onResume();

		Log.v(LOG_TAG, "Bye");
	}

	@Override
	protected void onPause()
	{
		Log.v(LOG_TAG, "Hello");

		super.onPause();

		Log.v(LOG_TAG, "Bye");
	}

	@Override
	public void onItemClick(AdapterView<?> parentView, View itemView, int position, long id)
	{
		if (itemView instanceof TextView)
		{
			TextView dateTextView = (TextView) itemView;
			String csvFilename = dateTextView.getText().toString();
			Log.d(LOG_TAG, "clicked csvFilename => " + csvFilename);

			File exportDir = getExportDir();
			File csvFile = new File(exportDir, csvFilename);
			Uri csvDataUri = Uri.fromFile(csvFile);
			Log.d(LOG_TAG, "csv dataUri => " + csvDataUri);

			Intent intent = new Intent(this, CsvFileViewerActivity.class);
			intent.setData(csvDataUri);
			startActivity(intent);
		}
	}

	@Override
	public boolean onCreateOptionsMenu(Menu menu)
	{
		Log.v(LOG_TAG, "Hello");

		MenuInflater menuInflater = getMenuInflater();
		menuInflater.inflate(R.menu.preference_menu, menu);
		menuInflater.inflate(R.menu.quit_menu, menu);

		Log.v(LOG_TAG, "Bye");
		return true;
	}

	@Override
	public boolean onOptionsItemSelected(MenuItem item)
	{
		boolean result = true;
		Log.v(LOG_TAG, "Hello");

		int itemId = item.getItemId();
		switch (itemId)
		{
			case R.id.preference_menuitem:
				Intent prefIntent = new Intent(this, BatteryLoggerPreferenceActivity.class);
				startActivity(prefIntent);
				break;
			case R.id.quit_menuitem:
				Log.d(LOG_TAG, "quit_menuitem is selected.");
				DashboardActivity.quitApplication(this);
				break;
			default:
				if (itemId == SystemResourceUtils.getResourceId("android.R$id.home"))
				{
					Log.d(LOG_TAG, "home is clicked.");
					DashboardActivity.goDashboardActivity(this);
				}
				else
				{
					result = super.onOptionsItemSelected(item);
				}
				break;
		}

		Log.v(LOG_TAG, "Bye");
		return result;
	}

	@Override
	public void onCreateContextMenu(ContextMenu menu, View view, ContextMenuInfo menuInfo)
	{
		Log.v(LOG_TAG, "Hello");

		super.onCreateContextMenu(menu, view, menuInfo);
		MenuInflater menuInflater = getMenuInflater();
		menuInflater.inflate(R.menu.csv_files_menu, menu);

		Log.v(LOG_TAG, "Bye");
	}

	@Override
	public boolean onContextItemSelected(MenuItem item)
	{
		boolean result = true;
		Log.v(LOG_TAG, "Hello");

		AdapterContextMenuInfo menuInfo = (AdapterContextMenuInfo) item.getMenuInfo();
		Log.d(LOG_TAG, "menuInfo.position => " + menuInfo.position);
		Log.d(LOG_TAG, "menuInfo.id => " + menuInfo.id);
		Log.d(LOG_TAG, "menuInfo.targetView => " + menuInfo.targetView);

		int itemId = item.getItemId();
		switch (itemId)
		{
			case R.id.send_menuitem:
				if (menuInfo.targetView instanceof TextView)
				{
					TextView dateTextView = (TextView) menuInfo.targetView;
					String csvFilename = dateTextView.getText().toString();
					Log.d(LOG_TAG, "send csvFilename => " + csvFilename);
					sendCsvFile(csvFilename);
				}
				break;
			default:
				result = super.onContextItemSelected(item);
		}

		Log.v(LOG_TAG, "Bye");
		return result;
	}

	private void initializeListData()
	{
		Log.v(LOG_TAG, "Hello");

		csvFilenameList = new ArrayList<String>();
		listAdapter = new ArrayAdapter<String>(this, android.R.layout.simple_list_item_1, csvFilenameList);
		setListAdapter(listAdapter);
		updateListData();

		Log.v(LOG_TAG, "Bye");
	}

	private void updateListData()
	{
		Log.v(LOG_TAG, "Hello");

		csvFilenameList.clear();

		String externalStorageState = Environment.getExternalStorageState();
		Log.i(LOG_TAG, "externalStorageState => " + externalStorageState);
		if (externalStorageState.equalsIgnoreCase(Environment.MEDIA_MOUNTED))
		{
			File exportDir = getExportDir();
			Log.d(LOG_TAG, "exportDir => " + exportDir.getAbsolutePath());

			if (exportDir.exists())
			{
				String[] csvFilenames = exportDir.list(new CsvFilenameFilter());
				for (String csvFilename : csvFilenames)
				{
					Log.v(LOG_TAG, "csvFile => " + csvFilename);
					csvFilenameList.add(csvFilename);
				}

				Collections.sort(csvFilenameList);
				Collections.reverse(csvFilenameList);
				listAdapter.notifyDataSetChanged();
			}
		}

		Log.v(LOG_TAG, " Bye");
	}

	private File getExportDir()
	{
		String csvExportDirectoryName = PreferenceUtils.getCsvExportDirectoryName(this);
		Log.d(LOG_TAG, "csvExportDirectoryName => " + csvExportDirectoryName);

		File externalStorageDirectory = Environment.getExternalStorageDirectory();
		File exportDir = new File(externalStorageDirectory, csvExportDirectoryName);
		return exportDir;
	}

	private void sendCsvFile(String csvFilename)
	{
		File exportDir = getExportDir();
		File csvFile = new File(exportDir, csvFilename);
		Uri csvDataUri = Uri.fromFile(csvFile);
		Log.d(LOG_TAG, "csv dataUri => " + csvDataUri);

		try
		{
			Intent intent = new Intent(Intent.ACTION_SEND);
			intent.setType("text/csv");
			intent.putExtra(Intent.EXTRA_STREAM, csvDataUri);
			startActivity(intent);
		}
		catch (ActivityNotFoundException e)
		{
			Log.e(LOG_TAG, "Failed to send a CSV file.", e);

			String message = getString(R.string.activity_not_found) + "\n" + csvDataUri;
			Toast.makeText(this, message, Toast.LENGTH_SHORT).show();
		}
	}
}
