﻿using System;
using System.ComponentModel;
using System.Diagnostics;
using System.IO;
using System.Runtime.InteropServices;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Input;
using System.Windows.Interop;
using Forms = System.Windows.Forms;
using System.Collections.Generic;

namespace JoinNotes
{
    /// <summary>
    /// Config.xaml の相互作用ロジック
    /// </summary>
    public partial class Config : Window
    {
        private static Dictionary<string, string> startupWindowClassNameComboSource = new Dictionary<string, string> { };

        public Config()
        {
            InitializeComponent();
            this.Loaded += new RoutedEventHandler(Config_Loaded);
            this.Closed += new EventHandler(Config_Closed);

            {
                //FIXME: タイプイニシャライザーで定義
                startupWindowClassNameComboSource["編集ウィンドウ"] = typeof(Editor).Name;
                startupWindowClassNameComboSource["全文検索ウィンドウ"] = typeof(Search).Name;
                startupWindowClassNameComboSource["設定ウィンドウ"] = typeof(Config).Name;
                this.startupWindowClassNameCombo.ItemsSource = startupWindowClassNameComboSource;
                this.startupWindowClassNameCombo.DisplayMemberPath = "Key";
                this.startupWindowClassNameCombo.SelectedValuePath = "Value";
            }
        }

        void Config_Closed(object sender, EventArgs e)
        {
            Application.configWindow = null;
        }

        void Config_Loaded(object sender, RoutedEventArgs e)
        {
            this.activateHotkeyText.Text = Properties.Settings.Default.ActivateHotkey.ToString();
            this.openWindowHotKeyText.Text = Properties.Settings.Default.OpenWindowHotKey.ToString();
            this.StartPageNameText.Text = Properties.Settings.Default.StartPageName;
            this.documentPathBox.Text = Properties.Settings.Default.DocumentPath;

            this.startupWindowClassNameCombo.SelectedValue = Properties.Settings.Default.StartupWindowClassName;

            this.search_RecentWordsCountBox.Text = Properties.Settings.Default.Search_RecentWordsCount.ToString();
        }

        void closeButton_Click(object sender, RoutedEventArgs e)
        {
            this.Close();
        }

        void hotKeyBox_KeyDown(object sender, KeyEventArgs e)
        {
            var c = (TextBox)sender;

            string hotKeyName;
            if (ReferenceEquals(c, this.activateHotkeyText))
                hotKeyName = "ActivateHotKey";
            else if (ReferenceEquals(c, this.openWindowHotKeyText))
                hotKeyName = "OpenWindowHotKey";
            else
                hotKeyName = null;

            try
            {
                Debug.Assert(hotKeyName != null, "hotKeyName");

                var hotKey = new HotKey(new WindowInteropHelper(Application.mainWindow).Handle,
                    Application.HotKeyId[hotKeyName],
                    Util.InputKeyToFormsKey(e.Key),
                    Util.InputModifierKeyToFormsKey(e.KeyboardDevice.Modifiers));
                Application.HotKey[hotKeyName] = hotKey;

                Properties.Settings.Default[hotKeyName] = hotKey.Keys;
                Properties.Settings.Default.Save();

                //FIXME: c.Text = hotKey.Keys.ToString() にしたい。
                c.Text = e.KeyboardDevice.Modifiers.ToString() + ", " + e.Key.ToString();

                Debug.WriteLine(c.Name + ": " + c.Text, "HotKeyBox_KeyDown");
            }
            catch (Win32Exception ex)
            {
                Debug.WriteLine(ex.Message, ex.Source);
                c.Text = "";
            }
        }

        void initializeButton_Click(object sender, RoutedEventArgs e)
        {
            //FIXME: 対象範囲だけを初期化
            Properties.Settings.Default.Properties.Clear();
        }

        void documentPathBrowseButton_Click(object sender, RoutedEventArgs e)
        {
            ShowDocumentPathBrowseDialog();
        }

        void documentPathBox_LostFocus(object sender, RoutedEventArgs e)
        {
            try
            {
                Properties.Settings.Default.DocumentPath = new DirectoryInfo(this.documentPathBox.Text).FullName;
                Properties.Settings.Default.Save();
                // 次回起動時に反映させるので、ここでは設定を書き換えるだけ
                //ApplyDocumentPath();
            }
            catch (Exception ex)
            {
                Debug.WriteLine(ex.Message, ex.Source);
            }
        }

        void startupWindowClassNameCombo_SelectionChanged(object sender, System.Windows.Controls.SelectionChangedEventArgs e)
        {
            var c = (ComboBox)sender;

            if (startupWindowClassNameComboSource.ContainsValue(c.SelectedValue as string))
                Properties.Settings.Default.StartupWindowClassName = c.SelectedValue as string;

            Properties.Settings.Default.Save();
        }

        void startPageBox_TextChanged(object sender, TextChangedEventArgs e)
        {
            var c = (TextBox)sender;

            Properties.Settings.Default.StartPageName = c.Text;
            Properties.Settings.Default.Save();
        }

        // 次回起動時に反映させるので、使わない。
        //private void ApplyDocumentPath()
        //{
        //    try
        //    {
        //        Application.DocumentPath = new DirectoryInfo(this.DocumentPathBox.Text);
        //    }
        //    catch (Exception ex)
        //    {
        //        Application.DocumentPath = Application.DocumentPathDefault;
        //        //MessageBox.Show(ex.Message);
        //    }
        //    finally
        //    {
        //        Properties.Settings.Default.DocumentPath = Application.DocumentPath.FullName;
        //    }
        //}

        void ShowDocumentPathBrowseDialog()
        {
            var dialog = new Forms.FolderBrowserDialog();
            dialog.RootFolder = Environment.SpecialFolder.Desktop;
            dialog.SelectedPath = this.documentPathBox.Text;
            dialog.ShowNewFolderButton = true;
            dialog.Description = "ドキュメントフォルダ―を指定してください。";
            var result = dialog.ShowDialog();

            if (result == Forms.DialogResult.OK)
            {
                this.documentPathBox.Text = dialog.SelectedPath;
                this.documentPathBox.Focus();
            }
        }

        private void search_HistoryItemsCountBox_TextChanged(object sender, TextChangedEventArgs e)
        {
            var c = (TextBox)sender;

            int i;
            if (int.TryParse(c.Text, out i))
            {
                Properties.Settings.Default.Search_RecentWordsCount = i;
                Properties.Settings.Default.Save();
            }
        }

        private void search_ClearRecentWords_Click(object sender, RoutedEventArgs e)
        {
            Properties.Settings.Default.Search_RecentWords = string.Empty;
            Properties.Settings.Default.Save();
        }
    }

    /// <summary>ホットキーの登録・解除を行うためのクラス</summary>
    public class HotKey
    {
        public IntPtr LParam { get { return lParam; } }
        public Forms.Keys Keys { get { return keys; } }

        IntPtr hWnd; // ホットキーの入力メッセージを受信するウィンドウのhWnd
        readonly int id; // ホットキーのID(0x0000〜0xBFFF)
        readonly IntPtr lParam; // WndProcメソッドで押下されたホットキーを識別するためのlParam値
        readonly Forms.Keys keys;

        public HotKey(IntPtr hWnd, int id, Forms.Keys keys)
        {
            this.hWnd = hWnd;
            this.id = id;
            this.keys = keys;

            var keycode = (int)(keys & Forms.Keys.KeyCode);
            var modifierscode = (int)(keys & Forms.Keys.Modifiers) >> 16;

            this.lParam = new IntPtr(keycode << 16 | modifierscode);

            Register(hWnd, id, modifierscode, keycode);
        }

        public HotKey(IntPtr hWnd, int id, Forms.Keys keys, Forms.Keys modifiers)
        {
            this.hWnd = hWnd;
            this.id = id;
            this.keys = keys | modifiers;

            var keycode = (int)(keys & Forms.Keys.KeyCode);
            var modifierscode = (int)(modifiers & Forms.Keys.Modifiers) >> 16;

            this.lParam = new IntPtr(keycode << 16 | modifierscode);

            Register(hWnd, id, modifierscode, keycode);
        }

        internal void Register(IntPtr hWnd, int id, int modifierscode, int keycode)
        {
            if (RegisterHotKey(hWnd, id, modifierscode, keycode) == 0)
                // ホットキーの登録に失敗
                throw new Win32Exception(Marshal.GetLastWin32Error());
        }

        internal void Unregister()
        {
            if (hWnd == IntPtr.Zero)
                return;

            if (UnregisterHotKey(hWnd, id) == 0)
                // ホットキーの解除に失敗
                throw new Win32Exception(Marshal.GetLastWin32Error());

            hWnd = IntPtr.Zero;
        }

        [DllImport("user32", SetLastError = true)]
        private static extern int RegisterHotKey(IntPtr hWnd, int id, int fsModifier, int vk);

        [DllImport("user32", SetLastError = true)]
        private static extern int UnregisterHotKey(IntPtr hWnd, int id);
    }
}
