;;; mf-lib-mp4.el -- This library for mf-tag-write.el -*- coding: utf-8-emacs -*-
;; Copyright (C) 2018, 2919 fubuki

;; Author: fubuki@frill.org
;; Version: $Revision: 1.1 $
;; Keywords: multimedia

;; This program is free software: you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation, either version 3 of the License.

;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs.  If not, see <http://www.gnu.org/licenses/>.

;;; Commentary:

;; This is the standard m4a, mp4 read/write module for mf-tag-write.el.

;; Add a list of '(regexp read-function write-function argument-conv-function conv-alias)
;; to variable `mf-function-list'.

;;; Installation:

;; (require 'mf-tag-write)

;;; Change Log:

;;; Code:

(defconst mf-lib-mp4-version     "1.0")
(defconst mf-lib-mp4-rcs-version "$Revision: 1.1 $")

(defcustom mf-no-one-patch nil
  "NON-NIL なら A30 前夜の Walkman を騙すパッチを充てない."
  :type  'boolean
  :group 'music-file)

(defcustom mf-no-mc-delete nil
  "NON-NIL なら MusicCenter が作る互換用のタグブロックを削除しない."
  :type  'boolean
  :group 'music-file)

(defvar mf-mp4-function-list
  '("\\.\\(m4a\\|mp4\\)\\'"
    mf-m4a-tag-read
    mf-mp4-write-buffer
    mf-list-convert
    mf-mp4-tag-alias))

(unless (boundp 'mf-function-list)
  (setq mf-function-list nil))
(add-to-list 'mf-function-list  mf-mp4-function-list)

(defvar mf-mp4-write-hook nil)

(defcustom mf-mp4-tag-alias
  '((title . "\251nam") (artist . "\251ART") (a-artist . "aART") (album . "\251alb") (date . "\251day") (year . "\251day") (genre . "\251gen") (track . "trkn") (disk . "disk") (writer . "\251wrt") (cover . "covr") (artwork . "covr") (lyric . "\251lyr") (s-album . "soal") (s-title . "sonm") (s-artist . "soar") (s-a-artist . "soaa") (copy . "cprt") (mpb . "iTunSMPB") (cpil . "cpil") (pgap . "pgap") (tempo .  "tmpo") (too . "\251too") (enc .  "Encoding Params") (norm . "iTunNORM") (cddb . "iTunes_CDDB_IDs") (ufid . "UFIDhttp://www.cddb.com/id3/taginfo1.html"))
  "mp4/m4a tag alias."
  :type  '(repeat (cons symbol string))
  :group 'music-file)

;; *** mp4 header 解析関連
(defconst mp4-container-alist
  '(("moov" . 8) ("trak" . 8) ("mdia" . 8)  ("minf" . 8)
    ("dinf" . 8) ("stbl" . 8) ("stsd" . 16) ("mp4a" . 36)
    ("udta" . 8) ("meta" . 12) ("ilst" . 8)
    ("\251nam" . 8) ("\251ART" . 8) ("\251alb" . 8) ("aART" . 8)
    ("\251gen" . 8) ("gnre"    . 8) ("\251day" . 8) ("trkn" . 8)
    ("\251wrt" . 8) ("disk"    . 8) ("sonm"    . 8) ("soar" . 8)
    ("soal"    . 8) ("soaa"    . 8) ("covr"    . 8) ("----" . 8)
    ("\251lyr" . 8) ("\251cmt" . 8)
    ("cpil" . 8) ("pgap" . 8) ("tmpo" . 8) ("\251too" . 8) ("cprt" . 8))
  "*コンテナ(子持ちボックスをコンテナと言う)の最初のボックスまでのオフセット.")

;; 予想で充てたシンボルであり公式ではない.
(defconst mf-ilst-data-type
  '((0  . binary) (1  . string) (13 . jpeg) (14 . png) (21 . number)) "ilst data type.")

;; :tag "----" :mean "com.apple.iTunes"
(defconst mf-itunes-tags '("iTunSMPB" "Encoding Params" "iTunNORM" "iTunes_CDDB_IDs"
                           "UFIDhttp://www.cddb.com/id3/taginfo1.html"))

(defvar mp4-cover-margin  24  "jpag の素のサイズに加えるアトムとコンテナの余長.")
(defvar mf-mp4-sort-order nil "tag の Sort order を保持しておく変数.")
(make-variable-buffer-local 'mf-mp4-sort-order)

(defun octput (val)
  "数値を 8進数にして表示."
  (format "%o" val))

(defun hexput (val)
  "数値の 16進表記を得る."
  (interactive "NVal: ")
  (format "%x" val))

(defun mp4-tag-type (tag)
  "TAG の ilst 管理番号を返す.
今調べるのが面倒なので `mf-list-convert' から 数値タグ等を指定しない前提の簡易版."
  (cond
   ((member tag '("disk" "trkn"))
    (car (rassq 'binary mf-ilst-data-type)))
   (t
    (car (rassq 'string mf-ilst-data-type)))))

(defvar mp4-container-p '("gnre" "cprt") "入れ子環境まで精査すべき重複利用シンボル.")

(defun mp4-container-p (key env)
  "KEY のタイプがコンテナなら最初のボックスまでのオフセット値を整数で返す.
さもなくば NIL.
ENV には親のタイプを指定する."
  (let ((r (assoc key mp4-container-alist)))
    (if (member key mp4-container-p)
        (and (string-equal env "ilst") r (cdr r))
      (and r (cdr r)))))

(defun mf-buffer-substring (start end)
  (ignore-errors (buffer-substring start end)))

(defun mf-mp4-tag-collect (&optional length env)
  "current buffer に読み込まれた mp4 file の atom tree list を返す.
そのとき point は atom 先頭になくてはならない.
LENGTH はスキャンする大きさ(atom の size), ENV は親の atom TYPE.
これらの引数はすべて再帰するとき自分自身に情報を渡す為のダミー."
  (let ((length (or length (point-max)))
        result)
    (while (and (< 0 length) (not (eobp)))
      (let* ((beg       (point))
             (size      (mf-buffer-read-long-word beg))
             (type      (mf-buffer-substring (+ beg 4) (+ beg 8)))
             (container (mp4-container-p type env)))
        (cond
         ((and size container)
          (forward-char container)
          (setq result
                (cons
                 (cons (list type beg size)
                       (mf-mp4-tag-collect (- size container) type))
                 result))
          (setq length (- length size)))
         (size
          (setq result (cons (list type beg size) result))
          (setq length (- length size))
          (forward-char size))
         (t
          (setq length 0)))))
    (reverse result)))

(defun mf-buffer-substring-margin (start length &optional margin)
  "START から LENGTH 分のバッファの内容を返す. MARGIN があれば START に追加される."
  (let ((margin (or margin 0)))
    (buffer-substring (+ start margin) (+ start length))))

(defun mf-ilst-string-p (c)
    (= c (car (rassq 'string mf-ilst-data-type))))

(defun mf-ilst-image-p (c)
  (or (= c (car (rassq 'jpeg mf-ilst-data-type)))
      (= c (car (rassq 'png  mf-ilst-data-type)))))

(defun mf-ilst-binary-p (c)
  (= c (car (rassq 'binary mf-ilst-data-type))))

(defun mf-string-to-number (str tag)
  " disk trakn の引数で \"1/1\" 等と書かれた文字列をバイナリ形式にパックして返す."
  (let* ((r      (split-string str "/" t))
         (first  (string-to-number (or (car r) "0")))
         (second (string-to-number (or (cadr r) "0"))))
    (if (string-equal tag "disk")
        (format "\0\0\0%c\0%c" first second)
      (format "\0\0\0%c\0%c\0\0" first second))))

(defun mf-mp4-tag-analyze (ilst &optional no-binary)
  "MP4-ATOMS の iTunes のタグやアドレス情報を展開し plist にして返す.
MP4-ATOMS の存在するバッファがカレントでなくてならない."
  (let (frame tag mean dsc str type result)
    (setq mf-mp4-sort-order nil)
    (dolist (frame (cdr ilst))
      (cond
       ((atom (car frame))
        (setq tag (car frame)
              str (buffer-substring (+ (cadr frame) 16)
                                    (+ (cadr frame) (caddr frame))))
        (setq mf-mp4-sort-order (cons (cons tag (cadr frame)) mf-mp4-sort-order))
        (setq result (cons (list :tag tag :data str) result)))
       ((and (consp (car frame)) (string-equal (caar frame) "----"))
        (let ((first  (mf-first frame))
              (second (mf-second frame))
              (third  (mf-third frame))
              (fourth (mf-fourth frame)))
          (setq tag  (mf-first first))
          (setq mean (mf-buffer-substring-margin (cadr second) (caddr second) 12))
          (setq dsc  (mf-buffer-substring-margin (cadr third) (caddr third) 12))   ; name
          (setq str  (mf-buffer-substring-margin (cadr fourth) (caddr fourth) 16)) ; data
          (setq type (string-to-char
                      (buffer-substring (+ (cadr fourth) 11) (+ (cadr fourth) 12))))
          (setq mf-mp4-sort-order (cons (cons dsc (cadr first)) mf-mp4-sort-order))
          (setq result (cons (list :tag tag :mean mean :dsc dsc :type type :data str) result))))
       ((and (consp (car frame)) (member (caar frame) '("trkn" "disk")))
        (let ((first  (mf-first frame))
              (second (mf-second frame))
              tmp r)
          (setq tag  (car first))
          (setq type (string-to-char
                      (buffer-substring (+ (cadr second) 11) (+ (cadr second) 12))))
          (setq tmp  (mf-buffer-substring-margin (cadr second) (caddr second) 16))
          (setq str
                (mapconcat 'number-to-string
                           (dolist (a (split-string tmp "[\0]+" t) (reverse r))
                             (setq r (cons (string-to-char a) r)))
                           "/"))
          (setq mf-mp4-sort-order (cons (cons tag (cadr first)) mf-mp4-sort-order))
          (setq result (cons (list :tag tag :type type :data str) result))))
       (t
        (let ((first  (mf-first frame))
              (second (mf-second frame)))
          (setq tag  (car first))
          (setq type (string-to-char
                      (buffer-substring (+ (cadr second) 11) (+ (cadr second) 12))))
          (setq str
                (cond
                 ((mf-ilst-string-p type)
                  (mf-chop
                   (decode-coding-string
                    (mf-buffer-substring-margin (cadr second) (caddr second) 16)
                    'utf-8)))
                 ((and no-binary (mf-ilst-image-p type))
                  nil)
                 (t
                  (mf-buffer-substring-margin (cadr second) (caddr second) 16))))
          (setq mf-mp4-sort-order (cons (cons tag (cadr first)) mf-mp4-sort-order))
          (setq result (cons (list :tag tag :type type :data str) result))))))
    (reverse result)))

(defun mf-get-ilst-1 (list)
  "完全な mp4 atom list から \"ilst\" のパートを得る.
`mp4-get-container' を忘れていて作った完全独立ヴァージョン. 結果は `mf-get-ilst' と同じ."
  (let (result)
    (catch 'break
      (while list
        (cond
         ((and (consp (car list)) (consp (caar list)))
          (setq result (mf-get-ilst (car list)))
          (if result (throw 'break result)))
         ((and (consp (car list)) (atom (caar list)) (string-equal (caar list) "ilst"))
          (throw 'break (setq result list))))
        (setq list (cdr list))))
    result))

(defun mf-get-ilst (list)
  "完全な mp4 atom LIST から \"ilst\" のパートを得る."
  (car (last (car (mp4-get-container "ilst" list)))))

;; mp4 & m4a
(defun mf-mp4-write-buffer (tags &optional no-backup no-one-patch no-mc-delete)
  "カレントバッファに読み込まれている mp4(m4a)バイナリのタグを TAG に差し替える.
NO-BACKUP が 非NIL なら元ファイイルを残さない.
NO-ONE-PATCH が NON-NIL なら meta 直下の titl を Titl に変更するパッチをしない.
NO-MC-DELETE が NON-NIL なら MusicCenter で作られた mp4 の 3つの重複アートワークを取り除かない."
  (let ((ilst-pack (mf-pack-mp4 tags))
        (file mf-current-file)
        (no-one-patch (or no-one-patch mf-no-one-patch))
        (no-mc-delete (or no-mc-delete mf-no-mc-delete))
        atoms depend ilst ilst-point offset meta mc-flag delete-list)

    (run-hooks 'mf-mp4-write-hook)
    
    (goto-char (point-min))

      ;;;; 必要な data 収集パート Collected atoms.
      (setq atoms  (mf-mp4-tag-collect)) ; 堅牢にする為更めてこのファイルから得る.
      ;; "ilst" を含めた "ilst" が依存するコンテナが集まる.
      (setq depend (car (mp4-get-container "ilst" atoms))
            ilst   (car (mp4-get-list "ilst" atoms)))

      ;; Set MusicCenter flag.
      ;; "mp42" 且つ "uuid" を持ち "meta" をふたつ持っていれば MusciCenter のデータ.
      (setq mc-flag (and (not no-mc-delete)
                         (string-equal (mf-mp4-get-type atoms) "mp42")
                         (mp4-get-list "uuid" atoms)
                         (= (length (mp4-get-list "meta" atoms)) 2)))

      ;; 削除する ilst と新しい ilst との大きさの差を offset にセット.
      (setq ilst-point (cadr ilst))
      (setq offset (- (length ilst-pack) (caddr ilst)))

      ;; ふたつある(なら) meta のうち削除する後方の方の情報を得る.
      (setq meta (if mc-flag (car (sort (mp4-get-list "meta" atoms) #'atom-point-more)) nil))

      ;; 後ほど削除するコンテナの降順リストを作る.
      (setq delete-list
            (sort
             (append
              (if mc-flag
                  (append (mp4-get-list "uuid" atoms)
                          (list meta)))
              (mp4-get-list "ilst" depend))
             #'atom-point-more))

      ;;;; 書き換えパート
      ;; Walkman に "ilst" の方を参照させるためのトリック.
      (if (and (mp4-get-list "titl" atoms) (or (not no-one-patch) mc-flag))
          (save-excursion (mf-m4a-one-patch atoms)))

      ;; 変更した "ilst" サイズに影響するバッファ上のアトムのサイズ情報にそのオフセットを加える.
      (dolist (d (butlast depend))
        (mf-point-add-long-word (cadr d) (- offset (mp4-meta-include (car d) meta atoms))))
    
      ;; *** ここで uuid , meta(ID32) と ilst を delete
      ;; (バッファ内のアトムの物理ポイントが変わる)
      ;; 末尾から削除していかないと整合性が取れなくなるので
      ;; `delete-list' は降順ソートされていなければならない.
      (dolist (atom delete-list) 
        (let* ((beg  (cadr atom))
               (end  (+ beg (caddr atom))))
          (delete-region beg end)))

      ;; 跡地に新 ilst の挿入.
      (goto-char ilst-point)
      (insert ilst-pack)
      
      ;; (ilst より後方にある)mdat の位置が変わったので
      ;; パケットテーブル(stco)の値にオフセットをかける.
      (packet-table-update
       (car (mp4-get-list "stco" atoms)) (if mc-flag (- offset (caddr meta)) offset))

      (if (and (not (stringp no-backup)) (null no-backup))
        (let ((name (make-backup-file-name file)))
          (if (file-exists-p name) (delete-file name 'trash))
        (rename-file file name)))
      (write-region (point-min) (point-max) file)))

(defun atom-point-more (a b)
  (> (cadr a) (cadr b)))

(defun mf-insert-long-word (value)
  "ポイントの後に VALUE を 4 bytes にしてバッファに書き込む. ポイントは書いた分進む."
  (insert (mf-long-word value)))

(defun mf-point-add-long-word (pos add)
  "POS から 4バイトを整数にし ADD を追加し書き戻す. 書いた分ポイントは前進する."
  (let (org)
    (goto-char pos)
    (setq org (mf-buffer-read-long-word))
    (delete-char 4)
    (mf-insert-long-word (+ org add)))) ;; この計算が正しいか未検証!! ***

;; #3 マッチしたリストを昇順のリストにまとめて返す
(defun mp4-get-container (type list &optional depend)
  "TYPE に依存したアトムすべてを LIST から新たなリストにして返す.
ここでのアトムとは lisp の atom ではなく MP4のアトム(入れ子のコンテナ)である.
DEPEND は子に渡すワーク用ダミーでユーザが指定することはない.

結果から目的のコンテナだけを得るには以下のようにして取り出す必要がある.

  (car (last (car (mp4-get-container \"udta\" foo))))

`mp4-get-container' は \"udta\" とこのコンテナを含んでいる親のコンテナを
ひとつであってもリストとして返すので, まず car 等で外側の括弧を取り外す必要がある.
取り出したリストは依存コンテナも含んでいるので,
そこから更に nth や last 等で目的のコンテナを取り出す.
この例では戻り値が list である  `last' で取り出しているので更に car している."
  (let (result ret)
    (dolist (lst list result)
      (when (and (consp lst) (consp (car lst)))
        (if (and (atom (caar lst)) (string-equal (caar lst) type))
            (setq result (reverse (cons (reverse (cons lst depend)) result)))
          (setq ret (mp4-get-container type (cdr lst) (cons (car lst) depend)))
          (if ret (setq result (append ret result)) nil))))))

;; #3 昇順で返す
(defun mp4-get-list (type list)
  "TYPE にマッチしたアトムを LIST 内から再帰的に探して list ですべて返す."
  (let (result)
    (dolist (lst list (reverse result))
      (if (and (consp lst) (consp (car lst)))
          (setq result (append (mp4-get-list type lst) result))
        (if (and (consp lst) (atom (car lst)))
            (if (string-equal type (car lst))
                (setq result (cons lst result))))))))

;;
;; レコチョクの m4a を Walkman で正常に扱えるようにするためのインチキパッチ.
;;;###autoload
(defun dired-do-m4a-one-patch (&optional prefix)
  "dired からマークしたファイルに one-path を実行する. PREFIX があればリバースパッチになる."
  (interactive "P")
  (let ((files (dired-get-marked-files)))
    (dolist (f files)
      (unless (m4a-one-patch f prefix) (message "Error: %s." f)))
    (revert-buffer)))

;;;###autoload
(defun m4a-one-patch (file &optional reverse)
  "`mf-m4a-one-patch' を FILE指定して単独実行するための関数.
prefix 起動すると REVERSE がオンになり逆パッチになる."
  (interactive "fm4a File: \nP")
  (with-temp-buffer
    (insert-file-contents-literally file)
    (set-buffer-multibyte nil)
    (goto-char (point-min))
    (unless (looking-at "....ftyp\\(mp4\\|m4a\\)") (error "Not MP4 or M4A."))
    (if (mf-m4a-one-patch (mf-mp4-tag-collect) reverse)
        (progn
          (let ((name (make-backup-file-name file)))
            (if (file-exists-p name) (delete-file name 'trash))
            (rename-file file name))
          (write-region (point-min) (point-max) file)
          t)
      nil)))

(defun mf-m4a-one-patch (atoms &optional reverse)
  "ジャケ表示されないレコチョクの m4a を Walkman で正常化するパッチを当てる.
\"titl\" タグを \"Titl\" に換えているだけ.
REVERSE が非NILなら逆パッチをする."
  (let* ((tag   (if reverse "Titl" "titl"))
         (patch (if reverse ?t ?T))
         (titl  (car (mp4-get-list tag (cadar (mp4-get-container "udta" atoms))))))
    (if (null titl)
        (progn
          (message "No %s tag." tag)
          nil)
      (goto-char (+ (cadr titl) 4))
      (delete-char 1)
      (insert-char patch)
      t)))

(defun mf-mp4-get-type (atoms)
  (let ((a (car (mp4-get-list "ftyp" atoms)))
        beg end)
    (setq beg (+ (cadr a) 8)
          end (+ beg 4))
    (buffer-substring beg end)))

(defun mp4-container-term (type atoms)
  (let ((atom (car (mp4-get-list type atoms))))
    (+ (cadr atom) (caddr atom))))

(defun mp4-meta-include (type meta atoms)
  "TYPE に META (アドレス含めて完全一致する) が含まれていれば
その META の長さを、さもなくば 0 を返す."
  (if (member meta (mp4-get-list (car meta) (car (mp4-get-container type atoms))))
      (caddr meta)
    0))

(defun packet-table-update (stco length)
  "パケットテーブル(STCO)の値に LENGTH 分オフセットを加える.
\"ilst\" の大きさが変わると \"mdat\" の位置が変わるので, これをやらないと音が鳴らない."
  (let* ((beg  (+ (cadr stco)  16)) ;; (stco (car (mp4-get-list "stco" atoms)))
         (end  (+ (cadr stco)  (caddr stco)))
         (size (- (caddr stco) 16))
         (i 0)
         (tmp  (buffer-substring-no-properties beg end))
         high low value)

    ;; 4バイト読んで整数化してオフセット(LENGTH)分を加えて元の位置に書き戻す
    (while (< i size)
      (setq high  (+ (* (aref tmp i) 256) (aref tmp (+ 1 i)))
            low   (+ (* (aref tmp (+ 2 i)) 256) (aref tmp (+ 3 i)))
            value (+ (+ (* high 65536) low) length))
      (aset tmp i (lsh value -24))
      (aset tmp (+ 1 i) (logand (lsh value -16) 255))
      (aset tmp (+ 2 i) (logand (lsh value  -8) 255))
      (aset tmp (+ 3 i) (logand value           255))
      (setq i (+ 4 i)))
    
    (delete-region beg end)
    (goto-char beg)
    (insert tmp)))

;; atoms tree utility
(defun mp4-point-per (now length)
  (ceiling (/ (* now 100.0) length)))

(defun atoms-tree-print (arg stream branch length)
  (if length
      (princ (format "%s%s %s%%\n" branch arg (mp4-point-per (cadr arg) length)) stream)
    (princ (format "%s%s\n" branch arg) stream)))

(defun atoms-tree (atoms &optional stream branch length)
  "ATOMS を tree 表示.
ATOMS は `mf-mp4-tag-collect' が出力する形式の MP4 の atom list."
  (let (leaf (branch (or branch "")))
    (while atoms
      (setq leaf  (car atoms)
            atoms (cdr atoms))
      (if (consp (car leaf))
          (progn
            (atoms-tree-print (car leaf) stream (concat branch (if atoms "|-- " "`-- ")) length)
            (atoms-tree (cdr leaf) stream (concat branch (if atoms "|   " "    ")) length))
        (atoms-tree-print leaf stream (concat branch (if atoms "|-- " "`-- ")) length)))))

(defun mp4-get-atoms (file)
  (with-temp-buffer
    (insert-file-contents-literally file)
    (set-buffer-multibyte nil)
    (goto-char (point-min))
    (unless (looking-at "....ftyp\\(mp4\\|m4a\\)") (error "Not MP4 or M4A."))
    (mf-mp4-tag-collect)))

;;;###autoload
(defun dired-mp4-atoms-tree (&optional prefix)
  "dired でカーソル位置のファイルの mp4 atom を tree 表示.
PREFIX があると位置のパーセント位置も追加する."
  (interactive "P")
  (let* ((file (dired-get-filename))
         (length (and prefix (mf-eighth (file-attributes file)))))
    (with-output-to-temp-buffer "*atoms tree*"
      (princ "(TYPE POINT SIZE)\n.\n")
      (atoms-tree (mp4-get-atoms file) nil nil length))))

;;;###autoload
(defun mp4-atoms-tree (arg &optional prefix)
  "ARG が list ならそのまま file なら開いて mp4 atom を tree 表示する."
  (interactive "fmp4 File: \nP")
  (let ((atm (if (consp arg) arg (mp4-get-atoms arg))))
    (if (not prefix)
        (with-output-to-temp-buffer "*atoms tree*"
          (princ "(TYPE POINT SIZE)\n.\n")
          (atoms-tree atm))
    (atoms-tree atm (current-buffer)))))

;;
;; m4a byte pack.
;;
(defun mf-make-mp4-frame (tag str)
  "TAG と STR をフレームにそれを合わせた長さを追加してパッキング."
  (let ((len (+ (length str) 8)))
    (concat (mf-long-word len) tag str)))

(defun mf-mp4-tag-sort (tags)
  "plist TAGS を読み込み時と同じ順列にソートして返す."
  (sort tags
        #'(lambda(a b)
            (> (or (cdr (assoc (or (plist-get a :dsc) (plist-get a :tag)) mf-mp4-sort-order))
                   #xffffffff)
               (or (cdr (assoc (or (plist-get b :dsc) (plist-get b :tag)) mf-mp4-sort-order))
                   #xffffffff)))))
     
(defun mf-pack-mp4 (tags)
  "plist TAGS を mp4 ilst のバイナリにパッキングして返す."
  (let ((tags (mf-mp4-tag-sort  tags))
        tag str type result mean dsc)
    (dolist (a tags result)
      (setq tag (plist-get a :tag))
      (cond
       ((string-equal "----" tag)
        (setq mean (plist-get a :mean) ; UTF-8 かもしれないが ASCII文字しかないのでそのまま.
              type (plist-get a :type)
              dsc  (plist-get a :dsc)) ; //
        (setq str (or (plist-get a :data) ""))
        ;; (setq str (if (= type (car (rassq 'string mf-ilst-data-type))) (encode-coding-string str 'utf-8) str))
        (setq result
              (cons
               (mf-make-mp4-frame
                tag
                (concat
                 (mf-make-mp4-frame "mean" (format "\0\0\0\0%s" mean))
                 (mf-make-mp4-frame "name" (format "\0\0\0\0%s" dsc))
                 (mf-make-mp4-frame "data" (format "\0\0\0%c\0\0\0\0%s" type str))))
               result)))
       ((assoc tag mp4-container-alist)
        (setq str  (or (plist-get a :data) ""))
        (setq type (or (plist-get a :type) 0))
        (setq str  (if (mf-ilst-string-p type)
                       (encode-coding-string str 'utf-8)
                     (if (mf-ilst-binary-p type)
                         (mf-string-to-number str tag)
                       str)))
        (when str
          (setq result
                (cons
                 (mf-make-mp4-frame
                  tag
                  (mf-make-mp4-frame "data" (concat (format "\0\0\0%c\0\0\0\0" type) str)))
                 result))))
       ((string-match "\\` " tag)
        "")
       (t
        (setq result
              (cons
               (mf-make-mp4-frame
                tag
                (or (plist-get a :data) ""))
               result)))))
    (mf-make-mp4-frame "ilst" (apply #'concat result))))

(defun mf-mp4-tag-read (file &optional length no-binary)
  "FILE のタグを plist にして返す.
`mf-type-dummy' を擬似タグとした TAG の種別も追加される.
LENGTH が非NIL ならその整数分だけ読み込む.
NO-BINARY が非NIL ならイメージタグは含めない."
  (let (hsize atoms tags ilst origin)
    (if length
        (insert-file-contents-literally file nil 0 length)
      (setq length (cadr (insert-file-contents-literally file))))

    (set-buffer-multibyte nil)
    (goto-char (point-min))
    (unless (looking-at "....ftyp") (error "Not MP4."))
    (goto-char (+ (mf-buffer-read-long-word) (point)))
    (setq hsize (mf-buffer-read-long-word))

    (when (and length (> hsize length))
      (let ((fsize  (mf-eighth (file-attributes file)))
            (margin (point)))
        (message "Reload file %s size %d header %d(%d%%)."
                 file fsize hsize (round (/ (* hsize 100.0) fsize)))
        (erase-buffer)
        (insert-file-contents-literally file nil 0 (+ hsize margin))
        (goto-char (point-min))))

    (setq atoms (mf-mp4-tag-collect hsize)
          ilst  (mf-get-ilst atoms))
    (goto-char (point-min))
    (setq mf-current-mode "mp4" origin (buffer-substring (+ (point) 8) (+ (point) 8 4)))
    (setq tags (mf-mp4-tag-analyze ilst no-binary))
    ;; (unless (assoc version func) (error "Bad music file"))
    (setq tags (cons (list :tag mf-type-dummy :data mf-current-mode :org origin) tags))))

(defalias 'mf-m4a-tag-read 'mf-mp4-tag-read)

(provide 'mf-lib-mp4)
;; fine.
