<?php
//  definition of class for OAI-PMH
//  $Revision: 1.25.2.3 $                                                                  //
//  --------------------------------------------------------------------------  //
//  XooNIps Xoops modules for Neuroinformatics Platforms                        //
//  Copyright (C) 2005 RIKEN, Japan. All rights reserved.                       //
//  http://sourceforge.jp/projects/xoonips/                                     //
//  --------------------------------------------------------------------------  //
//  This program is free software; you can redistribute it and/or               //
//  modify it under the terms of the GNU General Public License                 //
//  as published by the Free Software Foundation; either version 2              //
//  of the License, or (at your option) any later version.                      //
//                                                                              //
//  This program is distributed in the hope that it will be useful,             //
//  but WITHOUT ANY WARRANTY; without even the implied warranty of              //
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               //
//  GNU General Public License for more details.                                //
//                                                                              //
//  You should have received a copy of the GNU General Public License           //
//  along with this program; if not, write to the Free Software                 //
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. //
//  --------------------------------------------------------------------------  //

include_once "../../mainfile.php";

$config_handler =& xoops_gethandler('config');
$xoopsConfig =& $config_handler->getConfigsByCat(XOOPS_CONF);

$language = $xoopsConfig['language'];
if ( file_exists(XOOPS_ROOT_PATH."/language/$language/global.php") ) {
        include_once XOOPS_ROOT_PATH."/language/$language/global.php";
} else {
        include_once XOOPS_ROOT_PATH."/language/english/global.php";
}

include_once XOOPS_ROOT_PATH.'/modules/xoonips/include/encode.php';

class OAIPMH {

	/** hash to manage OAIPMHHandler ( key=name of metadataPrefix, value=instance of OAIPMHHandler) */
	var $handlers;

	/** base URL of repository */
	var $baseURL;

	/** name of repository */
	var $repositoryName;

	/** array of administrator's e-mail address */
	var $adminEmails;

	/**
	 * 
	 * 
	 * @param baseURL 
	 * @param repositoryName 
	 * @param adminEmails: administrator's e-mail address (charactor strings or array)
	 * 
	 */
	function OAIPMH( $baseURL = null, $repositoryName = null, $adminEmails = null ) {
		/* constructer for PHP3, PHP4 */
		$this->baseURL = $baseURL;
		$this->repositoryName = $repositoryName;
		if( !is_array( $adminEmails ) )
			$this->adminEmails = array( $adminEmails );
		else
			$this->adminEmails = $adminEmails;

		$this->handlers = array( );
	}

	/**
	 * 
	 * 
	 * @param baseURL 
	 * @param repositoryName 
	 * @param adminEmails: administrator's e-mail address (charactor strings or array)
	 * 
	 */
	function __construct( $baseURL = null, $repositoryName = null, $adminEmails = null ) {
		/* constructer for PHP5 */
		OAIPMH::OAIPMH( $baseURL, $repositoryName, $adminEmails );
	}
	function __destruct( ) {
	}

	/**
	 * 
	 * register OAIPMHHandler
	 * 
	 * @param class: instance of OAIPMHHandler
	 */
	function addHandler( $class ) {
		$this->handlers[$class->getName( )] = $class;
	}

	/**
	 * 
	 * delete OAIPMHHandler
	 * 
	 * @param class: instance of OAIPMHHandler
	 */
	function removeHandler( $class ) {
		$this->handlers[$class->getName( )] = null;
	}

	/**
	 * 
	 * return list of OAIPMHHandler
	 * 
	 * @return array of instances in registered handler
	 */
	function listHandlers( ) {
		return $this->handlers;
	}

	/**
	 * 
	 * generate <request>
	 * 
	 * @param attrs: hash parameters about demand of list <br/> ex:arary( 'verb' => 'GetRecord', 'identifier' => 'xxxx' )
	 * @return tag of <request>
	 */
	function request( $attrs ) {
		$request = "<request";
		foreach( array( 'verb', 'identifier', 'metadataPrefix', 'from', 'until', 'set', 'resumptionToken' ) as $k ) {
			if( array_key_exists( $k, $attrs ) ) {
				$request .= " ${k}=\"".$attrs[$k]."\"";
			}
		}
		$request .= ">";
		$request .= $this->baseURL;
		$request .= "</request>\n";
		return $request;
	}

	/**
	 * 
	 * generate <ListMetadataFormats>
	 * 
	 * @return <ListMetadataFormats>
	 * 
	 */
	function ListMetadataFormats( $attrs ) {
		$lines = array( );
		foreach( $this->handlers as $hdl ) {
			if( isset( $attrs['identifier'] ) )
				$fmt = $hdl->metadataFormat( $attrs['identifier'] );
			else
				$fmt = $hdl->metadataFormat( );
			if( !$fmt )
				continue;		// item specified by $identifier doesn't support this format.
			$lines[] = $fmt;
		}

		if( count( $lines ) == 0 )
			return $this->header( )
				.$this->request( $attrs )
				.$this->error( 'noMetadataFormats', '' )
				.$this->footer( );

		if( isset( $attrs['identifier'] ) )
			$identifier = $attrs['identifier'];
		else
			$identifier = null;
		return $this->header( )
			.$this->request( array( 'verb' => 'ListMetadataFormats', 'identifier' => $identifier ) )
			."<ListMetadataFormats>\n".implode( "\n", $lines )
			."</ListMetadataFormats>\n".$this->footer( );
	}

	/**
	 * 
	 * generate <Identify>
	 * 
	 * @return <Identify>
	 * 
	 */
	function Identify( ) {
		$xml = "<repositoryName>".$this->repositoryName."</repositoryName>
<baseURL>".$this->baseURL."</baseURL>
<protocolVersion>2.0</protocolVersion>
<deletedRecord>transient</deletedRecord>
<granularity>YYYY-MM-DDThh:mm:ssZ</granularity>
<earliestDatestamp>".gmdate( "Y-m-d\TH:i:s\Z", 0 )."</earliestDatestamp>
";
		foreach( $this->adminEmails as $email ) {
			$xml .= "<adminEmail>${email}</adminEmail>
";
		}
		return $this->header( )
			.$this->request( array( 'verb' => 'Identify' ) )
			."<Identify>\n".$xml."</Identify>\n".$this->footer( );
	}

	/**
	 * 
	 * 
	 * @return header of xml, <OAI-PMH>, <responseDate>
	 * 
	 */
	function header( ) {
		$date = gmdate( "Y-m-d\TH:i:s\Z", time( ) );
		return '<?xml version="1.0" encoding="UTF-8" ?>
<OAI-PMH xmlns="http://www.openarchives.org/OAI/2.0/"
         xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance"
         xsi:schemaLocation="http://www.openarchives.org/OAI/2.0/
         http://www.openarchives.org/OAI/2.0/OAI-PMH.xsd">
<responseDate>'.$date.'</responseDate>
';
	}

	/**
	 * 
	 * 
	 * @return </OAI-PMH>
	 * 
	 */
	function footer( ) {
		return '</OAI-PMH>
';
	}

	//call same name function of OAIPMHHandler class.
	//judges $metadataPrefix which OAIPMHHandler to use.
	//return error 'badArgument', if arguments are few.
	//return error 'cannnotDisseminateFormat', if Handler for $metadetaPrefix isn't registered.
	//The errors occered in handler aren't concerned(Handler return <error>.
	function GetRecord( $args ) {
		$attrs = array_merge( $args, array( 'verb' => 'GetRecord' ) );
		if( count( $args ) == 3 && isset( $args['verb'] ) && isset( $args['metadataPrefix'] ) && isset( $args['identifier'] ) );
		else
			return $this->header( )
				.$this->request( $attrs )
				.$this->error( 'badArguments', '' )
				.$this->footer( );

		if( !isset( $args['identifier'] ) )
			return $this->header( )
				.$this->request( $attrs )
				.$this->error( 'badArgument', 'identifier is not specified' )
				.$this->footer( );
		if( !array_key_exists( $args['metadataPrefix'], $this->handlers )
			|| !$this->handlers[$args['metadataPrefix']]->metadataFormat( $args['identifier'] ) )
			return $this->header( )
				.$this->request( $attrs )
				.$this->error( 'cannotDisseminateFormat', '' )
				.$this->footer( );

		return $this->header( )
			.$this->request( $attrs )
			.$this->handlers[$args['metadataPrefix']]->GetRecord( $args )
			.$this->footer( );
	}

	function ListIdentifiers( $args ) {
		$attrs = array_merge( $args, array( 'verb' => 'ListIdentifiers' ) );
		if( !isset( $args['metadataPrefix'] ) && !isset( $args['resumptionToken'] ) )
			return $this->header( )
				.$this->request( $attrs )
				.$this->error( 'badArgument', '' )
				.$this->footer( );

		if( isset( $args['resumptionToken'] ) ) {
			$result = getResumptionToken( $args['resumptionToken'] );
			if( !$result ) {
				return $this->header( )
					.$this->request( $attrs )
					.$this->error( 'badResumptionToken', '' )
					.$this->footer( );
			}
			$metadataPrefix = $result['metadata_prefix'];
		} else
			$metadataPrefix = $args['metadataPrefix'];

		return $this->header( )
			.$this->request( $attrs )
			.$this->handlers[$metadataPrefix]->ListIdentifiers( $args )
			.$this->footer( );
	}

	function ListRecords( $args ) {
		$attrs = array_merge( $args, array( 'verb' => 'ListRecords' ) );
		if( !isset( $args['metadataPrefix'] ) && !isset( $args['resumptionToken'] ) )
			return $this->header( )
				.$this->request( $attrs )
				.$this->error( 'badArgument', '' )
				.$this->footer( );
		if( isset( $args['resumptionToken'] ) ) {
			$result = getResumptionToken( $args['resumptionToken'] );
			if( !$result ) {
				return $this->header( )
					.$this->request( $attrs )
					.$this->error( 'badResumptionToken', '' )
					.$this->footer( );
			}
			$metadataPrefix = $result['metadata_prefix'];
		} else
			$metadataPrefix = $args['metadataPrefix'];

		if( !isset( $this->handlers[$metadataPrefix] ) ) {
			return $this->header( )
				.$this->request( $attrs )
				.$this->error( 'cannotDisseminateFormat', '' )
				.$this->footer( );
		}
		return $this->header( )
			.$this->request( $attrs )
			.$this->handlers[$metadataPrefix]->ListRecords( $args )
			.$this->footer( );
	}

	function ListSets( $args ) {
		$attrs = array_merge( $args, array( 'verb' => 'ListSets' ) );
		return $this->header( )
			.$this->request( $attrs )
			.$this -> error( 'noSetHierarchy', 'This ripository does not support sets' )
			.$this->footer( );
	}

	function error( $errorcode, $text ) {
		return "<error code='${errorcode}'>${text}</error>\n";
	}
}

class OAIPMHHandler {

	var $metadataPrefix;
	function OAIPMHHandler( ) {
		/* constructer for PHP3, PHP4 */
	} function __construct( ) {
		/* constructer for PHP5 */
	}
	function __destruct( ) {
	}

	function getName( ) {
		return $this->metadataPrefix;
	}

	/**
	 * 
	 * Analize $identifier, and return by dividing $identifier into nijc_code, item_id, item_type_id.
	 * 
	 * @param identifier: id to analize
	 * @return array( 'nijc_code' => NIJC issue code, 'item_type_id' => id of itemtypes, 'item_id' => id of item )
	 * @return false: failure in analysis
	 * 
	 */
	function parseIdentifier( $identifier ) {
		$match = array( );
		if( preg_match( "/([^\/]+)\/([0-9]+)\.([0-9]+)/", $identifier, $match ) == 0 )
			return false;
		return array( 'nijc_code' => $match[1], 'item_type_id' => $match[2], 'item_id' => $match[3] );
	}

	/**
	 *
	 * @param identifier
	 * @return xml   <metadataFormat> ... </metadataFormat>
	 * @return false not support this format
	 */
	function metadataFormat( $identifier ) {
		return false;
	}

	function GetRecord( $args ) {
	}
	function ListIdentifiers( $args ) {
	}
	function ListRecords( $attr ) {
	}
	function ListSets( $args ) {
	}

	function error( $errorcode, $text ) {
		return "<error code='${errorcode}'>${text}</error>\n";
	}
}

class OAIPMHHarvester {
	var $dateformat;
	var $earliestDatestamp;
	var $baseURL;
	var $metadataPrefix;
	var $lastError;
	var $lastStatus;

	//public
	function OAIPMHHarvester( $_baseURL ) {
		$this->baseURL = $_baseURL;
		$this->lastError = null;
		$this->lastStatus = null;
		$this->metadataPrefix = null;
		$this->dateformat = null;
		$this->earliestDatestamp = null;
	} function __construct( $_baseURL ) {
		/* constructer for PHP5 */
		$this->OAIPMHHarvester( $_baseURL );
	}
	function harvest( ) {
		global $xoopsDB;

		$ts =& MyTextSanitizer::getInstance( );

		//update repositories table
		$tbl_repositories = $xoopsDB->prefix( 'xoonips_oaipmh_repositories' );
		$sql = "UPDATE ${tbl_repositories} SET last_access_date=".time( ).", last_access_result=NULL WHERE URL='".$ts->addSlashes( $this->baseURL )."'";
		$result = $xoopsDB->queryF( $sql );
		if( $result ) {
			if( $this->Identify( ) ) {
				if( $this->ListMetadataFormats( ) ) {
					$args = array( 'metadataPrefix' => $this->metadataPrefix );
					$sql = "SELECT last_success_date FROM ${tbl_repositories} WHERE URL='".$ts->addSlashes( $this->baseURL )."'";
					$result = $xoopsDB->query( $sql );
					if( $result && $xoopsDB->getRowsNum( $result ) > 0 ) {
						list( $from_timestamp ) = $xoopsDB->fetchRow( $result );
						if( $from_timestamp == '' && $this->earliestDatestamp != null )
							$from_timestamp = $this->earliestDatestamp;
						$args['from'] = gmdate( $this->dateformat, $from_timestamp );
					}

					if( $this->ListRecords( $args ) ) {
						//update repositories table
						$tbl_repositories = $xoopsDB->prefix( 'xoonips_oaipmh_repositories' );
						$sql = "UPDATE ${tbl_repositories} SET last_success_date=".time( ).", last_access_result='".$ts->addSlashes( $this->lastStatus )."' WHERE URL='".$ts->addSlashes( $this->baseURL )."'";
						$result = $xoopsDB->queryF( $sql );
						if( $result )
							return true;
						else
							$this->lastError = $xoopsDB->error( );
					}
				}
			}
		} else
			$this->lastError = $xoopsDB->error( );

		//update repositories table(last_access_result)
		$tbl_repositories = $xoopsDB->prefix( 'xoonips_oaipmh_repositories' );
		$sql = "UPDATE ${tbl_repositories} SET last_access_result='".$ts->addSlashes( $this->lastError )."' WHERE URL='".$ts->addSlashes( $this->baseURL )."'";
		$result = $xoopsDB->queryF( $sql );
		return false;
	}
	function last_error( ) {
		return $this->lastError;
	}

	//private
	function Identify( ) {
		$hdl = fopen( $this->baseURL."?verb=Identify", 'r' );
		if( !$hdl ) {
			$this->lastError = "can't retrieve ".$this->baseURL."?verb=Identify";
			return false;
		}
		$metadata = stream_get_meta_data( $hdl );
		$this->lastError = $http_status = $metadata['wrapper_data'][0];

		$this->parser = xml_parser_create( 'UTF-8' );
		if( !$this->parser ) {
			fclose( $hdl );
			$this->lastError = "can't create XML parser";
			return false;
		}
		$handler = new IdentifyHandler( $this->parser );

		$result = $this->parse( $hdl );
		xml_parser_free( $this->parser );
		fclose( $hdl );
		if( !$result )
			return false;		//some error has occured in parse( $hdl );

		$this->dateformat = $handler->getDateFormat( );
		if( !$this->dateformat ) {
			$this->lastError = "value of <granularity> is wrong";
			return false;
		}

		$this->earliestDatestamp = $handler->getEarliestDatestamp( );

		return true;
	}

	function ListMetadataFormats( ) {
		$hdl = fopen( $this->baseURL."?verb=ListMetadataFormats", 'r' );
		if( !$hdl ) {
			$this->lastError = "can't retrieve ".$this->baseURL."verb=ListMetadataFormats";
			return false;
		}
		$metadata = stream_get_meta_data( $hdl );
		$this->lastError = $http_status = $metadata['wrapper_data'][0];

		$this->parser = xml_parser_create( 'UTF-8' );
		if( !$this->parser ) {
			fclose( $hdl );
			$this->lastError = "can't create XML parser";
			return false;
		}
		$handler = new ListMetadataFormatsHandler( $this->parser );

		$result = $this->parse( $hdl );
		xml_parser_free( $this->parser );
		fclose( $hdl );
		if( !$result )
			return false;		//some error has occured in parse( $hdl );

		$this->metadataPrefix = $handler->getMetadataPrefix( );
		if( !$this->metadataPrefix ) {
			$this->metadataPrefix = null;
			$this->lastError = "can't retrieve <metadataPrefix>";
			return false;
		}
		return true;
	}

	function ListRecords( $args ) {
		global $xoopsDB;
		if( !isset( $args['metadataPrefix'] ) ) {
			$this->lastError = "'metadataPrefix' is not specified.";
			return false;
		}
		$resumptionToken = null;
		do {
			$url = $this->baseURL."?verb=ListRecords";
			if( $resumptionToken == null )
				foreach( array( 'metadataPrefix', 'from', 'until', 'set' ) as $k ) {
				if( isset( $args[$k] ) )
					$url .= "&".urlencode( $k )."=".urlencode( $args[$k] );
			} else {
				$url .= "&resumptionToken=".htmlspecialchars( $resumptionToken );
			}
			$hdl = fopen( $url, 'r' );
			if( !$hdl ) {
				$this->lastError = "can't retrieve ${url}";
				$xoopsDB -> setLogger( XoopsLogger::instance() );
				return false;
			}
			$metadata = stream_get_meta_data( $hdl );
			$this->lastError = $this->lastStatus = $metadata['wrapper_data'][0];

			$this->parser = xml_parser_create( 'UTF-8' );
			if( !$this->parser ) {
				fclose( $hdl );
				$this->lastError = "can't create XML parser";
				$xoopsDB -> setLogger( XoopsLogger::instance() );
				return false;
			}

			$handler = "${resumptionToken}_handler";
//			$${handler} = new ListRecordsHandler( $this->parser, $this->baseURL, $args['metadataPrefix'] );
			$${'handler'} = new ListRecordsHandler( $this->parser, $this->baseURL, $args['metadataPrefix'] );
			$result = $this->parse( $hdl );
			xml_parser_free( $this->parser );
			fclose( $hdl );
			if( !$result ) {
				//some erorr has occured
//				if( $${handler}->getIdentifier( ) != null )
				if( $${'handler'}->getIdentifier( ) != null )
//					$this->lastError .= "[identifier]".$${handler}->getIdentifier( );
					$this->lastError .= "[identifier]".$${'handler'}->getIdentifier( );
				$xoopsDB -> setLogger( XoopsLogger::instance() );
				return false;	//some error has occured in parse( $hdl );    
			}
//			$resumptionToken = $${handler}->getResumptionToken( );	//
			$resumptionToken = $${'handler'}->getResumptionToken( );	//
		} while( $resumptionToken != null );
		$xoopsDB -> setLogger( XoopsLogger::instance() );
		return true;
	}

	function parse( $hdl ) {
//      $text = '';
		while( !feof( $hdl ) ) {
			$data = fread( $hdl, 8192 );
			//			$data = xnpEntity2Utf8( $data );
			if( !xml_parse( $this->parser, $data, feof( $hdl ) ) ) {
				$this->lastError = "[XMLParser]".xml_error_string( xml_get_error_code( $this->parser ) )." at line ".xml_get_current_line_number( $this->parser ).", column ".xml_get_current_column_number( $this->parser );

/*
				$lines = preg_split( "/\n/", $text );
				$this -> lastError .= "\n";
				for( $i = -3; $i < 3; $i++ ){
					$line = xml_get_current_line_number( $this -> parser ) + $i;
					if( $line < 0 || $line >= count( $lines ) ) continue;
					if( $i == 0 ) $this -> lastError .= ">>";
					else  $this -> lastError .= "  ";
					$this -> lastError .= $lines[$line]."\n";
				}
*/
				fclose( $hdl );
				xml_parser_free( $this->parser );
				return false;
			}
//          $text .= $data;
		}
		return true;
	}
}

class HarvesterHandler {
	var $parser;
	var $lastError;

	function HarvesterHandler( $_parser ) {
		$this->lastError = null;
		$this->parser = $_parser;
		xml_set_object( $this->parser, $this );
		xml_set_element_handler( $this->parser, 'startElementHandler', 'endElementHandler' );
		xml_set_character_data_handler( $this->parser, 'characterDataHandler' );
	} function __construct( $_parser ) {
		$this->HarvesterHandler( $_parser );
	}
	function startElementHandler( $parser, $name, $attribs ) {
	}
	function endElementHandler( $parser, $name ) {
	}
	function characterDataHandler( $parser, $data ) {
	}

	function last_error( ) {
		return $this->lastError;
	}
}

class IdentifyHandler extends HarvesterHandler {
	var $dateformat;
	var $earliestDatestamp;
	var $tagstack;
	function IdentifyHandler( $_parser ) {
		parent::HarvesterHandler( $_parser );
		$this->earliestDatestamp = null;
		$this->dateformat = null;
		$this->tagstack = array( );
	} function __construct( $_parser ) {
		$this->IdentifyHandler( $_parser );
	}

	function startElementHandler( $parser, $name, $attribs ) {
		array_push( $this->tagstack, $name );
	}

	function endElementHandler( $parser, $name ) {
		array_pop( $this->tagstack );
	}

	function characterDataHandler( $parser, $data ) {
		if( end( $this->tagstack ) == 'GRANULARITY' ) {
			if( $data == "YYYY-MM-DDThh:mm:ssZ" )
				$this->dateformat = "Y-m-d\TH:i:s\Z";
			else if( $data == "YYYY-MM-DD" )
				$this->dateformat = "Y-m-d";
			else
				$this->dateformat = false;
		} else if( end( $this->tagstack ) == 'EARLIESTDATESTAMP' ) {
			$this->earliestDatestamp = ISO8601toUTC( $data );
		}
	}

	function getDateFormat( ) {
		return $this->dateformat;
	}
	function getEarliestDatestamp( ) {
		return $this->earliestDatestamp;
	}
}

class ListMetadataFormatsHandler extends HarvesterHandler {
	var $metadataPrefix;
	var $tagstack;

	function ListMetadataFormatsHandler( $_parser ) {
		parent::HarvesterHandler( $_parser );

		$this->metadataPrefix = "oai_dc";
		$this->tagstack = array( );
	} function __construct( $_parser ) {
		$this->ListMetadataFormatsHandler( $_parser );
	}

	function startElementHandler( $parser, $name, $attribs ) {
		array_push( $this->tagstack, $name );
	}

	function endElementHandler( $parser, $name ) {
		array_pop( $this->tagstack );
	}

	function characterDataHandler( $parser, $data ) {
		if( end( $this->tagstack ) == 'METADATAPREFIX' ) {
			if( $data == "junii" ) {
				$this->metadataPrefix = $data;
			}
		}
	}

	function getMetadataPrefix( ) {
		return $this->metadataPrefix;
	}
}

class ListRecordsHandler extends HarvesterHandler {
	var $resumptionToken;
	var $identifier;
	var $title;
	var $metadata;
	var $metadataPrefix;
	var $search_text;
	var $baseURL;
	var $delete_flag;

	function ListRecordsHandler( $_parser, $_baseURL, $_metadataPrefix ) {
		parent::HarvesterHandler( $_parser );

		$this->resumptionToken = null;
		$this->identifier = null;
		$this->title = null;
		$this->metadata = array( );
		$this->metadataPrefix = $_metadataPrefix;
		$this->search_text = array( );
		$this->tagstack = array( );
		$this->baseURL = $_baseURL;
		$this->delete_flag = false;
	} function __construct( $_parser, $_baseURL, $_metadataPrefix ) {
		$this->ListRecordsHandler( $_parser, $_baseURL, $_metadataPrefix );
	}

	function startElementHandler( $parser, $name, $attribs ) {
		array_push( $this->tagstack, $name );
		if( $name == 'RECORD' ){
			global $xoopsDB;
			$xoopsDB -> setLogger( new XoopsLogger() );
			
			// initialize following value for each records
			$this->title = null;
			$this->metadata = array( );
			$this->search_text = array( );
		}
		if( $name == 'HEADER' ) {
			if( isset( $attribs['STATUS'] )
				&& $attribs['STATUS'] == 'deleted' ) {
				$this->delete_flag = true;
			} else {
				$this->delete_flag = false;
			}
		}
		if( !isset( $this->tagstack[2] ) || $this->tagstack[2] != 'RECORD' )
			return;

		array_push( $this->metadata, "<".strtolower( $name )."" );
		foreach( $attribs as $k => $v ) {
			array_push( $this->metadata, " ".strtolower($k)."='${v}'" );
		}
		array_push( $this->metadata, ">" );
	}

	function endElementHandler( $parser, $name ) {

		array_push( $this->search_text, ' ' );
		if( isset( $this->tagstack[2] ) && $this->tagstack[2] == 'RECORD' )
			array_push( $this->metadata, "</".strtolower( $name ).">" );
		if( $name == 'RECORD' ) {
			global $xoopsDB;

			$ts =& MyTextSanitizer::getInstance( );
			$tbl_repositories = $xoopsDB->prefix( 'xoonips_oaipmh_repositories' );
			$tbl_metadata = $xoopsDB->prefix( 'xoonips_oaipmh_metadata' );

			if( $this->delete_flag ) {
				//delete a metadata
				$sql = "DELETE FROM ${tbl_metadata} WHERE identifier='".$ts->addSlashes( $this->identifier )."'";
				$result = $xoopsDB->queryF( $sql );
			} else {
				//retrieve a repository id
				$sql = "SELECT repository_id FROM ${tbl_repositories} WHERE URL='".$ts->addSlashes( $this->baseURL )."'";
				$result = $xoopsDB->query( $sql );
				list( $id ) = $xoopsDB->fetchRow( $result );
				if( $id ) {
					$this->search_text = array_unique( $this->search_text );
					array_walk( $this -> search_text, 'array_walk_bin2hex' );
					//insert data into metadata table
/*
					$values = array( $ts->addSlashes( $this->identifier ),
									 $id,
									 $ts->addSlashes( $this->metadataPrefix ),
									 $ts->addSlashes( $this->title ),
									 $ts->addSlashes( implode( '', $this->metadata ) ),
									 $ts->addSlashes( implode( ' ', $this->search_text ) ) );
*/
					$values = array( addslashes( encodeMeta2Server( $this->identifier ) ),
									 $id,
									 addslashes( encodeMeta2Server( $this->metadataPrefix ) ),
									 addslashes( encodeMeta2Server( $this->title ) ),
									 addslashes( encodeMeta2Server( implode( '', $this->metadata ) ) ),
									 addslashes( encodeMeta2Server( implode( ' ', $this->search_text ) ) ) ); 
					$sql = "INSERT INTO ${tbl_metadata} (identifier, repository_id, format, title, metadata, search_text) VALUES ('".implode( "', '", $values )."')";
					$result = $xoopsDB->queryF( $sql );
					if( !$result && $xoopsDB->errno( ) == 1062 ) {	//1062: Can't write; duplicate key in table '%s' 
/*
						$update = array( 'title' => $ts->addSlashes( $this->title ),
										 'format' => $ts->addSlashes( $this->metadataPrefix ),
										 'metadata' => $ts->addSlashes( implode( '', $this->metadata ) ),
										 'search_text' => $ts->addSlashes( implode( ' ', $this->search_text ) ) );
*/
						$update = array( 'title' => addslashes( encodeMeta2Server( $this->title ) ),
										 'format' => addslashes( encodeMeta2Server( $this->metadataPrefix ) ),
										 'metadata' => addslashes( encodeMeta2Server( implode( '', $this->metadata ) ) ),
										 'search_text' => addslashes( encodeMeta2Server( implode( ' ', $this->search_text ) ) ) );
						$sql = "UPDATE ${tbl_metadata} SET ";
						$update2 = array( );
						foreach( $update as $k => $v ) $update2[] = "${k}='${v}'";
						$sql .= implode( ', ', $update2 );
						$sql .= " WHERE identifier='".$ts->addSlashes( $this->identifier )."'";
						$result = $xoopsDB->queryF( $sql );
					}
				}
			}
			$this->metadata = array( );
			$this->search_text = array( );
		}
		array_pop( $this->tagstack );
	}

	function characterDataHandler( $parser, $data ) {
		if( end( $this->tagstack ) == 'RESUMPTIONTOKEN' ) {
			$this->resumptionToken = $data;
			return;
		}
		if( end( $this->tagstack ) == 'IDENTIFIER' && prev( $this->tagstack ) == 'HEADER' ) {
			$this->identifier = $data;
			echo "${data}\n";
		}

		//<xxx:title> -> <title>
//		if( end( $tmp = split( ':', end( $this->tagstack ) ) ) == 'TITLE' ) {
		$tmp = split( ':', end( $this->tagstack ) );
		if( end( $tmp ) == 'TITLE' ) {
			$this->title .= $data;
		}

		if( !isset( $this->tagstack[2] ) || $this->tagstack[2] != 'RECORD' )
			return;
		// $data: unhtmlspecialchars() in characterDataHandler => to get back to XML, we need to do htmlspecialchars().
		array_push( $this->metadata, htmlspecialchars($data, ENT_QUOTES) );
		array_push( $this->search_text, $data );
		$this->search_text = array_merge( $this->search_text, $this -> word_separation( $data ) );
	}

	function getResumptionToken( ) {
		return $this->resumptionToken;
	}
	function getIdentifier( ) {
		return $this->identifier;
	}

	//private 
	function utf8bytes( $c1 ){
		if( ( $c1 & 0x80 ) == 0 ) return 1;
		if( ( $c1 & 0xe0 ) == 0xc0 ) return 2;
		if( ( $c1 & 0xe0 ) == 0xe0 ) return 3;
		if( ( $c1 & 0xf8 ) == 0xf0 ) return 4;
		return 0;
	}

	function window( $bytes ){
		$length = 8;
		$ret = array();
		$word = array();
		$len = 0;

		for( $len = 0; $len < $length && count( $bytes ) > 0 ; $len++ ){
			$c1 = reset( $bytes );
			for( $i = 0; $i < $this -> utf8bytes( $c1 ); $i++ ){
				array_push( $word, pack( "C", array_shift( $bytes ) ) );
			}
		}
		while( true ){
			array_push( $ret, implode( '', $word ) );
			if( count( $bytes ) > 0 && $len >= $length || $len >= 3 ){
//				$c1 = reset( unpack( "C", reset( $word ) ) );
				$tmp = unpack( "C", reset( $word ) );
				$c1 = reset( $tmp );
				$tmp = $this -> utf8bytes( $c1 );
//				for( $i = 0; $i < $this -> utf8bytes( $c1 ); $i++ ){
				for( $i = 0; $i < $tmp; $i++ ){
					array_shift( $word );
				}
				$len--;
			}else break;
			if( count( $bytes ) > 0 ){
				$c1 = reset( $bytes );
				for( $i = 0; $i < $this -> utf8bytes( $c1 ); $i++ ){
					array_push( $word, pack( "C", array_shift( $bytes ) ) );
				}
				$len++;
			}
		}
		return $ret;
	}

	/**
	 * 
	 * character strings are divided for search.
	 * Multi byte character strings is applied to window processing of length of 8 characters.
	 * don't change one byte character strings.
	 * 
	 */
	function word_separation( $text ){
		$multi = array();
		$bytes = unpack( "C*", $text );
		$c1 = 0;
		$words = array();
		while( count( $bytes ) >= 0 ){
			$tmp = array();
			if( count( $multi ) > 0 && ( $c1 & 0x80 ) != 0 && ( reset( $bytes ) & 0x80 ) == 0 ){
				$words = array_merge( $words, $this -> window( $multi ) );
				$multi = array();
			}else if( count( $multi ) > 0 && ( $c1 & 0x80 ) == 0 && ( reset( $bytes ) & 0x80 ) != 0 ){ 
				$tmp = array();
				while( count( $multi ) > 0 ) array_push( $tmp, pack( "C", array_shift( $multi ) ) );
				$words = array_merge( $words, explode( ' ', implode( '', $tmp ) ) );
				$multi = array();
			}
			if( count( $bytes ) == 0 ) break;
			$c1 = reset( $bytes );
			if( $this -> utf8bytes( $c1 ) == 1 ){
				array_push( $multi, array_shift( $bytes ) );
			}else{
				for( $i = 0; $i < $this -> utf8bytes( $c1 ); $i++ ){
					array_push( $multi, array_shift( $bytes ) );
				}
			}
		}
		if( count( $multi ) > 0 ){
			$tmp = array();
			while( count( $multi ) > 0 ) array_push( $tmp, pack( "C", array_shift( $multi ) ) );
			$words = array_merge( $words, explode( ' ', implode( '', $tmp ) ) );
		}
	
		return $words;
	}
	
}

function array_walk_bin2hex( &$item1, $key ){
	if ($item1 == '') return;
	$tmp = unpack( "C", $item1 );
//	if( ( reset( unpack( "C", $item1 ) ) & 0x80 ) == 0 || $item1 == ' ' || $item1 == "\n" ) return;
	if( ( reset( $tmp ) & 0x80 ) == 0 || $item1 == ' ' || $item1 == "\n" ) return;
	$item1 = bin2hex( $item1 );
}


/**
 * 
 * record resumptionToken and the related information.
 * 
 * @param resumption_token 
 * @param metadata_prefix 
 * @param verb 
 * @param args: hash in arguments specified verb
 * @param last_item_id: maximum value of item_id to applied in last demands.
 * @param limit_row: number of results applied this resumptionToken.
 * @param expire_date: Expiration date of this resumptionToken
 * @param publish_date: Of the date of this resumptionToken's issue
 * @return nothing
 * 
 */
function
setResumptionToken( $resumption_token, $metadata_prefix, $verb, $args, $last_item_id, $limit_row, $expire_date, $publish_date = null )
{
	global $xoopsDB;

	if( $publish_date == null )
		$publish_date = time( );
	$myts =& MyTextSanitizer::getInstance( );
	$table = $xoopsDB->prefix( "xoonips_oaipmh_resumption_token" );
	$sql = "INSERT INTO ${table} VALUES ('".$myts->addSlashes( $resumption_token )
		."', '${metadata_prefix}', '${verb}', '".$myts->addSlashes( serialize( $args ) )
		."', ${last_item_id}, ${limit_row}, ${publish_date}, ${expire_date} )";
	return $xoopsDB->queryF( $sql );
}

/**
 * 
 * return the related information of resumptionToken
 * 
 * @param resumption_token 
 * @return array( 
 * 'metadata_prefix' => metadataPrefix,
 * 'verb' => verb,
 * 'args' => hash in arguments specified verb,
 * 'last_item_id' => maximum value of item_id to applied in last demands,
 * 'limit_row' => number of results applied this resumptionToken,
 * 'expire_date' => Expiration date of this resumptionToken,
 * 'publish_date' => Of the date of this resumptionToken's issue )
 * @return false: not found resumptionToken
 * 
 */
function
getResumptionToken( $resumption_token )
{
	global $xoopsDB;

	$myts =& MyTextSanitizer::getInstance( );
	$table = $xoopsDB->prefix( "xoonips_oaipmh_resumption_token" );
	$sql = "SELECT resumption_token, metadata_prefix, args, last_item_id, publish_date, expire_date, verb FROM ${table} WHERE resumption_token=\"".$myts->stripSlashesGPC( $resumption_token )
		.'"';
	$result = $xoopsDB->query( $sql );
	$ret = $xoopsDB->fetchArray( $result );
	$ret['args'] = unserialize( $ret['args'] );
	return $ret;
}

/**
 * 
 * expire resumptionToken.
 * expire other tokens of cutting at expiration date, too.
 * 
 * @param resumption_token: Token to expire.
 * 
 */
function
expireResumptionToken( $resumptionToken )
{
	global $xoopsDB;

	$myts =& MyTextSanitizer::getInstance( );
	$table = $xoopsDB->prefix( "xoonips_oaipmh_resumption_token" );
	$sql = "DELETE FROM ${table} WHERE resumption_token=\"".$myts->stripSlashesGPC( $resumptionToken )
		.'" OR expire_date < UNIX_TIMESTAMP( NOW() )';
	$result = $xoopsDB->queryF( $sql );
}

/**
 * 
 * change expression of ISO8601 to UTC. return false when we can't change.
 * Usage: ISO8601toUTC( "2005-08-01T12:00:00Z" );
 * Usage: ISO8601toUTC( "2005-08-01" );
 * 
 */
function
ISO8601toUTC( $str )
{
	if( preg_match( "/([0-9]{4})-([0-9]{2})-([0-9]{2})(T([0-9]{2}):([0-9]{2}):([0-9]{2})Z)?/", $str, $match ) == 0 )
		return 0;
	if( !isset( $match[5] ) )
		$match[5] = 0;
	if( !isset( $match[6] ) )
		$match[6] = 0;
	if( !isset( $match[7] ) )
		$match[7] = 0;
	return gmmktime( $match[5], $match[6], $match[7], $match[2], $match[3], $match[1] );
}

// todo: $a = explode(',',$metajuniig[$num10]); $metajunii = $a[$num1];
function xnpGetMetadataJunii( $id ) {
	if (xnpChkServerLang()) {
		if (_CHARSET == 'EUC-JP') {
			include_once XOOPS_ROOT_PATH.'/modules/xoonips/language/japanese/metadata.php';
		}
		else {
			include_once XOOPS_ROOT_PATH.'/modules/xoonips/language/japaneseutf/metadata.php';
		}
	}
	else {
		include_once XOOPS_ROOT_PATH.'/modules/xoonips/language/english/metadata.php';
	}
	$metajuniig = explode( '/', _MD_XOONIPS_METADATA_JUNII );
	$i = 0;
	$num = $id;
	$num10 = ($num - ($num % 10)) / 10;
	$num1 = $num % 10;
	foreach ($metajuniig as $key => $value1) {
		$metajuniis = explode( ',', $value1 );
		$j = 0;
		foreach ($metajuniis as $key => $value2) {
			if (($i == $num10) && ($j == $num1)) {
				$metajunii = $value2;
			}
			$j++;
		}
		$i++;
	}
	return $metajunii;
}
?>
