/* $Id: davcp.c,v 1.14 2003/01/19 19:06:45 rtakano Exp $
 * davcp: WebDAV copy
 * Copyright (c) 2002 TAKANO Ryousei <rtakano@sourceforge.jp>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <sys/stat.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <getopt.h>
#include <signal.h>

#include <ne_session.h>
#include <ne_uri.h>
#include <ne_basic.h>
#include <ne_auth.h>

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include "davcp.h"


enum {
  METHOD_GET = 0,
  METHOD_PUT = 1,
};

int opt_recursive;
int opt_verbose = 1;
static char logname[256];

static void usage(void)
{
  printf("\
Usage: davcp [OPTION] SOURCE DEST\n\
OPTION:\n\
\t-l\tlogin name\n\
\t-r\tcopy directories recursively\n\
\t-v\tdisplay what is being done\n\
\t-h\tprint this usage message\n\
");
}

/* callback function of authentication. */
static int supply_creds_server(void* userdata, const char* realm, int attempt,
			       char* username, char* password)
{
  char* user;
  char* passwd;
  char* msg;

  if (attempt > 1) {
    return -1;
  }

  user = (logname[0] != '\0') ? logname : getenv("LOGNAME");
  strcpy(username, user);
  msg = ne_concat(username, "'s password: ", NULL);
  passwd = getpass(msg); /* TODO: getpass is a obsolete function. */
  strcpy(password, passwd);
  free(msg);
  return 0;
}


static ne_session* session;
static int open_connection(ne_uri* uri)
{
  char* proxy_host;
  int proxy_port = 0;

  /* Create session. */
  session = ne_session_create(uri->scheme, uri->host, uri->port);

  /* Get the proxy details */
  proxy_host = getenv("http_proxy");
  if (proxy_host != NULL) {
    char* tmp;
    if (strncmp(proxy_host, "http://", 7) == 0) {
      proxy_host = &proxy_host[7];
    }
    for (tmp = proxy_host; *tmp && *tmp != ':'; tmp++);
    if (*tmp) {
      *tmp = '\0';
      proxy_port = atoi(tmp + 1);
    }
  }

  if (proxy_port == 0) {
    proxy_port = 8080;
  }

  if (proxy_host) {
    ne_session_proxy(session, proxy_host, proxy_port);
  }

  ne_set_useragent(session, PACKAGE "/" VERSION);
  ne_set_server_auth(session, supply_creds_server, NULL);

  return 0;
}

static void close_connection(void)
{
  ne_session_destroy(session);
}

static void sig_handler(int signo)
{
  fprintf(stderr, "davcp: terminated by signal: %d\n", signo);
  close_connection();
  exit(1);
}


static int get_method(const char* local, const char* remote)
{
  struct resource* reslist = NULL;
  char* col;
  struct stat statbuf;
  int ret;

  /* check whether 'remote' is a collection? */
  col = ne_concat(remote, "/", NULL);
  ret = fetch_resource_list(session, col, NE_DEPTH_ZERO, 1, &reslist);
  if (ret == NE_OK) {
    struct resource* current = reslist;
    if (current->type == resr_collection) {
      /* get collection */
      if (opt_recursive) {
	char* p;
	char* lpath;
	p = strrchr(remote, '/');
	lpath = ne_concat(local, p, NULL);
	ret = get_collection(session, lpath, remote);
	free(lpath);
      } else {
	fprintf(stderr, "davcp: not recursive copy mode.\n");
	ret = -1;
      }
      goto out;
    }
  } else {
    fprintf(stderr, "davcp: Failed: %s\n", ne_get_error(session));
    ret = -1;
    goto out;
  }

  /* get file */
  ret = stat(local, &statbuf);
  if (ret != -1 && statbuf.st_mode & S_IFMT) {
    char* p;
    p = strrchr(remote, '/');
    if (p != NULL) {
      char* lpath;
      lpath = ne_concat(local, p, NULL);
      ret = get_file(session, lpath, remote);
      free(lpath);
    } else {
      ret = -1;
    }
  } else {
    ret = get_file(session, local, remote);
  }

 out:
  free_resource_list(reslist);
  free(col);
  return ret;
}

static int put_method(const char* local, const char* remote)
{
  struct stat statbuf;
  int ret;

  ret = stat(local, &statbuf);
  if (ret == -1) {
    fprintf(stderr, "davcp: local file does not find: %s.\n", local);
    return ret;
  }

  switch (statbuf.st_mode & S_IFMT) {
  case S_IFREG:
    {
      struct resource* reslist = NULL;
      char* col;
      char* p;

      p = strrchr(local, '/');

      /* check whether 'remote' is a collection? */
      col = ne_concat(remote, "/", NULL);
      ret = fetch_resource_list(session, col, NE_DEPTH_ZERO, 1, &reslist);
      if (ret == NE_OK) { /* resource already exists */
	char* rpath;
	struct resource* current = reslist;
	if (current->type == resr_collection) {
	  rpath = ne_concat(current->uri, (p ? p + 1 : local), NULL);
	} else {
	  rpath = strdup(current->uri);
	}

	ret = put_file(session, local, rpath);
	free(rpath);
      } else {            /* resource does not exist */
	ret = put_file(session, local, remote);
      }
      free_resource_list(reslist);
      free(col);
      break;
    }
  case S_IFDIR:
    {
      if (opt_recursive) {
	ret = put_collection(session, local, remote);
      } else {
	fprintf(stderr, "davcp: not recursive copy mode.\n");
	ret = -1;
      }
      break;
    }
  default:
    ret = -1;
    break;
  }
  return ret;
}


int main(int argc, char** argv)
{
  ne_uri src_uri = {0};
  ne_uri dst_uri = {0};
  ne_uri *uri;
  char* local_path;
  int direction;
  int ch;

  while ((ch = getopt(argc, argv, "l:nrvh")) != EOF) {
    switch(ch) {
    case 'l':
      strncpy(logname, optarg, sizeof(logname));
      break;
    case 'r':
      opt_recursive = 1;
      break;
    case 'v':
      opt_verbose = 1;
      break;
    case 'h':
      usage();
      return 1;
    }
  }

  argc -= optind;
  argv += optind;
  if (argc < 2) {
    usage();
    return 1;
  }

  signal(SIGINT, sig_handler);

  ne_uri_parse(argv[0], &src_uri);
  ne_uri_parse(argv[1], &dst_uri);
  if (src_uri.scheme != NULL && dst_uri.scheme == NULL) {
    direction = METHOD_GET;
    uri = &src_uri;
    local_path = argv[1];
  } else if (src_uri.scheme == NULL && dst_uri.scheme != NULL) {
    direction = METHOD_PUT;
    local_path = argv[0];
    uri = &dst_uri;
  } else {
    fprintf(stderr, "invalid copy direction: %s -> %s\n", argv[0], argv[1]);
    ne_uri_free(&src_uri);
    ne_uri_free(&dst_uri);
    return 1;
  }

  /* Set defaults. */
  if (uri->port == 0) {
    uri->port = ne_uri_defaultport(uri->scheme);
  }
  if (uri->path == NULL) {
    uri->path = "/";
  }

  /* Initialize socket libraries */
  if (ne_sock_init()) {
    fprintf(stderr, "davcp: Failed to initialize socket libraries.\n");
    ne_uri_free(&src_uri);
    ne_uri_free(&dst_uri);
    return 1;
  }

  /* Initialize connection */
  open_connection(uri);

  /* Dispatch a request */
  if (ne_path_has_trailing_slash(local_path)) {
    local_path[strlen(local_path) - 1] = '\0';
  }
  if (ne_path_has_trailing_slash(uri->path)) {
    uri->path[strlen(uri->path) - 1] = '\0';
  }

  if (direction == METHOD_GET) {
    get_method(local_path, uri->path);
  } else if (direction == METHOD_PUT) {
    put_method(local_path, uri->path);
  }

  close_connection();
  ne_uri_free(&src_uri);
  ne_uri_free(&dst_uri);

  return 0;
}
