/* $Id: list.c,v 1.2 2003/01/09 12:48:07 rtakano Exp $
 * davcp: WebDAV copy
 * Copyright (c) 2002 TAKANO Ryousei <rtakano@sourceforge.jp>
 *
 * the most part of this file comes from 'ls.c' in cadaver 0.20.5.
 * Copyright (C) 2000-2001, Joe Orton <joe@manyfish.co.uk>.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <unistd.h>
#include <stdlib.h>
#include <string.h>

#include <ne_session.h>
#include <ne_props.h>
#include <ne_uri.h>
#include <ne_basic.h>
#include <ne_auth.h>
#include <ne_utils.h>

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include "davcp.h"

struct fetch_context {
  struct resource **list;
  const char *target; /* Request-URI of the PROPFIND */
  unsigned int include_target; /* Include resource at href */
};    

static const ne_propname list_props[] = {
  { "DAV:", "getcontentlength" },
  { "DAV:", "getlastmodified" },
  { "http://apache.org/dav/props/", "executable" },
  { "DAV:", "resourcetype" },
  { NULL }
};

#define ELM_resourcetype (NE_ELM_207_UNUSED + 1)
#define ELM_collection (NE_ELM_207_UNUSED + 4)

static const struct ne_xml_elm complex_elms[] = {
  { "DAV:", "resourcetype", ELM_resourcetype, 0 },
  { "DAV:", "collection", ELM_collection, 0 },
  { NULL }
};

static int compare_resource(const struct resource *r1, 
			    const struct resource *r2)
{
  /* Sort errors first, then collections, then alphabetically */
  if (r1->type == resr_error) {
    return -1;
  } else if (r2->type == resr_error) {
    return 1;
  } else if (r1->type == resr_collection) {
    if (r2->type != resr_collection) {
      return -1;
    } else {
      return strcmp(r1->uri, r2->uri);
    }
  } else {
    if (r2->type != resr_collection) {
      return strcmp(r1->uri, r2->uri);
    } else {
      return 1;
    }
  }
}

static void results(void *userdata, const char *uri,
		    const ne_prop_result_set *set)
{
  struct fetch_context *ctx = userdata;
  struct resource *current, *previous, *newres;
  const char *clength, *modtime, *isexec;
  const ne_status *status = NULL;
  ne_uri u;

  newres = ne_propset_private(set);
    
  if (ne_uri_parse(uri, &u))
    return;
    
  if (u.path == NULL) {
    ne_uri_free(&u);
    return;
  }

  if (ne_path_compare(ctx->target, u.path) == 0 && !ctx->include_target) {
    /* This is the target URI */
    free(newres);
    ne_uri_free(&u);
    return;
  }

  newres->uri = ne_strdup(u.path);

  clength = ne_propset_value(set, &list_props[0]);    
  modtime = ne_propset_value(set, &list_props[1]);
  isexec = ne_propset_value(set, &list_props[2]);
    
  if (clength == NULL)
    status = ne_propset_status(set, &list_props[0]);
  if (modtime == NULL)
    status = ne_propset_status(set, &list_props[1]);

  if (newres->type == resr_normal && status) {
    /* It's an error! */
    newres->error_status = status->code;
    newres->type = resr_error;
  }

  if (isexec && strcasecmp(isexec, "T") == 0) {
    newres->is_executable = 1;
  } else {
    newres->is_executable = 0;
  }

  if (clength)
    newres->size = atoi(clength);

  for (current = *ctx->list, previous = NULL; current != NULL; 
       previous = current, current=current->next) {
    if (compare_resource(current, newres) >= 0) {
      break;
    }
  }
  if (previous) {
    previous->next = newres;
  } else {
    *ctx->list = newres;
  }
  newres->next = current;

  ne_uri_free(&u);
}

static int end_element(void *userdata, const struct ne_xml_elm *elm, const char *cdata)
{
  ne_propfind_handler *pfh = userdata;
  struct resource *r = ne_propfind_current_private(pfh);

  if (r == NULL) {
    return 0;
  }

  if (elm->id == ELM_collection) {
    r->type = resr_collection;
  }

  return 0;
}

static int check_context(void *ud, ne_xml_elmid parent, ne_xml_elmid child)
{
  if ((parent == NE_ELM_prop && child == ELM_resourcetype) ||
      (parent == ELM_resourcetype && child == ELM_collection)) {
    return NE_XML_VALID;
  }
  return NE_XML_DECLINE;
}

void free_resource(struct resource *res)
{
  NE_FREE(res->uri);
  free(res);
}

void free_resource_list(struct resource *res)
{
  struct resource *next;
  for (; res != NULL; res = next) {
    next = res->next;
    free_resource(res);
  }
}

static void *create_private(void *userdata, const char *uri)
{
  return ne_calloc(sizeof(struct resource));
}

int fetch_resource_list(ne_session *sess, const char *uri,
			int depth, int include_target,
			struct resource **reslist)
{
  ne_propfind_handler *pfh = ne_propfind_create(sess, uri, depth);
  int ret;
  struct fetch_context ctx = {0};

  *reslist = NULL;
  ctx.list = reslist;
  ctx.target = uri;
  ctx.include_target = include_target;

  ne_xml_push_handler(ne_propfind_get_parser(pfh), complex_elms, 
		      check_context, NULL, end_element, pfh);

  ne_propfind_set_private(pfh, create_private, NULL);

  ret = ne_propfind_named(pfh, list_props, results, &ctx);

  ne_propfind_destroy(pfh);

  return ret;
}
