"""
**256 bit Secure Hash Algorithm (SHA-256)**
"""
#====================================================================
# SHA256 python module
#
# Copyright (c) 2001, Bryan Mongeau <bryan@eevolved.com>
# All rights reserved.
#
# This code is hereby placed in the public domain.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# 1- Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
#
# 2- Redistributions in binary form must reproduce the above copyright notice,
#   this list of conditions and the following disclaimer in the documentation
#   and/or other materials provided with the distribution.
#
# 3- The names of its contributors may not be used to endorse or promote
#    products derived from this software without specific prior written
#    permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
# ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
# LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
# A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR
# CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
# EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
# PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
# PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
# LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
#  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
# SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
#====================================================================


from sha256c import *

class sha256:

  """
  **256 bit Secure Hash Algorithm (SHA-256)**

  The sha256 python class builds on NIST's SHA 256
  implementation written in ANSI C by Aaron D. Gifford.
  The interface is very much like the standard python
  sha, the primary difference being the length of the
  digest. This class extends sha from 160 to 256 bits.

  Usage Example::

    >>> from sha256.sha256 import sha256

    >>> hasher = sha256()

    >>> hasher.update('1234567890')

    >>> hasher.digest()

    'w\371\337\211\226q1\221\325\320\301?\355-M\377\255\301\223\251\360\252\023\222p@\205\022\015\357\330\263'

    >>> hasher.update('1234567890')

    >>> hasher.digest()

    'P~\374x\215\214\3123\204\2132\267c\023\244j +&\333\317\200A\331k:\305/\351\370\362q'


  The digest returned will invariably be 32 bytes (256 bits).
  """

  def __init__(self, bytes=None):

    """
    Digest Context Initialisation.
    A new digesting context is created. Optionally,
    a string can be hashed immediately.
    """

    self.ctx = new_SHA256_CTX()
    if bytes:
      self.update(bytes)


  def update(self, bytes):

    """
    Update the contents of the digest buffer by appending
    bytes to it. Accepts python strings with or without
    NULL bytes.
    """
    if str(type(bytes)) != "<type 'string'>" and len(bytes) > 0:
      raise TypeError("Please supply a string.")

    SHA256_Update(self.ctx,bytes,len(bytes))

  def digest(self, clear=0):

    """
    Execute an SHA256 hash on the data stored in the context
    buffer. Return the digest in a python string with NULL
    bytes. Set *clear* to 1 if you wish to reset the digest
    context.
    """
    if clear:
      del self.ctx
      self.ctx = new_SHA256_CTX()
    return SHA256_End(self.ctx)













