# Version: JEDEC JEP106BI.pdf

__all__ = ["jedec_mfg_name_from_bytes", "jedec_mfg_name_from_bank_num"]


def jedec_mfg_name_from_bytes(mfg_ids):
    bank = 0
    for mfg_id in mfg_ids:
        if mfg_id == 0x7F:
            bank += 1
        else:
            if bank >= len(_manufacturers):
                return None
            for (number, name, code) in _manufacturers[bank]:
                if code == mfg_id:
                    return name
            return None


def jedec_mfg_name_from_bank_num(bank, mfg_num):
    if bank >= len(_manufacturers):
        return None
    for (number, name, code) in _manufacturers[bank]:
        if number == mfg_num:
            return name
    return None

# Note: _manufacturers array is autogenerated
_manufacturers = [
    [ # Bank 1
        (1, "AMD", 0x01),
        (2, "AMI", 0x02),
        (3, "Fairchild", 0x83),
        (4, "Fujitsu", 0x04),
        (5, "GTE", 0x85),
        (6, "Harris", 0x86),
        (7, "Hitachi", 0x07),
        (8, "Inmos", 0x08),
        (9, "Intel", 0x89),
        (10, "I.T.T.", 0x8A),
        (11, "Intersil", 0x0B),
        (12, "Monolithic Memories", 0x8C),
        (13, "Mostek", 0x0D),
        (14, "Freescale (Motorola)", 0x0E),
        (15, "National", 0x8F),
        (16, "NEC", 0x10),
        (17, "RCA", 0x91),
        (18, "Raytheon", 0x92),
        (19, "Conexant (Rockwell)", 0x13),
        (20, "Seeq", 0x94),
        (21, "NXP (Philips)", 0x15),
        (22, "Synertek", 0x16),
        (23, "Texas Instruments", 0x97),
        (24, "Kioxia Corporation", 0x98),
        (25, "Xicor", 0x19),
        (26, "Zilog", 0x1A),
        (27, "Eurotechnique", 0x9B),
        (28, "Mitsubishi", 0x1C),
        (29, "Lucent (AT&T)", 0x9D),
        (30, "Exel", 0x9E),
        (31, "Atmel", 0x1F),
        (32, "STMicroelectronics", 0x20),
        (33, "Lattice Semi.", 0xA1),
        (34, "NCR", 0xA2),
        (35, "Wafer Scale Integration", 0x23),
        (36, "IBM", 0xA4),
        (37, "Tristar", 0x25),
        (38, "Visic", 0x26),
        (39, "Intl. CMOS Technology", 0xA7),
        (40, "SSSI", 0xA8),
        (41, "Microchip Technology", 0x29),
        (42, "Ricoh Ltd", 0x2A),
        (43, "VLSI", 0xAB),
        (44, "Micron Technology", 0x2C),
        (45, "SK Hynix", 0xAD),
        (46, "OKI Semiconductor", 0xAE),
        (47, "ACTEL", 0x2F),
        (48, "Sharp", 0xB0),
        (49, "Catalyst", 0x31),
        (50, "Panasonic", 0x32),
        (51, "IDT", 0xB3),
        (52, "Cypress", 0x34),
        (53, "DEC", 0xB5),
        (54, "LSI Logic", 0xB6),
        (55, "Zarlink (Plessey)", 0x37),
        (56, "UTMC", 0x38),
        (57, "Thinking Machine", 0xB9),
        (58, "Thomson CSF", 0xBA),
        (59, "Integrated CMOS (Vertex)", 0x3B),
        (60, "Honeywell", 0xBC),
        (61, "Tektronix", 0x3D),
        (62, "Oracle Corporation", 0x3E),
        (63, "Silicon Storage Technology", 0xBF),
        (64, "ProMos/Mosel Vitelic", 0x40),
        (65, "Infineon (Siemens)", 0xC1),
        (66, "Macronix", 0xC2),
        (67, "Xerox", 0x43),
        (68, "Plus Logic", 0xC4),
        (69, "Western Digital Technologies Inc", 0x45),
        (70, "Elan Circuit Tech.", 0x46),
        (71, "European Silicon Str.", 0xC7),
        (72, "Apple Computer", 0xC8),
        (73, "Xilinx", 0x49),
        (74, "Compaq", 0x4A),
        (75, "Protocol Engines", 0xCB),
        (76, "SCI", 0x4C),
        (77, "ABLIC", 0xCD),
        (78, "Samsung", 0xCE),
        (79, "I3 Design System", 0x4F),
        (80, "Klic", 0xD0),
        (81, "Crosspoint Solutions", 0x51),
        (82, "Alliance Memory Inc", 0x52),
        (83, "Tandem", 0xD3),
        (84, "Hewlett-Packard", 0x54),
        (85, "Integrated Silicon Solutions", 0xD5),
        (86, "Brooktree", 0xD6),
        (87, "New Media", 0x57),
        (88, "MHS Electronic", 0x58),
        (89, "Performance Semi.", 0xD9),
        (90, "Winbond Electronic", 0xDA),
        (91, "Kawasaki Steel", 0x5B),
        (92, "Bright Micro", 0xDC),
        (93, "TECMAR", 0x5D),
        (94, "Exar", 0x5E),
        (95, "PCMCIA", 0xDF),
        (96, "LG Semi (Goldstar)", 0xE0),
        (97, "Northern Telecom", 0x61),
        (98, "Sanyo", 0x62),
        (99, "Array Microsystems", 0xE3),
        (100, "Crystal Semiconductor", 0x64),
        (101, "Analog Devices", 0xE5),
        (102, "PMC-Sierra", 0xE6),
        (103, "Asparix", 0x67),
        (104, "Convex Computer", 0x68),
        (105, "Quality Semiconductor", 0xE9),
        (106, "Nimbus Technology", 0xEA),
        (107, "Transwitch", 0x6B),
        (108, "Micronas (ITT Intermetall)", 0xEC),
        (109, "Cannon", 0x6D),
        (110, "Altera", 0x6E),
        (111, "NEXCOM", 0xEF),
        (112, "Qualcomm", 0x70),
        (113, "Sony", 0xF1),
        (114, "Cray Research", 0xF2),
        (115, "AMS(Austria Micro)", 0x73),
        (116, "Vitesse", 0xF4),
        (117, "Aster Electronics", 0x75),
        (118, "Bay Networks (Synoptic)", 0x76),
        (119, "Zentrum/ZMD", 0xF7),
        (120, "TRW", 0xF8),
        (121, "Thesys", 0x79),
        (122, "Solbourne Computer", 0x7A),
        (123, "Allied-Signal", 0xFB),
        (124, "Dialog Semiconductor", 0x7C),
        (125, "Media Vision", 0xFD),
        (126, "Numonyx Corporation", 0xFE),
    ],
    [ # Bank 2
        (1, "Cirrus Logic", 0x01),
        (2, "National Instruments", 0x02),
        (3, "ILC Data Device", 0x83),
        (4, "Alcatel Mietec", 0x04),
        (5, "Micro Linear", 0x85),
        (6, "Univ. of NC", 0x86),
        (7, "JTAG Technologies", 0x07),
        (8, "BAE Systems (Loral)", 0x08),
        (9, "Nchip", 0x89),
        (10, "Galileo Tech", 0x8A),
        (11, "Bestlink Systems", 0x0B),
        (12, "Graychip", 0x8C),
        (13, "GENNUM", 0x0D),
        (14, "Imagination Technologies Limited", 0x0E),
        (15, "Robert Bosch", 0x8F),
        (16, "Chip Express", 0x10),
        (17, "DATARAM", 0x91),
        (18, "United Microelectronics Corp", 0x92),
        (19, "TCSI", 0x13),
        (20, "Smart Modular", 0x94),
        (21, "Hughes Aircraft", 0x15),
        (22, "Lanstar Semiconductor", 0x16),
        (23, "Qlogic", 0x97),
        (24, "Kingston", 0x98),
        (25, "Music Semi", 0x19),
        (26, "Ericsson Components", 0x1A),
        (27, "SpaSE", 0x9B),
        (28, "Eon Silicon Devices", 0x1C),
        (29, "Integrated Silicon Solution (ISSI)", 0x9D),
        (30, "DoD", 0x9E),
        (31, "Integ. Memories Tech.", 0x1F),
        (32, "Corollary Inc", 0x20),
        (33, "Dallas Semiconductor", 0xA1),
        (34, "Omnivision", 0xA2),
        (35, "EIV(Switzerland)", 0x23),
        (36, "Novatel Wireless", 0xA4),
        (37, "Zarlink (Mitel)", 0x25),
        (38, "Clearpoint", 0x26),
        (39, "Cabletron", 0xA7),
        (40, "STEC (Silicon Tech)", 0xA8),
        (41, "Vanguard", 0x29),
        (42, "Hagiwara Sys-Com", 0x2A),
        (43, "Vantis", 0xAB),
        (44, "Celestica", 0x2C),
        (45, "Century", 0xAD),
        (46, "Hal Computers", 0xAE),
        (47, "Rohm Company Ltd", 0x2F),
        (48, "Juniper Networks", 0xB0),
        (49, "Libit Signal Processing", 0x31),
        (50, "Mushkin Enhanced Memory", 0x32),
        (51, "Tundra Semiconductor", 0xB3),
        (52, "Adaptec Inc", 0x34),
        (53, "LightSpeed Semi.", 0xB5),
        (54, "ZSP Corp", 0xB6),
        (55, "AMIC Technology", 0x37),
        (56, "Adobe Systems", 0x38),
        (57, "Dynachip", 0xB9),
        (58, "PNY Technologies Inc", 0xBA),
        (59, "Newport Digital", 0x3B),
        (60, "MMC Networks", 0xBC),
        (61, "T Square", 0x3D),
        (62, "Seiko Epson", 0x3E),
        (63, "Broadcom", 0xBF),
        (64, "Viking Components", 0x40),
        (65, "V3 Semiconductor", 0xC1),
        (66, "Flextronics (Orbit Semiconductor)", 0xC2),
        (67, "Suwa Electronics", 0x43),
        (68, "Transmeta", 0xC4),
        (69, "Micron CMS", 0x45),
        (70, "American Computer & Digital Components Inc", 0x46),
        (71, "Enhance 3000 Inc", 0xC7),
        (72, "Tower Semiconductor", 0xC8),
        (73, "CPU Design", 0x49),
        (74, "Price Point", 0x4A),
        (75, "Maxim Integrated Product", 0xCB),
        (76, "Tellabs", 0x4C),
        (77, "Centaur Technology", 0xCD),
        (78, "Unigen Corporation", 0xCE),
        (79, "Transcend Information", 0x4F),
        (80, "Memory Card Technology", 0xD0),
        (81, "CKD Corporation Ltd", 0x51),
        (82, "Capital Instruments Inc", 0x52),
        (83, "Aica Kogyo Ltd", 0xD3),
        (84, "Linvex Technology", 0x54),
        (85, "MSC Vertriebs GmbH", 0xD5),
        (86, "AKM Company Ltd", 0xD6),
        (87, "Dynamem Inc", 0x57),
        (88, "NERA ASA", 0x58),
        (89, "GSI Technology", 0xD9),
        (90, "Dane-Elec (C Memory)", 0xDA),
        (91, "Acorn Computers", 0x5B),
        (92, "Lara Technology", 0xDC),
        (93, "Oak Technology Inc", 0x5D),
        (94, "Itec Memory", 0x5E),
        (95, "Tanisys Technology", 0xDF),
        (96, "Truevision", 0xE0),
        (97, "Wintec Industries", 0x61),
        (98, "Super PC Memory", 0x62),
        (99, "MGV Memory", 0xE3),
        (100, "Galvantech", 0x64),
        (101, "Gadzoox Networks", 0xE5),
        (102, "Multi Dimensional Cons.", 0xE6),
        (103, "GateField", 0x67),
        (104, "Integrated Memory System", 0x68),
        (105, "Triscend", 0xE9),
        (106, "XaQti", 0xEA),
        (107, "Goldenram", 0x6B),
        (108, "Clear Logic", 0xEC),
        (109, "Cimaron Communications", 0x6D),
        (110, "Nippon Steel Semi. Corp", 0x6E),
        (111, "Advantage Memory", 0xEF),
        (112, "AMCC", 0x70),
        (113, "LeCroy", 0xF1),
        (114, "Yamaha Corporation", 0xF2),
        (115, "Digital Microwave", 0x73),
        (116, "NetLogic Microsystems", 0xF4),
        (117, "MIMOS Semiconductor", 0x75),
        (118, "Advanced Fibre", 0x76),
        (119, "BF Goodrich Data.", 0xF7),
        (120, "Epigram", 0xF8),
        (121, "Acbel Polytech Inc", 0x79),
        (122, "Apacer Technology", 0x7A),
        (123, "Admor Memory", 0xFB),
        (124, "FOXCONN", 0x7C),
        (125, "Quadratics Superconductor", 0xFD),
        (126, "3COM", 0xFE),
    ],
    [ # Bank 3
        (1, "Camintonn Corporation", 0x01),
        (2, "ISOA Incorporated", 0x02),
        (3, "Agate Semiconductor", 0x83),
        (4, "ADMtek Incorporated", 0x04),
        (5, "HYPERTEC", 0x85),
        (6, "Adhoc Technologies", 0x86),
        (7, "MOSAID Technologies", 0x07),
        (8, "Ardent Technologies", 0x08),
        (9, "Switchcore", 0x89),
        (10, "Cisco Systems Inc", 0x8A),
        (11, "Allayer Technologies", 0x0B),
        (12, "WorkX AG (Wichman)", 0x8C),
        (13, "Oasis Semiconductor", 0x0D),
        (14, "Novanet Semiconductor", 0x0E),
        (15, "E-M Solutions", 0x8F),
        (16, "Power General", 0x10),
        (17, "Advanced Hardware Arch.", 0x91),
        (18, "Inova Semiconductors GmbH", 0x92),
        (19, "Telocity", 0x13),
        (20, "Delkin Devices", 0x94),
        (21, "Symagery Microsystems", 0x15),
        (22, "C-Port Corporation", 0x16),
        (23, "SiberCore Technologies", 0x97),
        (24, "Southland Microsystems", 0x98),
        (25, "Malleable Technologies", 0x19),
        (26, "Kendin Communications", 0x1A),
        (27, "Great Technology Microcomputer", 0x9B),
        (28, "Sanmina Corporation", 0x1C),
        (29, "HADCO Corporation", 0x9D),
        (30, "Corsair", 0x9E),
        (31, "Actrans System Inc", 0x1F),
        (32, "ALPHA Technologies", 0x20),
        (33, "Silicon Laboratories Inc (Cygnal)", 0xA1),
        (34, "Artesyn Technologies", 0xA2),
        (35, "Align Manufacturing", 0x23),
        (36, "Peregrine Semiconductor", 0xA4),
        (37, "Chameleon Systems", 0x25),
        (38, "Aplus Flash Technology", 0x26),
        (39, "MIPS Technologies", 0xA7),
        (40, "Chrysalis ITS", 0xA8),
        (41, "ADTEC Corporation", 0x29),
        (42, "Kentron Technologies", 0x2A),
        (43, "Win Technologies", 0xAB),
        (44, "Tezzaron Semiconductor", 0x2C),
        (45, "Extreme Packet Devices", 0xAD),
        (46, "RF Micro Devices", 0xAE),
        (47, "Siemens AG", 0x2F),
        (48, "Sarnoff Corporation", 0xB0),
        (49, "Itautec SA", 0x31),
        (50, "Radiata Inc", 0x32),
        (51, "Benchmark Elect. (AVEX)", 0xB3),
        (52, "Legend", 0x34),
        (53, "SpecTek Incorporated", 0xB5),
        (54, "Hi/fn", 0xB6),
        (55, "Enikia Incorporated", 0x37),
        (56, "SwitchOn Networks", 0x38),
        (57, "AANetcom Incorporated", 0xB9),
        (58, "Micro Memory Bank", 0xBA),
        (59, "ESS Technology", 0x3B),
        (60, "Virata Corporation", 0xBC),
        (61, "Excess Bandwidth", 0x3D),
        (62, "West Bay Semiconductor", 0x3E),
        (63, "DSP Group", 0xBF),
        (64, "Newport Communications", 0x40),
        (65, "Chip2Chip Incorporated", 0xC1),
        (66, "Phobos Corporation", 0xC2),
        (67, "Intellitech Corporation", 0x43),
        (68, "Nordic VLSI ASA", 0xC4),
        (69, "Ishoni Networks", 0x45),
        (70, "Silicon Spice", 0x46),
        (71, "Alchemy Semiconductor", 0xC7),
        (72, "Agilent Technologies", 0xC8),
        (73, "Centillium Communications", 0x49),
        (74, "W.L. Gore", 0x4A),
        (75, "HanBit Electronics", 0xCB),
        (76, "GlobeSpan", 0x4C),
        (77, "Element 14", 0xCD),
        (78, "Pycon", 0xCE),
        (79, "Saifun Semiconductors", 0x4F),
        (80, "Sibyte Incorporated", 0xD0),
        (81, "MetaLink Technologies", 0x51),
        (82, "Feiya Technology", 0x52),
        (83, "I & C Technology", 0xD3),
        (84, "Shikatronics", 0x54),
        (85, "Elektrobit", 0xD5),
        (86, "Megic", 0xD6),
        (87, "Com-Tier", 0x57),
        (88, "Malaysia Micro Solutions", 0x58),
        (89, "Hyperchip", 0xD9),
        (90, "Gemstone Communications", 0xDA),
        (91, "Anadigm (Anadyne)", 0x5B),
        (92, "3ParData", 0xDC),
        (93, "Mellanox Technologies", 0x5D),
        (94, "Tenx Technologies", 0x5E),
        (95, "Helix AG", 0xDF),
        (96, "Domosys", 0xE0),
        (97, "Skyup Technology", 0x61),
        (98, "HiNT Corporation", 0x62),
        (99, "Chiaro", 0xE3),
        (100, "MDT Technologies GmbH", 0x64),
        (101, "Exbit Technology A/S", 0xE5),
        (102, "Integrated Technology Express", 0xE6),
        (103, "AVED Memory", 0x67),
        (104, "Legerity", 0x68),
        (105, "Jasmine Networks", 0xE9),
        (106, "Caspian Networks", 0xEA),
        (107, "nCUBE", 0x6B),
        (108, "Silicon Access Networks", 0xEC),
        (109, "FDK Corporation", 0x6D),
        (110, "High Bandwidth Access", 0x6E),
        (111, "MultiLink Technology", 0xEF),
        (112, "BRECIS", 0x70),
        (113, "World Wide Packets", 0xF1),
        (114, "APW", 0xF2),
        (115, "Chicory Systems", 0x73),
        (116, "Xstream Logic", 0xF4),
        (117, "Fast-Chip", 0x75),
        (118, "Zucotto Wireless", 0x76),
        (119, "Realchip", 0xF7),
        (120, "Galaxy Power", 0xF8),
        (121, "eSilicon", 0x79),
        (122, "Morphics Technology", 0x7A),
        (123, "Accelerant Networks", 0xFB),
        (124, "Silicon Wave", 0x7C),
        (125, "SandCraft", 0xFD),
        (126, "Elpida", 0xFE),
    ],
    [ # Bank 4
        (1, "Solectron", 0x01),
        (2, "Optosys Technologies", 0x02),
        (3, "Buffalo (Formerly Melco)", 0x83),
        (4, "TriMedia Technologies", 0x04),
        (5, "Cyan Technologies", 0x85),
        (6, "Global Locate", 0x86),
        (7, "Optillion", 0x07),
        (8, "Terago Communications", 0x08),
        (9, "Ikanos Communications", 0x89),
        (10, "Princeton Technology", 0x8A),
        (11, "Nanya Technology", 0x0B),
        (12, "Elite Flash Storage", 0x8C),
        (13, "Mysticom", 0x0D),
        (14, "LightSand Communications", 0x0E),
        (15, "ATI Technologies", 0x8F),
        (16, "Agere Systems", 0x10),
        (17, "NeoMagic", 0x91),
        (18, "AuroraNetics", 0x92),
        (19, "Golden Empire", 0x13),
        (20, "Mushkin", 0x94),
        (21, "Tioga Technologies", 0x15),
        (22, "Netlist", 0x16),
        (23, "TeraLogic", 0x97),
        (24, "Cicada Semiconductor", 0x98),
        (25, "Centon Electronics", 0x19),
        (26, "Tyco Electronics", 0x1A),
        (27, "Magis Works", 0x9B),
        (28, "Zettacom", 0x1C),
        (29, "Cogency Semiconductor", 0x9D),
        (30, "Chipcon AS", 0x9E),
        (31, "Aspex Technology", 0x1F),
        (32, "F5 Networks", 0x20),
        (33, "Programmable Silicon Solutions", 0xA1),
        (34, "ChipWrights", 0xA2),
        (35, "Acorn Networks", 0x23),
        (36, "Quicklogic", 0xA4),
        (37, "Kingmax Semiconductor", 0x25),
        (38, "BOPS", 0x26),
        (39, "Flasys", 0xA7),
        (40, "BitBlitz Communications", 0xA8),
        (41, "eMemory Technology", 0x29),
        (42, "Procket Networks", 0x2A),
        (43, "Purple Ray", 0xAB),
        (44, "Trebia Networks", 0x2C),
        (45, "Delta Electronics", 0xAD),
        (46, "Onex Communications", 0xAE),
        (47, "Ample Communications", 0x2F),
        (48, "Memory Experts Intl", 0xB0),
        (49, "Astute Networks", 0x31),
        (50, "Azanda Network Devices", 0x32),
        (51, "Dibcom", 0xB3),
        (52, "Tekmos", 0x34),
        (53, "API NetWorks", 0xB5),
        (54, "Bay Microsystems", 0xB6),
        (55, "Firecron Ltd", 0x37),
        (56, "Resonext Communications", 0x38),
        (57, "Tachys Technologies", 0xB9),
        (58, "Equator Technology", 0xBA),
        (59, "Concept Computer", 0x3B),
        (60, "SILCOM", 0xBC),
        (61, "3Dlabs", 0x3D),
        (62, "c’t Magazine", 0x3E),
        (63, "Sanera Systems", 0xBF),
        (64, "Silicon Packets", 0x40),
        (65, "Viasystems Group", 0xC1),
        (66, "Simtek", 0xC2),
        (67, "Semicon Devices Singapore", 0x43),
        (68, "Satron Handelsges", 0xC4),
        (69, "Improv Systems", 0x45),
        (70, "INDUSYS GmbH", 0x46),
        (71, "Corrent", 0xC7),
        (72, "Infrant Technologies", 0xC8),
        (73, "Ritek Corp", 0x49),
        (74, "empowerTel Networks", 0x4A),
        (75, "Hypertec", 0xCB),
        (76, "Cavium Networks", 0x4C),
        (77, "PLX Technology", 0xCD),
        (78, "Massana Design", 0xCE),
        (79, "Intrinsity", 0x4F),
        (80, "Valence Semiconductor", 0xD0),
        (81, "Terawave Communications", 0x51),
        (82, "IceFyre Semiconductor", 0x52),
        (83, "Primarion", 0xD3),
        (84, "Picochip Designs Ltd", 0x54),
        (85, "Silverback Systems", 0xD5),
        (86, "Jade Star Technologies", 0xD6),
        (87, "Pijnenburg Securealink", 0x57),
        (88, "takeMS - Ultron AG", 0x58),
        (89, "Cambridge Silicon Radio", 0xD9),
        (90, "Swissbit", 0xDA),
        (91, "Nazomi Communications", 0x5B),
        (92, "eWave System", 0xDC),
        (93, "Rockwell Collins", 0x5D),
        (94, "Picocel Co Ltd (Paion)", 0x5E),
        (95, "Alphamosaic Ltd", 0xDF),
        (96, "Sandburst", 0xE0),
        (97, "SiCon Video", 0x61),
        (98, "NanoAmp Solutions", 0x62),
        (99, "Ericsson Technology", 0xE3),
        (100, "PrairieComm", 0x64),
        (101, "Mitac International", 0xE5),
        (102, "Layer N Networks", 0xE6),
        (103, "MtekVision (Atsana)", 0x67),
        (104, "Allegro Networks", 0x68),
        (105, "Marvell Semiconductors", 0xE9),
        (106, "Netergy Microelectronic", 0xEA),
        (107, "NVIDIA", 0x6B),
        (108, "Internet Machines", 0xEC),
        (109, "Memorysolution GmbH", 0x6D),
        (110, "Litchfield Communication", 0x6E),
        (111, "Accton Technology", 0xEF),
        (112, "Teradiant Networks", 0x70),
        (113, "Scaleo Chip", 0xF1),
        (114, "Cortina Systems", 0xF2),
        (115, "RAM Components", 0x73),
        (116, "Raqia Networks", 0xF4),
        (117, "ClearSpeed", 0x75),
        (118, "Matsushita Battery", 0x76),
        (119, "Xelerated", 0xF7),
        (120, "SimpleTech", 0xF8),
        (121, "Utron Technology", 0x79),
        (122, "Astec International", 0x7A),
        (123, "AVM gmbH", 0xFB),
        (124, "Redux Communications", 0x7C),
        (125, "Dot Hill Systems", 0xFD),
        (126, "TeraChip", 0xFE),
    ],
    [ # Bank 5
        (1, "T-RAM Incorporated", 0x01),
        (2, "Innovics Wireless", 0x02),
        (3, "Teknovus", 0x83),
        (4, "KeyEye Communications", 0x04),
        (5, "Runcom Technologies", 0x85),
        (6, "RedSwitch", 0x86),
        (7, "Dotcast", 0x07),
        (8, "Silicon Mountain Memory", 0x08),
        (9, "Signia Technologies", 0x89),
        (10, "Pixim", 0x8A),
        (11, "Galazar Networks", 0x0B),
        (12, "White Electronic Designs", 0x8C),
        (13, "Patriot Scientific", 0x0D),
        (14, "Neoaxiom Corporation", 0x0E),
        (15, "3Y Power Technology", 0x8F),
        (16, "Scaleo Chip", 0x10),
        (17, "Potentia Power Systems", 0x91),
        (18, "C-guys Incorporated", 0x92),
        (19, "Digital Communications Technology Inc", 0x13),
        (20, "Silicon-Based Technology", 0x94),
        (21, "Fulcrum Microsystems", 0x15),
        (22, "Positivo Informatica Ltd", 0x16),
        (23, "XIOtech Corporation", 0x97),
        (24, "PortalPlayer", 0x98),
        (25, "Zhiying Software", 0x19),
        (26, "ParkerVision Inc", 0x1A),
        (27, "Phonex Broadband", 0x9B),
        (28, "Skyworks Solutions", 0x1C),
        (29, "Entropic Communications", 0x9D),
        (30, "I’M Intelligent Memory Ltd", 0x9E),
        (31, "Zensys A/S", 0x1F),
        (32, "Legend Silicon Corp", 0x20),
        (33, "Sci-worx GmbH", 0xA1),
        (34, "SMSC (Standard Microsystems)", 0xA2),
        (35, "Renesas Electronics", 0x23),
        (36, "Raza Microelectronics", 0xA4),
        (37, "Phyworks", 0x25),
        (38, "MediaTek", 0x26),
        (39, "Non-cents Productions", 0xA7),
        (40, "US Modular", 0xA8),
        (41, "Wintegra Ltd", 0x29),
        (42, "Mathstar", 0x2A),
        (43, "StarCore", 0xAB),
        (44, "Oplus Technologies", 0x2C),
        (45, "Mindspeed", 0xAD),
        (46, "Just Young Computer", 0xAE),
        (47, "Radia Communications", 0x2F),
        (48, "OCZ", 0xB0),
        (49, "Emuzed", 0x31),
        (50, "LOGIC Devices", 0x32),
        (51, "Inphi Corporation", 0xB3),
        (52, "Quake Technologies", 0x34),
        (53, "Vixel", 0xB5),
        (54, "SolusTek", 0xB6),
        (55, "Kongsberg Maritime", 0x37),
        (56, "Faraday Technology", 0x38),
        (57, "Altium Ltd", 0xB9),
        (58, "Insyte", 0xBA),
        (59, "ARM Ltd", 0x3B),
        (60, "DigiVision", 0xBC),
        (61, "Vativ Technologies", 0x3D),
        (62, "Endicott Interconnect Technologies", 0x3E),
        (63, "Pericom", 0xBF),
        (64, "Bandspeed", 0x40),
        (65, "LeWiz Communications", 0xC1),
        (66, "CPU Technology", 0xC2),
        (67, "Ramaxel Technology", 0x43),
        (68, "DSP Group", 0xC4),
        (69, "Axis Communications", 0x45),
        (70, "Legacy Electronics", 0x46),
        (71, "Chrontel", 0xC7),
        (72, "Powerchip Semiconductor", 0xC8),
        (73, "MobilEye Technologies", 0x49),
        (74, "Excel Semiconductor", 0x4A),
        (75, "A-DATA Technology", 0xCB),
        (76, "VirtualDigm", 0x4C),
        (77, "G Skill Intl", 0xCD),
        (78, "Quanta Computer", 0xCE),
        (79, "Yield Microelectronics", 0x4F),
        (80, "Afa Technologies", 0xD0),
        (81, "KINGBOX Technology Co Ltd", 0x51),
        (82, "Ceva", 0x52),
        (83, "iStor Networks", 0xD3),
        (84, "Advance Modules", 0x54),
        (85, "Microsoft", 0xD5),
        (86, "Open-Silicon", 0xD6),
        (87, "Goal Semiconductor", 0x57),
        (88, "ARC International", 0x58),
        (89, "Simmtec", 0xD9),
        (90, "Metanoia", 0xDA),
        (91, "Key Stream", 0x5B),
        (92, "Lowrance Electronics", 0xDC),
        (93, "Adimos", 0x5D),
        (94, "SiGe Semiconductor", 0x5E),
        (95, "Fodus Communications", 0xDF),
        (96, "Credence Systems Corp", 0xE0),
        (97, "Genesis Microchip Inc", 0x61),
        (98, "Vihana Inc", 0x62),
        (99, "WIS Technologies", 0xE3),
        (100, "GateChange Technologies", 0x64),
        (101, "High Density Devices AS", 0xE5),
        (102, "Synopsys", 0xE6),
        (103, "Gigaram", 0x67),
        (104, "Enigma Semiconductor Inc", 0x68),
        (105, "Century Micro Inc", 0xE9),
        (106, "Icera Semiconductor", 0xEA),
        (107, "Mediaworks Integrated Systems", 0x6B),
        (108, "O’Neil Product Development", 0xEC),
        (109, "Supreme Top Technology Ltd", 0x6D),
        (110, "MicroDisplay Corporation", 0x6E),
        (111, "Team Group Inc", 0xEF),
        (112, "Sinett Corporation", 0x70),
        (113, "Toshiba Corporation", 0xF1),
        (114, "Tensilica", 0xF2),
        (115, "SiRF Technology", 0x73),
        (116, "Bacoc Inc", 0xF4),
        (117, "SMaL Camera Technologies", 0x75),
        (118, "Thomson SC", 0x76),
        (119, "Airgo Networks", 0xF7),
        (120, "Wisair Ltd", 0xF8),
        (121, "SigmaTel", 0x79),
        (122, "Arkados", 0x7A),
        (123, "Compete IT gmbH Co KG", 0xFB),
        (124, "Eudar Technology Inc", 0x7C),
        (125, "Focus Enhancements", 0xFD),
        (126, "Xyratex", 0xFE),
    ],
    [ # Bank 6
        (1, "Specular Networks", 0x01),
        (2, "Patriot Memory (PDP Systems)", 0x02),
        (3, "U-Chip Technology Corp", 0x83),
        (4, "Silicon Optix", 0x04),
        (5, "Greenfield Networks", 0x85),
        (6, "CompuRAM GmbH", 0x86),
        (7, "Stargen Inc", 0x07),
        (8, "NetCell Corporation", 0x08),
        (9, "Excalibrus Technologies Ltd", 0x89),
        (10, "SCM Microsystems", 0x8A),
        (11, "Xsigo Systems Inc", 0x0B),
        (12, "CHIPS & Systems Inc", 0x8C),
        (13, "Tier 1 Multichip Solutions", 0x0D),
        (14, "CWRL Labs", 0x0E),
        (15, "Teradici", 0x8F),
        (16, "Gigaram Inc", 0x10),
        (17, "g2 Microsystems", 0x91),
        (18, "PowerFlash Semiconductor", 0x92),
        (19, "P.A. Semi Inc", 0x13),
        (20, "NovaTech Solutions S.A.", 0x94),
        (21, "c2 Microsystems Inc", 0x15),
        (22, "Level5 Networks", 0x16),
        (23, "COS Memory AG", 0x97),
        (24, "Innovasic Semiconductor", 0x98),
        (25, "02IC Co Ltd", 0x19),
        (26, "Tabula Inc", 0x1A),
        (27, "Crucial Technology", 0x9B),
        (28, "Chelsio Communications", 0x1C),
        (29, "Solarflare Communications", 0x9D),
        (30, "Xambala Inc", 0x9E),
        (31, "EADS Astrium", 0x1F),
        (32, "Terra Semiconductor Inc", 0x20),
        (33, "Imaging Works Inc", 0xA1),
        (34, "Astute Networks Inc", 0xA2),
        (35, "Tzero", 0x23),
        (36, "Emulex", 0xA4),
        (37, "Power-One", 0x25),
        (38, "Pulse~LINK Inc", 0x26),
        (39, "Hon Hai Precision Industry", 0xA7),
        (40, "White Rock Networks Inc", 0xA8),
        (41, "Telegent Systems USA Inc", 0x29),
        (42, "Atrua Technologies Inc", 0x2A),
        (43, "Acbel Polytech Inc", 0xAB),
        (44, "eRide Inc", 0x2C),
        (45, "ULi Electronics Inc", 0xAD),
        (46, "Magnum Semiconductor Inc", 0xAE),
        (47, "neoOne Technology Inc", 0x2F),
        (48, "Connex Technology Inc", 0xB0),
        (49, "Stream Processors Inc", 0x31),
        (50, "Focus Enhancements", 0x32),
        (51, "Telecis Wireless Inc", 0xB3),
        (52, "uNav Microelectronics", 0x34),
        (53, "Tarari Inc", 0xB5),
        (54, "Ambric Inc", 0xB6),
        (55, "Newport Media Inc", 0x37),
        (56, "VMTS", 0x38),
        (57, "Enuclia Semiconductor Inc", 0xB9),
        (58, "Virtium Technology Inc", 0xBA),
        (59, "Solid State System Co Ltd", 0x3B),
        (60, "Kian Tech LLC", 0xBC),
        (61, "Artimi", 0x3D),
        (62, "Power Quotient International", 0x3E),
        (63, "Avago Technologies", 0xBF),
        (64, "ADTechnology", 0x40),
        (65, "Sigma Designs", 0xC1),
        (66, "SiCortex Inc", 0xC2),
        (67, "Ventura Technology Group", 0x43),
        (68, "eASIC", 0xC4),
        (69, "M.H.S. SAS", 0x45),
        (70, "Micro Star International", 0x46),
        (71, "Rapport Inc", 0xC7),
        (72, "Makway International", 0xC8),
        (73, "Broad Reach Engineering Co", 0x49),
        (74, "Semiconductor Mfg Intl Corp", 0x4A),
        (75, "SiConnect", 0xCB),
        (76, "FCI USA Inc", 0x4C),
        (77, "Validity Sensors", 0xCD),
        (78, "Coney Technology Co Ltd", 0xCE),
        (79, "Spans Logic", 0x4F),
        (80, "Neterion Inc", 0xD0),
        (81, "Qimonda", 0x51),
        (82, "New Japan Radio Co Ltd", 0x52),
        (83, "Velogix", 0xD3),
        (84, "Montalvo Systems", 0x54),
        (85, "iVivity Inc", 0xD5),
        (86, "Walton Chaintech", 0xD6),
        (87, "AENEON", 0x57),
        (88, "Lorom Industrial Co Ltd", 0x58),
        (89, "Radiospire Networks", 0xD9),
        (90, "Sensio Technologies Inc", 0xDA),
        (91, "Nethra Imaging", 0x5B),
        (92, "Hexon Technology Pte Ltd", 0xDC),
        (93, "CompuStocx (CSX)", 0x5D),
        (94, "Methode Electronics Inc", 0x5E),
        (95, "Connect One Ltd", 0xDF),
        (96, "Opulan Technologies", 0xE0),
        (97, "Septentrio NV", 0x61),
        (98, "Goldenmars Technology Inc", 0x62),
        (99, "Kreton Corporation", 0xE3),
        (100, "Cochlear Ltd", 0x64),
        (101, "Altair Semiconductor", 0xE5),
        (102, "NetEffect Inc", 0xE6),
        (103, "Spansion Inc", 0x67),
        (104, "Taiwan Semiconductor Mfg", 0x68),
        (105, "Emphany Systems Inc", 0xE9),
        (106, "ApaceWave Technologies", 0xEA),
        (107, "Mobilygen Corporation", 0x6B),
        (108, "Tego", 0xEC),
        (109, "Cswitch Corporation", 0x6D),
        (110, "Haier (Beijing) IC Design Co", 0x6E),
        (111, "MetaRAM", 0xEF),
        (112, "Axel Electronics Co Ltd", 0x70),
        (113, "Tilera Corporation", 0xF1),
        (114, "Aquantia", 0xF2),
        (115, "Vivace Semiconductor", 0x73),
        (116, "Redpine Signals", 0xF4),
        (117, "Octalica", 0x75),
        (118, "InterDigital Communications", 0x76),
        (119, "Avant Technology", 0xF7),
        (120, "Asrock Inc", 0xF8),
        (121, "Availink", 0x79),
        (122, "Quartics Inc", 0x7A),
        (123, "Element CXI", 0xFB),
        (124, "Innovaciones Microelectronicas", 0x7C),
        (125, "VeriSilicon Microelectronics", 0xFD),
        (126, "W5 Networks", 0xFE),
    ],
    [ # Bank 7
        (1, "MOVEKING", 0x01),
        (2, "Mavrix Technology Inc", 0x02),
        (3, "CellGuide Ltd", 0x83),
        (4, "Faraday Technology", 0x04),
        (5, "Diablo Technologies Inc", 0x85),
        (6, "Jennic", 0x86),
        (7, "Octasic", 0x07),
        (8, "Molex Incorporated", 0x08),
        (9, "3Leaf Networks", 0x89),
        (10, "Bright Micron Technology", 0x8A),
        (11, "Netxen", 0x0B),
        (12, "NextWave Broadband Inc", 0x8C),
        (13, "DisplayLink", 0x0D),
        (14, "ZMOS Technology", 0x0E),
        (15, "Tec-Hill", 0x8F),
        (16, "Multigig Inc", 0x10),
        (17, "Amimon", 0x91),
        (18, "Euphonic Technologies Inc", 0x92),
        (19, "BRN Phoenix", 0x13),
        (20, "InSilica", 0x94),
        (21, "Ember Corporation", 0x15),
        (22, "Avexir Technologies Corporation", 0x16),
        (23, "Echelon Corporation", 0x97),
        (24, "Edgewater Computer Systems", 0x98),
        (25, "XMOS Semiconductor Ltd", 0x19),
        (26, "GENUSION Inc", 0x1A),
        (27, "Memory Corp NV", 0x9B),
        (28, "SiliconBlue Technologies", 0x1C),
        (29, "Rambus Inc", 0x9D),
        (30, "Andes Technology Corporation", 0x9E),
        (31, "Coronis Systems", 0x1F),
        (32, "Achronix Semiconductor", 0x20),
        (33, "Siano Mobile Silicon Ltd", 0xA1),
        (34, "Semtech Corporation", 0xA2),
        (35, "Pixelworks Inc", 0x23),
        (36, "Gaisler Research AB", 0xA4),
        (37, "Teranetics", 0x25),
        (38, "Toppan Printing Co Ltd", 0x26),
        (39, "Kingxcon", 0xA7),
        (40, "Silicon Integrated Systems", 0xA8),
        (41, "I-O Data Device Inc", 0x29),
        (42, "NDS Americas Inc", 0x2A),
        (43, "Solomon Systech Limited", 0xAB),
        (44, "On Demand Microelectronics", 0x2C),
        (45, "Amicus Wireless Inc", 0xAD),
        (46, "SMARDTV SNC", 0xAE),
        (47, "Comsys Communication Ltd", 0x2F),
        (48, "Movidia Ltd", 0xB0),
        (49, "Javad GNSS Inc", 0x31),
        (50, "Montage Technology Group", 0x32),
        (51, "Trident Microsystems", 0xB3),
        (52, "Super Talent", 0x34),
        (53, "Optichron Inc", 0xB5),
        (54, "Future Waves UK Ltd", 0xB6),
        (55, "SiBEAM Inc", 0x37),
        (56, "InicoreInc", 0x38),
        (57, "Virident Systems", 0xB9),
        (58, "M2000 Inc", 0xBA),
        (59, "ZeroG Wireless Inc", 0x3B),
        (60, "Gingle Technology Co Ltd", 0xBC),
        (61, "Space Micro Inc", 0x3D),
        (62, "Wilocity", 0x3E),
        (63, "Novafora Inc", 0xBF),
        (64, "iKoa Corporation", 0x40),
        (65, "ASint Technology", 0xC1),
        (66, "Ramtron", 0xC2),
        (67, "Plato Networks Inc", 0x43),
        (68, "IPtronics AS", 0xC4),
        (69, "Infinite-Memories", 0x45),
        (70, "Parade Technologies Inc", 0x46),
        (71, "Dune Networks", 0xC7),
        (72, "GigaDevice Semiconductor", 0xC8),
        (73, "Modu Ltd", 0x49),
        (74, "CEITEC", 0x4A),
        (75, "Northrop Grumman", 0xCB),
        (76, "XRONET Corporation", 0x4C),
        (77, "Sicon Semiconductor AB", 0xCD),
        (78, "Atla Electronics Co Ltd", 0xCE),
        (79, "TOPRAM Technology", 0x4F),
        (80, "Silego Technology Inc", 0xD0),
        (81, "Kinglife", 0x51),
        (82, "Ability Industries Ltd", 0x52),
        (83, "Silicon Power Computer & Communications", 0xD3),
        (84, "Augusta Technology Inc", 0x54),
        (85, "Nantronics Semiconductors", 0xD5),
        (86, "Hilscher Gesellschaft", 0xD6),
        (87, "Quixant Ltd", 0x57),
        (88, "Percello Ltd", 0x58),
        (89, "NextIO Inc", 0xD9),
        (90, "Scanimetrics Inc", 0xDA),
        (91, "FS-Semi Company Ltd", 0x5B),
        (92, "Infinera Corporation", 0xDC),
        (93, "SandForce Inc", 0x5D),
        (94, "Lexar Media", 0x5E),
        (95, "Teradyne Inc", 0xDF),
        (96, "Memory Exchange Corp", 0xE0),
        (97, "Suzhou Smartek Electronics", 0x61),
        (98, "Avantium Corporation", 0x62),
        (99, "ATP Electronics Inc", 0xE3),
        (100, "Valens Semiconductor Ltd", 0x64),
        (101, "Agate Logic Inc", 0xE5),
        (102, "Netronome", 0xE6),
        (103, "Zenverge Inc", 0x67),
        (104, "N-trig Ltd", 0x68),
        (105, "SanMax Technologies Inc", 0xE9),
        (106, "Contour Semiconductor Inc", 0xEA),
        (107, "TwinMOS", 0x6B),
        (108, "Silicon Systems Inc", 0xEC),
        (109, "V-Color Technology Inc", 0x6D),
        (110, "Certicom Corporation", 0x6E),
        (111, "JSC ICC Milandr", 0xEF),
        (112, "PhotoFast Global Inc", 0x70),
        (113, "InnoDisk Corporation", 0xF1),
        (114, "Muscle Power", 0xF2),
        (115, "Energy Micro", 0x73),
        (116, "Innofidei", 0xF4),
        (117, "CopperGate Communications", 0x75),
        (118, "Holtek Semiconductor Inc", 0x76),
        (119, "Myson Century Inc", 0xF7),
        (120, "FIDELIX", 0xF8),
        (121, "Red Digital Cinema", 0x79),
        (122, "Densbits Technology", 0x7A),
        (123, "Zempro", 0xFB),
        (124, "MoSys", 0x7C),
        (125, "Provigent", 0xFD),
        (126, "Triad Semiconductor Inc", 0xFE),
    ],
    [ # Bank 8
        (1, "Siklu Communication Ltd", 0x01),
        (2, "A Force Manufacturing Ltd", 0x02),
        (3, "Strontium", 0x83),
        (4, "ALi Corp (Abilis Systems)", 0x04),
        (5, "Siglead Inc", 0x85),
        (6, "Ubicom Inc", 0x86),
        (7, "Unifosa Corporation", 0x07),
        (8, "Stretch Inc", 0x08),
        (9, "Lantiq Deutschland GmbH", 0x89),
        (10, "Visipro.", 0x8A),
        (11, "EKMemory", 0x0B),
        (12, "Microelectronics Institute ZTE", 0x8C),
        (13, "u-blox AG", 0x0D),
        (14, "Carry Technology Co Ltd", 0x0E),
        (15, "Nokia", 0x8F),
        (16, "King Tiger Technology", 0x10),
        (17, "Sierra Wireless", 0x91),
        (18, "HT Micron", 0x92),
        (19, "Albatron Technology Co Ltd", 0x13),
        (20, "Leica Geosystems AG", 0x94),
        (21, "BroadLight", 0x15),
        (22, "AEXEA", 0x16),
        (23, "ClariPhy Communications Inc", 0x97),
        (24, "Green Plug", 0x98),
        (25, "Design Art Networks", 0x19),
        (26, "Mach Xtreme Technology Ltd", 0x1A),
        (27, "ATO Solutions Co Ltd", 0x9B),
        (28, "Ramsta", 0x1C),
        (29, "Greenliant Systems Ltd", 0x9D),
        (30, "Teikon", 0x9E),
        (31, "Antec Hadron", 0x1F),
        (32, "NavCom Technology Inc", 0x20),
        (33, "Shanghai Fudan Microelectronics", 0xA1),
        (34, "Calxeda Inc", 0xA2),
        (35, "JSC EDC Electronics", 0x23),
        (36, "Kandit Technology Co Ltd", 0xA4),
        (37, "Ramos Technology", 0x25),
        (38, "Goldenmars Technology", 0x26),
        (39, "XeL Technology Inc", 0xA7),
        (40, "Newzone Corporation", 0xA8),
        (41, "ShenZhen MercyPower Tech", 0x29),
        (42, "Nanjing Yihuo Technology", 0x2A),
        (43, "Nethra Imaging Inc", 0xAB),
        (44, "SiTel Semiconductor BV", 0x2C),
        (45, "SolidGear Corporation", 0xAD),
        (46, "Topower Computer Ind Co Ltd", 0xAE),
        (47, "Wilocity", 0x2F),
        (48, "Profichip GmbH", 0xB0),
        (49, "Gerad Technologies", 0x31),
        (50, "Ritek Corporation", 0x32),
        (51, "Gomos Technology Limited", 0xB3),
        (52, "Memoright Corporation", 0x34),
        (53, "D-Broad Inc", 0xB5),
        (54, "HiSilicon Technologies", 0xB6),
        (55, "Syndiant Inc.", 0x37),
        (56, "Enverv Inc", 0x38),
        (57, "Cognex", 0xB9),
        (58, "Xinnova Technology Inc", 0xBA),
        (59, "Ultron AG", 0x3B),
        (60, "Concord Idea Corporation", 0xBC),
        (61, "AIM Corporation", 0x3D),
        (62, "Lifetime Memory Products", 0x3E),
        (63, "Ramsway", 0xBF),
        (64, "Recore Systems B.V.", 0x40),
        (65, "Haotian Jinshibo Science Tech", 0xC1),
        (66, "Being Advanced Memory", 0xC2),
        (67, "Adesto Technologies", 0x43),
        (68, "Giantec Semiconductor Inc", 0xC4),
        (69, "HMD Electronics AG", 0x45),
        (70, "Gloway International (HK)", 0x46),
        (71, "Kingcore", 0xC7),
        (72, "Anucell Technology Holding", 0xC8),
        (73, "Accord Software & Systems Pvt. Ltd", 0x49),
        (74, "Active-Semi Inc", 0x4A),
        (75, "Denso Corporation", 0xCB),
        (76, "TLSI Inc", 0x4C),
        (77, "Qidan", 0xCD),
        (78, "Mustang", 0xCE),
        (79, "Orca Systems", 0x4F),
        (80, "Passif Semiconductor", 0xD0),
        (81, "GigaDevice Semiconductor (Beijing)", 0x51),
        (82, "Memphis Electronic", 0x52),
        (83, "Beckhoff Automation GmbH", 0xD3),
        (84, "Harmony Semiconductor Corp", 0x54),
        (85, "Air Computers SRL", 0xD5),
        (86, "TMT Memory", 0xD6),
        (87, "Eorex Corporation", 0x57),
        (88, "Xingtera", 0x58),
        (89, "Netsol", 0xD9),
        (90, "Bestdon Technology Co Ltd", 0xDA),
        (91, "Baysand Inc", 0x5B),
        (92, "Uroad Technology Co Ltd", 0xDC),
        (93, "Wilk Elektronik S.A.", 0x5D),
        (94, "AAI", 0x5E),
        (95, "Harman", 0xDF),
        (96, "Berg Microelectronics Inc", 0xE0),
        (97, "ASSIA Inc", 0x61),
        (98, "Visiontek Products LLC", 0x62),
        (99, "OCMEMORY", 0xE3),
        (100, "Welink Solution Inc", 0x64),
        (101, "Shark Gaming", 0xE5),
        (102, "Avalanche Technology", 0xE6),
        (103, "R&D Center ELVEES OJSC", 0x67),
        (104, "KingboMars Technology Co Ltd", 0x68),
        (105, "High Bridge Solutions Industria Eletronica", 0xE9),
        (106, "Transcend Technology Co Ltd", 0xEA),
        (107, "Everspin Technologies", 0x6B),
        (108, "Hon-Hai Precision", 0xEC),
        (109, "Smart Storage Systems", 0x6D),
        (110, "Toumaz Group", 0x6E),
        (111, "Zentel Electronics Corporation", 0xEF),
        (112, "Panram International Corporation", 0x70),
        (113, "Silicon Space Technology", 0xF1),
        (114, "LITE-ON IT Corporation", 0xF2),
        (115, "Inuitive", 0x73),
        (116, "HMicro", 0xF4),
        (117, "BittWare Inc", 0x75),
        (118, "GLOBALFOUNDRIES", 0x76),
        (119, "ACPI Digital Co Ltd", 0xF7),
        (120, "Annapurna Labs", 0xF8),
        (121, "AcSiP Technology Corporation", 0x79),
        (122, "Idea! Electronic Systems", 0x7A),
        (123, "Gowe Technology Co Ltd", 0xFB),
        (124, "Hermes Testing Solutions Inc", 0x7C),
        (125, "Positivo BGH", 0xFD),
        (126, "Intelligence  Silicon Technology", 0xFE),
    ],
    [ # Bank 9
        (1, "3D PLUS", 0x01),
        (2, "Diehl Aerospace", 0x02),
        (3, "Fairchild", 0x83),
        (4, "Mercury Systems", 0x04),
        (5, "Sonics Inc", 0x85),
        (6, "Emerson Automation Solutions", 0x86),
        (7, "Shenzhen Jinge Information Co Ltd", 0x07),
        (8, "SCWW", 0x08),
        (9, "Silicon Motion Inc", 0x89),
        (10, "Anurag", 0x8A),
        (11, "King Kong", 0x0B),
        (12, "FROM30 Co Ltd", 0x8C),
        (13, "Gowin Semiconductor Corp", 0x0D),
        (14, "Fremont Micro Devices Ltd", 0x0E),
        (15, "Ericsson Modems", 0x8F),
        (16, "Exelis", 0x10),
        (17, "Satixfy Ltd", 0x91),
        (18, "Galaxy Microsystems Ltd", 0x92),
        (19, "Gloway International Co Ltd", 0x13),
        (20, "Lab", 0x94),
        (21, "Smart Energy Instruments", 0x15),
        (22, "Approved Memory Corporation", 0x16),
        (23, "Axell Corporation", 0x97),
        (24, "Essencore Limited", 0x98),
        (25, "Phytium", 0x19),
        (26, "Xi’an UniIC Semiconductors Co Ltd", 0x1A),
        (27, "Ambiq Micro", 0x9B),
        (28, "eveRAM Technology Inc", 0x1C),
        (29, "Infomax", 0x9D),
        (30, "Butterfly Network Inc", 0x9E),
        (31, "Shenzhen City Gcai Electronics", 0x1F),
        (32, "Stack Devices Corporation", 0x20),
        (33, "ADK Media Group", 0xA1),
        (34, "TSP Global Co Ltd", 0xA2),
        (35, "HighX", 0x23),
        (36, "Shenzhen Elicks Technology", 0xA4),
        (37, "XinKai/Silicon Kaiser", 0x25),
        (38, "Google Inc", 0x26),
        (39, "Dasima International Development", 0xA7),
        (40, "Leahkinn Technology Limited", 0xA8),
        (41, "HIMA Paul Hildebrandt GmbH Co KG", 0x29),
        (42, "Keysight Technologies", 0x2A),
        (43, "Techcomp International (Fastable)", 0xAB),
        (44, "Ancore Technology Corporation", 0x2C),
        (45, "Nuvoton", 0xAD),
        (46, "Korea Uhbele International Group Ltd", 0xAE),
        (47, "Ikegami Tsushinki Co Ltd", 0x2F),
        (48, "RelChip Inc", 0xB0),
        (49, "Baikal Electronics", 0x31),
        (50, "Nemostech Inc", 0x32),
        (51, "Memorysolution GmbH", 0xB3),
        (52, "Silicon Integrated Systems Corporation", 0x34),
        (53, "Xiede", 0xB5),
        (54, "BRC", 0xB6),
        (55, "Flash Chi", 0x37),
        (56, "Jone", 0x38),
        (57, "GCT Semiconductor Inc", 0xB9),
        (58, "Hong Kong Zetta Device Technology", 0xBA),
        (59, "Unimemory Technology(s) Pte Ltd", 0x3B),
        (60, "Cuso", 0xBC),
        (61, "Kuso", 0x3D),
        (62, "Uniquify Inc", 0x3E),
        (63, "Skymedi Corporation", 0xBF),
        (64, "Core Chance Co Ltd", 0x40),
        (65, "Tekism Co Ltd", 0xC1),
        (66, "Seagate Technology PLC", 0xC2),
        (67, "Hong Kong Gaia Group Co Limited", 0x43),
        (68, "Gigacom Semiconductor LLC", 0xC4),
        (69, "V2 Technologies", 0x45),
        (70, "TLi", 0x46),
        (71, "Neotion", 0xC7),
        (72, "Lenovo", 0xC8),
        (73, "Shenzhen Zhongteng Electronic Corp Ltd", 0x49),
        (74, "Compound Photonics", 0x4A),
        (75, "in2H2 inc", 0xCB),
        (76, "Shenzhen Pango Microsystems Co Ltd", 0x4C),
        (77, "Vasekey", 0xCD),
        (78, "Cal-Comp Industria de Semicondutores", 0xCE),
        (79, "Eyenix Co Ltd", 0x4F),
        (80, "Heoriady", 0xD0),
        (81, "Accelerated Memory Production Inc", 0x51),
        (82, "INVECAS Inc", 0x52),
        (83, "AP Memory", 0xD3),
        (84, "Douqi Technology", 0x54),
        (85, "Etron Technology Inc", 0xD5),
        (86, "Indie Semiconductor", 0xD6),
        (87, "Socionext Inc", 0x57),
        (88, "HGST", 0x58),
        (89, "EVGA", 0xD9),
        (90, "Audience Inc", 0xDA),
        (91, "EpicGear", 0x5B),
        (92, "Vitesse Enterprise Co", 0xDC),
        (93, "Foxtronn International Corporation", 0x5D),
        (94, "Bretelon Inc", 0x5E),
        (95, "Graphcore", 0xDF),
        (96, "Eoplex Inc", 0xE0),
        (97, "MaxLinear Inc", 0x61),
        (98, "ETA Devices", 0x62),
        (99, "LOKI", 0xE3),
        (100, "IMS Electronics Co Ltd", 0x64),
        (101, "Dosilicon Co Ltd", 0xE5),
        (102, "Dolphin Integration", 0xE6),
        (103, "Shenzhen Mic Electronics Technolog", 0x67),
        (104, "Boya Microelectronics Inc", 0x68),
        (105, "Geniachip (Roche)", 0xE9),
        (106, "Axign", 0xEA),
        (107, "Kingred Electronic Technology Ltd", 0x6B),
        (108, "Chao Yue Zhuo Computer Business Dept.", 0xEC),
        (109, "Guangzhou Si Nuo Electronic Technology.", 0x6D),
        (110, "Crocus Technology Inc", 0x6E),
        (111, "Creative Chips GmbH", 0xEF),
        (112, "GE Aviation Systems LLC.", 0x70),
        (113, "Asgard", 0xF1),
        (114, "Good Wealth Technology Ltd", 0xF2),
        (115, "TriCor Technologies", 0x73),
        (116, "Nova-Systems GmbH", 0xF4),
        (117, "JUHOR", 0x75),
        (118, "Zhuhai Douke Commerce Co Ltd", 0x76),
        (119, "DSL Memory", 0xF7),
        (120, "Anvo-Systems Dresden GmbH", 0xF8),
        (121, "Realtek", 0x79),
        (122, "AltoBeam", 0x7A),
        (123, "Wave Computing", 0xFB),
        (124, "Beijing TrustNet Technology Co Ltd", 0x7C),
        (125, "Innovium Inc", 0xFD),
        (126, "Starsway Technology Limited", 0xFE),
    ],
    [ # Bank 10
        (1, "Weltronics Co LTD", 0x01),
        (2, "VMware Inc", 0x02),
        (3, "Hewlett Packard Enterprise", 0x83),
        (4, "INTENSO", 0x04),
        (5, "Puya Semiconductor", 0x85),
        (6, "MEMORFI", 0x86),
        (7, "MSC Technologies GmbH", 0x07),
        (8, "Txrui", 0x08),
        (9, "SiFive Inc", 0x89),
        (10, "Spreadtrum Communications", 0x8A),
        (11, "XTX Technology Limited", 0x0B),
        (12, "UMAX Technology", 0x8C),
        (13, "Shenzhen Yong Sheng Technology", 0x0D),
        (14, "SNOAMOO (Shenzhen Kai Zhuo Yue)", 0x0E),
        (15, "Daten Tecnologia LTDA", 0x8F),
        (16, "Shenzhen XinRuiYan Electronics", 0x10),
        (17, "Eta Compute", 0x91),
        (18, "Energous", 0x92),
        (19, "Raspberry Pi Trading Ltd", 0x13),
        (20, "Shenzhen Chixingzhe Tech Co Ltd", 0x94),
        (21, "Silicon Mobility", 0x15),
        (22, "IQ-Analog Corporation", 0x16),
        (23, "Uhnder Inc", 0x97),
        (24, "Impinj", 0x98),
        (25, "DEPO Computers", 0x19),
        (26, "Nespeed Sysems", 0x1A),
        (27, "Yangtze Memory Technologies Co Ltd", 0x9B),
        (28, "MemxPro Inc", 0x1C),
        (29, "Tammuz Co Ltd", 0x9D),
        (30, "Allwinner Technology", 0x9E),
        (31, "Shenzhen City Futian District Qing Xuan Tong Computer Trading Firm", 0x1F),
        (32, "XMC", 0x20),
        (33, "Teclast", 0xA1),
        (34, "Maxsun", 0xA2),
        (35, "Haiguang Integrated Circuit Design", 0x23),
        (36, "RamCENTER Technology", 0xA4),
        (37, "Phison Electronics Corporation", 0x25),
        (38, "Guizhou Huaxintong Semi-Conductor", 0x26),
        (39, "Network Intelligence", 0xA7),
        (40, "Continental Technology (Holdings)", 0xA8),
        (41, "Guangzhou Huayan Suning Electronic", 0x29),
        (42, "Guangzhou Zhouji Electronic Co Ltd", 0x2A),
        (43, "Shenzhen Giant Hui Kang Tech Co Ltd", 0xAB),
        (44, "Shenzhen Yilong Innovative Co Ltd", 0x2C),
        (45, "Neo Forza", 0xAD),
        (46, "Lyontek Inc", 0xAE),
        (47, "Shanghai Kuxin Microelectronics Ltd", 0x2F),
        (48, "Shenzhen Larix Technology Co Ltd", 0xB0),
        (49, "Qbit Semiconductor Ltd", 0x31),
        (50, "Insignis Technology Corporation", 0x32),
        (51, "Lanson Memory Co Ltd", 0xB3),
        (52, "Shenzhen Superway Electronics Co Ltd", 0x34),
        (53, "Canaan-Creative Co Ltd", 0xB5),
        (54, "Black Diamond Memory", 0xB6),
        (55, "Shenzhen City Parker Baking Electronics", 0x37),
        (56, "Shenzhen Baihong Technology Co Ltd", 0x38),
        (57, "GEO Semiconductors", 0xB9),
        (58, "OCPC", 0xBA),
        (59, "Artery Technology Co Ltd", 0x3B),
        (60, "Jinyu", 0xBC),
        (61, "ShenzhenYing Chi Technology Development", 0x3D),
        (62, "Shenzhen Pengcheng Xin Technology", 0x3E),
        (63, "Pegasus Semiconductor (Shanghai) Co", 0xBF),
        (64, "Mythic Inc", 0x40),
        (65, "Elmos Semiconductor AG", 0xC1),
        (66, "Kllisre", 0xC2),
        (67, "Shenzhen Winconway Technology", 0x43),
        (68, "Shenzhen Xingmem Technology Corp", 0xC4),
        (69, "Gold Key Technology Co Ltd", 0x45),
        (70, "Habana Labs Ltd", 0x46),
        (71, "Hoodisk Electronics Co Ltd", 0xC7),
        (72, "SemsoTai (SZ) Technology Co Ltd", 0xC8),
        (73, "OM Nanotech Pvt. Ltd", 0x49),
        (74, "Shenzhen Zhifeng Weiye Technology", 0x4A),
        (75, "Xinshirui (Shenzhen) Electronics Co", 0xCB),
        (76, "Guangzhou Zhong Hao Tian Electronic", 0x4C),
        (77, "Shenzhen Longsys Electronics Co Ltd", 0xCD),
        (78, "Deciso B.V.", 0xCE),
        (79, "Puya Semiconductor (Shenzhen)", 0x4F),
        (80, "Shenzhen Veineda Technology Co Ltd", 0xD0),
        (81, "Antec Memory", 0x51),
        (82, "Cortus SAS", 0x52),
        (83, "Dust Leopard", 0xD3),
        (84, "MyWo AS", 0x54),
        (85, "J&A Information Inc", 0xD5),
        (86, "Shenzhen JIEPEI Technology Co Ltd", 0xD6),
        (87, "Heidelberg University", 0x57),
        (88, "Flexxon PTE Ltd", 0x58),
        (89, "Wiliot", 0xD9),
        (90, "Raysun Electronics International Ltd", 0xDA),
        (91, "Aquarius Production Company LLC", 0x5B),
        (92, "MACNICA DHW LTDA", 0xDC),
        (93, "Intelimem", 0x5D),
        (94, "Zbit Semiconductor Inc", 0x5E),
        (95, "Shenzhen Technology Co Ltd", 0xDF),
        (96, "Signalchip", 0xE0),
        (97, "Shenzen Recadata Storage Technology", 0x61),
        (98, "Hyundai Technology", 0x62),
        (99, "Shanghai Fudi Investment Development", 0xE3),
        (100, "Aixi Technology", 0x64),
        (101, "Tecon MT", 0xE5),
        (102, "Onda Electric Co Ltd", 0xE6),
        (103, "Jinshen", 0x67),
        (104, "Kimtigo Semiconductor (HK) Limited", 0x68),
        (105, "IIT Madras", 0xE9),
        (106, "Shenshan (Shenzhen) Electronic", 0xEA),
        (107, "Hefei Core Storage Electronic Limited", 0x6B),
        (108, "Colorful Technology Ltd", 0xEC),
        (109, "Visenta (Xiamen) Technology Co Ltd", 0x6D),
        (110, "Roa Logic BV", 0x6E),
        (111, "NSITEXE Inc", 0xEF),
        (112, "Hong Kong Hyunion Electronics", 0x70),
        (113, "ASK Technology Group Limited", 0xF1),
        (114, "GIGA-BYTE Technology Co Ltd", 0xF2),
        (115, "Terabyte Co Ltd", 0x73),
        (116, "Hyundai Inc", 0xF4),
        (117, "EXCELERAM", 0x75),
        (118, "PsiKick", 0x76),
        (119, "Netac Technology Co Ltd", 0xF7),
        (120, "PCCOOLER", 0xF8),
        (121, "Jiangsu Huacun Electronic Technology", 0x79),
        (122, "Shenzhen Micro Innovation Industry", 0x7A),
        (123, "Beijing Tongfang Microelectronics Co", 0xFB),
        (124, "XZN Storage Technology", 0x7C),
        (125, "ChipCraft Sp. z.o.o.", 0xFD),
        (126, "ALLFLASH Technology Limited", 0xFE),
    ],
    [ # Bank 11
        (1, "Foerd Technology Co Ltd", 0x01),
        (2, "KingSpec", 0x02),
        (3, "Codasip GmbH", 0x83),
        (4, "SL Link Co Ltd", 0x04),
        (5, "Shenzhen Kefu Technology Co Limited", 0x85),
        (6, "Shenzhen ZST Electronics Technology", 0x86),
        (7, "Kyokuto Electronic Inc", 0x07),
        (8, "Warrior Technology", 0x08),
        (9, "TRINAMIC Motion Control GmbH & Co", 0x89),
        (10, "PixelDisplay Inc", 0x8A),
        (11, "Shenzhen Futian District Bo Yueda Elec", 0x0B),
        (12, "Richtek Power", 0x8C),
        (13, "Shenzhen LianTeng Electronics Co Ltd", 0x0D),
        (14, "AITC Memory", 0x0E),
        (15, "UNIC Memory Technology Co Ltd", 0x8F),
        (16, "Shenzhen Huafeng Science Technology", 0x10),
        (17, "CXMT", 0x91),
        (18, "Guangzhou Xinyi Heng Computer Trading Firm", 0x92),
        (19, "SambaNova Systems", 0x13),
        (20, "V-GEN", 0x94),
        (21, "Jump Trading", 0x15),
        (22, "Ampere Computing", 0x16),
        (23, "Shenzhen Zhongshi Technology Co Ltd", 0x97),
        (24, "Shenzhen Zhongtian Bozhong Technology", 0x98),
        (25, "Tri-Tech International", 0x19),
        (26, "Silicon Intergrated Systems Corporation", 0x1A),
        (27, "Shenzhen HongDingChen Information", 0x9B),
        (28, "Plexton Holdings Limited", 0x1C),
        (29, "AMS (Jiangsu Advanced Memory Semi)", 0x9D),
        (30, "Wuhan Jing Tian Interconnected Tech Co", 0x9E),
        (31, "Axia Memory Technology", 0x1F),
        (32, "Chipset Technology Holding Limited", 0x20),
        (33, "Shenzhen Xinshida Technology Co Ltd", 0xA1),
        (34, "Shenzhen Chuangshifeida Technology", 0xA2),
        (35, "Guangzhou MiaoYuanJi Technology", 0x23),
        (36, "ADVAN Inc", 0xA4),
        (37, "Shenzhen Qianhai Weishengda Electronic Commerce Company Ltd", 0x25),
        (38, "Guangzhou Guang Xie Cheng Trading", 0x26),
        (39, "StarRam International Co Ltd", 0xA7),
        (40, "Shen Zhen XinShenHua Tech Co Ltd", 0xA8),
        (41, "UltraMemory Inc", 0x29),
        (42, "New Coastline Global Tech Industry Co", 0x2A),
        (43, "Sinker", 0xAB),
        (44, "Diamond", 0x2C),
        (45, "PUSKILL", 0xAD),
        (46, "Guangzhou Hao Jia Ye Technology Co", 0xAE),
        (47, "Ming Xin Limited", 0x2F),
        (48, "Barefoot Networks", 0xB0),
        (49, "Biwin Semiconductor (HK) Co Ltd", 0x31),
        (50, "UD INFO Corporation", 0x32),
        (51, "Trek Technology (S) PTE Ltd", 0xB3),
        (52, "Xiamen Kingblaze Technology Co Ltd", 0x34),
        (53, "Shenzhen Lomica Technology Co Ltd", 0xB5),
        (54, "Nuclei System Technology Co Ltd", 0xB6),
        (55, "Wuhan Xun Zhan Electronic Technology", 0x37),
        (56, "Shenzhen Ingacom Semiconductor Ltd", 0x38),
        (57, "Zotac Technology Ltd", 0xB9),
        (58, "Foxline", 0xBA),
        (59, "Shenzhen Farasia Science Technology", 0x3B),
        (60, "Efinix Inc", 0xBC),
        (61, "Hua Nan San Xian Technology Co Ltd", 0x3D),
        (62, "Goldtech Electronics Co Ltd", 0x3E),
        (63, "Shanghai Han Rong Microelectronics Co", 0xBF),
        (64, "Shenzhen Zhongguang Yunhe Trading", 0x40),
        (65, "Smart Shine(QingDao) Microelectronics", 0xC1),
        (66, "Thermaltake Technology Co Ltd", 0xC2),
        (67, "Shenzhen O’Yang Maile Technology Ltd", 0x43),
        (68, "UPMEM", 0xC4),
        (69, "Chun Well Technology Holding Limited", 0x45),
        (70, "Astera Labs Inc", 0x46),
        (71, "Winconway", 0xC7),
        (72, "Advantech Co Ltd", 0xC8),
        (73, "Chengdu Fengcai Electronic Technology", 0x49),
        (74, "The Boeing Company", 0x4A),
        (75, "Blaize Inc", 0xCB),
        (76, "Ramonster Technology Co Ltd", 0x4C),
        (77, "Wuhan Naonongmai Technology Co Ltd", 0xCD),
        (78, "Shenzhen Hui ShingTong Technology", 0xCE),
        (79, "Yourlyon", 0x4F),
        (80, "Fabu Technology", 0xD0),
        (81, "Shenzhen Yikesheng Technology Co Ltd", 0x51),
        (82, "NOR-MEM", 0x52),
        (83, "Cervoz Co Ltd", 0xD3),
        (84, "Bitmain Technologies Inc.", 0x54),
        (85, "Facebook Inc", 0xD5),
        (86, "Shenzhen Longsys Electronics Co Ltd", 0xD6),
        (87, "Guangzhou Siye Electronic Technology", 0x57),
        (88, "Silergy", 0x58),
        (89, "Adamway", 0xD9),
        (90, "PZG", 0xDA),
        (91, "Shenzhen King Power Electronics", 0x5B),
        (92, "Guangzhou ZiaoFu Tranding Co Ltd", 0xDC),
        (93, "Shenzhen SKIHOTAR Semiconductor", 0x5D),
        (94, "PulseRain Technology", 0x5E),
        (95, "Seeker Technology Limited", 0xDF),
        (96, "Shenzhen OSCOO Tech Co Ltd", 0xE0),
        (97, "Shenzhen Yze Technology Co Ltd", 0x61),
        (98, "Shenzhen Jieshuo Electronic Commerce", 0x62),
        (99, "Gazda", 0xE3),
        (100, "Hua Wei Technology Co Ltd", 0x64),
        (101, "Esperanto Technologies", 0xE5),
        (102, "JinSheng Electronic (Shenzhen) Co Ltd", 0xE6),
        (103, "Shenzhen Shi Bolunshuai Technology", 0x67),
        (104, "Shanghai Rui Zuan Information Tech", 0x68),
        (105, "Fraunhofer IIS", 0xE9),
        (106, "Kandou Bus SA", 0xEA),
        (107, "Acer", 0x6B),
        (108, "Artmem Technology Co Ltd", 0xEC),
        (109, "Gstar Semiconductor Co Ltd", 0x6D),
        (110, "ShineDisk", 0x6E),
        (111, "Shenzhen CHN Technology Co Ltd", 0xEF),
        (112, "UnionChip Semiconductor Co Ltd", 0x70),
        (113, "Tanbassh", 0xF1),
        (114, "Shenzhen Tianyu Jieyun Intl Logistics", 0xF2),
        (115, "MCLogic Inc", 0x73),
        (116, "Eorex Corporation", 0xF4),
        (117, "Arm Technology (China) Co Ltd", 0x75),
        (118, "Lexar Co Limited", 0x76),
        (119, "QinetiQ Group plc", 0xF7),
        (120, "Exascend", 0xF8),
        (121, "Hong Kong Hyunion Electronics Co Ltd", 0x79),
        (122, "Shenzhen Banghong Electronics Co Ltd", 0x7A),
        (123, "MBit Wireless Inc", 0xFB),
        (124, "Hex Five Security Inc", 0x7C),
        (125, "ShenZhen Juhor Precision Tech Co Ltd", 0xFD),
        (126, "Shenzhen Reeinno Technology Co Ltd", 0xFE),
    ],
    [ # Bank 12
        (1, "ABIT Electronics (Shenzhen) Co Ltd", 0x01),
        (2, "Semidrive", 0x02),
        (3, "MyTek Electronics Corp", 0x83),
        (4, "Wxilicon Technology Co Ltd", 0x04),
        (5, "Shenzhen Meixin Electronics Ltd", 0x85),
        (6, "Ghost Wolf", 0x86),
        (7, "LiSion Technologies Inc", 0x07),
        (8, "Power Active Co Ltd", 0x08),
        (9, "Pioneer High Fidelity Taiwan Co. Ltd", 0x89),
        (10, "LuoSilk", 0x8A),
        (11, "Shenzhen Chuangshifeida Technology", 0x0B),
        (12, "Black Sesame Technologies Inc", 0x8C),
        (13, "Jiangsu Xinsheng Intelligent Technology", 0x0D),
        (14, "MLOONG", 0x0E),
        (15, "Quadratica LLC", 0x8F),
        (16, "Anpec Electronics", 0x10),
        (17, "Xi’an Morebeck Semiconductor Tech Co", 0x91),
        (18, "Kingbank Technology Co Ltd", 0x92),
        (19, "ITRenew Inc", 0x13),
        (20, "Shenzhen Eaget Innovation Tech Ltd", 0x94),
        (21, "Jazer", 0x15),
        (22, "Xiamen Semiconductor Investment Group", 0x16),
        (23, "Guangzhou Longdao Network Tech Co", 0x97),
        (24, "Shenzhen Futian SEC Electronic Market", 0x98),
        (25, "Allegro Microsystems LLC", 0x19),
        (26, "Hunan RunCore Innovation Technology", 0x1A),
        (27, "C-Corsa Technology", 0x9B),
        (28, "Zhuhai Chuangfeixin Technology Co Ltd", 0x1C),
        (29, "Beijing InnoMem Technologies Co Ltd", 0x9D),
        (30, "YooTin", 0x9E),
        (31, "Shenzhen Pengxiong Technology Co Ltd", 0x1F),
        (32, "Dongguan Yingbang Commercial Trading Co", 0x20),
        (33, "Shenzhen Ronisys Electronics Co Ltd", 0xA1),
        (34, "Hongkong Xinlan Guangke Co Ltd", 0xA2),
        (35, "Apex Microelectronics Co Ltd", 0x23),
        (36, "Beijing Hongda Jinming Technology Co Ltd", 0xA4),
        (37, "Ling Rui Technology (Shenzhen) Co Ltd", 0x25),
        (38, "Hongkong Hyunion Electronics Co Ltd", 0x26),
        (39, "Starsystems Inc", 0xA7),
        (40, "Shenzhen Yingjiaxun Industrial Co Ltd", 0xA8),
        (41, "Dongguan Crown Code Electronic Commerce", 0x29),
        (42, "Monolithic Power Systems Inc", 0x2A),
        (43, "WuHan SenNaiBo E-Commerce Co Ltd", 0xAB),
        (44, "Hangzhou Hikstorage Technology Co", 0x2C),
        (45, "Shenzhen Goodix Technology Co Ltd", 0xAD),
        (46, "Aigo Electronic Technology Co Ltd", 0xAE),
        (47, "Hefei Konsemi Storage Technology Co Ltd", 0x2F),
        (48, "Cactus Technologies Limited", 0xB0),
        (49, "DSIN", 0x31),
        (50, "Blu Wireless Technology", 0x32),
        (51, "Nanjing UCUN Technology Inc", 0xB3),
        (52, "Acacia Communications", 0x34),
        (53, "Beijinjinshengyihe Technology Co Ltd", 0xB5),
        (54, "Zyzyx", 0xB6),
        (55, "T-HEAD Semiconductor Co Ltd", 0x37),
        (56, "Shenzhen Hystou Technology Co Ltd", 0x38),
        (57, "Syzexion", 0xB9),
        (58, "Kembona", 0xBA),
        (59, "Qingdao Thunderobot Technology Co Ltd", 0x3B),
        (60, "Morse Micro", 0xBC),
        (61, "Shenzhen Envida  Technology Co Ltd", 0x3D),
        (62, "UDStore Solution Limited", 0x3E),
        (63, "Shunlie", 0xBF),
        (64, "Shenzhen Xin Hong Rui Tech Ltd", 0x40),
        (65, "Shenzhen Yze Technology Co Ltd", 0xC1),
        (66, "Shenzhen Huang Pu He Xin Technology", 0xC2),
        (67, "Xiamen Pengpai Microelectronics Co Ltd", 0x43),
        (68, "JISHUN", 0xC4),
        (69, "Shenzhen WODPOSIT Technology Co", 0x45),
        (70, "Unistar", 0x46),
        (71, "UNICORE Electronic (Suzhou) Co Ltd", 0xC7),
        (72, "Axonne Inc", 0xC8),
        (73, "Shenzhen SOVERECA Technology Co", 0x49),
        (74, "Dire Wolf", 0x4A),
        (75, "Whampoa Core Technology Co Ltd", 0xCB),
        (76, "CSI Halbleiter GmbH", 0x4C),
        (77, "ONE Semiconductor", 0xCD),
        (78, "SimpleMachines Inc", 0xCE),
        (79, "Shenzhen Chengyi Qingdian Electronic", 0x4F),
        (80, "Shenzhen Xinlianxin Network Technology", 0xD0),
        (81, "Vayyar Imaging Ltd", 0x51),
        (82, "Paisen Network Technology Co Ltd", 0x52),
        (83, "Shenzhen Fengwensi Technology Co Ltd", 0xD3),
        (84, "Caplink Technology Limited", 0x54),
        (85, "JJT Solution Co Ltd", 0xD5),
        (86, "HOSIN Global Electronics Co Ltd", 0xD6),
        (87, "Shenzhen KingDisk Century Technology", 0x57),
        (88, "SOYO", 0x58),
        (89, "DIT Technology Co Ltd", 0xD9),
        (90, "iFound", 0xDA),
        (91, "Aril Computer Company", 0x5B),
        (92, "ASUS", 0xDC),
        (93, "Shenzhen Ruiyingtong Technology Co", 0x5D),
        (94, "HANA Micron", 0x5E),
        (95, "RANSOR", 0xDF),
        (96, "Axiado Corporation", 0xE0),
        (97, "Tesla Corporation", 0x61),
        (98, "Pingtouge (Shanghai) Semiconductor Co", 0x62),
        (99, "S3Plus Technologies SA", 0xE3),
        (100, "Integrated Silicon Solution Israel Ltd", 0x64),
        (101, "GreenWaves Technologies", 0xE5),
        (102, "NUVIA Inc", 0xE6),
        (103, "Guangzhou Shuvrwine Technology Co", 0x67),
        (104, "Shenzhen Hangshun Chip Technology", 0x68),
        (105, "Chengboliwei Electronic Business", 0xE9),
        (106, "Kowin Technology HK Limited", 0xEA),
        (107, "Euronet Technology Inc", 0x6B),
        (108, "SCY", 0xEC),
        (109, "Shenzhen Xinhongyusheng Electrical", 0x6D),
        (110, "PICOCOM", 0x6E),
        (111, "Shenzhen Toooogo Memory Technology", 0xEF),
        (112, "VLSI Solution", 0x70),
        (113, "Costar Electronics Inc", 0xF1),
        (114, "Shenzhen Huatop Technology Co Ltd", 0xF2),
        (115, "Inspur Electronic Information Industry", 0x73),
        (116, "Shenzhen Boyuan Computer Technology", 0xF4),
        (117, "Beijing Welldisk Electronics Co Ltd", 0x75),
        (118, "Suzhou EP Semicon Co Ltd", 0x76),
        (119, "Zhejiang Dahua Memory Technology", 0xF7),
        (120, "Virtu Financial", 0xF8),
        (121, "Datotek International Co Ltd", 0x79),
        (122, "Telecom and Microelectronics Industries", 0x7A),
        (123, "Echow Technology Ltd", 0xFB),
        (124, "APEX-INFO", 0x7C),
        (125, "Yingpark", 0xFD),
        (126, "Shenzhen Bigway Tech Co Ltd", 0xFE),
    ],
    [ # Bank 13
        (1, "Beijing Haawking Technology Co Ltd", 0x01),
        (2, "Open HW Group", 0x02),
        (3, "JHICC", 0x83),
        (4, "ncoder AG", 0x04),
        (5, "ThinkTech Information Technology Co", 0x85),
        (6, "Shenzhen Chixingzhe Technology Co Ltd", 0x86),
        (7, "Biao Ram Technology Co Ltd", 0x07),
        (8, "Shenzhen Kaizhuoyue Electronics Co Ltd", 0x08),
        (9, "Shenzhen YC Storage Technology Co Ltd", 0x89),
        (10, "Shenzhen Chixingzhe Technology Co", 0x8A),
        (11, "Wink Semiconductor  (Shenzhen) Co Ltd", 0x0B),
        (12, "AISTOR", 0x8C),
        (13, "Palma Ceia SemiDesign", 0x0D),
        (14, "EM Microelectronic-Marin SA", 0x0E),
        (15, "Shenzhen Monarch Memory Technology", 0x8F),
        (16, "Reliance Memory Inc", 0x10),
        (17, "Jesis", 0x91),
        (18, "Espressif Systems (Shanghai)  Co Ltd", 0x92),
        (19, "Shenzhen Sati Smart Technology Co Ltd", 0x13),
        (20, "NeuMem Co Ltd", 0x94),
        (21, "Lifelong", 0x15),
        (22, "Beijing Oitech Technology Co Ltd", 0x16),
        (23, "Groupe LDLC", 0x97),
        (24, "Semidynamics Technology Services SLU", 0x98),
        (25, "swordbill", 0x19),
        (26, "YIREN", 0x1A),
        (27, "Shenzhen Yinxiang Technology Co Ltd", 0x9B),
        (28, "PoweV Electronic Technology Co Ltd", 0x1C),
        (29, "LEORICE", 0x9D),
        (30, "Waymo LLC", 0x9E),
        (31, "Ventana Micro Systems", 0x1F),
        (32, "Hefei Guangxin Microelectronics Co Ltd", 0x20),
        (33, "Shenzhen Sooner Industrial Co Ltd", 0xA1),
        (34, "Horizon Robotics", 0xA2),
        (35, "Tangem AG", 0x23),
        (36, "FuturePath Technology (Shenzhen) Co", 0xA4),
        (37, "RC Module", 0x25),
        (38, "Timetec International Inc", 0x26),
        (39, "ICMAX Technologies Co Limited", 0xA7),
        (40, "Lynxi Technologies Ltd Co", 0xA8),
        (41, "Guangzhou Taisupanke Computer Equipment", 0x29),
        (42, "Ceremorphic Inc", 0x2A),
        (43, "Biwin Storage Technology Co Ltd", 0xAB),
        (44, "Beijing ESWIN Computing Technology", 0x2C),
        (45, "WeForce Co Ltd", 0xAD),
        (46, "Shenzhen Fanxiang Information Technology", 0xAE),
        (47, "Unisoc", 0x2F),
        (48, "YingChu", 0xB0),
        (49, "GUANCUN", 0x31),
        (50, "IPASON", 0x32),
        (51, "Ayar Labs", 0xB3),
        (52, "Amazon", 0x34),
        (53, "Shenzhen Xinxinshun Technology Co", 0xB5),
        (54, "Galois Inc", 0xB6),
        (55, "Ubilite Inc", 0x37),
        (56, "Shenzhen Quanxing Technology Co Ltd", 0x38),
        (57, "Group RZX Technology LTDA", 0xB9),
        (58, "Yottac Technology (XI’AN) Cooperation", 0xBA),
        (59, "Shenzhen RuiRen Technology Co Ltd", 0x3B),
        (60, "Group Star Technology Co Ltd", 0xBC),
        (61, "RWA (Hong Kong) Ltd", 0x3D),
        (62, "Genesys Logic Inc", 0x3E),
        (63, "T3 Robotics Inc.", 0xBF),
        (64, "Biostar Microtech International Corp", 0x40),
        (65, "Shenzhen SXmicro Technology Co Ltd", 0xC1),
        (66, "Shanghai Yili Computer Technology Co", 0xC2),
        (67, "Zhixin Semicoducotor Co Ltd", 0x43),
        (68, "uFound", 0xC4),
        (69, "Aigo Data Security Technology Co. Ltd", 0x45),
        (70, ".GXore Technologies", 0x46),
        (71, "Shenzhen Pradeon Intelligent Technology", 0xC7),
        (72, "Power LSI", 0xC8),
        (73, "PRIME", 0x49),
        (74, "Shenzhen Juyang Innovative Technology", 0x4A),
        (75, "CERVO", 0xCB),
        (76, "SiEngine Technology Co., Ltd.", 0x4C),
        (77, "Beijing Unigroup Tsingteng MicroSystem", 0xCD),
        (78, "Brainsao GmbH", 0xCE),
        (79, "Credo Technology Group Ltd", 0x4F),
        (80, "Shanghai Biren Technology Co Ltd", 0xD0),
        (81, "Nucleu Semiconductor", 0x51),
        (82, "Shenzhen Guangshuo Electronics Co Ltd", 0x52),
        (83, "ZhongsihangTechnology Co Ltd", 0xD3),
        (84, "Suzhou Mainshine Electronic Co Ltd.", 0x54),
        (85, "Guangzhou Riss Electronic Technology", 0xD5),
        (86, "Shenzhen Cloud Security Storage  Co", 0xD6),
        (87, "ROG", 0x57),
        (88, "Perceive", 0x58),
        (89, "e-peas", 0xD9),
        (90, "Fraunhofer IPMS", 0xDA),
        (91, "Shenzhen Daxinlang Electronic Tech Co", 0x5B),
        (92, "Abacus Peripherals Private Limited", 0xDC),
        (93, "OLOy Technology", 0x5D),
        (94, "Wuhan P&S Semiconductor Co Ltd", 0x5E),
        (95, "Sitrus Technology", 0xDF),
        (96, "AnHui Conner Storage Co Ltd", 0xE0),
        (97, "Rochester Electronics", 0x61),
        (98, "Wuxi Smart Memories Technologies Co", 0x62),
        (99, "Star Memory", 0xE3),
        (100, "Agile Memory Technology Co Ltd", 0x64),
        (101, "MEJEC", 0xE5),
        (102, "Rockchip Electronics Co Ltd", 0xE6),
        (103, "Dongguan Guanma e-commerce Co Ltd", 0x67),
        (104, "Rayson Hi-Tech (SZ) Limited", 0x68),
        (105, "MINRES Technologies GmbH", 0xE9),
        (106, "Himax Technologies Inc", 0xEA),
        (107, "Shenzhen Cwinner Technology Co Ltd", 0x6B),
        (108, "Tecmiyo", 0xEC),
        (109, "Shenzhen Suhuicun Technology Co Ltd", 0x6D),
        (110, "Vickter Electronics Co. Ltd.", 0x6E),
        (111, "lowRISC", 0xEF),
        (112, "EXEGate FZE", 0x70),
        (113, "Shenzhen 9 Chapter Technologies Co", 0xF1),
        (114, "Addlink", 0xF2),
        (115, "Starsway", 0x73),
        (116, "Pensando Systems Inc.", 0xF4),
        (117, "AirDisk", 0x75),
        (118, "Shenzhen Speedmobile Technology Co", 0x76),
        (119, "PEZY Computing", 0xF7),
        (120, "Extreme Engineering Solutions Inc", 0xF8),
        (121, "Shangxin Technology Co Ltd", 0x79),
        (122, "Shanghai Zhaoxin Semiconductor Co", 0x7A),
        (123, "Xsight Labs Ltd", 0xFB),
        (124, "Hangzhou Hikstorage Technology Co", 0x7C),
        (125, "Dell Technologies", 0xFD),
        (126, "Guangdong StarFive Technology Co", 0xFE),
    ],
    [ # Bank 14
        (1, "TECOTON", 0x01),
        (2, "Abko Co Ltd", 0x02),
        (3, "Shenzhen Feisrike Technology Co Ltd", 0x83),
        (4, "Shenzhen Sunhome Electronics Co Ltd", 0x04),
        (5, "Global Mixed-mode Technology Inc", 0x85),
        (6, "Shenzhen Weien Electronics Co. Ltd.", 0x86),
        (7, "Shenzhen Cooyes Technology Co Ltd", 0x07),
        (8, "Keymos Electronics Co., Limited", 0x08),
        (9, "E-Rockic Technology Company Limited", 0x89),
        (10, "Aerospace Science Memory Shenzhen", 0x8A),
        (11, "Shenzhen Quanji Technology Co Ltd", 0x0B),
        (12, "Dukosi", 0x8C),
        (13, "Maxell Corporation of America", 0x0D),
        (14, "Shenshen Xinxintao Electronics Co Ltd", 0x0E),
        (15, "Zhuhai Sanxia Semiconductor Co Ltd", 0x8F),
        (16, "Groq Inc", 0x10),
        (17, "AstraTek", 0x91),
        (18, "Shenzhen Xinyuze Technology  Co Ltd", 0x92),
        (19, "All Bit Semiconductor", 0x13),
        (20, "ACFlow", 0x94),
        (21, "Shenzhen Sipeed Technology Co Ltd", 0x15),
        (22, "Linzhi Hong Kong Co Limited", 0x16),
        (23, "Supreme Wise Limited", 0x97),
        (24, "Blue Cheetah Analog Design Inc", 0x98),
        (25, "Hefei Laiku Technology Co Ltd", 0x19),
        (26, "Zord", 0x1A),
        (27, "SBO Hearing A/S", 0x9B),
        (28, "Regent Sharp International Limited", 0x1C),
        (29, "Permanent Potential Limited", 0x9D),
        (30, "Creative World International Limited", 0x9E),
        (31, "Base Creation International Limited", 0x1F),
        (32, "Shenzhen Zhixin Chuanglian Technology", 0x20),
        (33, "Protected Logic Corporation", 0xA1),
        (34, "Sabrent", 0xA2),
        (35, "Union Memory", 0x23),
        (36, "NEUCHIPS Corporation", 0xA4),
        (37, "Ingenic Semiconductor Co Ltd", 0x25),
        (38, "SiPearl", 0x26),
        (39, "Shenzhen Actseno Information Technology", 0xA7),
        (40, "RIVAI Technologies (Shenzhen) Co Ltd", 0xA8),
        (41, "Shenzhen Sunny Technology Co Ltd", 0x29),
        (42, "Cott Electronics Ltd", 0x2A),
        (43, "Shanghai Synsense Technologies Co Ltd", 0xAB),
        (44, "Shenzhen Jintang Fuming Optoelectronics", 0x2C),
        (45, "CloudBEAR LLC", 0xAD),
        (46, "Emzior, LLC", 0xAE),
        (47, "Ehiway Microelectronic Science Tech Co", 0x2F),
        (48, "UNIM Innovation Technology (Wu XI)", 0xB0),
        (49, "GDRAMARS", 0x31),
        (50, "Meminsights Technology", 0x32),
        (51, "Zhuzhou Hongda Electronics Corp Ltd", 0xB3),
        (52, "Luminous Computing Inc", 0x34),
        (53, "PROXMEM", 0xB5),
        (54, "Draper Labs", 0xB6),
        (55, "ORICO  Technologies Co. Ltd.", 0x37),
        (56, "Space Exploration Technologies Corp", 0x38),
        (57, "AONDEVICES Inc", 0xB9),
        (58, "Shenzhen Netforward Micro Electronic", 0xBA),
        (59, "Syntacore Ltd", 0x3B),
        (60, "Shenzhen Secmem Microelectronics Co", 0xBC),
        (61, "ONiO As", 0x3D),
        (62, "Shenzhen Peladn Technology Co Ltd", 0x3E),
        (63, "O-Cubes Shanghai Microelectronics", 0xBF),
        (64, "ASTC", 0x40),
        (65, "UMIS", 0xC1),
        (66, "Paradromics", 0xC2),
        (67, "Sinh Micro Co Ltd", 0x43),
        (68, "Metorage Semiconductor Technology Co", 0xC4),
        (69, "Aeva Inc", 0x45),
        (70, "HongKong Hyunion Electronics Co Ltd", 0x46),
        (71, "China Flash Co Ltd", 0xC7),
        (72, "Sunplus Technology Co Ltd", 0xC8),
        (73, "Idaho Scientific", 0x49),
        (74, "Suzhou SF Micro Electronics Co Ltd", 0x4A),
        (75, "IMEX Cap AG", 0xCB),
        (76, "Fitipower Integrated  Technology Co Ltd", 0x4C),
        (77, "ShenzhenWooacme Technology Co Ltd", 0xCD),
        (78, "KeepData Original Chips", 0xCE),
        (79, "Rivos Inc", 0x4F),
        (80, "Big Innovation Company Limited", 0xD0),
        (81, "Wuhan YuXin Semiconductor Co Ltd", 0x51),
        (82, "United Memory Technology (Jiangsu)", 0x52),
        (83, "PQShield Ltd", 0xD3),
        (84, "ArchiTek Corporation", 0x54),
        (85, "ShenZhen AZW Technology Co Ltd", 0xD5),
        (86, "Hengchi Zhixin (Dongguan) Technology", 0xD6),
        (87, "Eggtronic  Engineering Spa", 0x57),
        (88, "Fusontai Technology", 0x58),
        (89, "PULP Platform", 0xD9),
        (90, "Koitek Electronic Technology (Shenzhen) Co", 0xDA),
        (91, "Shenzhen Jiteng Network Technology Co", 0x5B),
        (92, "Aviva Links Inc", 0xDC),
        (93, "Trilinear Technologies Inc", 0x5D),
        (94, "Shenzhen Developer Microelectronics Co", 0x5E),
        (95, "Guangdong OPPO Mobile Telecommunication", 0xDF),
        (96, "Akeana", 0xE0),
        (97, "Lyczar", 0x61),
        (98, "Shenzhen Qiji Technology Co Ltd", 0x62),
        (99, "Shenzhen Shangzhaoyuan Technology", 0xE3),
        (100, "Han Stor", 0x64),
        (101, "China Micro Semicon Co., Ltd.", 0xE5),
        (102, "Shenzhen Zhuqin Technology Co Ltd", 0xE6),
        (103, "Shanghai Ningyuan Electronic Technology", 0x67),
        (104, "Auradine", 0x68),
        (105, "Suzhou Yishuo Electronics Co Ltd", 0xE9),
        (106, "Faurecia Clarion Electronics", 0xEA),
        (107, "SiMa Technologies", 0x6B),
        (108, "CFD Sales Inc", 0xEC),
        (109, "Suzhou Comay Information Co Ltd", 0x6D),
        (110, "Yentek", 0x6E),
        (111, "Qorvo Inc", 0xEF),
        (112, "Shenzhen Youzhi Computer Technology", 0x70),
        (113, "Sychw Technology (Shenzhen) Co Ltd", 0xF1),
        (114, "MK Founder Technology Co Ltd", 0xF2),
        (115, "Siliconwaves Technologies Co Ltd", 0x73),
        (116, "Hongkong Hyunion Electronics Co Ltd", 0xF4),
        (117, "Shenzhen Xinxinzhitao Electronics Business", 0x75),
        (118, "Shenzhen HenQi Electronic Commerce Co", 0x76),
        (119, "Shenzhen Jingyi Technology Co Ltd", 0xF7),
        (120, "Xiaohua Semiconductor Co. Ltd.", 0xF8),
        (121, "Shenzhen Dalu Semiconductor Technology", 0x79),
        (122, "Shenzhen Ninespeed Electronics Co Ltd", 0x7A),
        (123, "ICYC Semiconductor Co Ltd", 0xFB),
        (124, "Shenzhen Jaguar Microsystems Co Ltd", 0x7C),
        (125, "Beijing EC-Founder Co Ltd", 0xFD),
        (126, "Shenzhen Taike Industrial Automation Co", 0xFE),
    ],
    [ # Bank 15
        (1, "Kalray SA", 0x01),
        (2, "Shanghai Iluvatar CoreX Semiconductor Co", 0x02),
        (3, "Fungible Inc", 0x83),
        (4, "Song Industria E Comercio de Eletronicos", 0x04),
        (5, "DreamBig Semiconductor Inc", 0x85),
        (6, "ChampTek Electronics Corp", 0x86),
        (7, "Fusontai Technology", 0x07),
        (8, "Endress Hauser AG", 0x08),
        (9, "altec ComputerSysteme GmbH", 0x89),
        (10, "UltraRISC Technology (Shanghai) Co Ltd", 0x8A),
        (11, "Shenzhen Jing Da Kang Technology Co Ltd", 0x0B),
        (12, "Hangzhou Hongjun Microelectronics Co Ltd", 0x8C),
        (13, "Pliops Ltd", 0x0D),
        (14, "Cix Technology (Shanghai) Co Ltd", 0x0E),
        (15, "TeraDevices Inc", 0x8F),
        (16, "SpacemiT (Hangzhou)Technology Co Ltd", 0x10),
        (17, "InnoPhase loT Inc", 0x91),
        (18, "InnoPhase loT Inc", 0x92),
        (19, "Yunhight Microelectronics", 0x13),
        (20, "Samnix", 0x94),
        (21, "HKC Storage Co Ltd", 0x15),
        (22, "Chiplego Technology (Shanghai) Co Ltd", 0x16),
        (23, "StoreSkill", 0x97),
        (24, "Shenzhen Astou Technology Company", 0x98),
        (25, "Guangdong LeafFive Technology Limited", 0x19),
        (26, "Jin JuQuan", 0x1A),
        (27, "Huaxuan Technology (Shenzhen) Co Ltd", 0x9B),
        (28, "Gigastone Corporation", 0x1C),
        (29, "Kinsotin", 0x9D),
        (30, "PengYing", 0x9E),
        (31, "Shenzhen Xunhi Technology Co Ltd", 0x1F),
        (32, "FOXX Storage Inc", 0x20),
        (33, "Shanghai Belling Corporation Ltd", 0xA1),
        (34, "Glenfy Tech Co Ltd", 0xA2),
        (35, "Sahasra Semiconductors Pvt Ltd", 0x23),
        (36, "Chongqing SeekWave Technology Co Ltd", 0xA4),
        (37, "Shenzhen Zhixing Intelligent Manufacturing", 0x25),
        (38, "Ethernovia", 0x26),
        (39, "Shenzhen Xinrongda Technology Co Ltd", 0xA7),
        (40, "Hangzhou Clounix Technology Limited", 0xA8),
        (41, "JGINYUE", 0x29),
        (42, "Shenzhen Xinwei Semiconductor Co Ltd", 0x2A),
        (43, "COLORFIRE Technology Co Ltd", 0xAB),
        (44, "B LKE", 0x2C),
        (45, "ZHUDIAN", 0xAD),
        (46, "REECHO", 0xAE),
        (47, "Enphase Energy Inc", 0x2F),
        (48, "Shenzhen Yingrui Storage Technology Co Ltd", 0xB0),
        (49, "Shenzhen Sinomos Semiconductor Technology", 0x31),
        (50, "O2micro International Limited", 0x32),
        (51, "Axelera AI BV", 0xB3),
        (52, "Silicon Legend Technology (Suzhou) Co Ltd", 0x34),
        (53, "Suzhou Novosense Microelectronics Co Ltd", 0xB5),
        (54, "Pirateman", 0xB6),
        (55, "Yangtze MasonSemi", 0x37),
        (56, "Shanghai Yunsilicon Technology Co Ltd", 0x38),
        (57, "Rayson", 0xB9),
        (58, "Alphawave IP", 0xBA),
        (59, "Shenzhen Visions Chip Electronic Technology", 0x3B),
        (60, "KYO Group", 0xBC),
        (61, "Shenzhen Aboison Technology Co Ltd", 0x3D),
        (62, "Shenzhen JingSheng Semiconducto Co Ltd", 0x3E),
        (63, "Shenzhen Dingsheng Technology Co Ltd", 0xBF),
        (64, "EVAS Intelligence Co Ltd", 0x40),
        (65, "Kaibright Electronic Technologies", 0xC1),
        (66, "Fraunhofer IMS", 0xC2),
        (67, "Shenzhen Xinrui Renhe Technology", 0x43),
        (68, "Beijing Vcore Technology Co Ltd", 0xC4),
        (69, "Silicon Innovation Technologies Co Ltd", 0x45),
        (70, "Shenzhen Zhengxinda Technology Co Ltd", 0x46),
        (71, "Shenzhen Remai Electronics Co Lttd", 0xC7),
        (72, "Shenzhen Xinruiyan Electronics Co Ltd", 0xC8),
        (73, "CEC Huada Electronic Design Co Ltd", 0x49),
        (74, "Westberry Technology Inc", 0x4A),
        (75, "Tongxin Microelectronics Co Ltd", 0xCB),
        (76, "UNIM Semiconductor (Shang Hai) Co Ltd", 0x4C),
        (77, "Shenzhen Qiaowenxingyu Industrial Co Ltd", 0xCD),
        (78, "ICC", 0xCE),
        (79, "Enfabrica Corporation", 0x4F),
        (80, "Niobium Microsystems Inc", 0xD0),
        (81, "Xiaoli AI Electronics (Shenzhen) Co Ltd", 0x51),
        (82, "Silicon Mitus", 0x52),
        (83, "Ajiatek Inc", 0xD3),
        (84, "HomeNet", 0x54),
        (85, "Shenzhen Shubang Technology Co Ltd", 0xD5),
        (86, "Exacta Technologies Ltd", 0xD6),
        (87, "Synology", 0x57),
        (88, "Trium Elektronik Bilgi Islem San Ve Dis", 0x58),
        (89, "Wuxi HippStor Technology Co Ltd", 0xD9),
        (90, "SSCT", 0xDA),
        (91, "Sichuan Heentai Semiconductor Co Ltd", 0x5B),
        (92, "Zhejiang University", 0xDC),
        (93, "www.shingroup.cn", 0x5D),
        (94, "Suzhou Nano Mchip Technology Company", 0x5E),
        (95, "Feature Integration Technology Inc", 0xDF),
        (96, "d-Matrix", 0xE0),
        (97, "Golden Memory", 0x61),
        (98, "Qingdao Thunderobot Technology Co Ltd", 0x62),
        (99, "Shenzhen Tianxiang Chuangxin Technology", 0xE3),
        (100, "HYPHY USA", 0x64),
        (101, "Valkyrie", 0xE5),
        (102, "Suzhou Hesetc Electronic Technology Co", 0xE6),
        (103, "Hainan Zhongyuncun Technology Co Ltd", 0x67),
        (104, "Shenzhen Yousheng Bona Technology Co", 0x68),
        (105, "Shenzhen Xinle Chuang Technology Co", 0xE9),
        (106, "DEEPX", 0xEA),
        (107, "iStarChip CA LLC", 0x6B),
        (108, "Shenzhen Vinreada Technology Co Ltd", 0xEC),
        (109, "Novatek Microelectronics Corp", 0x6D),
        (110, "Chemgdu EG Technology Co Ltd", 0x6E),
        (111, "AGI Technology", 0xEF),
        (112, "Syntiant", 0x70),
        (113, "AOC", 0xF1),
        (114, "GamePP", 0xF2),
        (115, "Yibai Electronic Technologies", 0x73),
        (116, "Hangzhou Rencheng Trading Co Ltd", 0xF4),
        (117, "HOGE Technology Co Ltd", 0x75),
        (118, "United Micro Technology (Shenzhen) Co", 0x76),
        (119, "Fabric of Truth Inc", 0xF7),
        (120, "Epitech", 0xF8),
        (121, "Elitestek", 0x79),
        (122, "Cornelis Networks Inc", 0x7A),
        (123, "WingSemi Technologies Co Ltd", 0xFB),
        (124, "ForwardEdge ASIC", 0x7C),
        (125, "Beijing Future Imprint Technology Co Ltd", 0xFD),
        (126, "Fine Made Microelectronics Group Co Ltd", 0xFE),
    ],
    [ # Bank 16
        (1, "Changxin Memory Technology (Shanghai)", 0x01),
        (2, "Synconv", 0x02),
        (3, "MULTIUNIT", 0x83),
        (4, "Zero ASIC Corporation", 0x04),
        (5, "NTT Innovative Devices Corporation", 0x85),
        (6, "Xbstor", 0x86),
        (7, "Shenzhen South Electron Co Ltd", 0x07),
        (8, "Iontra Inc", 0x08),
        (9, "SIEFFI Inc", 0x89),
        (10, "HK Winston Electronics Co Limited", 0x8A),
        (11, "Anhui SunChip Semiconductor Technology", 0x0B),
        (12, "HaiLa Technologies Inc", 0x8C),
        (13, "AUTOTALKS", 0x0D),
        (14, "Shenzhen Ranshuo Technology Co Limited", 0x0E),
        (15, "ScaleFlux", 0x8F),
        (16, "XC Memory", 0x10),
        (17, "Guangzhou Beimu Technology Co., Ltd", 0x91),
        (18, "Rays Semiconductor Nanjing Co Ltd", 0x92),
        (19, "Milli-Centi Intelligence Technology Jiangsu", 0x13),
        (20, "Zilia Technologioes", 0x94),
        (21, "Incore Semiconductors", 0x15),
        (22, "Kinetic Technologies", 0x16),
        (23, "Nanjing Houmo Technology Co Ltd", 0x97),
        (24, "Suzhou Yige Technology Co Ltd", 0x98),
        (25, "Shenzhen Techwinsemi Technology Co Ltd", 0x19),
        (26, "Pure Array Technology (Shanghai) Co. Ltd", 0x1A),
        (27, "Shenzhen Techwinsemi Technology Udstore", 0x9B),
        (28, "RISE MODE", 0x1C),
        (29, "NEWREESTAR", 0x9D),
        (30, "Hangzhou Hualan Microeletronique Co Ltd", 0x9E),
        (31, "Senscomm Semiconductor Co Ltd", 0x1F),
        (32, "Holt Integrated Circuits", 0x20),
        (33, "Tenstorrent Inc", 0xA1),
        (34, "SkyeChip", 0xA2),
        (35, "Guangzhou Kaishile Trading Co Ltd", 0x23),
        (36, "Jing Pai Digital Technology (Shenzhen) Co", 0xA4),
    ]
]

def _jedec_update_mfg_from_pdf():
    import fitz
    import re
    import argparse

    parser = argparse.ArgumentParser(description="Update manufactures ID from JEDEC's official list distributed via PDF")
    parser.add_argument("pdf_filepath", default="JEP106BF01.pdf", help="Offical JEP106 PDF")
    parser.add_argument("output_filepath", nargs='?', default=__file__, help="Optional output file. Default: 'jedec.py'")
    args = parser.parse_args()

    pdf_text = ''
    with fitz.open(args.pdf_filepath) as doc:
        for page in doc:
            pdf_text += page.get_text()
    jep106_matches = re.findall(r'^(\d+) (.+(?:\n.+){0,1})(?:(?:\n\d ){8})(?:\n)([0-9A-F]{2})', pdf_text, re.MULTILINE)

    output_text = ""
    with open(__file__) as source_file:
        output_text += source_file.read()

    # Update version string
    output_text = re.sub(r"^# Version.*", f"# Version: JEDEC {args.pdf_filepath}", output_text)

    manufacturers_array = "_manufacturers = [\n"
    bank_no = 0
    for entry in jep106_matches:
        entry = tuple(item.replace('\n', '').strip() for item in entry)
        if entry[0] == '1':
            bank_no += 1
            if (bank_no > 1):
                manufacturers_array += f"    ],\n"
            manufacturers_array += f"    [ # Bank {bank_no}\n"
        manufacturers_array += f"        ({int(entry[0])}, \"{entry[1]}\", 0x{int(entry[2], 16):02X}),\n"
    manufacturers_array += "    ]\n]"

    # Update Manufacturers array
    output_text = re.sub(r"_manufacturers = \[(.|\n)*^\]", manufacturers_array, output_text, flags=re.MULTILINE)

    with open(args.output_filepath, 'w') as f:
        print(output_text, file=f, end='')

if __name__ == "__main__":
    _jedec_update_mfg_from_pdf()
