searchData={"items":[{"type":"module","title":"mnesia","doc":"A distributed key-value DBMS\n\nThe following are some of the most important and attractive capabilities\nprovided by Mnesia:\n\n- A relational/object hybrid data model that is suitable for telecommunications\n  applications.\n- A DBMS query language, Query List Comprehension (QLC) as an add-on library.\n- Persistence. Tables can be coherently kept on disc and in the main memory.\n- Replication. Tables can be replicated at several nodes.\n- Atomic transactions. A series of table manipulation operations can be grouped\n  into a single atomic transaction.\n- Location transparency. Programs can be written without knowledge of the actual\n  data location.\n- Extremely fast real-time data searches.\n- Schema manipulation routines. The DBMS can be reconfigured at runtime without\n  stopping the system.\n\nThis Reference Manual describes the Mnesia API. This includes functions that\ndefine and manipulate Mnesia tables.\n\nAll functions in this Reference Manual can be used in any combination with\nqueries using the list comprehension notation. For information about the query\nnotation, see the `m:qlc` manual page in STDLIB.\n\nData in Mnesia is organized as a set of tables. Each table has a name that must\nbe an atom. Each table is made up of Erlang records. The user is responsible for\nthe record definitions. Each table also has a set of properties. The following\nare some of the properties that are associated with each table:\n\n- `type`. Each table can have `set`, `ordered_set`, or `bag` semantics. Notice\n  that currently `ordered_set` is not supported for `disc_only_copies`.\n\n  If a table is of type `set`, each key leads to either one or zero records.\n\n  If a new item is inserted with the same key as an existing record, the old\n  record is overwritten. However, if a table is of type `bag`, each key can map\n  to several records. All records in type `bag` tables are unique, only the keys\n  can be duplicated.\n\n- `record_name`. All records stored in a table must have the same name. The\n  records must be instances of the same record type.\n- `ram_copies`. A table can be replicated on a number of Erlang nodes. Property\n  `ram_copies` specifies a list of Erlang nodes where RAM copies are kept. These\n  copies can be dumped to disc at regular intervals. However, updates to these\n  copies are not written to disc on a transaction basis.\n- `disc_copies`. This property specifies a list of Erlang nodes where the table\n  is kept in RAM and on disc. All updates of the table are performed in the\n  actual table and are also logged to disc. If a table is of type `disc_copies`\n  at a certain node, the entire table is resident in RAM memory and on disc.\n  Each transaction performed on the table is appended to a `LOG` file and\n  written into the RAM table.\n- `disc_only_copies`. Some, or all, table replicas can be kept on disc only.\n  These replicas are considerably slower than the RAM-based replicas.\n- `index`. This is a list of attribute names, or integers, which specify the\n  tuple positions on which Mnesia is to build and maintain an extra index table.\n- `local_content`. When an application requires tables whose contents are local\n  to each node, `local_content` tables can be used. The table name is known to\n  all Mnesia nodes, but its content is unique on each node. This means that\n  access to such a table must be done locally. Set field `local_content` to\n  `true` to enable the `local_content` behavior. Default is `false`.\n- `majority`. This attribute is `true` or `false`; default is `false`. When\n  `true`, a majority of the table replicas must be available for an update to\n  succeed. Majority checking can be enabled on tables with mission-critical\n  data, where it is vital to avoid inconsistencies because of network splits.\n- `snmp`. Each (set-based) Mnesia table can be automatically turned into a\n  Simple Network Management Protocol (SNMP) ordered table as well. This property\n  specifies the types of the SNMP keys.\n- `attributes`. The names of the attributes for the records that are inserted in\n  the table.\n\nFor information about the complete set of table properties and their details,\nsee `mnesia:create_table/2`.\n\nThis Reference Manual uses a table of persons to illustrate various examples.\nThe following record definition is assumed:\n\n```erlang\n-record(person, {name,\n                 age = 0,\n                 address = unknown,\n                 salary = 0,\n                 children = []}),\n```\n\nThe first record attribute is the primary key, or key for short.\n\nThe function descriptions are sorted in alphabetical order. It is recommended to\nstart to read about `mnesia:create_table/2`, `mnesia:lock/2`, and\n `mnesia:activity/4` before you continue and learn about the rest.\n\nWriting or deleting in transaction-context creates a local copy of each modified\nrecord during the transaction. During iteration, that is, `mnesia:foldl/4`,\n`mnesia:foldr/4`, `mnesia:next/2`, `mnesia:prev/2`, and `mnesia:snmp_get_next_index/2`, Mnesia\ncompensates for every written or deleted record, which can reduce the\nperformance.\n\nIf possible, avoid writing or deleting records in the same transaction before\niterating over the table.","ref":"mnesia.html"},{"type":"module","title":"Configuration Parameters - mnesia","doc":"[](){: #configuration_parameters }\n\nMnesia reads the following application configuration parameters:\n\n- `-mnesia access_module Module`. The name of the Mnesia activity access\n  callback module. Default is `mnesia`.\n- `-mnesia auto_repair true | false`. This flag controls if Mnesia automatically\n  tries to repair files that have not been properly closed. Default is `true`.\n- `-mnesia backup_module Module`. The name of the Mnesia backup callback module.\n  Default is `mnesia_backup`.\n- `-mnesia debug Level`. Controls the debug level of Mnesia. The possible values\n  are as follows:\n\n  - **`none`** - No trace outputs. This is the default.\n\n  - **`verbose`** - Activates tracing of important debug events. These events\n    generate `{mnesia_info, Format, Args}` system events. Processes can\n    subscribe to these events with `mnesia:subscribe/1`. The events are always\n    sent to the Mnesia event handler.\n\n  - **`debug`** - Activates all events at the verbose level plus full trace of\n    all debug events. These debug events generate `{mnesia_info, Format, Args}`\n    system events. Processes can subscribe to these events with\n    `mnesia:subscribe/1`. The events are always sent to the Mnesia event\n    handler. On this debug level, the Mnesia event handler starts subscribing to\n    updates in the schema table.\n\n  - **`trace`** - Activates all events at the debug level. On this level, the\n    Mnesia event handler starts subscribing to updates on all Mnesia tables.\n    This level is intended only for debugging small toy systems, as many large\n    events can be generated.\n\n  - **`false`** - An alias for none.\n\n  - **`true`** - An alias for debug.\n\n- `-mnesia core_dir Directory`. The name of the directory where Mnesia core\n  files is stored, or false. Setting it implies that also RAM-only nodes\n  generate a core file if a crash occurs.\n- `-mnesia dc_dump_limit Number`. Controls how often `disc_copies` tables are\n  dumped from memory. Tables are dumped when\n  `filesize(Log) > (filesize(Tab)/Dc_dump_limit)`. Lower values reduce CPU\n  overhead but increase disk space and startup times. Default is 4.\n- `-mnesia dir Directory`. The name of the directory where all Mnesia data is\n  stored. The directory name must be unique for the current node. Two nodes must\n  never share the the same Mnesia directory. The results are unpredictable.\n- `-mnesia dump_disc_copies_at_startup true | false`. If set to false, this\n  disables the dumping of `disc_copies` tables during startup while tables are\n  being loaded. The default is true.\n- `-mnesia dump_log_load_regulation true | false`. Controls if log dumps are to\n  be performed as fast as possible, or if the dumper is to do its own load\n  regulation. Default is `false`.\n\n  This feature is temporary and will be removed in a future release\n\n- `-mnesia dump_log_update_in_place true | false`. Controls if log dumps are\n  performed on a copy of the original data file, or if the log dump is performed\n  on the original data file. Default is `true`\n- [](){: #dump_log_write_threshold } `-mnesia dump_log_write_threshold Max`.\n  `Max` is an integer that specifies the maximum number of writes allowed to the\n  transaction log before a new dump of the log is performed. Default is `1000`\n  log writes.\n- [](){: #dump_log_time_threshold } `-mnesia dump_log_time_threshold Max`. `Max`\n  is an integer that specifies the dump log interval in milliseconds. Default is\n  3 minutes. If a dump has not been performed within `dump_log_time_threshold`\n  milliseconds, a new dump is performed regardless of the number of writes\n  performed.\n- `-mnesia event_module Module`. The name of the Mnesia event handler callback\n  module. Default is `mnesia_event`.\n- `-mnesia extra_db_nodes Nodes` specifies a list of nodes, in addition to the\n  ones found in the schema, with which Mnesia is also to establish contact.\n  Default is `[]` (empty list).\n- `-mnesia fallback_error_function {UserModule, UserFunc}`. Specifies a\n  user-supplied callback function, which is called if a fallback is installed\n  and Mnesia goes down on another node. Mnesia calls the function with one\n  argument, the name of the dying node, for example,\n  `UserModule:UserFunc(DyingNode)`. Mnesia must be restarted, otherwise the\n  database can be inconsistent. The default behavior is to terminate Mnesia.\n- `-mnesia max_wait_for_decision Timeout`. Specifies how long Mnesia waits for\n  other nodes to share their knowledge about the outcome of an unclear\n  transaction. By default, `Timeout` is set to the atom `infinity`. This implies\n  that if Mnesia upon startup detects a \"heavyweight transaction\" whose outcome\n  is unclear, the local Mnesia waits until Mnesia is started on some (in the\n  worst case all) of the other nodes that were involved in the interrupted\n  transaction. This is a rare situation, but if it occurs, Mnesia does not guess\n  if the transaction on the other nodes was committed or terminated. Mnesia\n  waits until it knows the outcome and then acts accordingly.\n\n  If `Timeout` is set to an integer value in milliseconds, Mnesia forces\n  \"heavyweight transactions\" to be finished, even if the outcome of the\n  transaction for the moment is unclear. After `Timeout` milliseconds, Mnesia\n  commits or terminates the transaction and continues with the startup. This can\n  lead to a situation where the transaction is committed on some nodes and\n  terminated on other nodes. If the transaction is a schema transaction, the\n  inconsistency can be fatal.\n\n- `-mnesia no_table_loaders NUMBER`. Specifies the number of parallel table\n  loaders during start. More loaders can be good if the network latency is high\n  or if many tables contain few records. Default is `2`.\n- `-mnesia send_compressed Level`. Specifies the level of compression to be used\n  when copying a table from the local node to another one. Default is `0`.\n\n  `Level` must be an integer in the interval `[0, 9]`, where `0` means no\n  compression and `9` means maximum compression. Before setting it to a non-zero\n  value, ensure that the remote nodes understand this configuration.\n\n- `-mnesia max_transfer_size Number`. Specifies the estimated size in bytes of a\n  single packet of data to be used when copying a table from the local node to\n  another one. Default is `64000`.\n- `-mnesia schema_location Loc`. Controls where Mnesia looks for its schema.\n  Parameter `Loc` can be one of the following atoms:\n\n  - **`disc`** - Mandatory disc. The schema is assumed to be located in the\n    Mnesia directory. If the schema cannot be found, Mnesia refuses to start.\n    This is the old behavior.\n\n  - **`ram`** - Mandatory RAM. The schema resides in RAM only. At startup, a\n    tiny new schema is generated. This default schema only contains the\n    definition of the schema table and only resides on the local node. Since no\n    other nodes are found in the default schema, configuration parameter\n    `extra_db_nodes` must be used to let the node share its table definitions\n    with other nodes.\n\n    Parameter `extra_db_nodes` can also be used on disc based nodes.\n\n  - **`opt_disc`** - Optional disc. The schema can reside on disc or in RAM. If\n    the schema is found on disc, Mnesia starts as a disc-based node and the\n    storage type of the schema table is `disc_copies`. If no schema is found on\n    disc, Mnesia starts as a disc-less node and the storage type of the schema\n    table is `ram_copies`. Default value for the application parameter is\n    `opt_disc`.\n\nFirst, the SASL application parameters are checked, then the command-line flags\nare checked, and finally, the default value is chosen.","ref":"mnesia.html#module-configuration-parameters"},{"type":"module","title":"See Also - mnesia","doc":"`m:application`, `m:dets`, `m:disk_log`, `m:ets`, `m:qlc`","ref":"mnesia.html#module-see-also"},{"type":"function","title":"mnesia.abort/1","doc":"Terminate the current transaction.\n\nMakes the transaction silently return the tuple `{aborted, Reason}`. Termination\nof a Mnesia transaction means that an exception is thrown to an enclosing\n`catch`. Thus, the expression `catch mnesia:abort(x)` does not terminate the\ntransaction.","ref":"mnesia.html#abort/1"},{"type":"function","title":"mnesia.activate_checkpoint/1","doc":"Activate a checkpoint.\n\nA checkpoint is a consistent view of the system. A checkpoint can be activated\non a set of tables. This checkpoint can then be traversed and presents a view of\nthe system as it existed at the time when the checkpoint was activated, even if\nthe tables are being or have been manipulated.\n\n`Args` is a list of the following tuples:\n\n- `{name,Name}`. `Name` is the checkpoint name. Each checkpoint must have a name\n  that is unique to the associated nodes. The name can be reused only once the\n  checkpoint has been deactivated. By default, a name that is probably unique is\n  generated.\n- `{max,MaxTabs}`. `MaxTabs` is a list of tables that are to be included in the\n  checkpoint. Default is `[]`. For these tables, the redundancy is maximized and\n  checkpoint information is retained together with all replicas. The checkpoint\n  becomes more fault tolerant if the tables have several replicas. When a new\n  replica is added by the schema manipulation function\n  `mnesia:add_table_copy/3`, a retainer is also attached automatically.\n- `{min,MinTabs}`. `MinTabs` is a list of tables that are to be included in the\n  checkpoint. Default is []. For these tables, the redundancy is minimized and\n  the checkpoint information is only retained with one replica, preferably on\n  the local node.\n- `{allow_remote,Bool}`. `false` means that all retainers must be local. The\n  checkpoint cannot be activated if a table does not reside locally. `true`\n  allows retainers to be allocated on any node. Default is `true`.\n- `{ram_overrides_dump,Bool}`. Only applicable for `ram_copies`. `Bool` allows\n  you to choose to back up the table state as it is in RAM, or as it is on disc.\n  `true` means that the latest committed records in RAM are to be included in\n  the checkpoint. These are the records that the application accesses. `false`\n  means that the records dumped to `DAT` files are to be included in the\n  checkpoint. These records are loaded at startup. Default is `false`.\n\nReturns `{ok,Name,Nodes}` or `{error,Reason}`. `Name` is the (possibly\ngenerated) checkpoint name. `Nodes` are the nodes that are involved in the\ncheckpoint. Only nodes that keep a checkpoint retainer know about the\ncheckpoint.","ref":"mnesia.html#activate_checkpoint/1"},{"type":"function","title":"mnesia.activity/2","doc":"Execute `Fun` in `AccessContext`.\n\nCalls [`mnesia:activity(AccessContext, Fun, Args, AccessMod)`](`activity/4`), where `AccessMod`\nis the default access callback module obtained by\n`mnesia:system_info(access_module)`. `Args` defaults to `[]` (empty list).","ref":"mnesia.html#activity/2"},{"type":"function","title":"mnesia.activity/4","doc":"Execute `Fun` in `AccessContext`.\n\nExecutes the functional object `Fun` with argument `Args`.\n\nThe code that executes inside the activity can consist of a series of table\nmanipulation functions, which are performed in an `AccessContext`. Currently,\nthe following access contexts are supported:\n\n- **`transaction`** - Short for `{transaction, infinity}`\n\n- **`{transaction, Retries}`** - Calls [`mnesia:transaction(Fun, Args, Retries)`](`transaction/3`).\n  Notice that the result from `Fun` is returned if the transaction is successful\n  (atomic), otherwise the function exits with an abort reason.\n\n- **`sync_transaction`** - Short for `{sync_transaction, infinity}`\n\n- **`{sync_transaction, Retries}`** - Calls\n  [`mnesia:sync_transaction(Fun, Args, Retries)`](`sync_transaction/3`). Notice that the result from\n  `Fun` is returned if the transaction is successful (atomic), otherwise the\n  function exits with an abort reason.\n\n- **`async_dirty`** - Calls `mnesia:async_dirty(Fun, Args)`.\n\n- **`sync_dirty`** - Calls `mnesia:sync_dirty(Fun, Args)`.\n\n- **`ets`** - Calls `mnesia:ets(Fun, Args)`.\n\nThis function (`mnesia:activity/4`) differs in an important way from the\nfunctions `mnesia:transaction/3`, `mnesia:sync_transaction/3`, `mnesia:async_dirty/2`,\n`mnesia:sync_dirty/2`, and `mnesia:ets/2`. Argument `AccessMod` is the name of a\ncallback module, which implements the `mnesia_access` behavior.\n\nMnesia forwards calls to the following functions:\n\n- mnesia:lock/2 (read_lock_table/1, write_lock_table/1)\n- mnesia:write/3 (write/1, s_write/1)\n- mnesia:delete/3 (delete/1, s_delete/1)\n- mnesia:delete_object/3 (delete_object/1, s_delete_object/1)\n- mnesia:read/3 (read/1, wread/1)\n- mnesia:match_object/3 (match_object/1)\n- mnesia:all_keys/1\n- mnesia:first/1\n- mnesia:last/1\n- mnesia:prev/2\n- mnesia:next/2\n- mnesia:index_match_object/4 (index_match_object/2)\n- mnesia:index_read/3\n- mnesia:table_info/2\n\nto the corresponding:\n\n- AccessMod:lock(ActivityId, Opaque, LockItem, LockKind)\n- AccessMod:write(ActivityId, Opaque, Tab, Rec, LockKind)\n- AccessMod:delete(ActivityId, Opaque, Tab, Key, LockKind)\n- AccessMod:delete_object(ActivityId, Opaque, Tab, RecXS, LockKind)\n- AccessMod:read(ActivityId, Opaque, Tab, Key, LockKind)\n- AccessMod:match_object(ActivityId, Opaque, Tab, Pattern, LockKind)\n- AccessMod:all_keys(ActivityId, Opaque, Tab, LockKind)\n- AccessMod:first(ActivityId, Opaque, Tab)\n- AccessMod:last(ActivityId, Opaque, Tab)\n- AccessMod:prev(ActivityId, Opaque, Tab, Key)\n- AccessMod:next(ActivityId, Opaque, Tab, Key)\n- AccessMod:index_match_object(ActivityId, Opaque, Tab, Pattern, Attr, LockKind)\n- AccessMod:index_read(ActivityId, Opaque, Tab, SecondaryKey, Attr, LockKind)\n- AccessMod:table_info(ActivityId, Opaque, Tab, InfoItem)\n\n`ActivityId` is a record that represents the identity of the enclosing Mnesia\nactivity. The first field (obtained with\n[`element(1, ActivityId)`](`element/2`)) contains an atom, which can be\ninterpreted as the activity type: `ets`, `async_dirty`, `sync_dirty`, or `tid`.\n`tid` means that the activity is a transaction. The structure of the rest of the\nidentity record is internal to Mnesia.\n\n`Opaque` is an opaque data structure that is internal to Mnesia.","ref":"mnesia.html#activity/4"},{"type":"function","title":"mnesia.add_table_copy/3","doc":"Copy a table to a remote node.\n\nMakes another copy of a table at the node `Node`. Argument `Type` must be either\nof the atoms `ram_copies`, `disc_copies`, or `disc_only_copies`. For example,\nthe following call ensures that a disc replica of the `person` table also exists\nat node `Node`:\n\n```text\nmnesia:add_table_copy(person, Node, disc_copies)\n```\n\nThis function can also be used to add a replica of the table named `schema`.","ref":"mnesia.html#add_table_copy/3"},{"type":"function","title":"mnesia.add_table_index/2","doc":"Add table index.\n\nTable indexes can be used whenever the user wants to use frequently some other\nfield than the key field to look up records. If this other field has an\nassociated index, these lookups can occur in constant time and space. For\nexample, if your application wishes to use field `age` to find efficiently all\npersons with a specific age, it can be a good idea to have an index on field\n`age`. This can be done with the following call:\n\n```text\nmnesia:add_table_index(person, age)\n```\n\nIndexes do not come for free. They occupy space that is proportional to the\ntable size, and they cause insertions into the table to execute slightly slower.","ref":"mnesia.html#add_table_index/2"},{"type":"function","title":"mnesia.all_keys/1","doc":"Return all keys in a table.\n\nReturns a list of all keys in the table named `Tab`. The semantics of this\nfunction is context-sensitive. For more information, see `mnesia:activity/4`. In\ntransaction-context, it acquires a read lock on the entire table.","ref":"mnesia.html#all_keys/1"},{"type":"function","title":"mnesia.async_dirty/1","doc":"","ref":"mnesia.html#async_dirty/1"},{"type":"function","title":"mnesia.async_dirty/2","doc":"Call the `Fun` in a context that is not protected by a transaction.\n\nThe Mnesia function calls performed in the `Fun` are mapped to the\ncorresponding dirty functions. This still involves logging,\nreplication, and subscriptions, but there is no locking, local\ntransaction storage, or commit protocols involved.  Checkpoint\nretainers and indexes are updated, but they are updated dirty. As for\nnormal `mnesia:dirty_*` operations, the operations are performed\nsemi-asynchronously. For details, see `mnesia:activity/4` and the\nUser's Guide.\n\nThe Mnesia tables can be manipulated without using transactions. This has some\nserious disadvantages, but is considerably faster, as the transaction manager is\nnot involved and no locks are set. A dirty operation does, however, guarantee a\ncertain level of consistency, and the dirty operations cannot return garbled\nrecords. All dirty operations provide location transparency to the programmer,\nand a program does not have to be aware of the whereabouts of a certain table to\nfunction.\n\nNotice that it is more than ten times more efficient to read records dirty than\nwithin a transaction.\n\nDepending on the application, it can be a good idea to use the dirty functions\nfor certain operations. Almost all Mnesia functions that can be called within\ntransactions have a dirty equivalent, which is much more efficient.\n\nHowever, notice that there is a risk that the database can be left in an\ninconsistent state if dirty operations are used to update it. Dirty operations\nare only to be used for performance reasons when it is absolutely necessary.\n\nNotice that calling (nesting) `mnesia:[a]sync_dirty` inside a\ntransaction-context inherits the transaction semantics.","ref":"mnesia.html#async_dirty/2"},{"type":"function","title":"mnesia.backup/1","doc":"","ref":"mnesia.html#backup/1"},{"type":"function","title":"mnesia.backup/2","doc":"Back up all tables in the database.\n\nActivates a new checkpoint covering all Mnesia tables, including the schema,\nwith maximum degree of redundancy, and performs a backup using\n`backup_checkpoint/2/3`. The default value of the backup callback module\n`BackupMod` is obtained by `mnesia:system_info(backup_module)`.","ref":"mnesia.html#backup/2"},{"type":"function","title":"mnesia.backup_checkpoint/2","doc":"","ref":"mnesia.html#backup_checkpoint/2"},{"type":"function","title":"mnesia.backup_checkpoint/3","doc":"Back up all tables in a checkpoint.\n\nThe tables are backed up to external media using backup module `BackupMod`.\nTables with the local contents property are backed up as they exist on the\ncurrent node. `BackupMod` is the default backup callback module obtained by\n`mnesia:system_info(backup_module)`. For information about the exact callback\ninterface (the `mnesia_backup behavior`), see the User's Guide.","ref":"mnesia.html#backup_checkpoint/3"},{"type":"function","title":"mnesia.change_config/2","doc":"Change a configuration setting.\n\n`Config` is to be an atom of the following configuration parameters:\n\n- **`extra_db_nodes`** - `Value` is a list of nodes that Mnesia is to try to\n  connect to. `ReturnValue` is those nodes in `Value` that Mnesia is connected\n  to.\n\n  Notice that this function must only be used to connect to newly started RAM\n  nodes (N.D.R.S.N.) with an empty schema. If, for example, this function is\n  used after the network has been partitioned, it can lead to inconsistent\n  tables.\n\n  Notice that Mnesia can be connected to other nodes than those returned in\n  `ReturnValue`.\n\n- **`dc_dump_limit`** - `Value` is a number. See the description in\n  [Section Configuration Parameters](`m:mnesia#configuration_parameters`).\n  `ReturnValue` is the new value. Notice that this configuration parameter is\n  not persistent. It is lost when Mnesia has stopped.","ref":"mnesia.html#change_config/2"},{"type":"function","title":"mnesia.change_table_access_mode/2","doc":"Change table access mode.\n\n`AcccessMode` is by default the atom `read_write` but it can also be set to the\natom `read_only`. If `AccessMode` is set to `read_only`, updates to the table\ncannot be performed. At startup, Mnesia always loads `read_only` tables locally\nregardless of when and if Mnesia is terminated on other nodes.","ref":"mnesia.html#change_table_access_mode/2"},{"type":"function","title":"mnesia.change_table_copy_type/3","doc":"Change the storage type of a table.\n\nFor example:\n\n```erlang\nmnesia:change_table_copy_type(person, node(), disc_copies)\n```\n\nTransforms the `person` table from a RAM table into a disc-based table at\n`Node`.\n\nThis function can also be used to change the storage type of the table named\n`schema`. The schema table can only have `ram_copies` or `disc_copies` as the\nstorage type. If the storage type of the schema is `ram_copies`, no other table\ncan be disc-resident on that node.","ref":"mnesia.html#change_table_copy_type/3"},{"type":"function","title":"mnesia.change_table_frag/2","doc":"Reconfigure table fragment properties.\n\nArgument `FragProp` should have one of the following values:\n\n- **`{activate, FragProps}`** - Activates the fragmentation properties of an\n  existing table. `FragProps` is either to contain `{node_pool, Nodes}` or be\n  empty.\n\n- **`deactivate`** - Deactivates the fragmentation properties of a table. The\n  number of fragments must be `1`. No other table can refer to this table in its\n  foreign key.\n\n- **`{add_frag, NodesOrDist}`** - Adds a fragment to a fragmented table. All\n  records in one of the old fragments are rehashed and about half of them are\n  moved to the new (last) fragment. All other fragmented tables, which refer to\n  this table in their foreign key, automatically get a new fragment. Also, their\n  records are dynamically rehashed in the same manner as for the main table.\n\n  Argument `NodesOrDist` can either be a list of nodes or the result from the\n  function [mnesia:table_info(Tab, frag_dist)](`mnesia:table_info/2`). Argument\n  `NodesOrDist` is assumed to be a sorted list with the best nodes to host new\n  replicas first in the list. The new fragment gets the same number of replicas\n  as the first fragment (see `n_ram_copies`, `n_disc_copies`, and\n  `n_disc_only_copies`). The `NodesOrDist` list must at least contain one\n  element for each replica that needs to be allocated.\n\n- **`del_frag`** - Deletes a fragment from a fragmented table. All records in\n  the last fragment are moved to one of the other fragments. All other\n  fragmented tables, which refer to this table in their foreign key,\n  automatically lose their last fragment. Also, their records are dynamically\n  rehashed in the same manner as for the main table.\n\n- **`{add_node, Node}`** - Adds a node to `node_pool`. The new node pool affects\n  the list returned from the function\n  [mnesia:table_info(Tab, frag_dist)](`mnesia:table_info/2`).\n\n- **`{del_node, Node}`** - Deletes a node from `node_pool`. The new node pool\n  affects the list returned from the function\n  [mnesia:table_info(Tab, frag_dist)](`mnesia:table_info/2`).","ref":"mnesia.html#change_table_frag/2"},{"type":"function","title":"mnesia.change_table_load_order/2","doc":"Change table load order.\n\nThe `LoadOrder` priority is by default `0` (zero) but can be set to any integer.\nThe tables with the highest `LoadOrder` priority are loaded first at startup.","ref":"mnesia.html#change_table_load_order/2"},{"type":"function","title":"mnesia.change_table_majority/2","doc":"Change table majority.\n\n`Majority` must be a boolean. Default is `false`. When `true`, a majority of the\ntable replicas must be available for an update to succeed. When used on\nfragmented tables, `Tab` must be the base table name. Directly changing the\nmajority setting on individual fragments is not allowed.","ref":"mnesia.html#change_table_majority/2"},{"type":"function","title":"mnesia.clear_table/1","doc":"Delete all entries in the table `Tab`.","ref":"mnesia.html#clear_table/1"},{"type":"function","title":"mnesia.create_schema/1","doc":"Create a new schema on the specified nodes.\n\nCreates a new database on disc. Various files are created in the local Mnesia\ndirectory of each node. Notice that the directory must be unique for each node.\nTwo nodes must never share the same directory. If possible, use a local disc\ndevice to improve performance.\n\n`mnesia:create_schema/1` fails if any of the Erlang nodes given as `DiscNodes`\nare not alive, if Mnesia is running on any of the nodes, or if any of the nodes\nalready have a schema. Use `mnesia:delete_schema/1` to get rid of old faulty\nschemas.\n\nNotice that only nodes with disc are to be included in `DiscNodes`. Disc-less\nnodes, that is, nodes where all tables including the schema only resides in RAM,\nmust not be included.","ref":"mnesia.html#create_schema/1"},{"type":"function","title":"mnesia.create_table/2","doc":"Create a table.\n\nCreates a Mnesia table called `Name` according to argument `TabDef`. This list\nmust be a list of `{Item, Value}` tuples, where the following values are\nallowed:\n\n- `{access_mode, Atom}`. The access mode is by default the atom `read_write` but\n  it can also be set to the atom `read_only`. If `AccessMode` is set to\n  `read_only`, updates to the table cannot be performed.\n\n  At startup, Mnesia always loads `read_only` table locally regardless of when\n  and if Mnesia is terminated on other nodes. This argument returns the access\n  mode of the table. The access mode can be `read_only` or `read_write`.\n\n- `{attributes, AtomList}` is a list of the attribute names for the records that\n  are supposed to populate the table. Default is `[key, val]`. The table must at\n  least have one extra attribute in addition to the key.\n\n  When accessing single attributes in a record, it is not necessary, or even\n  recommended, to hard code any attribute names as atoms. Use construct\n  `record_info(fields, RecordName)` instead. It can be used for records of type\n  `RecordName`.\n\n- `{disc_copies, Nodelist}`, where `Nodelist` is a list of the nodes where this\n  table is supposed to have disc copies. If a table replica is of type\n  `disc_copies`, all write operations on this particular replica of the table\n  are written to disc and to the RAM copy of the table.\n\n  It is possible to have a replicated table of type `disc_copies` on one node\n  and another type on another node. Default is `[]`.\n\n- `{disc_only_copies, Nodelist}`, where `Nodelist` is a list of the nodes where\n  this table is supposed to have `disc_only_copies`. A disc only table replica\n  is kept on disc only and unlike the other replica types, the contents of the\n  replica do not reside in RAM. These replicas are considerably slower than\n  replicas held in RAM.\n- `{index, Intlist}`, where `Intlist` is a list of attribute names (atoms) or\n  record fields for which Mnesia is to build and maintain an extra index table.\n  The `qlc` query compiler _may_ be able to optimize queries if there are\n  indexes available.\n- `{load_order, Integer}`. The load order priority is by default `0` (zero) but\n  can be set to any integer. The tables with the highest load order priority are\n  loaded first at startup.\n- `{majority, Flag}`, where `Flag` must be a boolean. If `true`, any (non-dirty)\n  update to the table is aborted, unless a majority of the table replicas are\n  available for the commit. When used on a fragmented table, all fragments are\n  given the same majority setting.\n- `{ram_copies, Nodelist}`, where `Nodelist` is a list of the nodes where this\n  table is supposed to have RAM copies. A table replica of type `ram_copies` is\n  not written to disc on a per transaction basis. `ram_copies` replicas can be\n  dumped to disc with the function `mnesia:dump_tables(Tabs)`. Default value for\n  this attribute is `[node()]`.\n- `{record_name, Name}`, where `Name` must be an atom. All records stored in the\n  table must have this name as the first element. It defaults to the same name\n  as the table name.\n- `{snmp, SnmpStruct}`. For a description of `SnmpStruct`, see\n  `mnesia:snmp_open_table/2`. If this attribute is present in `ArgList` to\n  `mnesia:create_table/2`, the table is immediately accessible by SNMP.\n  Therefore applications that use SNMP to manipulate and control the system can\n  be designed easily, since Mnesia provides a direct mapping between the logical\n  tables that make up an SNMP control application and the physical data that\n  makes up a Mnesia table.\n- `{storage_properties, [{Backend, Properties}]` forwards more properties to the\n  back end storage. `Backend` can currently be `ets` or `dets`. `Properties` is\n  a list of options sent to the back end storage during table creation.\n  `Properties` cannot contain properties already used by Mnesia, such as `type`\n  or `named_table`.\n\n  For example:\n\n  ```erlang\n  mnesia:create_table(table, [{ram_copies, [node()]}, {disc_only_copies, nodes()},\n         {storage_properties,\n          [{ets, [compressed]}, {dets, [{auto_save, 5000}]} ]}])\n  ```\n\n- `{type, Type}`, where `Type` must be either of the atoms `set`, `ordered_set`,\n  or `bag`. Default is `set`. In a `set`, all records have unique keys. In a\n  `bag`, several records can have the same key, but the record content is\n  unique. If a non-unique record is stored, the old conflicting records are\n  overwritten.\n\n  Notice that currently `ordered_set` is not supported for `disc_only_copies`.\n\n- `{local_content, Bool}`, where `Bool` is `true` or `false`. Default is\n  `false`.\n\nFor example, the following call creates the `person` table (defined earlier) and\nreplicates it on two nodes:\n\n```erlang\nmnesia:create_table(person,\n    [{ram_copies, [N1, N2]},\n     {attributes, record_info(fields, person)}]).\n```\n\nIf it is required that Mnesia must build and maintain an extra index table on\nattribute `address` of all the `person` records that are inserted in the table,\nthe following code would be issued:\n\n```erlang\nmnesia:create_table(person,\n    [{ram_copies, [N1, N2]},\n     {index, [address]},\n     {attributes, record_info(fields, person)}]).\n```\n\nThe specification of `index` and `attributes` can be hard-coded as\n`{index, [2]}` and `{attributes, [name, age, address, salary, children]}`,\nrespectively.\n\n`mnesia:create_table/2` writes records into the table `schema`. This function,\nand all other schema manipulation functions, are implemented with the normal\ntransaction management system. This guarantees that schema updates are performed\non all nodes in an atomic manner.","ref":"mnesia.html#create_table/2"},{"type":"function","title":"mnesia.deactivate_checkpoint/1","doc":"Deactivate a checkpoint.\n\nThe checkpoint is automatically deactivated when some of the tables involved\nhave no retainer attached to them. This can occur when nodes go down or when a\nreplica is deleted. Checkpoints are also deactivated with this function. `Name`\nis the name of an active checkpoint.","ref":"mnesia.html#deactivate_checkpoint/1"},{"type":"function","title":"mnesia.del_table_copy/2","doc":"Delete the replica of table.\n\nDeletes the replica of table `Tab` at node `Node`. When the last replica is\ndeleted with this function, the table disappears entirely.\n\nThis function can also be used to delete a replica of the table named `schema`.\nThe Mnesia node is then removed. Notice that Mnesia must be stopped on the node\nfirst.","ref":"mnesia.html#del_table_copy/2"},{"type":"function","title":"mnesia.del_table_index/2","doc":"Delete table index.\n\nDeletes the index on attribute with name `AttrName` in a table.","ref":"mnesia.html#del_table_index/2"},{"type":"function","title":"mnesia.delete/1","doc":"","ref":"mnesia.html#delete/1"},{"type":"function","title":"mnesia.delete/3","doc":"Delete all records in table `Tab` with the key `Key`.\n\nThe semantics of this function is context-sensitive. For details, see\n`mnesia:activity/4`. In transaction-context, it acquires a lock of type\n`LockKind` in the record. Currently, the lock types `write` and `sticky_write`\nare supported.","ref":"mnesia.html#delete/3"},{"type":"function","title":"mnesia.delete_object/1","doc":"","ref":"mnesia.html#delete_object/1"},{"type":"function","title":"mnesia.delete_object/3","doc":"Delete a record.\n\nIf a table is of type `bag`, it can sometimes be needed to delete only some of\nthe records with a certain key. This can be done with the function\n[`delete_object/3`](`delete_object/3`). A complete record must be supplied to\nthis function.\n\nThe semantics of this function is context-sensitive. For details, see\n`mnesia:activity/4`. In transaction-context, it acquires a lock of type\n`LockKind` on the record. Currently, the lock types `write` and `sticky_write`\nare supported.","ref":"mnesia.html#delete_object/3"},{"type":"function","title":"mnesia.delete_schema/1","doc":"Delete the schema on the given nodes.\n\nDeletes a database created with `mnesia:create_schema/1`.\n`mnesia:delete_schema/1` fails if any of the Erlang nodes given as `DiscNodes`\nare not alive, or if Mnesia is running on any of the nodes.\n\nAfter the database is deleted, it can still be possible to start Mnesia as a\ndisc-less node. This depends on how configuration parameter `schema_location` is\nset.\n\n> #### Warning {: .warning }\n>\n> Use this function with extreme caution, as it makes existing persistent data\n> obsolete. Think twice before using it.","ref":"mnesia.html#delete_schema/1"},{"type":"function","title":"mnesia.delete_table/1","doc":"Permanently delete all replicas of table `Tab`.","ref":"mnesia.html#delete_table/1"},{"type":"function","title":"mnesia.dirty_all_keys/1","doc":"Dirty equivalent to `mnesia:all_keys/1`.","ref":"mnesia.html#dirty_all_keys/1"},{"type":"function","title":"mnesia.dirty_delete/1","doc":"","ref":"mnesia.html#dirty_delete/1"},{"type":"function","title":"mnesia.dirty_delete/2","doc":"Dirty equivalent to `mnesia:delete/3`.","ref":"mnesia.html#dirty_delete/2"},{"type":"function","title":"mnesia.dirty_delete_object/1","doc":"","ref":"mnesia.html#dirty_delete_object/1"},{"type":"function","title":"mnesia.dirty_delete_object/2","doc":"Dirty equivalent to `mnesia:delete_object/3`.","ref":"mnesia.html#dirty_delete_object/2"},{"type":"function","title":"mnesia.dirty_first/1","doc":"Return the key for the first record in a table.\n\nRecords in `set` or `bag` tables are not ordered. However, there is an ordering\nof the records that is unknown to the user. Therefore, a table can be traversed\nby this function with the function `mnesia:dirty_next/2`.\n\nIf there are no records in the table, this function returns the atom\n`'$end_of_table'`. It is therefore highly undesirable, but not disallowed, to\nuse this atom as the key for any user records.","ref":"mnesia.html#dirty_first/1"},{"type":"function","title":"mnesia.dirty_index_match_object/2","doc":"","ref":"mnesia.html#dirty_index_match_object/2"},{"type":"function","title":"mnesia.dirty_index_match_object/3","doc":"Dirty equivalent to `mnesia:index_match_object/4`.","ref":"mnesia.html#dirty_index_match_object/3"},{"type":"function","title":"mnesia.dirty_index_read/3","doc":"Dirty equivalent to `mnesia:index_read/3`.","ref":"mnesia.html#dirty_index_read/3"},{"type":"function","title":"mnesia.dirty_last/1","doc":"Return the key for the last record in a table.\n\nWorks exactly like `mnesia:dirty_first/1` but returns the last object in Erlang\nterm order for the `ordered_set` table type. For all other table types,\n`mnesia:dirty_first/1` and `mnesia:dirty_last/1` are synonyms.","ref":"mnesia.html#dirty_last/1"},{"type":"function","title":"mnesia.dirty_match_object/1","doc":"","ref":"mnesia.html#dirty_match_object/1"},{"type":"function","title":"mnesia.dirty_match_object/2","doc":"Dirty equivalent to `mnesia:match_object/3`.","ref":"mnesia.html#dirty_match_object/2"},{"type":"function","title":"mnesia.dirty_next/2","doc":"Return the next key in a table.\n\nTraverses a table and performs operations on all records in the table. When the\nend of the table is reached, the special key `'$end_of_table'` is returned.\nOtherwise, the function returns a key that can be used to read the actual\nrecord. The behavior is undefined if another Erlang process performs write\noperations on the table while it is being traversed with the function\n`mnesia:dirty_next/2`.","ref":"mnesia.html#dirty_next/2"},{"type":"function","title":"mnesia.dirty_prev/2","doc":"Return the previous key in a table.\n\nWorks exactly like `mnesia:dirty_next/2` but returns the previous object in\nErlang term order for the `ordered_set` table type. For all other table types,\n`mnesia:dirty_next/2` and `mnesia:dirty_prev/2` are synonyms.","ref":"mnesia.html#dirty_prev/2"},{"type":"function","title":"mnesia.dirty_read/1","doc":"","ref":"mnesia.html#dirty_read/1"},{"type":"function","title":"mnesia.dirty_read/2","doc":"Dirty equivalent to `mnesia:read/3`.","ref":"mnesia.html#dirty_read/2"},{"type":"function","title":"mnesia.dirty_select/2","doc":"Dirty equivalent to `mnesia:select/2`.","ref":"mnesia.html#dirty_select/2"},{"type":"function","title":"mnesia.dirty_update_counter/2","doc":"","ref":"mnesia.html#dirty_update_counter/2"},{"type":"function","title":"mnesia.dirty_update_counter/3","doc":"Dirty update of a counter record.\n\nMnesia has no special counter records. However, records of the form\n`{Tab, Key, Integer}` can be used as (possibly disc-resident) counters when\n`Tab` is a `set`. This function updates a counter with a positive or negative\nnumber. However, counters can never become less than zero. There are two\nsignificant differences between this function and the action of first reading\nthe record, performing the arithmetic, and then writing the record:\n\n- It is much more efficient.\n- `mnesia:dirty_update_counter/3` is performed as an atomic operation although\n  it is not protected by a transaction.\n\nIf two processes perform `mnesia:dirty_update_counter/3` simultaneously, both\nupdates take effect without the risk of losing one of the updates. The new value\n`NewVal` of the counter is returned.\n\nIf `Key` does not exist, a new record is created with value `Incr` if it is\nlarger than 0, otherwise it is set to 0.","ref":"mnesia.html#dirty_update_counter/3"},{"type":"function","title":"mnesia.dirty_write/1","doc":"","ref":"mnesia.html#dirty_write/1"},{"type":"function","title":"mnesia.dirty_write/2","doc":"Dirty equivalent to `mnesia:write/3`.","ref":"mnesia.html#dirty_write/2"},{"type":"function","title":"mnesia.dump_log/0","doc":"Perform a user-initiated dump of the local log file.\n\nThis is usually not necessary, as Mnesia by default manages this automatically. See configuration\nparameters [dump_log_time_threshold](`m:mnesia#dump_log_time_threshold`) and\n[dump_log_write_threshold](`m:mnesia#dump_log_write_threshold`).","ref":"mnesia.html#dump_log/0"},{"type":"function","title":"mnesia.dump_tables/1","doc":"Dump ram_copies tables to disc.\n\nDumps a set of `ram_copies` tables to disc. The next time the system is started,\nthese tables are initiated with the data found in the files that are the result\nof this dump. None of the tables can have disc-resident replicas.","ref":"mnesia.html#dump_tables/1"},{"type":"function","title":"mnesia.dump_to_textfile/1","doc":"Dump local tables into a text file.\n\nDumps all local tables of a Mnesia system into a text file, which can be edited\n(by a normal text editor) and then be reloaded with `mnesia:load_textfile/1`.\nOnly use this function for educational purposes. Use other functions to deal\nwith real backups.","ref":"mnesia.html#dump_to_textfile/1"},{"type":"function","title":"mnesia.error_description/1","doc":"Return a string describing a particular Mnesia error.\n\nAll Mnesia transactions, including all the schema update functions, either\nreturn value `{atomic, Val}` or the tuple `{aborted, Reason}`. `Reason` can be\neither of the atoms in the following list. The function\n[`error_description/1`](`error_description/1`) returns a descriptive string that\ndescribes the error.\n\n- `nested_transaction`. Nested transactions are not allowed in this context.\n- `badarg`. Bad or invalid argument, possibly bad type.\n- `no_transaction`. Operation not allowed outside transactions.\n- `combine_error`. Table options illegally combined.\n- `bad_index`. Index already exists, or was out of bounds.\n- `already_exists`. Schema option to be activated is already on.\n- `index_exists`. Some operations cannot be performed on tables with an index.\n- `no_exists`. Tried to perform operation on non-existing (not-alive) item.\n- `system_limit`. A system limit was exhausted.\n- `mnesia_down`. A transaction involves records on a remote node, which became\n  unavailable before the transaction was completed. Records are no longer\n  available elsewhere in the network.\n- `not_a_db_node`. A node was mentioned that does not exist in the schema.\n- `bad_type`. Bad type specified in argument.\n- `node_not_running`. Node is not running.\n- `truncated_binary_file`. Truncated binary in file.\n- `active`. Some delete operations require that all active records are removed.\n- `illegal`. Operation not supported on this record.\n\n`Error` can be `Reason`, `{error, Reason}`, or `{aborted, Reason}`. `Reason` can\nbe an atom or a tuple with `Reason` as an atom in the first field.\n\nThe following examples illustrate a function that returns an error, and the\nmethod to retrieve more detailed error information:\n\n- The function [mnesia:create_table(bar, \\[\\{attributes,\n  3.14\\}])](`create_table/2`) returns the tuple `{aborted,Reason}`, where\n  `Reason` is the tuple `{bad_type,bar,3.14000}`.\n- The function [mnesia:error_description(Reason)](`error_description/1`) returns\n  the term `{\"Bad type on some provided arguments\",bar,3.14000}`, which is an\n  error description suitable for display.","ref":"mnesia.html#error_description/1"},{"type":"function","title":"mnesia.ets/1","doc":"","ref":"mnesia.html#ets/1"},{"type":"function","title":"mnesia.ets/2","doc":"Call `Fun` in a raw context that is not protected by a transaction.\n\nThe Mnesia function call is performed in the `Fun` and performed directly on the\nlocal ETS tables on the assumption that the local storage type is `ram_copies`\nand the tables are not replicated to other nodes. Subscriptions are not\ntriggered and checkpoints are not updated, but it is extremely fast. This\nfunction can also be applied to `disc_copies` tables if all operations are read\nonly. For details, see `mnesia:activity/4` and the User's Guide.\n\nNotice that calling (nesting) a `mnesia:ets` inside a transaction-context\ninherits the transaction semantics.","ref":"mnesia.html#ets/2"},{"type":"function","title":"mnesia.first/1","doc":"Return the key for the first record in a table.\n\nRecords in `set` or `bag` tables are not ordered. However, there is an ordering\nof the records that is unknown to the user. A table can therefore be traversed\nby this function with the function `mnesia:next/2`.\n\nIf there are no records in the table, this function returns the atom\n`'$end_of_table'`. It is therefore highly undesirable, but not disallowed, to\nuse this atom as the key for any user records.","ref":"mnesia.html#first/1"},{"type":"function","title":"mnesia.foldl/3","doc":"","ref":"mnesia.html#foldl/3"},{"type":"function","title":"mnesia.foldl/4","doc":"Call `Fun` for each record in `Table`.\n\nIterates over the table `Table` and calls `Function(Record, NewAcc)` for each\n`Record` in the table. The term returned from `Function` is used as the second\nargument in the next call to `Function`.\n\n`foldl` returns the same term as the last call to `Function` returned.","ref":"mnesia.html#foldl/4"},{"type":"function","title":"mnesia.foldr/3","doc":"","ref":"mnesia.html#foldr/3"},{"type":"function","title":"mnesia.foldr/4","doc":"Call `Fun` for each record in `Table`.\n\nWorks exactly like [`foldl/3`](`foldl/3`) but iterates the table in the opposite\norder for the `ordered_set` table type. For all other table types,\n[`foldr/3`](`foldr/3`) and [`foldl/3`](`foldl/3`) are synonyms.","ref":"mnesia.html#foldr/4"},{"type":"function","title":"mnesia.force_load_table/1","doc":"Force a table to be loaded into the system.\n\nThe Mnesia algorithm for table load can lead to a situation where a table cannot\nbe loaded. This situation occurs when a node is started and Mnesia concludes, or\nsuspects, that another copy of the table was active after this local copy became\ninactive because of a system crash.\n\nIf this situation is not acceptable, this function can be used to override the\nstrategy of the Mnesia table load algorithm. This can lead to a situation where\nsome transaction effects are lost with an inconsistent database as result, but\nfor some applications high availability is more important than consistent data.","ref":"mnesia.html#force_load_table/1"},{"type":"function","title":"mnesia.index_match_object/2","doc":"Match records and uses index information.\n\nStarts `mnesia:index_match_object(Tab, Pattern, Pos, read)`, where `Tab` is\n[`element(1, Pattern)`](`element/2`).","ref":"mnesia.html#index_match_object/2"},{"type":"function","title":"mnesia.index_match_object/4","doc":"Match records and uses index information.\n\nIn a manner similar to the function `mnesia:index_read/3`, any index information\ncan be used when trying to match records. This function takes a pattern that\nobeys the same rules as the function `mnesia:match_object/3`, except that this\nfunction requires the following conditions:\n\n- The table `Tab` must have an index on position `Pos`.\n- The element in position `Pos` in `Pattern` must be bound. `Pos` is an integer\n  (`#record.Field`) or an attribute name.\n\nThe two index search functions described here are automatically started when\nsearching tables with `qlc` list comprehensions and also when using the\nlow-level `mnesia:[dirty_]match_object` functions.\n\nThe semantics of this function is context-sensitive. For details, see\n`mnesia:activity/4`. In transaction-context, it acquires a lock of type\n`LockKind` on the entire table or on a single record. Currently, the lock type\n`read` is supported.","ref":"mnesia.html#index_match_object/4"},{"type":"function","title":"mnesia.index_read/3","doc":"Read records through the index table.\n\nAssume that there is an index on position `Pos` for a certain record type. This\nfunction can be used to read the records without knowing the actual key for the\nrecord. For example, with an index in position 1 of table `person`, the call\n`mnesia:index_read(person, 36, #person.age)` returns a list of all persons with\nage 36. `Pos` can also be an attribute name (atom), but if the notation\n`mnesia:index_read(person, 36, age)` is used, the field position is searched for\nin runtime, for each call.\n\nThe semantics of this function is context-sensitive. For details, see\n`mnesia:activity/4`. In transaction-context, it acquires a read lock on the\nentire table.","ref":"mnesia.html#index_read/3"},{"type":"function","title":"mnesia.info/0","doc":"Print system information on the terminal.\n\nPrints system information on the terminal. This function can be used even if\nMnesia is not started. However, more information is displayed if Mnesia is\nstarted.","ref":"mnesia.html#info/0"},{"type":"function","title":"mnesia.install_fallback/1","doc":"","ref":"mnesia.html#install_fallback/1"},{"type":"function","title":"mnesia.install_fallback/2","doc":"Install a backup as fallback.\n\nThe fallback is used to restore the database at\nthe next startup. Installation of fallbacks requires Erlang to be operational on\nall the involved nodes, but it does not matter if Mnesia is running or not. The\ninstallation of the fallback fails if the local node is not one of the\ndisc-resident nodes in the backup.\n\n`Args` is a list of the following tuples:\n\n- `{module, BackupMod}`. All accesses of the backup media are performed through\n  a callback module named `BackupMod`. Argument `Opaque` is forwarded to the\n  callback module, which can interpret it as it wishes. The default callback\n  module is called `mnesia_backup` and it interprets argument `Opaque` as a\n  local filename. The default for this module is also configurable through\n  configuration parameter `-mnesia mnesia_backup`.\n- `{scope, Scope}`. The `Scope` of a fallback is either `global` for the entire\n  database or `local` for one node. By default, the installation of a fallback\n  is a global operation, which either is performed on all nodes with a\n  disc-resident schema or none. Which nodes that are disc-resident is determined\n  from the schema information in the backup.\n\n  If `Scope` of the operation is `local`, the fallback is only installed on the\n  local node.\n\n- `{mnesia_dir, AlternateDir}`. This argument is only valid if the scope of the\n  installation is `local`. Normally the installation of a fallback is targeted\n  to the Mnesia directory, as configured with configuration parameter\n  `-mnesia dir`. But by explicitly supplying an `AlternateDir`, the fallback is\n  installed there regardless of the Mnesia directory configuration parameter\n  setting. After installation of a fallback on an alternative Mnesia directory,\n  that directory is fully prepared for use as an active Mnesia directory.\n\n  This is a dangerous feature that must be used with care. By unintentional\n  mixing of directories, you can easily end up with an inconsistent database, if\n  the same backup is installed on more than one directory.","ref":"mnesia.html#install_fallback/2"},{"type":"function","title":"mnesia.is_transaction/0","doc":"Return true if inside a transaction context.\n\nWhen this function is executed inside a transaction-context, it returns `true`,\notherwise `false`.","ref":"mnesia.html#is_transaction/0"},{"type":"function","title":"mnesia.last/1","doc":"Return the key for the last record in a table.\n\nWorks exactly like `mnesia:first/1`, but returns the last object in Erlang term\norder for the `ordered_set` table type. For all other table types,\n`mnesia:first/1` and `mnesia:last/1` are synonyms.","ref":"mnesia.html#last/1"},{"type":"function","title":"mnesia.load_textfile/1","doc":"Load tables from a text file.\n\nLoads a series of definitions and data found in the text file (generated with\n`mnesia:dump_to_textfile/1`) into Mnesia. This function also starts Mnesia and\npossibly creates a new schema. This function is intended for educational\npurposes only. It is recommended to use other functions to deal with real\nbackups.","ref":"mnesia.html#load_textfile/1"},{"type":"function","title":"mnesia.lock/2","doc":"Explicitly grab lock.\n\nWrite locks are normally acquired on all nodes where a replica of the table\nresides (and is active). Read locks are acquired on one node (the local node if\na local replica exists). Most of the context-sensitive access functions acquire\nan implicit lock if they are started in a transaction-context. The granularity\nof a lock can either be a single record or an entire table.\n\nThe normal use is to call the function without checking the return value, as it\nexits if it fails and the transaction is restarted by the transaction manager.\nIt returns all the locked nodes if a write lock is acquired and `ok` if it was a\nread lock.\n\nThe function `mnesia:lock/2` is intended to support explicit locking on tables,\nbut is also intended for situations when locks need to be acquired regardless of\nhow tables are replicated. Currently, two kinds of `LockKind` are supported:\n\n- **`write`** - Write locks are exclusive. This means that if one transaction\n  manages to acquire a write lock on an item, no other transaction can acquire\n  any kind of lock on the same item.\n\n- **`read`** - Read locks can be shared. This means that if one transaction\n  manages to acquire a read lock on an item, other transactions can also acquire\n  a read lock on the same item. However, if someone has a read lock, no one can\n  acquire a write lock at the same item. If someone has a write lock, no one can\n  acquire either a read lock or a write lock at the same item.\n\nConflicting lock requests are automatically queued if there is no risk of a\ndeadlock. Otherwise the transaction must be terminated and executed again.\nMnesia does this automatically as long as the upper limit of the maximum\n`retries` is not reached. For details, see `mnesia:transaction/3`.\n\nFor the sake of completeness, sticky write locks are also described here even if\na sticky write lock is not supported by this function:\n\n- **`sticky_write`** - Sticky write locks are a mechanism that can be used to\n  optimize write lock acquisition. If your application uses replicated tables\n  mainly for fault tolerance (as opposed to read access optimization purpose),\n  sticky locks can be the best option available.\n\n  When a sticky write lock is acquired, all nodes are informed which node is\n  locked. Then, sticky lock requests from the same node are performed as a local\n  operation without any communication with other nodes. The sticky lock lingers\n  on the node even after the transaction ends. For details, see the User's\n  Guide.\n\nCurrently, this function supports two kinds of `LockItem`:\n\n- **`{table, Tab}`** - This acquires a lock of type `LockKind` on the entire\n  table `Tab`.\n\n- **`{global, GlobalKey, Nodes}`** - This acquires a lock of type `LockKind` on\n  the global resource `GlobalKey`. The lock is acquired on all active nodes in\n  the `Nodes` list.\n\nLocks are released when the outermost transaction ends.\n\nThe semantics of this function is context-sensitive. For details, see\n`mnesia:activity/4`. In transaction-context, it acquires locks, otherwise it\nignores the request.","ref":"mnesia.html#lock/2"},{"type":"function","title":"mnesia.match_object/1","doc":"","ref":"mnesia.html#match_object/1"},{"type":"function","title":"mnesia.match_object/3","doc":"Match `Pattern` for records.\n\nTakes a pattern with \"don't care\" variables denoted as a `'_'` parameter. This\nfunction returns a list of records that matched the pattern. Since the second\nelement of a record in a table is considered to be the key for the record, the\nperformance of this function depends on whether this key is bound or not.\n\nFor example, the call\n`mnesia:match_object(person, {person, '_', 36, '_', '_'}, read)` returns a list\nof all person records with an `age` field of 36.\n\nThe function `mnesia:match_object/3` automatically uses indexes if these exist.\nHowever, no heuristics are performed to select the best index.\n\nThe semantics of this function is context-sensitive. For details, see\n`mnesia:activity/4`. In transaction-context, it acquires a lock of type\n`LockKind` on the entire table or a single record. Currently, the lock type\n`read` is supported.","ref":"mnesia.html#match_object/3"},{"type":"function","title":"mnesia.move_table_copy/3","doc":"Move a table copy.\n\nMoves the copy of table `Tab` from node `From` to node `To`.\n\nThe storage type is preserved. For example, a RAM table moved from one node\nremains a RAM on the new node. Other transactions can still read and write in\nthe table while it is being moved.\n\nThis function cannot be used on `local_content` tables.","ref":"mnesia.html#move_table_copy/3"},{"type":"function","title":"mnesia.next/2","doc":"Return the next key in a table.\n\nTraverses a table and performs operations on all records in the table. When the\nend of the table is reached, the special key `'$end_of_table'` is returned.\nOtherwise the function returns a key that can be used to read the actual record.","ref":"mnesia.html#next/2"},{"type":"function","title":"mnesia.prev/2","doc":"Return the previous key in a table.\n\nWorks exactly like `mnesia:next/2`, but returns the previous object in Erlang\nterm order for the `ordered_set` table type. For all other table types,\n`mnesia:next/2` and `mnesia:prev/2` are synonyms.","ref":"mnesia.html#prev/2"},{"type":"function","title":"mnesia.read/1","doc":"","ref":"mnesia.html#read/1"},{"type":"function","title":"mnesia.read/2","doc":"","ref":"mnesia.html#read/2"},{"type":"function","title":"mnesia.read/3","doc":"Read records(s) with a given key.\n\nReads all records from table `Tab` with key `Key`. This function has the same\nsemantics regardless of the location of `Tab`. If the table is of type `bag`,\nthe function `mnesia:read(Tab, Key)` can return an arbitrarily long list. If the\ntable is of type `set`, the list is either of length 1, or `[]`.\n\nThe semantics of this function is context-sensitive. For details, see\n`mnesia:activity/4`. In transaction-context, it acquires a lock of type\n`LockKind`. Currently, the lock types `read`, `write`, and `sticky_write` are\nsupported.\n\nIf the user wants to update the record, it is more efficient to use\n`write/sticky_write` as the `LockKind`. If majority checking is active on the\ntable, it is checked as soon as a write lock is attempted. This can be used to\nend quickly if the majority condition is not met.","ref":"mnesia.html#read/3"},{"type":"function","title":"mnesia.read_lock_table/1","doc":"","ref":"mnesia.html#read_lock_table/1"},{"type":"function","title":"mnesia.report_event/1","doc":"Report a user event to the Mnesia event handler.\n\nWhen tracing a system of Mnesia applications it is useful to be able to\ninterleave Mnesia own events with application-related events that give\ninformation about the application context.\n\nWhenever the application begins a new and demanding Mnesia task, or if it enters\na new interesting phase in its execution, it can be a good idea to use\n`mnesia:report_event/1`. `Event` can be any term and generates a\n`{mnesia_user, Event}` event for any processes that subscribe to Mnesia system\nevents.","ref":"mnesia.html#report_event/1"},{"type":"function","title":"mnesia.restore/2","doc":"Restore a backup.\n\nWith this function, tables can be restored online from a backup without\nrestarting Mnesia. `Opaque` is forwarded to the backup module. `Args` is a list\nof the following tuples:\n\n- `{module,BackupMod}`. The backup module `BackupMod` is used to access the\n  backup media. If omitted, the default backup module is used.\n- `{skip_tables, TabList}`, where `TabList` is a list of tables that is not to\n  be read from the backup.\n- `{clear_tables, TabList}`, where `TabList` is a list of tables that is to be\n  cleared before the records from the backup are inserted. That is, all records\n  in the tables are deleted before the tables are restored. Schema information\n  about the tables is not cleared or read from the backup.\n- `{keep_tables, TabList}`, where `TabList` is a list of tables that is not to\n  be cleared before the records from the backup are inserted. That is, the\n  records in the backup are added to the records in the table. Schema\n  information about the tables is not cleared or read from the backup.\n- `{recreate_tables, TabList}`, where `TabList` is a list of tables that is to\n  be recreated before the records from the backup are inserted. The tables are\n  first deleted and then created with the schema information from the backup.\n  All the nodes in the backup need to be operational.\n- `{default_op, Operation}`, where `Operation` is either of the operations\n  `skip_tables`, `clear_tables`, `keep_tables`, or `recreate_tables`. The\n  default operation specifies which operation that is to be used on tables from\n  the backup that is not specified in any of the mentioned lists. If omitted,\n  operation `clear_tables` is used.\n\nThe affected tables are write-locked during the restoration. However, regardless\nof the lock conflicts caused by this, the applications can continue to do their\nwork while the restoration is being performed. The restoration is performed as\none single transaction.\n\nIf the database is huge, it it not always possible to restore it online. In such\ncases, restore the old database by installing a fallback and then restart.","ref":"mnesia.html#restore/2"},{"type":"function","title":"mnesia.s_delete/1","doc":"Call the function `mnesia:delete(Tab, Key, sticky_write)`","ref":"mnesia.html#s_delete/1"},{"type":"function","title":"mnesia.s_delete_object/1","doc":"","ref":"mnesia.html#s_delete_object/1"},{"type":"function","title":"mnesia.s_write/1","doc":"","ref":"mnesia.html#s_write/1"},{"type":"function","title":"mnesia.schema/0","doc":"Print information about all table definitions on the terminal.","ref":"mnesia.html#schema/0"},{"type":"function","title":"mnesia.schema/1","doc":"Print information about one table definition on the terminal.","ref":"mnesia.html#schema/1"},{"type":"function","title":"mnesia.select/1","doc":"Continue selecting objects.\n\nSelects more objects with the match specification initiated by\n`mnesia:select/4`.\n\nNotice that any modifying operations, that is, `mnesia:write` or\n`mnesia:delete`, that are done between the `mnesia:select/4` and\n`mnesia:select/1` calls are not visible in the result.","ref":"mnesia.html#select/1"},{"type":"function","title":"mnesia.select/2","doc":"","ref":"mnesia.html#select/2"},{"type":"function","title":"mnesia.select/3","doc":"Select the objects in `Tab` against `MatchSpec`.\n\nMatches the objects in table `Tab` using a `match_spec` as described in the\n`ets:select/3`. Optionally a lock `read` or `write` can be given as the third\nargument. Default is `read`. The return value depends on `MatchSpec`.\n\nNotice that for best performance, `select` is to be used before any modifying\noperations are done on that table in the same transaction. That is, do not use\n`write` or `delete` before a `select`.\n\nIn its simplest forms, the `match_spec` look as follows:\n\n- `MatchSpec = [MatchFunction]`\n- `MatchFunction = {MatchHead, [Guard], [Result]}`\n- `MatchHead = tuple() | record()`\n- `Guard = {\"Guardtest name\", ...}`\n- `Result = \"Term construct\"`\n\nFor a complete description of `select`, see the [ERTS](`e:erts:index.html`)\nUser's Guide and the `m:ets` manual page in STDLIB.\n\nFor example, to find the names of all male persons older than 30 in table `Tab`:\n\n```erlang\nMatchHead = #person{name='$1', sex=male, age='$2', _='_'},\nGuard = {'>', '$2', 30},\nResult = '$1',\nmnesia:select(Tab,[{MatchHead, [Guard], [Result]}]),\n```","ref":"mnesia.html#select/3"},{"type":"function","title":"mnesia.select/4","doc":"Select the objects in `Tab` against `MatchSpec`.\n\nMatches the objects in table `Tab` using a `match_spec` as described in the\n[ERTS](`e:erts:index.html`) User's Guide, and returns a chunk of terms and a\ncontinuation. The wanted number of returned terms is specified by argument\n`NObjects`. The lock argument can be `read` or `write`. The continuation is to\nbe used as argument to `mnesia:select/1`, if more or all answers are needed.\n\nNotice that for best performance, `select` is to be used before any modifying\noperations are done on that table in the same transaction. That is, do not use\n`mnesia:write` or `mnesia:delete` before a `mnesia:select`. For efficiency,\n`NObjects` is a recommendation only and the result can contain anything from an\nempty list to all available results.","ref":"mnesia.html#select/4"},{"type":"function","title":"mnesia.set_debug_level/1","doc":"Change the internal debug level of Mnesia.\n\nFor details, see Section Configuration Parameters](`m:mnesia#configuration_parameters`).","ref":"mnesia.html#set_debug_level/1"},{"type":"function","title":"mnesia.set_master_nodes/1","doc":"Set the master nodes for all tables.\n\nFor each table Mnesia determines its replica nodes (`TabNodes`) and starts\n`mnesia:set_master_nodes(Tab, TabMasterNodes)`. where `TabMasterNodes` is the\nintersection of `MasterNodes` and `TabNodes`. For semantics, see\n`mnesia:set_master_nodes/2`.","ref":"mnesia.html#set_master_nodes/1"},{"type":"function","title":"mnesia.set_master_nodes/2","doc":"Set the master nodes for a table.\n\nIf the application detects a communication failure (in a potentially partitioned\nnetwork) that can have caused an inconsistent database, it can use the function\n`mnesia:set_master_nodes(Tab, MasterNodes)` to define from which nodes each\ntable is to be loaded. At startup, the Mnesia normal table load algorithm is\nbypassed and the table is loaded from one of the master nodes defined for the\ntable, regardless of when and if Mnesia terminated on other nodes. `MasterNodes`\ncan only contain nodes where the table has a replica. If the `MasterNodes` list\nis empty, the master node recovery mechanism for the particular table is reset,\nand the normal load mechanism is used at the next restart.\n\nThe master node setting is always local. It can be changed regardless if Mnesia\nis started or not.\n\nThe database can also become inconsistent if configuration parameter\n`max_wait_for_decision` is used or if `mnesia:force_load_table/1` is used.","ref":"mnesia.html#set_master_nodes/2"},{"type":"function","title":"mnesia.snmp_close_table/1","doc":"Remove the possibility for SNMP to manipulate the table.","ref":"mnesia.html#snmp_close_table/1"},{"type":"function","title":"mnesia.snmp_get_mnesia_key/2","doc":"Get the corresponding Mnesia key from an SNMP index.\n\nTransforms an SNMP index to the corresponding Mnesia key. If the SNMP table has\nmultiple keys, the key is a tuple of the key columns.","ref":"mnesia.html#snmp_get_mnesia_key/2"},{"type":"function","title":"mnesia.snmp_get_next_index/2","doc":"Get the index of the next lexicographical row.\n\n`RowIndex` can specify a non-existing row. Specifically, it can be the empty\nlist. Returns the index of the next lexicographical row. If `RowIndex` is the\nempty list, this function returns the index of the first row in the table.","ref":"mnesia.html#snmp_get_next_index/2"},{"type":"function","title":"mnesia.snmp_get_row/2","doc":"Retrieve a row indexed by an SNMP index.\n\nReads a row by its SNMP index. This index is specified as an SNMP Object\nIdentifier, a list of integers.","ref":"mnesia.html#snmp_get_row/2"},{"type":"function","title":"mnesia.snmp_open_table/2","doc":"Organize a Mnesia table as an SNMP table.\n\nA direct one-to-one mapping can be established between Mnesia tables and SNMP\ntables. Many telecommunication applications are controlled and monitored by the\nSNMP protocol. This connection between Mnesia and SNMP makes it simple and\nconvenient to achieve this mapping.\n\nArgument `SnmpStruct` is a list of SNMP information. Currently, the only\ninformation needed is information about the key types in the table. Multiple\nkeys cannot be handled in Mnesia, but many SNMP tables have multiple keys.\nTherefore, the following convention is used: if a table has multiple keys, these\nmust always be stored as a tuple of the keys. Information about the key types is\nspecified as a tuple of atoms describing the types. The only significant type is\n`fix_string`. This means that a string has a fixed size.\n\nFor example, the following causes table `person` to be ordered as an SNMP table:\n\n```text\nmnesia:snmp_open_table(person, [{key, string}])\n```\n\nConsider the following schema for a table of company employees. Each employee is\nidentified by department number and name. The other table column stores the\ntelephone number:\n\n```erlang\nmnesia:create_table(employee,\n    [{snmp, [{key, {integer, string}}]},\n     {attributes, record_info(fields, employees)}]),\n```\n\nThe corresponding SNMP table would have three columns: `department`, `name`, and\n`telno`.\n\nAn option is to have table columns that are not visible through the SNMP\nprotocol. These columns must be the last columns of the table. In the previous\nexample, the SNMP table could have columns `department` and `name` only. The\napplication could then use column `telno` internally, but it would not be\nvisible to the SNMP managers.\n\nIn a table monitored by SNMP, all elements must be integers, strings, or lists\nof integers.\n\nWhen a table is SNMP ordered, modifications are more expensive than usual,\nO(logN). Also, more memory is used.\n\nNotice that only the lexicographical SNMP ordering is implemented in Mnesia, not\nthe actual SNMP monitoring.","ref":"mnesia.html#snmp_open_table/2"},{"type":"function","title":"mnesia.start/0","doc":"Start a local Mnesia system.\n\nMnesia startup is asynchronous. The function call `mnesia:start()` returns the\natom `ok` and then starts to initialize the different tables. Depending on the\nsize of the database, this can take some time, and the application programmer\nmust wait for the tables that the application needs before they can be used.\nThis is achieved by using the function `mnesia:wait_for_tables/2`.\n\nThe startup procedure for a set of Mnesia nodes is a fairly complicated\noperation. A Mnesia system consists of a set of nodes, with Mnesia started\nlocally on all participating nodes. Normally, each node has a directory where\nall the Mnesia files are written. This directory is referred to as the Mnesia\ndirectory. Mnesia can also be started on disc-less nodes. For more information\nabout disc-less nodes, see `mnesia:create_schema/1` and the User's Guide.\n\nThe set of nodes that makes up a Mnesia system is kept in a schema. Mnesia nodes\ncan be added to or removed from the schema. The initial schema is normally\ncreated on disc with the function `mnesia:create_schema/1`. On disc-less nodes,\na tiny default schema is generated each time Mnesia is started. During the\nstartup procedure, Mnesia exchanges schema information between the nodes to\nverify that the table definitions are compatible.\n\nEach schema has a unique cookie, which can be regarded as a unique schema\nidentifier. The cookie must be the same on all nodes where Mnesia is supposed to\nrun. For details, see the User's Guide.\n\nThe schema file and all other files that Mnesia needs are kept in the Mnesia\ndirectory. The command-line option `-mnesia dir Dir` can be used to specify the\nlocation of this directory to the Mnesia system. If no such command-line option\nis found, the name of the directory defaults to `Mnesia.Node`.\n\n`application:start(mnesia)` can also be used.","ref":"mnesia.html#start/0"},{"type":"function","title":"mnesia.stop/0","doc":"Stop Mnesia locally on the current node.\n\n`application:stop(mnesia)` can also be used.","ref":"mnesia.html#stop/0"},{"type":"function","title":"mnesia.subscribe/1","doc":"Subscribe to events of type `EventCategory`.\n\nEnsures that a copy of all events of type `EventCategory` is sent to the caller.\nThe available event types are described in the\n[User's Guide](mnesia_chap5.md#event_handling).","ref":"mnesia.html#subscribe/1"},{"type":"function","title":"mnesia.sync_dirty/1","doc":"","ref":"mnesia.html#sync_dirty/1"},{"type":"function","title":"mnesia.sync_dirty/2","doc":"Call the `Fun` in a context that is not protected by a transaction.\n\nThe Mnesia function calls performed in the `Fun` are mapped to the corresponding dirty\nfunctions. It is performed in almost the same context as\n`mnesia:async_dirty/1,2`. The difference is that the operations are performed\nsynchronously. The caller waits for the updates to be performed on all active\nreplicas before the `Fun` returns. For details, see `mnesia:activity/4` and the\nUser's Guide.","ref":"mnesia.html#sync_dirty/2"},{"type":"function","title":"mnesia.sync_log/0","doc":"Perform a file sync of the local log file.\n\nEnsures that the local transaction log file is synced to disk. On a single node\nsystem, data written to disk tables since the last dump can be lost if there is\na power outage. See `dump_log/0`.","ref":"mnesia.html#sync_log/0"},{"type":"function","title":"mnesia.sync_transaction/1","doc":"","ref":"mnesia.html#sync_transaction/1"},{"type":"function","title":"mnesia.sync_transaction/2","doc":"","ref":"mnesia.html#sync_transaction/2"},{"type":"function","title":"mnesia.sync_transaction/3","doc":"Synchronously execute a transaction.\n\nWaits until data have been committed and logged to disk (if disk is used) on\nevery involved node before it returns, otherwise it behaves as\n`mnesia:transaction/[1,2,3]`.\n\nThis functionality can be used to avoid that one process overloads a database on\nanother node.","ref":"mnesia.html#sync_transaction/3"},{"type":"function","title":"mnesia.system_info/1","doc":"Return information about the Mnesia system.\n\nSuch as transaction statistics, `db_nodes`, and configuration\nparameters.\n\nThe valid keys are as follows:\n\n- `all`. Returns a list of all local system information. Each element is a\n  `{InfoKey, InfoVal}` tuple.\n\n  New `InfoKey`s can be added and old undocumented `InfoKey`s can be removed\n  without notice.\n\n- `access_module`. Returns the name of module that is configured to be the\n  activity access callback module.\n- `auto_repair`. Returns `true` or `false` to indicate if Mnesia is configured\n  to start the auto-repair facility on corrupted disc files.\n- `backup_module`. Returns the name of the module that is configured to be the\n  backup callback module.\n- `checkpoints`. Returns a list of the names of the checkpoints currently active\n  on this node.\n- `event_module`. Returns the name of the module that is the event handler\n  callback module.\n- `db_nodes`. Returns the nodes that make up the persistent database. Disc-less\n  nodes are only included in the list of nodes if they explicitly have been\n  added to the schema, for example, with `mnesia:add_table_copy/3`. The function\n  can be started even if Mnesia is not yet running.\n- `debug`. Returns the current debug level of Mnesia.\n- `directory`. Returns the name of the Mnesia directory. It can be called even\n  if Mnesia is not yet running.\n- `dump_log_load_regulation`. Returns a boolean that tells if Mnesia is\n  configured to regulate the dumper process load.\n\n  This feature is temporary and will be removed in future releases.\n\n- `dump_log_time_threshold`. Returns the time threshold for transaction log\n  dumps in milliseconds.\n- `dump_log_update_in_place`. Returns a boolean that tells if Mnesia is\n  configured to perform the updates in the Dets files directly, or if the\n  updates are to be performed in a copy of the Dets files.\n- `dump_log_write_threshold`. Returns the write threshold for transaction log\n  dumps as the number of writes to the transaction log.\n- `extra_db_nodes`. Returns a list of extra `db_nodes` to be contacted at\n  startup.\n- `fallback_activated`. Returns `true` if a fallback is activated, otherwise\n  `false`.\n- `held_locks`. Returns a list of all locks held by the local Mnesia lock\n  manager.\n- `is_running`. Returns `yes` or `no` to indicate if Mnesia is running. It can\n  also return `starting` or `stopping`. Can be called even if Mnesia is not yet\n  running.\n- `local_tables`. Returns a list of all tables that are configured to reside\n  locally.\n- `lock_queue`. Returns a list of all transactions that are queued for execution\n  by the local lock manager.\n- `log_version`. Returns the version number of the Mnesia transaction log\n  format.\n- `master_node_tables`. Returns a list of all tables with at least one master\n  node.\n- `protocol_version`. Returns the version number of the Mnesia inter-process\n  communication protocol.\n- `running_db_nodes`. Returns a list of nodes where Mnesia currently is running.\n  This function can be called even if Mnesia is not yet running, but it then has\n  slightly different semantics.\n\n  If Mnesia is down on the local node, the function returns those other\n  `db_nodes` and `extra_db_nodes` that for the moment are operational.\n\n  If Mnesia is started, the function returns those nodes that Mnesia on the\n  local node is fully connected to. Only those nodes that Mnesia has exchanged\n  schema information with are included as `running_db_nodes`. After the merge of\n  schemas, the local Mnesia system is fully operable and applications can\n  perform access of remote replicas. Before the schema merge, Mnesia only\n  operates locally. Sometimes there are more nodes included in the\n  `running_db_nodes` list than all `db_nodes` and `extra_db_nodes` together.\n\n- `schema_location`. Returns the initial schema location.\n- `subscribers`. Returns a list of local processes currently subscribing to\n  system events.\n- `tables`. Returns a list of all locally known tables.\n- `transactions`. Returns a list of all currently active local transactions.\n- `transaction_failures`. Returns a number that indicates how many transactions\n  have failed since Mnesia was started.\n- `transaction_commits`. Returns a number that indicates how many transactions\n  have terminated successfully since Mnesia was started.\n- `transaction_restarts`. Returns a number that indicates how many transactions\n  have been restarted since Mnesia was started.\n- `transaction_log_writes`. Returns a number that indicates how many write\n  operations that have been performed to the transaction log since startup.\n- `use_dir`. Returns a boolean that indicates if the Mnesia directory is used or\n  not. Can be started even if Mnesia is not yet running.\n- `version`. Returns the current version number of Mnesia.","ref":"mnesia.html#system_info/1"},{"type":"function","title":"mnesia.table/1","doc":"","ref":"mnesia.html#table/1"},{"type":"function","title":"mnesia.table/2","doc":"Return a QLC query handle.\n\nReturns a Query List Comprehension (QLC) query handle, see the `m:qlc` manual\npage in STDLIB. The module `qlc` implements a query language that can use Mnesia\ntables as sources of data. Calling `mnesia:table/1,2` is the means to make the\n`mnesia` table `Tab` usable to QLC.\n\n`Option` can contain Mnesia options or QLC options. Mnesia recognizes the\nfollowing options (any other option is forwarded to QLC).\n\n- `{lock, Lock}`, where `lock` can be `read` or `write`. Default is `read`.\n- `{n_objects,Number}`, where `n_objects` specifies (roughly) the number of\n  objects returned from Mnesia to QLC. Queries to remote tables can need a\n  larger chunk to reduce network overhead. By default, `100` objects at a time\n  are returned.\n- `{traverse, SelectMethod}`, where `traverse` determines the method to traverse\n  the whole table (if needed). The default method is `select`.\n\nThere are two alternatives for `select`:\n\n- `select`. The table is traversed by calling `mnesia:select/4` and\n  `mnesia:select/1`. The match specification (the second argument of\n  [`select/3`](`select/3`)) is assembled by QLC: simple filters are translated\n  into equivalent match specifications. More complicated filters need to be\n  applied to all objects returned by [`select/3`](`select/3`) given a match\n  specification that matches all objects.\n- `{select, MatchSpec}`. As for `select`, the table is traversed by calling\n  `mnesia:select/3` and `mnesia:select/1`. The difference is that the match\n  specification is explicitly given. This is how to state match specifications\n  that cannot easily be expressed within the syntax provided by QLC.","ref":"mnesia.html#table/2"},{"type":"function","title":"mnesia.table_info/2","doc":"Return local information about table.\n\nThe [`table_info/2`](`table_info/2`) function takes two arguments. The first is\nthe name of a Mnesia table. The second is one of the following keys:\n\n- `all`. Returns a list of all local table information. Each element is a\n  `{InfoKey, ItemVal}` tuple.\n\n  New `InfoItem`s can be added and old undocumented `InfoItem`s can be removed\n  without notice.\n\n- `access_mode`. Returns the access mode of the table. The access mode can be\n  `read_only` or `read_write`.\n- `arity`. Returns the arity of records in the table as specified in the schema.\n- `attributes`. Returns the table attribute names that are specified in the\n  schema.\n- `checkpoints`. Returns the names of the currently active checkpoints, which\n  involve this table on this node.\n- `cookie`. Returns a table cookie, which is a unique system-generated\n  identifier for the table. The cookie is used internally to ensure that two\n  different table definitions using the same table name cannot accidentally be\n  intermixed. The cookie is generated when the table is created initially.\n- `disc_copies`. Returns the nodes where a `disc_copy` of the table resides\n  according to the schema.\n- `disc_only_copies`. Returns the nodes where a `disc_only_copy` of the table\n  resides according to the schema.\n- `index`. Returns the list of index position integers for the table.\n- `load_node`. Returns the name of the node that Mnesia loaded the table from.\n  The structure of the returned value is unspecified, but can be useful for\n  debugging purposes.\n- `load_order`. Returns the load order priority of the table. It is an integer\n  and defaults to `0` (zero).\n- `load_reason`. Returns the reason of why Mnesia decided to load the table. The\n  structure of the returned value is unspecified, but can be useful for\n  debugging purposes.\n- `local_content`. Returns `true` or `false` to indicate if the table is\n  configured to have locally unique content on each node.\n- `master_nodes`. Returns the master nodes of a table.\n- `memory`. Returns for `ram_copies` and `disc_copies` tables the number of\n  words allocated in memory to the table on this node. For `disc_only_copies`\n  tables the number of bytes stored on disc is returned.\n- `ram_copies`. Returns the nodes where a `ram_copy` of the table resides\n  according to the schema.\n- `record_name`. Returns the record name, common for all records in the table.\n- `size`. Returns the number of records inserted in the table.\n- `snmp`. Returns the SNMP struct. `[]` means that the table currently has no\n  SNMP properties.\n- `storage_type`. Returns the local storage type of the table. It can be\n  `disc_copies`, `ram_copies`, `disc_only_copies`, or the atom `unknown`.\n  `unknown` is returned for all tables that only reside remotely.\n- `subscribers`. Returns a list of local processes currently subscribing to\n  local table events that involve this table on this node.\n- `type`. Returns the table type, which is `bag`, `set`, or `ordered_set`.\n- `user_properties`. Returns the user-associated table properties of the table.\n  It is a list of the stored property records.\n- `version`. Returns the current version of the table definition. The table\n  version is incremented when the table definition is changed. The table\n  definition can be incremented directly when it has been changed in a schema\n  transaction, or when a committed table definition is merged with table\n  definitions from other nodes during startup.\n- `where_to_read`. Returns the node where the table can be read. If value\n  `nowhere` is returned, either the table is not loaded or it resides at a\n  remote node that is not running.\n- `where_to_write`. Returns a list of the nodes that currently hold an active\n  replica of the table.\n- `wild_pattern`. Returns a structure that can be given to the various match\n  functions for a certain table. A record tuple is where all record fields have\n  value `'_'`.","ref":"mnesia.html#table_info/2"},{"type":"function","title":"mnesia.transaction/1","doc":"","ref":"mnesia.html#transaction/1"},{"type":"function","title":"mnesia.transaction/2","doc":"","ref":"mnesia.html#transaction/2"},{"type":"function","title":"mnesia.transaction/3","doc":"Execute `Fun` with arguments `Args` as a transaction.\n\nThe code that executes inside the transaction can consist of a series of table\nmanipulation functions. If something goes wrong inside the transaction as a\nresult of a user error or a certain table not being available, the entire\ntransaction is terminated and the function [`transaction/1`](`transaction/1`)\nreturns the tuple `{aborted, Reason}`.\n\nIf all is going well, `{atomic, ResultOfFun}` is returned, where `ResultOfFun`\nis the value of the last expression in `Fun`.\n\nA function that adds a family to the database can be written as follows if there\nis a structure `{family, Father, Mother, ChildrenList}`:\n\n```erlang\nadd_family({family, F, M, Children}) ->\n    ChildOids = lists:map(fun oid/1, Children),\n    Trans = fun() ->\n        mnesia:write(F#person{children = ChildOids}),\n        mnesia:write(M#person{children = ChildOids}),\n        Write = fun(Child) -> mnesia:write(Child) end,\n        lists:foreach(Write, Children)\n    end,\n    mnesia:transaction(Trans).\n\noid(Rec) -> {element(1, Rec), element(2, Rec)}.\n```\n\nThis code adds a set of people to the database. Running this code within one\ntransaction ensures that either the whole family is added to the database, or\nthe whole transaction terminates. For example, if the last child is badly\nformatted, or the executing process terminates because of an `'EXIT'` signal\nwhile executing the family code, the transaction terminates. Thus, the situation\nwhere half a family is added can never occur.\n\nIt is also useful to update the database within a transaction if several\nprocesses concurrently update the same records. For example, the function\n`raise(Name, Amount)`, which adds `Amount` to the salary field of a person, is\nto be implemented as follows:\n\n```erlang\nraise(Name, Amount) ->\n    mnesia:transaction(fun() ->\n        case mnesia:wread({person, Name}) of\n            [P] ->\n                Salary = Amount + P#person.salary,\n                P2 = P#person{salary = Salary},\n                mnesia:write(P2);\n            _ ->\n                mnesia:abort(\"No such person\")\n        end\n    end).\n```\n\nWhen this function executes within a transaction, several processes running on\ndifferent nodes can concurrently execute the function `raise/2` without\ninterfering with each other.\n\nSince Mnesia detects deadlocks, a transaction can be restarted any number of\ntimes and therefore the `Fun` shall not have any side effects such as waiting\nfor specific messages. This function attempts a restart as many times as\nspecified in `Retries`. `Retries` must be an integer greater than 0 or the atom\n`infinity`, default is `infinity`. Mnesia uses `exit` exceptions to signal that\na transaction needs to be restarted, thus a `Fun` must not catch `exit`\nexceptions with reason `{aborted, term()}`.","ref":"mnesia.html#transaction/3"},{"type":"function","title":"mnesia.transform_table/3","doc":"","ref":"mnesia.html#transform_table/3"},{"type":"function","title":"mnesia.transform_table/4","doc":"Change format on all records in table.\n\nApplies argument `Fun` to all records in the table. `Fun` is a function that\ntakes a record of the old type and returns a transformed record of the new type.\nArgument `Fun` can also be the atom `ignore`, which indicates that only the\nmetadata about the table is updated. Use of `ignore` is not recommended, but\nincluded as a possibility for the user do to an own transformation.\n\n`NewAttributeList` and `NewRecordName` specify the attributes and the new record\ntype of the converted table. Table name always remains unchanged. If\n`record_name` is changed, only the Mnesia functions that use table identifiers\nwork, for example, `mnesia:write/3` works, but not `mnesia:write/1`.","ref":"mnesia.html#transform_table/4"},{"type":"function","title":"mnesia.traverse_backup/4","doc":"","ref":"mnesia.html#traverse_backup/4"},{"type":"function","title":"mnesia.traverse_backup/6","doc":"Traverse a backup.\n\nIterates over a backup, either to transform it into a new backup, or read it.\nThe arguments are explained briefly here. For details, see the User's Guide.\n\n- `SourceMod` and `TargetMod` are the names of the modules that actually access\n  the backup media.\n- `Source` and `Target` are opaque data used exclusively by modules `SourceMod`\n  and `TargetMod` to initialize the backup media.\n- `Acc` is an initial accumulator value.\n- `Fun(BackupItems, Acc)` is applied to each item in the backup. The `Fun` must\n  return a tuple `{BackupItems,NewAcc}`, where `BackupItems` is a list of valid\n  backup items, and `NewAcc` is a new accumulator value. The returned backup\n  items are written in the target backup.\n- `LastAcc` is the last accumulator value. This is the last `NewAcc` value that\n  was returned by `Fun`.","ref":"mnesia.html#traverse_backup/6"},{"type":"function","title":"mnesia.uninstall_fallback/0","doc":"","ref":"mnesia.html#uninstall_fallback/0"},{"type":"function","title":"mnesia.uninstall_fallback/1","doc":"Uninstall a fallback.\n\nDeinstalls a fallback before it has been used to restore the database. This is\nnormally a distributed operation that is either performed on all nodes with disc\nresident schema, or none. Uninstallation of fallbacks requires Erlang to be\noperational on all involved nodes, but it does not matter if Mnesia is running\nor not. Which nodes that are considered as disc-resident nodes is determined\nfrom the schema information in the local fallback.\n\n`Args` is a list of the following tuples:\n\n- `{module, BackupMod}`. For semantics, see `mnesia:install_fallback/2`.\n- `{scope, Scope}`. For semantics, see `mnesia:install_fallback/2`.\n- `{mnesia_dir, AlternateDir}`. For semantics, see `mnesia:install_fallback/2`.","ref":"mnesia.html#uninstall_fallback/1"},{"type":"function","title":"mnesia.unsubscribe/1","doc":"Stop sending events of type `EventCategory` to the caller.\n\n`Node` is the local node.","ref":"mnesia.html#unsubscribe/1"},{"type":"function","title":"mnesia.wait_for_tables/2","doc":"Wait for tables to be accessible.\n\nSome applications need to wait for certain tables to be accessible to do useful\nwork. `mnesia:wait_for_tables/2` either hangs until all tables in `TabList` are\naccessible, or until `timeout` is reached.","ref":"mnesia.html#wait_for_tables/2"},{"type":"function","title":"mnesia.wread/1","doc":"","ref":"mnesia.html#wread/1"},{"type":"function","title":"mnesia.write/1","doc":"Write a record into the database.\n\nCalls the function `mnesia:write(Tab, Record, write)`, where `Tab` is\n[`element(1, Record)`](`element/2`).","ref":"mnesia.html#write/1"},{"type":"function","title":"mnesia.write/3","doc":"Write `Record` to table `Tab`.\n\nThe function returns `ok`, or terminates if an error occurs. For example, the\ntransaction terminates if no `person` table exists.\n\nThe semantics of this function is context-sensitive. For details, see\n`mnesia:activity/4`. In transaction-context, it acquires a lock of type\n`LockKind`. The lock types `write` and `sticky_write` are supported.","ref":"mnesia.html#write/3"},{"type":"function","title":"mnesia.write_lock_table/1","doc":"","ref":"mnesia.html#write_lock_table/1"},{"type":"type","title":"mnesia.activity/0","doc":"","ref":"mnesia.html#t:activity/0"},{"type":"type","title":"mnesia.config_key/0","doc":"","ref":"mnesia.html#t:config_key/0"},{"type":"type","title":"mnesia.config_result/0","doc":"","ref":"mnesia.html#t:config_result/0"},{"type":"type","title":"mnesia.config_value/0","doc":"","ref":"mnesia.html#t:config_value/0"},{"type":"type","title":"mnesia.create_option/0","doc":"","ref":"mnesia.html#t:create_option/0"},{"type":"type","title":"mnesia.debug_level/0","doc":"","ref":"mnesia.html#t:debug_level/0"},{"type":"type","title":"mnesia.index_attr/0","doc":"","ref":"mnesia.html#t:index_attr/0"},{"type":"type","title":"mnesia.lock_kind/0","doc":"","ref":"mnesia.html#t:lock_kind/0"},{"type":"type","title":"mnesia.read_locks/0","doc":"","ref":"mnesia.html#t:read_locks/0"},{"type":"type","title":"mnesia.result/0","doc":"","ref":"mnesia.html#t:result/0"},{"type":"type","title":"mnesia.select_continuation/0","doc":"","ref":"mnesia.html#t:select_continuation/0"},{"type":"type","title":"mnesia.snmp_struct/0","doc":"","ref":"mnesia.html#t:snmp_struct/0"},{"type":"type","title":"mnesia.snmp_type/0","doc":"","ref":"mnesia.html#t:snmp_type/0"},{"type":"type","title":"mnesia.storage_type/0","doc":"","ref":"mnesia.html#t:storage_type/0"},{"type":"type","title":"mnesia.t_result/1","doc":"","ref":"mnesia.html#t:t_result/1"},{"type":"type","title":"mnesia.table/0","doc":"","ref":"mnesia.html#t:table/0"},{"type":"type","title":"mnesia.tuple_of/1","doc":"","ref":"mnesia.html#t:tuple_of/1"},{"type":"type","title":"mnesia.write_locks/0","doc":"","ref":"mnesia.html#t:write_locks/0"},{"type":"module","title":"mnesia_frag_hash","doc":"Defines mnesia_frag_hash callback behavior\n\nThis module defines a callback behavior for user-defined hash functions of\nfragmented tables.\n\nWhich module that is selected to implement the `mnesia_frag_hash` behavior for a\nparticular fragmented table is specified together with the other\n`frag_properties`. The `hash_module` defines the module name. The `hash_state`\ndefines the initial hash state.\n\nThis module implements dynamic hashing, which is a kind of hashing that grows\nnicely when new fragments are added. It is well suited for scalable hash tables.","ref":"mnesia_frag_hash.html"},{"type":"module","title":"See Also - mnesia_frag_hash","doc":"`m:mnesia`","ref":"mnesia_frag_hash.html#module-see-also"},{"type":"function","title":"mnesia_frag_hash.add_frag/1","doc":"add_frag(State) -> {NewState, IterFrags, AdditionalLockFrags} | abort(Reason)\n\nTo scale well, it is a good idea to ensure that the records are evenly\ndistributed over all fragments, including the new one.\n\n`NewState` is stored as `hash_state` among the other `frag_properties`.\n\nAs a part of the `add_frag` procedure, Mnesia iterates over all fragments\ncorresponding to the `IterFrags` numbers and starts\n[`key_to_frag_number(NewState,RecordKey)`](`key_to_frag_number/2`) for each\nrecord. If the new fragment differs from the old fragment, the record is moved\nto the new fragment.\n\nAs the `add_frag` procedure is a part of a schema transaction, Mnesia acquires\nwrite locks on the affected tables. That is, both the fragments corresponding to\n`IterFrags` and those corresponding to `AdditionalLockFrags`.","ref":"mnesia_frag_hash.html#add_frag/1"},{"type":"function","title":"mnesia_frag_hash.del_frag/1","doc":"del_frag(State) -> {NewState, IterFrags, AdditionalLockFrags} | abort(Reason)\n\n`NewState` is stored as `hash_state` among the other `frag_properties`.\n\nAs a part of the `del_frag` procedure, Mnesia iterates over all fragments\ncorresponding to the `IterFrags` numbers and starts\n[`key_to_frag_number(NewState,RecordKey)`](`key_to_frag_number/2`) for each\nrecord. If the new fragment differs from the old fragment, the record is moved\nto the new fragment.\n\nNotice that all records in the last fragment must be moved to another fragment,\nas the entire fragment is deleted.\n\nAs the `del_frag` procedure is a part of a schema transaction, Mnesia acquires\nwrite locks on the affected tables. That is, both the fragments corresponding to\n`IterFrags` and those corresponding to `AdditionalLockFrags`.","ref":"mnesia_frag_hash.html#del_frag/1"},{"type":"function","title":"mnesia_frag_hash.init_state/2","doc":"init_state(Tab, State) -> NewState | abort(Reason)\n\nStarts when a fragmented table is created with the function\n`mnesia:create_table/2` or when a normal (unfragmented) table is converted to be\na fragmented table with `mnesia:change_table_frag/2`.\n\nNotice that the function `add_frag/2` is started one time for each of the other\nfragments (except number 1) as a part of the table creation procedure.\n\n`State` is the initial value of the `hash_state` `frag_property`. `NewState` is\nstored as `hash_state` among the other `frag_properties`.","ref":"mnesia_frag_hash.html#init_state/2"},{"type":"function","title":"mnesia_frag_hash.key_to_frag_number/2","doc":"key_to_frag_number(State, Key) -> FragNum | abort(Reason)\n\nStarts whenever Mnesia needs to determine which fragment a certain record\nbelongs to. It is typically started at `read`, `write`, and `delete`.","ref":"mnesia_frag_hash.html#key_to_frag_number/2"},{"type":"function","title":"mnesia_frag_hash.match_spec_to_frag_numbers/2","doc":"match_spec_to_frag_numbers(State, MatchSpec) -> FragNums | abort(Reason)\n\nThis function is called whenever Mnesia needs to determine which fragments that\nneed to be searched for a `MatchSpec`. It is typically called by `select` and\n`match_object`.","ref":"mnesia_frag_hash.html#match_spec_to_frag_numbers/2"},{"type":"module","title":"mnesia_registry","doc":"This module is deprecated and the functions should not be used.\n\nThis module was intended for internal use within OTP by the `erl_interface` application,\nbut it has two functions that are exported for public use.\n\nSince the `erl_interface` have removed the registry functionality a long time ago,\nthese functions are deprecated.","ref":"mnesia_registry.html"},{"type":"module","title":"See Also - mnesia_registry","doc":"`m:mnesia`","ref":"mnesia_registry.html#module-see-also"},{"type":"function","title":"mnesia_registry.create_table/1","doc":"> #### Warning {: .warning }\n>\n> _This function is deprecated. Do not use it._\n>\n\nA wrapper function for `mnesia:create_table/2`, which creates a table (if there\nis no existing table) with an appropriate set of `attributes`. The table only\nresides on the local node and its storage type is the same as the `schema` table\non the local node, that is, `{ram_copies,[node()]}` or `{disc_copies,[node()]}`.\n\nThis function is used by `erl_interface` to create the Mnesia table if it does\nnot already exist.","ref":"mnesia_registry.html#create_table/1"},{"type":"function","title":"mnesia_registry.create_table/2","doc":"> #### Warning {: .warning }\n>\n> _This function is deprecated. Do not use it._\n>\n\nA wrapper function for `mnesia:create_table/2`, which creates a table (if there\nis no existing table) with an appropriate set of `attributes`. The attributes\nand `TabDef` are forwarded to `mnesia:create_table/2`. For example, if the table\nis to reside as `disc_only_copies` on all nodes, a call looks as follows:\n\n```erlang\n          TabDef = [{{disc_only_copies, node()|nodes()]}],\n          mnesia_registry:create_table(my_reg, TabDef)\n```","ref":"mnesia_registry.html#create_table/2"},{"type":"extras","title":"Mnesia Release Notes","doc":"<!--\n%CopyrightBegin%\n\nCopyright Ericsson AB 2023-2024. All Rights Reserved.\n\nLicensed under the Apache License, Version 2.0 (the \"License\");\nyou may not use this file except in compliance with the License.\nYou may obtain a copy of the License at\n\n    http://www.apache.org/licenses/LICENSE-2.0\n\nUnless required by applicable law or agreed to in writing, software\ndistributed under the License is distributed on an \"AS IS\" BASIS,\nWITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.\nSee the License for the specific language governing permissions and\nlimitations under the License.\n\n%CopyrightEnd%\n-->\n# Mnesia Release Notes\n\nThis document describes the changes made to the Mnesia system from version to\nversion. The intention of this document is to list all incompatibilities as well\nas all enhancements and bugfixes for every release of Mnesia. Each release of\nMnesia thus constitutes one section in this document. The title of each section\nis the version number of Mnesia.","ref":"notes.html"},{"type":"extras","title":"Mnesia 4.23.2 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-23-2"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Mnesia Release Notes","doc":"- The `mnesia_registry` module have been deprecated.\n\n  Own Id: OTP-18994","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Improvements and New Features - Mnesia Release Notes","doc":"- The documentation has been migrated to use Markdown and ExDoc.\n\n  Own Id: OTP-18955 Aux Id: [PR-8026]\n\n[PR-8026]: https://github.com/erlang/otp/pull/8026","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Mnesia 4.23.1 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-23-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Mnesia Release Notes","doc":"* Mnesia could crash during startup if `del_table_copy/2` and `add_table_copy/3` was invoked when the table was loading.\n\n  Own Id: OTP-19076 Aux Id: ERIERL-1073","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Mnesia 4.23 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-23"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Mnesia Release Notes","doc":"- Document `mnesia:foldl/4` and `mnesia:foldr/4`.\n\n  Own Id: OTP-18798\n\n- `mnesia:add_table_copy/3` no longer fails with reason system_limit when the\n  node is starting.\n\n  Own Id: OTP-18850","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Improvements and New Features - Mnesia Release Notes","doc":"- Restore recreate of disc_only tables could crash if they had an index.\n\n  Own Id: OTP-18843 Aux Id: GH-7766","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Mnesia 4.22.1 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-22-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Mnesia Release Notes","doc":"- Do not delete old backup file if the new backup fails.\n\n  Own Id: OTP-18711 Aux Id: ERIERL-963","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Mnesia 4.22 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-22"},{"type":"extras","title":"Improvements and New Features - Mnesia Release Notes","doc":"- Added debug statistics for active transactions.\n\n  Own Id: OTP-18309 Aux Id: PR-6377\n\n- The implementation has been fixed to use `proc_lib:init_fail/2,3` where\n  appropriate, instead of `proc_lib:init_ack/1,2`.\n\n  \\*** POTENTIAL INCOMPATIBILITY \\***\n\n  Own Id: OTP-18490 Aux Id: OTP-18471, GH-6339, PR-6843","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Mnesia 4.21.4.3 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-21-4-3"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Mnesia Release Notes","doc":"* Mnesia could crash during startup if `del_table_copy/2` and `add_table_copy/3` was invoked when the table was loading.\n\n  Own Id: OTP-19076 Aux Id: ERIERL-1073","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Mnesia 4.21.4.2 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-21-4-2"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Mnesia Release Notes","doc":"- `mnesia:add_table_copy/3` no longer fails with reason system_limit when the\n  node is starting.\n\n  Own Id: OTP-18850","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Mnesia 4.21.4.1 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-21-4-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Mnesia Release Notes","doc":"- Do not delete old backup file if the new backup fails.\n\n  Own Id: OTP-18711 Aux Id: ERIERL-963","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Mnesia 4.21.4 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-21-4"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Mnesia Release Notes","doc":"- Improved consistency for dirty writes when a table was added with\n  `add_table_copy/3`.\n\n  Fixed a problem with sticky write, which could lead to inconsistent data.\n\n  Own Id: OTP-18412","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Improvements and New Features - Mnesia Release Notes","doc":"- Replace size/1 with either tuple_size/1 or byte_size/1\n\n  The [`size/1`](`size/1`) BIF is not optimized by the JIT, and its use can\n  result in worse types for Dialyzer.\n\n  When one knows that the value being tested must be a tuple,\n  [`tuple_size/1`](`tuple_size/1`) should always be preferred.\n\n  When one knows that the value being tested must be a binary,\n  [`byte_size/1`](`byte_size/1`) should be preferred. However,\n  [`byte_size/1`](`byte_size/1`) also accepts a bitstring (rounding up size to a\n  whole number of bytes), so one must make sure that the call to `byte_size/` is\n  preceded by a call to [`is_binary/1`](`is_binary/1`) to ensure that bitstrings\n  are rejected. Note that the compiler removes redundant calls to\n  [`is_binary/1`](`is_binary/1`), so if one is not sure whether previous code\n  had made sure that the argument is a binary, it does not harm to add an\n  [`is_binary/1`](`is_binary/1`) test immediately before the call to\n  [`byte_size/1`](`byte_size/1`).\n\n  Own Id: OTP-18432 Aux Id:\n  GH-6672,PR-6793,PR-6784,PR-6787,PR-6785,PR-6682,PR-6800,PR-6797,PR-6798,PR-6799,PR-6796,PR-6813,PR-6671,PR-6673,PR-6684,PR-6694,GH-6677,PR-6696,PR-6670,PR-6674","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Mnesia 4.21.3 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-21-3"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Mnesia Release Notes","doc":"- Fixed crash which could happen during startup if too many decisions where sent\n  from remote nodes.\n\n  Own Id: OTP-18319 Aux Id: ERIERL-875","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Mnesia 4.21.2 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-21-2"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Mnesia Release Notes","doc":"- Don't fill the logs if mnesia can't connect to all nodes, due to partitioned\n  network.\n\n  Own Id: OTP-18288 Aux Id: ERIERL-868","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Mnesia 4.21.1 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-21-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Mnesia Release Notes","doc":"- Fixed `add_table_copy` which could leave a table lock if the receiving node\n  went down during the operation.\n\n  Own Id: OTP-18128 Aux Id: PR-6013","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Mnesia 4.21 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-21"},{"type":"extras","title":"Improvements and New Features - Mnesia Release Notes","doc":"- Documentation fixes.\n\n  Own Id: OTP-17930","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Mnesia 4.20.4.4 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-20-4-4"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Mnesia Release Notes","doc":"- `mnesia:add_table_copy/3` no longer fails with reason system_limit when the\n  node is starting.\n\n  Own Id: OTP-18850","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Mnesia 4.20.4.3 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-20-4-3"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Mnesia Release Notes","doc":"- Do not delete old backup file if the new backup fails.\n\n  Own Id: OTP-18711 Aux Id: ERIERL-963","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Mnesia 4.20.4.2 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-20-4-2"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Mnesia Release Notes","doc":"- Don't fill the logs if mnesia can't connect to all nodes, due to partitioned\n  network.\n\n  Own Id: OTP-18288 Aux Id: ERIERL-868\n\n- Fixed crash which could happen during startup if too many decisions where sent\n  from remote nodes.\n\n  Own Id: OTP-18319 Aux Id: ERIERL-875","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Mnesia 4.20.4.1 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-20-4-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Mnesia Release Notes","doc":"- Fixed `add_table_copy` which could leave a table lock if the receiving node\n  went down during the operation.\n\n  Own Id: OTP-18128 Aux Id: PR-6013","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Mnesia 4.20.4 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-20-4"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Mnesia Release Notes","doc":"- Fixed `mnesia:add_table_copy/3` so that calling it when mnesia started on\n  another node does not fail or cause hanging nodes.\n\n  Own Id: OTP-18056","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Mnesia 4.20.3 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-20-3"},{"type":"extras","title":"Improvements and New Features - Mnesia Release Notes","doc":"- Optimize locker to handle many read locks on the same record.\n\n  Own Id: OTP-17973 Aux Id: ERIERL-772","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Mnesia 4.20.2 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-20-2"},{"type":"extras","title":"Improvements and New Features - Mnesia Release Notes","doc":"- Reduce the number of locks taken during table copying, should reduce the\n  startup time on large systems.\n\n  Own Id: OTP-17656 Aux Id: ERIERL-688","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Mnesia 4.20.1 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-20-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Mnesia Release Notes","doc":"- Documentation and minor code cleanup.\n\n  Own Id: OTP-17727","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Mnesia 4.20 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-20"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Mnesia Release Notes","doc":"- Fixed that index keys was deleted for set tables when mnesia:delete_object/1\n  tried to delete a non-existing record.\n\n  Own Id: OTP-17564 Aux Id: GH-5040","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Improvements and New Features - Mnesia Release Notes","doc":"- Optimized table loading and added `max_transfer_size` configuration parameter.\n\n  Own Id: OTP-17508","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Mnesia 4.19.1 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-19-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Mnesia Release Notes","doc":"- Suppression of deprecation warnings has been added to the source files of the\n  Mnesia application.\n\n  Own Id: OTP-17217\n\n- Fixed that the backend plugin initialization is done only once.\n\n  Own Id: OTP-17294 Aux Id: GH-4525 PR-4674","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Mnesia 4.19 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-19"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Mnesia Release Notes","doc":"- Fixed the type spec for `disc_only_copies`.\n\n  Own Id: OTP-17249 Aux Id: PR-4578\n\n- Do not crash in `mnesia:change_config/2` if mnesia is stopping or starting.\n\n  Own Id: OTP-17274 Aux Id: GH-4616","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Improvements and New Features - Mnesia Release Notes","doc":"- Optimized table loading time for tables that are updated during the loading.\n\n  Own Id: OTP-17271 Aux Id: PR-4575","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Mnesia 4.18.1 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-18-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Mnesia Release Notes","doc":"- Avoid potential performance issue, if the input queue to `mnesia_tm` is long.\n\n  Own Id: OTP-17066 Aux Id: PR-2889","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Mnesia 4.18 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-18"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Mnesia Release Notes","doc":"- FIx mnesia delete object handling in transaction storage. In a transaction\n  `mnesia:read/1` could indicate that exiting objects did not exist after\n  another object was deleted.\n\n  Own Id: OTP-16782 Aux Id: PR-2663","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Improvements and New Features - Mnesia Release Notes","doc":"- Fixed crash during startup, which could happen if a table was deleted on\n  another node.\n\n  Own Id: OTP-16815 Aux Id: ERIERL-500","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Mnesia 4.17 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-17"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Mnesia Release Notes","doc":"- Make `mnesia:create_table/2` return correct badarg value.\n\n  Own Id: OTP-16072 Aux Id: PR-2320\n\n- Fixed a bug where mnesia was sometimes not waiting during start for a commit\n  decision on asymmetric transactions.\n\n  Own Id: OTP-16634 Aux Id: PR-2610 ERL-1227","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Improvements and New Features - Mnesia Release Notes","doc":"- Remove usage and documentation of old requests of the I/O-protocol.\n\n  Own Id: OTP-15695\n\n- Avoid using `rpc` calls to do table reads, which will reduce the load on rpc\n  server and improve performance.\n\n  Own Id: OTP-16189","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Mnesia 4.16.3.1 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-16-3-1"},{"type":"extras","title":"Improvements and New Features - Mnesia Release Notes","doc":"- Fixed crash during startup, which could happen if a table was deleted on\n  another node.\n\n  Own Id: OTP-16815 Aux Id: ERIERL-500","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Mnesia 4.16.3 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-16-3"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Mnesia Release Notes","doc":"- Fixed a timing issue in uninstall fallback functionality.\n\n  Own Id: OTP-16468 Aux Id: ERL-1151","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Mnesia 4.16.2 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-16-2"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Mnesia Release Notes","doc":"- Fixed mnesia crash which could happen when trying to recover from failures in\n  transactions containing `sticky_locks`.\n\n  Own Id: OTP-16286 Aux Id: ERL-1077\n\n- Fixed mnesia index issue. Could happen when updating records with a index\n  plugin backend.\n\n  Own Id: OTP-16291 Aux Id: ERL-1091","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Mnesia 4.16.1 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-16-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Mnesia Release Notes","doc":"- `mnesia:add_table_copy/3` could cause a deadlock if called when a new node was\n  starting.\n\n  Own Id: OTP-15933 Aux Id: ERL-872\n\n- Transactions with sticky locks could with async_asym transactions be committed\n  in the wrong order, since asym transaction are spawned on the remote nodes.\n\n  To fix this bug the communication protocol between mnesia nodes had to be\n  updated, thus mnesia will no longer be able to connect to nodes earlier than\n  mnesia-4.14 , OTP-19.0.\n\n  \\*** POTENTIAL INCOMPATIBILITY \\***\n\n  Own Id: OTP-15979 Aux Id: ERL-768","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Mnesia 4.16 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-16"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Mnesia Release Notes","doc":"- Optimize mnesia:read/1 if data have been written in the same transaction.\n\n  Own Id: OTP-15550 Aux Id: PR-2029\n\n- Fixed bugs in table index plugin handling.\n\n  Own Id: OTP-15689 Aux Id: PR-1695 ERL-556","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Improvements and New Features - Mnesia Release Notes","doc":"- Optimized dumping of tables with plugin backends.\n\n  Own Id: OTP-15588 Aux Id: PR-2102\n\n- Include stacktrace in exception if a dirty activity errors, thus if user have\n  matched on the error thrown it may not match any more.\n\n  \\*** POTENTIAL INCOMPATIBILITY \\***\n\n  Own Id: OTP-15804 Aux Id: PR-2216","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Mnesia 4.15.6 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-15-6"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Mnesia Release Notes","doc":"- Avoid overload warnings caused by a race condition.\n\n  Own Id: OTP-15619 Aux Id: ERIERL-310","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Mnesia 4.15.5 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-15-5"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Mnesia Release Notes","doc":"- Fixed type spec for `mnesia:change_config/2`.\n\n  Own Id: OTP-15201 Aux Id: PR-1881\n\n- When master node is set do not force a load from ram_copies replica when there\n  are no available disc_copies, since that would load an empty table. Wait until\n  a disk replica is available or until user explicitly force_loads the table.\n\n  Own Id: OTP-15221 Aux Id: ERIERL-217\n\n- Allow to add replicas even if all other replicas are down when the other\n  replicas are not stored on disk.\n\n  Own Id: OTP-15226 Aux Id: ERIERL-221\n\n- Fixed `mnesia:delete_object/1` bug, where delete_object was deleting the\n  record if it was written in the same transaction even if it was written to a\n  different value.\n\n  Own Id: OTP-15231 Aux Id: PR-1858\n\n- Fixed a bug where the bag table index data was not deleted when objects were\n  deleted.\n\n  Own Id: OTP-15243","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Mnesia 4.15.4 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-15-4"},{"type":"extras","title":"Improvements and New Features - Mnesia Release Notes","doc":"- Calls to `erlang:get_stacktrace()` are removed.\n\n  Own Id: OTP-14861","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Mnesia 4.15.3.1 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-15-3-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Mnesia Release Notes","doc":"- When master node is set do not force a load from ram_copies replica when there\n  are no available disc_copies, since that would load an empty table. Wait until\n  a disk replica is available or until user explicitly force_loads the table.\n\n  Own Id: OTP-15221 Aux Id: ERIERL-217\n\n- Allow to add replicas even if all other replicas are down when the other\n  replicase are not stored on disk.\n\n  Own Id: OTP-15226 Aux Id: ERIERL-221","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Mnesia 4.15.3 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-15-3"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Mnesia Release Notes","doc":"- Removed a quadratic behavior in startup. This change implies that backend\n  plugins (if used) must be set when the schema is created or via configuration\n  parameters before mnesia is started.\n\n  Own Id: OTP-14829 Aux Id: ERIERL-84\n\n- Bad timing could crash mnesia after a checkpoint was deactivated and\n  reactivated with the same checkpoint name on different tables.\n\n  Own Id: OTP-14841 Aux Id: ERIERL-113","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Mnesia 4.15.2 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-15-2"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Mnesia Release Notes","doc":"- Fix backup error handling, the real failure reason was not returned.\n\n  Own Id: OTP-14776 Aux Id: ERIERL-103","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Mnesia 4.15.1 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-15-1"},{"type":"extras","title":"Improvements and New Features - Mnesia Release Notes","doc":"- General Unicode improvements.\n\n  Own Id: OTP-14462","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Mnesia 4.15 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-15"},{"type":"extras","title":"Improvements and New Features - Mnesia Release Notes","doc":"- Removed the wrapping of select continuations in extension plugin handling.\n  This might require the user to rewrite user backend plugin if used.\n\n  \\*** POTENTIAL INCOMPATIBILITY \\***\n\n  Own Id: OTP-14039","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Mnesia 4.14.3 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-14-3"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Mnesia Release Notes","doc":"- Fixed crash in checkpoint handling when table was deleted during backup.\n\n  Own Id: OTP-14167","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Mnesia 4.14.2 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-14-2"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Mnesia Release Notes","doc":"- A continuation returned by mnesia:select/\\[14] should be reusable in\n  different, non-transactional activities.\n\n  Own Id: OTP-13944 Aux Id: PR-1184\n\n- Fixed crash when calling block_table multiple times. Could happen when having\n  locks for a long time and restarting mnesia.\n\n  Own Id: OTP-13970 Aux Id: Seq-13198\n\n- Change mnesia_tm process to have off-heap messages since mnesia_tm can be the\n  receiver of many non-synchronized message from other nodes.\n\n  Own Id: OTP-14074","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Mnesia 4.14.1 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-14-1"},{"type":"extras","title":"Improvements and New Features - Mnesia Release Notes","doc":"- Correct some minor documentation issues.\n\n  Own Id: OTP-13891","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Mnesia 4.14 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-14"},{"type":"extras","title":"Improvements and New Features - Mnesia Release Notes","doc":"- Added experimental external backend plugin api. This adds the possibility for\n  the user to write other storage backends for data, for example by using shared\n  memory or ram-cached disk storage.\n\n  The plugin api may change in future versions after being battle tested.\n\n  Own Id: OTP-13058","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Mnesia 4.13.4 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-13-4"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Mnesia Release Notes","doc":"- Mnesia transactions could hang while waiting on a response from a node who had\n  stopped.\n\n  Own Id: OTP-13423","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Mnesia 4.13.3 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-13-3"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Mnesia Release Notes","doc":"- Avoid deadlock possibility in `mnesia:del_table_copy/2`\n\n  Own Id: OTP-13284","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Mnesia 4.13.2 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-13-2"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Mnesia Release Notes","doc":"- Fixed a process and file descriptor leak in mnesia:restore/2.\n\n  Own Id: OTP-13025 Aux Id: seq12957","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Mnesia 4.13.1 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-13-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Mnesia Release Notes","doc":"- Improved index updates to avoid a timing glitch in dirty_index_read.\n\n  Own Id: OTP-12972","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Mnesia 4.13 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-13"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Mnesia Release Notes","doc":"- Mnesia's dirty functions did not always exit with `{aborted, Reason}` as\n  documented when an error occurred.\n\n  Own Id: OTP-12714\n\n- Consider file descriptors limits (emfile) as a fatal error and do not delete\n  log files. Previously the error was seen as a corrupted disk and the log files\n  deleted which caused data loss.\n\n  Own Id: OTP-12807","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Improvements and New Features - Mnesia Release Notes","doc":"- Make Mnesia DCD dump behavior at start up optional, when turned off mnesia\n  loads large disc_copies tables faster.\n\n  Own Id: OTP-12481","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Mnesia 4.12.5 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-12-5"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Mnesia Release Notes","doc":"- Fixed race condition in protocol negotiation.\n\n  Own Id: OTP-12473","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Improvements and New Features - Mnesia Release Notes","doc":"- Grammar corrections. (Thanks to Derek Brown)\n\n  Own Id: OTP-12400","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Mnesia 4.12.4 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-12-4"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Mnesia Release Notes","doc":"- Fixed a spelling mistake in mnesia documentation.\n\n  Own Id: OTP-12278\n\n- Matching data with `mnesia:match_object/1` did not work as expected in some\n  cases, when data was written in the same transaction before the matching was\n  invoked.\n\n  Own Id: OTP-12304 Aux Id: Seq12745","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Mnesia 4.12.3 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-12-3"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Mnesia Release Notes","doc":"- Various logging fixes, including: Add run queue index to the process dump in\n  crash dumps.  \n  Add thread index to enomem slogan when crashing.  \n  Remove error logger message for sending messages to old instances of the same\n  node.\n\n  Own Id: OTP-12115","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Mnesia 4.12.2 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-12-2"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Mnesia Release Notes","doc":"- Fixed a race which could make create_table fail if a node was going down\n  during the transaction.\n\n  Own Id: OTP-12124 Aux Id: seq12694","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Mnesia 4.12.1 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-12-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Mnesia Release Notes","doc":"- Force load table could hang when a node went away during start up.\n\n  Own Id: OTP-11948 Aux Id: seq12585\n\n- The time for inserting locks for a transaction with large number of locks is\n  reduced significantly.\n\n  Own Id: OTP-11981","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Mnesia 4.12 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-12"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Mnesia Release Notes","doc":"- Some local implementations of removing the last element from a list are\n  replaced by `lists:droplast/1`. Note that this requires at least `stdlib-2.0`,\n  which is the stdlib version delivered in OTP 17.0. (Thanks to Hans Svensson)\n\n  Own Id: OTP-11678\n\n- Application upgrade (appup) files are corrected for the following\n  applications:\n\n  `asn1, common_test, compiler, crypto, debugger, dialyzer, edoc, eldap, erl_docgen, et, eunit, gs, hipe, inets, observer, odbc, os_mon, otp_mibs, parsetools, percept, public_key, reltool, runtime_tools, ssh, syntax_tools, test_server, tools, typer, webtool, wx, xmerl`\n\n  A new test utility for testing appup files is added to test_server. This is\n  now used by most applications in OTP.\n\n  (Thanks to Tobias Schlager)\n\n  Own Id: OTP-11744","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Improvements and New Features - Mnesia Release Notes","doc":"- To prevent a race condition if there is a short communication problem when\n  node-down and node-up events are received. They are now stored and later\n  checked if the node came up just before mnesia flagged the node as down.\n  (Thanks to Jonas Falkevik )\n\n  Own Id: OTP-11497\n\n- Added `mnesia:sync_log/0` to explicit sync mnesias transaction log.\n\n  Own Id: OTP-11729","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Mnesia 4.11 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-11"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Mnesia Release Notes","doc":"- Fixed a race in mnesia which could cause hanging transaction when sticky locks\n  had been used. Thanks janchochol.\n\n  Own Id: OTP-11375\n\n- Fixed dirty_update_counter which could return ok, thanks Anton Ryabkov.\n\n  Own Id: OTP-11485","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Mnesia 4.10 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-10"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Mnesia Release Notes","doc":"- Fix timing issues in checkpoint creation.\n\n  Own Id: OTP-10957","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Improvements and New Features - Mnesia Release Notes","doc":"- Fixed a problem where the fallback BUP file is removed when calling\n  mnesia:uninstall_fallback and mnesia is not started.\n\n  Own Id: OTP-11241","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Mnesia 4.9 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-9"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Mnesia Release Notes","doc":"- If mnesia:clear_table/2 was called during a table load on that table, the\n  schema record was written to the table instead of clearing table.\n\n  Own Id: OTP-11030 Aux Id: seq12267","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Improvements and New Features - Mnesia Release Notes","doc":"- Optimize index creation for Mnesia set tables. Thanks to Nick Marino.\n\n  Own Id: OTP-11103","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Mnesia 4.8 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-8"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Mnesia Release Notes","doc":"- Use chained send_after instead of send_interval, to make decrease the number\n  of messages sent after a sleep (Thanks to James Wheare)\n\n  Own Id: OTP-10636\n\n- Fix format of mnesia overload message (Thanks to Ahmed Omar)\n\n  Own Id: OTP-10639","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Improvements and New Features - Mnesia Release Notes","doc":"- Added a general framework for executing benchmarks of Erlang/OTP. Benchmarks\n  for the Erlang VM and mnesia have been incorporated in the framework.\n\n  For details about how to add more benchmarks see $ERL_TOP/HOWTO/BENCHMARKS.md\n  in the source distribution.\n\n  Own Id: OTP-10156\n\n- Where necessary a comment stating encoding has been added to Erlang files. The\n  comment is meant to be removed in Erlang/OTP R17B when UTF-8 becomes the\n  default encoding.\n\n  Own Id: OTP-10630\n\n- Remove support for the query keyword and query expressions. Thanks to Loïc\n  Hoguin.\n\n  Own Id: OTP-10729","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Mnesia 4.7.1 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-7-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Mnesia Release Notes","doc":"- Add tests showing that trying to delete non-existing object may corrupt the\n\n  In case of bag tables, trying to delete a non-existing object leads to the\n  index becoming corrupt. This happens if the non-existing object we try to\n  delete happens to share its key and index field value with a single existing\n  object in the table. Result: The index entry corresponding to the existing\n  object is removed.\n\n  Prevent index from being corrupted if a nonexistent item is deleted\n\n  We have to ensure that we actually delete the last object with a given (key,\n  index) pair before removing the index. Thanks to Bartlomiej Puzon\n\n  Own Id: OTP-10220","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Mnesia 4.7 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-7"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Mnesia Release Notes","doc":"- Returns the same value for mnesia_loader:disc_load_table/2 as\n  mnesia_loader:net_load_table/4 if a table copy cannot be found. (Thanks to Uwe\n  Dauernheim)\n\n  Own Id: OTP-10015","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Improvements and New Features - Mnesia Release Notes","doc":"- Improved table lock algorithm.\n\n  Own Id: OTP-9890","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Mnesia 4.6 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-6"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Mnesia Release Notes","doc":"- Reduce calls to phash in key_to_frag_number\n\n  Original code calls phash 1..2 times, based on which fragment the hashed key\n  targets and how many fragments exist. New code always calls phash only once.\n\n  Add mnesia_frag_hash test (Thanks to Philip Robinson)\n\n  Own Id: OTP-9722\n\n- Fixed a sticky lock bug which caused mnesia:read(Tab, Key, write) return\n  undefined.\n\n  Own Id: OTP-9786\n\n- Use the synchronous log_terms instead of alog_terms in mnesia_log:ets2dcd()\n\n  This avoids the situation where mnesia could dump a very large ets table in\n  its entirety into the message queue of the disk_log process, causing memory\n  blowup and choking the disk logger. (Thanks to Richard Carlsson)\n\n  Own Id: OTP-9804","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Improvements and New Features - Mnesia Release Notes","doc":"- Implemented a new option to mnesia:create_table/2 which allows the user to\n  assign 'ets' and 'dets' options not available in mnesia.\n\n  Own Id: OTP-8970","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Mnesia 4.5.1 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-5-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Mnesia Release Notes","doc":"- Fix deadlock in mnesia:del_table_copy/2.\n\n  Own Id: OTP-9689 Aux Id: seq11927","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Improvements and New Features - Mnesia Release Notes","doc":"- Allow schema operations when using different mnesia versions.\n\n  Own Id: OTP-9657 Aux Id: seq11926","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Mnesia 4.5 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-5"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Mnesia Release Notes","doc":"- Fix protocol issues. Mnesia-4.4.19 could not communicate with to older nodes.\n\n  Own Id: OTP-9473\n\n- XML files have been corrected.\n\n  Own Id: OTP-9550 Aux Id: OTP-9541","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Improvements and New Features - Mnesia Release Notes","doc":"- Dump the log even if no transactions have been invoked on local node,\n  otherwise the log will grow forever with decisions from the other nodes who\n  have tables on disk. Thanks Marek Majkowski.\n\n  Own Id: OTP-9551\n\n- Use dedicated api for clear_table, i.e. instead of match_delete use\n  delete_all_objects. Thanks KukHyun Lee.\n\n  Own Id: OTP-9558","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Mnesia 4.4.19 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-4-19"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Mnesia Release Notes","doc":"- Mnesia could crash if mnesia:add_table_index/2 was invoked before the table\n  was loaded on all nodes.\n\n  Own Id: OTP-9285 Aux Id: seq11844\n\n- Add \\{majority, boolean()\\} per-table option.\n\n  With \\{majority, true\\} set for a table, write transactions will abort if they\n  cannot commit to a majority of the nodes that have a copy of the table.\n  Currently, the implementation hooks into the prepare_commit, and forces an\n  asymmetric transaction if the commit set affects any table with the majority\n  flag set. In the commit itself, the transaction will abort if it cannot\n  satisfy the majority requirement for all tables involved in the\n  transaction.(Thanks to Ulf Wiger)\n\n  Own Id: OTP-9304","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Mnesia 4.4.18 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-4-18"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Mnesia Release Notes","doc":"- Call chmod without the \"-f\" flag\n\n  \"-f\" is a non-standard chmod option which at least SGI IRIX and HP UX do not\n  support. As the only effect of the \"-f\" flag is to suppress warning messages,\n  it can be safely omitted. (Thanks to Holger Weiß)\n\n  Own Id: OTP-9170\n\n- Mnesia sometimes failed to update meta-information in large systems, which\n  could cause table content to be inconsistent between nodes.\n\n  Own Id: OTP-9186 Aux Id: seq11728","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Mnesia 4.4.17 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-4-17"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Mnesia Release Notes","doc":"- Calling mnesia:first/1 on empty fragmented table works. Thanks Magnus Henoch.\n\n  Own Id: OTP-9108\n\n- If Mnesia detects that the network is not fully connected during start, Mnesia\n  will not start until all nodes are reachable.\n\n  Own Id: OTP-9115 Aux Id: seq-11728","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Improvements and New Features - Mnesia Release Notes","doc":"- Fix issues reported by dialyzer.\n\n  Own Id: OTP-9107","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Mnesia 4.4.16 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-4-16"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Mnesia Release Notes","doc":"- Sometimes a 'log_header' record was added to tables when invoking\n  mnesia:restore/2 with the option 'recreate_tables'. Thanks Vance Shipley.\n\n  Own Id: OTP-8960","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Improvements and New Features - Mnesia Release Notes","doc":"- Compiler warnings were eliminated.\n\n  Own Id: OTP-8855","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Mnesia 4.4.15 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-4-15"},{"type":"extras","title":"Improvements and New Features - Mnesia Release Notes","doc":"- Eliminated warnings for auto-imported BIF clashes.\n\n  Own Id: OTP-8840","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Mnesia 4.4.14 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-4-14"},{"type":"extras","title":"Improvements and New Features - Mnesia Release Notes","doc":"- Added mnesia:subscribe(activity) contributed by Bernard Duggan.\n\n  Own Id: OTP-8519","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Mnesia 4.4.13 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-4-13"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Mnesia Release Notes","doc":"- Transactions could be left hanging if a node went down when invoking\n  mnesia:sync_transaction/\\[1,2]. Thanks Igor Ribeiro Sucupira.\n\n  Own Id: OTP-8402","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Improvements and New Features - Mnesia Release Notes","doc":"- Igor Ribeiro Sucupira added the option to compress data when copying tables\n  between Mnesia nodes.\n\n  Own Id: OTP-8406","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Mnesia 4.4.12 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-4-12"},{"type":"extras","title":"Improvements and New Features - Mnesia Release Notes","doc":"- The documentation is now built with open source tools (xsltproc and fop) that\n  exists on most platforms. One visible change is that the frames are removed.\n\n  Own Id: OTP-8250","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Mnesia 4.4.11 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-4-11"},{"type":"extras","title":"Improvements and New Features - Mnesia Release Notes","doc":"- Fixed duplicate results with mnesia:index_read() on ordered_set tables.\n  Reported by Sam Bobroff.\n\n  Fixed locking in mnesia:index_read() which now grabs a read table lock to\n  ensure correctness, this may slow down the operation or block other processes\n  trying to reach the same table.\n\n  Calling mnesia:dump_log() could crash mnesia, Reported by Igor Ribeiro\n  Sucupira.\n\n  Own Id: OTP-8074","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Mnesia 4.4.10 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-4-10"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Mnesia Release Notes","doc":"- Mnesia crashed if a qlc query was running inside a transaction when mnesia\n  stopped at another node. Thanks Teemu Antti-Poika.\n\n  Own Id: OTP-7968\n\n- Mnesia could crash when loading local_content tables.\n\n  Own Id: OTP-8002 Aux Id: seq11277","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Improvements and New Features - Mnesia Release Notes","doc":"- Minor (smp) optimizations.\n\n  Own Id: OTP-7928","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Mnesia 4.4.9 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-4-9"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Mnesia Release Notes","doc":"- mnesia:clear_table/1 crashed instead of returning `{aborted,..}` if it was\n  called inside a transaction.\n\n  Own Id: OTP-7911","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Mnesia 4.4.8 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-4-8"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Mnesia Release Notes","doc":"- With bad timing several api functions could return or exit with a bad error\n  message when mnesia was shutting down.\n\n  Own Id: OTP-7753 Aux Id: seq11179\n\n- `mnesia:clear_table/1` cleared all nodes table content even if the table was\n  `local_content` only type.\n\n  Own Id: OTP-7835","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Mnesia 4.4.7 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-4-7"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Mnesia Release Notes","doc":"- Disallowed match patterns ('\\_', and '$n') as argument to\n  `mnesia:delete_object/1` and friends.\n\n  Own Id: OTP-7524","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Improvements and New Features - Mnesia Release Notes","doc":"- Introduced a few new functions in Mnesia: `mnesia:read/2`, `mnesia:first/3`,\n  `mnesia:last/3`, `mnesia:prev/4`, `mnesia:next/4`, `mnesia_frag:first/1`,\n  `mnesia_frag:last/1`, `mnesia_frag:prev/2`, `mnesia_frag:next/2`.\n\n  Own Id: OTP-7625","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Mnesia 4.4.6 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-4-6"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Mnesia Release Notes","doc":"- `mnesia:restore/2` aborted if a `EXIT` message appeared in the client message\n  queue.\n\n  Own Id: OTP-7585 Aux Id: seq11046","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Mnesia 4.4.5 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-4-5"},{"type":"extras","title":"Improvements and New Features - Mnesia Release Notes","doc":"- mnesia:clear_table/1 does not require that all replicas of the table are\n  available anymore.\n\n  Own Id: OTP-7466 Aux Id: seq11015","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Mnesia 4.4.4 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-4-4"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Mnesia Release Notes","doc":"- Mnesia did not garbage collect transaction decisions on disk based nodes if no\n  transactions where made on the local node.\n\n  Own Id: OTP-7419","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Mnesia 4.4.3 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-4-3"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Mnesia Release Notes","doc":"- Table referred to by foreign key did not have node_pool properly cleaned up\n  when a node was removed from the schema. Thanks Paul Mineiro.\n\n  Own Id: OTP-7340\n\n- Mnesia crashed and generated a core dump if a schema_transaction was running\n  when mnesia stopped.\n\n  Own Id: OTP-7378 Aux Id: seq10964","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Improvements and New Features - Mnesia Release Notes","doc":"- It is now possible to delete a db node even when other disk resident nodes are\n  down. Thanks Paul Mineiro.\n\n  Own Id: OTP-7383","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Mnesia 4.4.2 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-4-2"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Mnesia Release Notes","doc":"- Sticky locks could lead to hanging transactions.\n\n  Own Id: OTP-7205 Aux Id: seq10793\n\n- `mnesia:snmp_get_next_index/2` didn't work with partial index keys. Argument\n  checking is now done according to documentation, in functions\n  `mnesia:snmp_get_row/2`, `mnesia:snmp_get_mnesia_key/2` and\n  `mnesia:snmp_get_next_index/2`. These functions now require that `RowIndex` is\n  a list.\n\n  \\*** POTENTIAL INCOMPATIBILITY \\***\n\n  Own Id: OTP-7208","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Mnesia 4.4.1 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-4-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Mnesia Release Notes","doc":"- Snmp index tables was not initialized correctly in `mnesia-4.4`.\n\n  Own Id: OTP-7170 Aux Id: seq10870","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Known Bugs and Problems - Mnesia Release Notes","doc":"- Rearranging fragmented tables is an O(N^2) operation.\n\n  Own Id: OTP-6300","ref":"notes.html#known-bugs-and-problems"},{"type":"extras","title":"Mnesia 4.4 - Mnesia Release Notes","doc":"","ref":"notes.html#mnesia-4-4"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Mnesia Release Notes","doc":"- Mnesia ignored the module argument to `mnesia:restore/2`. Thanks Paul Minerio.\n\n  Own Id: OTP-6981","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Improvements and New Features - Mnesia Release Notes","doc":"- Mnesia's snmp operations `snmp_get_row/2`, `snmp_get_next_index/2` and\n  `snmp_get_mnesia_key/2` have been made context aware, i.e. inside a\n  transaction they will compensate for table updates made in earlier in the same\n  transaction. This might cause a performance drop if a lot of updates have been\n  made before the invocation of these functions.\n\n  \\*** POTENTIAL INCOMPATIBILITY \\***\n\n  Own Id: OTP-6856 Aux Id: seq10671\n\n- Introduced erlang:phash/2 as new default for fragmented tables. Already\n  existing tables will continue to use whatever hash function they where using.\n\n  Own Id: OTP-6923\n\n- Introduced `mnesia:is_transaction/0`.\n\n  Own Id: OTP-6995 Aux Id: seq10812","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Known Bugs and Problems - Mnesia Release Notes","doc":"- Rearranging fragmented tables is an O(N^2) operation.\n\n  Own Id: OTP-6300","ref":"notes.html#known-bugs-and-problems"},{"type":"extras","title":"Introduction","doc":"<!--\n%CopyrightBegin%\n\nCopyright Ericsson AB 2023-2024. All Rights Reserved.\n\nLicensed under the Apache License, Version 2.0 (the \"License\");\nyou may not use this file except in compliance with the License.\nYou may obtain a copy of the License at\n\n    http://www.apache.org/licenses/LICENSE-2.0\n\nUnless required by applicable law or agreed to in writing, software\ndistributed under the License is distributed on an \"AS IS\" BASIS,\nWITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.\nSee the License for the specific language governing permissions and\nlimitations under the License.\n\n%CopyrightEnd%\n-->\n# Introduction\n\nThe Mnesia application provides a heavy-duty real-time distributed database.","ref":"mnesia_chap1.html"},{"type":"extras","title":"Scope - Introduction","doc":"This User's Guide describes how to build Mnesia-backed applications, and how to\nintegrate and use the Mnesia database management system with OTP. Programming\nconstructs are described, and numerous programming examples are included to\nillustrate the use of Mnesia.\n\nThis User's Guide is organized as follows:\n\n- [Mnesia](mnesia_overview.md) provides an introduction to Mnesia.\n- [Getting Started](mnesia_chap2.md) introduces Mnesia with an example database.\n  Examples are included on how to start an Erlang session, specify a Mnesia\n  database directory, initialize a database schema, start Mnesia, and create\n  tables. Initial prototyping of record definitions is also discussed.\n- [Build a Mnesia Database](mnesia_chap3.md) more formally describes the steps\n  introduced in the previous section, namely the Mnesia functions that define a\n  database schema, start Mnesia, and create the required tables.\n- [Transactions and Other Access Contexts](mnesia_chap4.md) describes the\n  transactions properties that make Mnesia into a fault-tolerant, real-time\n  distributed database management system. This section also describes the\n  concept of locking to ensure consistency in tables, and \"dirty operations\", or\n  shortcuts, which bypass the transaction system to improve speed and reduce\n  overheads.\n- [Miscellaneous Mnesia Features](mnesia_chap5.md) describes features that\n  enable the construction of more complex database applications. These features\n  include indexing, checkpoints, distribution and fault tolerance, disc-less\n  nodes, replica manipulation, local content tables, concurrency, and\n  object-based programming in Mnesia.\n- [Mnesia System Information](mnesia_chap7.md) describes the files contained in\n  the Mnesia database directory, database configuration data, core and table\n  dumps, as well as the functions used for backup, restore, fallback, and\n  disaster recovery.\n- [Combine Mnesia with SNMP](mnesia_chap8.md) is a short section that outlines\n  the integration between Mnesia and SNMP.\n- [Appendix A: Backup Callback Interface](mnesia_app_a.md) is a program listing\n  of the default implementation of this facility.\n- [Appendix B: Activity Access Callback Interface](mnesia_app_b.md) is a program\n  outlining one possible implementation of this facility.\n- [Appendix C: Fragmented Table Hashing Callback Interface](mnesia_app_c.md) is\n  a program outlining one possible implementation of this facility.","ref":"mnesia_chap1.html#scope"},{"type":"extras","title":"Prerequisites - Introduction","doc":"It is assumed that the reader is familiar with the Erlang programming language,\nsystem development principles, and database management systems.","ref":"mnesia_chap1.html#prerequisites"},{"type":"extras","title":"Overview","doc":"<!--\n%CopyrightBegin%\n\nCopyright Ericsson AB 2023-2024. All Rights Reserved.\n\nLicensed under the Apache License, Version 2.0 (the \"License\");\nyou may not use this file except in compliance with the License.\nYou may obtain a copy of the License at\n\n    http://www.apache.org/licenses/LICENSE-2.0\n\nUnless required by applicable law or agreed to in writing, software\ndistributed under the License is distributed on an \"AS IS\" BASIS,\nWITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.\nSee the License for the specific language governing permissions and\nlimitations under the License.\n\n%CopyrightEnd%\n-->\n# Overview\n\nThe management of data in telecommunications systems has many aspects of which\nsome, but not all, are addressed by traditional Database Management Systems\n(DBMSs). In particular, the high level of fault tolerance required in many\nnonstop systems, combined with requirements on the DBMS to run in the same\naddress space as the applications, have led us to implement a new DBMS, called\nMnesia.\n\nMnesia is implemented in, and tightly coupled to Erlang. It provides the\nfunctionality that is necessary for the implementation of fault-tolerant\ntelecommunications systems.\n\nMnesia is a multiuser distributed DBMS specifically designed for\nindustrial-grade telecommunications applications written in Erlang, which is\nalso the intended target language. Mnesia tries to address all the data\nmanagement issues required for typical telecommunications systems and has a\nnumber of features not normally found in traditional DBMSs.\n\nTelecommunications applications need a mix of a broad range of features\ngenerally not provided by traditional DBMSs. Mnesia is designed to meet\nrequirements such as:\n\n- Fast real-time key-value lookup\n- Complex non-real-time queries (mainly for operation and maintenance tasks)\n- Distributed data (due to the distributed nature of the applications)\n- High fault tolerance\n- Dynamic reconfiguration\n- Complex objects\n\nMnesia addresses the typical data management issues required for\ntelecommunications applications which sets it apart from most other DBMSs. It\ncombines many concepts found in traditional DBMSs, such as transactions and\nqueries, with concepts found in data management systems for telecommunications\napplications such as:\n\n- Fast real-time operations\n- Configurable replication for fault tolerance\n- Dynamic reconfiguration without service disruption\n\nMnesia is also unique due to its tight coupling to Erlang. It almost turns\nErlang into a database programming language, which yields many benefits. The\nforemost is that the impedance mismatch between the data format used by the DBMS\nand the data format used by the programming language, which is used to\nmanipulate the data, completely disappears.","ref":"mnesia_overview.html"},{"type":"extras","title":"The Mnesia Database Management System - Overview","doc":"","ref":"mnesia_overview.html#the-mnesia-database-management-system"},{"type":"extras","title":"Features - Overview","doc":"Mnesia has the following features that combine to produce a fault-tolerant\ndistributed database management system (DBMS) written in Erlang:\n\n- Database schema can be dynamically reconfigured at runtime.\n- Tables can be declared to have properties such as location, replication, and\n  persistence.\n- Tables can be moved or replicated to several nodes to improve fault tolerance.\n  Other nodes in the system can still access the tables to read, write, and\n  delete records.\n- Table locations are transparent to the programmer. Programs address table\n  names and the system itself keeps track of table locations.\n- Transactions can be distributed and multiple operations can be executed within\n  a single transaction.\n- Multiple transactions can run concurrently and their execution is fully\n  synchronized by Mnesia, ensuring that no two processes manipulate the same\n  data simultaneously.\n- Transactions can be assigned the property of being executed on all nodes in\n  the system, or on none.\n- Transactions can be bypassed using dirty operations, which reduce overheads\n  and run fast.\n\nAll of the above features are described in detail in the coming sections.","ref":"mnesia_overview.html#features"},{"type":"extras","title":"Query List Comprehension - Overview","doc":"Query List Comprehension (QLC) can be used with Mnesia to produce specialized\nfunctions that enhance its operational ability. QLC has its own documentation as\npart of the OTP documentation set. The main QLC advantages when used with Mnesia\nare:\n\n- QLC can optimize the query compiler for Mnesia, essentially making the system\n  more efficient.\n- QLC can be used as a database programming language for Mnesia. It includes a\n  notation called list comprehensions which can be used to execute complex\n  database queries over a set of tables.\n\nFor more information about QLC, please see the `m:qlc` manual page in STDLIB.","ref":"mnesia_overview.html#query-list-comprehension"},{"type":"extras","title":"When to Use Mnesia - Overview","doc":"Mnesia is a great fit for applications that:\n\n- Need to replicate data.\n- Perform complex data queries.\n- Need to use atomic transactions to safely update several records\n  simultaneously.\n- Require soft real-time characteristics.\n\nMnesia is not as appropriate for applications that:\n\n- Process plain text or binary data files.\n- Merely need a lookup dictionary that can be stored on disc. Such applications\n  may use the standard library module `dets`, which is a disc-based version of\n  the `ets` module. For more information about `dets`, please see the `m:dets`\n  manual page in STDLIB.\n- Need disc logging facilities. Such applications may use the module `disk_log`.\n  For more information about `disk_log`, please see the `m:disk_log` manual page\n  in Kernel.\n- Require hard real-time characteristics.","ref":"mnesia_overview.html#when-to-use-mnesia"},{"type":"extras","title":"Getting Started","doc":"<!--\n%CopyrightBegin%\n\nCopyright Ericsson AB 2023-2024. All Rights Reserved.\n\nLicensed under the Apache License, Version 2.0 (the \"License\");\nyou may not use this file except in compliance with the License.\nYou may obtain a copy of the License at\n\n    http://www.apache.org/licenses/LICENSE-2.0\n\nUnless required by applicable law or agreed to in writing, software\ndistributed under the License is distributed on an \"AS IS\" BASIS,\nWITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.\nSee the License for the specific language governing permissions and\nlimitations under the License.\n\n%CopyrightEnd%\n-->\n# Getting Started\n\n[](){: #getting_started }\n\nThis section introduces `Mnesia` with an example database. This example is\nreferenced in the following sections, where the example is modified to\nillustrate various program constructs. This section illustrates the following\nmandatory procedures through examples:\n\n- Starting the Erlang session.\n- Specifying the `Mnesia` directory where the database is to be stored.\n- Initializing a new database schema with an attribute that specifies on which\n  node, or nodes, that database is to operate.\n- Starting `Mnesia`.\n- Creating and populating the database tables.","ref":"mnesia_chap2.html"},{"type":"extras","title":"Starting Mnesia for the First Time - Getting Started","doc":"This section provides a simplified demonstration of a `Mnesia` system startup.\nThe dialogue from the Erlang shell is as follows:\n\n```erlang\nunix> erl -mnesia dir '\"/tmp/funky\"'\nErlang (BEAM) emulator version 4.9\n\nEshell V4.9  (abort with ^G)\n1>\n1> mnesia:create_schema([node()]).\nok\n2> mnesia:start().\nok\n3> mnesia:create_table(funky, []).\n{atomic,ok}\n4> mnesia:info().\n---> Processes holding locks <---\n---> Processes waiting for locks <---\n---> Pending (remote) transactions <---\n---> Active (local) transactions <---\n---> Uncertain transactions <---\n---> Active tables <---\nfunky          : with 0 records occupying 269 words of mem\nschema         : with 2 records occupying 353 words of mem\n===> System info in version \"1.0\", debug level = none <===\nopt_disc. Directory \"/tmp/funky\" is used.\nuse fall-back at restart = false\nrunning db nodes = [nonode@nohost]\nstopped db nodes = []\nremote           = []\nram_copies       = [funky]\ndisc_copies      = [schema]\ndisc_only_copies = []\n[{nonode@nohost,disc_copies}] = [schema]\n[{nonode@nohost,ram_copies}] = [funky]\n1 transactions committed, 0 aborted, 0 restarted, 1 logged to disc\n0 held locks, 0 in queue; 0 local transactions, 0 remote\n0 transactions waits for other nodes: []\nok\n```\n\nIn this example, the following actions are performed:\n\n- _Step 1:_ The Erlang system is started from the UNIX prompt with a flag\n  `-mnesia dir '\"/tmp/funky\"'`, which indicates in which directory to store the\n  data.\n- _Step 2:_ A new empty schema is initialized on the local node by evaluating\n  [`mnesia:create_schema([node()])`](`mnesia:create_schema/1`). The schema\n  contains information about the database in general. This is explained in\n  detail later.\n- _Step 3:_ The DBMS is started by evaluating\n  [`mnesia:start()`](`mnesia:start/0`).\n- _Step 4:_ A first table is created, called `funky`, by evaluating the\n  expression [`mnesia:create_table(funky, [])`](`mnesia:create_table/2`). The\n  table is given default properties.\n- _Step 5:_ [`mnesia:info()`](`mnesia:info/0`) is evaluated to display information\n  on the terminal about the status of the database.","ref":"mnesia_chap2.html#starting-mnesia-for-the-first-time"},{"type":"extras","title":"Example - Getting Started","doc":"A `Mnesia` database is organized as a set of tables. Each table is populated\nwith instances (Erlang records). A table has also a number of properties, such\nas location and persistence.","ref":"mnesia_chap2.html#example"},{"type":"extras","title":"Database - Getting Started","doc":"This example shows how to create a database called `Company` and the\nrelationships shown in the following diagram:\n\n```mermaid\n---\ntitle: Company Entity-Relation Diagram\n---\n\nerDiagram\n    Dept {\n        atom id\n        string name\n    }\n\n    Employee {\n        int emp_no\n        string name\n        int salary\n        atom sex\n        int phone\n        tuple room_no\n\n    }\n\n    Project {\n        atom name\n        int number\n    }\n\n    Dept ||--|| Employee: Manager\n    Employee }|--|| Dept: At_dep\n    Employee }|--|{ Project: in_proj\n```\n\n\nThe database model is as follows:\n\n- There are three entities: department, employee, and project.\n- There are three relationships between these entities:\n\n  1. A department is managed by an employee, hence the `manager` relationship.\n  1. An employee works at a department, hence the `at_dep` relationship.\n  1. Each employee works on a number of projects, hence the `in_proj`\n     relationship.","ref":"mnesia_chap2.html#database"},{"type":"extras","title":"Defining Structure and Content - Getting Started","doc":"First the record definitions are entered into a text file named `company.hrl`.\nThis file defines the following structure for the example database:\n\n```erlang\n-record(employee, {emp_no,\n                   name,\n                   salary,\n                   sex,\n                   phone,\n                   room_no}).\n\n-record(dept, {id,\n               name}).\n\n-record(project, {name,\n                  number}).\n\n\n-record(manager, {emp,\n                  dept}).\n\n-record(at_dep, {emp,\n                 dept_id}).\n\n-record(in_proj, {emp,\n                  proj_name}).\n```\n\nThe structure defines six tables in the database. In `Mnesia`, the function\n[`mnesia:create_table(Name, ArgList)`](`mnesia:create_table/2`) creates tables.\n`Name` is the table name.\n\n> #### Note {: .info }\n>\n> The current version of `Mnesia` does not require that the name of the table is\n> the same as the record name, see\n> [Record Names versus Table Names.](mnesia_chap4.md#recordnames_tablenames).\n\nFor example, the table for employees is created with the function\n`mnesia:create_table(employee, [{attributes, record_info(fields, employee)}])`.\nThe table name `employee` matches the name for records specified in `ArgList`.\nThe expression `record_info(fields, RecordName)` is processed by the Erlang\npreprocessor and evaluates to a list containing the names of the different\nfields for a record.","ref":"mnesia_chap2.html#defining-structure-and-content"},{"type":"extras","title":"Program - Getting Started","doc":"The following shell interaction starts `Mnesia` and initializes the schema for\nthe `Company` database:\n\n```erlang\n% erl -mnesia dir '\"/ldisc/scratch/Mnesia.Company\"'\nErlang (BEAM) emulator version 4.9\n\nEshell V4.9  (abort with ^G)\n1> mnesia:create_schema([node()]).\nok\n2> mnesia:start().\nok\n```\n\nThe following program module creates and populates previously defined tables:\n\n```erlang\n-include_lib(\"stdlib/include/qlc.hrl\").\n-include(\"company.hrl\").\n\ninit() ->\n    mnesia:create_table(employee,\n                        [{attributes, record_info(fields, employee)}]),\n    mnesia:create_table(dept,\n                        [{attributes, record_info(fields, dept)}]),\n    mnesia:create_table(project,\n                        [{attributes, record_info(fields, project)}]),\n    mnesia:create_table(manager, [{type, bag},\n                                  {attributes, record_info(fields, manager)}]),\n    mnesia:create_table(at_dep,\n                         [{attributes, record_info(fields, at_dep)}]),\n    mnesia:create_table(in_proj, [{type, bag},\n                                  {attributes, record_info(fields, in_proj)}]).\n```","ref":"mnesia_chap2.html#program"},{"type":"extras","title":"Program Explained - Getting Started","doc":"The following commands and functions are used to initiate the `Company`\ndatabase:\n\n- `% erl -mnesia dir '\"/ldisc/scratch/Mnesia.Company\"'`. This is a UNIX\n  command-line entry that starts the Erlang system. The flag `-mnesia dir Dir`\n  specifies the location of the database directory. The system responds and\n  waits for further input with the prompt `1>`.\n- [`mnesia:create_schema([node()])`](`mnesia:create_schema/1`). This function has\n  the format `mnesia:create_schema(DiscNodeList)` and initiates a new schema. In\n  this example, a non-distributed system using only one node is created. Schemas\n  are fully explained in [Define a Schema](mnesia_chap3.md#def_schema).\n- [`mnesia:start()`](`mnesia:start/0`). This function starts `Mnesia` and is fully\n  explained in [Start Mnesia](mnesia_chap3.md#start_mnesia).\n\nContinuing the dialogue with the Erlang shell produces the following:\n\n```erlang\n3> company:init().\n{atomic,ok}\n4> mnesia:info().\n---> Processes holding locks <---\n---> Processes waiting for locks <---\n---> Pending (remote) transactions <---\n---> Active (local) transactions <---\n---> Uncertain transactions <---\n---> Active tables <---\nin_proj        : with 0 records occuping 269 words of mem\nat_dep         : with 0 records occuping 269 words of mem\nmanager        : with 0 records occuping 269 words of mem\nproject        : with 0 records occuping 269 words of mem\ndept           : with 0 records occuping 269 words of mem\nemployee       : with 0 records occuping 269 words of mem\nschema         : with 7 records occuping 571 words of mem\n===> System info in version \"1.0\", debug level = none <===\nopt_disc. Directory \"/ldisc/scratch/Mnesia.Company\" is used.\nuse fall-back at restart = false\nrunning db nodes = [nonode@nohost]\nstopped db nodes = []\nremote           = []\nram_copies       =\n    [at_dep,dept,employee,in_proj,manager,project]\ndisc_copies      = [schema]\ndisc_only_copies = []\n[{nonode@nohost,disc_copies}] = [schema]\n[{nonode@nohost,ram_copies}] =\n    [employee,dept,project,manager,at_dep,in_proj]\n6 transactions committed, 0 aborted, 0 restarted, 6 logged to disc\n0 held locks, 0 in queue; 0 local transactions, 0 remote\n0 transactions waits for other nodes: []\nok\n```\n\nA set of tables is created. The function\n[`mnesia:create_table(Name, ArgList)`](`mnesia:create_table/2`) creates the\nrequired database tables. The options available with `ArgList` are explained in\n[Create New Tables](mnesia_chap3.md#create_tables).\n\nThe function `company:init/0` creates the tables. Two tables are of type `bag`.\nThis is the `manager` relation as well the `in_proj` relation. This is\ninterpreted as: an employee can be manager over several departments, and an\nemployee can participate in several projects. However, the `at_dep` relation is\n`set`, as an employee can only work in one department. In this data model, there\nare examples of relations that are 1-to-1 (`set`) and 1-to-many (`bag`).\n\n[`mnesia:info()`](`mnesia:info/0`) now indicates that a database has seven local\ntables, where six are the user-defined tables and one is the schema. Six\ntransactions have been committed, as six successful transactions were run when\ncreating the tables.\n\nTo write a function that inserts an employee record into the database, there\nmust be an `at_dep` record and a set of `in_proj` records inserted. Examine the\nfollowing code used to complete this action:\n\n```erlang\ninsert_emp(Emp, DeptId, ProjNames) ->\n    Ename = Emp#employee.name,\n    Fun = fun() ->\n                  mnesia:write(Emp),\n                  AtDep = #at_dep{emp = Ename, dept_id = DeptId},\n                  mnesia:write(AtDep),\n                  mk_projs(Ename, ProjNames)\n          end,\n    mnesia:transaction(Fun).\n\n\nmk_projs(Ename, [ProjName|Tail]) ->\n    mnesia:write(#in_proj{emp = Ename, proj_name = ProjName}),\n    mk_projs(Ename, Tail);\nmk_projs(_, []) -> ok.\n```\n\n- The `insert_emp/3` arguments are as follows:\n\n  1. `Emp` is an employee record.\n  1. `DeptId` is the identity of the department where the employee works.\n  1. `ProjNames` is a list of the names of the projects where the employee\n     works.\n\nThe function `insert_emp/3` creates a Functional Object (Fun). `Fun` is passed\nas a single argument to the function\n[`mnesia:transaction(Fun)`](`mnesia:transaction/1`). This means that `Fun` is run\nas a transaction with the following properties:\n\n- A `Fun` either succeeds or fails.\n- Code that manipulates the same data records can be run concurrently without\n  the different processes interfering with each other.\n\nThe function can be used as follows:\n\n```erlang\nEmp = #employee{emp_no = 104732,\n                name = klacke,\n                salary = 7,\n                sex = male,\n                phone = 98108,\n                room_no = {221, 015}},\ninsert_emp(Emp, 'B/SFR', [Erlang, mnesia, otp]).\n```\n\n> #### Note {: .info }\n>\n> For information about Funs, see \"Fun Expressions\" in section\n> `Erlang Reference Manual` in System Documentation..","ref":"mnesia_chap2.html#program-explained"},{"type":"extras","title":"Initial Database Content - Getting Started","doc":"After the insertion of the employee named `klacke`, the database has the\nfollowing records:\n\n| emp_no | name   | salary | sex  | phone | room_no      |\n| ------ | ------ | ------ | ---- | ----- | ------------ |\n| 104732 | klacke | 7      | male | 98108 | \\{221, 015\\} |\n\n{: #table2_1 }\n\n_Table: employee Database Record_\n\nThis `employee` record has the Erlang record/tuple representation\n`{employee, 104732, klacke, 7, male, 98108, {221, 015}}`.\n\n| emp    | dept_name |\n| ------ | --------- |\n| klacke | B/SFR     |\n\n{: #table2_2 }\n\n_Table: at_dep Database Record_\n\nThis `at_dep` record has the Erlang tuple representation\n`{at_dep, klacke, 'B/SFR'}`.\n\n| emp    | proj_name |\n| ------ | --------- |\n| klacke | Erlang    |\n| klacke | otp       |\n| klacke | mnesia    |\n\n{: #table3_3 }\n\n_Table: in_proj Database Record_\n\nThis `in_proj` record has the Erlang tuple representation\n`{in_proj, klacke, 'Erlang', klacke, 'otp', klacke, 'mnesia'}`.\n\nThere is no difference between rows in a table and `Mnesia` records. Both\nconcepts are the same and are used interchangeably throughout this User's Guide.\n\nA `Mnesia` table is populated by `Mnesia` records. For example, the tuple\n`{boss, klacke, bjarne}` is a record. The second element in this tuple is the\nkey. To identify a table uniquely, both the key and the table name is needed.\nThe term Object Identifier (OID) is sometimes used for the arity two tuple\n`{Tab, Key}`. The OID for the record `{boss, klacke, bjarne}` is the arity two\ntuple `{boss, klacke}`. The first element of the tuple is the type of the record\nand the second element is the key. An OID can lead to zero, one, or more records\ndepending on whether the table type is `set` or `bag`.\n\nThe record `{boss, klacke, bjarne}` can also be inserted. This record contains\nan implicit reference to another employee that does not yet exist in the\ndatabase. `Mnesia` does not enforce this.","ref":"mnesia_chap2.html#initial-database-content"},{"type":"extras","title":"Adding Records and Relationships to Database - Getting Started","doc":"After adding more records to the `Company` database, the result can be the\nfollowing records:\n\n`employees`:\n\n```erlang\n{employee, 104465, \"Johnson Torbjorn\",   1, male,  99184, {242,038}}.\n{employee, 107912, \"Carlsson Tuula\",     2, female,94556, {242,056}}.\n{employee, 114872, \"Dacker Bjarne\",      3, male,  99415, {221,035}}.\n{employee, 104531, \"Nilsson Hans\",       3, male,  99495, {222,026}}.\n{employee, 104659, \"Tornkvist Torbjorn\", 2, male,  99514, {222,022}}.\n{employee, 104732, \"Wikstrom Claes\",     2, male,  99586, {221,015}}.\n{employee, 117716, \"Fedoriw Anna\",       1, female,99143, {221,031}}.\n{employee, 115018, \"Mattsson Hakan\",     3, male,  99251, {203,348}}.\n```\n\n`dept`:\n\n```erlang\n{dept, 'B/SF',  \"Open Telecom Platform\"}.\n{dept, 'B/SFP', \"OTP - Product Development\"}.\n{dept, 'B/SFR', \"Computer Science Laboratory\"}.\n```\n\n`projects`:\n\n```erlang\n%% projects\n{project, erlang, 1}.\n{project, otp, 2}.\n{project, beam, 3}.\n{project, mnesia, 5}.\n{project, wolf, 6}.\n{project, documentation, 7}.\n{project, www, 8}.\n```\n\nThese three tables, `employees`, `dept`, and `projects`, are made up of real\nrecords. The following database content is stored in the tables and is built on\nrelationships. These tables are `manager`, `at_dep`, and `in_proj`.\n\n`manager`:\n\n```erlang\n{manager, 104465, 'B/SF'}.\n{manager, 104465, 'B/SFP'}.\n{manager, 114872, 'B/SFR'}.\n```\n\n`at_dep`:\n\n```erlang\n{at_dep, 104465, 'B/SF'}.\n{at_dep, 107912, 'B/SF'}.\n{at_dep, 114872, 'B/SFR'}.\n{at_dep, 104531, 'B/SFR'}.\n{at_dep, 104659, 'B/SFR'}.\n{at_dep, 104732, 'B/SFR'}.\n{at_dep, 117716, 'B/SFP'}.\n{at_dep, 115018, 'B/SFP'}.\n```\n\n`in_proj`:\n\n```erlang\n{in_proj, 104465, otp}.\n{in_proj, 107912, otp}.\n{in_proj, 114872, otp}.\n{in_proj, 104531, otp}.\n{in_proj, 104531, mnesia}.\n{in_proj, 104545, wolf}.\n{in_proj, 104659, otp}.\n{in_proj, 104659, wolf}.\n{in_proj, 104732, otp}.\n{in_proj, 104732, mnesia}.\n{in_proj, 104732, erlang}.\n{in_proj, 117716, otp}.\n{in_proj, 117716, documentation}.\n{in_proj, 115018, otp}.\n{in_proj, 115018, mnesia}.\n```\n\nThe room number is an attribute of the employee record. This is a structured\nattribute that consists of a tuple. The first element of the tuple identifies a\ncorridor, and the second element identifies the room in that corridor. An\nalternative is to represent this as a record `-record(room, {corr, no}).`\ninstead of an anonymous tuple representation.\n\nThe `Company` database is now initialized and contains data.","ref":"mnesia_chap2.html#adding-records-and-relationships-to-database"},{"type":"extras","title":"Writing Queries - Getting Started","doc":"Retrieving data from DBMS is usually to be done with the functions\n`mnesia:read/3` or [`mnesia:read/1`](`mnesia:read/2`). The following function\nraises the salary:\n\n```erlang\nraise(Eno, Raise) ->\n    F = fun() ->\n                [E] = mnesia:read(employee, Eno, write),\n                Salary = E#employee.salary + Raise,\n                New = E#employee{salary = Salary},\n                mnesia:write(New)\n        end,\n    mnesia:transaction(F).\n```\n\nSince it is desired to update the record using the function `mnesia:write/1`\nafter the salary has been increased, a write lock (third argument to `read`) is\nacquired when the record from the table is read.\n\nTo read the values from the table directly is not always possible. It can be\nneeded to search one or more tables to get the wanted data, and this is done by\nwriting database queries. Queries are always more expensive operations than\ndirect lookups done with `mnesia:read/1`. Therefore, avoid queries in\nperformance-critical code.\n\nTwo methods are available for writing database queries:\n\n- `Mnesia` functions\n- QLC\n\n#### Using Mnesia Functions\n\nThe following function extracts the names of the female employees stored in the\ndatabase:\n\n```erlang\nmnesia:select(employee, [{#employee{sex = female, name = '$1', _ = '_'},[], ['$1']}]).\n```\n\n`select` must always run within an activity, such as a transaction. The\nfollowing function can be constructed to call from the shell:\n\n```erlang\nall_females() ->\n    F = fun() ->\n\t\tFemale = #employee{sex = female, name = '$1', _ = '_'},\n\t\tmnesia:select(employee, [{Female, [], ['$1']}])\n        end,\n    mnesia:transaction(F).\n```\n\nThe `select` expression matches all entries in table employee with the field\n`sex` set to `female`.\n\nThis function can be called from the shell as follows:\n\n```erlang\n(klacke@gin)1> company:all_females().\n{atomic, [\"Carlsson Tuula\", \"Fedoriw Anna\"]}\n```\n\nFor a description of `select` and its syntax, see\n[Pattern Matching](mnesia_chap4.md#matching).\n\n#### Using QLC\n\nThis section contains simple introductory examples only. For a full description\nof the QLC query language, see the `m:qlc` manual page in `STDLIB`.\n\nUsing QLC can be more expensive than using `Mnesia` functions directly but\noffers a nice syntax.\n\nThe following function extracts a list of female employees from the database:\n\n```erlang\nQ = qlc:q([E#employee.name || E <- mnesia:table(employee),\n                              E#employee.sex == female]),\nqlc:e(Q),\n```\n\nAccessing `Mnesia` tables from a QLC list comprehension must always be done\nwithin a transaction. Consider the following function:\n\n```erlang\nfemales() ->\n    F = fun() ->\n\t\tQ = qlc:q([E#employee.name || E <- mnesia:table(employee),\n\t\t\t\t\t      E#employee.sex == female]),\n\t\tqlc:e(Q)\n\tend,\n    mnesia:transaction(F).\n```\n\nThis function can be called from the shell as follows:\n\n```erlang\n(klacke@gin)1> company:females().\n{atomic, [\"Carlsson Tuula\", \"Fedoriw Anna\"]}\n```\n\nIn traditional relational database terminology, this operation is called a\nselection, followed by a projection.\n\nThe previous list comprehension expression contains a number of syntactical\nelements:\n\n- The first `[` bracket is read as \"build the list\".\n- The `||` \"such that\" and the arrow `<-` is read as \"taken from\".\n\nHence, the previous list comprehension demonstrates the formation of the list\n`E#employee.name` such that `E` is taken from the table of employees, and\nattribute `sex` of each record is equal to the atom `female`.\n\nThe whole list comprehension must be given to the function `qlc:q/1`.\n\nList comprehensions with low-level `Mnesia` functions can be combined in the\nsame transaction. To raise the salary of all female employees, execute the\nfollowing:\n\n```erlang\nraise_females(Amount) ->\n    F = fun() ->\n                Q = qlc:q([E || E <- mnesia:table(employee),\n                                E#employee.sex == female]),\n\t\tFs = qlc:e(Q),\n                over_write(Fs, Amount)\n        end,\n    mnesia:transaction(F).\n\nover_write([E|Tail], Amount) ->\n    Salary = E#employee.salary + Amount,\n    New = E#employee{salary = Salary},\n    mnesia:write(New),\n    1 + over_write(Tail, Amount);\nover_write([], _) ->\n    0.\n```\n\nThe function `raise_females/1` returns the tuple `{atomic, Number}`, where\n`Number` is the number of female employees who received a salary increase. If an\nerror occurs, the value `{aborted, Reason}` is returned, and `Mnesia` guarantees\nthat the salary is not raised for any employee.\n\n_Example:_\n\n```erlang\n33> company:raise_females(33).\n{atomic,2}\n```","ref":"mnesia_chap2.html#writing-queries"},{"type":"extras","title":"Build a Mnesia Database","doc":"<!--\n%CopyrightBegin%\n\nCopyright Ericsson AB 2023-2024. All Rights Reserved.\n\nLicensed under the Apache License, Version 2.0 (the \"License\");\nyou may not use this file except in compliance with the License.\nYou may obtain a copy of the License at\n\n    http://www.apache.org/licenses/LICENSE-2.0\n\nUnless required by applicable law or agreed to in writing, software\ndistributed under the License is distributed on an \"AS IS\" BASIS,\nWITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.\nSee the License for the specific language governing permissions and\nlimitations under the License.\n\n%CopyrightEnd%\n-->\n# Build a Mnesia Database\n\nThis section describes the basic steps when designing a `Mnesia` database and\nthe programming constructs that make different solutions available to the\nprogrammer. The following topics are included:\n\n- Define a schema\n- Data model\n- Start `Mnesia`\n- Create tables\n\n[](){: #def_schema }","ref":"mnesia_chap3.html"},{"type":"extras","title":"Define a Schema - Build a Mnesia Database","doc":"The configuration of a `Mnesia` system is described in a schema. The schema is a\nspecial table that includes information such as the table names and the storage\ntype of each table (that is, whether a table is to be stored in RAM, on disc, or\non both, as well as its location).\n\nUnlike data tables, information in schema tables can only be accessed and\nmodified by using the schema-related functions described in this section.\n\n`Mnesia` has various functions for defining the database schema. Tables can be\nmoved or deleted, and the table layout can be reconfigured.\n\nAn important aspect of these functions is that the system can access a table\nwhile it is being reconfigured. For example, it is possible to move a table and\nsimultaneously perform write operations to the same table. This feature is\nessential for applications that require continuous service.\n\nThis section describes the functions available for schema management, all which\nreturn either of the following tuples:\n\n- `{atomic, ok}` if successful\n- `{aborted, Reason}` if unsuccessful","ref":"mnesia_chap3.html#define-a-schema"},{"type":"extras","title":"Schema Functions - Build a Mnesia Database","doc":"The schema functions are as follows:\n\n- [`mnesia:create_schema(NodeList)`](`mnesia:create_schema/1`) initializes a new,\n  empty schema. This is a mandatory requirement before `Mnesia` can be started.\n  `Mnesia` is a truly distributed DBMS and the schema is a system table that is\n  replicated on all nodes in a `Mnesia` system. This function fails if a schema\n  is already present on any of the nodes in `NodeList`. The function requires\n  `Mnesia` to be stopped on the all `db_nodes` contained in parameter\n  `NodeList`. Applications call this function only once, as it is usually a\n  one-time activity to initialize a new database.\n- [`mnesia:delete_schema(DiscNodeList)`](`mnesia:delete_schema/1`) erases any old\n  schemas on the nodes in `DiscNodeList`. It also removes all old tables\n  together with all data. This function requires `Mnesia` to be stopped on all\n  `db_nodes`.\n- [`mnesia:delete_table(Tab)`](`mnesia:delete_table/1`) permanently deletes all\n  replicas of table `Tab`.\n- [`mnesia:clear_table(Tab)`](`mnesia:clear_table/1`) permanently deletes all\n  entries in table `Tab`.\n- [`mnesia:move_table_copy(Tab, From, To)`](`mnesia:move_table_copy/3`) moves the\n  copy of table `Tab` from node `From` to node `To`. The table storage type\n  `{type}` is preserved, so if a RAM table is moved from one node to another, it\n  remains a RAM table on the new node. Other transactions can still perform read\n  and write operation to the table while it is being moved.\n- [`mnesia:add_table_copy(Tab, Node, Type)`](`mnesia:add_table_copy/3`) creates a\n  replica of table `Tab` at node `Node`. Argument `Type` must be either of the\n  atoms `ram_copies`, `disc_copies`, or `disc_only_copies`. If you add a copy of\n  the system table `schema` to a node, you want the `Mnesia` schema to reside\n  there as well. This action extends the set of nodes that comprise this\n  particular `Mnesia` system.\n- [`mnesia:del_table_copy(Tab, Node)`](`mnesia:del_table_copy/2`) deletes the\n  replica of table `Tab` at node `Node`. When the last replica of a table is\n  removed, the table is deleted.\n- [`mnesia:transform_table(Tab, Fun, NewAttributeList, NewRecordName)`](`mnesia:transform_table/4`)\n  changes the format on all records in table `Tab`. It applies argument `Fun` to\n  all records in the table. `Fun` must be a function that takes a record of the\n  old type, and returns the record of the new type. The table key must not be\n  changed.\n\n  _Example:_\n\n  ```erlang\n  -record(old, {key, val}).\n  -record(new, {key, val, extra}).\n\n  Transformer =\n     fun(X) when record(X, old) ->\n        #new{key = X#old.key,\n             val = X#old.val,\n             extra = 42}\n     end,\n  {atomic, ok} = mnesia:transform_table(foo, Transformer,\n                                        record_info(fields, new),\n                                        new),\n  ```\n\n  Argument `Fun` can also be the atom `ignore`, which indicates that only the\n  metadata about the table is updated. Use of `ignore` is not recommended (as it\n  creates inconsistencies between the metadata and the actual data) but it is\n  included as a possibility for the user do to an own (offline) transform.\n\n- [`mnesia:change_table_copy_type(Tab, Node, ToType)`](`mnesia:change_table_copy_type/3`)\n  changes the storage type of a table. For example, a RAM table is changed to a\n  `disc_table` at the node specified as `Node`.","ref":"mnesia_chap3.html#schema-functions"},{"type":"extras","title":"Data Model - Build a Mnesia Database","doc":"The data model employed by `Mnesia` is an extended relational data model. Data\nis organized as a set of tables and relations between different data records can\nbe modeled as more tables describing the relationships. Each table contains\ninstances of Erlang records. The records are represented as Erlang tuples.\n\nEach Object Identifier (OID) is made up of a table name and a key. For example,\nif an employee record is represented by the tuple\n`{employee, 104732, klacke, 7, male, 98108, {221, 015}}`, this record has an\nOID, which is the tuple `{employee, 104732}`.\n\nThus, each table is made up of records, where the first element is a record name\nand the second element of the table is a key, which identifies the particular\nrecord in that table. The combination of the table name and a key is an arity\ntwo tuple `{Tab, Key}` called the OID. For more information about the\nrelationship between the record name and the table name, see\n[Record Names versus Table Names](mnesia_chap4.md#recordnames_tablenames).\n\nWhat makes the `Mnesia` data model an extended relational model is the ability\nto store arbitrary Erlang terms in the attribute fields. One attribute value\ncan, for example, be a whole tree of OIDs leading to other terms in other\ntables. This type of record is difficult to model in traditional relational\nDBMSs.\n\n[](){: #start_mnesia }","ref":"mnesia_chap3.html#data-model"},{"type":"extras","title":"Start Mnesia - Build a Mnesia Database","doc":"Before starting `Mnesia`, the following must be done:\n\n- An empty schema must be initialized on all the participating nodes.\n- The Erlang system must be started.\n- Nodes with disc database schema must be defined and implemented with the\n  function [`mnesia:create_schema(NodeList)`](`mnesia:create_schema/1`).\n\nWhen running a distributed system with two or more participating nodes, the\nfunction [`mnesia:start()`](`mnesia:start/0`) must be executed on each\nparticipating node. This would typically be part of the boot script in an\nembedded environment. In a test environment or an interactive environment,\n`mnesia:start()` can also be used either from the Erlang shell or another\nprogram.","ref":"mnesia_chap3.html#start-mnesia"},{"type":"extras","title":"Initialize a Schema and Start Mnesia - Build a Mnesia Database","doc":"Let us use the example database `Company`, described in\n[Getting Started](mnesia_chap2.md#getting_started) to illustrate how to run a\ndatabase on two separate nodes, called `a@gin` and `b@skeppet`. Each of these\nnodes must have a `Mnesia` directory and an initialized schema before `Mnesia`\ncan be started. There are two ways to specify the `Mnesia` directory to be used:\n\n- Specify the `Mnesia` directory by providing an application parameter either\n  when starting the Erlang shell or in the application script. Previously, the\n  following example was used to create the directory for the `Company` database:\n\n  ```text\n  % erl -mnesia dir '\"/ldisc/scratch/Mnesia.Company\"'\n  ```\n\n- If no command-line flag is entered, the `Mnesia` directory becomes the current\n  working directory on the node where the Erlang shell is started.\n\nTo start the `Company` database and get it running on the two specified nodes,\nenter the following commands:\n\n1. On the node `a@gin`:\n\n```text\n gin % erl -sname a  -mnesia dir '\"/ldisc/scratch/Mnesia.company\"'\n```\n\n1. On the node `b@skeppet`:\n\n```text\nskeppet % erl -sname b -mnesia dir '\"/ldisc/scratch/Mnesia.company\"'\n```\n\n1. On one of the two nodes:\n\n```erlang\n(a@gin)1> mnesia:create_schema([a@gin, b@skeppet]).\n```\n\n1. The function [`mnesia:start()`](`mnesia:start/0`) is called on both nodes.\n1. To initialize the database, execute the following code on one of the two\n   nodes:\n\n```erlang\ndist_init() ->\n    mnesia:create_table(employee,\n                         [{ram_copies, [a@gin, b@skeppet]},\n                          {attributes, record_info(fields,\n\t\t\t\t\t\t   employee)}]),\n    mnesia:create_table(dept,\n                         [{ram_copies, [a@gin, b@skeppet]},\n                          {attributes, record_info(fields, dept)}]),\n    mnesia:create_table(project,\n                         [{ram_copies, [a@gin, b@skeppet]},\n                          {attributes, record_info(fields, project)}]),\n    mnesia:create_table(manager, [{type, bag},\n                                  {ram_copies, [a@gin, b@skeppet]},\n                                  {attributes, record_info(fields,\n\t\t\t\t\t\t\t   manager)}]),\n    mnesia:create_table(at_dep,\n                         [{ram_copies, [a@gin, b@skeppet]},\n                          {attributes, record_info(fields, at_dep)}]),\n    mnesia:create_table(in_proj,\n                        [{type, bag},\n                         {ram_copies, [a@gin, b@skeppet]},\n                         {attributes, record_info(fields, in_proj)}]).\n```\n\nAs illustrated, the two directories reside on different nodes, because\n`/ldisc/scratch` (the \"local\" disc) exists on the two different nodes.\n\nBy executing these commands, two Erlang nodes are configured to run the\n`Company` database, and therefore, initialize the database. This is required\nonly once when setting up. The next time the system is started,\n[`mnesia:start()`](`mnesia:start/0`) is called on both nodes, to initialize the\nsystem from disc.\n\nIn a system of `Mnesia` nodes, every node is aware of the current location of\nall tables. In this example, data is replicated on both nodes and functions that\nmanipulate the data in the tables can be executed on either of the two nodes.\nCode that manipulate `Mnesia` data behaves identically regardless of where the\ndata resides.\n\nThe function [`mnesia:stop()`](`mnesia:stop/0`) stops `Mnesia` on the node where\nthe function is executed. The functions `mnesia:start/0` and `mnesia:stop/0`\nwork on the \"local\" `Mnesia` system. No functions start or stop a set of nodes.","ref":"mnesia_chap3.html#initialize-a-schema-and-start-mnesia"},{"type":"extras","title":"Startup Procedure - Build a Mnesia Database","doc":"Start `Mnesia` by calling the following function:\n\n```erlang\nmnesia:start().\n```\n\nThis function initiates the DBMS locally.\n\nThe choice of configuration alters the location and load order of the tables.\nThe alternatives are as follows:\n\n1. Tables that are only stored locally are initialized from the local `Mnesia`\n   directory.\n1. Replicated tables that reside locally as well as somewhere else are either\n   initiated from disc or by copying the entire table from the other node,\n   depending on which of the different replicas are the most recent. `Mnesia`\n   determines which of the tables are the most recent.\n1. Tables that reside on remote nodes are available to other nodes as soon as\n   they are loaded.\n\nTable initialization is asynchronous. The function call\n[`mnesia:start()`](`mnesia:start/0`) returns the atom `ok` and then starts to\ninitialize the different tables. Depending on the size of the database, this can\ntake some time, and the application programmer must wait for the tables that the\napplication needs before they can be used. This is achieved by using the\nfunction [`mnesia:wait_for_tables(TabList, Timeout)`](`mnesia:wait_for_tables/2`),\nwhich suspends the caller until all tables specified in `TabList` are properly\ninitiated.\n\nA problem can arise if a replicated table on one node is initiated, but `Mnesia`\ndeduces that another (remote) replica is more recent than the replica existing\non the local node, and the initialization procedure does not proceed. In this\nsituation, a call to `mnesia:wait_for_tables/2`, suspends the caller until the\nremote node has initialized the table from its local disc and the node has\ncopied the table over the network to the local node.\n\nHowever, this procedure can be time-consuming, the shortcut function\n[`mnesia:force_load_table(Tab)`](`mnesia:force_load_table/1`) loads all the tables\nfrom disc at a faster rate. The function forces tables to be loaded from disc\nregardless of the network situation.\n\nThus, it can be assumed that if an application wants to use tables `a` and `b`,\nthe application must perform some action similar to following before it can use\nthe tables:\n\n```erlang\ncase mnesia:wait_for_tables([a, b], 20000) of\n  {timeout, RemainingTabs} ->\n    panic(RemainingTabs);\n  ok ->\n    synced\nend.\n```\n\n> #### Warning {: .warning }\n>\n> When tables are forcefully loaded from the local disc, all operations that\n> were performed on the replicated table while the local node was down, and the\n> remote replica was alive, are lost. This can cause the database to become\n> inconsistent.\n\nIf the startup procedure fails, the function [`mnesia:start()`](`mnesia:start/0`)\nreturns the cryptic tuple\n`{error,{shutdown, {mnesia_sup,start_link,[normal,[]]}}}`. To get more\ninformation about the start failure, use command-line arguments\n`-boot start_sasl` as argument to the `erl` script.\n\n[](){: #create_tables }","ref":"mnesia_chap3.html#startup-procedure"},{"type":"extras","title":"Create Tables - Build a Mnesia Database","doc":"The function [`mnesia:create_table(Name, ArgList)`](`mnesia:create_table/2`)\ncreates tables. When executing this function, it returns one of the following\nresponses:\n\n- `{atomic, ok}` if the function executes successfully\n- `{aborted, Reason}` if the function fails\n\nThe function arguments are as follows:\n\n- `Name` is the name of the table. It is usually the same name as the name of\n  the records that constitute the table. For details, see `record_name`.\n- `ArgList` is a list of `{Key,Value}` tuples. The following arguments are\n  valid:\n\n  - `{type, Type}`, where `Type` must be either of the atoms `set`,\n    `ordered_set`, or `bag`. Default is `set`.\n\n    Notice that currently `ordered_set` is not supported for `disc_only_copies`\n    tables.\n\n    A table of type `set` or `ordered_set` has either zero or one record per\n    key, whereas a table of type `bag` can have an arbitrary number of records\n    per key. The key for each record is always the first attribute of the\n    record.\n\n    The following example illustrates the difference between type `set` and\n    `bag`:\n\n    ```erlang\n     f() ->\n        F = fun() ->\n              mnesia:write({foo, 1, 2}),\n              mnesia:write({foo, 1, 3}),\n              mnesia:read({foo, 1})\n            end,\n        mnesia:transaction(F).\n    ```\n\n    This transaction returns the list `[{foo,1,3}]` if table `foo` is of type\n    `set`. However, the list `[{foo,1,2}, {foo,1,3}]` is returned if the table\n    is of type `bag`.\n\n    `Mnesia` tables can never contain duplicates of the same record in the same\n    table. Duplicate records have attributes with the same contents and key.\n\n  - `{disc_copies, NodeList}`, where `NodeList` is a list of the nodes where\n    this table is to reside on disc.\n\n    Write operations to a table replica of type `disc_copies` write data to the\n    disc copy and to the RAM copy of the table.\n\n    It is possible to have a replicated table of type `disc_copies` on one node,\n    and the same table stored as a different type on another node. Default is\n    `[]`. This arrangement is desirable if the following operational\n    characteristics are required:\n\n    1. Read operations must be fast and performed in RAM.\n    1. All write operations must be written to persistent storage.\n\n    A write operation on a `disc_copies` table replica is performed in two\n    steps. First the write operation is appended to a log file, then the actual\n    operation is performed in RAM.\n\n  - `{ram_copies, NodeList}`, where `NodeList` is a list of the nodes where this\n    table is stored in RAM. Default is `[node()]`. If the default value is used\n    to create a table, it is located on the local node only.\n\n    Table replicas of type `ram_copies` can be dumped to disc with the function\n    [`mnesia:dump_tables(TabList)`](`mnesia:dump_tables/1`).\n\n  - `{disc_only_copies, NodeList}`. These table replicas are stored on disc only\n    and are therefore slower to access. However, a disc-only replica consumes\n    less memory than a table replica of the other two storage types.\n  - `{index, AttributeNameList}`, where `AttributeNameList` is a list of atoms\n    specifying the names of the attributes `Mnesia` is to build and maintain. An\n    index table exists for every element in the list. The first field of a\n    `Mnesia` record is the key and thus need no extra index.\n\n    The first field of a record is the second element of the tuple, which is the\n    representation of the record.\n\n  - `{snmp, SnmpStruct}`. `SnmpStruct` is described in the\n    [SNMP](`e:snmp:index.html`) User's Guide. Basically, if this attribute is\n    present in `ArgList` of `mnesia:create_table/2`, the table is immediately\n    accessible the SNMP.\n\n    It is easy to design applications that use SNMP to manipulate and control\n    the system. `Mnesia` provides a direct mapping between the logical tables\n    that make up an SNMP control application and the physical data that makes up\n    a `Mnesia` table. The default value is `[]`.\n\n  - `{local_content, true}`. When an application needs a table whose contents is\n    to be locally unique on each node, `local_content` tables can be used. The\n    name of the table is known to all `Mnesia` nodes, but its contents is unique\n    for each node. Access to this type of table must be done locally.\n  - `{attributes, AtomList}` is a list of the attribute names for the records\n    that are supposed to populate the table. Default is the list `[key, val]`.\n    The table must at least have one extra attribute besides the key. When\n    accessing single attributes in a record, it is not recommended to hard code\n    the attribute names as atoms. Use the construct\n    `record_info(fields, record_name)` instead.\n\n    The expression `record_info(fields, record_name)` is processed by the Erlang\n    preprocessor and returns a list of the record field names. With the record\n    definition `-record(foo, {x,y,z}).`, the expression\n    `record_info(fields,foo)` is expanded to the list `[x,y,z]`. It is therefore\n    possible for you to provide the attribute names or to use the\n    `record_info/2` notation.\n\n    It is recommended to use the `record_info/2` notation, as it becomes easier\n    to maintain the program and the program becomes more robust with regards to\n    future record changes.\n\n  - `{record_name, Atom}` specifies the common name of all records stored in the\n    table. All records stored in the table must have this name as their first\n    element. `record_name` defaults to the name of the table. For more\n    information, see\n    [Record Names versus Table Names](mnesia_chap4.md#recordnames_tablenames).\n\nAs an example, consider the following record definition:\n\n```erlang\n-record(funky, {x, y}).\n```\n\nThe following call would create a table that is replicated on two nodes, has an\nextra index on attribute `y`, and is of type `bag`.\n\n```erlang\nmnesia:create_table(funky, [{disc_copies, [N1, N2]}, {index, [y]},\n                            {type, bag}, {attributes, record_info(fields, funky)}]).\n```\n\nWhereas a call to the following default code values would return a table with a\nRAM copy on the local node, no extra indexes, and the attributes defaulted to\nthe list `[key,val]`.\n\n```erlang\nmnesia:create_table(stuff, [])\n```","ref":"mnesia_chap3.html#create-tables"},{"type":"extras","title":"Transactions and Other Access Contexts","doc":"<!--\n%CopyrightBegin%\n\nCopyright Ericsson AB 2023-2024. All Rights Reserved.\n\nLicensed under the Apache License, Version 2.0 (the \"License\");\nyou may not use this file except in compliance with the License.\nYou may obtain a copy of the License at\n\n    http://www.apache.org/licenses/LICENSE-2.0\n\nUnless required by applicable law or agreed to in writing, software\ndistributed under the License is distributed on an \"AS IS\" BASIS,\nWITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.\nSee the License for the specific language governing permissions and\nlimitations under the License.\n\n%CopyrightEnd%\n-->\n# Transactions and Other Access Contexts\n\nThis section describes the `Mnesia` transaction system and the transaction\nproperties that make `Mnesia` a fault-tolerant, distributed Database Management\nSystem (DBMS).\n\nThis section also describes the locking functions, including table locks and\nsticky locks, as well as alternative functions that bypass the transaction\nsystem in favor of improved speed and reduced overhead. These functions are\ncalled \"dirty operations\". The use of nested transactions is also described. The\nfollowing topics are included:\n\n- Transaction properties, which include atomicity, consistency, isolation, and\n  durability\n- Locking\n- Dirty operations\n- Record names versus table names\n- Activity concept and various access contexts\n- Nested transactions\n- Pattern matching\n- Iteration\n\n[](){: #trans_prop }","ref":"mnesia_chap4.html"},{"type":"extras","title":"Transaction Properties - Transactions and Other Access Contexts","doc":"Transactions are important when designing fault-tolerant, distributed systems. A\n`Mnesia` transaction is a mechanism by which a series of database operations can\nbe executed as one functional block. The functional block that is run as a\ntransaction is called a Functional Object (Fun), and this code can read, write,\nand delete `Mnesia` records. The Fun is evaluated as a transaction that either\ncommits or terminates. If a transaction succeeds in executing the Fun, it\nreplicates the action on all nodes involved, or terminates if an error occurs.\n\nThe following example shows a transaction that raises the salary of certain\nemployee numbers:\n\n```erlang\nraise(Eno, Raise) ->\n    F = fun() ->\n                [E] = mnesia:read(employee, Eno, write),\n                Salary = E#employee.salary + Raise,\n                New = E#employee{salary = Salary},\n                mnesia:write(New)\n        end,\n    mnesia:transaction(F).\n```\n\nThe function `raise/2` contains a Fun made up of four code lines. This Fun is\ncalled by the statement `mnesia:transaction(F)` and returns a value.\n\nThe `Mnesia` transaction system facilitates the construction of reliable,\ndistributed systems by providing the following important properties:\n\n- The transaction handler ensures that a Fun, which is placed inside a\n  transaction, does not interfere with operations embedded in other transactions\n  when it executes a series of operations on tables.\n- The transaction handler ensures that either all operations in the transaction\n  are performed successfully on all nodes atomically, or the transaction fails\n  without permanent effect on any node.\n- The `Mnesia` transactions have four important properties, called *A*tomicity,\n  *C*onsistency, *I*solation, and *D*urability (ACID). These properties are\n  described in the following sections.","ref":"mnesia_chap4.html#transaction-properties"},{"type":"extras","title":"Atomicity - Transactions and Other Access Contexts","doc":"Atomicity means that database changes that are executed by a transaction take\neffect on all nodes involved, or on none of the nodes. That is, the transaction\neither succeeds entirely, or it fails entirely.\n\nAtomicity is important when it is needed to write atomically more than one\nrecord in the same transaction. The function `raise/2`, shown in the previous\nexample, writes one record only. The function `insert_emp/3`, shown in the\nprogram listing in [Getting Started](mnesia_chap2.md#getting_started), writes\nthe record `employee` as well as employee relations, such as `at_dep` and\n`in_proj`, into the database. If this latter code is run inside a transaction,\nthe transaction handler ensures that the transaction either succeeds completely,\nor not at all.\n\n`Mnesia` is a distributed DBMS where data can be replicated on several nodes. In\nmany applications, it is important that a series of write operations are\nperformed atomically inside a transaction. The atomicity property ensures that a\ntransaction takes effect on all nodes, or none.","ref":"mnesia_chap4.html#atomicity"},{"type":"extras","title":"Consistency - Transactions and Other Access Contexts","doc":"The consistency property ensures that a transaction always leaves the DBMS in a\nconsistent state. For example, `Mnesia` ensures that no inconsistencies occur if\nErlang, `Mnesia`, or the computer crashes while a write operation is in\nprogress.","ref":"mnesia_chap4.html#consistency"},{"type":"extras","title":"Isolation - Transactions and Other Access Contexts","doc":"The isolation property ensures that transactions that execute on different nodes\nin a network, and access and manipulate the same data records, do not interfere\nwith each other. The isolation property makes it possible to execute the\nfunction `raise/2` concurrently. A classical problem in concurrency control\ntheory is the \"lost update problem\".\n\nThe isolation property is in particular useful if the following circumstances\noccur where an employee (with employee number 123) and two processes (P1 and P2)\nare concurrently trying to raise the salary for the employee:\n\n- _Step 1:_ The initial value of the employees salary is, for example, 5.\n  Process P1 starts to execute, reads the employee record, and adds 2 to the\n  salary.\n- _Step 2:_ Process P1 is for some reason pre-empted and process P2 has the\n  opportunity to run.\n- _Step 3:_ Process P2 reads the record, adds 3 to the salary, and finally\n  writes a new employee record with the salary set to 8.\n- _Step 4:_ Process P1 starts to run again and writes its employee record with\n  salary set to 7, thus effectively overwriting and undoing the work performed\n  by process P2. The update performed by P2 is lost.\n\nA transaction system makes it possible to execute two or more processes\nconcurrently that manipulate the same record. The programmer does not need to\ncheck that the updates are synchronous; this is overseen by the transaction\nhandler. All programs accessing the database through the transaction system can\nbe written as if they had sole access to the data.","ref":"mnesia_chap4.html#isolation"},{"type":"extras","title":"Durability - Transactions and Other Access Contexts","doc":"The durability property ensures that changes made to the DBMS by a transaction\nare permanent. Once a transaction is committed, all changes made to the database\nare durable, that is, they are written safely to disc and do not become\ncorrupted and do not disappear.\n\n> #### Note {: .info }\n>\n> The described durability feature does not entirely apply to situations where\n> `Mnesia` is configured as a \"pure\" primary memory database.","ref":"mnesia_chap4.html#durability"},{"type":"extras","title":"Locking - Transactions and Other Access Contexts","doc":"Different transaction managers employ different strategies to satisfy the\nisolation property. `Mnesia` uses the standard technique of two phase locking.\nThat is, locks are set on records before they are read or written. `Mnesia` uses\nthe following lock types:\n\n- _Read locks_. A read lock is set on one replica of a record before it can be\n  read.\n- _Write locks_. Whenever a transaction writes to a record, write locks are\n  first set on all replicas of that particular record.\n- _Read table locks_. If a transaction traverses an entire table in search for a\n  record that satisfies some particular property, it is most inefficient to set\n  read locks on the records one by one. It is also memory consuming, as the read\n  locks themselves can take up considerable space if the table is large.\n  Therefore, `Mnesia` can set a read lock on an entire table.\n- _Write table locks_. If a transaction writes many records to one table, a\n  write lock can be set on the entire table.\n- _Sticky locks_. These are write locks that stay in place at a node after the\n  transaction that initiated the lock has terminated.\n\n`Mnesia` employs a strategy whereby functions, such as `mnesia:read/1` acquire\nthe necessary locks dynamically as the transactions execute. `Mnesia`\nautomatically sets and releases the locks and the programmer does not need to\ncode these operations.\n\nDeadlocks can occur when concurrent processes set and release locks on the same\nrecords. `Mnesia` employs a \"wait-die\" strategy to resolve these situations. If\n`Mnesia` suspects that a deadlock can occur when a transaction tries to set a\nlock, the transaction is forced to release all its locks and sleep for a while.\nThe Fun in the transaction is evaluated once more.\n\nIt is therefore important that the code inside the Fun given to\n`mnesia:transaction/1` is pure. Some strange results can occur if, for example,\nmessages are sent by the transaction Fun. The following example illustrates this\nsituation:\n\n```erlang\nbad_raise(Eno, Raise) ->\n    F = fun() ->\n                [E] = mnesia:read({employee, Eno}),\n                Salary = E#employee.salary + Raise,\n                New = E#employee{salary = Salary},\n                io:format(\"Trying to write ... ~n\", []),\n                mnesia:write(New)\n        end,\n    mnesia:transaction(F).\n```\n\nThis transaction can write the text `\"Trying to write ... \"` 1000 times to the\nterminal. However, `Mnesia` guarantees that each transaction will eventually\nrun. As a result, `Mnesia` is not only deadlock free, but also livelock free.\n\nThe `Mnesia` programmer cannot prioritize one particular transaction to execute\nbefore other transactions that are waiting to execute. As a result, the `Mnesia`\nDBMS transaction system is not suitable for hard real-time applications.\nHowever, `Mnesia` contains other features that have real-time properties.\n\n`Mnesia` dynamically sets and releases locks as transactions execute. It is\ntherefore dangerous to execute code with transaction side-effects. In\nparticular, a `receive` statement inside a transaction can lead to a situation\nwhere the transaction hangs and never returns, which in turn can cause locks not\nto release. This situation can bring the whole system to a standstill, as other\ntransactions that execute in other processes, or on other nodes, are forced to\nwait for the defective transaction.\n\nIf a transaction terminates abnormally, `Mnesia` automatically releases the\nlocks held by the transaction.\n\nUp to now, examples of a number of functions that can be used inside a\ntransaction have been shown. The following list shows the _simplest_ `Mnesia`\nfunctions that work with transactions. Notice that these functions must be\nembedded in a transaction. If no enclosing transaction (or other enclosing\n`Mnesia` activity) exists, they all fail.\n\n- [`mnesia:transaction(Fun) -> {aborted, Reason} | {atomic, Value}`](`mnesia:transaction/1`)\n  executes one transaction with the functional object `Fun` as the single\n  parameter.\n- [`mnesia:read({Tab, Key}) -> transaction abort | RecordList`](`mnesia:read/1`)\n  reads all records with `Key` as key from table `Tab`. This function has the\n  same semantics regardless of the location of `Table`. If the table is of type\n  `bag`, `read({Tab, Key})` can return an arbitrarily long list. If the table is\n  of type `set`, the list is either of length one or `[]`.\n- [`mnesia:wread({Tab, Key}) -> transaction abort | RecordList`](`mnesia:wread/1`)\n  behaves the same way as the previously listed function `read/1`, except that\n  it acquires a write lock instead of a read lock. To execute a transaction that\n  reads a record, modifies the record, and then writes the record, it is\n  slightly more efficient to set the write lock immediately. When a\n  `mnesia:read/1` is issued, followed by a `mnesia:write/1` the first read lock\n  must be upgraded to a write lock when the write operation is executed.\n- [`mnesia:write(Record) -> transaction abort | ok`](`mnesia:write/1`) writes a\n  record into the database. Argument `Record` is an instance of a record. The\n  function returns `ok`, or terminates the transaction if an error occurs.\n- [`mnesia:delete({Tab, Key}) -> transaction abort | ok`](`mnesia:delete/1`)\n  deletes all records with the given key.\n- [`mnesia:delete_object(Record) -> transaction abort | ok`](`mnesia:delete_object/1`)\n  deletes records with the OID `Record`. Use this function to delete only some\n  records in a table of type `bag`.","ref":"mnesia_chap4.html#locking"},{"type":"extras","title":"Sticky Locks - Transactions and Other Access Contexts","doc":"As previously stated, the locking strategy used by `Mnesia` is to lock one\nrecord when reading a record, and lock all replicas of a record when writing a\nrecord. However, some applications use `Mnesia` mainly for its fault-tolerant\nqualities. These applications can be configured with one node doing all the\nheavy work, and a standby node that is ready to take over if the main node\nfails. Such applications can benefit from using sticky locks instead of the\nnormal locking scheme.\n\nA sticky lock is a lock that stays in place at a node, after the transaction\nthat first acquired the lock has terminated. To illustrate this, assume that the\nfollowing transaction is executed:\n\n```erlang\nF = fun() ->\n      mnesia:write(#foo{a = kalle})\n    end,\nmnesia:transaction(F).\n```\n\nThe `foo` table is replicated on the two nodes `N1` and `N2`.\n\nNormal locking requires the following:\n\n- One network RPC (two messages) to acquire the write lock\n- Three network messages to execute the two-phase commit protocol\n\nIf sticky locks are used, the code must first be changed as follows:\n\n```erlang\nF = fun() ->\n      mnesia:s_write(#foo{a = kalle})\n    end,\nmnesia:transaction(F).\n```\n\nThis code uses the function [`s_write/1`](`mnesia:s_write/1`) instead of the\nfunction [`write/1`](`mnesia:write/1`) The function `s_write/1` sets a sticky lock\ninstead of a normal lock. If the table is not replicated, sticky locks have no\nspecial effect. If the table is replicated, and a sticky lock is set on node\n`N1`, this lock then sticks to node `N1`. The next time you try to set a sticky\nlock on the same record at node `N1`, `Mnesia` detects that the lock is already\nset and do no network operation to acquire the lock.\n\nIt is more efficient to set a local lock than it is to set a networked lock.\nSticky locks can therefore benefit an application that uses a replicated table\nand perform most of the work on only one of the nodes.\n\nIf a record is stuck at node `N1` and you try to set a sticky lock for the\nrecord on node `N2`, the record must be unstuck. This operation is expensive and\nreduces performance. The unsticking is done automatically if you issue\n`s_write/1` requests at `N2`.","ref":"mnesia_chap4.html#sticky-locks"},{"type":"extras","title":"Table Locks - Transactions and Other Access Contexts","doc":"`Mnesia` supports read and write locks on whole tables as a complement to the\nnormal locks on single records. As previously stated, `Mnesia` sets and releases\nlocks automatically, and the programmer does not need to code these operations.\nHowever, transactions that read and write many records in a specific table\nexecute more efficiently if the transaction is started by setting a table lock\non this table. This blocks other concurrent transactions from the table. The\nfollowing two functions are used to set explicit table locks for read and write\noperations:\n\n- [`mnesia:read_lock_table(Tab)`](`mnesia:read_lock_table/1`) sets a read lock on\n  table `Tab`.\n- [`mnesia:write_lock_table(Tab)`](`mnesia:write_lock_table/1`) sets a write lock\n  on table `Tab`.\n\nAlternative syntax for acquisition of table locks is as follows:\n\n```erlang\nmnesia:lock({table, Tab}, read)\nmnesia:lock({table, Tab}, write)\n```\n\nThe matching operations in `Mnesia` can either lock the entire table or only a\nsingle record (when the key is bound in the pattern).","ref":"mnesia_chap4.html#table-locks"},{"type":"extras","title":"Global Locks - Transactions and Other Access Contexts","doc":"Write locks are normally acquired on all nodes where a replica of the table\nresides (and is active). Read locks are acquired on one node (the local one if a\nlocal replica exists).\n\nThe function `mnesia:lock/2` is intended to support table locks (as mentioned\npreviously) but also for situations when locks need to be acquired regardless of\nhow tables have been replicated:\n\n```text\nmnesia:lock({global, GlobalKey, Nodes}, LockKind)\n\nLockKind ::= read | write | ...\n```\n\nThe lock is acquired on `LockItem` on all nodes in the node list.","ref":"mnesia_chap4.html#global-locks"},{"type":"extras","title":"Dirty Operations - Transactions and Other Access Contexts","doc":"In many applications, the overhead of processing a transaction can result in a\nloss of performance. Dirty operation are short cuts that bypass much of the\nprocessing and increase the speed of the transaction.\n\nDirty operation are often useful, for example, in a datagram routing application\nwhere `Mnesia` stores the routing table, and it is time consuming to start a\nwhole transaction every time a packet is received. `Mnesia` has therefore\nfunctions that manipulate tables without using transactions. This alternative to\nprocessing is known as a dirty operation. However, notice the trade-off in\navoiding the overhead of transaction processing:\n\n- The atomicity and the isolation properties of `Mnesia` are lost.\n- The isolation property is compromised, because other Erlang processes, which\n  use transaction to manipulate the data, do not get the benefit of isolation if\n  dirty operations simultaneously are used to read and write records from the\n  same table.\n\nThe major advantage of dirty operations is that they execute much faster than\nequivalent operations that are processed as functional objects within a\ntransaction.\n\nDirty operations are written to disc if they are performed on a table of type\n`disc_copies` or type `disc_only_copies`. `Mnesia` also ensures that all\nreplicas of a table are updated if a dirty write operation is performed on a\ntable.\n\nA dirty operation ensures a certain level of consistency. For example, dirty\noperations cannot return garbled records. Hence, each individual read or write\noperation is performed in an atomic manner.\n\nAll dirty functions execute a call to [`exit({aborted, Reason})`](`exit/1`) on\nfailure. Even if the following functions are executed inside a transaction no\nlocks are acquired. The following functions are available:\n\n- [`mnesia:dirty_read({Tab, Key})`](`mnesia:dirty_read/1`) reads one or more\n  records from `Mnesia`.\n- [`mnesia:dirty_write(Record)`](`mnesia:dirty_write/1`) writes the record\n  `Record`.\n- [`mnesia:dirty_delete({Tab, Key})`](`mnesia:dirty_delete/1`) deletes one or\n  more records with key `Key`.\n- [`mnesia:dirty_delete_object(Record)`](`mnesia:dirty_delete_object/1`) is the\n  dirty operation alternative to the function\n  [`delete_object/1`](`mnesia:delete_object/1`).\n- [`mnesia:dirty_first(Tab)`](`mnesia:dirty_first/1`) returns the \"first\" key in\n  table `Tab`.\n\n  Records in `set` or `bag` tables are not sorted. However, there is a record\n  order that is unknown to the user. This means that a table can be traversed by\n  this function with the function `mnesia:dirty_next/2`.\n\n  If there are no records in the table, this function returns the atom\n  `'$end_of_table'`. It is not recommended to use this atom as the key for any\n  user records.\n\n- [`mnesia:dirty_next(Tab, Key)`](`mnesia:dirty_next/2`) returns the \"next\" key in\n  table `Tab`. This function makes it possible to traverse a table and perform\n  some operation on all records in the table. When the end of the table is\n  reached, the special key `'$end_of_table'` is returned. Otherwise, the\n  function returns a key that can be used to read the actual record.\n\n  The behavior is undefined if any process performs a write operation on the\n  table while traversing the table with the function\n  [`dirty_next/2`](`mnesia:dirty_next/2`) This is because `write` operations on a\n  `Mnesia` table can lead to internal reorganizations of the table itself. This\n  is an implementation detail, but remember that the dirty functions are\n  low-level functions.\n\n- [`mnesia:dirty_last(Tab)`](`mnesia:dirty_last/1`) works exactly like\n  `mnesia:dirty_first/1` but returns the last object in Erlang term order for\n  the table type `ordered_set`. For all other table types,\n  `mnesia:dirty_first/1` and `mnesia:dirty_last/1` are synonyms.\n- [`mnesia:dirty_prev(Tab, Key)`](`mnesia:dirty_prev/2`) works exactly like\n  `mnesia:dirty_next/2` but returns the previous object in Erlang term order for\n  the table type `ordered_set`. For all other table types, `mnesia:dirty_next/2`\n  and `mnesia:dirty_prev/2` are synonyms.\n- The behavior of this function is undefined if the table is written on while\n  being traversed. The function\n  [`mnesia:read_lock_table(Tab)`](`mnesia:read_lock_table/1`) can be used to\n  ensure that no transaction-protected writes are performed during the\n  iteration.\n- [`mnesia:dirty_update_counter({Tab, Key}, Val)`](`mnesia:dirty_update_counter/2`).\n  Counters are positive integers with a value greater than or equal to zero.\n  Updating a counter adds `Val` and the counter where `Val` is a positive or\n  negative integer.\n\n  `Mnesia` has no special counter records. However, records of the form\n  `{TabName, Key, Integer}` can be used as counters, and can be persistent.\n\n  Transaction-protected updates of counter records are not possible.\n\n  There are two significant differences when using this function instead of\n  reading the record, performing the arithmetic, and writing the record:\n\n  1. It is much more efficient.\n  1. The function [`dirty_update_counter/2`](`mnesia:dirty_update_counter/2`) is\n     performed as an atomic operation although it is not protected by a\n     transaction. Therefore no table update is lost if two processes\n     simultaneously execute the function `dirty_update_counter/2`.\n\n- [`mnesia:dirty_match_object(Pat)`](`mnesia:dirty_match_object/2`) is the dirty\n  equivalent of `mnesia:match_object/1`.\n- [`mnesia:dirty_select(Tab, Pat)`](`mnesia:dirty_select/2`) is the dirty\n  equivalent of `mnesia:select/2`.\n- [`mnesia:dirty_index_match_object(Pat, Pos)`](`mnesia:dirty_index_match_object/2`)\n  is the dirty equivalent of `mnesia:index_match_object/2`.\n- [`mnesia:dirty_index_read(Tab, SecondaryKey, Pos)`](`mnesia:dirty_index_read/3`)\n  is the dirty equivalent of `mnesia:index_read/3`.\n- [`mnesia:dirty_all_keys(Tab)`](`mnesia:dirty_all_keys/1`) is the dirty\n  equivalent of `mnesia:all_keys/1`.\n\n[](){: #recordnames_tablenames }","ref":"mnesia_chap4.html#dirty-operations"},{"type":"extras","title":"Record Names versus Table Names - Transactions and Other Access Contexts","doc":"In `Mnesia`, all records in a table must have the same name. All the records\nmust be instances of the same record type. The record name, however, does not\nnecessarily have to be the same as the table name, although this is the case in\nmost of the examples in this User's Guide. If a table is created without\nproperty `record_name`, the following code ensures that all records in the\ntables have the same name as the table:\n\n```erlang\nmnesia:create_table(subscriber, [])\n```\n\nHowever, if the table is created with an explicit record name as argument, as\nshown in the following example, subscriber records can be stored in both of the\ntables regardless of the table names:\n\n```erlang\nTabDef = [{record_name, subscriber}],\nmnesia:create_table(my_subscriber, TabDef),\nmnesia:create_table(your_subscriber, TabDef).\n```\n\nTo access such tables, simplified access functions (as described earlier) cannot\nbe used. For example, writing a subscriber record into a table requires the\nfunction `mnesia:write/3` instead of the simplified functions `mnesia:write/1`\nand `mnesia:s_write/1`:\n\n```erlang\nmnesia:write(subscriber, #subscriber{}, write)\nmnesia:write(my_subscriber, #subscriber{}, sticky_write)\nmnesia:write(your_subscriber, #subscriber{}, write)\n```\n\nThe following simple code illustrates the relationship between the simplified\naccess functions used in most of the examples and their more flexible\ncounterparts:\n\n```erlang\nmnesia:dirty_write(Record) ->\n  Tab = element(1, Record),\n  mnesia:dirty_write(Tab, Record).\n\nmnesia:dirty_delete({Tab, Key}) ->\n  mnesia:dirty_delete(Tab, Key).\n\nmnesia:dirty_delete_object(Record) ->\n  Tab = element(1, Record),\n  mnesia:dirty_delete_object(Tab, Record)\n\nmnesia:dirty_update_counter({Tab, Key}, Incr) ->\n  mnesia:dirty_update_counter(Tab, Key, Incr).\n\nmnesia:dirty_read({Tab, Key}) ->\n  Tab = element(1, Record),\n  mnesia:dirty_read(Tab, Key).\n\nmnesia:dirty_match_object(Pattern) ->\n  Tab = element(1, Pattern),\n  mnesia:dirty_match_object(Tab, Pattern).\n\nmnesia:dirty_index_match_object(Pattern, Attr)\n  Tab = element(1, Pattern),\n  mnesia:dirty_index_match_object(Tab, Pattern, Attr).\n\nmnesia:write(Record) ->\n  Tab = element(1, Record),\n  mnesia:write(Tab, Record, write).\n\nmnesia:s_write(Record) ->\n  Tab = element(1, Record),\n  mnesia:write(Tab, Record, sticky_write).\n\nmnesia:delete({Tab, Key}) ->\n  mnesia:delete(Tab, Key, write).\n\nmnesia:s_delete({Tab, Key}) ->\n  mnesia:delete(Tab, Key, sticky_write).\n\nmnesia:delete_object(Record) ->\n  Tab = element(1, Record),\n  mnesia:delete_object(Tab, Record, write).\n\nmnesia:s_delete_object(Record) ->\n  Tab = element(1, Record),\n  mnesia:delete_object(Tab, Record, sticky_write).\n\nmnesia:read({Tab, Key}) ->\n  mnesia:read(Tab, Key, read).\n\nmnesia:wread({Tab, Key}) ->\n  mnesia:read(Tab, Key, write).\n\nmnesia:match_object(Pattern) ->\n  Tab = element(1, Pattern),\n  mnesia:match_object(Tab, Pattern, read).\n\nmnesia:index_match_object(Pattern, Attr) ->\n  Tab = element(1, Pattern),\n  mnesia:index_match_object(Tab, Pattern, Attr, read).\n```","ref":"mnesia_chap4.html#record-names-versus-table-names"},{"type":"extras","title":"Activity Concept and Various Access Contexts - Transactions and Other Access Contexts","doc":"As previously described, a Functional Object (Fun) performing table access\noperations, as listed here, can be passed on as arguments to the function\n[`mnesia:transaction/1,2,3`](`mnesia:transaction/1`):\n\n- [`mnesia:write/3` (`write/1`, `s_write/1`)](`mnesia:write/3`)\n- `mnesia:delete/3` (`mnesia:delete/1`, `mnesia:s_delete/1`)\n- `mnesia:delete_object/3` (`mnesia:delete_object/1`,\n  `mnesia:s_delete_object/1`)\n- `mnesia:read/3` (`mnesia:read/1`, `mnesia:wread/1`)\n- [`mnesia:match_object/2`](`mnesia:match_object/3`) (`mnesia:match_object/1`)\n- [`mnesia:select/3`](`mnesia:select/2`) (`mnesia:select/2`)\n- `mnesia:foldl/3` (`mnesia:foldl/4`, `mnesia:foldr/3`, `mnesia:foldr/4`)\n- `mnesia:all_keys/1`\n- `mnesia:index_match_object/4` (`mnesia:index_match_object/2`)\n- `mnesia:index_read/3`\n- `mnesia:lock/2` (`mnesia:read_lock_table/1`, `mnesia:write_lock_table/1`)\n- `mnesia:table_info/2`\n\nThese functions are performed in a transaction context involving mechanisms,\nsuch as locking, logging, replication, checkpoints, subscriptions, and commit\nprotocols. However, the same function can also be evaluated in other activity\ncontexts.\n\nThe following activity access contexts are currently supported:\n\n- `transaction`\n- `sync_transaction`\n- `async_dirty`\n- `sync_dirty`\n- `ets`\n\nBy passing the same \"fun\" as argument to the function\n[`mnesia:sync_transaction(Fun [, Args])`](`mnesia:sync_transaction/1`) it is\nperformed in synced transaction context. Synced transactions wait until all\nactive replicas has committed the transaction (to disc) before returning from\nthe `mnesia:sync_transaction` call. Using `sync_transaction` is useful in the\nfollowing cases:\n\n- When an application executes on several nodes and wants to be sure that the\n  update is performed on the remote nodes before a remote process is spawned or\n  a message is sent to a remote process.\n- When a combining transaction writes with \"dirty reads\", that is, the functions\n  `dirty_match_object`, `dirty_read`, `dirty_index_read`, `dirty_select`, and so\n  on.\n- When an application performs frequent or voluminous updates that can overload\n  `Mnesia` on other nodes.\n\nBy passing the same \"fun\" as argument to the function [`mnesia:async_dirty(Fun\n[, Args])`](`mnesia:async_dirty/1`), it is performed in dirty context. The\nfunction calls are mapped to the corresponding dirty functions. This still\ninvolves logging, replication, and subscriptions but no locking, local\ntransaction storage, or commit protocols are involved. Checkpoint retainers are\nupdated but updated \"dirty\". Thus, they are updated asynchronously. The\nfunctions wait for the operation to be performed on one node but not the others.\nIf the table resides locally, no waiting occurs.\n\nBy passing the same \"fun\" as an argument to the function [`mnesia:sync_dirty(Fun\n[, Args])`](`mnesia:sync_dirty/1`), it is performed in almost the same context\nas the function [`mnesia:async_dirty/1,2`](`mnesia:async_dirty/1`). The difference\nis that the operations are performed synchronously. The caller waits for the\nupdates to be performed on all active replicas. Using `mnesia:sync_dirty/1,2` is\nuseful in the following cases:\n\n- When an application executes on several nodes and wants to be sure that the\n  update is performed on the remote nodes before a remote process is spawned or\n  a message is sent to a remote process.\n- When an application performs frequent or voluminous updates that can overload\n  `Mnesia` on the nodes.\n\nTo check if your code is executed within a transaction, use the function\n`mnesia:is_transaction/0`. It returns `true` when called inside a transaction\ncontext, otherwise `false`.\n\n`Mnesia` tables with storage type `RAM_copies` and `disc_copies` are implemented\ninternally as `ets` tables. Applications can access the these tables directly.\nThis is only recommended if all options have been weighed and the possible\noutcomes are understood. By passing the earlier mentioned \"fun\" to the function\n[`mnesia:ets(Fun [, Args])`](`mnesia:ets/1`), it is performed but in a raw\ncontext. The operations are performed directly on the local `ets` tables,\nassuming that the local storage type is `RAM_copies` and that the table is not\nreplicated on other nodes.\n\nSubscriptions are not triggered and no checkpoints are updated, but this\noperation is blindingly fast. Disc resident tables are not to be updated with\nthe `ets` function, as the disc is not updated.\n\nThe Fun can also be passed as an argument to the function\n[`mnesia:activity/2,3,4`](`mnesia:activity/2`), which enables use of customized\nactivity access callback modules. It can either be obtained directly by stating\nthe module name as argument, or implicitly by use of configuration parameter\n`access_module`. A customized callback module can be used for several purposes,\nsuch as providing triggers, integrity constraints, runtime statistics, or\nvirtual tables.\n\nThe callback module does not have to access real `Mnesia` tables, it is free to\ndo whatever it wants as long as the callback interface is fulfilled.\n\n[Appendix B, Activity Access Callback Interface](mnesia_app_b.md) provides the\nsource code, `mnesia_frag.erl`, for one alternative implementation. The\ncontext-sensitive function `mnesia:table_info/2` can be used to provide virtual\ninformation about a table. One use of this is to perform `QLC` queries within an\nactivity context with a customized callback module. By providing table\ninformation about table indexes and other `QLC` requirements, `QLC` can be used\nas a generic query language to access virtual tables.\n\nQLC queries can be performed in all these activity contexts (`transaction`,\n`sync_transaction`, `async_dirty`, `sync_dirty`, and `ets`). The `ets` activity\nonly works if the table has no indexes.\n\n> #### Note {: .info }\n>\n> The function `mnesia:dirty_*` always executes with `async_dirty` semantics\n> regardless of which activity access contexts that are started. It can even\n> start contexts without any enclosing activity access context.","ref":"mnesia_chap4.html#activity-concept-and-various-access-contexts"},{"type":"extras","title":"Nested Transactions - Transactions and Other Access Contexts","doc":"Transactions can be nested in an arbitrary fashion. A child transaction must run\nin the same process as its parent. When a child transaction terminates, the\ncaller of the child transaction gets return value `{aborted, Reason}` and any\nwork performed by the child is erased. If a child transaction commits, the\nrecords written by the child are propagated to the parent.\n\nNo locks are released when child transactions terminate. Locks created by a\nsequence of nested transactions are kept until the topmost transaction\nterminates. Furthermore, any update performed by a nested transaction is only\npropagated in such a manner so that the parent of the nested transaction sees\nthe updates. No final commitment is done until the top-level transaction\nterminates. So, although a nested transaction returns `{atomic, Val}`, if the\nenclosing parent transaction terminates, the entire nested operation terminates.\n\nThe ability to have nested transaction with identical semantics as top-level\ntransaction makes it easier to write library functions that manipulate `Mnesia`\ntables.\n\nConsider a function that adds a subscriber to a telephony system:\n\n```erlang\nadd_subscriber(S) ->\n    mnesia:transaction(fun() ->\n        case mnesia:read( ..........\n```\n\nThis function needs to be called as a transaction. Assume that you wish to write\na function that both calls the function `add_subscriber/1` and is in itself\nprotected by the context of a transaction. By calling `add_subscriber/1` from\nwithin another transaction, a nested transaction is created.\n\nAlso, different activity access contexts can be mixed while nesting. However,\nthe dirty ones (`async_dirty`, `sync_dirty`, and `ets`) inherit the transaction\nsemantics if they are called inside a transaction and thus grab locks and use\ntwo or three phase commit.\n\n_Example:_\n\n```erlang\nadd_subscriber(S) ->\n    mnesia:transaction(fun() ->\n       %% Transaction context\n       mnesia:read({some_tab, some_data}),\n       mnesia:sync_dirty(fun() ->\n           %% Still in a transaction context.\n           case mnesia:read( ..) ..end), end).\nadd_subscriber2(S) ->\n    mnesia:sync_dirty(fun() ->\n       %% In dirty context\n       mnesia:read({some_tab, some_data}),\n       mnesia:transaction(fun() ->\n           %% In a transaction context.\n           case mnesia:read( ..) ..end), end).\n```","ref":"mnesia_chap4.html#nested-transactions"},{"type":"extras","title":"Pattern Matching - Transactions and Other Access Contexts","doc":"[](){: #matching }\n\nWhen the function `mnesia:read/3` cannot be used, `Mnesia` provides the\nprogrammer with several functions for matching records against a pattern. The\nmost useful ones are the following:\n\n```erlang\nmnesia:select(Tab, MatchSpecification, LockKind) ->\n    transaction abort | [ObjectList]\nmnesia:select(Tab, MatchSpecification, NObjects, Lock) ->\n    transaction abort | {[Object],Continuation} | '$end_of_table'\nmnesia:select(Cont) ->\n    transaction abort | {[Object],Continuation} | '$end_of_table'\nmnesia:match_object(Tab, Pattern, LockKind) ->\n    transaction abort | RecordList\n```\n\nThese functions match a `Pattern` against all records in table `Tab`. In a\n[`mnesia:select`](`mnesia:select/2`) call, `Pattern` is a part of\n`MatchSpecification` described in the following. It is not necessarily performed\nas an exhaustive search of the entire table. By using indexes and bound values\nin the key of the pattern, the actual work done by the function can be condensed\ninto a few hash lookups. Using `ordered_set` tables can reduce the search space\nif the keys are partially bound.\n\nThe pattern provided to the functions must be a valid record, and the first\nelement of the provided tuple must be the `record_name` of the table. The\nspecial element `'_'` matches any data structure in Erlang (also known as an\nErlang term). The special elements `'$ '` behave as Erlang variables,\nthat is, they match anything, bind the first occurrence, and match the coming\noccurrences of that variable against the bound value.\n\nUse function [`mnesia:table_info(Tab, wild_pattern)`](`mnesia:table_info/2`) to\nobtain a basic pattern, which matches all records in a table, or use the default\nvalue in record creation. Do not make the pattern hard-coded, as this makes the\ncode more vulnerable to future changes of the record definition.\n\n_Example:_\n\n```erlang\nWildpattern = mnesia:table_info(employee, wild_pattern),\n%% Or use\nWildpattern = #employee{_ = '_'},\n```\n\nFor the employee table, the wild pattern looks as follows:\n\n```erlang\n{employee, '_', '_', '_', '_', '_',' _'}.\n```\n\nTo constrain the match, it is needed to replace some of the `'_'` elements. The\ncode for matching out all female employees looks as follows:\n\n```erlang\nPat = #employee{sex = female, _ = '_'},\nF = fun() -> mnesia:match_object(Pat) end,\nFemales = mnesia:transaction(F).\n```\n\nThe match function can also be used to check the equality of different\nattributes. For example, to find all employees with an employee number equal to\ntheir room number:\n\n```erlang\nPat = #employee{emp_no = '$1', room_no = '$1', _ = '_'},\nF = fun() -> mnesia:match_object(Pat) end,\nOdd = mnesia:transaction(F).\n```\n\nThe function `mnesia:match_object/3` lacks some important features that\n[`mnesia:select/3`](`mnesia:select/2`) have. For example, `mnesia:match_object/3`\ncan only return the matching records, and it cannot express constraints other\nthan equality. To find the names of the male employees on the second floor:\n\n```erlang\nMatchHead = #employee{name='$1', sex=male, room_no={'$2', '_'}, _='_'},\nGuard = [{'>=', '$2', 220},{'<', '$2', 230}],\nResult = '$1',\nmnesia:select(employee,[{MatchHead, Guard, [Result]}])\n```\n\nThe function `select` can be used to add more constraints and create output that\ncannot be done with `mnesia:match_object/3`.\n\nThe second argument to `select` is a `MatchSpecification`. A\n`MatchSpecification` is a list of `MatchFunction`s, where each `MatchFunction`\nconsists of a tuple containing `{MatchHead, MatchCondition, MatchBody}`:\n\n- `MatchHead` is the same pattern as used in `mnesia:match_object/3` described\n  earlier.\n- `MatchCondition` is a list of extra constraints applied to each record.\n- `MatchBody` constructs the return values.\n\nFor details about the match specifications, see \"Match Specifications in Erlang\"\nin [ERTS](`e:erts:index.html`) User's Guide. For more information, see the\n`m:ets` and `m:dets` manual pages in `STDLIB`.\n\nThe functions [`select/4`](`mnesia:select/4`) and [`select/1`](`mnesia:select/2`)\nare used to get a limited number of results, where `Continuation` gets the next\nchunk of results. `Mnesia` uses `NObjects` as a recommendation only. Thus, more\nor less results than specified with `NObjects` can be returned in the result\nlist, even the empty list can be returned even if there are more results to\ncollect.\n\n> #### Warning {: .warning }\n>\n> There is a severe performance penalty in using `mnesia:select/1,2,3,4` after\n> any modifying operation is done on that table in the same transaction. That\n> is, avoid using `mnesia:write/1` or `mnesia:delete/1` before `mnesia:select`\n> in the same transaction.\n\nIf the key attribute is bound in a pattern, the match operation is efficient.\nHowever, if the key attribute in a pattern is given as `'_'` or `'$1'`, the\nwhole `employee` table must be searched for records that match. Hence if the\ntable is large, this can become a time-consuming operation, but it can be\nremedied with indexes (see [Indexing](mnesia_chap5.md#indexing)) if the function\n[`mnesia:match_object`](`mnesia:match_object/1`) is used.\n\nQLC queries can also be used to search `Mnesia` tables. By using the function\n[`mnesia:table/1,2`](`mnesia:table/1`) as the generator inside a QLC query, you\nlet the query operate on a `Mnesia` table. `Mnesia`-specific options to\n`mnesia:table/2` are `{lock, Lock}`, `{n_objects, Integer}`, and\n`{traverse, SelMethod}`:\n\n- `lock` specifies whether `Mnesia` is to acquire a read or write lock on the\n  table.\n- `n_objects` specifies how many results are to be returned in each chunk to\n  QLC.\n- `traverse` specifies which function `Mnesia` is to use to traverse the table.\n  Default `select` is used, but by using\n  `{traverse, {select, MatchSpecification}}` as an option to\n  [`mnesia:table/2`](`mnesia:table/1`) the user can specify its own view of the\n  table.\n\nIf no options are specified, a read lock is acquired, 100 results are returned\nin each chunk, and `select` is used to traverse the table, that is:\n\n```erlang\nmnesia:table(Tab) ->\n    mnesia:table(Tab, [{n_objects, 100},{lock, read}, {traverse, select}]).\n```\n\nThe function [`mnesia:all_keys(Tab)`](`mnesia:all_keys/1`) returns all keys in a\ntable.","ref":"mnesia_chap4.html#pattern-matching"},{"type":"extras","title":"Iteration - Transactions and Other Access Contexts","doc":"`Mnesia` provides the following functions that iterate over all the records in a\ntable:\n\n```erlang\nmnesia:foldl(Fun, Acc0, Tab) -> NewAcc | transaction abort\nmnesia:foldr(Fun, Acc0, Tab) -> NewAcc | transaction abort\nmnesia:foldl(Fun, Acc0, Tab, LockType) -> NewAcc | transaction abort\nmnesia:foldr(Fun, Acc0, Tab, LockType) -> NewAcc | transaction abort\n```\n\nThese functions iterate over the `Mnesia` table `Tab` and apply the function\n`Fun` to each record. `Fun` takes two arguments, the first is a record from the\ntable, and the second is the accumulator. `Fun` returns a new accumulator.\n\nThe first time `Fun` is applied, `Acc0` is the second argument. The next time\n`Fun` is called, the return value from the previous call is used as the second\nargument. The term the last call to `Fun` returns is the return value of the\nfunction `mnesia:foldl/3` or `mnesia:foldr/3`.\n\nThe difference between these functions is the order the table is accessed for\n`ordered_set` tables. For other table types the functions are equivalent.\n\n`LockType` specifies what type of lock that is to be acquired for the iteration,\ndefault is `read`. If records are written or deleted during the iteration, a\nwrite lock is to be acquired.\n\nThese functions can be used to find records in a table when it is impossible to\nwrite constraints for the function `mnesia:match_object/3`, or when you want to\nperform some action on certain records.\n\nFor example, finding all the employees who have a salary less than 10 can look\nas follows:\n\n```erlang\nfind_low_salaries() ->\n  Constraint =\n       fun(Emp, Acc) when Emp#employee.salary < 10 ->\n              [Emp | Acc];\n          (_, Acc) ->\n              Acc\n       end,\n  Find = fun() -> mnesia:foldl(Constraint, [], employee) end,\n  mnesia:transaction(Find).\n```\n\nTo raise the salary to 10 for everyone with a salary less than 10 and return the\nsum of all raises:\n\n```erlang\nincrease_low_salaries() ->\n   Increase =\n       fun(Emp, Acc) when Emp#employee.salary < 10 ->\n              OldS = Emp#employee.salary,\n              ok = mnesia:write(Emp#employee{salary = 10}),\n              Acc + 10 - OldS;\n          (_, Acc) ->\n              Acc\n       end,\n  IncLow = fun() -> mnesia:foldl(Increase, 0, employee, write) end,\n  mnesia:transaction(IncLow).\n```\n\nMany nice things can be done with the iterator functions but take some caution\nabout performance and memory use for large tables.\n\nCall these iteration functions on nodes that contain a replica of the table.\nEach call to the function `Fun` access the table and if the table resides on\nanother node it generates much unnecessary network traffic.\n\n`Mnesia` also provides some functions that make it possible for the user to\niterate over the table. The order of the iteration is unspecified if the table\nis not of type `ordered_set`:\n\n```erlang\nmnesia:first(Tab) ->  Key | transaction abort\nmnesia:last(Tab)  ->  Key | transaction abort\nmnesia:next(Tab,Key)  ->  Key | transaction abort\nmnesia:prev(Tab,Key)  ->  Key | transaction abort\nmnesia:snmp_get_next_index(Tab,Index) -> {ok, NextIndex} | endOfTable\n```\n\nThe order of `first`/`last` and `next`/`prev` is only valid for `ordered_set`\ntables, they are synonyms for other tables. When the end of the table is\nreached, the special key `'$end_of_table'` is returned.\n\nIf records are written and deleted during the traversal, use the function\n`mnesia:foldl/3` or `mnesia:foldr/3` with a `write` lock. Or the function\n`mnesia:write_lock_table/1` when using `first` and `next`.\n\nWriting or deleting in transaction context creates a local copy of each modified\nrecord. Thus, modifying each record in a large table uses much memory. `Mnesia`\ncompensates for every written or deleted record during the iteration in a\ntransaction context, which can reduce the performance. If possible, avoid\nwriting or deleting records in the same transaction before iterating over the\ntable.\n\nIn dirty context, that is, `sync_dirty` or `async_dirty`, the modified records\nare not stored in a local copy; instead, each record is updated separately. This\ngenerates much network traffic if the table has a replica on another node and\nhas all the other drawbacks that dirty operations have. Especially for commands\n`mnesia:first/1` and `mnesia:next/2`, the same drawbacks as described previously\nfor `mnesia:dirty_first/1` and `mnesia:dirty_next/2` applies, that is, no\nwriting to the table is to be done during iteration.","ref":"mnesia_chap4.html#iteration"},{"type":"extras","title":"Miscellaneous Mnesia Features","doc":"<!--\n%CopyrightBegin%\n\nCopyright Ericsson AB 2023-2024. All Rights Reserved.\n\nLicensed under the Apache License, Version 2.0 (the \"License\");\nyou may not use this file except in compliance with the License.\nYou may obtain a copy of the License at\n\n    http://www.apache.org/licenses/LICENSE-2.0\n\nUnless required by applicable law or agreed to in writing, software\ndistributed under the License is distributed on an \"AS IS\" BASIS,\nWITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.\nSee the License for the specific language governing permissions and\nlimitations under the License.\n\n%CopyrightEnd%\n-->\n# Miscellaneous Mnesia Features\n\nThe previous sections describe how to get started with `Mnesia` and how to build\na `Mnesia` database. This section describes the more advanced features available\nwhen building a distributed, fault-tolerant `Mnesia` database. The following\ntopics are included:\n\n- Indexing\n- Distribution and fault tolerance\n- Table fragmentation\n- Local content tables\n- Disc-less nodes\n- More about schema management\n- `Mnesia` event handling\n- Debugging `Mnesia` applications\n- Concurrent processes in `Mnesia`\n- Prototyping\n- Object-based programming with `Mnesia`","ref":"mnesia_chap5.html"},{"type":"extras","title":"Indexing - Miscellaneous Mnesia Features","doc":"Data retrieval and matching can be performed efficiently if the key for the\nrecord is known. Conversely, if the key is unknown, all records in a table must\nbe searched. The larger the table, the more time consuming it becomes. To remedy\nthis problem, `Mnesia` indexing capabilities are used to improve data retrieval\nand matching of records.\n\nThe following two functions manipulate indexes on existing tables:\n\n- [`mnesia:add_table_index(Tab, AttributeName) -> {aborted, R} | {atomic, ok}`](`mnesia:add_table_index/2`)\n- [`mnesia:del_table_index(Tab, AttributeName) -> {aborted, R} | {atomic, ok}`](`mnesia:del_table_index/2`)\n\nThese functions create or delete a table index on a field defined by\n`AttributeName`. To illustrate this, add an index to the table definition\n`(employee, {emp_no, name, salary, sex, phone, room_no})`, which is the example\ntable from the `Company` database. The function that adds an index on element\n`salary` can be expressed as `mnesia:add_table_index(employee, salary)`.\n\nThe indexing capabilities of `Mnesia` are used with the following three\nfunctions, which retrieve and match records based on index entries in the\ndatabase:\n\n- [`mnesia:index_read(Tab, SecondaryKey, AttributeName) -> transaction abort | RecordList`](`mnesia:index_read/3`)\n  avoids an exhaustive search of the entire table, by looking up `SecondaryKey`\n  in the index to find the primary keys.\n- [`mnesia:index_match_object(Pattern, AttributeName) -> transaction abort | RecordList`](`mnesia:index_match_object/2`)\n  avoids an exhaustive search of the entire table, by looking up the secondary\n  key in the index to find the primary keys. The secondary key is found in field\n  `AttributeName` of `Pattern`. The secondary key must be bound.\n- [`mnesia:match_object(Pattern) -> transaction abort | RecordList`](`mnesia:match_object/1`)\n  uses indexes to avoid exhaustive search of the entire table. Unlike the\n  previous functions, this function can use any index as long as the secondary\n  key is bound.\n\nThese functions are further described and exemplified in\n[Pattern Matching](mnesia_chap4.md#matching).","ref":"mnesia_chap5.html#indexing"},{"type":"extras","title":"Distribution and Fault Tolerance - Miscellaneous Mnesia Features","doc":"`Mnesia` is a distributed, fault-tolerant DBMS. Tables can be replicated on\ndifferent Erlang nodes in various ways. The `Mnesia` programmer does not need to\nstate where the different tables reside, only the names of the different tables\nneed to be specified in the program code. This is known as \"location\ntransparency\" and is an important concept. In particular:\n\n- A program works regardless of the data location. It makes no difference\n  whether the data resides on the local node or on a remote node.\n\n  Notice that the program runs slower if the data is located on a remote node.\n\n- The database can be reconfigured, and tables can be moved between nodes. These\n  operations do not affect the user programs.\n\nIt has previously been shown that each table has a number of system attributes,\nsuch as `index` and `type`.\n\nTable attributes are specified when the table is created. For example, the\nfollowing function creates a table with two RAM replicas:\n\n```erlang\nmnesia:create_table(foo,\n                    [{ram_copies, [N1, N2]},\n                     {attributes, record_info(fields, foo)}]).\n```\n\nTables can also have the following properties, where each attribute has a list\nof Erlang nodes as its value:\n\n- `ram_copies`. The value of the node list is a list of Erlang nodes, and a RAM\n  replica of the table resides on each node in the list.\n\n  Notice that no disc operations are performed when a program executes write\n  operations to these replicas. However, if permanent RAM replicas are required,\n  the following alternatives are available:\n\n  1. The function `mnesia:dump_tables/1` can be used to dump RAM table replicas\n     to disc.\n  1. The table replicas can be backed up, either from RAM, or from disc if\n     dumped there with this function.\n\n- `disc_copies`. The value of the attribute is a list of Erlang nodes, and a\n  replica of the table resides both in RAM and on disc on each node in the list.\n  Write operations addressed to the table address both the RAM and the disc copy\n  of the table.\n- `disc_only_copies`. The value of the attribute is a list of Erlang nodes, and\n  a replica of the table resides only as a disc copy on each node in the list.\n  The major disadvantage of this type of table replica is the access speed. The\n  major advantage is that the table does not occupy space in memory.\n\nIn addition, table properties can be set and changed. For details, see\n[Define a Schema](mnesia_chap3.md#def_schema).\n\nThere are basically two reasons for using more than one table replica: fault\ntolerance and speed. Notice that table replication provides a solution to both\nof these system requirements.\n\nIf there are two active table replicas, all information is still available if\none replica fails. This can be an important property in many applications.\nFurthermore, if a table replica exists at two specific nodes, applications that\nexecute at either of these nodes can read data from the table without accessing\nthe network. Network operations are considerably slower and consume more\nresources than local operations.\n\nIt can be advantageous to create table replicas for a distributed application\nthat reads data often, but writes data seldom, to achieve fast read operations\non the local node. The major disadvantage with replication is the increased time\nto write data. If a table has two replicas, every write operation must access\nboth table replicas. Since one of these write operations must be a network\noperation, it is considerably more expensive to perform a write operation to a\nreplicated table than to a non-replicated table.","ref":"mnesia_chap5.html#distribution-and-fault-tolerance"},{"type":"extras","title":"Table Fragmentation - Miscellaneous Mnesia Features","doc":"","ref":"mnesia_chap5.html#table-fragmentation"},{"type":"extras","title":"Concept - Miscellaneous Mnesia Features","doc":"A concept of table fragmentation has been introduced to cope with large tables.\nThe idea is to split a table into several manageable fragments. Each fragment is\nimplemented as a first class `Mnesia` table and can be replicated, have indexes,\nand so on, as any other table. But the tables cannot have `local_content` or\nhave the `snmp` connection activated.\n\nTo be able to access a record in a fragmented table, `Mnesia` must determine to\nwhich fragment the actual record belongs. This is done by module `mnesia_frag`,\nwhich implements the `mnesia_access` callback behavior. It is recommended to\nread the documentation about the function `mnesia:activity/4` to see how\n`mnesia_frag` can be used as a `mnesia_access` callback module.\n\nAt each record access, `mnesia_frag` first computes a hash value from the record\nkey. Second, the name of the table fragment is determined from the hash value.\nFinally the actual table access is performed by the same functions as for\nnon-fragmented tables. When the key is not known beforehand, all fragments are\nsearched for matching records.\n\nNotice that in `ordered_set` tables, the records are ordered per fragment, and\nthe order is undefined in results returned by `select` and `match_object`, as\nwell as `first`, `next`, `prev` and `last`.\n\nThe following code illustrates how a `Mnesia` table is converted to be a\nfragmented table and how more fragments are added later:\n\n```erlang\nEshell V4.7.3.3  (abort with ^G)\n(a@sam)1> mnesia:start().\nok\n(a@sam)2> mnesia:system_info(running_db_nodes).\n[b@sam,c@sam,a@sam]\n(a@sam)3> Tab = dictionary.\ndictionary\n(a@sam)4> mnesia:create_table(Tab, [{ram_copies, [a@sam, b@sam]}]).\n{atomic,ok}\n(a@sam)5> Write = fun(Keys) -> [mnesia:write({Tab,K,-K}) || K <- Keys], ok end.\n#Fun \n(a@sam)6> mnesia:activity(sync_dirty, Write, [lists:seq(1, 256)], mnesia_frag).\nok\n(a@sam)7> mnesia:change_table_frag(Tab, {activate, []}).\n{atomic,ok}\n(a@sam)8> mnesia:table_info(Tab, frag_properties).\n[{base_table,dictionary},\n {foreign_key,undefined},\n {n_doubles,0},\n {n_fragments,1},\n {next_n_to_split,1},\n {node_pool,[a@sam,b@sam,c@sam]}]\n(a@sam)9> Info = fun(Item) -> mnesia:table_info(Tab, Item) end.\n#Fun \n(a@sam)10> Dist = mnesia:activity(sync_dirty, Info, [frag_dist], mnesia_frag).\n[{c@sam,0},{a@sam,1},{b@sam,1}]\n(a@sam)11> mnesia:change_table_frag(Tab, {add_frag, Dist}).\n{atomic,ok}\n(a@sam)12> Dist2 = mnesia:activity(sync_dirty, Info, [frag_dist], mnesia_frag).\n[{b@sam,1},{c@sam,1},{a@sam,2}]\n(a@sam)13> mnesia:change_table_frag(Tab, {add_frag, Dist2}).\n{atomic,ok}\n(a@sam)14> Dist3 = mnesia:activity(sync_dirty, Info, [frag_dist], mnesia_frag).\n[{a@sam,2},{b@sam,2},{c@sam,2}]\n(a@sam)15> mnesia:change_table_frag(Tab, {add_frag, Dist3}).\n{atomic,ok}\n(a@sam)16> Read = fun(Key) -> mnesia:read({Tab, Key}) end.\n#Fun \n(a@sam)17> mnesia:activity(transaction, Read, [12], mnesia_frag).\n[{dictionary,12,-12}]\n(a@sam)18> mnesia:activity(sync_dirty, Info, [frag_size], mnesia_frag).\n[{dictionary,64},\n {dictionary_frag2,64},\n {dictionary_frag3,64},\n {dictionary_frag4,64}]\n(a@sam)19>\n```","ref":"mnesia_chap5.html#concept"},{"type":"extras","title":"Fragmentation Properties - Miscellaneous Mnesia Features","doc":"The table property `frag_properties` can be read with the function\n[`mnesia:table_info(Tab, frag_properties)`](`mnesia:table_info/2`). The\nfragmentation properties are a list of tagged tuples with arity 2. By default\nthe list is empty, but when it is non-empty it triggers `Mnesia` to regard the\ntable as fragmented. The fragmentation properties are as follows:\n\n- **`{n_fragments, Int}`** - `n_fragments` regulates how many fragments that the\n  table currently has. This property can explicitly be set at table creation and\n  later be changed with `{add_frag, NodesOrDist}` or `del_frag`. `n_fragments`\n  defaults to `1`.\n\n- **`{node_pool, List}`** - The node pool contains a list of nodes and can\n  explicitly be set at table creation and later be changed with\n  `{add_node, Node}` or `{del_node, Node}`. At table creation `Mnesia` tries to\n  distribute the replicas of each fragment evenly over all the nodes in the node\n  pool. Hopefully all nodes end up with the same number of replicas. `node_pool`\n  defaults to the return value from the function\n  [`mnesia:system_info(db_nodes)`](`mnesia:system_info/1`).\n\n- **`{n_ram_copies, Int}`** - Regulates how many `ram_copies` replicas that each\n  fragment is to have. This property can explicitly be set at table creation.\n  Defaults is `0`, but if `n_disc_copies` and `n_disc_only_copies` also are `0`,\n  `n_ram_copies` defaults to `1`.\n\n- **`{n_disc_copies, Int}`** - Regulates how many `disc_copies` replicas that\n  each fragment is to have. This property can explicitly be set at table\n  creation. Default is `0`.\n\n- **`{n_disc_only_copies, Int}`** - Regulates how many `disc_only_copies`\n  replicas that each fragment is to have. This property can explicitly be set at\n  table creation. Defaults is `0`.\n\n- **`{foreign_key, ForeignKey}`** - `ForeignKey` can either be the atom\n  `undefined` or the tuple `{ForeignTab, Attr}`, where `Attr` denotes an\n  attribute that is to be interpreted as a key in another fragmented table named\n  `ForeignTab`. `Mnesia` ensures that the number of fragments in this table and\n  in the foreign table are always the same.\n\n  When fragments are added or deleted, `Mnesia` automatically propagates the\n  operation to all fragmented tables that have a foreign key referring to this\n  table. Instead of using the record key to determine which fragment to access,\n  the value of field `Attr` is used. This feature makes it possible to colocate\n  records automatically in different tables to the same node. `foreign_key`\n  defaults to `undefined`. However, if the foreign key is set to something else,\n  it causes the default values of the other fragmentation properties to be the\n  same values as the actual fragmentation properties of the foreign table.\n\n- **`{hash_module, Atom}`** - Enables definition of an alternative hashing\n  scheme. The module must implement the `m:mnesia_frag_hash` callback behavior.\n  This property can explicitly be set at table creation. Default is\n  `mnesia_frag_hash`.\n\n- **`{hash_state, Term}`** - Enables a table-specific parameterization of a\n  generic hash module. This property can explicitly be set at table creation.\n  Default is `undefined`.\n\n  ```erlang\n  Eshell V4.7.3.3  (abort with ^G)\n  (a@sam)1> mnesia:start().\n  ok\n  (a@sam)2> PrimProps = [{n_fragments, 7}, {node_pool, [node()]}].\n  [{n_fragments,7},{node_pool,[a@sam]}]\n  (a@sam)3> mnesia:create_table(prim_dict,\n                                [{frag_properties, PrimProps},\n                                 {attributes,[prim_key,prim_val]}]).\n  {atomic,ok}\n  (a@sam)4> SecProps = [{foreign_key, {prim_dict, sec_val}}].\n  [{foreign_key,{prim_dict,sec_val}}]\n  (a@sam)5> mnesia:create_table(sec_dict,\n                                [{frag_properties, SecProps},\n  (a@sam)5>                      {attributes, [sec_key, sec_val]}]).\n  {atomic,ok}\n  (a@sam)6> Write = fun(Rec) -> mnesia:write(Rec) end.\n  #Fun \n  (a@sam)7> PrimKey = 11.\n  11\n  (a@sam)8> SecKey = 42.\n  42\n  (a@sam)9> mnesia:activity(sync_dirty, Write,\n                            [{prim_dict, PrimKey, -11}], mnesia_frag).\n  ok\n  (a@sam)10> mnesia:activity(sync_dirty, Write,\n                             [{sec_dict, SecKey, PrimKey}], mnesia_frag).\n  ok\n  (a@sam)11> mnesia:change_table_frag(prim_dict, {add_frag, [node()]}).\n  {atomic,ok}\n  (a@sam)12> SecRead = fun(PrimKey, SecKey) ->\n                 mnesia:read({sec_dict, PrimKey}, SecKey, read) end.\n  #Fun \n  (a@sam)13> mnesia:activity(transaction, SecRead,\n                             [PrimKey, SecKey], mnesia_frag).\n  [{sec_dict,42,11}]\n  (a@sam)14> Info = fun(Tab, Item) -> mnesia:table_info(Tab, Item) end.\n  #Fun \n  (a@sam)15> mnesia:activity(sync_dirty, Info,\n                             [prim_dict, frag_size], mnesia_frag).\n  [{prim_dict,0},\n   {prim_dict_frag2,0},\n   {prim_dict_frag3,0},\n   {prim_dict_frag4,1},\n   {prim_dict_frag5,0},\n   {prim_dict_frag6,0},\n   {prim_dict_frag7,0},\n   {prim_dict_frag8,0}]\n  (a@sam)16> mnesia:activity(sync_dirty, Info,\n                             [sec_dict, frag_size], mnesia_frag).\n  [{sec_dict,0},\n   {sec_dict_frag2,0},\n   {sec_dict_frag3,0},\n   {sec_dict_frag4,1},\n   {sec_dict_frag5,0},\n   {sec_dict_frag6,0},\n   {sec_dict_frag7,0},\n   {sec_dict_frag8,0}]\n  (a@sam)17>\n  ```","ref":"mnesia_chap5.html#fragmentation-properties"},{"type":"extras","title":"Management of Fragmented Tables - Miscellaneous Mnesia Features","doc":"The function `mnesia:change_table_frag(Tab, Change)` is intended to be used for\nreconfiguration of fragmented tables. Argument `Change` is to have one of the\nfollowing values:\n\n- **`{activate, FragProps}`** - Activates the fragmentation properties of an\n  existing table. `FragProps` is either to contain `{node_pool, Nodes}` or be\n  empty.\n\n- **`deactivate`** - Deactivates the fragmentation properties of a table. The\n  number of fragments must be `1`. No other table can refer to this table in its\n  foreign key.\n\n- **`{add_frag, NodesOrDist}`** - Adds a fragment to a fragmented table. All\n  records in one of the old fragments are rehashed and about half of them are\n  moved to the new (last) fragment. All other fragmented tables, which refer to\n  this table in their foreign key, automatically get a new fragment. Also, their\n  records are dynamically rehashed in the same manner as for the main table.\n\n  Argument `NodesOrDist` can either be a list of nodes or the result from the\n  function [`mnesia:table_info(Tab, frag_dist)`](`mnesia:table_info/2`). Argument\n  `NodesOrDist` is assumed to be a sorted list with the best nodes to host new\n  replicas first in the list. The new fragment gets the same number of replicas\n  as the first fragment (see `n_ram_copies`, `n_disc_copies`, and\n  `n_disc_only_copies`). The `NodesOrDist` list must at least contain one\n  element for each replica that needs to be allocated.\n\n- **`del_frag`** - Deletes a fragment from a fragmented table. All records in\n  the last fragment are moved to one of the other fragments. All other\n  fragmented tables, which refer to this table in their foreign key,\n  automatically lose their last fragment. Also, their records are dynamically\n  rehashed in the same manner as for the main table.\n\n- **`{add_node, Node}`** - Adds a node to `node_pool`. The new node pool affects\n  the list returned from the function\n  [`mnesia:table_info(Tab, frag_dist)`](`mnesia:table_info/2`).\n\n- **`{del_node, Node}`** - Deletes a node from `node_pool`. The new node pool\n  affects the list returned from the function\n  [`mnesia:table_info(Tab, frag_dist)`](`mnesia:table_info/2`).","ref":"mnesia_chap5.html#management-of-fragmented-tables"},{"type":"extras","title":"Extensions of Existing Functions - Miscellaneous Mnesia Features","doc":"The function `mnesia:create_table/2` creates a brand new fragmented table, by\nsetting table property `frag_properties` to some proper values.\n\nThe function `mnesia:delete_table/1` deletes a fragmented table including all\nits fragments. There must however not exist any other fragmented tables that\nrefer to this table in their foreign key.\n\nThe function `mnesia:table_info/2` now understands item `frag_properties`.\n\nIf the function `mnesia:table_info/2` is started in the activity context of\nmodule `mnesia_frag`, information of several new items can be obtained:\n\n- **`base_table`** - The name of the fragmented table\n\n- **`n_fragments`** - The actual number of fragments\n\n- **`node_pool`** - The pool of nodes\n\n- **`n_ram_copies`**\n\n- **`n_disc_copies`**\n\n- **`n_disc_only_copies`** - The number of replicas with storage type\n  `ram_copies`, `disc_copies`, and `disc_only_copies`, respectively. The actual\n  values are dynamically derived from the first fragment. The first fragment\n  serves as a protype. When the actual values need to be computed (for example,\n  when adding new fragments) they are determined by counting the number of each\n  replica for each storage type. This means that when the functions\n  `mnesia:add_table_copy/3`, `mnesia:del_table_copy/2`, and\n  [`mnesia:change_table_copy_type/2`](`mnesia:change_table_copy_type/3`) are\n  applied on the first fragment, it affects the settings on `n_ram_copies`,\n  `n_disc_copies`, and `n_disc_only_copies`.\n\n- **`foreign_key`** - The foreign key\n\n- **`foreigners`** - All other tables that refer to this table in their foreign\n  key\n\n- **`frag_names`** - The names of all fragments\n\n- **`frag_dist`** - A sorted list of `{Node, Count}` tuples that are sorted in\n  increasing `Count` order. `Count` is the total number of replicas that this\n  fragmented table hosts on each `Node`. The list always contains at least all\n  nodes in `node_pool`. Nodes that do not belong to `node_pool` are put last in\n  the list even if their `Count` is lower.\n\n- **`frag_size`** - A list of `{Name, Size}` tuples, where `Name` is a fragment\n  `Name`, and `Size` is how many records it contains\n\n- **`frag_memory`** - A list of `{Name, Memory}` tuples, where `Name` is a\n  fragment `Name`, and `Memory` is how much memory it occupies\n\n- **`size`** - Total size of all fragments\n\n- **`memory`** - Total memory of all fragments","ref":"mnesia_chap5.html#extensions-of-existing-functions"},{"type":"extras","title":"Load Balancing - Miscellaneous Mnesia Features","doc":"There are several algorithms for distributing records in a fragmented table\nevenly over a pool of nodes. No one is best, it depends on the application\nneeds. The following examples of situations need some attention:\n\n- `permanent change of nodes`. When a new permanent `db_node` is introduced or\n  dropped, it can be time to change the pool of nodes and redistribute the\n  replicas evenly over the new pool of nodes. It can also be time to add or\n  delete a fragment before the replicas are redistributed.\n- `size/memory threshold`. When the total size or total memory of a fragmented\n  table (or a single fragment) exceeds some application-specific threshold, it\n  can be time to add a new fragment dynamically to obtain a better distribution\n  of records.\n- `temporary node down`. When a node temporarily goes down, it can be time to\n  compensate some fragments with new replicas to keep the desired level of\n  redundancy. When the node comes up again, it can be time to remove the\n  superfluous replica.\n- `overload threshold`. When the load on some node exceeds some\n  application-specific threshold, it can be time to either add or move some\n  fragment replicas to nodes with lower load. Take extra care if the table has a\n  foreign key relation to some other table. To avoid severe performance\n  penalties, the same redistribution must be performed for all the related\n  tables.\n\nUse the function `mnesia:change_table_frag/2` to add new fragments and apply the\nusual schema manipulation functions (such as `mnesia:add_table_copy/3`,\n`mnesia:del_table_copy/2`, and\n[`mnesia:change_table_copy_type/2`](`mnesia:change_table_copy_type/3`)) on each\nfragment to perform the actual redistribution.","ref":"mnesia_chap5.html#load-balancing"},{"type":"extras","title":"Local Content Tables - Miscellaneous Mnesia Features","doc":"Replicated tables have the same content on all nodes where they are replicated.\nHowever, it is sometimes advantageous to have tables, but different content on\ndifferent nodes.\n\nIf attribute `{local_content, true}` is specified when you create the table, the\ntable resides on the nodes where you specify the table to exist, but the write\noperations on the table are only performed on the local copy.\n\nFurthermore, when the table is initialized at startup, the table is only\ninitialized locally, and the table content is not copied from another node.","ref":"mnesia_chap5.html#local-content-tables"},{"type":"extras","title":"Disc-Less Nodes - Miscellaneous Mnesia Features","doc":"`Mnesia` can be run on nodes that do not have a disc. Replicas of `disc_copies`\nor `disc_only_copies` are not possible on such nodes. This is especially\ntroublesome for the `schema` table, as `Mnesia` needs the schema to initialize\nitself.\n\nThe schema table can, as other tables, reside on one or more nodes. The storage\ntype of the schema table can either be `disc_copies` or `ram_copies` (but not\n`disc_only_copies`). At startup, `Mnesia` uses its schema to determine with\nwhich nodes it is to try to establish contact. If any other node is started\nalready, the starting node merges its table definitions with the table\ndefinitions brought from the other nodes. This also applies to the definition of\nthe schema table itself. Application parameter `extra_db_nodes` contains a list\nof nodes that `Mnesia` also is to establish contact with besides those found in\nthe schema. Default is `[]` (empty list).\n\nHence, when a disc-less node needs to find the schema definitions from a remote\nnode on the network, this information must be supplied through application\nparameter `-mnesia extra_db_nodes NodeList`. Without this configuration\nparameter set, `Mnesia` starts as a single node system. Also, the function\n`mnesia:change_config/2` can be used to assign a value to `extra_db_nodes` and\nforce a connection after `Mnesia` has been started, that is,\n`mnesia:change_config(extra_db_nodes, NodeList)`.\n\nApplication parameter `schema_location` controls where `Mnesia` searches for its\nschema. The parameter can be one of the following atoms:\n\n- **`disc`** - Mandatory disc. The schema is assumed to be located in the\n  `Mnesia` directory. If the schema cannot be found, `Mnesia` refuses to start.\n\n- **`ram`** - Mandatory RAM. The schema resides in RAM only. At startup, a tiny\n  new schema is generated. This default schema contains only the definition of\n  the schema table and resides on the local node only. Since no other nodes are\n  found in the default schema, configuration parameter `extra_db_nodes` must be\n  used to let the node share its table definitions with other nodes. (Parameter\n  `extra_db_nodes` can also be used on disc-full nodes.)\n\n- **`opt_disc`** - Optional disc. The schema can reside on either disc or RAM.\n  If the schema is found on disc, `Mnesia` starts as a disc-full node (the\n  storage type of the schema table is disc_copies). If no schema is found on\n  disc, `Mnesia` starts as a disc-less node (the storage type of the schema\n  table is `ram_copies`). The default for the application parameter is\n  `opt_disc`.\n\nWhen `schema_location` is set to `opt_disc`, the function\n`mnesia:change_table_copy_type/3` can be used to change the storage type of the\nschema. This is illustrated as follows:\n\n```erlang\n1> mnesia:start().\nok\n2> mnesia:change_table_copy_type(schema, node(), disc_copies).\n{atomic, ok}\n```\n\nAssuming that the call to `mnesia:start/0` does not find any schema to read on\nthe disc, `Mnesia` starts as a disc-less node, and then change it to a node that\nuse the disc to store the schema locally.","ref":"mnesia_chap5.html#disc-less-nodes"},{"type":"extras","title":"More about Schema Management - Miscellaneous Mnesia Features","doc":"Nodes can be added to and removed from a `Mnesia` system. This can be done by\nadding a copy of the schema to those nodes.\n\nThe functions `mnesia:add_table_copy/3` and `mnesia:del_table_copy/2` can be\nused to add and delete replicas of the schema table. Adding a node to the list\nof nodes where the schema is replicated affects the following:\n\n- It allows other tables to be replicated to this node.\n- It causes `Mnesia` to try to contact the node at startup of disc-full nodes.\n\nThe function call `mnesia:del_table_copy(schema, mynode@host)` deletes node\n`mynode@host` from the `Mnesia` system. The call fails if `Mnesia` is running on\n`mynode@host`. The other `Mnesia` nodes never try to connect to that node again.\nNotice that if there is a disc resident schema on node `mynode@host`, the entire\n`Mnesia` directory is to be deleted. This is done with the function\n`mnesia:delete_schema/1`. If `Mnesia` is started again on node `mynode@host` and\nthe directory has not been cleared, the behavior of `Mnesia` is undefined.\n\nIf the storage type of the schema is `ram_copies`, that is, a disc-less node,\n`Mnesia` does not use the disc on that particular node. The disc use is enabled\nby changing the storage type of table `schema` to `disc_copies`.\n\nNew schemas are created explicitly with the function `mnesia:create_schema/1` or\nimplicitly by starting `Mnesia` without a disc resident schema. Whenever a table\n(including the schema table) is created, it is assigned its own unique cookie.\nThe schema table is not created with the function `mnesia:create_table/2` as\nnormal tables.\n\nAt startup, `Mnesia` connects different nodes to each other, then they exchange\ntable definitions with each other, and the table definitions are merged. During\nthe merge procedure, `Mnesia` performs a sanity test to ensure that the table\ndefinitions are compatible with each other. If a table exists on several nodes,\nthe cookie must be the same, otherwise `Mnesia` shut down one of the nodes. This\nunfortunate situation occurs if a table has been created on two nodes\nindependently of each other while they were disconnected. To solve this, one of\nthe tables must be deleted (as the cookies differ, it is regarded to be two\ndifferent tables even if they have the same name).\n\nMerging different versions of the schema table does not always require the\ncookies to be the same. If the storage type of the schema table is\n`disc_copies`, the cookie is immutable, and all other `db_nodes` must have the\nsame cookie. When the schema is stored as type `ram_copies`, its cookie can be\nreplaced with a cookie from another node (`ram_copies` or `disc_copies`). The\ncookie replacement (during merge of the schema table definition) is performed\neach time a RAM node connects to another node.\n\nFurther, the following applies:\n\n- [`mnesia:system_info(schema_location)`](`mnesia:system_info/1`) and\n  [`mnesia:system_info(extra_db_nodes)`](`mnesia:system_info/1`) can be used to\n  determine the actual values of `schema_location` and `extra_db_nodes`,\n  respectively.\n- [`mnesia:system_info(use_dir)`](`mnesia:system_info/1`) can be used to determine\n  whether `Mnesia` is actually using the `Mnesia` directory.\n- `use_dir` can be determined even before `Mnesia` is started.\n\nThe function `mnesia:info/0` can now be used to print some system information\neven before `Mnesia` is started. When `Mnesia` is started, the function prints\nmore information.\n\nTransactions that update the definition of a table requires that `Mnesia` is\nstarted on all nodes where the storage type of the schema is `disc_copies`. All\nreplicas of the table on these nodes must also be loaded. There are a few\nexceptions to these availability rules:\n\n- Tables can be created and new replicas can be added without starting all the\n  disc-full nodes.\n- New replicas can be added before all other replicas of the table have been\n  loaded, provided that at least one other replica is active.\n\n[](){: #event_handling }","ref":"mnesia_chap5.html#more-about-schema-management"},{"type":"extras","title":"Mnesia Event Handling - Miscellaneous Mnesia Features","doc":"System events and table events are the two event categories that `Mnesia`\ngenerates in various situations.\n\nA user process can subscribe on the events generated by `Mnesia`. The following\ntwo functions are provided:\n\n- [`mnesia:subscribe(Event-Category)`](`mnesia:subscribe/1`) - Ensures that a\n  copy of all events of type `Event-Category` are sent to the calling process\n\n- [`mnesia:unsubscribe(Event-Category)`](`mnesia:unsubscribe/1`) - Removes the\n  subscription on events of type `Event-Category`\n\n`Event-Category` can be either of the following:\n\n- The atom `system`\n- The atom `activity`\n- The tuple `{table, Tab, simple}`\n- The tuple `{table, Tab, detailed}`\n\nThe old event category `{table, Tab}` is the same event category as\n`{table, Tab, simple}`.\n\nThe subscribe functions activate a subscription of events. The events are\ndelivered as messages to the process evaluating the function\n`mnesia:subscribe/1` The syntax is as follows:\n\n- `{mnesia_system_event, Event}` for system events\n- `{mnesia_activity_event, Event}` for activity events\n- `{mnesia_table_event, Event}` for table events\n\nThe event types are described in the next sections.\n\nAll system events are subscribed by the `Mnesia` `gen_event` handler. The\ndefault `gen_event` handler is `mnesia_event`, but it can be changed by using\napplication parameter `event_module`. The value of this parameter must be the\nname of a module implementing a complete handler, as specified by the\n`m:gen_event` module in `STDLIB`.\n\n[`mnesia:system_info(subscribers)`](`mnesia:system_info/1`) and\n[`mnesia:table_info(Tab, subscribers)`](`mnesia:table_info/2`) can be used to\ndetermine which processes are subscribed to various events.","ref":"mnesia_chap5.html#mnesia-event-handling"},{"type":"extras","title":"System Events - Miscellaneous Mnesia Features","doc":"The system events are as follows:\n\n- **`{mnesia_up, Node}`** - Mnesia is started on a node. `Node` is the node\n  name. By default this event is ignored.\n\n- **`{mnesia_down, Node}`** - Mnesia is stopped on a node. `Node` is the node\n  name. By default this event is ignored.\n\n- **`{mnesia_checkpoint_activated, Checkpoint}`** - A checkpoint with the name\n  `Checkpoint` is activated and the current node is involved in the checkpoint.\n  Checkpoints can be activated explicitly with the function\n  `mnesia:activate_checkpoint/1` or implicitly at backup, when adding table\n  replicas, at internal transfer of data between nodes, and so on. By default\n  this event is ignored.\n\n- **`{mnesia_checkpoint_deactivated, Checkpoint}`** - A checkpoint with the name\n  `Checkpoint` is deactivated and the current node is involved in the\n  checkpoint. Checkpoints can be deactivated explicitly with the function\n  [`mnesia:deactivate/1`](`mnesia:deactivate_checkpoint/1`) or implicitly when the\n  last replica of a table (involved in the checkpoint) becomes unavailable, for\n  example, at node-down. By default this event is ignored.\n\n- **`{mnesia_overload, Details}`** - `Mnesia` on the current node is overloaded\n  and the subscriber is to take action.\n\n  A typical overload situation occurs when the applications perform more updates\n  on disc resident tables than `Mnesia` can handle. Ignoring this kind of\n  overload can lead to a situation where the disc space is exhausted (regardless\n  of the size of the tables stored on disc).\n\n  Each update is appended to the transaction log and occasionally (depending on\n  how it is configured) dumped to the tables files. The table file storage is\n  more compact than the transaction log storage, especially if the same record\n  is updated repeatedly. If the thresholds for dumping the transaction log are\n  reached before the previous dump is finished, an overload event is triggered.\n\n  Another typical overload situation is when the transaction manager cannot\n  commit transactions at the same pace as the applications perform updates of\n  disc resident tables. When this occurs, the message queue of the transaction\n  manager continues to grow until the memory is exhausted or the load decreases.\n\n  The same problem can occur for dirty updates. The overload is detected locally\n  on the current node, but its cause can be on another node. Application\n  processes can cause high load if any table resides on another node (replicated\n  or not). By default this event is reported to `error_logger.`\n\n- **`{inconsistent_database, Context, Node}`** - `Mnesia` regards the database\n  as potential inconsistent and gives its applications a chance to recover from\n  the inconsistency. For example, by installing a consistent backup as fallback\n  and then restart the system. An alternative is to pick a `MasterNode` from\n  [`mnesia:system_info(db_nodes)`](`mnesia:system_info/1`) and invoke\n  [`mnesia:set_master_nodes([MasterNode])`](`mnesia:set_master_nodes/1`). By\n  default an error is reported to `error_logger`.\n\n- **`{mnesia_fatal, Format, Args, BinaryCore}`** - `Mnesia` detected a fatal\n  error and terminates soon. The fault reason is explained in `Format` and\n  `Args`, which can be given as input to `io:format/2` or sent to\n  `error_logger`. By default it is sent to `error_logger`.\n\n  `BinaryCore` is a binary containing a summary of the `Mnesia` internal state\n  at the time when the fatal error was detected. By default the binary is\n  written to a unique filename on the current directory. On RAM nodes, the core\n  is ignored.\n\n- **`{mnesia_info, Format, Args}`** - `Mnesia` detected something that can be of\n  interest when debugging the system. This is explained in `Format` and `Args`,\n  which can appear as input to `io:format/2` or sent to `error_logger`. By\n  default this event is printed with `io:format/2`.\n\n- **`{mnesia_error, Format, Args}`** - `Mnesia` has detected an error. The fault\n  reason is explained in `Format` and `Args`, which can be given as input to\n  `io:format/2` or sent to `error_logger`. By default this event is reported to\n  `error_logger`.\n\n- **`{mnesia_user, Event}`** - An application started the function\n  [`mnesia:report_event(Event)`](`mnesia:report_event/1`). `Event` can be any\n  Erlang data structure. When tracing a system of `Mnesia` applications, it is\n  useful to be able to interleave own events of `Mnesia` with\n  application-related events that give information about the application\n  context. Whenever the application starts with a new and demanding `Mnesia`\n  activity, or enters a new and interesting phase in its execution, it can be a\n  good idea to use `mnesia:report_event/1`.","ref":"mnesia_chap5.html#system-events"},{"type":"extras","title":"Activity Events - Miscellaneous Mnesia Features","doc":"Currently, there is only one type of activity event:\n\n- **`{complete, ActivityID}`** - This event occurs when a transaction that\n  caused a modification to the database is completed. It is useful for\n  determining when a set of table events (see the next section), caused by a\n  given activity, have been sent. Once this event is received, it is guaranteed\n  that no further table events with the same `ActivityID` will be received.\n  Notice that this event can still be received even if no table events with a\n  corresponding `ActivityID` were received, depending on the tables to which the\n  receiving process is subscribed.\n\n  Dirty operations always contain only one update and thus no activity event is\n  sent.","ref":"mnesia_chap5.html#activity-events"},{"type":"extras","title":"Table Events - Miscellaneous Mnesia Features","doc":"Table events are events related to table updates. There are two types of table\nevents, simple and detailed.\n\nThe _simple table events_ are tuples like `{Oper, Record, ActivityId}`, where:\n\n- `Oper` is the operation performed.\n- `Record` is the record involved in the operation.\n- `ActivityId` is the identity of the transaction performing the operation.\n\nNotice that the record name is the table name even when `record_name` has\nanother setting.\n\nThe table-related events that can occur are as follows:\n\n- **`{write, NewRecord, ActivityId}`** - A new record has been written.\n  `NewRecord` contains the new record value.\n\n- **`{delete_object, OldRecord, ActivityId}`** - A record has possibly been\n  deleted with `mnesia:delete_object/1`. `OldRecord` contains the value of the\n  old record, as stated as argument by the application. Notice that other\n  records with the same key can remain in the table if it is of type `bag`.\n\n- **`{delete, {Tab, Key}, ActivityId}`** - One or more records have possibly\n  been deleted. All records with the key `Key` in the table `Tab` have been\n  deleted.\n\nThe _detailed table events_ are tuples like\n`{Oper, Table, Data, [OldRecs], ActivityId}`, where:\n\n- `Oper` is the operation performed.\n- `Table` is the table involved in the operation.\n- `Data` is the record/OID written/deleted.\n- `OldRecs` is the contents before the operation.\n- `ActivityId` is the identity of the transaction performing the operation.\n\nThe table-related events that can occur are as follows:\n\n- **`{write, Table, NewRecord, [OldRecords], ActivityId}`** - A new record has\n  been written. `NewRecord` contains the new record value and `OldRecords`\n  contains the records before the operation is performed. Notice that the new\n  content depends on the table type.\n\n- **`{delete, Table, What, [OldRecords], ActivityId}`** - Records have possibly\n  been deleted. `What` is either `{Table, Key}` or a record\n  `{RecordName, Key, ...}` that was deleted. Notice that the new content depends\n  on the table type.","ref":"mnesia_chap5.html#table-events"},{"type":"extras","title":"Debugging Mnesia Applications - Miscellaneous Mnesia Features","doc":"Debugging a `Mnesia` application can be difficult for various reasons, primarily\nrelated to difficulties in understanding how the transaction and table load\nmechanisms work. Another source of confusion can be the semantics of nested\ntransactions.\n\nThe debug level of `Mnesia` is set by calling the function\n[`mnesia:set_debug_level(Level)`](`mnesia:set_debug_level/1`), where `Level` is\none of the following:\n\n- **`none`** - No trace outputs. This is the default.\n\n- **`verbose`** - Activates tracing of important debug events. These events\n  generate `{mnesia_info, Format, Args}` system events. Processes can subscribe\n  to these events with the function `mnesia:subscribe/1`. The events are always\n  sent to the `Mnesia` event handler.\n\n- **`debug`** - Activates all events at the verbose level plus traces of all\n  debug events. These debug events generate `{mnesia_info, Format, Args}` system\n  events. Processes can subscribe to these events with `mnesia:subscribe/1`. The\n  events are always sent to the `Mnesia` event handler. On this debug level, the\n  `Mnesia` event handler starts subscribing to updates in the schema table.\n\n- **`trace`** - Activates all events at the debug level. On this level, the\n  `Mnesia` event handler starts subscribing to updates on all `Mnesia` tables.\n  This level is intended only for debugging small toy systems, as many large\n  events can be generated.\n\n- **`false`** - An alias for none.\n\n- **`true`** - An alias for debug.\n\nThe debug level of `Mnesia` itself is also an application parameter, making it\npossible to start an Erlang system to turn on `Mnesia` debug in the initial\nstartup phase by using the following code:\n\n```text\n% erl -mnesia debug verbose\n```","ref":"mnesia_chap5.html#debugging-mnesia-applications"},{"type":"extras","title":"Concurrent Processes in Mnesia - Miscellaneous Mnesia Features","doc":"Programming concurrent Erlang systems is the subject of a separate book.\nHowever, it is worthwhile to draw attention to the following features, which\npermit concurrent processes to exist in a `Mnesia` system:\n\n- A group of functions or processes can be called within a transaction. A\n  transaction can include statements that read, write, or delete data from the\n  DBMS. Many such transactions can run concurrently, and the programmer does not\n  need to explicitly synchronize the processes that manipulate the data.\n\n  All programs accessing the database through the transaction system can be\n  written as if they had sole access to the data. This is a desirable property,\n  as all synchronization is taken care of by the transaction handler. If a\n  program reads or writes data, the system ensures that no other program tries\n  to manipulate the same data at the same time.\n\n- Tables can be moved or deleted, and the layout of a table can be reconfigured\n  in various ways. An important aspect of the implementation of these functions\n  is that user programs can continue to use a table while it is being\n  reconfigured. For example, it is possible to move a table and perform write\n  operations to the table at the same time. This is important for many\n  applications that require continuously available services. For more\n  information, see\n  [Transactions and Other Access Contexts](mnesia_chap4.md#trans_prop).","ref":"mnesia_chap5.html#concurrent-processes-in-mnesia"},{"type":"extras","title":"Prototyping - Miscellaneous Mnesia Features","doc":"If and when you would like to start and manipulate `Mnesia`, it is often easier\nto write the definitions and data into an ordinary text file. Initially, no\ntables and no data exist, or which tables are required. At the initial stages of\nprototyping, it is prudent to write all data into one file, process that file,\nand have the data in the file inserted into the database. `Mnesia` can be\ninitialized with data read from a text file. The following two functions can be\nused to work with text files.\n\n- [`mnesia:load_textfile(Filename)`](`mnesia:load_textfile/1`) loads a series of\n  local table definitions and data found in the file into `Mnesia`. This\n  function also starts `Mnesia` and possibly creates a new schema. The function\n  operates on the local node only.\n- [`mnesia:dump_to_textfile(Filename)`](`mnesia:dump_to_textfile/1`) dumps all\n  local tables of a `Mnesia` system into a text file, which can be edited (with\n  a normal text editor) and later reloaded.\n\nThese functions are much slower than the ordinary store and load functions of\n`Mnesia`. However, this is mainly intended for minor experiments and initial\nprototyping. The major advantage of these functions is that they are easy to\nuse.\n\nThe format of the text file is as follows:\n\n```erlang\n{tables, [{Typename, [Options]},\n{Typename2 ......}]}.\n\n{Typename, Attribute1, Attribute2 ....}.\n{Typename, Attribute1, Attribute2 ....}.\n```\n\n`Options` is a list of `{Key,Value}` tuples conforming to the options that you\ncan give to `mnesia:create_table/2`.\n\nFor example, to start playing with a small database for healthy foods, enter the\nfollowing data into file `FRUITS`:\n\n```erlang\n{tables,\n [{fruit, [{attributes, [name, color, taste]}]},\n  {vegetable, [{attributes, [name, color, taste, price]}]}]}.\n\n\n{fruit, orange, orange, sweet}.\n{fruit, apple, green, sweet}.\n{vegetable, carrot, orange, carrotish, 2.55}.\n{vegetable, potato, yellow, none, 0.45}.\n```\n\nThe following session with the Erlang shell shows how to load the `FRUITS`\ndatabase:\n\n```erlang\n% erl\nErlang (BEAM) emulator version 4.9\n\nEshell V4.9  (abort with ^G)\n1> mnesia:load_textfile(\"FRUITS\").\nNew table fruit\nNew table vegetable\n{atomic,ok}\n2> mnesia:info().\n---> Processes holding locks <---\n---> Processes waiting for locks <---\n---> Pending (remote) transactions <---\n---> Active (local) transactions <---\n---> Uncertain transactions <---\n---> Active tables <---\nvegetable      : with 2 records occuping 299 words of mem\nfruit          : with 2 records occuping 291 words of mem\nschema         : with 3 records occuping 401 words of mem\n===> System info in version \"1.1\", debug level = none <===\nopt_disc. Directory \"/var/tmp/Mnesia.nonode@nohost\" is used.\nuse fallback at restart = false\nrunning db nodes = [nonode@nohost]\nstopped db nodes = []\nremote           = []\nram_copies       = [fruit,vegetable]\ndisc_copies      = [schema]\ndisc_only_copies = []\n[{nonode@nohost,disc_copies}] = [schema]\n[{nonode@nohost,ram_copies}] = [fruit,vegetable]\n3 transactions committed, 0 aborted, 0 restarted, 2 logged to disc\n0 held locks, 0 in queue; 0 local transactions, 0 remote\n0 transactions waits for other nodes: []\nok\n3>\n```\n\nIt can be seen that the DBMS was initiated from a regular text file.","ref":"mnesia_chap5.html#prototyping"},{"type":"extras","title":"Object-Based Programming with Mnesia - Miscellaneous Mnesia Features","doc":"The `Company` database, introduced in\n[Getting Started](mnesia_chap2.md#getting_started), has three tables that store\nrecords (`employee`, `dept`, `project`), and three tables that store\nrelationships (`manager`, `at_dep`, `in_proj`). This is a normalized data model,\nwhich has some advantages over a non-normalized data model.\n\nIt is more efficient to do a generalized search in a normalized database. Some\noperations are also easier to perform on a normalized data model. For example,\none project can easily be removed, as the following example illustrates:\n\n```erlang\nremove_proj(ProjName) ->\n    F = fun() ->\n                Ip = qlc:e(qlc:q([X || X <- mnesia:table(in_proj),\n\t\t\t\t       X#in_proj.proj_name == ProjName]\n\t\t\t\t)),\n                mnesia:delete({project, ProjName}),\n                del_in_projs(Ip)\n        end,\n    mnesia:transaction(F).\n\ndel_in_projs([Ip|Tail]) ->\n    mnesia:delete_object(Ip),\n    del_in_projs(Tail);\ndel_in_projs([]) ->\n    done.\n```\n\nIn reality, data models are seldom fully normalized. A realistic alternative to\na normalized database model would be a data model that is not even in first\nnormal form. `Mnesia` is suitable for applications such as telecommunications,\nbecause it is easy to organize data in a flexible manner. A `Mnesia` database is\nalways organized as a set of tables. Each table is filled with rows, objects,\nand records. What sets `Mnesia` apart is that individual fields in a record can\ncontain any type of compound data structures. An individual field in a record\ncan contain lists, tuples, functions, and even record code.\n\nMany telecommunications applications have unique requirements on lookup times\nfor certain types of records. If the `Company` database had been a part of a\ntelecommunications system, it could be to minimize the lookup time of an\nemployee _together_ with a list of the projects the employee is working on. If\nthis is the case, a drastically different data model without direct\nrelationships can be chosen. You would then have only the records themselves,\nand different records could contain either direct references to other records,\nor contain other records that are not part of the `Mnesia` schema.\n\nThe following record definitions can be created:\n\n```erlang\n-record(employee, {emp_no,\n\t\t   name,\n\t\t   salary,\n\t\t   sex,\n\t\t   phone,\n\t\t   room_no,\n\t\t   dept,\n\t\t   projects,\n\t\t   manager}).\n\n\n-record(dept, {id,\n               name}).\n\n-record(project, {name,\n                  number,\n                  location}).\n```\n\nA record that describes an employee can look as follows:\n\n```erlang\nMe = #employee{emp_no = 104732,\n               name = klacke,\n               salary = 7,\n               sex = male,\n               phone = 99586,\n               room_no = {221, 015},\n               dept = 'B/SFR',\n               projects = [erlang, mnesia, otp],\n               manager = 114872},\n```\n\nThis model has only three different tables, and the employee records contain\nreferences to other records. The record has the following references:\n\n- `'B/SFR'` refers to a `dept` record.\n- `[erlang, mnesia, otp]` is a list of three direct references to three\n  different `projects` records.\n- `114872` refers to another employee record.\n\nThe `Mnesia` record identifiers (`{Tab, Key}`) can also be used as references.\nIn this case, attribute `dept` would be set to value `{dept, 'B/SFR'}` instead\nof `'B/SFR'`.\n\nWith this data model, some operations execute considerably faster than they do\nwith the normalized data model in the `Company` database. However, some other\noperations become much more complicated. In particular, it becomes more\ndifficult to ensure that records do not contain dangling pointers to other\nnon-existent, or deleted, records.\n\nThe following code exemplifies a search with a non-normalized data model. To\nfind all employees at department `Dep` with a salary higher than `Salary`, use\nthe following code:\n\n```erlang\nget_emps(Salary, Dep) ->\n    Q = qlc:q(\n          [E || E <- mnesia:table(employee),\n                E#employee.salary > Salary,\n                E#employee.dept == Dep]\n\t ),\n    F = fun() -> qlc:e(Q) end,\n    transaction(F).\n```\n\nThis code is easier to write and to understand, and it also executes much\nfaster.\n\nIt is easy to show examples of code that executes faster if a non-normalized\ndata model is used, instead of a normalized model. The main reason is that fewer\ntables are required. Therefore, data from different tables can more easily be\ncombined in join operations. In the previous example, the function `get_emps/2`\nis transformed from a join operation into a simple query, which consists of a\nselection and a projection on one single table.","ref":"mnesia_chap5.html#object-based-programming-with-mnesia"},{"type":"extras","title":"Mnesia System Information","doc":"<!--\n%CopyrightBegin%\n\nCopyright Ericsson AB 2023-2024. All Rights Reserved.\n\nLicensed under the Apache License, Version 2.0 (the \"License\");\nyou may not use this file except in compliance with the License.\nYou may obtain a copy of the License at\n\n    http://www.apache.org/licenses/LICENSE-2.0\n\nUnless required by applicable law or agreed to in writing, software\ndistributed under the License is distributed on an \"AS IS\" BASIS,\nWITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.\nSee the License for the specific language governing permissions and\nlimitations under the License.\n\n%CopyrightEnd%\n-->\n# Mnesia System Information\n\nThe following topics are included:\n\n- Database configuration data\n- Core dumps\n- Dumping tables\n- Checkpoints\n- Startup files, log file, and data files\n- Loading tables at startup\n- Recovery from communication failure\n- Recovery of transactions\n- Backup, restore, fallback, and disaster recovery","ref":"mnesia_chap7.html"},{"type":"extras","title":"Database Configuration Data - Mnesia System Information","doc":"The following two functions can be used to retrieve system information. For\ndetails, see the Reference Manual.\n\n- [`mnesia:table_info(Tab, Key) -> Info | exit({aborted, Reason})`](`mnesia:table_info/2`)\n  returns information about one table, for example, the current size of the\n  table and on which nodes it resides.\n- [`mnesia:system_info(Key) -> Info | exit({aborted, Reason})`](`mnesia:system_info/1`)\n  returns information about the `Mnesia` system, for example, transaction\n  statistics, `db_nodes`, and configuration parameters.","ref":"mnesia_chap7.html#database-configuration-data"},{"type":"extras","title":"Core Dumps - Mnesia System Information","doc":"If `Mnesia` malfunctions, system information is dumped to file\n`MnesiaCore.Node.When`. The type of system information contained in this file\ncan also be generated with the function `mnesia_lib:coredump()`. If a `Mnesia`\nsystem behaves strangely, it is recommended that a `Mnesia` core dump file is\nincluded in the bug report.","ref":"mnesia_chap7.html#core-dumps"},{"type":"extras","title":"Dumping Tables - Mnesia System Information","doc":"Tables of type `ram_copies` are by definition stored in memory only. However,\nthese tables can be dumped to disc, either at regular intervals or before the\nsystem is shut down. The function\n[`mnesia:dump_tables(TabList)`](`mnesia:dump_tables/1`) dumps all replicas of a\nset of RAM tables to disc. The tables can be accessed while being dumped to\ndisc. To dump the tables to disc, all replicas must have the storage type\n`ram_copies`.\n\nThe table content is placed in a `.DCD` file on the disc. When the `Mnesia`\nsystem is started, the RAM table is initially loaded with data from its `.DCD`\nfile.","ref":"mnesia_chap7.html#dumping-tables"},{"type":"extras","title":"Checkpoints - Mnesia System Information","doc":"A checkpoint is a transaction consistent state that spans over one or more\ntables. When a checkpoint is activated, the system remembers the current content\nof the set of tables. The checkpoint retains a transaction consistent state of\nthe tables, allowing the tables to be read and updated while the checkpoint is\nactive. A checkpoint is typically used to back up tables to external media, but\nthey are also used internally in `Mnesia` for other purposes. Each checkpoint is\nindependent and a table can be involved in several checkpoints simultaneously.\n\nEach table retains its old contents in a checkpoint retainer. For performance\ncritical applications, it can be important to realize the processing overhead\nassociated with checkpoints. In a worst case scenario, the checkpoint retainer\nconsumes more memory than the table itself. Also, each update becomes slightly\nslower on those nodes where checkpoint retainers are attached to the tables.\n\nFor each table, it is possible to choose if there is to be one checkpoint\nretainer attached to all replicas of the table, or if it is enough to have only\none checkpoint retainer attached to a single replica. With a single checkpoint\nretainer per table, the checkpoint consumes less memory, but it is vulnerable to\nnode crashes. With several redundant checkpoint retainers, the checkpoint\nsurvives as long as there is at least one active checkpoint retainer attached to\neach table.\n\nCheckpoints can be explicitly deactivated with the function\n[`mnesia:deactivate_checkpoint(Name)`](`mnesia:deactivate_checkpoint/1`), where\n`Name` is the name of an active checkpoint. This function returns `ok` if\nsuccessful or `{error, Reason}` if there is an error. All tables in a checkpoint\nmust be attached to at least one checkpoint retainer. The checkpoint is\nautomatically deactivated by `Mnesia`, when any table lacks a checkpoint\nretainer. This can occur when a node goes down or when a replica is deleted. Use\narguments `min` and `max` (described in the following list) to control the\ndegree of checkpoint retainer redundancy.\n\n[](){: #mnesia%3Achkpt%28Args%29 }\n\nCheckpoints are activated with the function\n[`mnesia:activate_checkpoint(Args)`](`mnesia:activate_checkpoint/1`), where `Args`\nis a list of the following tuples:\n\n- `{name, Name}`, where `Name` specifies a temporary name of the checkpoint. The\n  name can be reused when the checkpoint has been deactivated. If no name is\n  specified, a name is generated automatically.\n- `{max, MaxTabs}`, where `MaxTabs` is a list of tables that are to be included\n  in the checkpoint. Default is `[]` (empty list). For these tables, the\n  redundancy is maximized. The old content of the table is retained in the\n  checkpoint retainer when the main table is updated by the applications. The\n  checkpoint is more fault tolerant if the tables have several replicas. When\n  new replicas are added by the schema manipulation function\n  `mnesia:add_table_copy/3` it also attaches a local checkpoint retainer.\n- `{min, MinTabs}`, where `MinTabs` is a list of tables that are to be included\n  in the checkpoint. Default is `[]`. For these tables, the redundancy is\n  minimized, and there is to be single checkpoint retainer per table, preferably\n  at the local node.\n- `{allow_remote, Bool}`, where `false` means that all checkpoint retainers must\n  be local. If a table does not reside locally, the checkpoint cannot be\n  activated. `true` allows checkpoint retainers to be allocated on any node.\n  Default is `true`.\n- `{ram_overrides_dump, Bool}`. This argument only applies to tables of type\n  `ram_copies`. `Bool` specifies if the table state in RAM is to override the\n  table state on disc. `true` means that the latest committed records in RAM are\n  included in the checkpoint retainer. These are the records that the\n  application accesses. `false` means that the records on the disc `.DAT` file\n  are included in the checkpoint retainer. These records are loaded on startup.\n  Default is `false`.\n\nThe function [`mnesia:activate_checkpoint(Args)`](`mnesia:activate_checkpoint/1`)\nreturns one of the following values:\n\n- `{ok, Name, Nodes}`\n- `{error, Reason}`\n\n`Name` is the checkpoint name. `Nodes` are the nodes where the checkpoint is\nknown.\n\nA list of active checkpoints can be obtained with the following functions:\n\n- [`mnesia:system_info(checkpoints)`](`mnesia:system_info/1`) returns all active\n  checkpoints on the current node.\n- [`mnesia:table_info(Tab, checkpoints)`](`mnesia:table_info/2`) returns active\n  checkpoints on a specific table.","ref":"mnesia_chap7.html#checkpoints"},{"type":"extras","title":"Startup Files, Log File, and Data Files - Mnesia System Information","doc":"This section describes the internal files that are created and maintained by the\n`Mnesia` system. In particular, the workings of the `Mnesia` log are described.","ref":"mnesia_chap7.html#startup-files-log-file-and-data-files"},{"type":"extras","title":"Startup Files - Mnesia System Information","doc":"[Start Mnesia](mnesia_chap3.md#start_mnesia) states the following prerequisites\nfor starting `Mnesia`:\n\n- An Erlang session must be started and a `Mnesia` directory must be specified\n  for the database.\n- A database schema must be initiated, using the function\n  `mnesia:create_schema/1`.\n\nThe following example shows how these tasks are performed:\n\n_Step 1:_ Start an Erlang session and specify a `Mnesia` directory for the\ndatabase:\n\n```text\n% erl -sname klacke -mnesia dir '\"/ldisc/scratch/klacke\"'\n```\n\n```erlang\nErlang (BEAM) emulator version 4.9\n\nEshell V4.9  (abort with ^G)\n(klacke@gin)1> mnesia:create_schema([node()]).\nok\n(klacke@gin)2>\n^Z\nSuspended\n```\n\n_Step 2:_ You can inspect the `Mnesia` directory to see what files have been\ncreated:\n\n```text\n% ls -l /ldisc/scratch/klacke\n-rw-rw-r--   1 klacke   staff       247 Aug 12 15:06 FALLBACK.BUP\n```\n\nThe response shows that the file `FALLBACK.BUP` has been created. This is called\na backup file, and it contains an initial schema. If more than one node in the\nfunction `mnesia:create_schema/1` had been specified, identical backup files\nwould have been created on all nodes.\n\n_Step 3:_ Start `Mnesia`:\n\n```erlang\n(klacke@gin)3>mnesia:start( ).\nok\n```\n\n_Step 4:_ You can see the following listing in the `Mnesia` directory:\n\n```text\n-rw-rw-r--   1 klacke   staff         86 May 26 19:03 LATEST.LOG\n-rw-rw-r--   1 klacke   staff      34507 May 26 19:03 schema.DAT\n```\n\nThe schema in the backup file `FALLBACK.BUP` has been used to generate the file\n`schema.DAT`. Since there are no other disc resident tables than the schema, no\nother data files were created. The file `FALLBACK.BUP` was removed after the\nsuccessful \"restoration\". You also see some files that are for internal use by\n`Mnesia`.\n\n_Step 5:_ Create a table:\n\n```erlang\n(klacke@gin)4> mnesia:create_table(foo,[{disc_copies, [node()]}]).\n{atomic,ok}\n```\n\n_Step 6:_ You can see the following listing in the `Mnesia` directory:\n\n```text\n% ls -l /ldisc/scratch/klacke\n-rw-rw-r-- 1 klacke staff    86 May 26 19:07 LATEST.LOG\n-rw-rw-r-- 1 klacke staff    94 May 26 19:07 foo.DCD\n-rw-rw-r-- 1 klacke staff  6679 May 26 19:07 schema.DAT\n```\n\nThe file `foo.DCD` has been created. This file will eventually store all data\nthat is written into the `foo` table.","ref":"mnesia_chap7.html#startup-files"},{"type":"extras","title":"Log File - Mnesia System Information","doc":"When starting `Mnesia`, a `.LOG` file called `LATEST.LOG` is created and placed\nin the database directory. This file is used by `Mnesia` to log disc-based\ntransactions. This includes all transactions that write at least one record in a\ntable that is of storage type `disc_copies` or `disc_only_copies`. The file also\nincludes all operations that manipulate the schema itself, such as creating new\ntables. The log format can vary with different implementations of `Mnesia`. The\n`Mnesia` log is currently implemented in the standard library module\n`m:disk_log` in `Kernel`.\n\nThe log file grows continuously and must be dumped at regular intervals.\n\"Dumping the log file\" means that `Mnesia` performs all the operations listed in\nthe log and place the records in the corresponding `.DAT`, `.DCD`, and `.DCL`\ndata files. For example, if the operation \"write record `{foo, 4, elvis, 6}`\" is\nlisted in the log, `Mnesia` inserts the operation into the file `foo.DCL`.\nLater, when `Mnesia` thinks that the `.DCL` file is too large, the data is moved\nto the `.DCD` file. The dumping operation can be time consuming if the log is\nlarge. Notice that the `Mnesia` system continues to operate during log dumps.\n\nBy default `Mnesia` either dumps the log whenever 1000 records have been written\nin the log or when three minutes have passed. This is controlled by the two\napplication parameters `-mnesia dump_log_write_threshold WriteOperations` and\n`-mnesia dump_log_time_threshold MilliSecs`.\n\nBefore the log is dumped, the file `LATEST.LOG` is renamed to `PREVIOUS.LOG`,\nand a new `LATEST.LOG` file is created. Once the log has been successfully\ndumped, the file `PREVIOUS.LOG` is deleted.\n\nThe log is also dumped at startup and whenever a schema operation is performed.","ref":"mnesia_chap7.html#log-file"},{"type":"extras","title":"Data Files - Mnesia System Information","doc":"The directory listing also contains one `.DAT` file, which contains the schema\nitself, contained in the `schema.DAT` file. The `DAT` files are indexed files,\nand it is efficient to insert and search for records in these files with a\nspecific key. The `.DAT` files are used for the schema and for\n`disc_only_copies` tables. The `Mnesia` data files are currently implemented in\nthe standard library module `m:dets` in `STDLIB`.\n\nAll operations that can be performed on `dets` files can also be performed on\nthe `Mnesia` data files. For example, `dets` contains the function\n`dets:traverse/2`, which can be used to view the contents of a `Mnesia` `DAT`\nfile. However, this can only be done when `Mnesia` is not running. So, to view\nthe schema file, do as follows;\n\n```erlang\n{ok, N} = dets:open_file(schema, [{file, \"./schema.DAT\"},{repair,false},\n{keypos, 2}]),\nF = fun(X) -> io:format(\"~p~n\", [X]), continue end,\ndets:traverse(N, F),\ndets:close(N).\n```\n\n> #### Warning {: .warning }\n>\n> The `DAT` files must always be opened with option `{repair, false}`. This\n> ensures that these files are not automatically repaired. Without this option,\n> the database can become inconsistent, because `Mnesia` can believe that the\n> files were properly closed. For information about configuration parameter\n> `auto_repair`, see the Reference Manual.\n\n> #### Warning {: .warning }\n>\n> It is recommended that the data files are not tampered with while `Mnesia` is\n> running. While not prohibited, the behavior of `Mnesia` is unpredictable.\n\nThe `disc_copies` tables are stored on disk with `.DCL` and `.DCD` files, which\nare standard `disk_log` files.","ref":"mnesia_chap7.html#data-files"},{"type":"extras","title":"Loading Tables at Startup - Mnesia System Information","doc":"At startup, `Mnesia` loads tables to make them accessible for its applications.\nSometimes `Mnesia` decides to load all tables that reside locally, and sometimes\nthe tables are not accessible until `Mnesia` brings a copy of the table from\nanother node.\n\nTo understand the behavior of `Mnesia` at startup, it is essential to understand\nhow `Mnesia` reacts when it loses contact with `Mnesia` on another node. At this\nstage, `Mnesia` cannot distinguish between a communication failure and a\n\"normal\" node-down. When this occurs, `Mnesia` assumes that the other node is no\nlonger running, whereas, in reality, the communication between the nodes has\nfailed.\n\nTo overcome this situation, try to restart the ongoing transactions that are\naccessing tables on the failing node, and write a `mnesia_down` entry to a log\nfile.\n\nAt startup, notice that all tables residing on nodes without a `mnesia_down`\nentry can have fresher replicas. Their replicas can have been updated after the\ntermination of `Mnesia` on the current node. To catch up with the latest\nupdates, transfer a copy of the table from one of these other \"fresh\" nodes. If\nyou are unlucky, other nodes can be down and you must wait for the table to be\nloaded on one of these nodes before receiving a fresh copy of the table.\n\nBefore an application makes its first access to a table,\n[`mnesia:wait_for_tables(TabList, Timeout)`](`mnesia:wait_for_tables/2`) is to be\nexecuted to ensure that the table is accessible from the local node. If the\nfunction times out, the application can choose to force a load of the local\nreplica with [`mnesia:force_load_table(Tab)`](`mnesia:force_load_table/1`) and\ndeliberately lose all updates that can have been performed on the other nodes\nwhile the local node was down. If `Mnesia` has loaded the table on another node\nalready, or intends to do so, copy the table from that node to avoid unnecessary\ninconsistency.\n\n> #### Warning {: .warning }\n>\n> Only one table is loaded by\n> [`mnesia:force_load_table(Tab)`](`mnesia:force_load_table/1`). Since committed\n> transactions can have caused updates in several tables, the tables can become\n> inconsistent because of the forced load.\n\nThe allowed `AccessMode` of a table can be defined to be `read_only` or\n`read_write`. It can be toggled with the function\n[`mnesia:change_table_access_mode(Tab, AccessMode)`](`mnesia:change_table_access_mode/2`)\nin runtime. `read_only` tables and `local_content` tables are always loaded\nlocally, as there is no need for copying the table from other nodes. Other\ntables are primarily loaded remotely from active replicas on other nodes if the\ntable has been loaded there already, or if the running `Mnesia` has decided to\nload the table there already.\n\nAt startup, `Mnesia` assumes that its local replica is the most recent version\nand loads the table from disc if either of the following situations is detected:\n\n- `mnesia_down` is returned from all other nodes that hold a disc resident\n  replica of the table.\n- All replicas are `ram_copies`.\n\nThis is normally a wise decision, but it can be disastrous if the nodes have\nbeen disconnected because of a communication failure, as the `Mnesia` normal\ntable load mechanism does not cope with communication failures.\n\nWhen `Mnesia` loads many tables, the default load order is used. However, the\nload order can be affected, by explicitly changing property `load_order` for the\ntables, with the function\n[`mnesia:change_table_load_order(Tab, LoadOrder)`](`mnesia:change_table_load_order/2`).\n`LoadOrder` is by default `0` for all tables, but it can be set to any integer.\nThe table with the highest `load_order` is loaded first. Changing the load order\nis especially useful for applications that need to ensure early availability of\nfundamental tables. Large peripheral tables are to have a low load order value,\nperhaps less than `0`","ref":"mnesia_chap7.html#loading-tables-at-startup"},{"type":"extras","title":"Recovery from Communication Failure - Mnesia System Information","doc":"There are several occasions when `Mnesia` can detect that the network has been\npartitioned because of a communication failure, for example:\n\n- `Mnesia` is operational already and the Erlang nodes gain contact again. Then\n  `Mnesia` tries to contact `Mnesia` on the other node to see if it also thinks\n  that the network has been partitioned for a while. If `Mnesia` on both nodes\n  has logged `mnesia_down` entries from each other, `Mnesia` generates a system\n  event, called `{inconsistent_database, running_partitioned_network, Node}`,\n  which is sent to the `Mnesia` event handler and other possible subscribers.\n  The default event handler reports an error to the error logger.\n- If `Mnesia` detects at startup that both the local node and another node\n  received `mnesia_down` from each other, `Mnesia` generates an\n  `{inconsistent_database, starting_partitioned_network, Node}` system event and\n  acts as described in the previous item.\n\nIf the application detects that there has been a communication failure that can\nhave caused an inconsistent database, it can use the function\n[`mnesia:set_master_nodes(Tab, Nodes)`](`mnesia:set_master_nodes/2`) to pinpoint\nfrom which nodes each table can be loaded.\n\nAt startup, the `Mnesia` normal table load algorithm is bypassed and the table\nis loaded from one of the master nodes defined for the table, regardless of\npotential `mnesia_down` entries in the log. `Nodes` can only contain nodes where\nthe table has a replica. If `Nodes` is empty, the master node recovery mechanism\nfor the particular table is reset and the normal load mechanism is used at the\nnext restart.\n\nThe function [`mnesia:set_master_nodes(Nodes)`](`mnesia:set_master_nodes/1`) sets\nmaster nodes for all tables. For each table it determines its replica nodes and\nstarts [`mnesia:set_master_nodes(Tab, TabNodes)`](`mnesia:set_master_nodes/2`)\nwith those replica nodes that are included in the `Nodes` list (that is,\n`TabNodes` is the intersection of `Nodes` and the replica nodes of the table).\nIf the intersection is empty, the master node recovery mechanism for the\nparticular table is reset and the normal load mechanism is used at the next\nrestart.\n\nThe functions [`mnesia:system_info(master_node_tables)`](`mnesia:system_info/1`)\nand [`mnesia:table_info(Tab, master_nodes)`](`mnesia:table_info/2`) can be used to\nobtain information about the potential master nodes.\n\nDetermining what data to keep after a communication failure is outside the scope\nof `Mnesia`. One approach is to determine which \"island\" contains most of the\nnodes. Using option `{majority,true}` for critical tables can be a way to ensure\nthat nodes that are not part of a \"majority island\" cannot update those tables.\nNotice that this constitutes a reduction in service on the minority nodes. This\nwould be a tradeoff in favor of higher consistency guarantees.\n\nThe function [`mnesia:force_load_table(Tab)`](`mnesia:force_load_table/1`) can be\nused to force load the table regardless of which table load mechanism that is\nactivated.","ref":"mnesia_chap7.html#recovery-from-communication-failure"},{"type":"extras","title":"Recovery of Transactions - Mnesia System Information","doc":"A `Mnesia` table can reside on one or more nodes. When a table is updated,\n`Mnesia` ensures that the updates are replicated to all nodes where the table\nresides. If a replica is inaccessible (for example, because of a temporary\nnode-down), `Mnesia` performs the replication later.\n\nOn the node where the application is started, there is a transaction coordinator\nprocess. If the transaction is distributed, there is also a transaction\nparticipant process on all the other nodes where commit-work needs to be\nperformed.\n\nInternally `Mnesia` uses several commit protocols. The selected protocol depends\non which table that has been updated in the transaction. If all the involved\ntables are symmetrically replicated (that is, they all have the same\n`ram_nodes`, `disc_nodes`, and `disc_only_nodes` currently accessible from the\ncoordinator node), a lightweight transaction commit protocol is used.\n\nThe number of messages that the transaction coordinator and its participants\nneed to exchange is few, as the `Mnesia` table load mechanism takes care of the\ntransaction recovery if the commit protocol gets interrupted. Since all involved\ntables are replicated symmetrically, the transaction is automatically recovered\nby loading the involved tables from the same node at startup of a failing node.\nIt does not matter if the transaction was committed or terminated as long as the\nACID properties can be ensured. The lightweight commit protocol is non-blocking,\nthat is, the surviving participants and their coordinator finish the\ntransaction, even if any node crashes in the middle of the commit protocol.\n\nIf a node goes down in the middle of a dirty operation, the table load mechanism\nensures that the update is performed on all replicas, or none. Both asynchronous\ndirty updates and synchronous dirty updates use the same recovery principle as\nlightweight transactions.\n\nIf a transaction involves updates of asymmetrically replicated tables or updates\nof the schema table, a heavyweight commit protocol is used. This protocol can\nfinish the transaction regardless of how the tables are replicated. The typical\nuse of a heavyweight transaction is when a replica is to be moved from one node\nto another. Then ensure that the replica either is entirely moved or left as it\nwas. Do never end up in a situation with replicas on both nodes, or on no node\nat all. Even if a node crashes in the middle of the commit protocol, the\ntransaction must be guaranteed to be atomic. The heavyweight commit protocol\ninvolves more messages between the transaction coordinator and its participants\nthan a lightweight protocol, and it performs recovery work at startup to finish\nthe terminating or commit work.\n\nThe heavyweight commit protocol is also non-blocking, which allows the surviving\nparticipants and their coordinator to finish the transaction regardless (even if\na node crashes in the middle of the commit protocol). When a node fails at\nstartup, `Mnesia` determines the outcome of the transaction and recovers it.\nLightweight protocols, heavyweight protocols, and dirty updates, are dependent\non other nodes to be operational to make the correct heavyweight transaction\nrecovery decision.\n\nIf `Mnesia` has not started on some of the nodes that are involved in the\ntransaction _and_ neither the local node nor any of the already running nodes\nknow the outcome of the transaction, `Mnesia` waits for one, by default. In the\nworst case scenario, all other involved nodes must start before `Mnesia` can\nmake the correct decision about the transaction and finish its startup.\n\nThus, `Mnesia` (on one node) can hang if a double fault occurs, that is, when\ntwo nodes crash simultaneously and one attempts to start when the other refuses\nto start, for example, because of a hardware error.\n\nThe maximum time that `Mnesia` waits for other nodes to respond with a\ntransaction recovery decision can be specified. The configuration parameter\n`max_wait_for_decision` defaults to `infinity`, which can cause the indefinite\nhanging as mentioned earlier. However, if the parameter is set to a definite\ntime period (for example, three minutes), `Mnesia` then enforces a transaction\nrecovery decision, if needed, to allow `Mnesia` to continue with its startup\nprocedure.\n\nThe downside of an enforced transaction recovery decision is that the decision\ncan be incorrect, because of insufficient information about the recovery\ndecisions from the other nodes. This can result in an inconsistent database\nwhere `Mnesia` has committed the transaction on some nodes but terminated it on\nothers.\n\nIn fortunate cases, the inconsistency is only visible in tables belonging to a\nspecific application. However, if a schema transaction is inconsistently\nrecovered because of the enforced transaction recovery decision, the effects of\nthe inconsistency can be fatal. However, if the higher priority is availability\nrather than consistency, it can be worth the risk.\n\nIf `Mnesia` detects an inconsistent transaction decision, an\n`{inconsistent_database, bad_decision, Node}` system event is generated to give\nthe application a chance to install a fallback or other appropriate measures to\nresolve the inconsistency. The default behavior of the `Mnesia` event handler is\nthe same as if the database became inconsistent as a result of partitioned\nnetwork (as described earlier).","ref":"mnesia_chap7.html#recovery-of-transactions"},{"type":"extras","title":"Backup, Restore, Fallback, and Disaster Recovery - Mnesia System Information","doc":"The following functions are used to back up data, to install a backup as\nfallback, and for disaster recovery:\n\n- [`mnesia:backup_checkpoint(Name, Opaque, [Mod])`](`mnesia:backup_checkpoint/2`)\n  performs a backup of the tables included in the checkpoint.\n- [`mnesia:backup(Opaque, [Mod])`](`mnesia:backup/1`) activates a new checkpoint\n  that covers all `Mnesia` tables and performs a backup. It is performed with\n  maximum degree of redundancy (see also the function\n  [`mnesia:activate_checkpoint(Args)`](mnesia_chap7.md#checkpoints),\n  `{max, MaxTabs} and {min, MinTabs})`.\n- [`mnesia:traverse_backup(Source, [SourceMod,] Target, [TargetMod,] Fun,\n  Acc)`](`mnesia:traverse_backup/4`) can be used to read an existing backup,\n  create a backup from an existing one, or to copy a backup from one type media\n  to another.\n- [`mnesia:uninstall_fallback()`](`mnesia:uninstall_fallback/0`) removes\n  previously installed fallback files.\n- [`mnesia:restore(Opaque, Args)`](`mnesia:restore/2`) restores a set of tables\n  from a previous backup.\n- [`mnesia:install_fallback(Opaque, [Mod])`](`mnesia:install_fallback/1`) can be\n  configured to restart `Mnesia` and the reload data tables, and possibly the\n  schema tables, from an existing backup. This function is typically used for\n  disaster recovery purposes, when data or schema tables are corrupted.\n\nThese functions are explained in the following sections. See also\n[Checkpoints](mnesia_chap7.md#checkpoints), which describes the two functions\nused to activate and deactivate checkpoints.","ref":"mnesia_chap7.html#backup-restore-fallback-and-disaster-recovery"},{"type":"extras","title":"Backup - Mnesia System Information","doc":"Backup operation are performed with the following functions:\n\n- [`mnesia:backup_checkpoint(Name, Opaque, [Mod])`](`mnesia:backup_checkpoint/2`)\n- [`mnesia:backup(Opaque, [Mod])`](`mnesia:backup/1`)\n- [`mnesia:traverse_backup(Source, [SourceMod,] Target, [TargetMod,] Fun,\n  Acc)`](`mnesia:traverse_backup/4`)\n\nBy default, the actual access to the backup media is performed through module\n`mnesia_backup` for both read and write. Currently `mnesia_backup` is\nimplemented with the standard library module `disk_log`. However, you can write\nyour own module with the same interface as `mnesia_backup` and configure\n`Mnesia` so that the alternative module performs the actual accesses to the\nbackup media. The user can therefore put the backup on a media that `Mnesia`\ndoes not know about, possibly on hosts where Erlang is not running. Use\nconfiguration parameter `-mnesia backup_module  ` for this purpose.\n\nThe source for a backup is an activated checkpoint. The backup function\n[`mnesia:backup_checkpoint(Name, Opaque,[Mod])`](`mnesia:backup_checkpoint/2`) is\nmost commonly used and returns `ok` or `{error,Reason}`. It has the following\narguments:\n\n- `Name` is the name of an activated checkpoint. For details on how to include\n  table names in checkpoints, see the function\n  `mnesia:activate_checkpoint(ArgList)` in\n  [Checkpoints](mnesia_chap7.md#checkpoints).\n- `Opaque`. `Mnesia` does not interpret this argument, but it is forwarded to\n  the backup module. The `Mnesia` default backup module `mnesia_backup`\n  interprets this argument as a local filename.\n- `Mod` is the name of an alternative backup module.\n\nThe function [`mnesia:backup(Opaque [,Mod])`](`mnesia:backup/1`) activates a new\ncheckpoint that covers all `Mnesia` tables with maximum degree of redundancy and\nperforms a backup. Maximum redundancy means that each table replica has a\ncheckpoint retainer. Tables with property `local_contents` are backed up as they\nlook on the current node.\n\nYou can iterate over a backup, either to transform it into a new backup, or only\nread it. The function [`mnesia:traverse_backup(Source, [SourceMod,] Target,\n[TargetMod,] Fun, Acc)`](`mnesia:traverse_backup/4`), which normally returns\n`{ok, LastAcc}`, is used for both of these purposes.\n\nBefore the traversal starts, the source backup media is opened with\n`SourceMod:open_read(Source)`, and the target backup media is opened with\n`TargetMod:open_write(Target)`. The arguments are as follows:\n\n- `SourceMod` and `TargetMod` are module names.\n- `Source` and `Target` are opaque data used exclusively by the modules\n  `SourceMod` and `TargetMod` for initializing the backup media.\n- `Acc` is an initial accumulator value.\n- `Fun(BackupItems, Acc)` is applied to each item in the backup. The Fun must\n  return a tuple `{ValGoodBackupItems, NewAcc}`, where `ValidBackupItems` is a\n  list of valid backup items. `NewAcc` is a new accumulator value. The\n  `ValidBackupItems` are written to the target backup with the function\n  `TargetMod:write/2`.\n- `LastAcc` is the last accumulator value, that is, the last `NewAcc` value that\n  was returned by `Fun`.\n\nAlso, a read-only traversal of the source backup can be performed without\nupdating a target backup. If `TargetMod==read_only`, no target backup is\naccessed.\n\nBy setting `SourceMod` and `TargetMod` to different modules, a backup can be\ncopied from one backup media to another.\n\nValid `BackupItems` are the following tuples:\n\n- `{schema, Tab}` specifies a table to be deleted.\n- `{schema, Tab, CreateList}` specifies a table to be created. For more\n  information about `CreateList`, see `mnesia:create_table/2`.\n- `{Tab, Key}` specifies the full identity of a record to be deleted.\n- `{Record}` specifies a record to be inserted. It can be a tuple with `Tab` as\n  first field. Notice that the record name is set to the table name regardless\n  of what `record_name` is set to.\n\nThe backup data is divided into two sections. The first section contains\ninformation related to the schema. All schema-related items are tuples where the\nfirst field equals the atom schema. The second section is the record section.\nSchema records cannot be mixed with other records and all schema records must be\nlocated first in the backup.\n\nThe schema itself is a table and is possibly included in the backup. Each node\nwhere the schema table resides is regarded as a `db_node`.\n\nThe following example shows how\n[`mnesia:traverse_backup`](`mnesia:traverse_backup/4`) can be used to rename a\n`db_node` in a backup file:\n\n```erlang\nchange_node_name(Mod, From, To, Source, Target) ->\n    Switch =\n        fun(Node) when Node == From -> To;\n           (Node) when Node == To -> throw({error, already_exists});\n           (Node) -> Node\n        end,\n    Convert =\n        fun({schema, db_nodes, Nodes}, Acc) ->\n                {[{schema, db_nodes, lists:map(Switch,Nodes)}], Acc};\n           ({schema, version, Version}, Acc) ->\n                {[{schema, version, Version}], Acc};\n           ({schema, cookie, Cookie}, Acc) ->\n                {[{schema, cookie, Cookie}], Acc};\n           ({schema, Tab, CreateList}, Acc) ->\n                Keys = [ram_copies, disc_copies, disc_only_copies],\n                OptSwitch =\n                    fun({Key, Val}) ->\n                            case lists:member(Key, Keys) of\n                                true -> {Key, lists:map(Switch, Val)};\n                                false-> {Key, Val}\n                            end\n                    end,\n                {[{schema, Tab, lists:map(OptSwitch, CreateList)}], Acc};\n           (Other, Acc) ->\n                {[Other], Acc}\n        end,\n    mnesia:traverse_backup(Source, Mod, Target, Mod, Convert, switched).\n\nview(Source, Mod) ->\n    View = fun(Item, Acc) ->\n                   io:format(\"~p.~n\",[Item]),\n                   {[Item], Acc + 1}\n           end,\n    mnesia:traverse_backup(Source, Mod, dummy, read_only, View, 0).\n```","ref":"mnesia_chap7.html#backup"},{"type":"extras","title":"Restore - Mnesia System Information","doc":"Tables can be restored online from a backup without restarting `Mnesia`. A\nrestore is performed with the function\n[`mnesia:restore(Opaque, Args)`](`mnesia:restore/2`), where `Args` can contain the\nfollowing tuples:\n\n- `{module,Mod}`. The backup module `Mod` is used to access the backup media. If\n  omitted, the default backup module is used.\n- `{skip_tables, TableList}`, where `TableList` is a list of tables, which is\n  not to be read from the backup.\n- `{clear_tables, TableList}`, where `TableList` is a list of tables, which is\n  to be cleared before the records from the backup are inserted. That is, all\n  records in the tables are deleted before the tables are restored. Schema\n  information about the tables is not cleared or read from the backup.\n- `{keep_tables, TableList}`, where `TableList` is a list of tables, which is\n  not to be cleared before the records from the backup are inserted. That is,\n  the records in the backup are added to the records in the table. Schema\n  information about the tables is not cleared or read from the backup.\n- `{recreate_tables, TableList}`, where `TableList` is a list of tables, which\n  is to be recreated before the records from the backup are inserted. The tables\n  are first deleted and then created with the schema information from the\n  backup. All the nodes in the backup need to be operational.\n- `{default_op, Operation}`, where `Operation` is one of the operations\n  `skip_tables`, `clear_tables`, `keep_tables`, or `recreate_tables`. The\n  default operation specifies which operation is to be used on tables from the\n  backup that are not specified in any of the previous lists. If omitted, the\n  operation `clear_tables` is used.\n\nThe argument `Opaque` is forwarded to the backup module. It returns\n`{atomic, TabList}` if successful, or the tuple `{aborted, Reason}` if there is\nan error. `TabList` is a list of the restored tables. Tables that are restored\nare write-locked during the restore operation. However, regardless of any lock\nconflict caused by this, applications can continue to do their work during the\nrestore operation.\n\nThe restoration is performed as a single transaction. If the database is large,\nit cannot always be restored online. The old database must then be restored by\ninstalling a fallback, followed by a restart.","ref":"mnesia_chap7.html#restore"},{"type":"extras","title":"Fallback - Mnesia System Information","doc":"The function [`mnesia:install_fallback(Opaque,\n[Mod])`](`mnesia:install_fallback/2`) installs a backup as fallback. It uses the\nbackup module `Mod`, or the default backup module, to access the backup media.\nThe function returns `ok` if successful, or `{error, Reason}` if there is an\nerror.\n\nInstalling a fallback is a distributed operation, which is _only_ performed on\nall `db_nodes`. The fallback restores the database the next time the system is\nstarted. If a `Mnesia` node with a fallback installed detects that `Mnesia` on\nanother node has died, it unconditionally terminates itself.\n\nA fallback is typically used when a system upgrade is performed. A system\ntypically involves the installation of new software versions, and `Mnesia`\ntables are often transformed into new layouts. If the system crashes during an\nupgrade, it is highly probable that reinstallation of the old applications is\nrequired, and restoration of the database to its previous state. This can be\ndone if a backup is performed and installed as a fallback before the system\nupgrade begins.\n\nIf the system upgrade fails, `Mnesia` must be restarted on all `db_nodes` to\nrestore the old database. The fallback is automatically deinstalled after a\nsuccessful startup. The function\n[`mnesia:uninstall_fallback()`](`mnesia:uninstall_fallback/0`) can also be used to\ndeinstall the fallback after a successful system upgrade. Again, this is a\ndistributed operation that is either performed on all `db_nodes` or none. Both\nthe installation and deinstallation of fallbacks require Erlang to be\noperational on all `db_nodes`, but it does not matter if `Mnesia` is running or\nnot.","ref":"mnesia_chap7.html#fallback"},{"type":"extras","title":"Disaster Recovery - Mnesia System Information","doc":"The system can become inconsistent as a result of a power failure. The UNIX\nfeature `fsck` can possibly repair the file system, but there is no guarantee\nthat the file content is consistent.\n\nIf `Mnesia` detects that a file has not been properly closed, possibly as a\nresult of a power failure, it tries to repair the bad file in a similar manner.\nData can be lost, but `Mnesia` can be restarted even if the data is\ninconsistent. Configuration parameter `-mnesia auto_repair  ` can be used\nto control the behavior of `Mnesia` at startup. If ` ` has the value\n`true`, `Mnesia` tries to repair the file. If ` ` has the value `false`,\n`Mnesia` does not restart if it detects a suspect file. This configuration\nparameter affects the repair behavior of log files, `DAT` files, and the default\nbackup media.\n\nConfiguration parameter `-mnesia dump_log_update_in_place  ` controls the\nsafety level of the function [`mnesia:dump_log()`](`mnesia:dump_log/0`) By\ndefault, `Mnesia` dumps the transaction log directly into the `DAT` files. If a\npower failure occurs during the dump, this can cause the randomly accessed `DAT`\nfiles to become corrupt. If the parameter is set to `false`, `Mnesia` copies the\n`DAT` files and target the dump to the new temporary files. If the dump is\nsuccessful, the temporary files are renamed to their normal `DAT` suffixes. The\npossibility for unrecoverable inconsistencies in the data files becomes much\nsmaller with this strategy. However, the actual dumping of the transaction log\nbecomes considerably slower. The system designer must decide whether speed or\nsafety is the higher priority.\n\nReplicas of type `disc_only_copies` are only affected by this parameter during\nthe initial dump of the log file at startup. When designing applications with\n_very_ high requirements, it can be appropriate not to use `disc_only_copies`\ntables at all. The reason for this is the random access nature of normal\noperating system files. If a node goes down for a reason such as a power\nfailure, these files can be corrupted because they are not properly closed. The\n`DAT` files for `disc_only_copies` are updated on a per transaction basis.\n\nIf a disaster occurs and the `Mnesia` database is corrupted, it can be\nreconstructed from a backup. Regard this as a last resort, as the backup\ncontains old data. The data is hopefully consistent, but data is definitely lost\nwhen an old backup is used to restore the database.","ref":"mnesia_chap7.html#disaster-recovery"},{"type":"extras","title":"Combine Mnesia with SNMP","doc":"<!--\n%CopyrightBegin%\n\nCopyright Ericsson AB 2023-2024. All Rights Reserved.\n\nLicensed under the Apache License, Version 2.0 (the \"License\");\nyou may not use this file except in compliance with the License.\nYou may obtain a copy of the License at\n\n    http://www.apache.org/licenses/LICENSE-2.0\n\nUnless required by applicable law or agreed to in writing, software\ndistributed under the License is distributed on an \"AS IS\" BASIS,\nWITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.\nSee the License for the specific language governing permissions and\nlimitations under the License.\n\n%CopyrightEnd%\n-->\n# Combine Mnesia with SNMP","ref":"mnesia_chap8.html"},{"type":"extras","title":"Combine Mnesia and SNMP - Combine Mnesia with SNMP","doc":"Many telecommunications applications must be controlled and reconfigured\nremotely. It is sometimes an advantage to perform this remote control with an\nopen protocol such as the Simple Network Management Protocol (SNMP). The\nalternatives to this would be the following:\n\n- Not being able to control the application remotely\n- Using a proprietary control protocol\n- Using a bridge that maps control messages in a proprietary protocol to a\n  standardized management protocol and conversely\n\nAll these approaches have different advantages and disadvantages. Mnesia\napplications can easily be opened to the SNMP protocol. A direct 1-to-1 mapping\ncan be established between Mnesia tables and SNMP tables. This means that a\nMnesia table can be configured to be _both_ a Mnesia table and an SNMP table. A\nnumber of functions to control this behavior are described in the Reference\nManual.","ref":"mnesia_chap8.html#combine-mnesia-and-snmp"},{"type":"extras","title":"Appendix A: Backup Callback Interface","doc":"<!--\n%CopyrightBegin%\n\nCopyright Ericsson AB 2023-2024. All Rights Reserved.\n\nLicensed under the Apache License, Version 2.0 (the \"License\");\nyou may not use this file except in compliance with the License.\nYou may obtain a copy of the License at\n\n    http://www.apache.org/licenses/LICENSE-2.0\n\nUnless required by applicable law or agreed to in writing, software\ndistributed under the License is distributed on an \"AS IS\" BASIS,\nWITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.\nSee the License for the specific language governing permissions and\nlimitations under the License.\n\n%CopyrightEnd%\n-->\n# Appendix A: Backup Callback Interface","ref":"mnesia_app_a.html"},{"type":"extras","title":"mnesia_backup Callback Behavior - Appendix A: Backup Callback Interface","doc":"```erlang\n%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%\n%%\n%% This module contains one implementation of callback functions\n%% used by Mnesia at backup and restore. The user may however\n%% write an own module the same interface as mnesia_backup and\n%% configure Mnesia so the alternate module performs the actual\n%% accesses to the backup media. This means that the user may put\n%% the backup on media that Mnesia does not know about, possibly\n%% on hosts where Erlang is not running.\n%%\n%% The OpaqueData argument is never interpreted by other parts of\n%% Mnesia. It is the property of this module. Alternate implementations\n%% of this module may have different interpretations of OpaqueData.\n%% The OpaqueData argument given to open_write/1 and open_read/1\n%% are forwarded directly from the user.\n%%\n%% All functions must return {ok, NewOpaqueData} or {error, Reason}.\n%%\n%% The NewOpaqueData arguments returned by backup callback functions will\n%% be given as input when the next backup callback function is invoked.\n%% If any return value does not match {ok, _} the backup will be aborted.\n%%\n%% The NewOpaqueData arguments returned by restore callback functions will\n%% be given as input when the next restore callback function is invoked\n%% If any return value does not match {ok, _} the restore will be aborted.\n%%\n%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%\n\n-module(mnesia_backup).\n\n-include_lib(\"kernel/include/file.hrl\").\n\n-export([\n\t %% Write access\n         open_write/1,\n\t write/2,\n\t commit_write/1,\n\t abort_write/1,\n\n\t %% Read access\n         open_read/1,\n\t read/1,\n\t close_read/1\n        ]).\n\n%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%\n%% Backup callback interface\n-record(backup, {tmp_file, file, file_desc}).\n\n%% Opens backup media for write\n%%\n%% Returns {ok, OpaqueData} or {error, Reason}\nopen_write(OpaqueData) ->\n    File = OpaqueData,\n    Tmp = lists:concat([File,\".BUPTMP\"]),\n    file:delete(Tmp),\n    case disk_log:open([{name, make_ref()},\n\t\t\t{file, Tmp},\n\t\t\t{repair, false},\n\t\t\t{linkto, self()}]) of\n\t{ok, Fd} ->\n\t    {ok, #backup{tmp_file = Tmp, file = File, file_desc = Fd}};\n\t{error, Reason} ->\n\t    {error, Reason}\n    end.\n\n%% Writes BackupItems to the backup media\n%%\n%% Returns {ok, OpaqueData} or {error, Reason}\nwrite(OpaqueData, BackupItems) ->\n    B = OpaqueData,\n    case disk_log:log_terms(B#backup.file_desc, BackupItems) of\n        ok ->\n            {ok, B};\n        {error, Reason} ->\n            abort_write(B),\n            {error, Reason}\n    end.\n\n%% Closes the backup media after a successful backup\n%%\n%% Returns {ok, ReturnValueToUser} or {error, Reason}\ncommit_write(OpaqueData) ->\n    B = OpaqueData,\n    case disk_log:sync(B#backup.file_desc) of\n        ok ->\n            case disk_log:close(B#backup.file_desc) of\n                ok ->\n                    file:delete(B#backup.file),\n\t\t    case file:rename(B#backup.tmp_file, B#backup.file) of\n\t\t       ok ->\n\t\t\t    {ok, B#backup.file};\n\t\t       {error, Reason} ->\n\t\t\t    {error, Reason}\n\t\t    end;\n                {error, Reason} ->\n\t\t    {error, Reason}\n            end;\n        {error, Reason} ->\n            {error, Reason}\n    end.\n\n%% Closes the backup media after an interrupted backup\n%%\n%% Returns {ok, ReturnValueToUser} or {error, Reason}\nabort_write(BackupRef) ->\n    Res = disk_log:close(BackupRef#backup.file_desc),\n    file:delete(BackupRef#backup.tmp_file),\n    case Res of\n        ok ->\n            {ok, BackupRef#backup.file};\n        {error, Reason} ->\n            {error, Reason}\n    end.\n\n%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%\n%% Restore callback interface\n\n-record(restore, {file, file_desc, cont}).\n\n%% Opens backup media for read\n%%\n%% Returns {ok, OpaqueData} or {error, Reason}\nopen_read(OpaqueData) ->\n    File = OpaqueData,\n    case file:read_file_info(File) of\n\t{error, Reason} ->\n\t    {error, Reason};\n\t_FileInfo -> %% file exists\n\t    case disk_log:open([{file, File},\n\t\t\t\t{name, make_ref()},\n\t\t\t\t{repair, false},\n\t\t\t\t{mode, read_only},\n\t\t\t\t{linkto, self()}]) of\n\t\t{ok, Fd} ->\n\t\t    {ok, #restore{file = File, file_desc = Fd, cont = start}};\n\t\t{repaired, Fd, _, {badbytes, 0}} ->\n\t\t    {ok, #restore{file = File, file_desc = Fd, cont = start}};\n\t\t{repaired, Fd, _, _} ->\n\t\t    {ok, #restore{file = File, file_desc = Fd, cont = start}};\n\t\t{error, Reason} ->\n\t\t    {error, Reason}\n\t    end\n    end.\n\n%% Reads BackupItems from the backup media\n%%\n%% Returns {ok, OpaqueData, BackupItems} or {error, Reason}\n%%\n%% BackupItems == [] is interpreted as eof\nread(OpaqueData) ->\n    R = OpaqueData,\n    Fd = R#restore.file_desc,\n    case disk_log:chunk(Fd, R#restore.cont) of\n        {error, Reason} ->\n            {error, {\"Possibly truncated\", Reason}};\n        eof ->\n            {ok, R, []};\n        {Cont, []} ->\n            read(R#restore{cont = Cont});\n        {Cont, BackupItems, _BadBytes} ->\n            {ok, R#restore{cont = Cont}, BackupItems};\n        {Cont, BackupItems} ->\n            {ok, R#restore{cont = Cont}, BackupItems}\n    end.\n\n%% Closes the backup media after restore\n%%\n%% Returns {ok, ReturnValueToUser} or {error, Reason}\nclose_read(OpaqueData) ->\n    R = OpaqueData,\n    case disk_log:close(R#restore.file_desc) of\n        ok -> {ok, R#restore.file};\n        {error, Reason} -> {error, Reason}\n    end.\n```","ref":"mnesia_app_a.html#mnesia_backup-callback-behavior"},{"type":"extras","title":"Appendix B: Activity Access Callback Interface","doc":"<!--\n%CopyrightBegin%\n\nCopyright Ericsson AB 2023-2024. All Rights Reserved.\n\nLicensed under the Apache License, Version 2.0 (the \"License\");\nyou may not use this file except in compliance with the License.\nYou may obtain a copy of the License at\n\n    http://www.apache.org/licenses/LICENSE-2.0\n\nUnless required by applicable law or agreed to in writing, software\ndistributed under the License is distributed on an \"AS IS\" BASIS,\nWITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.\nSee the License for the specific language governing permissions and\nlimitations under the License.\n\n%CopyrightEnd%\n-->\n# Appendix B: Activity Access Callback Interface","ref":"mnesia_app_b.html"},{"type":"extras","title":"mnesia_access Callback Behavior - Appendix B: Activity Access Callback Interface","doc":"```erlang\n-module(mnesia_frag).\n\n%% Callback functions when accessed within an activity\n-export([\n\t lock/4,\n\t write/5, delete/5, delete_object/5,\n\t read/5, match_object/5, all_keys/4,\n\t select/5,select/6,select_cont/3,\n\t index_match_object/6, index_read/6,\n\t foldl/6, foldr/6, table_info/4,\n\t first/3, next/4, prev/4, last/3,\n\t clear_table/4\n       ]).\n```\n\n```erlang\n%% Callback functions which provides transparent\n%% access of fragmented tables from any activity\n%% access context.\n\nlock(ActivityId, Opaque, {table , Tab}, LockKind) ->\n    case frag_names(Tab) of\n\t[Tab] ->\n\t    mnesia:lock(ActivityId, Opaque, {table, Tab}, LockKind);\n\tFrags ->\n\t    DeepNs = [mnesia:lock(ActivityId, Opaque, {table, F}, LockKind) ||\n\t\t\t F <- Frags],\n\t    mnesia_lib:uniq(lists:append(DeepNs))\n    end;\n\nlock(ActivityId, Opaque, LockItem, LockKind) ->\n    mnesia:lock(ActivityId, Opaque, LockItem, LockKind).\n\nwrite(ActivityId, Opaque, Tab, Rec, LockKind) ->\n    Frag = record_to_frag_name(Tab, Rec),\n    mnesia:write(ActivityId, Opaque, Frag, Rec, LockKind).\n\ndelete(ActivityId, Opaque, Tab, Key, LockKind) ->\n    Frag = key_to_frag_name(Tab, Key),\n    mnesia:delete(ActivityId, Opaque, Frag, Key, LockKind).\n\ndelete_object(ActivityId, Opaque, Tab, Rec, LockKind) ->\n    Frag = record_to_frag_name(Tab, Rec),\n    mnesia:delete_object(ActivityId, Opaque, Frag, Rec, LockKind).\n\nread(ActivityId, Opaque, Tab, Key, LockKind) ->\n    Frag = key_to_frag_name(Tab, Key),\n    mnesia:read(ActivityId, Opaque, Frag, Key, LockKind).\n\nmatch_object(ActivityId, Opaque, Tab, HeadPat, LockKind) ->\n    MatchSpec = [{HeadPat, [], ['$_']}],\n    select(ActivityId, Opaque, Tab, MatchSpec, LockKind).\n\nselect(ActivityId, Opaque, Tab, MatchSpec, LockKind) ->\n    do_select(ActivityId, Opaque, Tab, MatchSpec, LockKind).\n\n\nselect(ActivityId, Opaque, Tab, MatchSpec, Limit, LockKind) ->\n    init_select(ActivityId, Opaque, Tab, MatchSpec, Limit, LockKind).\n\n\nall_keys(ActivityId, Opaque, Tab, LockKind) ->\n    Match = [mnesia:all_keys(ActivityId, Opaque, Frag, LockKind)\n\t     || Frag <- frag_names(Tab)],\n    lists:append(Match).\n\nclear_table(ActivityId, Opaque, Tab, Obj) ->\n    [mnesia:clear_table(ActivityId, Opaque, Frag, Obj)  || Frag <- frag_names(Tab)],\n    ok.\n\nindex_match_object(ActivityId, Opaque, Tab, Pat, Attr, LockKind) ->\n    Match =\n\t[mnesia:index_match_object(ActivityId, Opaque, Frag, Pat, Attr, LockKind)\n\t || Frag <- frag_names(Tab)],\n    lists:append(Match).\n\nindex_read(ActivityId, Opaque, Tab, Key, Attr, LockKind) ->\n    Match =\n\t[mnesia:index_read(ActivityId, Opaque, Frag, Key, Attr, LockKind)\n\t     || Frag <- frag_names(Tab)],\n    lists:append(Match).\n\nfoldl(ActivityId, Opaque, Fun, Acc, Tab, LockKind) ->\n    Fun2 = fun(Frag, A) ->\n\t\t   mnesia:foldl(ActivityId, Opaque, Fun, A, Frag, LockKind)\n\t   end,\n    lists:foldl(Fun2, Acc, frag_names(Tab)).\n\nfoldr(ActivityId, Opaque, Fun, Acc, Tab, LockKind) ->\n    Fun2 = fun(Frag, A) ->\n\t\t   mnesia:foldr(ActivityId, Opaque, Fun, A, Frag, LockKind)\n\t   end,\n    lists:foldr(Fun2, Acc, frag_names(Tab)).\n\ntable_info(ActivityId, Opaque, {Tab, Key}, Item) ->\n    Frag = key_to_frag_name(Tab, Key),\n    table_info2(ActivityId, Opaque, Tab, Frag, Item);\ntable_info(ActivityId, Opaque, Tab, Item) ->\n    table_info2(ActivityId, Opaque, Tab, Tab, Item).\n\ntable_info2(ActivityId, Opaque, Tab, Frag, Item) ->\n    case Item of\n\tsize ->\n\t    SumFun = fun({_, Size}, Acc) -> Acc + Size end,\n\t    lists:foldl(SumFun, 0, frag_size(ActivityId, Opaque, Tab));\n\tmemory ->\n\t    SumFun = fun({_, Size}, Acc) -> Acc + Size end,\n\t    lists:foldl(SumFun, 0, frag_memory(ActivityId, Opaque, Tab));\n\tbase_table ->\n\t    lookup_prop(Tab, base_table);\n\tnode_pool ->\n\t    lookup_prop(Tab, node_pool);\n\tn_fragments ->\n\t    FH = lookup_frag_hash(Tab),\n\t    FH#frag_state.n_fragments;\n\tforeign_key ->\n\t    FH = lookup_frag_hash(Tab),\n\t    FH#frag_state.foreign_key;\n\tforeigners ->\n\t    lookup_foreigners(Tab);\n\tn_ram_copies ->\n\t    length(val({Tab, ram_copies}));\n\tn_disc_copies ->\n\t    length(val({Tab, disc_copies}));\n\tn_disc_only_copies ->\n\t    length(val({Tab, disc_only_copies}));\n\tn_external_copies ->\n\t    length(val({Tab, external_copies}));\n\n\tfrag_names ->\n\t    frag_names(Tab);\n\tfrag_dist ->\n\t    frag_dist(Tab);\n\tfrag_size ->\n\t    frag_size(ActivityId, Opaque, Tab);\n\tfrag_memory ->\n\t    frag_memory(ActivityId, Opaque, Tab);\n\t_ ->\n\t    mnesia:table_info(ActivityId, Opaque, Frag, Item)\n    end.\n\nfirst(ActivityId, Opaque, Tab) ->\n    case ?catch_val({Tab, frag_hash}) of\n\t{'EXIT', _} ->\n\t    mnesia:first(ActivityId, Opaque, Tab);\n\tFH ->\n\t    FirstFrag = Tab,\n\t    case mnesia:first(ActivityId, Opaque, FirstFrag) of\n\t\t'$end_of_table' ->\n\t\t    search_first(ActivityId, Opaque, Tab, 1, FH);\n\t\tNext ->\n\t\t    Next\n\t    end\n    end.\n\nsearch_first(ActivityId, Opaque, Tab, N, FH) when N  \n    NextN = N + 1,\n    NextFrag = n_to_frag_name(Tab, NextN),\n    case mnesia:first(ActivityId, Opaque, NextFrag) of\n\t'$end_of_table' ->\n\t    search_first(ActivityId, Opaque, Tab, NextN, FH);\n\tNext ->\n\t    Next\n    end;\nsearch_first(_ActivityId, _Opaque, _Tab, _N, _FH) ->\n    '$end_of_table'.\n\nlast(ActivityId, Opaque, Tab) ->\n    case ?catch_val({Tab, frag_hash}) of\n\t{'EXIT', _} ->\n\t    mnesia:last(ActivityId, Opaque, Tab);\n\tFH ->\n\t    LastN = FH#frag_state.n_fragments,\n\t    search_last(ActivityId, Opaque, Tab, LastN, FH)\n    end.\n\nsearch_last(ActivityId, Opaque, Tab, N, FH) when N >= 1 ->\n    Frag = n_to_frag_name(Tab, N),\n    case mnesia:last(ActivityId, Opaque, Frag) of\n\t'$end_of_table' ->\n\t    PrevN = N - 1,\n\t    search_last(ActivityId, Opaque, Tab, PrevN, FH);\n\tPrev ->\n\t    Prev\n    end;\nsearch_last(_ActivityId, _Opaque, _Tab, _N, _FH) ->\n    '$end_of_table'.\n\nprev(ActivityId, Opaque, Tab, Key) ->\n    case ?catch_val({Tab, frag_hash}) of\n\t{'EXIT', _} ->\n\t    mnesia:prev(ActivityId, Opaque, Tab, Key);\n\tFH ->\n\t    N = key_to_n(FH, Key),\n\t    Frag = n_to_frag_name(Tab, N),\n\t    case mnesia:prev(ActivityId, Opaque, Frag, Key) of\n\t\t'$end_of_table' ->\n\t\t    search_prev(ActivityId, Opaque, Tab, N);\n\t\tPrev ->\n\t\t    Prev\n\t    end\n    end.\n\nsearch_prev(ActivityId, Opaque, Tab, N) when N > 1 ->\n    PrevN = N - 1,\n    PrevFrag = n_to_frag_name(Tab, PrevN),\n    case mnesia:last(ActivityId, Opaque, PrevFrag) of\n\t'$end_of_table' ->\n\t    search_prev(ActivityId, Opaque, Tab, PrevN);\n\tPrev ->\n\t    Prev\n    end;\nsearch_prev(_ActivityId, _Opaque, _Tab, _N) ->\n    '$end_of_table'.\n\nnext(ActivityId, Opaque, Tab, Key) ->\n    case ?catch_val({Tab, frag_hash}) of\n\t{'EXIT', _} ->\n\t    mnesia:next(ActivityId, Opaque, Tab, Key);\n\tFH ->\n\t    N = key_to_n(FH, Key),\n\t    Frag = n_to_frag_name(Tab, N),\n\t    case mnesia:next(ActivityId, Opaque, Frag, Key) of\n\t\t'$end_of_table' ->\n\t\t    search_next(ActivityId, Opaque, Tab, N, FH);\n\t\tPrev ->\n\t\t    Prev\n\t    end\n    end.\n\nsearch_next(ActivityId, Opaque, Tab, N, FH) when N  \n    NextN = N + 1,\n    NextFrag = n_to_frag_name(Tab, NextN),\n    case mnesia:first(ActivityId, Opaque, NextFrag) of\n\t'$end_of_table' ->\n\t    search_next(ActivityId, Opaque, Tab, NextN, FH);\n\tNext ->\n\t    Next\n    end;\nsearch_next(_ActivityId, _Opaque, _Tab, _N, _FH) ->\n    '$end_of_table'.\n```","ref":"mnesia_app_b.html#mnesia_access-callback-behavior"},{"type":"extras","title":"Appendix C: Fragmented Table Hashing Callback Interface","doc":"<!--\n%CopyrightBegin%\n\nCopyright Ericsson AB 2023-2024. All Rights Reserved.\n\nLicensed under the Apache License, Version 2.0 (the \"License\");\nyou may not use this file except in compliance with the License.\nYou may obtain a copy of the License at\n\n    http://www.apache.org/licenses/LICENSE-2.0\n\nUnless required by applicable law or agreed to in writing, software\ndistributed under the License is distributed on an \"AS IS\" BASIS,\nWITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.\nSee the License for the specific language governing permissions and\nlimitations under the License.\n\n%CopyrightEnd%\n-->\n# Appendix C: Fragmented Table Hashing Callback Interface","ref":"mnesia_app_c.html"},{"type":"extras","title":"mnesia_frag_hash Callback Behavior - Appendix C: Fragmented Table Hashing Callback Interface","doc":"```erlang\n-module(mnesia_frag_hash).\n-compile([{nowarn_deprecated_function, [{erlang,phash,2}]}]).\n\n%% Fragmented Table Hashing callback functions\n-export([\n\t init_state/2,\n\t add_frag/1,\n\t del_frag/1,\n\t key_to_frag_number/2,\n\t match_spec_to_frag_numbers/2\n\t]).\n```\n\n```erlang\n-record(hash_state,\n\t{n_fragments,\n\t next_n_to_split,\n\t n_doubles,\n\t function}).\n\n%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%\n-spec init_state(Tab, State) -> NewState when\n      Tab :: atom(),\n      State :: term(),\n      NewState :: term().\ninit_state(_Tab, State) when State == undefined ->\n    #hash_state{n_fragments     = 1,\n\t\tnext_n_to_split = 1,\n\t\tn_doubles       = 0,\n\t\tfunction        = phash2}.\n\nconvert_old_state({hash_state, N, P, L}) ->\n    #hash_state{n_fragments     = N,\n\t\tnext_n_to_split = P,\n\t\tn_doubles       = L,\n\t\tfunction        = phash}.\n\n%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%\n\n-spec add_frag(State :: term()) -> {NewState, IterFrags, AdditionalLockFrags} when\n      NewState :: term(),\n      IterFrags :: [integer()],\n      AdditionalLockFrags :: [integer()].\nadd_frag(#hash_state{next_n_to_split = SplitN, n_doubles = L, n_fragments = N} = State) ->\n    P = SplitN + 1,\n    NewN = N + 1,\n    State2 = case power2(L) + 1 of\n\t\t P2 when P2 == P ->\n\t\t     State#hash_state{n_fragments      = NewN,\n\t\t\t\t      n_doubles        = L + 1,\n\t\t\t\t      next_n_to_split = 1};\n\t\t _ ->\n\t\t     State#hash_state{n_fragments     = NewN,\n\t\t\t\t      next_n_to_split = P}\n\t     end,\n    {State2, [SplitN], [NewN]};\nadd_frag(OldState) ->\n    State = convert_old_state(OldState),\n    add_frag(State).\n\n%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%\n\n-spec del_frag(State :: term()) -> {NewState, IterFrags, AdditionalLockFrags} when\n      NewState :: term(),\n      IterFrags :: [integer()],\n      AdditionalLockFrags :: [integer()].\ndel_frag(#hash_state{next_n_to_split = SplitN, n_doubles = L, n_fragments = N} = State) ->\n    P = SplitN - 1,\n    if\n\tP < 1 ->\n\t    L2 = L - 1,\n\t    MergeN = power2(L2),\n\t    State2 = State#hash_state{n_fragments     = N - 1,\n\t\t\t\t      next_n_to_split = MergeN,\n\t\t\t\t      n_doubles       = L2},\n\t    {State2, [N], [MergeN]};\n\ttrue ->\n\t    MergeN = P,\n\t    State2 = State#hash_state{n_fragments     = N - 1,\n\t\t\t\t      next_n_to_split = MergeN},\n\t    {State2, [N], [MergeN]}\n\tend;\ndel_frag(OldState) ->\n    State = convert_old_state(OldState),\n    del_frag(State).\n\n%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%\n-spec key_to_frag_number(State, Key) -> Fragnum when\n      State :: term(),\n      Key :: term(),\n      Fragnum :: integer().\nkey_to_frag_number(#hash_state{function = phash, n_fragments = N, n_doubles = L}, Key) ->\n    A = erlang:phash(Key, power2(L + 1)),\n    if\n\tA > N ->\n\t    A - power2(L);\n\ttrue ->\n\t    A\n    end;\nkey_to_frag_number(#hash_state{function = phash2, n_fragments = N, n_doubles = L}, Key) ->\n    A = erlang:phash2(Key, power2(L + 1)) + 1,\n    if\n\tA > N ->\n\t    A - power2(L);\n\ttrue ->\n\t    A\n    end;\nkey_to_frag_number(OldState, Key) ->\n    State = convert_old_state(OldState),\n    key_to_frag_number(State, Key).\n\n%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%\n-spec match_spec_to_frag_numbers(State, MatchSpec) -> Fragnums when\n      State :: term(),\n      MatchSpec :: ets:match_spec(),\n      Fragnums :: [integer()].\nmatch_spec_to_frag_numbers(#hash_state{n_fragments = N} = State, MatchSpec) ->\n    case MatchSpec of\n\t[{HeadPat, _, _}] when is_tuple(HeadPat), tuple_size(HeadPat) > 2 ->\n\t    KeyPat = element(2, HeadPat),\n\t    case has_var(KeyPat) of\n\t\tfalse ->\n\t\t    [key_to_frag_number(State, KeyPat)];\n\t\ttrue ->\n\t\t    lists:seq(1, N)\n\t    end;\n\t_ ->\n\t    lists:seq(1, N)\n    end;\nmatch_spec_to_frag_numbers(OldState, MatchSpec) ->\n    State = convert_old_state(OldState),\n    match_spec_to_frag_numbers(State, MatchSpec).\n\npower2(Y) ->\n    1 bsl Y. % trunc(math:pow(2, Y)).\n```","ref":"mnesia_app_c.html#mnesia_frag_hash-callback-behavior"}],"content_type":"text/plain","producer":{"name":"ex_doc","version":[48,46,51,52,46,49]}}