import logging
from typing import Dict

from multiqc.base_module import BaseMultiqcModule, ModuleNoSamplesFound

from . import stats

log = logging.getLogger(__name__)


class MultiqcModule(BaseMultiqcModule):
    """
    The module parses `bamtools stats` logs generated by Bamtools.

    Supported commands: `stats`
    """

    def __init__(self):
        # Initialise the parent object
        super(MultiqcModule, self).__init__(
            name="Bamtools",
            anchor="bamtools",
            href="https://github.com/pezmaster31/bamtools",
            info="Provides both a programmer's API and an end-user's toolkit for handling BAM files.",
            doi="10.1093/bioinformatics/btr174",
        )

        # Set up class objects to hold parsed data
        self.general_stats_headers: Dict = dict()
        self.general_stats_data: Dict = dict()
        n = dict()

        # Call submodule functions
        n["stats"] = stats.parse_reports(self)
        if n["stats"] > 0:
            log.info(f"Found {n['stats']} bamtools stats reports")

        # Exit if we didn't find anything
        if sum(n.values()) == 0:
            raise ModuleNoSamplesFound

        # Add to the General Stats table (has to be called once per MultiQC module)
        self.general_stats_addcols(self.general_stats_data, self.general_stats_headers)
