# -*- coding: utf-8 -*-
# Copyright 2011-2012 Antoine Bertin <diaoulael@gmail.com>
#
# This file is part of subliminal.
#
# subliminal is free software; you can redistribute it and/or modify it under
# the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# subliminal is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with subliminal.  If not, see <http://www.gnu.org/licenses/>.
from . import ServiceBase
from ..language import language_set, Language
from ..subtitles import get_subtitle_path, ResultSubtitle
from ..videos import Episode, Movie, UnknownVideo
import logging


logger = logging.getLogger("subliminal")


class TheSubDB(ServiceBase):
    server_url = 'http://api.thesubdb.com'
    site_url = 'http://www.thesubdb.com/'
    user_agent = 'SubDB/1.0 (subliminal/0.6; https://github.com/Diaoul/subliminal)'
    api_based = True
    # Source: http://api.thesubdb.com/?action=languages
    languages = language_set(['af', 'cs', 'da', 'de', 'en', 'es', 'fi', 'fr', 'hu', 'id', 'it',
                              'la', 'nl', 'no', 'oc', 'pl', 'pb', 'ro', 'ru', 'sl', 'sr', 'sv',
                              'tr'])
    videos = [Movie, Episode, UnknownVideo]
    require_video = True

    def list_checked(self, video, languages):
        return self.query(video.path, video.hashes['TheSubDB'], languages)

    def query(self, filepath, filehash, languages):
        r = self.session.get(self.server_url, params={'action': 'search', 'hash': filehash})
        if 404 == r.status_code or (200 == r.status_code and not r.text):
            logger.debug(u'Could not find subtitles for hash %s' % filehash)
            return []
        if 200 != r.status_code:
            logger.error(u'Request %s returned status code %d' % (r.url, r.status_code))
            return []
        available_languages = language_set(r.text.split(','))
        # this is needed because for theSubDB pt languages is Portuguese Brazil and not Portuguese #
        # So we are deleting pt language and adding pb language
        if Language('pt') in available_languages:
            available_languages = available_languages - language_set(['pt']) | language_set(['pb'])
        languages &= available_languages
        if not languages:
            logger.debug(u'Could not find subtitles for hash %s with languages %r (only %r available)' % (
                filehash, languages, available_languages))
            return []
        subtitles = []
        for language in languages:
            path = get_subtitle_path(filepath, language, self.config.multi)
            subtitle = ResultSubtitle(
                path, language, self.__class__.__name__.lower(),
                '%s?action=download&hash=%s&language=%s' % (self.server_url, filehash, language.alpha2))
            subtitles.append(subtitle)
        return subtitles


Service = TheSubDB
