/** @jsx h */
import { Component, h } from 'preact';
import './SystemComponentView.css';

class SystemComponentView extends Component {
  render() {
    const labelAndClass = this.getStatusLabelAndClass();
    const statusLabel = labelAndClass[0];
    const statusClass = labelAndClass[1];
    return (
      <div className={'system-component-view ' + statusClass}>
        <div className="icon">{this.getIcon()}</div>
        <div className="name">{this.props.children}</div>
        <div className="status-icon">{this.getStatusIcon()}</div>
        <div className="status-label">{statusLabel}</div>
      </div>
    );
  }

  getIcon() {
    if (this.props.type == 'APP_SERVER') {
      return this.getCoreServerIcon();
    } else if (this.props.type == 'PREPARATION_WORK') {
      return this.getPreparationWorkIcon();
    } else if (this.props.type == 'APP') {
      return this.getAppIcon();
    } else {
      return (<span class="glyphicon glyphicon-asterisk" aria-hidden="true"></span>);
    }
  }

  getStatusIcon() {
    if (this.props.status === 'WORKING' || this.props.status == 'DONE') {
      return (<span class="glyphicon glyphicon-ok" aria-hidden="true"></span>);
    } else if (this.props.status === 'ERROR') {
      return (<span class="glyphicon glyphicon-remove" aria-hidden="true"></span>);
    } else if (this.props.status === 'NOT_REACHED') {
      return (<span class="glyphicon glyphicon-minus-sign" aria-hidden="true"></span>);
    } else {
      return (<span class="glyphicon glyphicon-question-sign" aria-hidden="true"></span>);
    }
  }

  getStatusLabelAndClass() {
    if (this.props.status === 'WORKING') {
      return ['Working', 'working'];
    } else if (this.props.status === 'DONE') {
      return ['Done', 'done'];
    } else if (this.props.status === 'ERROR') {
      return ['Error', 'error'];
    } else if (this.props.status === 'NOT_REACHED') {
      return ['Not reached', 'not_reached'];
    } else {
      return ['Unknown', 'unknown'];
    }
  }


  getCoreServerIcon() {
    return (
      <svg width="76px" height="64px" viewBox="0 0 76 64" version="1.1">
        <defs>
          <linearGradient x1="50%" y1="0%" x2="50%" y2="100%" id="linearGradient-1">
            <stop stop-color="#888B9E" offset="0%"></stop>
            <stop stop-color="#3D4048" offset="100%"></stop>
          </linearGradient>
        </defs>
        <g id="Page-1" stroke="none" stroke-width="1" fill="none" fill-rule="evenodd">
          <g id="64">
            <path d="M40.0317529,63.6095765 L35.7398406,63.6095765 C35.4232667,63.6095765 35.1234596,63.4559002 34.9378267,63.1985725 L31.1337497,57.9027436 C31.054706,57.7938895 30.9992157,57.6694278 30.9700734,57.534961 L30.882247,57.12996 L30.7616855,57.0303105 C30.5413213,57.4329103 30.1357234,57.9359601 29.4742315,57.9871855 C29.0802108,58.0176006 27.3528268,57.9999919 26.628659,57.9903871 C26.3943224,57.9875857 26.1675708,57.9003424 25.9899221,57.7442649 L19.7913804,52.3187716 C19.6001586,52.1514886 19.4819923,51.9209742 19.4572412,51.6696494 L18.9566312,46.6511582 L16.6288347,44.5112959 C16.3917036,44.3536176 16.0352087,43.9926384 16.0352087,43.2554725 C16.0352087,42.9449183 15.9481808,40.1203162 15.8918921,38.3538393 L14.5617226,37.4321818 C14.2942516,37.2460894 14.1349666,36.941138 14.1349666,36.6165769 L14.1349666,31.794183 C14.1349666,31.3255504 14.4399634,30.7292543 15.108641,30.5283546 C15.2256097,30.4919365 16.1094618,30.1209524 17.3705678,29.5714796 C17.2300457,29.1188549 16.8963057,28.1823901 16.4711465,27.0710384 C16.3441976,26.931369 16.2352131,26.834521 16.1928968,26.8049063 C15.9326115,26.6792439 14.4806829,25.7615883 0.618097645,16.3132981 C0.469990543,16.212448 0.35182423,16.0723785 0.276373442,15.9082971 C0.194535286,15.7310091 -0.0657499724,15.0938928 0.0156889734,14.3579274 C0.0747721301,13.825263 0.486358174,11.8838993 0.660813171,11.0742974 C0.697540539,10.9070144 0.7757858,10.7525377 0.887165535,10.6264751 C1.27759342,10.1834552 1.9825992,9.81167062 2.79060129,10.0958117 C3.28522312,10.2710987 19.4967631,15.6889882 21.8333422,16.4693756 C21.855698,16.4349585 21.8808483,16.3925374 21.908793,16.3409118 C22.4165888,15.39044 24.3288072,13.7620316 24.9032711,13.2857952 C25.0697419,13.1473265 25.2721416,13.0692877 25.4889129,13.0592828 L25.8737519,13.0432748 C26.6334495,12.3209162 28.815135,10.4543897 31.5920433,9.88250579 C31.5924425,9.88130519 31.5928418,9.8801046 31.5936402,9.878904 C31.7213875,9.56514825 32.588872,5.90012889 33.5685346,1.53716307 C33.5944833,1.41750366 33.6447838,1.30224645 33.7170409,1.19579361 C33.976927,0.815204677 34.3933035,0.329763698 34.902297,0.176887809 C35.0017004,0.146872909 35.2943217,0.0588292035 37.4456672,0.0188093372 L37.4456672,0 L37.856854,0.0108053639 L37.8855972,0.411004027 L37.9003679,0.0112055626 L38.3055666,0.00520258262 L38.3059658,0.0184091385 C40.4752758,0.0580288062 40.7694939,0.146872909 40.8696958,0.176887809 C41.3774916,0.329763698 41.7942673,0.815204677 42.0537542,1.19579361 C42.1256121,1.30024546 42.1763118,1.41630307 42.2030589,1.53916406 C43.1831207,5.90132949 44.0506052,9.56634884 44.1787518,9.88170539 C44.179151,9.88210559 44.179151,9.88250579 44.179151,9.88250579 C46.957257,10.4543897 49.138144,12.3209162 49.8978417,13.0432748 L50.2810838,13.0592828 C50.4986535,13.0692877 50.7010532,13.1473265 50.8679232,13.285395 C51.4427864,13.7620316 53.355404,15.3908402 53.8628005,16.341312 C53.890346,16.3925374 53.9154963,16.4349585 53.9382513,16.4697758 C56.3255302,15.6717796 72.4943546,10.2678971 72.980593,10.0958117 C73.7885951,9.81167062 74.4936009,10.1834552 74.8848272,10.6268753 C74.9966061,10.7533381 75.0744522,10.908215 75.1103811,11.0738972 C75.2461127,11.7014087 75.6932285,13.7932471 75.7555053,14.3579274 C75.8373435,15.0970944 75.5762598,15.7322096 75.4944217,15.9094977 C75.4201685,16.0719783 75.3020022,16.2116476 75.1538951,16.3128979 C60.9707438,25.9792964 59.7787012,26.7168626 59.5623291,26.81251 C59.5435662,26.8277176 59.4329849,26.9249659 59.3000478,27.0710384 C58.8708965,28.1939958 58.5351604,29.1368639 58.397832,29.570279 C59.6617325,30.1209524 60.546383,30.4919365 60.6645493,30.5287548 C61.3316301,30.7296545 61.6362277,31.3255504 61.6362277,31.794183 L61.6362277,36.6165769 C61.6362277,36.941138 61.4773419,37.2460894 61.2106693,37.4317816 L59.8793022,38.3538393 C59.8230136,40.1203162 59.7359857,42.9449183 59.7359857,43.2554725 C59.7359857,43.9922382 59.3798899,44.3532174 59.143158,44.5112959 L56.8145631,46.6511582 L56.3139531,51.6700496 C56.2892021,51.9213744 56.1710357,52.1514886 55.9810115,52.3183714 L49.7804738,57.7450653 C49.6044219,57.9003424 49.3772711,57.9875857 49.1417369,57.9903871 C48.4187667,57.9999919 46.691782,58.0172004 46.297362,57.9871855 C45.6366686,57.9359601 45.2306715,57.4329103 45.0099081,57.0299103 L44.8893465,57.12996 L44.8019194,57.5337604 C44.7723778,57.6694278 44.7168875,57.7938895 44.6370454,57.9039442 L40.8337668,63.1977721 C40.6485332,63.4555 40.3487261,63.6095765 40.0317529,63.6095765" id="Fill-1" fill="url(#linearGradient-1)"></path>
            <path d="M49.1630548,41.072749 C49.1630548,41.072749 48.6879943,49.7638633 48.3079458,54.1452383 C48.3079458,54.1452383 48.1893803,55.0260756 47.7378732,55.1453348 C47.2867653,55.2641938 47.4292835,53.5021191 47.4292835,53.5021191 L48.0704156,41.191608 C48.0704156,41.191608 48.2368864,40.4536416 48.7119469,40.4300299 C49.3055729,40.4292295 49.1630548,41.072749 49.1630548,41.072749" id="Fill-5" fill="#F6F6F6"></path>
            <path d="M46.6571704,36.4414499 C46.6571704,36.4414499 47.500303,37.8345415 46.3601578,39.7871108 C45.2200126,41.7392798 42.5361202,40.2749529 42.5361202,40.2749529 C42.5361202,40.2749529 41.0039503,39.4297334 42.6431087,38.3700073 C44.2818678,37.3106814 45.8850973,36.4414499 45.8850973,36.4414499 C45.8850973,36.4414499 46.5058696,36.0872741 46.6571704,36.4414499" id="Fill-7" fill="#F6F6F6"></path>
            <path d="M49.2818199,37.3701109 C49.4007846,37.7270882 49.7213507,37.3701109 49.7213507,37.3701109 L51.8826764,34.8580639 L52.0607243,34.4174452 C52.0607243,34.4174452 52.1202067,33.6674729 52.084677,33.4769783 C52.0491472,33.2864838 52.1557364,33.1912365 52.1557364,33.1912365 C52.1557364,33.1912365 55.6001248,28.9887503 55.7186903,28.5957553 C55.837655,28.2027602 55.6001248,28.2980075 55.6001248,28.2980075 C52.6307969,29.940823 51.0990262,31.0481727 51.0990262,31.0481727 C51.0990262,31.0481727 48.2247105,33.2508661 48.0111328,33.4769783 C47.797156,33.7030906 48.0111328,33.9888324 48.0111328,33.9888324" id="Fill-9" fill="#F6F6F6"></path>
            <path d="M54.6340353,38.5431332 L53.398878,49.8907663 C53.398878,49.8907663 53.4304156,50.2085241 53.2088538,50.5258816 C52.9872919,50.8432392 49.376832,54.2405256 49.376832,54.2405256 C49.376832,54.2405256 49.1692425,54.4942516 49.1780252,53.637026 C49.1868078,52.7798005 49.7253428,48.747799 49.7568804,40.5885486 L49.8203549,40.0178653 C49.8203549,40.0178653 49.8518925,39.5408285 50.1684665,39.2550867 C50.4854396,38.9693448 53.6523767,35.381964 53.6523767,35.381964 C53.6523767,35.381964 53.9054761,35.1914694 53.9054761,34.8737117 L53.9054761,33.2228922 C53.9054761,33.2228922 53.8424009,32.873919 54.348999,32.4929299 C54.8555972,32.1119407 55.5841564,31.7309516 55.5841564,31.7309516 L59.0996042,32.5245456 C59.0996042,32.5245456 59.3846405,32.6197928 59.3846405,33.1596608 L59.3846405,35.0325906 C59.3846405,35.0325906 59.3527036,35.4135797 58.9726552,35.6677059 C58.5926068,35.9214318 54.6340353,38.5431332 54.6340353,38.5431332" id="Fill-11" fill="#F6F6F6"></path>
            <path d="M66.7953051,18.8092171 L46.1150026,31.3182267 C46.1150026,31.3182267 44.9668732,32.1038167 45.322969,30.873606 C45.6714797,29.4765125 46.5900631,26.5242469 46.5900631,26.5242469 C46.5900631,26.5242469 46.707431,26.0824276 46.9573368,25.9207474 C47.539785,25.6350055 58.6641853,20.1983067 59.0677871,19.9838002 C59.4083137,19.8093136 59.7013342,19.1585905 59.7013342,19.1585905 L60.5772021,17.4313331 C60.5772021,17.4313331 60.7253092,17.2576469 60.9480686,17.175206 C61.1708281,17.092765 67.5869396,14.713584 67.5869396,14.713584 C67.5869396,14.713584 68.0304624,14.459858 67.8719759,14.9677101 C67.7138885,15.4755622 67.238828,18.2061177 67.238828,18.2061177 C67.238828,18.2061177 67.1861322,18.5730999 66.7953051,18.8092171" id="Fill-13" fill="#F6F6F6"></path>
            <path d="M59.4953815,18.1187143 L59.067827,18.951928 C59.067827,18.951928 58.8542494,19.4045526 58.4027423,19.6662826 C57.9516344,19.9284127 46.7166528,25.1430013 46.7166528,25.1430013 C46.7166528,25.1430013 46.5421978,25.2622605 46.4076639,24.8096358 C46.2651457,24.262164 45.2567399,23.1428084 45.2567399,23.1428084 C45.2567399,23.1428084 44.9589289,22.7698232 45.5765076,22.5713247 C46.3366044,22.3095947 59.4953815,17.5712426 59.4953815,17.5712426 C59.4953815,17.5712426 59.8159475,17.4940042 59.4953815,18.1187143" id="Fill-15" fill="#F6F6F6"></path>
            <path d="M32.0477022,13.8564785 C29.3438494,15.0222572 27.5789397,16.754317 27.3022868,17.0252515 C27.2527847,17.0740758 27.2156581,17.0072426 27.2156581,17.0072426 L26.3154384,15.9995424 C26.0938766,15.7138005 25.6663221,16.0787817 25.6663221,16.0787817 C24.446734,17.0312545 23.7341432,18.0629667 23.544119,18.3963321 C23.3808419,18.6752706 23.6710679,18.7617135 23.6710679,18.7617135 C23.6710679,18.7617135 29.7989494,20.9363931 30.0524481,20.9840167 C30.5334967,21.127688 30.5722201,21.1100793 30.6524615,20.8267386 C30.6975723,20.6686602 32.1111767,14.4595779 32.1427143,14.1262124 C32.212177,13.7612312 32.0477022,13.8564785 32.0477022,13.8564785" id="Fill-17" fill="#F6F6F6"></path>
            <path d="M52.2271153,18.3964922 C52.0370911,18.0631267 51.3245003,17.0314146 50.1053114,16.0789418 C50.1053114,16.0789418 49.6777569,15.7139606 49.4557958,15.9993022 L48.5555761,17.0074027 C48.5555761,17.0074027 48.5188488,17.0738356 48.4693467,17.0254116 C48.1926938,16.7540769 46.4273849,15.0224173 43.7239312,13.8566386 C43.7239312,13.8566386 43.5594565,13.7613913 43.6289191,14.1263725 C43.6604568,14.459738 45.073662,20.66842 45.1187728,20.8264985 C45.1994133,21.1098392 45.2381367,21.1278481 45.7187862,20.9841768 C45.9722849,20.9365531 52.1005655,18.7614734 52.1005655,18.7614734 C52.1005655,18.7614734 52.3903924,18.6754307 52.2271153,18.3964922" id="Fill-19" fill="#F6F6F6"></path>
            <path d="M40.3746747,17.7853488 L37.8907869,17.7853488 L37.8808066,17.7853488 L35.3969188,17.7853488 C35.3969188,17.7853488 35.0168704,17.7793459 35.0168704,17.568041 C35.0168704,17.3567361 36.1570156,6.23681603 36.1570156,6.23681603 C36.1570156,6.23681603 36.3705932,6.11795702 36.3945459,6.33206331 C36.4180993,6.54656979 36.2787748,13.8209809 36.2787748,13.8209809 C36.2787748,13.8209809 36.2636048,14.0350872 36.537064,14.0470931 C36.7749935,14.0574983 37.6456716,14.0586989 37.8983719,14.0586989 L37.8983719,14.0586989 L37.9319056,14.0586989 L37.9678345,14.0586989 L37.9678345,14.0586989 C38.20776,14.0586989 38.9985961,14.0570981 39.2345295,14.0470931 C39.5075895,14.0350872 39.4928187,13.8209809 39.4928187,13.8209809 C39.4928187,13.8209809 39.3534942,6.54656979 39.3770477,6.33206331 C39.4010003,6.11795702 39.6145779,6.23681603 39.6145779,6.23681603 C39.6145779,6.23681603 40.7547232,17.3567361 40.7547232,17.568041 C40.7547232,17.7793459 40.3746747,17.7853488 40.3746747,17.7853488 M39.8760608,2.85553752 C39.8285548,2.54618395 39.5195658,2.54618395 39.5195658,2.54618395 L37.8983719,2.47454839 L37.8728224,2.47454839 L36.2520277,2.54618395 C36.2520277,2.54618395 35.9430388,2.54618395 35.8955327,2.85553752 C35.8480267,3.16529129 32.5701091,20.3806372 32.5701091,20.3806372 C32.5701091,20.3806372 32.475097,20.8332619 32.9026515,20.6903909 C34.9302417,19.6702846 37.7794072,19.5974484 37.8855972,19.5950472 C37.9921864,19.5974484 40.8409526,19.6702846 42.8685428,20.6903909 C43.2960973,20.8332619 43.2010852,20.3806372 43.2010852,20.3806372 C43.2010852,20.3806372 39.9235669,3.16529129 39.8760608,2.85553752" id="Fill-21" fill="#F6F6F6"></path>
            <path d="M33.1284449,38.3701674 C31.4896858,37.3104413 29.8860571,36.44161 29.8860571,36.44161 C29.8860571,36.44161 29.265684,36.087034 29.1143832,36.44161 C29.1143832,36.44161 28.2712506,37.8343014 29.4109966,39.7868707 C30.5511418,41.7394399 33.2354334,40.275113 33.2354334,40.275113 C33.2354334,40.275113 34.7672041,39.4298934 33.1284449,38.3701674" id="Fill-23" fill="#F6F6F6"></path>
            <path d="M24.6725673,31.0482127 C24.6725673,31.0482127 23.1403974,29.940863 20.1714688,28.2980475 C20.1714688,28.2980475 19.9339385,28.2028002 20.0529032,28.5957953 C20.1714688,28.9883902 23.6154579,33.1912765 23.6154579,33.1912765 C23.6154579,33.1912765 23.7224463,33.2865238 23.6869166,33.4770184 C23.6513868,33.6675129 23.71047,34.4174852 23.71047,34.4174852 L23.8889171,34.858104 L26.0502428,37.370151 C26.0502428,37.370151 26.3708089,37.7271282 26.4897736,37.370151 L27.7604607,33.9888725 C27.7604607,33.9888725 27.9740383,33.7031306 27.7604607,33.4770184 C27.5464839,33.2509061 24.6725673,31.0482127 24.6725673,31.0482127" id="Fill-25" fill="#F6F6F6"></path>
            <path d="M28.34231,53.5022391 L27.7011779,41.191728 C27.7011779,41.191728 27.5347072,40.4537617 27.0596466,40.4297498 C26.4660206,40.4293496 26.6085388,41.072869 26.6085388,41.072869 C26.6085388,41.072869 27.0835993,49.7639834 27.4636477,54.1449582 C27.4636477,54.1449582 27.5822132,55.0261956 28.0337203,55.1450546 C28.4848282,55.2643138 28.34231,53.5022391 28.34231,53.5022391" id="Fill-27" fill="#F6F6F6"></path>
            <path d="M26.0145933,40.5885886 L25.9511189,40.0179053 C25.9511189,40.0179053 25.9195812,39.5408685 25.6030073,39.2551267 C25.2860341,38.9693848 22.1190971,35.382004 22.1190971,35.382004 C22.1190971,35.382004 21.8659976,35.1915095 21.8659976,34.8737517 L21.8659976,33.2229322 C21.8659976,33.2229322 21.9290729,32.8735588 21.4224748,32.4925697 C20.9158766,32.1119807 20.1873174,31.7309916 20.1873174,31.7309916 L16.6718696,32.5245856 C16.6718696,32.5245856 16.3868333,32.6198329 16.3868333,33.1593007 L16.3868333,35.0326306 C16.3868333,35.0326306 16.4187701,35.4136197 16.7988185,35.6677459 C17.1788669,35.9214718 21.1374384,38.5431733 21.1374384,38.5431733 L22.3725958,49.8908064 C22.3725958,49.8908064 22.3410582,50.2081639 22.56262,50.5259216 C22.7841818,50.8432792 26.3946418,54.2405656 26.3946418,54.2405656 C26.3946418,54.2405656 26.6022312,54.4942916 26.5934486,53.637066 C26.584666,52.7798405 26.046131,48.747839 26.0145933,40.5885886" id="Fill-29" fill="#F6F6F6"></path>
            <path d="M30.448385,30.873606 C30.0998743,29.4765125 29.1816901,26.5242469 29.1816901,26.5242469 C29.1816901,26.5242469 29.063923,26.0824276 28.8140172,25.9207474 C28.231569,25.6350055 17.1075679,20.1983067 16.7035669,19.9838002 C16.3630403,19.8093136 16.070419,19.1585905 16.070419,19.1585905 L15.1941519,17.4313331 C15.1941519,17.4313331 15.0460448,17.2576469 14.8232854,17.175206 C14.6005259,17.092765 8.18441444,14.713584 8.18441444,14.713584 C8.18441444,14.713584 7.74089155,14.459858 7.89937813,14.9677101 C8.05786471,15.4755622 8.53292522,18.2061177 8.53292522,18.2061177 C8.53292522,18.2061177 8.58562101,18.5730999 8.9760489,18.8092171 L29.6563514,31.3182267 C29.6563514,31.3182267 30.8044808,32.1038167 30.448385,30.873606" id="Fill-31" fill="#F6F6F6"></path>
            <path d="M16.7036068,18.952088 C16.7036068,18.952088 16.9171844,19.4043125 17.3686915,19.6664426 C17.8197994,19.9281726 29.054781,25.1431613 29.054781,25.1431613 C29.054781,25.1431613 29.229236,25.2620204 29.36377,24.8097959 C29.5062881,24.2619239 30.5146939,23.1425682 30.5146939,23.1425682 C30.5146939,23.1425682 30.812505,22.7699833 30.1949263,22.5714847 C29.4348295,22.3093546 16.2760524,17.5710024 16.2760524,17.5710024 C16.2760524,17.5710024 15.9554863,17.4941643 16.2760524,18.1188744 L16.7036068,18.952088 Z" id="Fill-33" fill="#F6F6F6"></path>
            <path d="M37.8857169,32.1730511 C35.663312,32.1322308 33.7814336,30.3161293 33.7814336,28.0798191 C33.7814336,25.843509 35.663312,24.0274075 37.8857169,23.9865872 C40.1081219,24.0274075 41.9900003,25.843509 41.9900003,28.0798191 C41.9900003,30.3161293 40.1081219,32.1322308 37.8857169,32.1730511 M45.3549457,25.5241505 C42.9141726,21.123566 38.4194614,21.236422 37.8857169,21.2664369 C37.3519725,21.236422 32.8572612,21.123566 30.4164881,25.5241505 C30.4164881,25.5241505 30.2899384,25.8255001 30.2899384,26.0636183 C30.2899384,26.3017365 32.7287154,35.1753414 32.7287154,35.1753414 C32.7287154,35.1753414 32.7762215,35.2865967 32.9818149,35.5883465 C33.1878075,35.8896961 35.6265846,37.6677787 35.6265846,37.6677787 C35.6265846,37.6677787 35.8006404,37.9058969 36.4182191,37.9375126 C36.9483706,37.9647261 37.6869101,37.9455166 37.8857169,37.9391134 C38.0845238,37.9455166 38.8230632,37.9647261 39.3532148,37.9375126 C39.9707935,37.9058969 40.1448492,37.6677787 40.1448492,37.6677787 C40.1448492,37.6677787 42.5836263,35.8896961 42.7896189,35.5883465 C42.9952123,35.2865967 43.0427184,35.1753414 43.0427184,35.1753414 C43.0427184,35.1753414 45.4814955,26.3017365 45.4814955,26.0636183 C45.4814955,25.8255001 45.3549457,25.5241505 45.3549457,25.5241505" id="Fill-35" fill="#F6F6F6"></path>
            <path d="M46.1148429,51.795752 L46.1148429,49.6050645 C46.1148429,49.3193227 45.8138382,49.6210725 45.8138382,49.6210725 L44.3730874,50.6687926 C43.9135961,51.0337737 43.5810538,50.7640398 43.5810538,50.7640398 C43.5810538,50.7640398 42.0293225,49.1764517 41.8077607,48.8590942 C41.5861988,48.5417367 41.2851941,48.5733524 41.2851941,48.5733524 L34.4862398,48.5733524 C34.4862398,48.5733524 34.185235,48.5417367 33.9636732,48.8590942 C33.7421113,49.1764517 32.1903801,50.7640398 32.1903801,50.7640398 C32.1903801,50.7640398 31.8578377,51.0337737 31.3983464,50.6687926 L29.9575956,49.6210725 C29.9575956,49.6210725 29.6565909,49.3193227 29.6565909,49.6050645 L29.6565909,51.795752 C29.6565909,51.795752 29.6406225,52.2083568 29.9891333,52.4784909 C30.337644,52.7482248 32.7919903,54.4150522 32.7919903,54.4150522 C32.7919903,54.4150522 33.2650548,54.6807842 33.5995932,54.0024474 L35.1197868,51.255884 C35.1197868,51.255884 35.2147989,51.0337737 35.8004408,51.0177658 C36.3250034,51.0037588 39.4332565,51.0033586 39.9709931,51.0177658 C40.5566349,51.0337737 40.651647,51.255884 40.651647,51.255884 L42.1718407,54.0024474 C42.5063791,54.6807842 42.9794435,54.4150522 42.9794435,54.4150522 C42.9794435,54.4150522 45.4337898,52.7482248 45.7823006,52.4784909 C46.1308114,52.2083568 46.1148429,51.795752 46.1148429,51.795752" id="Fill-37" fill="#F6F6F6"></path>
            <path d="M45.9958383,41.7191899 L43.918746,42.6184363 C43.9159515,42.6196369 43.913157,42.6208375 43.9103625,42.6224383 C43.8640541,42.6508524 43.342286,42.9477998 42.6468612,42.4935743 C42.6468612,42.4935743 38.5517598,40.0179454 37.9796911,40.0179454 C37.9764974,40.0179454 37.9737029,40.019146 37.9705092,40.0195462 C37.9677148,40.0195462 37.9653195,40.0179454 37.9621258,40.0179454 L37.7401648,40.0179454 C37.7210027,40.0179454 37.7042358,40.0247487 37.6886666,40.0335531 C36.8447356,40.2444578 33.1244928,42.4935743 33.1244928,42.4935743 C32.4294672,42.9477998 31.9072999,42.6508524 31.8609915,42.6224383 C31.858197,42.6208375 31.8558017,42.6196369 31.852608,42.6184363 L29.7755157,41.7191899 C29.770326,41.7167887 29.0281936,41.2929783 29.0549407,41.9376983 C29.0569368,41.985322 29.4665268,47.2559384 29.4665268,47.2559384 C29.4816968,48.1847995 29.8349981,48.4489306 29.8749191,48.4757439 C29.8769152,48.4773447 29.8789112,48.4781451 29.8813065,48.4797459 L30.9819299,49.0784431 C30.9851236,49.0804441 31.5875323,49.4514282 31.8474183,49.229318 C31.8534065,49.2241154 31.8597938,49.2221144 31.865782,49.2169118 C31.9316517,49.1556814 32.3899454,48.7218661 32.7129067,48.1924033 C33.0478443,47.6429305 33.5879762,47.5925055 33.6286956,47.5893039 C33.6306917,47.5893039 33.6322885,47.5889037 33.6338854,47.5889037 L42.1374686,47.5889037 C42.1394647,47.5889037 42.1410615,47.5893039 42.1426584,47.5893039 C42.183777,47.5925055 42.7235097,47.6429305 43.0588465,48.1924033 C43.3814086,48.7218661 43.8401015,49.1556814 43.9059712,49.2169118 C43.9115602,49.2221144 43.9183467,49.2241154 43.9243349,49.229318 C44.1838217,49.4514282 44.7862304,49.0804441 44.7894241,49.0784431 L45.8904467,48.4797459 C45.892842,48.4781451 45.8944388,48.4773447 45.8968341,48.4757439 C45.9363559,48.4489306 46.2900564,48.1847995 46.3048272,47.2559384 C46.3048272,47.2559384 46.7144172,41.985322 46.7164133,41.9376983 C46.7431604,41.2929783 46.0014272,41.7167887 45.9958383,41.7191899" id="Fill-39" fill="#F6F6F6"></path>
          </g>
        </g>
      </svg>
    );
  }

  getPreparationWorkIcon() {
    return (
      <svg width="60px" height="63px" viewBox="0 0 60 63" version="1.1">
        <defs>
          <path d="M14.4973229,0.57613464 L14.4973229,0.57613464 L14.4973229,0.57613464 C15.7389717,0.483042186 16.8213633,1.4132827 16.9159945,2.65481525 L17.018698,4.0022539 C19.48808,4.31319289 21.7290427,5.43245748 23.4489057,7.1006423 L25.501023,5.70083779 L25.501023,5.70083779 C26.5297293,4.99912951 27.9324051,5.26330879 28.6353673,6.29115865 L28.6353673,6.29115865 L28.6353673,6.29115865 C29.3374289,7.31769165 29.0743928,8.71899441 28.0478598,9.42105596 C28.0472866,9.42144796 28.0467133,9.42183969 28.0461397,9.42223115 L25.9825488,10.8307522 C26.4405558,11.9204863 26.7542672,13.0895474 26.8489064,14.3315663 C26.9441298,15.5735852 26.8121023,16.7773881 26.5246796,17.921551 L28.7834642,19.0089268 L28.7834642,19.0089268 C29.9037141,19.5482136 30.3750531,20.8932569 29.8364672,22.013844 L29.8364672,22.013844 L29.8364672,22.013844 C29.2980038,23.1341764 27.9532841,23.6058744 26.8329517,23.0674109 C26.8329218,23.0673966 26.8328919,23.0673822 26.832862,23.0673679 L24.5863286,21.9874982 C23.1386994,23.8930852 21.0940254,25.3342903 18.7011726,26.0105971 L18.8039409,27.3588855 L18.8039409,27.3588855 C18.8983997,28.5981567 17.9703459,29.6793586 16.7310746,29.7738175 C16.7302653,29.7738792 16.7294558,29.7739404 16.7286464,29.7740012 L16.7286464,29.7740012 L16.7286464,29.7740012 C15.4874259,29.8672578 14.405203,28.9376026 14.3102175,27.6965132 L14.2069792,26.3475925 C11.7381814,26.0360744 9.49780284,24.9173889 7.77677152,23.2492041 L5.7247553,24.6491485 L5.7247553,24.6491485 C4.69629315,25.350795 3.29387177,25.0868194 2.59089406,24.0592667 L2.59089406,24.0592667 L2.59089406,24.0592667 C1.88884172,23.0330667 2.15161535,21.6320413 3.17781539,20.929989 C3.1783965,20.9295914 3.17897779,20.9291942 3.17955927,20.9287972 L5.24429672,19.5190942 C4.78570555,18.4305182 4.47199414,17.260878 4.37677075,16.0182801 C4.28213156,14.7768402 4.41474327,13.5736163 4.70216601,12.4282954 L2.44258307,11.3415899 L2.44258307,11.3415899 C1.32193216,10.8026332 0.850231251,9.45736218 1.38891784,8.33658138 L1.38891784,8.33658138 L1.38891784,8.33658138 C1.92748633,7.2160463 3.27245552,6.74426916 4.39299061,7.28283765 C4.39309115,7.28288597 4.39319169,7.2829343 4.39329222,7.28298264 L6.63934858,8.36292724 C8.08697778,6.4573402 10.1310676,5.01613506 12.5250888,4.33924924 L12.421943,2.99175511 L12.421943,2.99175511 C12.3270934,1.75264107 13.254703,0.671249865 14.493817,0.576400243 C14.4949856,0.576310796 14.4961542,0.576222261 14.4973229,0.57613464 Z M8.93931474,15.8092503 C9.2097959,19.3477014 12.1663945,22.0401899 15.6592819,22.0407689 C15.8292819,22.0407689 15.9998662,22.0343996 16.1716188,22.0210819 C19.8841626,21.7431476 22.6631667,18.5104243 22.3798333,14.8005804 C22.0964999,11.0901575 18.8577362,8.30965649 15.1451924,8.58874885 C11.433817,8.86726218 8.65539715,12.0994065 8.93931474,15.8092503 Z" id="path-1"></path>
          <linearGradient x1="50%" y1="15.6382541%" x2="50%" y2="140.143913%" id="linearGradient-3">
            <stop stop-color="#888B9E" offset="0%"></stop>
            <stop stop-color="#3D4048" offset="100%"></stop>
          </linearGradient>
          <polygon id="path-4" points="0.11958422 43.4272341 39.5907306 43.4272341 39.5907306 0.274518023 0.11958422 0.274518023 0.11958422 43.4272341"></polygon>
          <linearGradient x1="50%" y1="-42.489415%" x2="50%" y2="100%" id="linearGradient-6">
            <stop stop-color="#888B9E" offset="0%"></stop>
            <stop stop-color="#3D4048" offset="100%"></stop>
          </linearGradient>
          <path d="M14.9556219,14.9517072 L14.9556219,14.9517072 L14.9556219,14.9517072 C14.4428638,15.4204211 13.6473938,15.3853707 13.1778483,14.873374 L12.2481815,13.859657 C11.1014115,14.5469654 9.74374791,14.8712221 8.36505344,14.7293598 L8.13563373,15.4494351 L8.13563373,15.4494351 C7.92518674,16.1099612 7.22061964,16.4763434 6.55902211,16.2692895 L6.55902211,16.2692895 L6.55902211,16.2692895 C5.90179748,16.0636041 5.53575255,15.3640774 5.74143789,14.7068528 C5.74207508,14.7048167 5.74271751,14.7027823 5.74336515,14.7007496 L5.97278486,13.9806742 C4.76175366,13.3124739 3.8381452,12.2736944 3.29835126,11.0588899 L1.93918358,11.3619848 L1.93918358,11.3619848 C1.26646216,11.5120019 0.599500784,11.0882663 0.449483669,10.4155449 C0.449207184,10.4143051 0.448932593,10.4130648 0.448659896,10.4118241 L0.448659896,10.4118241 L0.448659896,10.4118241 C0.30009348,9.73589873 0.726203899,9.06713019 1.40160583,8.91620181 L2.7684886,8.61075198 C2.75972571,7.9541322 2.84209686,7.2853528 3.05299038,6.62352175 C3.26329971,5.96226973 3.58343724,5.36818517 3.97134111,4.83547776 L3.03641137,3.81568872 L3.03641137,3.81568872 C2.56784991,3.30459807 2.60232663,2.51043311 3.11341728,2.04187165 C3.11391608,2.04141435 3.11441526,2.04095746 3.1149148,2.04050097 L3.1149148,2.04050097 L3.1149148,2.04050097 C3.62785783,1.57176559 4.42347869,1.60688511 4.89312629,2.11899306 L5.82264737,3.13255116 C6.96941739,2.4452428 8.32708093,2.12040709 9.7051912,2.26226938 L9.93442176,1.54300535 L9.93442176,1.54300535 C10.144983,0.882320592 10.8497253,0.515855343 11.5115146,0.722918691 L11.5115146,0.722918691 L11.5115146,0.722918691 C12.1687965,0.928571712 12.534914,1.62811864 12.329261,2.28540049 C12.3286044,2.28749903 12.3279422,2.28959583 12.3272745,2.29169086 L12.098044,3.0109549 C13.308491,3.67915528 14.2320994,4.71793472 14.7724776,5.93331824 L16.1314275,5.62996705 L16.1314275,5.62996705 C16.8040917,5.4798118 17.4711186,5.90338894 17.6212738,6.57605308 C17.6215744,6.57739973 17.6218728,6.57874688 17.6221689,6.58009452 L17.6221689,6.58009452 L17.6221689,6.58009452 C17.7707262,7.2561383 17.344428,7.92497222 16.6688758,8.07574904 L15.301756,8.38087716 C15.3111031,9.03807597 15.2281478,9.70685537 15.0178385,10.3681074 C14.8075291,11.0293594 14.4873916,11.624023 14.0994877,12.1567304 L15.0341301,13.1765398 L15.0341301,13.1765398 C15.5026777,13.6877827 15.4680664,14.4820598 14.9568235,14.9506074 C14.9564232,14.9509743 14.9560227,14.9513409 14.9556219,14.9517072 Z M9.98794041,5.50599426 C8.33525962,4.98892066 6.57041386,5.90841996 6.04464055,7.55981292 C5.51828304,9.21120588 6.43079184,10.9685613 8.08288843,11.4856349 C9.73498503,12.0027085 11.5009992,11.0837882 12.0267725,9.43239525 C12.5525458,7.78042327 11.640037,6.02306786 9.98794041,5.50599426 Z" id="path-7"></path>
          <linearGradient x1="50%" y1="-80.5345186%" x2="50%" y2="100%" id="linearGradient-9">
            <stop stop-color="#888B9E" offset="0%"></stop>
            <stop stop-color="#3D4048" offset="100%"></stop>
          </linearGradient>
        </defs>
        <g id="Page-1" stroke="none" stroke-width="1" fill="none" fill-rule="evenodd">
          <g transform="translate(-1.000000, 0.000000)">
            <g id="Group-5" transform="translate(0.000000, 0.300516)">
              <mask id="mask-2" fill="white">
                <use href="#path-1"></use>
              </mask>
              <g id="Clip-4"></g>
              <polygon id="Fill-3" fill="url(#linearGradient-3)" mask="url(#mask-2)" points="-5.33017303 39.3361571 39.98915 35.933778 36.5564344 -8.9861949 -8.76288858 -5.58381586"></polygon>
            </g>
            <g id="Group-15" transform="translate(2.920963, 19.408499)">
              <g id="Group-8" transform="translate(19.278355, 0.000000)">
                <mask id="mask-5" fill="white">
                  <use href="#path-4"></use>
                </mask>
                <g id="Clip-7"></g>
                <path d="M37.1135237,9.019334 L37.2064838,9.14183334 L37.2064838,9.14183334 C38.2080665,10.4616811 37.9500618,12.3435723 36.6302141,13.345155 C36.61783,13.3545529 36.6053729,13.3638543 36.5928442,13.3730585 L33.8385953,15.39646 C35.2500046,18.3356152 35.7167744,21.6974621 35.0303481,24.9950368 L36.8128658,25.7538381 L36.8128658,25.7538381 C38.3373407,26.402793 39.0470898,28.1647059 38.3981349,29.6891808 C38.3955936,29.6951506 38.3930329,29.7011122 38.3904529,29.7070654 L38.3491533,29.8023629 L38.3491533,29.8023629 C37.6932916,31.3157406 35.9387276,32.015813 34.4211959,31.3696211 L32.603028,30.5954129 C30.6629245,33.3591221 27.8809994,35.3382459 24.7584901,36.3469158 L25.1744593,39.7221436 L25.1744593,39.7221436 C25.3771197,41.3665568 24.2083482,42.8639059 22.563935,43.0665663 C22.5547682,43.067696 22.5455963,43.0687834 22.5364196,43.0698284 L22.3493673,43.0911291 L22.3493673,43.0911291 C20.7144126,43.277311 19.2346282,42.1117484 19.0326243,40.478673 L18.605774,37.0278548 C16.9875606,36.8894667 15.3611684,36.526415 13.7856011,35.8553195 C12.2100337,35.184803 10.8260815,34.2658827 9.60979256,33.1975728 L6.76317569,35.28773 L6.76317569,35.28773 C5.44260009,36.257376 3.58833504,35.9882559 2.59789995,34.6831997 L2.50453301,34.5601739 L2.50453301,34.5601739 C1.50290079,33.2403637 1.76083499,31.3584628 3.08064517,30.3568306 C3.09311631,30.347366 3.10566136,30.3379992 3.11827915,30.3287309 L5.87212848,28.3059291 C4.46071922,25.3661949 3.99394936,22.0043479 4.68037563,18.7067733 L2.89778102,17.9477129 L2.89778102,17.9477129 C1.37337593,17.2985941 0.663816144,15.5366049 1.31293488,14.0121998 C1.3154839,14.0062137 1.3180524,14.0002358 1.32064031,13.9942663 L1.36190542,13.8990813 L1.36190542,13.8990813 C2.01792018,12.3858712 3.77242404,11.6859973 5.28984046,12.332223 L7.10769577,13.1063972 C9.04838349,10.342688 11.8297243,8.36356417 14.9522336,7.35489428 L14.5362803,3.9797948 L14.5362803,3.9797948 C14.3336199,2.3353816 15.5023914,0.838032582 17.1468046,0.635372155 C17.1560138,0.634237206 17.1652281,0.633144989 17.1744474,0.632095537 L17.362743,0.610661354 L17.362743,0.610661354 C18.998074,0.424507426 20.4780291,1.59068713 20.679491,3.22420286 L21.1049498,6.67395525 C22.7243316,6.81292239 24.3489712,7.17539504 25.9251227,7.84649057 C27.5012743,8.51758609 28.8863949,9.4370854 30.1009312,10.5042373 L32.9487859,8.41426933 L32.9487859,8.41426933 C34.2694039,7.44510151 36.1232879,7.71443894 37.1135237,9.019334 Z M15.6894847,31.6312972 C21.0576302,33.9173068 27.2600027,31.4946462 29.5477009,26.2179477 C31.8342306,20.9412492 29.3379757,14.8122189 23.9698302,12.5267883 C18.6011004,10.2407787 12.3975595,12.6657555 10.1110298,17.942454 C7.82391586,23.2185734 10.3207549,29.3458666 15.6894847,31.6312972 Z" id="Fill-6" fill="url(#linearGradient-6)" mask="url(#mask-5)"></path>
              </g>
              <g id="Group-14" transform="translate(0.000000, 15.633804)">
                <g id="Group-13">
                  <mask id="mask-8" fill="white">
                    <use href="#path-7"></use>
                  </mask>
                  <g id="Clip-12"></g>
                  <polygon id="Fill-11" fill="url(#linearGradient-9)" mask="url(#mask-8)" points="-10.0293597 16.6790689 14.1007145 24.2301963 21.7191699 0.312791892 -2.41090433 -7.23833558"></polygon>
                </g>
              </g>
            </g>
          </g>
        </g>
      </svg>
    );
  }

  getAppIcon() {
    return (
      <svg width="64px" height="64px" viewBox="0 0 64 64" version="1.1">
          <defs>
            <linearGradient x1="50%" y1="0%" x2="50%" y2="100%" id="linearGradient-1">
              <stop stop-color="#888B9E" offset="0%"></stop>
              <stop stop-color="#3D4048" offset="100%"></stop>
            </linearGradient>
          </defs>
          <g id="Page-1" stroke="none" stroke-width="1" fill="none" fill-rule="evenodd">
            <g id="Group-21">
              <path d="M16,10 C16,11.106 16.896,12 18,12 C19.104,12 20,11.106 20,10 C20,8.896 19.104,8 18,8 C16.896,8 16,8.896 16,10 Z M24,10 C24,11.106 24.896,12 26,12 C27.104,12 28,11.106 28,10 C28,8.896 27.104,8 26,8 C24.896,8 24,8.896 24,10 Z M56,60 L8,60 C5.792,60 4,58.208 4,56 L4,18 L60,18 L60,56 C60,58.208 58.208,60 56,60 Z M8,0 C3.582,0 0,3.582 0,8 L0,56 C0,60.418 3.582,64 8,64 L56,64 C60.418,64 64,60.418 64,56 L64,16 C64,11.582 60.418,8 56,8 L43.4909668,8 C33.5795898,8 38.9165039,0 28.5126953,0 L8,0 Z M8,10 C8,11.106 8.896,12 10,12 C11.104,12 12,11.106 12,10 C12,8.896 11.104,8 10,8 C8.896,8 8,8.896 8,10 Z" id="Fill-109" fill="url(#linearGradient-1)"></path>
              <path d="M15.2918884,33.9521484 L17.6908142,33.9521484 L19.0687439,40.4682617 L20.4798767,33.9521484 L22.9452087,33.9521484 L20.3221619,43 L17.8900329,43 L16.4705994,36.4091797 L15.0345642,43 L12.5775329,43 L10.0374939,33.9521484 L12.5775329,33.9521484 L13.9886658,40.4433594 L15.2918884,33.9521484 Z M30.0110331,33.9521484 L32.4099589,33.9521484 L33.7878886,40.4682617 L35.1990214,33.9521484 L37.6643534,33.9521484 L35.0413066,43 L32.6091776,43 L31.1897441,36.4091797 L29.7537089,43 L27.2966776,43 L24.7566386,33.9521484 L27.2966776,33.9521484 L28.7078105,40.4433594 L30.0110331,33.9521484 Z M44.7301778,33.9521484 L47.1291036,33.9521484 L48.5070333,40.4682617 L49.9181661,33.9521484 L52.3834981,33.9521484 L49.7604513,43 L47.3283224,43 L45.9088888,36.4091797 L44.4728536,43 L42.0158224,43 L39.4757833,33.9521484 L42.0158224,33.9521484 L43.4269552,40.4433594 L44.7301778,33.9521484 Z" id="www" fill="#5C606C"></path>
            </g>
          </g>
      </svg>
    );
  }
}

export default SystemComponentView;
