/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.security.authc;

import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.io.UncheckedIOException;
import java.security.NoSuchAlgorithmException;
import java.time.Clock;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Base64;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Function;
import java.util.stream.Collectors;
import javax.crypto.SecretKeyFactory;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.elasticsearch.ElasticsearchException;
import org.elasticsearch.ElasticsearchSecurityException;
import org.elasticsearch.ExceptionsHelper;
import org.elasticsearch.Version;
import org.elasticsearch.action.Action;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.ActionRequest;
import org.elasticsearch.action.DocWriteResponse;
import org.elasticsearch.action.bulk.BulkItemResponse;
import org.elasticsearch.action.bulk.BulkRequest;
import org.elasticsearch.action.bulk.BulkRequestBuilder;
import org.elasticsearch.action.get.GetRequest;
import org.elasticsearch.action.index.IndexAction;
import org.elasticsearch.action.index.IndexRequest;
import org.elasticsearch.action.index.IndexRequestBuilder;
import org.elasticsearch.action.search.SearchAction;
import org.elasticsearch.action.search.SearchRequest;
import org.elasticsearch.action.support.WriteRequest;
import org.elasticsearch.action.update.UpdateRequest;
import org.elasticsearch.action.update.UpdateResponse;
import org.elasticsearch.client.Client;
import org.elasticsearch.cluster.service.ClusterService;
import org.elasticsearch.common.CharArrays;
import org.elasticsearch.common.Nullable;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.UUIDs;
import org.elasticsearch.common.bytes.BytesReference;
import org.elasticsearch.common.cache.Cache;
import org.elasticsearch.common.cache.CacheBuilder;
import org.elasticsearch.common.logging.DeprecationLogger;
import org.elasticsearch.common.logging.LoggerMessageFormat;
import org.elasticsearch.common.settings.SecureString;
import org.elasticsearch.common.settings.Setting;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.common.util.concurrent.FutureUtils;
import org.elasticsearch.common.util.concurrent.ListenableFuture;
import org.elasticsearch.common.util.concurrent.ThreadContext;
import org.elasticsearch.common.xcontent.DeprecationHandler;
import org.elasticsearch.common.xcontent.NamedXContentRegistry;
import org.elasticsearch.common.xcontent.XContent;
import org.elasticsearch.common.xcontent.XContentBuilder;
import org.elasticsearch.common.xcontent.XContentFactory;
import org.elasticsearch.common.xcontent.XContentParser;
import org.elasticsearch.common.xcontent.XContentType;
import org.elasticsearch.index.query.BoolQueryBuilder;
import org.elasticsearch.index.query.QueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.license.LicenseUtils;
import org.elasticsearch.license.XPackLicenseState;
import org.elasticsearch.search.SearchService;
import org.elasticsearch.threadpool.ThreadPool;
import org.elasticsearch.xpack.core.ClientHelper;
import org.elasticsearch.xpack.core.XPackSettings;
import org.elasticsearch.xpack.core.security.ScrollHelper;
import org.elasticsearch.xpack.core.security.action.ApiKey;
import org.elasticsearch.xpack.core.security.action.CreateApiKeyRequest;
import org.elasticsearch.xpack.core.security.action.CreateApiKeyResponse;
import org.elasticsearch.xpack.core.security.action.GetApiKeyResponse;
import org.elasticsearch.xpack.core.security.action.InvalidateApiKeyResponse;
import org.elasticsearch.xpack.core.security.authc.Authentication;
import org.elasticsearch.xpack.core.security.authc.AuthenticationResult;
import org.elasticsearch.xpack.core.security.authc.support.Hasher;
import org.elasticsearch.xpack.core.security.authz.RoleDescriptor;
import org.elasticsearch.xpack.core.security.user.User;
import org.elasticsearch.xpack.security.authc.ExpiredApiKeysRemover;
import org.elasticsearch.xpack.security.support.SecurityIndexManager;

public class ApiKeyService {
    private static final Logger logger = LogManager.getLogger(ApiKeyService.class);
    private static final DeprecationLogger deprecationLogger = new DeprecationLogger(logger);
    private static final String TYPE = "doc";
    static final String API_KEY_ID_KEY = "_security_api_key_id";
    static final String API_KEY_ROLE_DESCRIPTORS_KEY = "_security_api_key_role_descriptors";
    static final String API_KEY_LIMITED_ROLE_DESCRIPTORS_KEY = "_security_api_key_limited_by_role_descriptors";
    public static final Setting<String> PASSWORD_HASHING_ALGORITHM = new Setting("xpack.security.authc.api_key.hashing.algorithm", "pbkdf2", Function.identity(), v -> {
        if (!Hasher.getAvailableAlgoStoredHash().contains(v.toLowerCase(Locale.ROOT))) {
            throw new IllegalArgumentException("Invalid algorithm: " + v + ". Valid values for password hashing are " + Hasher.getAvailableAlgoStoredHash().toString());
        }
        if (v.regionMatches(true, 0, "pbkdf2", 0, "pbkdf2".length())) {
            try {
                SecretKeyFactory.getInstance("PBKDF2withHMACSHA512");
            }
            catch (NoSuchAlgorithmException e) {
                throw new IllegalArgumentException("Support for PBKDF2WithHMACSHA512 must be available in order to use any of the PBKDF2 algorithms for the [xpack.security.authc.api_key.hashing.algorithm] setting.", e);
            }
        }
    }, new Setting.Property[]{Setting.Property.NodeScope});
    public static final Setting<TimeValue> DELETE_TIMEOUT = Setting.timeSetting((String)"xpack.security.authc.api_key.delete.timeout", (TimeValue)TimeValue.MINUS_ONE, (Setting.Property[])new Setting.Property[]{Setting.Property.NodeScope});
    public static final Setting<TimeValue> DELETE_INTERVAL = Setting.timeSetting((String)"xpack.security.authc.api_key.delete.interval", (TimeValue)TimeValue.timeValueHours((long)24L), (Setting.Property[])new Setting.Property[]{Setting.Property.NodeScope});
    public static final Setting<String> CACHE_HASH_ALGO_SETTING = Setting.simpleString((String)"xpack.security.authc.api_key.cache.hash_algo", (String)"ssha256", (Setting.Property[])new Setting.Property[]{Setting.Property.NodeScope});
    public static final Setting<TimeValue> CACHE_TTL_SETTING = Setting.timeSetting((String)"xpack.security.authc.api_key.cache.ttl", (TimeValue)TimeValue.timeValueHours((long)24L), (Setting.Property[])new Setting.Property[]{Setting.Property.NodeScope});
    public static final Setting<Integer> CACHE_MAX_KEYS_SETTING = Setting.intSetting((String)"xpack.security.authc.api_key.cache.max_keys", (int)10000, (Setting.Property[])new Setting.Property[]{Setting.Property.NodeScope});
    private final Clock clock;
    private final Client client;
    private final XPackLicenseState licenseState;
    private final SecurityIndexManager securityIndex;
    private final ClusterService clusterService;
    private final Hasher hasher;
    private final boolean enabled;
    private final Settings settings;
    private final ExpiredApiKeysRemover expiredApiKeysRemover;
    private final TimeValue deleteInterval;
    private final Cache<String, ListenableFuture<CachedApiKeyHashResult>> apiKeyAuthCache;
    private final Hasher cacheHasher;
    private final ThreadPool threadPool;
    private volatile long lastExpirationRunMs;

    public ApiKeyService(Settings settings, Clock clock, Client client, XPackLicenseState licenseState, SecurityIndexManager securityIndex, ClusterService clusterService, ThreadPool threadPool) {
        this.clock = clock;
        this.client = client;
        this.licenseState = licenseState;
        this.securityIndex = securityIndex;
        this.clusterService = clusterService;
        this.enabled = (Boolean)XPackSettings.API_KEY_SERVICE_ENABLED_SETTING.get(settings);
        this.hasher = Hasher.resolve((String)((String)PASSWORD_HASHING_ALGORITHM.get(settings)));
        this.settings = settings;
        this.deleteInterval = (TimeValue)DELETE_INTERVAL.get(settings);
        this.expiredApiKeysRemover = new ExpiredApiKeysRemover(settings, client);
        this.threadPool = threadPool;
        this.cacheHasher = Hasher.resolve((String)((String)CACHE_HASH_ALGO_SETTING.get(settings)));
        TimeValue ttl = (TimeValue)CACHE_TTL_SETTING.get(settings);
        this.apiKeyAuthCache = ttl.getNanos() > 0L ? CacheBuilder.builder().setExpireAfterWrite(ttl).setMaximumWeight((long)((Integer)CACHE_MAX_KEYS_SETTING.get(settings)).intValue()).build() : null;
    }

    public void createApiKey(Authentication authentication, CreateApiKeyRequest request, Set<RoleDescriptor> userRoles, ActionListener<CreateApiKeyResponse> listener) {
        this.ensureEnabled();
        if (authentication == null) {
            listener.onFailure((Exception)new IllegalArgumentException("authentication must be provided"));
        } else {
            this.checkDuplicateApiKeyNameAndCreateApiKey(authentication, request, userRoles, listener);
        }
    }

    private void checkDuplicateApiKeyNameAndCreateApiKey(Authentication authentication, CreateApiKeyRequest request, Set<RoleDescriptor> userRoles, ActionListener<CreateApiKeyResponse> listener) {
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery().filter((QueryBuilder)QueryBuilders.termQuery((String)"doc_type", (String)"api_key")).filter((QueryBuilder)QueryBuilders.termQuery((String)"name", (String)request.getName())).filter((QueryBuilder)QueryBuilders.termQuery((String)"api_key_invalidated", (boolean)false));
        BoolQueryBuilder expiredQuery = QueryBuilders.boolQuery().should((QueryBuilder)QueryBuilders.rangeQuery((String)"expiration_time").lte((Object)Instant.now().toEpochMilli())).should((QueryBuilder)QueryBuilders.boolQuery().mustNot((QueryBuilder)QueryBuilders.existsQuery((String)"expiration_time")));
        boolQuery.filter((QueryBuilder)expiredQuery);
        SearchRequest searchRequest = (SearchRequest)this.client.prepareSearch(new String[]{".security"}).setScroll((TimeValue)SearchService.DEFAULT_KEEPALIVE_SETTING.get(this.settings)).setQuery((QueryBuilder)boolQuery).setVersion(false).setSize(1).request();
        this.securityIndex.prepareIndexIfNeededThenExecute(arg_0 -> listener.onFailure(arg_0), () -> ClientHelper.executeAsyncWithOrigin((Client)this.client, (String)"security", (Action)SearchAction.INSTANCE, (ActionRequest)searchRequest, (ActionListener)ActionListener.wrap(indexResponse -> {
            if (indexResponse.getHits().getTotalHits() > 0L) {
                listener.onFailure((Exception)this.traceLog("create api key", new ElasticsearchSecurityException("Error creating api key as api key with name [{}] already exists", new Object[]{request.getName()})));
            } else {
                this.createApiKeyAndIndexIt(authentication, request, userRoles, listener);
            }
        }, arg_0 -> ((ActionListener)listener).onFailure(arg_0))));
    }

    private void createApiKeyAndIndexIt(Authentication authentication, CreateApiKeyRequest request, Set<RoleDescriptor> roleDescriptorSet, ActionListener<CreateApiKeyResponse> listener) {
        Instant created = this.clock.instant();
        Instant expiration = this.getApiKeyExpiration(created, request);
        SecureString apiKey = UUIDs.randomBase64UUIDSecureString();
        Version version = this.clusterService.state().nodes().getMinNodeVersion();
        if (version.before(Version.V_6_7_0)) {
            logger.warn("nodes prior to the minimum supported version for api keys {} exist in the cluster; these nodes will not be able to use api keys", (Object)Version.V_6_7_0);
        }
        try (XContentBuilder builder = this.newDocument(apiKey, request.getName(), authentication, roleDescriptorSet, created, expiration, request.getRoleDescriptors(), version);){
            IndexRequest indexRequest = (IndexRequest)((IndexRequestBuilder)this.client.prepareIndex(".security", TYPE).setSource(builder).setRefreshPolicy(request.getRefreshPolicy())).request();
            this.securityIndex.prepareIndexIfNeededThenExecute(arg_0 -> listener.onFailure(arg_0), () -> ClientHelper.executeAsyncWithOrigin((Client)this.client, (String)"security", (Action)IndexAction.INSTANCE, (ActionRequest)indexRequest, (ActionListener)ActionListener.wrap(indexResponse -> listener.onResponse((Object)new CreateApiKeyResponse(request.getName(), indexResponse.getId(), apiKey, expiration)), arg_0 -> ((ActionListener)listener).onFailure(arg_0))));
        }
        catch (IOException e) {
            listener.onFailure((Exception)e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    XContentBuilder newDocument(SecureString apiKey, String name, Authentication authentication, Set<RoleDescriptor> userRoles, Instant created, Instant expiration, List<RoleDescriptor> keyRoles, Version version) throws IOException {
        XContentBuilder builder = XContentFactory.jsonBuilder();
        builder.startObject().field("doc_type", "api_key").field("creation_time", created.toEpochMilli()).field("expiration_time", expiration == null ? null : Long.valueOf(expiration.toEpochMilli())).field("api_key_invalidated", false);
        byte[] utf8Bytes = null;
        char[] keyHash = this.hasher.hash(apiKey);
        try {
            utf8Bytes = CharArrays.toUtf8Bytes((char[])keyHash);
            builder.field("api_key_hash").utf8Value(utf8Bytes, 0, utf8Bytes.length);
        }
        finally {
            if (utf8Bytes != null) {
                Arrays.fill(utf8Bytes, (byte)0);
            }
            Arrays.fill(keyHash, '\u0000');
        }
        builder.startObject("role_descriptors");
        if (keyRoles != null && !keyRoles.isEmpty()) {
            for (RoleDescriptor descriptor : keyRoles) {
                builder.field(descriptor.getName(), (contentBuilder, params) -> descriptor.toXContent(contentBuilder, params, true));
            }
        }
        builder.endObject();
        builder.startObject("limited_by_role_descriptors");
        for (RoleDescriptor descriptor : userRoles) {
            builder.field(descriptor.getName(), (contentBuilder, params) -> descriptor.toXContent(contentBuilder, params, true));
        }
        builder.endObject();
        builder.field("name", name).field("version", version.id).startObject("creator").field("principal", authentication.getUser().principal()).field("metadata", authentication.getUser().metadata()).field("realm", authentication.getLookedUpBy() == null ? authentication.getAuthenticatedBy().getName() : authentication.getLookedUpBy().getName()).endObject().endObject();
        return builder;
    }

    void authenticateWithApiKeyIfPresent(ThreadContext ctx, ActionListener<AuthenticationResult> listener) {
        if (this.isEnabled()) {
            ApiKeyCredentials credentials;
            try {
                credentials = ApiKeyService.getCredentialsFromHeader(ctx);
            }
            catch (IllegalArgumentException iae) {
                listener.onResponse((Object)AuthenticationResult.unsuccessful((String)iae.getMessage(), (Exception)iae));
                return;
            }
            if (credentials != null) {
                this.loadApiKeyAndValidateCredentials(ctx, credentials, (ActionListener<AuthenticationResult>)ActionListener.wrap(response -> {
                    credentials.close();
                    listener.onResponse(response);
                }, e -> {
                    credentials.close();
                    listener.onFailure(e);
                }));
            } else {
                listener.onResponse((Object)AuthenticationResult.notHandled());
            }
        } else {
            listener.onResponse((Object)AuthenticationResult.notHandled());
        }
    }

    private void loadApiKeyAndValidateCredentials(ThreadContext ctx, ApiKeyCredentials credentials, ActionListener<AuthenticationResult> listener) {
        String docId = credentials.getId();
        GetRequest getRequest = (GetRequest)this.client.prepareGet(".security", TYPE, docId).setFetchSource(true).request();
        ClientHelper.executeAsyncWithOrigin((ThreadContext)ctx, (String)"security", (ActionRequest)getRequest, (ActionListener)ActionListener.wrap(response -> {
            if (response.isExists()) {
                Map source = response.getSource();
                this.validateApiKeyCredentials(docId, source, credentials, this.clock, listener);
            } else {
                listener.onResponse((Object)AuthenticationResult.unsuccessful((String)("unable to find apikey with id " + credentials.getId()), null));
            }
        }, e -> listener.onResponse((Object)AuthenticationResult.unsuccessful((String)("apikey authentication for id " + credentials.getId() + " encountered a failure"), (Exception)e))), (arg_0, arg_1) -> ((Client)this.client).get(arg_0, arg_1));
    }

    public void getRoleForApiKey(Authentication authentication, ActionListener<ApiKeyRoleDescriptors> listener) {
        if (authentication.getAuthenticationType() != Authentication.AuthenticationType.API_KEY) {
            throw new IllegalStateException("authentication type must be api key but is " + authentication.getAuthenticationType());
        }
        Map metadata = authentication.getMetadata();
        String apiKeyId = (String)metadata.get(API_KEY_ID_KEY);
        Map roleDescriptors = (Map)metadata.get(API_KEY_ROLE_DESCRIPTORS_KEY);
        Map authnRoleDescriptors = (Map)metadata.get(API_KEY_LIMITED_ROLE_DESCRIPTORS_KEY);
        if (roleDescriptors == null && authnRoleDescriptors == null) {
            listener.onFailure((Exception)new ElasticsearchSecurityException("no role descriptors found for API key", new Object[0]));
        } else if (roleDescriptors == null || roleDescriptors.isEmpty()) {
            List<RoleDescriptor> authnRoleDescriptorsList = this.parseRoleDescriptors(apiKeyId, authnRoleDescriptors);
            listener.onResponse((Object)new ApiKeyRoleDescriptors(apiKeyId, authnRoleDescriptorsList, null));
        } else {
            List<RoleDescriptor> roleDescriptorList = this.parseRoleDescriptors(apiKeyId, roleDescriptors);
            List<RoleDescriptor> authnRoleDescriptorsList = this.parseRoleDescriptors(apiKeyId, authnRoleDescriptors);
            listener.onResponse((Object)new ApiKeyRoleDescriptors(apiKeyId, roleDescriptorList, authnRoleDescriptorsList));
        }
    }

    private List<RoleDescriptor> parseRoleDescriptors(String apiKeyId, Map<String, Object> roleDescriptors) {
        if (roleDescriptors == null) {
            return null;
        }
        return roleDescriptors.entrySet().stream().map(entry -> {
            String name = (String)entry.getKey();
            Map rdMap = (Map)entry.getValue();
            try (XContentBuilder builder = XContentBuilder.builder((XContent)XContentType.JSON.xContent());){
                RoleDescriptor roleDescriptor;
                block14: {
                    builder.map(rdMap);
                    XContentParser parser = XContentType.JSON.xContent().createParser(NamedXContentRegistry.EMPTY, (DeprecationHandler)new ApiKeyLoggingDeprecationHandler(deprecationLogger, apiKeyId), (InputStream)BytesReference.bytes((XContentBuilder)builder).streamInput());
                    try {
                        roleDescriptor = RoleDescriptor.parse((String)name, (XContentParser)parser, (boolean)false);
                        if (parser == null) break block14;
                    }
                    catch (Throwable throwable) {
                        if (parser != null) {
                            try {
                                parser.close();
                            }
                            catch (Throwable throwable2) {
                                throwable.addSuppressed(throwable2);
                            }
                        }
                        throw throwable;
                    }
                    parser.close();
                }
                return roleDescriptor;
            }
            catch (IOException e) {
                throw new UncheckedIOException(e);
            }
        }).collect(Collectors.toList());
    }

    void validateApiKeyCredentials(String docId, Map<String, Object> source, ApiKeyCredentials credentials, Clock clock, ActionListener<AuthenticationResult> listener) {
        String docType = (String)source.get("doc_type");
        Boolean invalidated = (Boolean)source.get("api_key_invalidated");
        if (!"api_key".equals(docType)) {
            listener.onResponse((Object)AuthenticationResult.unsuccessful((String)("document [" + docId + "] is [" + docType + "] not an api key"), null));
        } else if (invalidated == null) {
            listener.onResponse((Object)AuthenticationResult.unsuccessful((String)"api key document is missing invalidated field", null));
        } else if (invalidated.booleanValue()) {
            listener.onResponse((Object)AuthenticationResult.unsuccessful((String)"api key has been invalidated", null));
        } else {
            String apiKeyHash = (String)source.get("api_key_hash");
            if (apiKeyHash == null) {
                throw new IllegalStateException("api key hash is missing");
            }
            if (this.apiKeyAuthCache != null) {
                ListenableFuture listenableCacheEntry;
                AtomicBoolean valueAlreadyInCache = new AtomicBoolean(true);
                try {
                    listenableCacheEntry = (ListenableFuture)this.apiKeyAuthCache.computeIfAbsent((Object)credentials.getId(), k -> {
                        valueAlreadyInCache.set(false);
                        return new ListenableFuture();
                    });
                }
                catch (ExecutionException e) {
                    listener.onFailure((Exception)e);
                    return;
                }
                if (valueAlreadyInCache.get()) {
                    listenableCacheEntry.addListener(ActionListener.wrap(result -> {
                        if (result.success) {
                            if (((CachedApiKeyHashResult)result).verify(credentials.getKey())) {
                                this.validateApiKeyExpiration(source, credentials, clock, listener);
                            } else {
                                listener.onResponse((Object)AuthenticationResult.unsuccessful((String)"invalid credentials", null));
                            }
                        } else if (((CachedApiKeyHashResult)result).verify(credentials.getKey())) {
                            listener.onResponse((Object)AuthenticationResult.unsuccessful((String)"invalid credentials", null));
                        } else {
                            this.apiKeyAuthCache.invalidate((Object)credentials.getId(), (Object)listenableCacheEntry);
                            this.validateApiKeyCredentials(docId, source, credentials, clock, listener);
                        }
                    }, arg_0 -> listener.onFailure(arg_0)), this.threadPool.generic(), this.threadPool.getThreadContext());
                } else {
                    boolean verified = this.verifyKeyAgainstHash(apiKeyHash, credentials);
                    listenableCacheEntry.onResponse((Object)new CachedApiKeyHashResult(verified, credentials.getKey()));
                    if (verified) {
                        this.validateApiKeyExpiration(source, credentials, clock, listener);
                    } else {
                        listener.onResponse((Object)AuthenticationResult.unsuccessful((String)"invalid credentials", null));
                    }
                }
            } else {
                boolean verified = this.verifyKeyAgainstHash(apiKeyHash, credentials);
                if (verified) {
                    this.validateApiKeyExpiration(source, credentials, clock, listener);
                } else {
                    listener.onResponse((Object)AuthenticationResult.unsuccessful((String)"invalid credentials", null));
                }
            }
        }
    }

    CachedApiKeyHashResult getFromCache(String id) {
        return this.apiKeyAuthCache == null ? null : (CachedApiKeyHashResult)FutureUtils.get((Future)((Future)this.apiKeyAuthCache.get((Object)id)), (long)0L, (TimeUnit)TimeUnit.MILLISECONDS);
    }

    private void validateApiKeyExpiration(Map<String, Object> source, ApiKeyCredentials credentials, Clock clock, ActionListener<AuthenticationResult> listener) {
        Long expirationEpochMilli = (Long)source.get("expiration_time");
        if (expirationEpochMilli == null || Instant.ofEpochMilli(expirationEpochMilli).isAfter(clock.instant())) {
            Map creator = Objects.requireNonNull((Map)source.get("creator"));
            String principal = Objects.requireNonNull((String)creator.get("principal"));
            Map metadata = (Map)creator.get("metadata");
            Map roleDescriptors = (Map)source.get("role_descriptors");
            Map limitedByRoleDescriptors = (Map)source.get("limited_by_role_descriptors");
            String[] roleNames = roleDescriptors != null ? roleDescriptors.keySet().toArray(Strings.EMPTY_ARRAY) : limitedByRoleDescriptors.keySet().toArray(Strings.EMPTY_ARRAY);
            User apiKeyUser = new User(principal, roleNames, null, null, metadata, true);
            HashMap<String, Object> authResultMetadata = new HashMap<String, Object>();
            authResultMetadata.put(API_KEY_ROLE_DESCRIPTORS_KEY, roleDescriptors);
            authResultMetadata.put(API_KEY_LIMITED_ROLE_DESCRIPTORS_KEY, limitedByRoleDescriptors);
            authResultMetadata.put(API_KEY_ID_KEY, credentials.getId());
            listener.onResponse((Object)AuthenticationResult.success((User)apiKeyUser, authResultMetadata));
        } else {
            listener.onResponse((Object)AuthenticationResult.unsuccessful((String)"api key is expired", null));
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    static ApiKeyCredentials getCredentialsFromHeader(ThreadContext threadContext) {
        String header = threadContext.getHeader("Authorization");
        if (Strings.hasText((String)header) && header.regionMatches(true, 0, "ApiKey ", 0, "ApiKey ".length()) && header.length() > "ApiKey ".length()) {
            byte[] decodedApiKeyCredBytes = Base64.getDecoder().decode(header.substring("ApiKey ".length()));
            char[] apiKeyCredChars = null;
            try {
                apiKeyCredChars = CharArrays.utf8BytesToChars((byte[])decodedApiKeyCredBytes);
                int colonIndex = -1;
                for (int i = 0; i < apiKeyCredChars.length; ++i) {
                    if (apiKeyCredChars[i] != ':') continue;
                    colonIndex = i;
                    break;
                }
                if (colonIndex < 1) {
                    throw new IllegalArgumentException("invalid ApiKey value");
                }
                ApiKeyCredentials apiKeyCredentials = new ApiKeyCredentials(new String(Arrays.copyOfRange(apiKeyCredChars, 0, colonIndex)), new SecureString(Arrays.copyOfRange(apiKeyCredChars, colonIndex + 1, apiKeyCredChars.length)));
                return apiKeyCredentials;
            }
            finally {
                if (apiKeyCredChars != null) {
                    Arrays.fill(apiKeyCredChars, '\u0000');
                }
            }
        }
        return null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected boolean verifyKeyAgainstHash(String apiKeyHash, ApiKeyCredentials credentials) {
        char[] apiKeyHashChars = apiKeyHash.toCharArray();
        try {
            Hasher hasher = Hasher.resolveFromHash((char[])apiKeyHash.toCharArray());
            boolean bl = hasher.verify(credentials.getKey(), apiKeyHashChars);
            return bl;
        }
        finally {
            Arrays.fill(apiKeyHashChars, '\u0000');
        }
    }

    private Instant getApiKeyExpiration(Instant now, CreateApiKeyRequest request) {
        if (request.getExpiration() != null) {
            return now.plusSeconds(request.getExpiration().getSeconds());
        }
        return null;
    }

    private boolean isEnabled() {
        return this.enabled && this.licenseState.isApiKeyServiceAllowed();
    }

    private void ensureEnabled() {
        if (!this.licenseState.isApiKeyServiceAllowed()) {
            throw LicenseUtils.newComplianceException((String)"api keys");
        }
        if (!this.enabled) {
            throw new IllegalStateException("api keys are not enabled");
        }
    }

    public void invalidateApiKeysForRealmAndUser(String realmName, String userName, ActionListener<InvalidateApiKeyResponse> invalidateListener) {
        this.ensureEnabled();
        if (!Strings.hasText((String)realmName) && !Strings.hasText((String)userName)) {
            logger.trace("No realm name or username provided");
            invalidateListener.onFailure((Exception)new IllegalArgumentException("realm name or username must be provided"));
        } else {
            this.findApiKeysForUserAndRealm(userName, realmName, true, false, (ActionListener<Collection<ApiKey>>)ActionListener.wrap(apiKeyIds -> {
                if (apiKeyIds.isEmpty()) {
                    logger.warn("No active api keys to invalidate for realm [{}] and username [{}]", (Object)realmName, (Object)userName);
                    invalidateListener.onResponse((Object)InvalidateApiKeyResponse.emptyResponse());
                } else {
                    this.invalidateAllApiKeys(apiKeyIds.stream().map(apiKey -> apiKey.getId()).collect(Collectors.toSet()), invalidateListener);
                }
            }, arg_0 -> invalidateListener.onFailure(arg_0)));
        }
    }

    private void invalidateAllApiKeys(Collection<String> apiKeyIds, ActionListener<InvalidateApiKeyResponse> invalidateListener) {
        this.indexInvalidation(apiKeyIds, invalidateListener, null);
    }

    public void invalidateApiKeyForApiKeyId(String apiKeyId, ActionListener<InvalidateApiKeyResponse> invalidateListener) {
        this.ensureEnabled();
        if (!Strings.hasText((String)apiKeyId)) {
            logger.trace("No api key id provided");
            invalidateListener.onFailure((Exception)new IllegalArgumentException("api key id must be provided"));
        } else {
            this.findApiKeysForApiKeyId(apiKeyId, true, false, (ActionListener<Collection<ApiKey>>)ActionListener.wrap(apiKeyIds -> {
                if (apiKeyIds.isEmpty()) {
                    logger.warn("No api key to invalidate for api key id [{}]", (Object)apiKeyId);
                    invalidateListener.onResponse((Object)InvalidateApiKeyResponse.emptyResponse());
                } else {
                    this.invalidateAllApiKeys(apiKeyIds.stream().map(apiKey -> apiKey.getId()).collect(Collectors.toSet()), invalidateListener);
                }
            }, arg_0 -> invalidateListener.onFailure(arg_0)));
        }
    }

    public void invalidateApiKeyForApiKeyName(String apiKeyName, ActionListener<InvalidateApiKeyResponse> invalidateListener) {
        this.ensureEnabled();
        if (!Strings.hasText((String)apiKeyName)) {
            logger.trace("No api key name provided");
            invalidateListener.onFailure((Exception)new IllegalArgumentException("api key name must be provided"));
        } else {
            this.findApiKeyForApiKeyName(apiKeyName, true, false, (ActionListener<Collection<ApiKey>>)ActionListener.wrap(apiKeyIds -> {
                if (apiKeyIds.isEmpty()) {
                    logger.warn("No api key to invalidate for api key name [{}]", (Object)apiKeyName);
                    invalidateListener.onResponse((Object)InvalidateApiKeyResponse.emptyResponse());
                } else {
                    this.invalidateAllApiKeys(apiKeyIds.stream().map(apiKey -> apiKey.getId()).collect(Collectors.toSet()), invalidateListener);
                }
            }, arg_0 -> invalidateListener.onFailure(arg_0)));
        }
    }

    private void findApiKeysForUserAndRealm(String userName, String realmName, boolean filterOutInvalidatedKeys, boolean filterOutExpiredKeys, ActionListener<Collection<ApiKey>> listener) {
        SecurityIndexManager frozenSecurityIndex = this.securityIndex.freeze();
        if (!frozenSecurityIndex.indexExists()) {
            listener.onResponse(Collections.emptyList());
        } else if (!frozenSecurityIndex.isAvailable()) {
            listener.onFailure((Exception)frozenSecurityIndex.getUnavailableReason());
        } else {
            BoolQueryBuilder boolQuery = QueryBuilders.boolQuery().filter((QueryBuilder)QueryBuilders.termQuery((String)"doc_type", (String)"api_key"));
            if (Strings.hasText((String)userName)) {
                boolQuery.filter((QueryBuilder)QueryBuilders.termQuery((String)"creator.principal", (String)userName));
            }
            if (Strings.hasText((String)realmName)) {
                boolQuery.filter((QueryBuilder)QueryBuilders.termQuery((String)"creator.realm", (String)realmName));
            }
            this.findApiKeys(boolQuery, filterOutInvalidatedKeys, filterOutExpiredKeys, listener);
        }
    }

    private void findApiKeys(BoolQueryBuilder boolQuery, boolean filterOutInvalidatedKeys, boolean filterOutExpiredKeys, ActionListener<Collection<ApiKey>> listener) {
        if (filterOutInvalidatedKeys) {
            boolQuery.filter((QueryBuilder)QueryBuilders.termQuery((String)"api_key_invalidated", (boolean)false));
        }
        if (filterOutExpiredKeys) {
            BoolQueryBuilder expiredQuery = QueryBuilders.boolQuery();
            expiredQuery.should((QueryBuilder)QueryBuilders.rangeQuery((String)"expiration_time").lte((Object)Instant.now().toEpochMilli()));
            expiredQuery.should((QueryBuilder)QueryBuilders.boolQuery().mustNot((QueryBuilder)QueryBuilders.existsQuery((String)"expiration_time")));
            boolQuery.filter((QueryBuilder)expiredQuery);
        }
        SearchRequest request = (SearchRequest)this.client.prepareSearch(new String[]{".security"}).setScroll((TimeValue)SearchService.DEFAULT_KEEPALIVE_SETTING.get(this.settings)).setQuery((QueryBuilder)boolQuery).setVersion(false).setSize(1000).setFetchSource(true).request();
        this.securityIndex.checkIndexVersionThenExecute(arg_0 -> listener.onFailure(arg_0), () -> ScrollHelper.fetchAllByEntity((Client)this.client, (SearchRequest)request, (ActionListener)listener, hit -> {
            Map source = hit.getSourceAsMap();
            String name = (String)source.get("name");
            String id = hit.getId();
            Long creation = (Long)source.get("creation_time");
            Long expiration = (Long)source.get("expiration_time");
            Boolean invalidated = (Boolean)source.get("api_key_invalidated");
            String username = (String)((Map)source.get("creator")).get("principal");
            String realm = (String)((Map)source.get("creator")).get("realm");
            return new ApiKey(name, id, Instant.ofEpochMilli(creation), expiration != null ? Instant.ofEpochMilli(expiration) : null, invalidated.booleanValue(), username, realm);
        }));
    }

    private void findApiKeyForApiKeyName(String apiKeyName, boolean filterOutInvalidatedKeys, boolean filterOutExpiredKeys, ActionListener<Collection<ApiKey>> listener) {
        SecurityIndexManager frozenSecurityIndex = this.securityIndex.freeze();
        if (!frozenSecurityIndex.indexExists()) {
            listener.onResponse(Collections.emptyList());
        } else if (!frozenSecurityIndex.isAvailable()) {
            listener.onFailure((Exception)frozenSecurityIndex.getUnavailableReason());
        } else {
            BoolQueryBuilder boolQuery = QueryBuilders.boolQuery().filter((QueryBuilder)QueryBuilders.termQuery((String)"doc_type", (String)"api_key"));
            if (Strings.hasText((String)apiKeyName)) {
                boolQuery.filter((QueryBuilder)QueryBuilders.termQuery((String)"name", (String)apiKeyName));
            }
            this.findApiKeys(boolQuery, filterOutInvalidatedKeys, filterOutExpiredKeys, listener);
        }
    }

    private void findApiKeysForApiKeyId(String apiKeyId, boolean filterOutInvalidatedKeys, boolean filterOutExpiredKeys, ActionListener<Collection<ApiKey>> listener) {
        SecurityIndexManager frozenSecurityIndex = this.securityIndex.freeze();
        if (!frozenSecurityIndex.indexExists()) {
            listener.onResponse(Collections.emptyList());
        } else if (!frozenSecurityIndex.isAvailable()) {
            listener.onFailure((Exception)frozenSecurityIndex.getUnavailableReason());
        } else {
            BoolQueryBuilder boolQuery = QueryBuilders.boolQuery().filter((QueryBuilder)QueryBuilders.termQuery((String)"doc_type", (String)"api_key")).filter((QueryBuilder)QueryBuilders.termQuery((String)"_id", (String)apiKeyId));
            this.findApiKeys(boolQuery, filterOutInvalidatedKeys, filterOutExpiredKeys, listener);
        }
    }

    private void indexInvalidation(Collection<String> apiKeyIds, ActionListener<InvalidateApiKeyResponse> listener, @Nullable InvalidateApiKeyResponse previousResult) {
        this.maybeStartApiKeyRemover();
        if (apiKeyIds.isEmpty()) {
            listener.onFailure((Exception)new ElasticsearchSecurityException("No api key ids provided for invalidation", new Object[0]));
        } else {
            BulkRequestBuilder bulkRequestBuilder = this.client.prepareBulk();
            for (String apiKeyId : apiKeyIds) {
                UpdateRequest request = (UpdateRequest)this.client.prepareUpdate(".security", TYPE, apiKeyId).setDoc(Collections.singletonMap("api_key_invalidated", true)).request();
                bulkRequestBuilder.add(request);
            }
            bulkRequestBuilder.setRefreshPolicy(WriteRequest.RefreshPolicy.WAIT_UNTIL);
            this.securityIndex.prepareIndexIfNeededThenExecute(ex -> listener.onFailure(this.traceLog("prepare security index", ex)), () -> ClientHelper.executeAsyncWithOrigin((ThreadContext)this.client.threadPool().getThreadContext(), (String)"security", (ActionRequest)((BulkRequest)bulkRequestBuilder.request()), (ActionListener)ActionListener.wrap(bulkResponse -> {
                ArrayList<ElasticsearchException> failedRequestResponses = new ArrayList<ElasticsearchException>();
                ArrayList<String> previouslyInvalidated = new ArrayList<String>();
                ArrayList<String> invalidated = new ArrayList<String>();
                if (null != previousResult) {
                    failedRequestResponses.addAll(previousResult.getErrors());
                    previouslyInvalidated.addAll(previousResult.getPreviouslyInvalidatedApiKeys());
                    invalidated.addAll(previousResult.getInvalidatedApiKeys());
                }
                for (BulkItemResponse bulkItemResponse : bulkResponse.getItems()) {
                    if (bulkItemResponse.isFailed()) {
                        Exception cause = bulkItemResponse.getFailure().getCause();
                        String failedApiKeyId = bulkItemResponse.getFailure().getId();
                        this.traceLog("invalidate api key", failedApiKeyId, cause);
                        failedRequestResponses.add(new ElasticsearchException("Error invalidating api key", (Throwable)cause, new Object[0]));
                        continue;
                    }
                    UpdateResponse updateResponse = (UpdateResponse)bulkItemResponse.getResponse();
                    if (updateResponse.getResult() == DocWriteResponse.Result.UPDATED) {
                        logger.debug("Invalidated api key for doc [{}]", (Object)updateResponse.getId());
                        invalidated.add(updateResponse.getId());
                        continue;
                    }
                    if (updateResponse.getResult() != DocWriteResponse.Result.NOOP) continue;
                    previouslyInvalidated.add(updateResponse.getId());
                }
                InvalidateApiKeyResponse result = new InvalidateApiKeyResponse(invalidated, previouslyInvalidated, failedRequestResponses);
                listener.onResponse((Object)result);
            }, e -> {
                Throwable cause = ExceptionsHelper.unwrapCause((Throwable)e);
                this.traceLog("invalidate api keys", cause);
                listener.onFailure(e);
            }), (arg_0, arg_1) -> ((Client)this.client).bulk(arg_0, arg_1)));
        }
    }

    private <E extends Throwable> E traceLog(String action, String identifier, E exception) {
        if (logger.isTraceEnabled()) {
            if (exception instanceof ElasticsearchException) {
                ElasticsearchException esEx = (ElasticsearchException)exception;
                List detail = esEx.getHeader("error_description");
                if (detail != null) {
                    logger.trace(() -> new ParameterizedMessage("Failure in [{}] for id [{}] - [{}]", new Object[]{action, identifier, detail}), (Throwable)esEx);
                } else {
                    logger.trace(() -> new ParameterizedMessage("Failure in [{}] for id [{}]", (Object)action, (Object)identifier), (Throwable)esEx);
                }
            } else {
                logger.trace(() -> new ParameterizedMessage("Failure in [{}] for id [{}]", (Object)action, (Object)identifier), exception);
            }
        }
        return exception;
    }

    private <E extends Throwable> E traceLog(String action, E exception) {
        if (logger.isTraceEnabled()) {
            if (exception instanceof ElasticsearchException) {
                ElasticsearchException esEx = (ElasticsearchException)exception;
                List detail = esEx.getHeader("error_description");
                if (detail != null) {
                    logger.trace(() -> new ParameterizedMessage("Failure in [{}] - [{}]", (Object)action, detail), (Throwable)esEx);
                } else {
                    logger.trace(() -> new ParameterizedMessage("Failure in [{}]", (Object)action), (Throwable)esEx);
                }
            } else {
                logger.trace(() -> new ParameterizedMessage("Failure in [{}]", (Object)action), exception);
            }
        }
        return exception;
    }

    boolean isExpirationInProgress() {
        return this.expiredApiKeysRemover.isExpirationInProgress();
    }

    long lastTimeWhenApiKeysRemoverWasTriggered() {
        return this.lastExpirationRunMs;
    }

    private void maybeStartApiKeyRemover() {
        if (this.securityIndex.isAvailable() && this.client.threadPool().relativeTimeInMillis() - this.lastExpirationRunMs > this.deleteInterval.getMillis()) {
            this.expiredApiKeysRemover.submit(this.client.threadPool());
            this.lastExpirationRunMs = this.client.threadPool().relativeTimeInMillis();
        }
    }

    public void getApiKeysForRealmAndUser(String realmName, String userName, ActionListener<GetApiKeyResponse> listener) {
        this.ensureEnabled();
        if (!Strings.hasText((String)realmName) && !Strings.hasText((String)userName)) {
            logger.trace("No realm name or username provided");
            listener.onFailure((Exception)new IllegalArgumentException("realm name or username must be provided"));
        } else {
            this.findApiKeysForUserAndRealm(userName, realmName, false, false, (ActionListener<Collection<ApiKey>>)ActionListener.wrap(apiKeyInfos -> {
                if (apiKeyInfos.isEmpty()) {
                    logger.warn("No active api keys found for realm [{}] and username [{}]", (Object)realmName, (Object)userName);
                    listener.onResponse((Object)GetApiKeyResponse.emptyResponse());
                } else {
                    listener.onResponse((Object)new GetApiKeyResponse(apiKeyInfos));
                }
            }, arg_0 -> listener.onFailure(arg_0)));
        }
    }

    public void getApiKeyForApiKeyId(String apiKeyId, ActionListener<GetApiKeyResponse> listener) {
        this.ensureEnabled();
        if (!Strings.hasText((String)apiKeyId)) {
            logger.trace("No api key id provided");
            listener.onFailure((Exception)new IllegalArgumentException("api key id must be provided"));
        } else {
            this.findApiKeysForApiKeyId(apiKeyId, false, false, (ActionListener<Collection<ApiKey>>)ActionListener.wrap(apiKeyInfos -> {
                if (apiKeyInfos.isEmpty()) {
                    logger.warn("No api key found for api key id [{}]", (Object)apiKeyId);
                    listener.onResponse((Object)GetApiKeyResponse.emptyResponse());
                } else {
                    listener.onResponse((Object)new GetApiKeyResponse(apiKeyInfos));
                }
            }, arg_0 -> listener.onFailure(arg_0)));
        }
    }

    public void getApiKeyForApiKeyName(String apiKeyName, ActionListener<GetApiKeyResponse> listener) {
        this.ensureEnabled();
        if (!Strings.hasText((String)apiKeyName)) {
            logger.trace("No api key name provided");
            listener.onFailure((Exception)new IllegalArgumentException("api key name must be provided"));
        } else {
            this.findApiKeyForApiKeyName(apiKeyName, false, false, (ActionListener<Collection<ApiKey>>)ActionListener.wrap(apiKeyInfos -> {
                if (apiKeyInfos.isEmpty()) {
                    logger.warn("No api key found for api key name [{}]", (Object)apiKeyName);
                    listener.onResponse((Object)GetApiKeyResponse.emptyResponse());
                } else {
                    listener.onResponse((Object)new GetApiKeyResponse(apiKeyInfos));
                }
            }, arg_0 -> listener.onFailure(arg_0)));
        }
    }

    final class CachedApiKeyHashResult {
        final boolean success;
        final char[] hash;

        CachedApiKeyHashResult(boolean success, SecureString apiKey) {
            this.success = success;
            this.hash = ApiKeyService.this.cacheHasher.hash(apiKey);
        }

        private boolean verify(SecureString password) {
            return this.hash != null && ApiKeyService.this.cacheHasher.verify(password, this.hash);
        }
    }

    private static class ApiKeyLoggingDeprecationHandler
    implements DeprecationHandler {
        private final DeprecationLogger deprecationLogger;
        private final String apiKeyId;

        private ApiKeyLoggingDeprecationHandler(DeprecationLogger logger, String apiKeyId) {
            this.deprecationLogger = logger;
            this.apiKeyId = apiKeyId;
        }

        public void usedDeprecatedName(String usedName, String modernName) {
            this.deprecationLogger.deprecated("Deprecated field [{}] used in api key [{}], expected [{}] instead", new Object[]{usedName, this.apiKeyId, modernName});
        }

        public void usedDeprecatedField(String usedName, String replacedWith) {
            this.deprecationLogger.deprecated("Deprecated field [{}] used in api key [{}], replaced by [{}]", new Object[]{usedName, this.apiKeyId, replacedWith});
        }

        public void deprecated(String message, Object ... params) {
            String formattedMessage = LoggerMessageFormat.format((String)message, (Object[])params);
            this.deprecationLogger.deprecated(formattedMessage, new Object[0]);
        }
    }

    static final class ApiKeyCredentials
    implements Closeable {
        private final String id;
        private final SecureString key;

        ApiKeyCredentials(String id, SecureString key) {
            this.id = id;
            this.key = key;
        }

        String getId() {
            return this.id;
        }

        SecureString getKey() {
            return this.key;
        }

        @Override
        public void close() {
            this.key.close();
        }
    }

    public static class ApiKeyRoleDescriptors {
        private final String apiKeyId;
        private final List<RoleDescriptor> roleDescriptors;
        private final List<RoleDescriptor> limitedByRoleDescriptors;

        public ApiKeyRoleDescriptors(String apiKeyId, List<RoleDescriptor> roleDescriptors, List<RoleDescriptor> limitedByDescriptors) {
            this.apiKeyId = apiKeyId;
            this.roleDescriptors = roleDescriptors;
            this.limitedByRoleDescriptors = limitedByDescriptors;
        }

        public String getApiKeyId() {
            return this.apiKeyId;
        }

        public List<RoleDescriptor> getRoleDescriptors() {
            return this.roleDescriptors;
        }

        public List<RoleDescriptor> getLimitedByRoleDescriptors() {
            return this.limitedByRoleDescriptors;
        }
    }
}

