<?php

declare(strict_types=1);

/*
 * This file is part of the TYPO3 CMS project.
 *
 * It is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License, either version 2
 * of the License, or any later version.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 *
 * The TYPO3 project - inspiring people to share!
 */

namespace TYPO3\CMS\Workspaces\Controller\Remote;

use Psr\EventDispatcher\EventDispatcherInterface;
use Psr\Http\Message\ServerRequestInterface;
use Symfony\Component\DependencyInjection\Attribute\Autoconfigure;
use TYPO3\CMS\Backend\Backend\Avatar\Avatar;
use TYPO3\CMS\Backend\Utility\BackendUtility;
use TYPO3\CMS\Backend\View\ValueFormatter\FlexFormValueFormatter;
use TYPO3\CMS\Core\Authentication\BackendUserAuthentication;
use TYPO3\CMS\Core\Database\Connection;
use TYPO3\CMS\Core\Database\ConnectionPool;
use TYPO3\CMS\Core\Imaging\IconFactory;
use TYPO3\CMS\Core\Imaging\IconSize;
use TYPO3\CMS\Core\Localization\LanguageService;
use TYPO3\CMS\Core\Log\LogDataTrait;
use TYPO3\CMS\Core\Resource\FileReference;
use TYPO3\CMS\Core\Resource\ProcessedFile;
use TYPO3\CMS\Core\Schema\SearchableSchemaFieldsCollector;
use TYPO3\CMS\Core\Schema\VisibleSchemaFieldsCollector;
use TYPO3\CMS\Core\SysLog\Action\Database as DatabaseAction;
use TYPO3\CMS\Core\Utility\DiffGranularity;
use TYPO3\CMS\Core\Utility\DiffUtility;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Core\Utility\MathUtility;
use TYPO3\CMS\Core\Utility\StringUtility;
use TYPO3\CMS\Core\Versioning\VersionState;
use TYPO3\CMS\Workspaces\Domain\Model\CombinedRecord;
use TYPO3\CMS\Workspaces\Domain\Model\WorkspaceStage;
use TYPO3\CMS\Workspaces\Domain\Repository\WorkspaceRepository;
use TYPO3\CMS\Workspaces\Domain\Repository\WorkspaceStageRepository;
use TYPO3\CMS\Workspaces\Event\ModifyVersionDifferencesEvent;
use TYPO3\CMS\Workspaces\Exception\WorkspaceStageNotFoundException;
use TYPO3\CMS\Workspaces\Service\GridDataService;
use TYPO3\CMS\Workspaces\Service\HistoryService;
use TYPO3\CMS\Workspaces\Service\IntegrityService;
use TYPO3\CMS\Workspaces\Service\StagesService;
use TYPO3\CMS\Workspaces\Service\WorkspaceService;

/**
 * @internal This is a specific Backend Controller implementation and is not considered part of the Public TYPO3 API.
 */
#[Autoconfigure(public: true)]
readonly class RemoteServer
{
    use LogDataTrait;

    public function __construct(
        protected GridDataService $gridDataService,
        protected StagesService $stagesService,
        protected WorkspaceService $workspaceService,
        protected EventDispatcherInterface $eventDispatcher,
        protected FlexFormValueFormatter $flexFormValueFormatter,
        private DiffUtility $diffUtility,
        protected IconFactory $iconFactory,
        protected Avatar $avatar,
        protected ConnectionPool $connectionPool,
        protected SearchableSchemaFieldsCollector $searchableSchemaFieldsCollector,
        protected VisibleSchemaFieldsCollector $visibleSchemaFieldsCollector,
        protected WorkspaceRepository $workspaceRepository,
        protected WorkspaceStageRepository $workspaceStageRepository,
    ) {}

    /**
     * Checks integrity of elements before performing actions on them.
     *
     * @return array
     */
    public function checkIntegrity(\stdClass $parameters)
    {
        $integrity = $this->createIntegrityService($this->getAffectedElements($parameters));
        $integrity->check();
        $response = [
            'result' => $integrity->getStatusRepresentation(),
        ];
        return $response;
    }

    /**
     * Get List of workspace changes
     *
     * @param \stdClass $parameter
     * @return array $data
     */
    public function getWorkspaceInfos($parameter, ServerRequestInterface $request)
    {
        // To avoid too much work we use -1 to indicate that every page is relevant
        $pageId = $parameter->id > 0 ? $parameter->id : -1;
        if (!isset($parameter->language) || !MathUtility::canBeInterpretedAsInteger($parameter->language)) {
            $parameter->language = null;
        }
        if (!isset($parameter->stage) || !MathUtility::canBeInterpretedAsInteger($parameter->stage)) {
            // -99 disables stage filtering
            $parameter->stage = -99;
        }
        $backendUser = $this->getBackendUser();
        $currentWorkspace = $backendUser->workspace;
        $workspaceRecord = $this->workspaceRepository->findByUid($currentWorkspace);
        $stages = $this->workspaceStageRepository->findAllStagesByWorkspace($backendUser, $workspaceRecord);
        $versions = $this->workspaceService->selectVersionsInWorkspace(
            $this->getCurrentWorkspace(),
            (int)$parameter->stage,
            $pageId,
            (int)$parameter->depth,
            'tables_select',
            $parameter->language !== null ? (int)$parameter->language : null
        );
        return $this->gridDataService->generateGridListFromVersions($stages, $versions, $parameter);
    }

    /**
     * Fetch further information to current selected workspace record.
     *
     * @param WorkspaceStage[] $stages
     */
    public function getRowDetails(array $stages, \stdClass $parameter): array
    {
        if (!BackendUtility::isTableWorkspaceEnabled($parameter->table)
            || !$this->getBackendUser()->check('tables_modify', $parameter->table)
        ) {
            throw new \RuntimeException(sprintf('Invalid access to table "%s"', $parameter->table), 1756882012);
        }

        $diffReturnArray = [];
        $liveReturnArray = [];
        $plainLiveRecord = $liveRecord = (array)BackendUtility::getRecord($parameter->table, $parameter->t3ver_oid);
        $plainVersionRecord = $versionRecord = (array)BackendUtility::getRecord($parameter->table, $parameter->uid);
        $versionState = VersionState::tryFrom($versionRecord['t3ver_state'] ?? 0);

        try {
            $currentStage = $this->stagesService->getStage($stages, $parameter->stage);
        } catch (WorkspaceStageNotFoundException) {
            $currentStage = null;
        }
        $nextStageSendToTitle = false;
        $previousStageSendToTitle = false;
        if ($currentStage?->isAllowed) {
            try {
                $nextStage = $this->stagesService->getNextStage($stages, $currentStage->uid);
                if ($nextStage->isExecuteStage) {
                    $nextStageSendToTitle = $this->getLanguageService()->sL('LLL:EXT:workspaces/Resources/Private/Language/locallang.xlf:publish_execute_action_option');
                } else {
                    $nextStageSendToTitle = $this->getLanguageService()->sL('LLL:EXT:workspaces/Resources/Private/Language/locallang.xlf:actionSendToStage') . ' "' . $nextStage->title . '"';
                }
            } catch (WorkspaceStageNotFoundException) {
                // keep false as title
            }
            try {
                $previousStage = $this->stagesService->getPreviousStage($stages, $currentStage->uid);
                $previousStageSendToTitle = $this->getLanguageService()->sL('LLL:EXT:workspaces/Resources/Private/Language/locallang.xlf:actionSendToStage') . ' "' . $previousStage->title . '"';
            } catch (WorkspaceStageNotFoundException) {
                // keep false as title
            }
        }

        $iconWorkspace = $this->iconFactory->getIconForRecord($parameter->table, $versionRecord, IconSize::SMALL);
        $fieldsOfRecords = array_keys($liveRecord);
        $isNewOrDeletePlaceholder = $versionState === VersionState::NEW_PLACEHOLDER || $versionState === VersionState::DELETE_PLACEHOLDER;
        $suitableFields = ($isNewOrDeletePlaceholder && ($parameter->filterFields ?? false)) ? array_flip($this->getSuitableFields($parameter->table, $liveRecord)) : [];
        foreach ($fieldsOfRecords as $fieldName) {
            if (
                empty($GLOBALS['TCA'][$parameter->table]['columns'][$fieldName]['config'])
            ) {
                continue;
            }
            // Disable internal fields
            // l10n_diffsource is not needed, see #91667
            if (($GLOBALS['TCA'][$parameter->table]['ctrl']['transOrigDiffSourceField'] ?? '') === $fieldName) {
                continue;
            }
            if (($GLOBALS['TCA'][$parameter->table]['ctrl']['origUid'] ?? '') === $fieldName) {
                continue;
            }
            // Get the field's label. If not available, use the field name
            $fieldTitle = $this->getLanguageService()->sL(BackendUtility::getItemLabel($parameter->table, $fieldName));
            if (empty($fieldTitle)) {
                $fieldTitle = $fieldName;
            }
            // Gets the TCA configuration for the current field
            $configuration = $GLOBALS['TCA'][$parameter->table]['columns'][$fieldName]['config'];
            // check for exclude fields
            $isFieldExcluded = (bool)($GLOBALS['TCA'][$parameter->table]['columns'][$fieldName]['exclude'] ?? false);
            if ($this->getBackendUser()->isAdmin()
                || (
                    ($GLOBALS['TCA'][$parameter->table]['columns'][$fieldName]['displayCond'] ?? null) !== 'HIDE_FOR_NON_ADMINS'
                    && (!$isFieldExcluded || GeneralUtility::inList($this->getBackendUser()->groupData['non_exclude_fields'], $parameter->table . ':' . $fieldName))
                )
            ) {
                // call diff class only if there is a difference
                if ($configuration['type'] === 'file') {
                    $useThumbnails = false;
                    if (!empty($configuration['allowed']) && !empty($GLOBALS['TYPO3_CONF_VARS']['GFX']['imagefile_ext'])) {
                        $fileExtensions = GeneralUtility::trimExplode(',', $GLOBALS['TYPO3_CONF_VARS']['GFX']['imagefile_ext'], true);
                        $allowedExtensions = GeneralUtility::trimExplode(',', $configuration['allowed'], true);
                        $differentExtensions = array_diff($allowedExtensions, $fileExtensions);
                        $useThumbnails = empty($differentExtensions);
                    }
                    $liveFileReferences = (array)BackendUtility::resolveFileReferences(
                        $parameter->table,
                        $fieldName,
                        $liveRecord,
                        0
                    );
                    $versionFileReferences = (array)BackendUtility::resolveFileReferences(
                        $parameter->table,
                        $fieldName,
                        $versionRecord,
                        $this->getCurrentWorkspace()
                    );
                    $fileReferenceDifferences = $this->prepareFileReferenceDifferences(
                        $liveFileReferences,
                        $versionFileReferences,
                        $useThumbnails
                    );

                    if ($fileReferenceDifferences === null) {
                        continue;
                    }

                    $diffReturnArray[] = [
                        'field' => $fieldName,
                        'label' => $fieldTitle,
                        'content' => $fileReferenceDifferences['differences'],
                    ];
                    $liveReturnArray[] = [
                        'field' => $fieldName,
                        'label' => $fieldTitle,
                        'content' => $fileReferenceDifferences['live'],
                    ];
                } elseif ($isNewOrDeletePlaceholder && isset($suitableFields[$fieldName])) {
                    // If this is a new or delete placeholder, add diff view for all appropriate fields
                    $newOrDeleteRecord[$fieldName] = $this->formatValue($parameter->table, $fieldName, (string)$liveRecord[$fieldName], $liveRecord['uid'], $configuration, $plainLiveRecord);

                    // Don't add empty fields
                    if ($newOrDeleteRecord[$fieldName] === '') {
                        continue;
                    }

                    $granularity = ($configuration['type'] ?? '') === 'flex' ? DiffGranularity::CHARACTER : DiffGranularity::WORD;
                    $diffReturnArray[] = [
                        'field' => $fieldName,
                        'label' => $fieldTitle,
                        'content' => $versionState === VersionState::NEW_PLACEHOLDER
                            ? $this->diffUtility->diff('', strip_tags($newOrDeleteRecord[$fieldName]), $granularity)
                            : $this->diffUtility->diff(strip_tags($newOrDeleteRecord[$fieldName]), '', $granularity),
                    ];

                    // Generally not needed by Core, but let's make it available for further processing in hooks
                    $liveReturnArray[] = [
                        'field' => $fieldName,
                        'label' => $fieldTitle,
                        'content' => $newOrDeleteRecord[$fieldName],
                    ];
                } elseif ((string)$liveRecord[$fieldName] !== (string)$versionRecord[$fieldName]) {
                    // Select the human-readable values before diff
                    $liveRecord[$fieldName] = $this->formatValue($parameter->table, $fieldName, (string)$liveRecord[$fieldName], $liveRecord['uid'], $configuration, $plainLiveRecord);
                    $versionRecord[$fieldName] = $this->formatValue($parameter->table, $fieldName, (string)$versionRecord[$fieldName], $versionRecord['uid'], $configuration, $plainVersionRecord);
                    $fieldDifferences = ($configuration['type'] ?? '') === 'flex'
                        ? $this->diffUtility->diff(strip_tags($liveRecord[$fieldName]), strip_tags($versionRecord[$fieldName]), DiffGranularity::CHARACTER)
                        : $this->diffUtility->diff(strip_tags($liveRecord[$fieldName]), strip_tags($versionRecord[$fieldName]));
                    $diffReturnArray[] = [
                        'field' => $fieldName,
                        'label' => $fieldTitle,
                        'content' => $fieldDifferences,
                    ];
                    $liveReturnArray[] = [
                        'field' => $fieldName,
                        'label' => $fieldTitle,
                        'content' => $liveRecord[$fieldName],
                    ];
                }
            }
        }

        $versionDifferencesEvent = $this->eventDispatcher->dispatch(new ModifyVersionDifferencesEvent($diffReturnArray, $liveReturnArray, $parameter));
        $historyService = GeneralUtility::makeInstance(HistoryService::class);
        $history = $historyService->getHistory($parameter->table, $parameter->t3ver_oid);
        $stageChanges = $historyService->getStageChanges($parameter->table, (int)$parameter->t3ver_oid);
        $stageChangesFromSysLog = $this->getStageChangesFromSysLog($parameter->table, (int)$parameter->t3ver_oid);
        $commentsForRecord = $this->getCommentsForRecord($stageChanges, $stageChangesFromSysLog);

        return [
            'total' => 1,
            'data' => [
                [
                    // these parts contain HTML (don't escape)
                    'diff' => $versionDifferencesEvent->getVersionDifferences(),
                    'icon_Workspace' => $iconWorkspace->getIdentifier(),
                    'icon_Workspace_Overlay' => $iconWorkspace->getOverlayIcon()?->getIdentifier() ?? '',
                    // this part is already escaped in getCommentsForRecord()
                    'comments' => $commentsForRecord,
                    // escape/sanitize the others
                    'path_Live' => htmlspecialchars(BackendUtility::getRecordPath($liveRecord['pid'], '', 999)),
                    'label_Stage' => htmlspecialchars($currentStage->title),
                    'label_PrevStage' => $previousStageSendToTitle ? ['title' => $previousStageSendToTitle] : false,
                    'label_NextStage' => $nextStageSendToTitle ? ['title' => $nextStageSendToTitle] : false,
                    'stage_position' => $this->stagesService->getPositionOfCurrentStage($stages, $currentStage->uid),
                    'stage_count' => count($stages) - 1, // Do not count 'pseudo' execute stage
                    'parent' => [
                        'table' => htmlspecialchars($parameter->table),
                        'uid' => (int)$parameter->uid,
                    ],
                    'history' => [
                        'data' => $history,
                        'total' => count($history),
                    ],
                ],
            ],
        ];
    }

    protected function formatValue(string $table, string $fieldName, string $value, int $uid, array $tcaConfiguration, array $fullRow): string
    {
        if (($tcaConfiguration['type'] ?? '') === 'flex') {
            return $this->flexFormValueFormatter->format($table, $fieldName, $value, $uid, $tcaConfiguration);
        }
        return (string)BackendUtility::getProcessedValue($table, $fieldName, $value, 0, true, false, $uid, true, (int)$fullRow['pid'], $fullRow);
    }

    /**
     * Prepares difference view for file references.
     *
     * @param FileReference[] $liveFileReferences
     * @param FileReference[] $versionFileReferences
     * @param bool|false $useThumbnails
     * @return array|null
     */
    protected function prepareFileReferenceDifferences(array $liveFileReferences, array $versionFileReferences, $useThumbnails = false)
    {
        $randomValue = StringUtility::getUniqueId('file');

        $liveValues = [];
        $versionValues = [];
        $candidates = [];
        $substitutes = [];

        // Process live references
        foreach ($liveFileReferences as $liveFileReference) {
            $identifierWithRandomValue = $randomValue . '__' . $liveFileReference->getUid() . '__' . $randomValue;
            $candidates[$identifierWithRandomValue] = $liveFileReference;
            $liveValues[] = $identifierWithRandomValue;
        }

        // Process version references
        foreach ($versionFileReferences as $versionFileReference) {
            $identifierWithRandomValue = $randomValue . '__' . $versionFileReference->getUid() . '__' . $randomValue;
            $candidates[$identifierWithRandomValue] = $versionFileReference;
            $versionValues[] = $identifierWithRandomValue;
        }

        // Combine values and surround by spaces
        // (to reduce the chunks Diff will find)
        $liveInformation = ' ' . implode(' ', $liveValues) . ' ';
        $versionInformation = ' ' . implode(' ', $versionValues) . ' ';

        // Return if information has not changed
        if ($liveInformation === $versionInformation) {
            return null;
        }

        foreach ($candidates as $identifierWithRandomValue => $fileReference) {
            if ($useThumbnails) {
                $thumbnailFile = $fileReference->getOriginalFile()->process(
                    ProcessedFile::CONTEXT_IMAGEPREVIEW,
                    ['width' => 40, 'height' => 40]
                );
                $thumbnailMarkup = '<img src="' . htmlspecialchars($thumbnailFile->getPublicUrl() ?? '') . '" />';
                $substitutes[$identifierWithRandomValue] = $thumbnailMarkup;
            } else {
                $substitutes[$identifierWithRandomValue] = $fileReference->getPublicUrl();
            }
        }

        $differences = $this->diffUtility->diff(strip_tags($liveInformation), strip_tags($versionInformation));
        $liveInformation = str_replace(array_keys($substitutes), array_values($substitutes), trim($liveInformation));
        $differences = str_replace(array_keys($substitutes), array_values($substitutes), trim($differences));

        return [
            'live' => $liveInformation,
            'differences' => $differences,
        ];
    }

    /**
     * Prepares all comments of the stage change history entries for returning the JSON structure
     *
     * @param array $additionalChangesFromLog this is not in use since 2022 anymore, and can be removed in TYPO3 v13.0 the latest.
     */
    protected function getCommentsForRecord(array $historyEntries, array $additionalChangesFromLog): array
    {
        $allStageChanges = [];

        foreach ($historyEntries as $entry) {
            $preparedEntry = [];
            $beUserRecord = BackendUtility::getRecord('be_users', $entry['userid']);
            $preparedEntry['stage_title'] = $this->stagesService->getStageTitle((int)$entry['history_data']['next']);
            $preparedEntry['previous_stage_title'] = $this->stagesService->getStageTitle((int)$entry['history_data']['current']);
            $preparedEntry['user_uid'] = (int)$entry['userid'];
            $preparedEntry['user_username'] = is_array($beUserRecord) ? $beUserRecord['username'] : '';
            $preparedEntry['tstamp'] = BackendUtility::datetime($entry['tstamp']);
            $preparedEntry['user_comment'] = $entry['history_data']['comment'];
            $preparedEntry['user_avatar'] = $beUserRecord ? $this->avatar->render($beUserRecord) : '';
            $allStageChanges[] = $preparedEntry;
        }

        // see if there are more
        foreach ($additionalChangesFromLog as $sysLogRow) {
            $sysLogEntry = [];
            $data = $this->unserializeLogData($sysLogRow['log_data'] ?? '');
            $beUserRecord = BackendUtility::getRecord('be_users', $sysLogRow['userid']);
            $sysLogEntry['stage_title'] = $this->stagesService->getStageTitle((int)$data['stage']);
            $sysLogEntry['previous_stage_title'] = '';
            $sysLogEntry['user_uid'] = (int)$sysLogRow['userid'];
            $sysLogEntry['user_username'] = is_array($beUserRecord) ? $beUserRecord['username'] : '';
            $sysLogEntry['tstamp'] = BackendUtility::datetime($sysLogRow['tstamp']);
            $sysLogEntry['user_comment'] = $data['comment'];
            $sysLogEntry['user_avatar'] = $this->avatar->render($beUserRecord);
            $allStageChanges[] = $sysLogEntry;
        }

        // There might be "old" sys_log entries, so they need to be checked as well
        return $allStageChanges;
    }

    /**
     * Find all stage changes from sys_log that do not have a historyId. Can safely be removed in future TYPO3
     * versions as this fallback layer only makes sense in TYPO3 v11 when old records want to have a history.
     */
    protected function getStageChangesFromSysLog(string $table, int $uid): array
    {
        $queryBuilder = $this->connectionPool->getQueryBuilderForTable('sys_log');

        return $queryBuilder
            ->select('log_data', 'tstamp', 'userid')
            ->from('sys_log')
            ->where(
                $queryBuilder->expr()->eq(
                    'action',
                    $queryBuilder->createNamedParameter(DatabaseAction::UPDATE, Connection::PARAM_INT)
                ),
                $queryBuilder->expr()->eq(
                    'details_nr',
                    $queryBuilder->createNamedParameter(30, Connection::PARAM_INT)
                ),
                $queryBuilder->expr()->eq(
                    'tablename',
                    $queryBuilder->createNamedParameter($table)
                ),
                $queryBuilder->expr()->eq(
                    'recuid',
                    $queryBuilder->createNamedParameter($uid, Connection::PARAM_INT)
                )
            )
            ->orderBy('tstamp', 'DESC')
            ->executeQuery()
            ->fetchAllAssociative();
    }

    protected function getBackendUser(): BackendUserAuthentication
    {
        return $GLOBALS['BE_USER'];
    }

    protected function getLanguageService(): LanguageService
    {
        return $GLOBALS['LANG'];
    }

    /**
     * Creates a new instance of the integrity service for the
     * given set of affected elements.
     *
     * @param CombinedRecord[] $affectedElements
     * @see getAffectedElements
     */
    protected function createIntegrityService(array $affectedElements): IntegrityService
    {
        $integrityService = GeneralUtility::makeInstance(IntegrityService::class);
        $integrityService->setAffectedElements($affectedElements);
        return $integrityService;
    }

    /**
     * Gets affected elements on publishing/swapping actions.
     * Affected elements have a dependency, e.g. translation overlay
     * and the default origin record - thus, the default record would be
     * affected if the translation overlay shall be published.
     */
    protected function getAffectedElements(\stdClass $parameters): array
    {
        $affectedElements = [];
        if ($parameters->type === 'selection') {
            foreach ((array)$parameters->selection as $element) {
                $affectedElements[] = CombinedRecord::create($element->table, (int)$element->liveId, (int)$element->versionId);
            }
        }
        return $affectedElements;
    }

    /**
     * Gets the current workspace ID.
     */
    protected function getCurrentWorkspace(): int
    {
        return $this->workspaceService->getCurrentWorkspace();
    }

    /**
     * Gets the fields suitable for being displayed in new and delete diff views
     */
    protected function getSuitableFields(string $table, array $row): array
    {
        // @todo Usage of searchableSchemaFieldsCollector seems like a misuse here, or at least it's unexpected
        return $this->searchableSchemaFieldsCollector->getUniqueFieldList(
            $table,
            $this->visibleSchemaFieldsCollector->getFieldNames($table, $row),
            false
        );
    }
}
