---- QuerySessionCurrentInfo ----
select pg_catalog.current_database(),
       current_schema,
       current_timestamp as current_timestamp
;


---- IsSuperUser ----
select usesuper
from pg_user
where usename = current_user
;


---- SchemaAges ----
select pg_catalog.age(:schema_txid::varchar::xid) as schema_age, pg_catalog.age(:sources_txid::varchar::xid) as sources_age
;


---- DbAge ----
select pg_catalog.age(:db_age::varchar::xid) as database_age
;


---- ListDatabases ----
select N.oid::bigint as id,
       N.xmin as state_number,
       datname as name,
       D.description,
       pg_catalog.pg_get_userbyid(N.datdba) as "owner"
from pg_catalog.pg_database N
  left join pg_catalog.pg_shdescription D on N.oid = D.objoid
where not datistemplate
order by case when datname = pg_catalog.current_database() then -1::bigint else N.oid::bigint end
;


---- ListSchemas ----
select N.oid::bigint as id,
       N.xmin as state_number,
       nspname as name,
       D.description,
       pg_catalog.pg_get_userbyid(N.nspowner) as "owner"
from pg_catalog.pg_namespace N
  left join pg_catalog.pg_description D on N.oid = D.objoid
order by case when nspname = current_schema then -1::bigint else N.oid::bigint end
;


---- RetrieveRoles ----
select R.oid::bigint as role_id, rolname as role_name,
  rolsuper is_super, rolinherit is_inherit,
  rolcreaterole can_createrole, rolcreatedb can_createdb,
  rolcanlogin can_login, #V91: rolreplication #: false #. is_replication,
  rolconnlimit conn_limit, rolvaliduntil valid_until,
  #V95: rolbypassrls #: false #. bypass_rls, rolconfig config,
  D.description
from pg_catalog.pg_roles R
  left join pg_catalog.pg_shdescription D on D.objoid = R.oid
;


---- StateNumber ----
select L.transactionid::varchar::bigint as transaction_id
from pg_catalog.pg_locks L
where L.transactionid is not null
order by pg_catalog.age(L.transactionid) desc
limit 1
;


---- CurrentXid ----
select pg_catalog.txid_current()::varchar::bigint as current_txid
;


---- CheckSchemaChanges ----
select pg_catalog.age(C.last_tx) <= greatest(pg_catalog.age(:tx::varchar::xid), pg_catalog.age(:src_tx::varchar::xid)) as has_changes from
(
  select X.last_tx as last_tx
  from (
    (
    select xmin as last_tx
      from pg_catalog.pg_type
      where typnamespace = :schema_id::oid
      order by pg_catalog.age(xmin)
      limit 1
    )
    union all
    (
    select xmin as last_tx
      from pg_catalog.pg_class
      where relnamespace = :schema_id::oid
      order by pg_catalog.age(xmin)
      limit 1
    )
    union all
    (
    select xmin as last_tx
      from pg_catalog.pg_proc
      where pronamespace = :schema_id::oid
      order by pg_catalog.age(xmin)
      limit 1
    )
    union all
    (
    select xmin as last_tx
      from pg_catalog.pg_operator
      where oprnamespace = :schema_id::oid
      order by pg_catalog.age(xmin)
      limit 1
    )
    union all
    (
    select A.xmin as last_tx
      from pg_catalog.pg_attribute A join pg_catalog.pg_class K on K.oid = A.attrelid
      where K.relnamespace = :schema_id::oid
      order by pg_catalog.age(A.xmin)
      limit 1
    )
    union all
    (
    select E.xmin as last_tx
      from pg_catalog.pg_enum E join pg_catalog.pg_type T on T.oid = E.enumtypid
      where T.typnamespace = :schema_id::oid
      order by pg_catalog.age(E.xmin)
      limit 1
    )
    union all
    (
    select xmin as last_tx
      from pg_catalog.pg_constraint
      where connamespace = :schema_id::oid
      order by pg_catalog.age(xmin)
      limit 1
    )
    union all
    #V91: (
    select CO.xmin as last_tx
      from pg_catalog.pg_collation CO
      where CO.collnamespace = :schema_id::oid
      order by pg_catalog.age(CO.xmin)
      limit 1
    )
    union all
    #.
    (
    select IX.xmin as last_tx
      from pg_catalog.pg_index IX,
           pg_catalog.pg_class IC
      where IC.oid = IX.indrelid
        and IC.relnamespace = :schema_id::oid
        order by pg_catalog.age(IX.xmin)
        limit 1
    )
    union all
    (
    select RU.xmin as last_tx
      from pg_catalog.pg_rewrite RU,
           pg_catalog.pg_class RC
      where RC.oid = RU.ev_class
        and RC.relnamespace = :schema_id::oid
        order by pg_catalog.age(RU.xmin)
        limit 1
    )
    union all
    (
    select TG.xmin as last_tx
      from pg_catalog.pg_trigger TG,
           pg_catalog.pg_class TC
      where TC.oid = TG.tgrelid
        and TC.relnamespace = :schema_id::oid
        order by pg_catalog.age(TG.xmin)
        limit 1
    )
    union all
    (
    select D.xmin as last_tx
      from pg_catalog.pg_description D
        join pg_catalog.pg_class C on C.oid = D.objoid
      where C.relnamespace = :schema_id::oid
        order by pg_catalog.age(D.xmin)
        limit 1
    )
    union all
    (
    select D.xmin as last_tx
      from pg_catalog.pg_description D
        join pg_catalog.pg_constraint C on C.oid = D.objoid
      where C.connamespace = :schema_id::oid
        order by pg_catalog.age(D.xmin)
        limit 1
    )
    union all
    (
    select D.xmin as last_tx
      from pg_catalog.pg_description D
        join pg_catalog.pg_trigger T on T.oid = D.objoid
        join pg_catalog.pg_class C on C.oid = T.tgrelid
      where C.relnamespace = :schema_id::oid
        order by pg_catalog.age(D.xmin)
        limit 1
    )
    union all
    (
    select D.xmin as last_tx
      from pg_catalog.pg_description D
        join pg_catalog.pg_rewrite R on R.oid = D.objoid
        join pg_catalog.pg_class C on C.oid = R.ev_class
      where C.relnamespace = :schema_id::oid
        order by pg_catalog.age(D.xmin)
        limit 1
    )
    union all
    (
    select D.xmin as last_tx
      from pg_catalog.pg_description D
        join pg_catalog.pg_proc P on P.oid = D.objoid
      where P.pronamespace = :schema_id::oid
        order by pg_catalog.age(D.xmin)
        limit 1
    )
    union all
    (
    select D.xmin as last_tx
      from pg_catalog.pg_description D
        join pg_catalog.pg_type T on T.oid = D.objoid
      where T.typnamespace = :schema_id::oid
        order by pg_catalog.age(D.xmin)
        limit 1
    )
    union all
    (
    select D.xmin as last_tx
      from pg_catalog.pg_description D
        join pg_catalog.pg_operator O on O.oid = D.objoid
      where O.oprnamespace = :schema_id::oid
        order by pg_catalog.age(D.xmin)
        limit 1
    )
    #V91:
    union all
    (
    select D.xmin as last_tx
      from pg_catalog.pg_description D
        join pg_catalog.pg_collation C on C.oid = D.objoid
      where C.collnamespace = :schema_id::oid
        order by pg_catalog.age(D.xmin)
        limit 1
    )
    #.
  ) X
  order by pg_catalog.age(X.last_tx)
  limit 1
) C
;


---- ListExistentSchemaComments ----
select D.objoid id, pg_catalog.array_agg(D.objsubid) sub_ids
from pg_catalog.pg_description D
  join pg_catalog.pg_class C on D.objoid = C.oid
where C.relnamespace = :schema_id::oid and C.relkind != 'c'
group by D.objoid
union all
select T.oid id, pg_catalog.array_agg(D.objsubid)
from pg_catalog.pg_description D
  join pg_catalog.pg_type T on T.oid = D.objoid or T.typrelid = D.objoid
  left join pg_catalog.pg_class C on T.typrelid = C.oid
where T.typnamespace = :schema_id::oid and (C.relkind = 'c' or C.relkind is null)
group by T.oid
union all
select D.objoid id, array[D.objsubid]
from pg_catalog.pg_description D
  join pg_catalog.pg_constraint C on D.objoid = C.oid
where C.connamespace = :schema_id::oid
union all
select D.objoid id, array[D.objsubid]
from pg_catalog.pg_description D
  join pg_catalog.pg_trigger T on T.oid = D.objoid
  join pg_catalog.pg_class C on C.oid = T.tgrelid
where C.relnamespace = :schema_id::oid
union all
select D.objoid id, array[D.objsubid]
from pg_catalog.pg_description D
  join pg_catalog.pg_rewrite R on R.oid = D.objoid
  join pg_catalog.pg_class C on C.oid = R.ev_class
where C.relnamespace = :schema_id::oid
union all
select D.objoid id, array[D.objsubid]
from pg_catalog.pg_description D
  join pg_catalog.pg_proc P on P.oid = D.objoid
where P.pronamespace = :schema_id::oid
union all
select D.objoid id, array[D.objsubid]
from pg_catalog.pg_description D
  join pg_catalog.pg_operator O on O.oid = D.objoid
where O.oprnamespace = :schema_id::oid
#V91:union all
select D.objoid id, array[D.objsubid]
from pg_catalog.pg_description D
  join pg_catalog.pg_collation C on C.oid = D.objoid
where C.collnamespace = :schema_id::oid#.
;

---- ListExistentDatabaseComments ----
select D.objoid id
from pg_catalog.pg_description D
where D.classoid in (
  'pg_catalog.pg_am'::regclass,
  'pg_catalog.pg_foreign_data_wrapper'::regclass,
  'pg_catalog.pg_foreign_server'::regclass
  #V91:,
  'pg_catalog.pg_extension'::regclass
  #.
)
;

---- RetrieveSchemaComments ----
select D.objoid id, C.relkind as kind, D.objsubid sub_id, D.description
from pg_catalog.pg_description D
  join pg_catalog.pg_class C on D.objoid = C.oid
where C.relnamespace = :schema_id::oid and C.relkind != 'c'
#INC> and pg_catalog.age(D.xmin) <= pg_catalog.age(:tx::varchar::xid)
union all
select T.oid id, 'T' as kind, D.objsubid sub_id, D.description
from pg_catalog.pg_description D
  join pg_catalog.pg_type T on T.oid = D.objoid or T.typrelid = D.objoid
  left join pg_catalog.pg_class C on T.typrelid = C.oid
where T.typnamespace = :schema_id::oid and (C.relkind = 'c' or C.relkind is null)
#INC> and pg_catalog.age(D.xmin) <= pg_catalog.age(:tx::varchar::xid)
union all
select D.objoid id, pg_catalog.translate(C.contype, 'pufc', 'kkxz') as kind, D.objsubid sub_id, D.description
from pg_catalog.pg_description D
  join pg_catalog.pg_constraint C on D.objoid = C.oid
where C.connamespace = :schema_id::oid
#INC> and pg_catalog.age(D.xmin) <= pg_catalog.age(:tx::varchar::xid)
union all
select D.objoid id, 't' as kind, D.objsubid sub_id, D.description
from pg_catalog.pg_description D
  join pg_catalog.pg_trigger T on T.oid = D.objoid
  join pg_catalog.pg_class C on C.oid = T.tgrelid
where C.relnamespace = :schema_id::oid
#INC> and pg_catalog.age(D.xmin) <= pg_catalog.age(:tx::varchar::xid)
union all
select D.objoid id, 'R' as kind, D.objsubid sub_id, D.description
from pg_catalog.pg_description D
  join pg_catalog.pg_rewrite R on R.oid = D.objoid
  join pg_catalog.pg_class C on C.oid = R.ev_class
where C.relnamespace = :schema_id::oid
#INC> and pg_catalog.age(D.xmin) <= pg_catalog.age(:tx::varchar::xid)
union all
select D.objoid id, 'F' as kind, D.objsubid sub_id, D.description
from pg_catalog.pg_description D
  join pg_catalog.pg_proc P on P.oid = D.objoid
where P.pronamespace = :schema_id::oid
#INC> and pg_catalog.age(D.xmin) <= pg_catalog.age(:tx::varchar::xid)
union all
select D.objoid id, 'O' as kind, D.objsubid sub_id, D.description
from pg_catalog.pg_description D
  join pg_catalog.pg_operator O on O.oid = D.objoid
where O.oprnamespace = :schema_id::oid
#INC> and pg_catalog.age(D.xmin) <= pg_catalog.age(:tx::varchar::xid)
#V91:union all
select D.objoid id, 'C' as kind, D.objsubid sub_id, D.description
from pg_catalog.pg_description D
  join pg_catalog.pg_collation C on C.oid = D.objoid
where C.collnamespace = :schema_id::oid
#INC> and pg_catalog.age(D.xmin) <= pg_catalog.age(:tx::varchar::xid)
#.


---- RetrieveDatabaseComments ----
select D.objoid id, case
    when 'pg_catalog.pg_am'::regclass = classoid then 'A'
    when 'pg_catalog.pg_foreign_data_wrapper'::regclass = classoid then 'W'
    when 'pg_catalog.pg_foreign_server'::regclass = classoid then 'S'
    #V91:
    when 'pg_catalog.pg_extension'::regclass = classoid then 'E'
    #.
  end as kind,
  D.objsubid sub_id, D.description
from pg_catalog.pg_description D
where classoid in (
  'pg_catalog.pg_am'::regclass,
  'pg_catalog.pg_foreign_data_wrapper'::regclass,
  'pg_catalog.pg_foreign_server'::regclass
  #V91:,
  'pg_catalog.pg_extension'::regclass
  #.
)
#INC> and pg_catalog.age(D.xmin) <= pg_catalog.age(:tx::varchar::xid)


---- ListExistentClasses ----
select distinct oid
from pg_catalog.pg_type
where typnamespace = :schema_id::oid
union
select distinct oid
from pg_catalog.pg_class
where relnamespace = :schema_id::oid
union
select distinct oid
from pg_catalog.pg_proc
where pronamespace = :schema_id::oid
union
select distinct oid
from pg_catalog.pg_operator
where oprnamespace = :schema_id::oid
union
select distinct oid
from pg_catalog.pg_constraint
where connamespace = :schema_id::oid
union
#V91:select distinct oid
from pg_catalog.pg_collation
where collnamespace = :schema_id::oid
union
#.
select distinct IX.indexrelid
from pg_catalog.pg_index IX,
     pg_catalog.pg_class IC
where IC.oid = IX.indrelid
  and IC.relnamespace = :schema_id::oid
union
select distinct RU.oid
from pg_catalog.pg_rewrite RU,
     pg_catalog.pg_class RC
where RC.oid = RU.ev_class
  and RC.relnamespace = :schema_id::oid
  and not RU.rulename = '_RETURN'
union
select distinct TG.oid
from pg_catalog.pg_trigger TG,
     pg_catalog.pg_class TC
where TC.oid = TG.tgrelid
  and TC.relnamespace = :schema_id::oid
;


---- ListDatabaseExistentClasses ----
select distinct oid
from pg_catalog.pg_am
union
select distinct oid
from pg_catalog.pg_foreign_data_wrapper
union
select distinct oid
from pg_catalog.pg_foreign_server
#V91:union
select distinct oid
from pg_catalog.pg_extension
#.
#SUPER:union
select distinct oid
from pg_catalog.pg_user_mapping
#.
;


---- RetrieveStates ----
select distinct oid, xmin
from pg_catalog.pg_database
where oid in (OIDS)
union
select distinct oid, xmin
from pg_catalog.pg_namespace
where oid in (OIDS)
union
select distinct oid, xmin
from pg_catalog.pg_type
where oid in (OIDS)
union
select distinct oid, xmin
from pg_catalog.pg_class
where oid in (OIDS)
union
select distinct oid, xmin
from pg_catalog.pg_proc
where oid in (OIDS)
union
select distinct oid, xmin
from pg_catalog.pg_operator
where oid in (OIDS)
union
select distinct oid, xmin
from pg_catalog.pg_constraint
where oid in (OIDS)
union
#V91:select distinct oid, xmin
from pg_catalog.pg_collation
where oid in (OIDS)
union
#.
select distinct oid, xmin
from pg_catalog.pg_rewrite
where oid in (OIDS)
union
select distinct oid, xmin
from pg_catalog.pg_trigger
where oid in (OIDS)
union
select distinct oid, xmin
from pg_catalog.pg_am
where oid in (OIDS)
union
select distinct oid, xmin
from pg_catalog.pg_foreign_data_wrapper
where oid in (OIDS)
union
select distinct oid, xmin
from pg_catalog.pg_foreign_server
where oid in (OIDS)
#V91:union
select distinct oid, xmin
from pg_catalog.pg_extension
where oid in (OIDS)
#.
#SUPER:union
select distinct oid, xmin
from pg_catalog.pg_user_mapping
where oid in (OIDS)
#.
;


---- RetrieveSequences10 ----
select cls.xmin as sequence_state_number,
       sq.seqrelid as sequence_id,
       cls.relname as sequence_name,
       pg_catalog.format_type(sq.seqtypid, null) as data_type,
       sq.seqstart as next_value,
       sq.seqincrement as inc_value,
       sq.seqmin as min_value,
       sq.seqmax as max_value,
       sq.seqcache as cache_size,
       sq.seqcycle as cycle_option,
       pg_catalog.pg_get_userbyid(cls.relowner) as "owner"
from pg_catalog.pg_sequence sq
    join pg_class cls on sq.seqrelid = cls.oid
    where cls.relnamespace = :schema_id::oid
          #INC> and pg_catalog.age(cls.xmin) <= pg_catalog.age(:tx::varchar::xid)
;


---- RetrieveSequences ----
with #V91>U as (select usesysid from pg_catalog.pg_user where usename = current_user),
          T as (
                select oid as sequence_id,
                       relname as sequence_name,
                       pg_catalog.pg_get_userbyid(relowner) as "owner",
                       #V91:
                       case when relowner = (select usesysid from U) then trim(both '()' from pg_catalog.pg_sequence_parameters(oid)::varchar)
                       else null::varchar
                       end as sequence_params,
                       #.
                       xmin as sequence_state_number
                       from pg_catalog.pg_class
                         where relnamespace = :schema_id::oid
                         and relkind = 'S'
                         #INC> and pg_catalog.age(xmin) <= pg_catalog.age(:tx::varchar::xid)
                )
select sequence_id,
       sequence_name,
       #V91:
       pg_catalog.split_part(sequence_params, ',', 1)::bigint as next_value,
       pg_catalog.split_part(sequence_params, ',', 2)::bigint as min_value,
       pg_catalog.split_part(sequence_params, ',', 3)::bigint as max_value,
       pg_catalog.split_part(sequence_params, ',', 4)::bigint as inc_value,
       pg_catalog.lower(pg_catalog.split_part(sequence_params, ',', 5)) LIKE '%t%' as cycle_option,
       #.
       sequence_state_number,
       "owner"
from T
;


---- RetrieveDataTypes ----
select T.oid as type_id,
       T.xmin as type_state_number,
       T.typname as type_name,
       T.typtype as type_sub_kind,
       T.typcategory as type_category,
       T.typrelid as class_id,
       T.typbasetype as base_type_id,
       case when T.typtype in ('c','e') then null
            else pg_catalog.format_type(T.typbasetype, T.typtypmod) end as type_def,
       T.typdefault as default_expression,
       T.typnotnull as mandatory,
       pg_catalog.pg_get_userbyid(T.typowner) as "owner"
from pg_catalog.pg_type T
         left outer join pg_catalog.pg_class C
             on T.typrelid = C.oid
where T.typnamespace = :schema_id::oid
  #INC> and pg_catalog.age(T.xmin) <= pg_catalog.age(:tx::varchar::xid)
  and (T.typtype in ('d','e') or C.relkind = 'c'::"char" or T.typtype = 'b' and (T.typelem = 0 OR T.typcategory <> 'A'))
order by 1
;


---- RetrieveDataTypeChecks ----
select T.oid::bigint as type_id,
       C.oid::bigint as constraint_id,
       C.xmin as constraint_state_number,
       C.conname as constraint_name,
       C.consrc as predicate
from pg_catalog.pg_type T,
     pg_catalog.pg_constraint C
where T.typnamespace = :schema_id::oid
  and T.typtype = 'd'::"char"
  and T.oid = C.contypid
  #INC> and (pg_catalog.age(T.xmin) <= pg_catalog.age(:tx::varchar::xid) or pg_catalog.age(C.xmin) <= pg_catalog.age(:tx::varchar::xid))
order by constraint_name
;


---- RetrieveTables ----
select T.relkind as table_kind,
       T.relname as table_name,
       T.oid as table_id,
       T.xmin as table_state_number,
       T.relhasoids as table_with_oids,
       T.relnamespace as tablespace_id,
       T.reloptions as options,
       (select pg_catalog.array_agg(inhparent)::varchar from (select inhparent::bigint from pg_catalog.pg_inherits where T.oid = inhrelid order by inhseqno) as bts) as ancestors,
       (select pg_catalog.array_agg(inhrelid)::varchar from (select inhrelid::bigint from pg_catalog.pg_inherits where T.oid = inhparent order by inhrelid) as its) as successors,
       #V99: T.relispartition #: false #. as is_partition,
       #V99: pg_catalog.pg_get_partkeydef(T.oid) #: null #. as partition_key,
       #V99: pg_catalog.pg_get_expr(T.relpartbound, T.oid) #: null #. as partition_expression,
       pg_catalog.pg_get_userbyid(T.relowner) as "owner"
from pg_catalog.pg_class T
where relnamespace = :schema_id::oid
  and relkind in ('r', 'm', 'v', 'f', 'p')
  #INC> and pg_catalog.age(T.xmin) <= pg_catalog.age(:tx::varchar::xid)
order by table_kind, table_id
;


---- RetrieveForeignTables ----
select ft.ftrelid as table_id,
       srv.srvname as table_server,
       ft.ftoptions as table_options,
       pg_catalog.pg_get_userbyid(cls.relowner) as "owner"
from pg_catalog.pg_foreign_table ft
     left outer join pg_catalog.pg_foreign_server srv on ft.ftserver = srv.oid
     join pg_catalog.pg_class cls on ft.ftrelid = cls.oid
where cls.relnamespace = :schema_id::oid
  #INC> and pg_catalog.age(ft.xmin) <= pg_catalog.age(:tx::varchar::xid)
order by table_id
;


---- RetrieveStructAttributes ----
with types      as (select distinct
                           T.oid as type_id,
                           C.oid as class_id
                    from pg_catalog.pg_type T
                    join pg_catalog.pg_class C
                      on T.typrelid = C.oid
                    join pg_catalog.pg_attribute A
                      on C.oid = A.attrelid
                    where T.typnamespace = :schema_id::oid
                      #INC> and pg_catalog.age(A.xmin) <= pg_catalog.age(:tx::varchar::xid)
                      and T.typtype = 'c'::"char"
                      and C.relkind = 'c'::"char"),
     attributes as (select A.attrelid as class_id,
                           A.attnum   as att_position,
                           A.attname  as att_name,
                           pg_catalog.format_type(A.atttypid, A.atttypmod) as att_type,
                           A.attisdropped as att_dropped
                    from pg_catalog.pg_attribute A)
select type_id, att_position, att_name, att_type, att_dropped
from types natural join attributes
order by type_id::bigint, att_position
;


---- RetrieveEnumLabels ----
with types as (select distinct T.oid as type_id
               from pg_catalog.pg_type T
               join pg_catalog.pg_enum E
                 on T.oid = E.enumtypid
               where T.typnamespace = :schema_id::oid
                 and T.typtype = 'e'::"char"
               #INC> and pg_catalog.age(E.xmin) <= pg_catalog.age(:tx::varchar::xid)
               )
select T.type_id as enum_id,
       pg_catalog.array_agg(E.enumlabel::varchar #V91: order by E.enumsortorder #.) as enum_labels
from types T
join pg_catalog.pg_enum E
  on T.type_id = E.enumtypid
group by T.type_id
;


---- RetrieveColumns ----
with T as ( select distinct T.oid as table_id, T.relname as table_name
            from pg_catalog.pg_class T,
                 pg_catalog.pg_attribute A
            where T.relnamespace = :schema_id::oid
              and T.relkind in ('r', 'm', 'v', 'f', 'p')
              #INC> and pg_catalog.age(A.xmin) <= pg_catalog.age(:tx::varchar::xid)
              and A.attrelid = T.oid )
select T.table_id,
       C.attnum as column_position,
       C.attname as column_name,
       C.xmin as column_state_number,
       C.atttypmod as type_mod,
       C.attndims as dimensions_number,
       pg_catalog.format_type(C.atttypid, C.atttypmod) as type_spec,
       C.atttypid as type_id,
       C.attnotnull as mandatory,
       D.adsrc as column_default_expression,
       not C.attislocal as column_is_inherited,
       #V92> C.attfdwoptions as options,
       C.attisdropped as column_is_dropped
from T
  inner join pg_catalog.pg_attribute C on T.table_id = C.attrelid
  left outer join pg_catalog.pg_attrdef D on (C.attrelid, C.attnum) = (D.adrelid, D.adnum)
where attnum > 0
order by table_id, attnum
;


---- RetrieveIndices ----
with tab as ( select relnamespace as schema_id,
                     oid as table_id,
                     relkind as table_kind
              from pg_catalog.pg_class
              where relnamespace = :schema_id::oid
                and relkind in ('r','m','v','p')),
    ind_head as ( select indrelid as table_id,
                         indexrelid as index_id,
                         indisunique as is_unique,
                         indisprimary as is_primary,
                         indkey as column_positions,
                         indclass as column_operators,
                         indoption as column_options,
                         pg_catalog.pg_get_expr(indpred, indrelid) as condition,
                         case when indexprs is not null
                             then pg_catalog.array_to_string(array(
                                      select case when indkey[k] = 0
                                             then chr(27) || pg_catalog.pg_get_indexdef(indexrelid, k + 1, true)
                                             else pg_catalog.pg_get_indexdef(indexrelid, k + 1, true)
                                             end
                                      from pg_catalog.generate_subscripts(indkey, 1) as k
                                      order by k),
                                  chr(28))
                             else null end as expressions
                  from pg_catalog.pg_index ),
    ind_stor as ( select cls.relnamespace as schema_id,
                         cls.oid as index_id,
                         cls.xmin as state_number,
                         cls.relname as index_name,
                         #V96:pg_catalog.pg_indexam_has_property(cls.relam, 'can_order') #: am.amcanorder #. as can_order
                  from pg_catalog.pg_class cls
                    #V96:#:join pg_catalog.pg_am am on (cls.relam = am.oid)#.
                  where cls.relkind = 'i'
                    #INC: and pg_catalog.age(cls.xmin) <= pg_catalog.age(:tx::varchar::xid) #.)
select table_id, table_kind,
       index_name, index_id, state_number,
       is_unique, is_primary,
       column_positions, column_operators, column_options,
       condition, expressions, can_order
from tab natural join ind_head natural join ind_stor
;


---- RetrieveConstraints ----
with T as ( select oid as table_id,
                   relkind as table_kind,
                   xmin as table_state_id
            from pg_catalog.pg_class
            where relkind in ('r','v','f','p')
              and relnamespace = :schema_id::oid ),
     C as ( select conrelid as table_id,
                   oid::bigint as con_id,
                   xmin as con_state_id,
                   conname as con_name,
                   contype as con_kind,
                   conkey as con_columns,
                   conindid as index_id,
                   confrelid as ref_table_id,
                   confkey as ref_columns,
                   condeferrable as is_deferrable,
                   condeferred as is_init_deferred,
                   confupdtype as on_update,
                   confdeltype as on_delete,
                   consrc as con_expression
            from pg_catalog.pg_constraint
            where contype in ('p','u','f','c')
              and connamespace = :schema_id::oid )
select table_id,
       table_kind,
       con_id,
       con_state_id::varchar::bigint as con_state_id,
       con_name,
       con_kind,
       con_columns,
       index_id,
       ref_table_id,
       is_deferrable,
       is_init_deferred,
       on_update,
       on_delete,
       con_expression,
       ref_columns
from T natural join C
#INC> where pg_catalog.age(table_state_id) <= pg_catalog.age(:tx::varchar::xid) or pg_catalog.age(con_state_id) <= pg_catalog.age(:tx::varchar::xid)
order by pg_catalog.strpos('pufc', con_kind)
;


---- RetrieveUsedArgumentTypes ----
with schema_procs as (select prorettype, proargtypes, proallargtypes
                      from pg_catalog.pg_proc
                      where pronamespace = :schema_id::oid
                        #INC: and pg_catalog.age(xmin) <= pg_catalog.age(:tx::varchar::xid) #. ),
     schema_opers as (select oprleft, oprright, oprresult
                      from pg_catalog.pg_operator
                      where oprnamespace = :schema_id::oid
                        #INC: and pg_catalog.age(xmin) <= pg_catalog.age(:tx::varchar::xid) #. ),
     schema_aggregates as (select A.aggtranstype #V94:, A.aggmtranstype #.
                           from pg_catalog.pg_aggregate A
                           join pg_catalog.pg_proc P
                             on A.aggfnoid = P.oid
                           where P.pronamespace = :schema_id::oid
                           #INC: and (pg_catalog.age(A.xmin) <= pg_catalog.age(:tx::varchar::xid) or pg_catalog.age(P.xmin) <= pg_catalog.age(:tx::varchar::xid)) #.),
     schema_arg_types as ( select prorettype as type_id
                           from schema_procs
                           union
                           select distinct unnest(proargtypes) as type_id
                           from schema_procs
                           union
                           select distinct unnest(proallargtypes) as type_id
                           from schema_procs
                           union
                           select oprleft as type_id
                           from schema_opers
                           where oprleft is not null
                           union
                           select oprright as type_id
                           from schema_opers
                           where oprright is not null
                           union
                           select oprresult as type_id
                           from schema_opers
                           where oprresult is not null
                           union
                           select aggtranstype::oid as type_id
                           from schema_aggregates
                           #V94:
                           union
                           select aggmtranstype::oid as type_id
                           from schema_aggregates
                           #.
                           )
select type_id, pg_catalog.format_type(type_id, null) as type_spec
from schema_arg_types
where type_id <> 0
;


---- RetrieveRoutines ----
with languages as (select oid as lang_oid, lanname as lang
                   from pg_catalog.pg_language),
     routines as (select proname as r_name,
                         prolang as lang_oid,
                         oid as r_id,
                         xmin as r_state_number,
                         proargnames as arg_names,
                         proargmodes as arg_modes,
                         proargtypes::int[] as in_arg_types,
                         proallargtypes::int[] as all_arg_types,
                         pg_catalog.pg_get_expr(proargdefaults, 0) as arg_defaults,
                         provariadic as arg_variadic_id,
                         prorettype as ret_type_id,
                         proretset as ret_set,
                         #V110: prokind #: case when proiswindow then 'w'
                                                when proisagg then 'a'
                                                else 'f'
                                           end #. as kind,
                         provolatile as volatile_kind,
                         proisstrict as is_strict,
                         prosecdef as is_security_definer,
                         proconfig as configuration_parameters,
                         procost as cost,
                         pg_catalog.pg_get_userbyid(proowner) as "owner",
                         prorows as rows #V92:,
                         proleakproof as is_leakproof #. #V96:,
                         proparallel as concurrency_kind #.
                  from pg_catalog.pg_proc
                  where pronamespace = :schema_id::oid
                    and not #V110: (prokind = 'a') #: proisagg #.
                    #INC: and pg_catalog.age(xmin) <= pg_catalog.age(:tx::varchar::xid) #.)
select *
from routines natural join languages
;


---- RetrieveAggregates ----
with procedures as (
  select P.oid as function_id,
         pg_catalog.quote_ident(N.nspname) || '.' || pg_catalog.quote_ident(P.proname) as function_name,
         P.prorettype as return_type
  from pg_catalog.pg_proc P
  join pg_catalog.pg_namespace N
    on P.pronamespace = N.oid
),
operators as (
  select O.oid as operator_id,
         pg_catalog.quote_ident(N.nspname) || '.' || O.oprname as operator_name
  from pg_catalog.pg_operator O
  join pg_catalog.pg_namespace N
    on O.oprnamespace = N.oid
)
select P.oid as aggregate_id,
       P.xmin as state_number,
       P.proname as aggregate_name,
       P.proargnames as arg_names,
       P.proargmodes as arg_modes,
       P.proargtypes::int[] as in_arg_types,
       P.proallargtypes::int[] as all_arg_types,
       A.aggtransfn::oid as transition_function_id,
       PT.function_name::text as transition_function_name,
       A.aggtranstype as transition_type,
       A.aggfinalfn::oid as final_function_id,
       case when A.aggfinalfn::oid = 0 then null else PF.function_name end as final_function_name,
       case when A.aggfinalfn::oid = 0 then 0 else PF.return_type end as final_return_type,
       A.agginitval as initial_value,
       A.aggsortop as sort_operator_id,
       case when A.aggsortop = 0 then null else O.operator_name end as sort_operator_name,
       pg_catalog.pg_get_userbyid(P.proowner) as "owner"
       #V94:,
       A.aggfinalextra as final_extra,
       A.aggtransspace as state_size,
       A.aggmtransfn::oid as moving_transition_id,
       case when A.aggmtransfn::oid = 0 then null else FTP.function_name end as moving_transition_name,
       A.aggminvtransfn::oid as inverse_transition_id,
       case when A.aggminvtransfn::oid = 0 then null else ITP.function_name end as inverse_transition_name,
       A.aggmtranstype::oid as moving_state_type,
       A.aggmtransspace as moving_state_size,
       A.aggmfinalfn::oid as moving_final_id,
       case when A.aggmfinalfn::oid = 0 then null else FFP.function_name end as moving_final_name,
       A.aggmfinalextra as moving_final_extra,
       A.aggminitval as moving_initial_value,
       A.aggkind as aggregate_kind,
       A.aggnumdirectargs as direct_args
       #.
       #V96:,
       A.aggcombinefn::oid as combine_function_id,
       case when A.aggcombinefn::oid = 0 then null else CP.function_name end as combine_function_name,
       A.aggserialfn::oid as serialization_function_id,
       case when A.aggserialfn::oid = 0 then null else SP.function_name end as serialization_function_name,
       A.aggdeserialfn::oid as deserialization_function_id,
       case when A.aggdeserialfn::oid = 0 then null else DP.function_name end as deserialization_function_name,
       P.proparallel as concurrency_kind
       #.
from pg_catalog.pg_aggregate A
join pg_catalog.pg_proc P
  on A.aggfnoid = P.oid
join procedures PT
  on A.aggtransfn::oid = PT.function_id
left outer join procedures PF
  on A.aggfinalfn::oid = PF.function_id
left outer join operators O
  on A.aggsortop = O.operator_id
#V94:
left outer join procedures FTP
  on A.aggmtransfn = FTP.function_id
left outer join procedures ITP
  on A.aggminvtransfn = ITP.function_id
left outer join procedures FFP
  on A.aggmfinalfn = FFP.function_id
#.
#V96:
left outer join procedures CP
  on A.aggcombinefn = CP.function_id
left outer join procedures SP
  on A.aggserialfn = SP.function_id
left outer join procedures DP
  on A.aggdeserialfn = DP.function_id
#.
where P.pronamespace = :schema_id::oid
#INC> and (pg_catalog.age(A.xmin) <= pg_catalog.age(:tx::varchar::xid) or pg_catalog.age(P.xmin) <= pg_catalog.age(:tx::varchar::xid))
order by P.oid
;


---- RetrieveExtensions ----
select pg_extension.oid as id,
       pg_extension.xmin as state_number,
       extname as name,
       extversion as version,
       extnamespace as schema_id,
       nspname as schema_name,
       array(select target
             from pg_extension_update_paths(extname)
             where source = extversion and path is not null) as available_updates
from pg_catalog.pg_extension
  join pg_namespace on pg_extension.extnamespace = pg_namespace.oid
#INC> where pg_catalog.age(pg_extension.xmin) <= pg_catalog.age(:tx::varchar::xid)
;


---- RetrieveOperators ----
select O.oid as op_id,
       O.xmin as state_number,
       oprname as op_name,
       oprkind as op_kind,
       oprleft as arg_left_type_id,
       oprright as arg_right_type_id,
       oprresult as arg_result_type_id,
       oprcode as func_main,
       pg_catalog.pg_get_userbyid(O.oprowner) as "owner"
from pg_catalog.pg_operator O
where oprnamespace = :schema_id::oid
  #INC> and pg_catalog.age(xmin) <= pg_catalog.age(:tx::varchar::xid)
;


---- RetrieveRules ----
select R.ev_class as table_id,
       R.oid as rule_id,
       R.xmin as rule_state_number,
       R.rulename as rule_name,
       pg_catalog.translate(ev_type,'1234','SUID') as rule_event_code,
       R.ev_enabled as rule_fire_mode,
       R.is_instead as rule_is_instead
from pg_catalog.pg_rewrite R
where R.ev_class in (select oid
                     from pg_catalog.pg_class
                     where relnamespace = :schema_id::oid)
  #INC> and pg_catalog.age(R.xmin) <= pg_catalog.age(:tx::varchar::xid)
  and R.rulename != '_RETURN'::name
order by R.ev_class::bigint, ev_type
;


---- RetrieveTriggers ----
select T.tgrelid as table_id,
       T.oid as trigger_id,
       T.xmin as trigger_state_number,
       T.tgname as trigger_name,
       T.tgfoid as function_id,
       T.tgtype & 31 as bits,
       T.tgdeferrable as is_deferrable,
       T.tginitdeferred as is_init_deferred,
       T.tgenabled as trigger_fire_mode
from pg_catalog.pg_trigger T
where T.tgrelid in (select oid
                    from pg_catalog.pg_class
                    where relnamespace = :schema_id::oid)
  #INC> and pg_catalog.age(T.xmin) <= pg_catalog.age(:tx::varchar::xid)
  and not T.tgisinternal
;


---- RetrieveDescriptions ----
select T.relkind as table_kind,
       T.oid as table_id,
       D.objsubid as column_position,
       D.description
from pg_catalog.pg_class T,
     pg_catalog.pg_description D
where T.relnamespace = :schema_id::oid
  and T.relkind in ('r', 'm', 'v', 'p')
  and T.oid = D.objoid
  #INC> and pg_catalog.age(D.xmin) <= pg_catalog.age(:tx::varchar::xid)
order by table_id, column_position
;


---- RetrieveViewSources ----
select distinct
       T.relkind as view_kind,
       T.oid as view_id,
       pg_catalog.pg_get_viewdef(T.oid) as source_text
from pg_catalog.pg_class T
  join pg_catalog.pg_namespace N on T.relnamespace = N.oid
  left outer join pg_catalog.pg_attribute A on (A.attrelid = T.oid)
where N.oid = :schema_id::oid
  and T.relkind in ('m','v')
  #INCSRC> and (pg_catalog.age(T.xmin) <= pg_catalog.age(:src_tx::varchar::xid) or pg_catalog.age(A.xmin) <= pg_catalog.age(:src_tx::varchar::xid))
;


---- RetrieveRuleSources ----
with A as ( select oid as table_id, pg_catalog.upper(relkind) as table_kind
            from pg_catalog.pg_class
            where relnamespace = :schema_id::oid
              and relkind in ('r', 'm', 'v', 'f', 'p') )
select table_kind,
       table_id,
       R.oid as rule_id,
       pg_catalog.pg_get_ruledef(R.oid) as source_text
from A join pg_catalog.pg_rewrite R
        on A.table_id = R.ev_class
where R.rulename != '_RETURN'::name
  #INC> and pg_catalog.age(R.xmin) <= pg_catalog.age(:src_tx::varchar::xid)
;


---- RetrieveFunctionSources ----
with system_languages as ( select oid as lang
                           from pg_catalog.pg_language
                           where lanname in ('c','internal') )
select oid as id,
       pg_catalog.pg_get_function_arguments(oid) as arguments_def,
       pg_catalog.pg_get_function_result(oid) as result_def,
       prosrc as source_text
from pg_catalog.pg_proc
where pronamespace = :schema_id::oid
  #INC> and pg_catalog.age(xmin) <= pg_catalog.age(:src_tx::varchar::xid)
  and not #V110: (prokind = 'a') #: proisagg #.
  and prolang not in (select lang from system_languages)
  and prosrc is not null
;


---- RetrieveAccessMethods ----
select A.oid as access_method_id,
       A.xmin as state_number,
       A.amname as access_method_name
       #V96:
       ,
       A.amhandler::oid as handler_id,
       pg_catalog.quote_ident(N.nspname) || '.' || pg_catalog.quote_ident(P.proname) as handler_name,
       A.amtype as access_method_type
       #.
from pg_am A
  #V96:
  join pg_proc P on A.amhandler::oid = P.oid
  join pg_namespace N on P.pronamespace = N.oid
  #.
#INC> where pg_catalog.age(A.xmin) <= pg_catalog.age(:tx::varchar::xid)
;


---- RetrieveCollations ----
select oid as id,
       xmin as state_number,
       collname as name,
       collcollate as lc_collate,
       collctype as lc_ctype,
       pg_catalog.pg_get_userbyid(collowner) as "owner"
from pg_catalog.pg_collation
where collnamespace = :schema_id::oid
  #INC> and pg_catalog.age(xmin) <= pg_catalog.age(:tx::varchar::xid)
;


---- RetrieveForeignDataWrappers ----
select fdw.oid as id,
       fdw.xmin as state_number,
       fdw.fdwname as name,
       #V91>pr.proname as handler,
       #V91>nspc.nspname as handler_schema,
       pr2.proname as validator,
       nspc2.nspname as validator_schema,
       fdw.fdwoptions as options,
       pg_catalog.pg_get_userbyid(fdw.fdwowner) as "owner"
from pg_catalog.pg_foreign_data_wrapper fdw
     #V91>left outer join pg_catalog.pg_proc pr on fdw.fdwhandler = pr.oid
     #V91>left outer join pg_catalog.pg_namespace nspc on pr.pronamespace = nspc.oid
     left outer join pg_catalog.pg_proc pr2 on fdw.fdwvalidator = pr2.oid
     left outer join pg_catalog.pg_namespace nspc2 on pr2.pronamespace = nspc2.oid
  #INC> where pg_catalog.age(fdw.xmin) <= pg_catalog.age(:tx::varchar::xid)
;


---- RetrieveForeignServers ----
select srv.oid as id,
       srv.srvfdw as fdw_id,
       srv.xmin as state_number,
       srv.srvname as name,
       srv.srvtype as type,
       srv.srvversion as version,
       srv.srvoptions as options,
       pg_catalog.pg_get_userbyid(srv.srvowner) as "owner"
from pg_catalog.pg_foreign_server srv
  #INC> where pg_catalog.age(srv.xmin) <= pg_catalog.age(:tx::varchar::xid)
;


---- RetrieveUserMappingsSuperUser ----
select um.oid as id,
       um.xmin as state_number,
       case when um.umuser = 0 then null else pg_catalog.pg_get_userbyid(um.umuser) end as user,
       um.umoptions as options
from pg_catalog.pg_user_mapping um
where um.umserver = :server_id::oid
  #INC> and pg_catalog.age(um.xmin) <= pg_catalog.age(:tx::varchar::xid)
;


---- RetrieveUserMappings ----
select um.umid as id,
       um.usename as user,
       um.umoptions as options
from pg_catalog.pg_user_mappings um
where um.srvid = :server_id::oid
;


---- ListReferencingSchemas ----
select distinct connamespace as schema_id
from pg_catalog.pg_constraint F,
     pg_catalog.pg_class O
where F.contype = 'f'
  and F.confrelid = O.oid
  and O.relnamespace in (SCHEMAIDS)
;


---- RetrieveRelations ----
select 'sequence' as dependent_kind,
       'table' as owner_kind,
       D.objid as dependent_id,
       D.refobjid as owner_id,
       D.refobjsubid as owner_subobject_id
from pg_depend D
  join pg_class C_SEQ on D.objid = C_SEQ.oid
  join pg_class C_TAB on D.refobjid = C_TAB.oid
where C_SEQ.relkind = 'S'
  and C_TAB.relkind = 'r'
  and D.refobjsubid <> 0
  and D.deptype = 'a'