// Copyright 2000-2018 JetBrains s.r.o.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
// http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
package com.jetbrains.php.lang.classAlias;

import com.intellij.openapi.util.text.StringUtil;
import com.intellij.psi.PsiElement;
import com.jetbrains.php.lang.psi.elements.FunctionReference;
import com.jetbrains.php.lang.psi.elements.MethodReference;
import com.jetbrains.php.lang.psi.visitors.PhpRecursiveElementVisitor;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.function.BiConsumer;

import static com.intellij.util.containers.ContainerUtil.newArrayList;

public final class PhpClassAliasUtil {

  /**
   * The method is intended to return a list of providers relevant to the text.
   *
   * @param text to be searched for class alias declarations
   * @return list of providers
   * @see PhpClassAliasProvider#getSearchString()
   */
  public static List<PhpClassAliasProvider> getRelevantProviders(@NotNull final CharSequence text) {
    final List<PhpClassAliasProvider> providers = newArrayList();
    for (final PhpClassAliasProvider provider : PhpClassAliasProvider.PROVIDERS) {
      if (StringUtil.contains(text, provider.getSearchString())) {
        providers.add(provider);
      }
    }
    return providers;
  }

  /**
   * The method is intended to help processing function/method references which are class alias declarations.
   *
   * @param scope    to find class alias declarations in
   * @param consumer to be provided with a reference and an extract result, both are never {@code null}
   * @see PhpClassAliasUtil#processClassAliasDeclarations(PsiElement, Collection, BiConsumer)
   */
  public static void processClassAliasDeclarations(@NotNull final PsiElement scope,
                                                   @NotNull final BiConsumer<? super FunctionReference, ? super ExtractResult> consumer) {
    processClassAliasDeclarations(scope, Arrays.asList(PhpClassAliasProvider.PROVIDERS), consumer);
  }

  /**
   * The method is intended to help processing function/method references which are class alias declarations.
   * Working with a file it typically makes sense to filter providers first using {@link PhpClassAliasProvider#getSearchString()}
   * and only then checking against applicable providers or skipping the check at all.
   *
   * @param scope     to find class alias declarations in
   * @param providers to be checked against
   * @param consumer  to be provided with a reference and an extract result, both are never {@code null}
   * @see PhpClassAliasUtil#processClassAliasDeclarations(PsiElement, BiConsumer)
   */
  public static void processClassAliasDeclarations(@NotNull final PsiElement scope,
                                                   @NotNull final Collection<? extends PhpClassAliasProvider> providers,
                                                   @NotNull final BiConsumer<? super FunctionReference, ? super ExtractResult> consumer) {
    if (!providers.isEmpty()) {
      scope.accept(new PhpRecursiveElementVisitor() {
        @Override
        public void visitPhpFunctionCall(final FunctionReference reference) {
          process(reference);
        }

        @Override
        public void visitPhpMethodReference(final MethodReference reference) {
          process(reference);
        }

        private void process(final FunctionReference reference) {
          final ExtractResult result = extractClassAlias(reference, providers);
          if (result != null) {
            consumer.accept(reference, result);
          }
        }
      });
    }
  }

  /**
   * The method is intended to extract a class alias declaration from a function/method reference.
   *
   * @param reference which probably contains a class alias declaration
   * @return a pair of FQNs or {@code null} if the reference doesn't contain a class alias
   * @see PhpClassAliasUtil#extractClassAlias(FunctionReference, Collection)
   */
  @Nullable
  public static ExtractResult extractClassAlias(@NotNull final FunctionReference reference) {
    return extractClassAlias(reference, Arrays.asList(PhpClassAliasProvider.PROVIDERS));
  }

  /**
   * The method is intended to extract a class alias declaration from a function/method reference.
   * Working with a file it typically makes sense to filter providers first using {@link PhpClassAliasProvider#getSearchString()}
   * and only then checking against applicable providers or skipping the check at all.
   *
   * @param reference which probably contains a class alias declaration
   * @param providers to be checked against
   * @return a pair of FQNs or {@code null} if the reference doesn't contain a class alias
   * @see PhpClassAliasUtil#extractClassAlias(FunctionReference)
   */
  @Nullable
  public static ExtractResult extractClassAlias(@NotNull final FunctionReference reference,
                                                @NotNull final Collection<? extends PhpClassAliasProvider> providers) {
    for (final PhpClassAliasProvider provider : providers) {
      if (provider.isClassAliasDeclaration(reference)) {
        final String aliasFqn = provider.getAliasFqn(reference);
        final String originalFqn = provider.getOriginalFqn(reference);
        if (aliasFqn != null && originalFqn != null) {
          return new ExtractResult(aliasFqn, originalFqn);
        }
      }
    }
    return null;
  }

  public static final class ExtractResult {

    private final String myAliasFqn;
    private final String myOriginalFqn;

    private ExtractResult(@NotNull final String aliasFqn, @NotNull final String originalFqn) {
      myAliasFqn = aliasFqn;
      myOriginalFqn = originalFqn;
    }

    @NotNull
    public String getAliasFqn() {
      return myAliasFqn;
    }

    @NotNull
    public String getOriginalFqn() {
      return myOriginalFqn;
    }
  }
}
