/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db.compaction;

import com.google.common.base.Predicate;
import com.google.common.base.Predicates;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;
import org.apache.cassandra.db.ColumnFamily;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.DecoratedKey;
import org.apache.cassandra.db.Memtable;
import org.apache.cassandra.db.RowPosition;
import org.apache.cassandra.db.lifecycle.SSTableIntervalTree;
import org.apache.cassandra.io.sstable.format.SSTableReader;
import org.apache.cassandra.utils.AlwaysPresentFilter;
import org.apache.cassandra.utils.OverlapIterator;
import org.apache.cassandra.utils.concurrent.Refs;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CompactionController
implements AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(CompactionController.class);
    static final boolean NEVER_PURGE_TOMBSTONES = Boolean.getBoolean("cassandra.never_purge_tombstones");
    public final ColumnFamilyStore cfs;
    private Refs<SSTableReader> overlappingSSTables;
    private OverlapIterator<RowPosition, SSTableReader> overlapIterator;
    private final Iterable<SSTableReader> compacting;
    public final int gcBefore;

    protected CompactionController(ColumnFamilyStore cfs, int maxValue) {
        this(cfs, null, maxValue);
    }

    public CompactionController(ColumnFamilyStore cfs, Set<SSTableReader> compacting, int gcBefore) {
        assert (cfs != null);
        this.cfs = cfs;
        this.gcBefore = gcBefore;
        this.compacting = compacting;
        this.refreshOverlaps();
        if (NEVER_PURGE_TOMBSTONES) {
            logger.warn("You are running with -Dcassandra.never_purge_tombstones=true, this is dangerous!");
        }
    }

    void maybeRefreshOverlaps() {
        if (NEVER_PURGE_TOMBSTONES) {
            logger.debug("not refreshing overlaps - running with -Dcassandra.never_purge_tombstones=true");
            return;
        }
        for (SSTableReader reader : this.overlappingSSTables) {
            if (!reader.isMarkedCompacted()) continue;
            this.refreshOverlaps();
            return;
        }
    }

    private void refreshOverlaps() {
        if (NEVER_PURGE_TOMBSTONES) {
            return;
        }
        if (this.overlappingSSTables != null) {
            this.overlappingSSTables.release();
        }
        this.overlappingSSTables = this.compacting == null ? Refs.tryRef(Collections.emptyList()) : this.cfs.getAndReferenceOverlappingSSTables(this.compacting);
        this.overlapIterator = new OverlapIterator(SSTableIntervalTree.buildIntervals(this.overlappingSSTables));
    }

    public Set<SSTableReader> getFullyExpiredSSTables() {
        return CompactionController.getFullyExpiredSSTables(this.cfs, this.compacting, this.overlappingSSTables, this.gcBefore);
    }

    public static Set<SSTableReader> getFullyExpiredSSTables(ColumnFamilyStore cfStore, Iterable<SSTableReader> compacting, Iterable<SSTableReader> overlapping, int gcBefore) {
        logger.trace("Checking droppable sstables in {}", (Object)cfStore);
        if (compacting == null || NEVER_PURGE_TOMBSTONES) {
            return Collections.emptySet();
        }
        ArrayList<SSTableReader> candidates = new ArrayList<SSTableReader>();
        long minTimestamp = Long.MAX_VALUE;
        for (SSTableReader sstable : overlapping) {
            if (sstable.getSSTableMetadata().maxLocalDeletionTime < gcBefore) continue;
            minTimestamp = Math.min(minTimestamp, sstable.getMinTimestamp());
        }
        for (SSTableReader candidate : compacting) {
            if (candidate.getSSTableMetadata().maxLocalDeletionTime < gcBefore) {
                candidates.add(candidate);
                continue;
            }
            minTimestamp = Math.min(minTimestamp, candidate.getMinTimestamp());
        }
        for (Memtable memtable : cfStore.getTracker().getView().getAllMemtables()) {
            minTimestamp = Math.min(minTimestamp, memtable.getMinTimestamp());
        }
        Iterator iterator = candidates.iterator();
        while (iterator.hasNext()) {
            SSTableReader candidate;
            candidate = (SSTableReader)iterator.next();
            if (candidate.getMaxTimestamp() >= minTimestamp) {
                iterator.remove();
                continue;
            }
            logger.trace("Dropping expired SSTable {} (maxLocalDeletionTime={}, gcBefore={})", new Object[]{candidate, candidate.getSSTableMetadata().maxLocalDeletionTime, gcBefore});
        }
        return new HashSet<SSTableReader>(candidates);
    }

    public String getKeyspace() {
        return this.cfs.keyspace.getName();
    }

    public String getColumnFamily() {
        return this.cfs.name;
    }

    public Predicate<Long> getPurgeEvaluator(DecoratedKey key) {
        if (NEVER_PURGE_TOMBSTONES) {
            return Predicates.alwaysFalse();
        }
        this.overlapIterator.update(key);
        Set<SSTableReader> filteredSSTables = this.overlapIterator.overlaps();
        Iterable<Memtable> memtables = this.cfs.getTracker().getView().getAllMemtables();
        long minTimestampSeen = Long.MAX_VALUE;
        boolean hasTimestamp = false;
        for (SSTableReader sstable : filteredSSTables) {
            if ((!(sstable.getBloomFilter() instanceof AlwaysPresentFilter) || sstable.getPosition(key, SSTableReader.Operator.EQ, false) == null) && !sstable.getBloomFilter().isPresent(key)) continue;
            minTimestampSeen = Math.min(minTimestampSeen, sstable.getMinTimestamp());
            hasTimestamp = true;
        }
        for (Memtable memtable : memtables) {
            ColumnFamily cf = memtable.getColumnFamily(key);
            if (cf == null) continue;
            minTimestampSeen = Math.min(minTimestampSeen, memtable.getMinTimestamp());
            hasTimestamp = true;
        }
        if (!hasTimestamp) {
            return Predicates.alwaysTrue();
        }
        final long finalTimestamp = minTimestampSeen;
        return new Predicate<Long>(){

            public boolean apply(Long time) {
                return time < finalTimestamp;
            }
        };
    }

    @Override
    public void close() {
        if (this.overlappingSSTables != null) {
            this.overlappingSSTables.release();
        }
    }
}

