# 1 "compiler/lib/js_lexer.mll"
 

(* Js_of_ocaml compiler
 * Copyright (C) 2013 Hugo Heuzard
 *)

(* Yoann Padioleau
 *
 * Copyright (C) 2010 Facebook
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation, with the
 * special exception on linking described in file license.txt.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the file
 * license.txt for more details.
 *)

open Stdlib
open Js_token

let tok lexbuf = Lexing.lexeme lexbuf

let keyword_table =
  let h = Hashtbl.create 17 in
  List.iter ~f:(fun (s,f) -> Hashtbl.add h s f ) [
    "break",      (fun ii -> T_BREAK ii);
    "case",       (fun ii -> T_CASE ii);
    "catch",      (fun ii -> T_CATCH ii);
    "continue",   (fun ii -> T_CONTINUE ii);
    "debugger",   (fun ii -> T_DEBUGGER ii);
    "default",    (fun ii -> T_DEFAULT ii);
    "delete",     (fun ii -> T_DELETE ii);
    "do",         (fun ii -> T_DO ii);
    "else",       (fun ii -> T_ELSE ii);
    "false",      (fun ii -> T_FALSE ii);
    "finally",    (fun ii -> T_FINALLY ii);
    "for",        (fun ii -> T_FOR ii);
    "function",   (fun ii -> T_FUNCTION ii);
    "if",         (fun ii -> T_IF ii);
    "in",         (fun ii -> T_IN ii);
    "instanceof", (fun ii -> T_INSTANCEOF ii);
    "new",        (fun ii -> T_NEW ii);
    "null",       (fun ii -> T_NULL ii);
    "return",     (fun ii -> T_RETURN ii);
    "switch",     (fun ii -> T_SWITCH ii);
    "this",       (fun ii -> T_THIS ii);
    "throw",      (fun ii -> T_THROW ii);
    "true",       (fun ii -> T_TRUE ii);
    "try",        (fun ii -> T_TRY ii);
    "typeof",     (fun ii -> T_TYPEOF ii);
    "var",        (fun ii -> T_VAR ii);
    "void",       (fun ii -> T_VOID ii);
    "while",      (fun ii -> T_WHILE ii);
    "while",      (fun ii -> T_WHILE ii);
    "with",       (fun ii -> T_WITH ii);
  ];
  h

(* Update the current location with file name and line number. *)

let update_loc lexbuf ?file ~line ~absolute chars =
  let pos = lexbuf.Lexing.lex_curr_p in
  let new_file = match file with
                 | None -> pos.pos_fname
                 | Some s -> s
  in
  lexbuf.Lexing.lex_curr_p <- { pos with
    pos_fname = new_file;
    pos_lnum = if absolute then line else pos.pos_lnum + line;
    pos_bol = pos.pos_cnum - chars;
                              }

let tokinfo lexbuf = Parse_info.t_of_lexbuf lexbuf

let with_pos lexbuf f =
  let p = lexbuf.Lexing.lex_start_p in
  let pos = lexbuf.Lexing.lex_start_pos in
  let r = f () in
  lexbuf.Lexing.lex_start_p <- p;
  lexbuf.Lexing.lex_start_pos <- pos;
  r

# 89 "compiler/lib/js_lexer.ml"
let __ocaml_lex_tables = {
  Lexing.lex_base =
   "\000\000\194\255\195\255\198\255\081\000\103\000\160\000\205\255\
    \003\000\031\000\035\000\083\000\100\000\078\000\081\000\084\000\
    \085\000\127\000\109\000\240\255\241\255\242\255\243\255\237\000\
    \245\255\246\255\247\255\248\255\249\255\250\255\251\255\001\000\
    \003\000\120\000\196\255\155\000\255\255\153\000\030\001\040\001\
    \161\000\166\000\174\000\159\000\192\000\254\255\004\000\050\001\
    \102\000\225\255\072\001\082\001\092\001\218\255\239\255\217\255\
    \238\255\103\000\237\255\112\000\236\255\116\000\235\255\229\255\
    \124\000\234\255\117\000\227\255\226\255\222\255\231\255\221\255\
    \230\255\220\255\219\255\216\255\127\001\139\001\149\001\204\001\
    \209\000\252\255\253\255\003\002\026\002\255\255\064\002\254\255\
    \087\002\125\002\148\002\228\001\250\255\251\255\253\255\005\000\
    \006\000\255\255\254\255\007\000\133\000\255\255\065\001\250\255\
    \251\255\252\255\253\255\254\255\028\000\255\255\210\000\251\255\
    \252\255\253\255\041\000\255\255\254\255\186\002\221\001\251\255\
    \021\002\254\255\008\000\142\000\255\255";
  Lexing.lex_backtrk =
   "\255\255\255\255\255\255\255\255\056\000\056\000\052\000\255\255\
    \048\000\045\000\044\000\043\000\042\000\041\000\040\000\049\000\
    \051\000\046\000\047\000\255\255\255\255\255\255\255\255\011\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\004\000\
    \003\000\058\000\255\255\002\000\255\255\002\000\002\000\002\000\
    \002\000\002\000\002\000\255\255\255\255\255\255\001\000\056\000\
    \255\255\255\255\255\255\055\000\255\255\255\255\255\255\255\255\
    \255\255\022\000\255\255\023\000\255\255\027\000\255\255\255\255\
    \032\000\255\255\031\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\056\000\054\000\255\255\053\000\
    \255\255\255\255\255\255\002\000\002\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\002\000\
    \003\000\255\255\255\255\001\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\003\000\255\255\255\255\255\255\
    \255\255\255\255\002\000\255\255\255\255\000\000\255\255\255\255\
    \002\000\255\255\001\000\003\000\255\255";
  Lexing.lex_default =
   "\001\000\000\000\000\000\000\000\255\255\255\255\255\255\000\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\000\000\000\000\000\000\000\000\255\255\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\255\255\
    \255\255\255\255\000\000\037\000\000\000\037\000\037\000\037\000\
    \040\000\037\000\037\000\043\000\255\255\000\000\255\255\255\255\
    \255\255\000\000\255\255\255\255\255\255\000\000\000\000\000\000\
    \000\000\255\255\000\000\255\255\000\000\255\255\000\000\000\000\
    \255\255\000\000\255\255\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\255\255\255\255\255\255\255\255\
    \082\000\000\000\000\000\255\255\255\255\000\000\255\255\000\000\
    \255\255\255\255\255\255\093\000\000\000\000\000\000\000\255\255\
    \255\255\000\000\000\000\255\255\255\255\000\000\105\000\000\000\
    \000\000\000\000\000\000\000\000\109\000\000\000\113\000\000\000\
    \000\000\000\000\116\000\000\000\000\000\255\255\120\000\000\000\
    \120\000\000\000\255\255\255\255\000\000";
  Lexing.lex_trans =
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\032\000\030\000\030\000\032\000\031\000\045\000\094\000\
    \098\000\098\000\121\000\099\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \032\000\015\000\003\000\032\000\006\000\009\000\018\000\003\000\
    \027\000\026\000\010\000\012\000\021\000\011\000\023\000\033\000\
    \005\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\020\000\022\000\014\000\016\000\013\000\019\000\
    \075\000\006\000\006\000\006\000\006\000\006\000\006\000\006\000\
    \006\000\006\000\006\000\006\000\006\000\006\000\006\000\006\000\
    \006\000\006\000\006\000\006\000\006\000\006\000\006\000\006\000\
    \006\000\006\000\006\000\025\000\074\000\024\000\008\000\006\000\
    \073\000\006\000\006\000\006\000\006\000\006\000\006\000\006\000\
    \006\000\006\000\006\000\006\000\006\000\006\000\006\000\006\000\
    \006\000\006\000\006\000\006\000\006\000\006\000\006\000\006\000\
    \006\000\006\000\006\000\029\000\017\000\028\000\007\000\076\000\
    \072\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\065\000\064\000\061\000\062\000\070\000\
    \071\000\059\000\057\000\054\000\049\000\076\000\050\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\004\000\
    \004\000\069\000\036\000\255\255\058\000\255\255\255\255\035\000\
    \255\255\255\255\053\000\255\255\050\000\060\000\043\000\041\000\
    \255\255\063\000\068\000\255\255\101\000\034\000\050\000\042\000\
    \045\000\067\000\066\000\046\000\055\000\124\000\038\000\078\000\
    \000\000\044\000\000\000\042\000\006\000\000\000\041\000\000\000\
    \040\000\044\000\045\000\000\000\050\000\046\000\042\000\000\000\
    \006\000\006\000\006\000\006\000\006\000\006\000\006\000\006\000\
    \006\000\006\000\000\000\000\000\112\000\000\000\000\000\078\000\
    \044\000\006\000\006\000\006\000\006\000\006\000\006\000\006\000\
    \006\000\006\000\006\000\006\000\006\000\006\000\006\000\006\000\
    \006\000\006\000\006\000\006\000\006\000\006\000\006\000\006\000\
    \006\000\006\000\006\000\056\000\000\000\000\000\000\000\006\000\
    \002\000\006\000\006\000\006\000\006\000\006\000\006\000\006\000\
    \006\000\006\000\006\000\006\000\006\000\006\000\006\000\006\000\
    \006\000\006\000\006\000\006\000\006\000\006\000\006\000\006\000\
    \006\000\006\000\006\000\048\000\255\255\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\038\000\
    \255\255\255\255\000\000\255\255\000\000\085\000\114\000\115\000\
    \000\000\041\000\255\255\000\000\000\000\255\255\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\038\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\083\000\000\000\
    \041\000\084\000\040\000\104\000\000\000\000\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\000\000\000\000\000\000\000\000\
    \107\000\000\000\000\000\052\000\000\000\052\000\000\000\050\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\000\000\050\000\
    \000\000\255\255\000\000\255\255\106\000\108\000\000\000\255\255\
    \000\000\255\255\000\000\000\000\000\000\000\000\255\255\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\255\255\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\076\000\000\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\004\000\004\000\079\000\079\000\079\000\
    \079\000\079\000\079\000\079\000\079\000\079\000\079\000\000\000\
    \050\000\081\000\111\000\000\000\000\000\000\000\079\000\079\000\
    \079\000\079\000\079\000\079\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\121\000\
    \000\000\000\000\122\000\000\000\000\000\000\000\094\000\000\000\
    \050\000\095\000\000\000\000\000\000\000\000\000\079\000\079\000\
    \079\000\079\000\079\000\079\000\079\000\079\000\079\000\079\000\
    \079\000\079\000\079\000\079\000\079\000\079\000\097\000\123\000\
    \000\000\000\000\000\000\097\000\000\000\079\000\079\000\079\000\
    \079\000\079\000\079\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\255\255\255\255\
    \000\000\000\000\255\255\000\000\000\000\000\000\000\000\000\000\
    \255\255\000\000\000\000\000\000\000\000\079\000\079\000\079\000\
    \079\000\079\000\079\000\088\000\088\000\088\000\088\000\088\000\
    \088\000\088\000\088\000\088\000\088\000\000\000\000\000\255\255\
    \096\000\103\000\000\000\000\000\088\000\088\000\088\000\088\000\
    \088\000\088\000\086\000\086\000\086\000\086\000\086\000\086\000\
    \086\000\086\000\086\000\086\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\086\000\086\000\086\000\086\000\086\000\
    \086\000\000\000\000\000\000\000\088\000\088\000\088\000\088\000\
    \088\000\088\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \087\000\087\000\087\000\087\000\087\000\087\000\087\000\087\000\
    \087\000\087\000\000\000\086\000\086\000\086\000\086\000\086\000\
    \086\000\087\000\087\000\087\000\087\000\087\000\087\000\089\000\
    \089\000\089\000\089\000\089\000\089\000\089\000\089\000\089\000\
    \089\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \089\000\089\000\089\000\089\000\089\000\089\000\000\000\000\000\
    \000\000\087\000\087\000\087\000\087\000\087\000\087\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\090\000\090\000\090\000\
    \090\000\090\000\090\000\090\000\090\000\090\000\090\000\000\000\
    \089\000\089\000\089\000\089\000\089\000\089\000\090\000\090\000\
    \090\000\090\000\090\000\090\000\087\000\087\000\087\000\087\000\
    \087\000\087\000\087\000\087\000\087\000\087\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\087\000\087\000\087\000\
    \087\000\087\000\087\000\000\000\000\000\119\000\090\000\090\000\
    \090\000\090\000\090\000\090\000\092\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\087\000\087\000\087\000\
    \087\000\087\000\087\000\117\000\117\000\117\000\117\000\117\000\
    \117\000\117\000\117\000\117\000\117\000\117\000\117\000\117\000\
    \117\000\117\000\117\000\117\000\117\000\117\000\117\000\117\000\
    \117\000\117\000\117\000\117\000\117\000\255\255\000\000\000\000\
    \000\000\000\000\000\000\117\000\117\000\117\000\117\000\117\000\
    \117\000\117\000\117\000\117\000\117\000\117\000\117\000\117\000\
    \117\000\117\000\117\000\117\000\117\000\117\000\117\000\117\000\
    \117\000\117\000\117\000\117\000\117\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000";
  Lexing.lex_check =
   "\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\000\000\000\000\031\000\032\000\000\000\046\000\095\000\
    \096\000\099\000\122\000\096\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \000\000\000\000\000\000\032\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \008\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\009\000\000\000\000\000\000\000\
    \010\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\004\000\
    \011\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\013\000\013\000\014\000\014\000\012\000\
    \011\000\015\000\016\000\018\000\048\000\005\000\004\000\005\000\
    \005\000\005\000\005\000\005\000\005\000\005\000\005\000\005\000\
    \005\000\012\000\033\000\037\000\057\000\035\000\037\000\033\000\
    \035\000\043\000\018\000\040\000\005\000\059\000\040\000\041\000\
    \041\000\061\000\066\000\041\000\100\000\033\000\004\000\042\000\
    \042\000\064\000\064\000\042\000\017\000\123\000\035\000\005\000\
    \255\255\043\000\255\255\040\000\006\000\255\255\041\000\255\255\
    \041\000\044\000\044\000\255\255\005\000\044\000\042\000\255\255\
    \006\000\006\000\006\000\006\000\006\000\006\000\006\000\006\000\
    \006\000\006\000\255\255\255\255\110\000\255\255\255\255\005\000\
    \044\000\006\000\006\000\006\000\006\000\006\000\006\000\006\000\
    \006\000\006\000\006\000\006\000\006\000\006\000\006\000\006\000\
    \006\000\006\000\006\000\006\000\006\000\006\000\006\000\006\000\
    \006\000\006\000\006\000\017\000\255\255\255\255\255\255\006\000\
    \000\000\006\000\006\000\006\000\006\000\006\000\006\000\006\000\
    \006\000\006\000\006\000\006\000\006\000\006\000\006\000\006\000\
    \006\000\006\000\006\000\006\000\006\000\006\000\006\000\006\000\
    \006\000\006\000\006\000\023\000\108\000\023\000\023\000\023\000\
    \023\000\023\000\023\000\023\000\023\000\023\000\023\000\038\000\
    \038\000\114\000\255\255\038\000\255\255\080\000\110\000\110\000\
    \255\255\039\000\039\000\255\255\255\255\039\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\038\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\080\000\255\255\
    \039\000\080\000\039\000\102\000\255\255\255\255\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\255\255\255\255\255\255\255\255\
    \102\000\255\255\255\255\050\000\255\255\050\000\255\255\047\000\
    \050\000\050\000\050\000\050\000\050\000\050\000\050\000\050\000\
    \050\000\050\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\052\000\052\000\052\000\052\000\
    \052\000\052\000\052\000\052\000\052\000\052\000\255\255\047\000\
    \255\255\037\000\255\255\035\000\102\000\102\000\255\255\043\000\
    \255\255\040\000\255\255\255\255\255\255\255\255\041\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\042\000\076\000\
    \076\000\076\000\076\000\076\000\076\000\076\000\076\000\076\000\
    \076\000\077\000\255\255\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\255\255\
    \077\000\080\000\110\000\255\255\255\255\255\255\078\000\078\000\
    \078\000\078\000\078\000\078\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\118\000\
    \255\255\255\255\118\000\255\255\255\255\255\255\091\000\255\255\
    \077\000\091\000\255\255\255\255\255\255\255\255\078\000\078\000\
    \078\000\078\000\078\000\078\000\079\000\079\000\079\000\079\000\
    \079\000\079\000\079\000\079\000\079\000\079\000\091\000\118\000\
    \255\255\255\255\255\255\091\000\255\255\079\000\079\000\079\000\
    \079\000\079\000\079\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\038\000\120\000\
    \255\255\255\255\120\000\255\255\255\255\255\255\255\255\255\255\
    \039\000\255\255\255\255\255\255\255\255\079\000\079\000\079\000\
    \079\000\079\000\079\000\083\000\083\000\083\000\083\000\083\000\
    \083\000\083\000\083\000\083\000\083\000\255\255\255\255\120\000\
    \091\000\102\000\255\255\255\255\083\000\083\000\083\000\083\000\
    \083\000\083\000\084\000\084\000\084\000\084\000\084\000\084\000\
    \084\000\084\000\084\000\084\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\084\000\084\000\084\000\084\000\084\000\
    \084\000\255\255\255\255\255\255\083\000\083\000\083\000\083\000\
    \083\000\083\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \086\000\086\000\086\000\086\000\086\000\086\000\086\000\086\000\
    \086\000\086\000\255\255\084\000\084\000\084\000\084\000\084\000\
    \084\000\086\000\086\000\086\000\086\000\086\000\086\000\088\000\
    \088\000\088\000\088\000\088\000\088\000\088\000\088\000\088\000\
    \088\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \088\000\088\000\088\000\088\000\088\000\088\000\255\255\255\255\
    \255\255\086\000\086\000\086\000\086\000\086\000\086\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\089\000\089\000\089\000\
    \089\000\089\000\089\000\089\000\089\000\089\000\089\000\255\255\
    \088\000\088\000\088\000\088\000\088\000\088\000\089\000\089\000\
    \089\000\089\000\089\000\089\000\090\000\090\000\090\000\090\000\
    \090\000\090\000\090\000\090\000\090\000\090\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\090\000\090\000\090\000\
    \090\000\090\000\090\000\255\255\255\255\118\000\089\000\089\000\
    \089\000\089\000\089\000\089\000\091\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\090\000\090\000\090\000\
    \090\000\090\000\090\000\117\000\117\000\117\000\117\000\117\000\
    \117\000\117\000\117\000\117\000\117\000\117\000\117\000\117\000\
    \117\000\117\000\117\000\117\000\117\000\117\000\117\000\117\000\
    \117\000\117\000\117\000\117\000\117\000\120\000\255\255\255\255\
    \255\255\255\255\255\255\117\000\117\000\117\000\117\000\117\000\
    \117\000\117\000\117\000\117\000\117\000\117\000\117\000\117\000\
    \117\000\117\000\117\000\117\000\117\000\117\000\117\000\117\000\
    \117\000\117\000\117\000\117\000\117\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255";
  Lexing.lex_base_code =
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\010\000\
    \000\000\002\000\005\000\005\000\008\000\023\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000";
  Lexing.lex_backtrk_code =
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\023\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000";
  Lexing.lex_default_code =
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \012\000\000\000\000\000\012\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000";
  Lexing.lex_trans_code =
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\001\000\000\000\000\000\000\000\000\000\015\000\000\000\
    \000\000\015\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \001\000\000\000\015\000\001\000\018\000\015\000\000\000\015\000\
    \015\000\000\000\000\000\000\000\007\000\000\000\000\000\000\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000";
  Lexing.lex_check_code =
   "\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\038\000\040\000\255\255\255\255\255\255\042\000\043\000\
    \255\255\044\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \038\000\255\255\040\000\035\000\041\000\042\000\255\255\043\000\
    \044\000\255\255\255\255\255\255\039\000\255\255\255\255\255\255\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \040\000\255\255\255\255\255\255\255\255\043\000\255\255\255\255\
    \255\255\255\255\255\255";
  Lexing.lex_code =
   "\255\005\255\255\006\255\255\008\255\007\255\255\008\255\255\009\
    \255\255\007\255\008\255\255\004\009\000\005\001\006\002\007\003\
    \008\255";
}

let rec main lexbuf =
  lexbuf.Lexing.lex_mem <- Array.make 10 (-1); __ocaml_lex_main_rec lexbuf 0
and __ocaml_lex_main_rec lexbuf __ocaml_lex_state =
  match Lexing.new_engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 100 "compiler/lib/js_lexer.mll"
         (
      with_pos lexbuf (fun () ->
      let info = tokinfo lexbuf in
      let buf = Buffer.create 127 in
      Buffer.add_string buf (tok lexbuf);
      st_comment buf lexbuf;
      let content = Buffer.contents buf in
      TComment(content, info))
    )
# 526 "compiler/lib/js_lexer.ml"

  | 1 ->
let
# 110 "compiler/lib/js_lexer.mll"
                    line
# 532 "compiler/lib/js_lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(0) lexbuf.Lexing.lex_mem.(1)
and
# 111 "compiler/lib/js_lexer.mll"
                           file
# 537 "compiler/lib/js_lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(2) lexbuf.Lexing.lex_mem.(3)
and
# 112 "compiler/lib/js_lexer.mll"
         raw
# 542 "compiler/lib/js_lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_mem.(4) in
# 112 "compiler/lib/js_lexer.mll"
                     (
      let info = tokinfo lexbuf in
      let line = int_of_string line in
      update_loc lexbuf ~file ~line ~absolute:true 0;
      TCommentLineDirective (raw, info)
    )
# 551 "compiler/lib/js_lexer.ml"

  | 2 ->
let
# 119 "compiler/lib/js_lexer.mll"
                              cmt
# 557 "compiler/lib/js_lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 119 "compiler/lib/js_lexer.mll"
                                  ( TComment(cmt, tokinfo lexbuf) )
# 561 "compiler/lib/js_lexer.ml"

  | 3 ->
# 121 "compiler/lib/js_lexer.mll"
                 (
      main lexbuf
    )
# 568 "compiler/lib/js_lexer.ml"

  | 4 ->
# 124 "compiler/lib/js_lexer.mll"
            (
      update_loc lexbuf ~line:1 ~absolute:false 0;
      main lexbuf
    )
# 576 "compiler/lib/js_lexer.ml"

  | 5 ->
# 133 "compiler/lib/js_lexer.mll"
        ( T_LCURLY (tokinfo lexbuf); )
# 581 "compiler/lib/js_lexer.ml"

  | 6 ->
# 134 "compiler/lib/js_lexer.mll"
        ( T_RCURLY (tokinfo lexbuf); )
# 586 "compiler/lib/js_lexer.ml"

  | 7 ->
# 136 "compiler/lib/js_lexer.mll"
        ( T_LPAREN (tokinfo lexbuf); )
# 591 "compiler/lib/js_lexer.ml"

  | 8 ->
# 137 "compiler/lib/js_lexer.mll"
        ( T_RPAREN (tokinfo lexbuf); )
# 596 "compiler/lib/js_lexer.ml"

  | 9 ->
# 139 "compiler/lib/js_lexer.mll"
        ( T_LBRACKET (tokinfo lexbuf); )
# 601 "compiler/lib/js_lexer.ml"

  | 10 ->
# 140 "compiler/lib/js_lexer.mll"
        ( T_RBRACKET (tokinfo lexbuf); )
# 606 "compiler/lib/js_lexer.ml"

  | 11 ->
# 141 "compiler/lib/js_lexer.mll"
        ( T_PERIOD (tokinfo lexbuf); )
# 611 "compiler/lib/js_lexer.ml"

  | 12 ->
# 142 "compiler/lib/js_lexer.mll"
        ( T_SEMICOLON (tokinfo lexbuf); )
# 616 "compiler/lib/js_lexer.ml"

  | 13 ->
# 143 "compiler/lib/js_lexer.mll"
        ( T_COMMA (tokinfo lexbuf); )
# 621 "compiler/lib/js_lexer.ml"

  | 14 ->
# 144 "compiler/lib/js_lexer.mll"
        ( T_COLON (tokinfo lexbuf); )
# 626 "compiler/lib/js_lexer.ml"

  | 15 ->
# 145 "compiler/lib/js_lexer.mll"
        ( T_PLING (tokinfo lexbuf); )
# 631 "compiler/lib/js_lexer.ml"

  | 16 ->
# 146 "compiler/lib/js_lexer.mll"
         ( T_AND (tokinfo lexbuf); )
# 636 "compiler/lib/js_lexer.ml"

  | 17 ->
# 147 "compiler/lib/js_lexer.mll"
         ( T_OR (tokinfo lexbuf); )
# 641 "compiler/lib/js_lexer.ml"

  | 18 ->
# 148 "compiler/lib/js_lexer.mll"
          ( T_STRICT_EQUAL (tokinfo lexbuf); )
# 646 "compiler/lib/js_lexer.ml"

  | 19 ->
# 149 "compiler/lib/js_lexer.mll"
          ( T_STRICT_NOT_EQUAL (tokinfo lexbuf); )
# 651 "compiler/lib/js_lexer.ml"

  | 20 ->
# 150 "compiler/lib/js_lexer.mll"
         ( T_LESS_THAN_EQUAL (tokinfo lexbuf); )
# 656 "compiler/lib/js_lexer.ml"

  | 21 ->
# 151 "compiler/lib/js_lexer.mll"
         ( T_GREATER_THAN_EQUAL (tokinfo lexbuf); )
# 661 "compiler/lib/js_lexer.ml"

  | 22 ->
# 152 "compiler/lib/js_lexer.mll"
         ( T_EQUAL (tokinfo lexbuf); )
# 666 "compiler/lib/js_lexer.ml"

  | 23 ->
# 153 "compiler/lib/js_lexer.mll"
         ( T_NOT_EQUAL (tokinfo lexbuf); )
# 671 "compiler/lib/js_lexer.ml"

  | 24 ->
# 154 "compiler/lib/js_lexer.mll"
         ( T_INCR (tokinfo lexbuf); )
# 676 "compiler/lib/js_lexer.ml"

  | 25 ->
# 155 "compiler/lib/js_lexer.mll"
         ( T_DECR (tokinfo lexbuf); )
# 681 "compiler/lib/js_lexer.ml"

  | 26 ->
# 156 "compiler/lib/js_lexer.mll"
          ( T_LSHIFT_ASSIGN (tokinfo lexbuf); )
# 686 "compiler/lib/js_lexer.ml"

  | 27 ->
# 157 "compiler/lib/js_lexer.mll"
         ( T_LSHIFT (tokinfo lexbuf); )
# 691 "compiler/lib/js_lexer.ml"

  | 28 ->
# 158 "compiler/lib/js_lexer.mll"
          ( T_RSHIFT_ASSIGN (tokinfo lexbuf); )
# 696 "compiler/lib/js_lexer.ml"

  | 29 ->
# 159 "compiler/lib/js_lexer.mll"
           ( T_RSHIFT3_ASSIGN (tokinfo lexbuf); )
# 701 "compiler/lib/js_lexer.ml"

  | 30 ->
# 160 "compiler/lib/js_lexer.mll"
          ( T_SPREAD (tokinfo lexbuf); )
# 706 "compiler/lib/js_lexer.ml"

  | 31 ->
# 161 "compiler/lib/js_lexer.mll"
          ( T_RSHIFT3 (tokinfo lexbuf); )
# 711 "compiler/lib/js_lexer.ml"

  | 32 ->
# 162 "compiler/lib/js_lexer.mll"
         ( T_RSHIFT (tokinfo lexbuf); )
# 716 "compiler/lib/js_lexer.ml"

  | 33 ->
# 163 "compiler/lib/js_lexer.mll"
         ( T_PLUS_ASSIGN (tokinfo lexbuf); )
# 721 "compiler/lib/js_lexer.ml"

  | 34 ->
# 164 "compiler/lib/js_lexer.mll"
         ( T_MINUS_ASSIGN (tokinfo lexbuf); )
# 726 "compiler/lib/js_lexer.ml"

  | 35 ->
# 166 "compiler/lib/js_lexer.mll"
         ( T_MULT_ASSIGN (tokinfo lexbuf); )
# 731 "compiler/lib/js_lexer.ml"

  | 36 ->
# 167 "compiler/lib/js_lexer.mll"
         ( T_MOD_ASSIGN (tokinfo lexbuf); )
# 736 "compiler/lib/js_lexer.ml"

  | 37 ->
# 168 "compiler/lib/js_lexer.mll"
         ( T_BIT_AND_ASSIGN (tokinfo lexbuf); )
# 741 "compiler/lib/js_lexer.ml"

  | 38 ->
# 169 "compiler/lib/js_lexer.mll"
         ( T_BIT_OR_ASSIGN (tokinfo lexbuf); )
# 746 "compiler/lib/js_lexer.ml"

  | 39 ->
# 170 "compiler/lib/js_lexer.mll"
         ( T_BIT_XOR_ASSIGN (tokinfo lexbuf); )
# 751 "compiler/lib/js_lexer.ml"

  | 40 ->
# 171 "compiler/lib/js_lexer.mll"
        ( T_LESS_THAN (tokinfo lexbuf); )
# 756 "compiler/lib/js_lexer.ml"

  | 41 ->
# 172 "compiler/lib/js_lexer.mll"
        ( T_GREATER_THAN (tokinfo lexbuf); )
# 761 "compiler/lib/js_lexer.ml"

  | 42 ->
# 173 "compiler/lib/js_lexer.mll"
        ( T_PLUS (tokinfo lexbuf); )
# 766 "compiler/lib/js_lexer.ml"

  | 43 ->
# 174 "compiler/lib/js_lexer.mll"
        ( T_MINUS (tokinfo lexbuf); )
# 771 "compiler/lib/js_lexer.ml"

  | 44 ->
# 175 "compiler/lib/js_lexer.mll"
        ( T_MULT (tokinfo lexbuf); )
# 776 "compiler/lib/js_lexer.ml"

  | 45 ->
# 177 "compiler/lib/js_lexer.mll"
        ( T_MOD (tokinfo lexbuf); )
# 781 "compiler/lib/js_lexer.ml"

  | 46 ->
# 178 "compiler/lib/js_lexer.mll"
        ( T_BIT_OR (tokinfo lexbuf); )
# 786 "compiler/lib/js_lexer.ml"

  | 47 ->
# 179 "compiler/lib/js_lexer.mll"
        ( T_BIT_AND (tokinfo lexbuf); )
# 791 "compiler/lib/js_lexer.ml"

  | 48 ->
# 180 "compiler/lib/js_lexer.mll"
        ( T_BIT_XOR (tokinfo lexbuf); )
# 796 "compiler/lib/js_lexer.ml"

  | 49 ->
# 181 "compiler/lib/js_lexer.mll"
        ( T_NOT (tokinfo lexbuf); )
# 801 "compiler/lib/js_lexer.ml"

  | 50 ->
# 182 "compiler/lib/js_lexer.mll"
        ( T_BIT_NOT (tokinfo lexbuf); )
# 806 "compiler/lib/js_lexer.ml"

  | 51 ->
# 183 "compiler/lib/js_lexer.mll"
        ( T_ASSIGN (tokinfo lexbuf); )
# 811 "compiler/lib/js_lexer.ml"

  | 52 ->
# 188 "compiler/lib/js_lexer.mll"
                                                         (
      let s = tok lexbuf in
      let info = tokinfo lexbuf in
      try
        let f = Hashtbl.find keyword_table s in
        f info
      with
        | Not_found -> T_IDENTIFIER (s, info)
    )
# 824 "compiler/lib/js_lexer.ml"

  | 53 ->
# 202 "compiler/lib/js_lexer.mll"
                       (
      let s = tok lexbuf in
      let info = tokinfo lexbuf in
      T_NUMBER (s, info)
    )
# 833 "compiler/lib/js_lexer.ml"

  | 54 ->
# 207 "compiler/lib/js_lexer.mll"
                  (
      let s = tok lexbuf in
      let info = tokinfo lexbuf in
      T_NUMBER (s, info)
    )
# 842 "compiler/lib/js_lexer.ml"

  | 55 ->
# 213 "compiler/lib/js_lexer.mll"
                                                                    (
      let s = tok lexbuf in
      let info = tokinfo lexbuf in
      T_NUMBER (s, info)
    )
# 851 "compiler/lib/js_lexer.ml"

  | 56 ->
# 219 "compiler/lib/js_lexer.mll"
                            (
      let s = tok lexbuf in
      let info = tokinfo lexbuf in
      T_NUMBER (s, info)
    )
# 860 "compiler/lib/js_lexer.ml"

  | 57 ->
let
# 228 "compiler/lib/js_lexer.mll"
                 quote
# 866 "compiler/lib/js_lexer.ml"
= Lexing.sub_lexeme_char lexbuf lexbuf.Lexing.lex_start_pos in
# 228 "compiler/lib/js_lexer.mll"
                       (
      with_pos lexbuf (fun () ->
      let from = lexbuf.Lexing.lex_start_p.pos_cnum in
      let info = tokinfo lexbuf in
      let buf = Buffer.create 127 in
      string_quote quote buf lexbuf;
      let s = Buffer.contents buf in
      (* s does not contain the enclosing "'" but the info does *)
      let to_ = lexbuf.Lexing.lex_curr_p.pos_cnum in
      T_STRING (s, info, to_ - 1 - from))
    )
# 880 "compiler/lib/js_lexer.ml"

  | 58 ->
# 239 "compiler/lib/js_lexer.mll"
        ( T_DIV (tokinfo lexbuf) )
# 885 "compiler/lib/js_lexer.ml"

  | 59 ->
# 240 "compiler/lib/js_lexer.mll"
         ( T_DIV_ASSIGN (tokinfo lexbuf) )
# 890 "compiler/lib/js_lexer.ml"

  | 60 ->
# 245 "compiler/lib/js_lexer.mll"
        ( EOF (tokinfo lexbuf) )
# 895 "compiler/lib/js_lexer.ml"

  | 61 ->
# 247 "compiler/lib/js_lexer.mll"
      (
      TUnknown (tok lexbuf, tokinfo lexbuf)
    )
# 902 "compiler/lib/js_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_main_rec lexbuf __ocaml_lex_state

and string_escape quote buf lexbuf =
   __ocaml_lex_string_escape_rec quote buf lexbuf 80
and __ocaml_lex_string_escape_rec quote buf lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 253 "compiler/lib/js_lexer.mll"
        ( Buffer.add_string buf "\\\\" )
# 914 "compiler/lib/js_lexer.ml"

  | 1 ->
# 255 "compiler/lib/js_lexer.mll"
                            (
      Buffer.add_char buf '\\';
      Buffer.add_string buf (tok lexbuf) )
# 921 "compiler/lib/js_lexer.ml"

  | 2 ->
let
# 258 "compiler/lib/js_lexer.mll"
          c
# 927 "compiler/lib/js_lexer.ml"
= Lexing.sub_lexeme_char lexbuf lexbuf.Lexing.lex_start_pos in
# 259 "compiler/lib/js_lexer.mll"
    ( if Char.(c <> '\'') && Char.(c <> '\"') then Buffer.add_char buf '\\';
      Buffer.add_char buf c )
# 932 "compiler/lib/js_lexer.ml"

  | 3 ->
# 261 "compiler/lib/js_lexer.mll"
        ( Format.eprintf  "LEXER: WEIRD end of file in string_escape@."; ())
# 937 "compiler/lib/js_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_string_escape_rec quote buf lexbuf __ocaml_lex_state

and string_quote q buf lexbuf =
   __ocaml_lex_string_quote_rec q buf lexbuf 91
and __ocaml_lex_string_quote_rec q buf lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
let
# 264 "compiler/lib/js_lexer.mll"
                 q'
# 950 "compiler/lib/js_lexer.ml"
= Lexing.sub_lexeme_char lexbuf lexbuf.Lexing.lex_start_pos in
# 264 "compiler/lib/js_lexer.mll"
                    (
    if Char.(q = q')
    then ()
    else (Buffer.add_char buf q'; string_quote q buf lexbuf) )
# 957 "compiler/lib/js_lexer.ml"

  | 1 ->
# 268 "compiler/lib/js_lexer.mll"
                 (
    update_loc lexbuf ~line:1 ~absolute:false 0;
    string_quote q buf lexbuf )
# 964 "compiler/lib/js_lexer.ml"

  | 2 ->
# 271 "compiler/lib/js_lexer.mll"
            (
    Format.eprintf  "LEXER: WEIRD newline in quoted string@.";
    update_loc lexbuf ~line:1 ~absolute:false 0;
    Buffer.add_string buf (tok lexbuf);
    string_quote q buf lexbuf )
# 973 "compiler/lib/js_lexer.ml"

  | 3 ->
# 276 "compiler/lib/js_lexer.mll"
         (
      string_escape q buf lexbuf;
      string_quote q buf lexbuf
    )
# 981 "compiler/lib/js_lexer.ml"

  | 4 ->
let
# 280 "compiler/lib/js_lexer.mll"
          x
# 987 "compiler/lib/js_lexer.ml"
= Lexing.sub_lexeme_char lexbuf lexbuf.Lexing.lex_start_pos in
# 280 "compiler/lib/js_lexer.mll"
                   ( Buffer.add_char buf x; string_quote q buf lexbuf )
# 991 "compiler/lib/js_lexer.ml"

  | 5 ->
# 281 "compiler/lib/js_lexer.mll"
        ( Format.eprintf  "LEXER: WEIRD end of file in quoted string@."; ())
# 996 "compiler/lib/js_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_string_quote_rec q buf lexbuf __ocaml_lex_state

and main_regexp lexbuf =
   __ocaml_lex_main_regexp_rec lexbuf 100
and __ocaml_lex_main_regexp_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 285 "compiler/lib/js_lexer.mll"
        (
      with_pos lexbuf (fun () ->
      let info = tokinfo lexbuf in
      let buf = Buffer.create 127 in
      Buffer.add_string buf (Lexing.lexeme lexbuf);
      regexp buf lexbuf;
      T_REGEX (Buffer.contents buf, info)) )
# 1014 "compiler/lib/js_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_main_regexp_rec lexbuf __ocaml_lex_state

and regexp buf lexbuf =
   __ocaml_lex_regexp_rec buf lexbuf 102
and __ocaml_lex_regexp_rec buf lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
let
# 294 "compiler/lib/js_lexer.mll"
               x
# 1027 "compiler/lib/js_lexer.ml"
= Lexing.sub_lexeme_char lexbuf (lexbuf.Lexing.lex_start_pos + 1) in
# 294 "compiler/lib/js_lexer.mll"
                  ( Buffer.add_char buf '\\';
                    Buffer.add_char buf x;
                    regexp buf lexbuf )
# 1033 "compiler/lib/js_lexer.ml"

  | 1 ->
# 297 "compiler/lib/js_lexer.mll"
        ( Buffer.add_char buf '/'; regexp_maybe_ident buf lexbuf )
# 1038 "compiler/lib/js_lexer.ml"

  | 2 ->
# 298 "compiler/lib/js_lexer.mll"
        ( Buffer.add_char buf '['; regexp_class buf lexbuf )
# 1043 "compiler/lib/js_lexer.ml"

  | 3 ->
let
# 299 "compiler/lib/js_lexer.mll"
                 x
# 1049 "compiler/lib/js_lexer.ml"
= Lexing.sub_lexeme_char lexbuf lexbuf.Lexing.lex_start_pos in
# 299 "compiler/lib/js_lexer.mll"
                          ( Buffer.add_char buf x; regexp buf lexbuf )
# 1053 "compiler/lib/js_lexer.ml"

  | 4 ->
# 300 "compiler/lib/js_lexer.mll"
         ( Format.eprintf "LEXER: WEIRD newline in regexp@.";
           update_loc lexbuf ~line:1 ~absolute:false 0;
           ())
# 1060 "compiler/lib/js_lexer.ml"

  | 5 ->
# 303 "compiler/lib/js_lexer.mll"
        ( Format.eprintf "LEXER: WEIRD end of file in regexp@."; ())
# 1065 "compiler/lib/js_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_regexp_rec buf lexbuf __ocaml_lex_state

and regexp_class buf lexbuf =
   __ocaml_lex_regexp_class_rec buf lexbuf 110
and __ocaml_lex_regexp_class_rec buf lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 306 "compiler/lib/js_lexer.mll"
        ( Buffer.add_char buf ']';
             regexp buf lexbuf )
# 1078 "compiler/lib/js_lexer.ml"

  | 1 ->
let
# 308 "compiler/lib/js_lexer.mll"
               x
# 1084 "compiler/lib/js_lexer.ml"
= Lexing.sub_lexeme_char lexbuf (lexbuf.Lexing.lex_start_pos + 1) in
# 308 "compiler/lib/js_lexer.mll"
                  ( Buffer.add_char buf '\\';
                    Buffer.add_char buf x;
                    regexp_class buf lexbuf )
# 1090 "compiler/lib/js_lexer.ml"

  | 2 ->
let
# 311 "compiler/lib/js_lexer.mll"
                 x
# 1096 "compiler/lib/js_lexer.ml"
= Lexing.sub_lexeme_char lexbuf lexbuf.Lexing.lex_start_pos in
# 311 "compiler/lib/js_lexer.mll"
                    ( Buffer.add_char buf x; regexp_class buf lexbuf )
# 1100 "compiler/lib/js_lexer.ml"

  | 3 ->
# 312 "compiler/lib/js_lexer.mll"
         ( Format.eprintf "LEXER: WEIRD newline in regexp_class@.";
           update_loc lexbuf ~line:1 ~absolute:false 0;
           ())
# 1107 "compiler/lib/js_lexer.ml"

  | 4 ->
# 315 "compiler/lib/js_lexer.mll"
        ( Format.eprintf "LEXER: WEIRD end of file in regexp_class@."; ())
# 1112 "compiler/lib/js_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_regexp_class_rec buf lexbuf __ocaml_lex_state

and regexp_maybe_ident buf lexbuf =
   __ocaml_lex_regexp_maybe_ident_rec buf lexbuf 117
and __ocaml_lex_regexp_maybe_ident_rec buf lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 318 "compiler/lib/js_lexer.mll"
                      ( Buffer.add_string buf (tok lexbuf) )
# 1124 "compiler/lib/js_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_regexp_maybe_ident_rec buf lexbuf __ocaml_lex_state

and st_comment buf lexbuf =
   __ocaml_lex_st_comment_rec buf lexbuf 118
and __ocaml_lex_st_comment_rec buf lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 323 "compiler/lib/js_lexer.mll"
         ( Buffer.add_string buf (tok lexbuf) )
# 1136 "compiler/lib/js_lexer.ml"

  | 1 ->
# 324 "compiler/lib/js_lexer.mll"
            (
      update_loc lexbuf ~line:1 ~absolute:false 0;
      Buffer.add_string buf (tok lexbuf);
      st_comment buf lexbuf )
# 1144 "compiler/lib/js_lexer.ml"

  | 2 ->
# 328 "compiler/lib/js_lexer.mll"
                       ( Buffer.add_string buf (tok lexbuf);st_comment buf lexbuf )
# 1149 "compiler/lib/js_lexer.ml"

  | 3 ->
# 329 "compiler/lib/js_lexer.mll"
            ( Buffer.add_char buf '*';st_comment buf lexbuf )
# 1154 "compiler/lib/js_lexer.ml"

  | 4 ->
# 331 "compiler/lib/js_lexer.mll"
        ( Format.eprintf "LEXER: end of file in comment@."; Buffer.add_string buf "*/")
# 1159 "compiler/lib/js_lexer.ml"

  | 5 ->
# 332 "compiler/lib/js_lexer.mll"
       (
      let s = tok lexbuf in
      Format.eprintf "LEXER: unrecognised symbol in comment: %s@." s;
      Buffer.add_string buf s;
      st_comment buf lexbuf
    )
# 1169 "compiler/lib/js_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_st_comment_rec buf lexbuf __ocaml_lex_state

;;

