# Copyright (C) 2008 LottaNZB Development Team
# 
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; version 3.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA.

"""Contains classes for all messages put out by the Core module."""

from lottanzb.hellalog import LogRecord, _FatalErrorHandler, _ExceptionHandler
from lottanzb.util import _

class ConfigNotLoadableError(LogRecord): # findAndLoadConfig(...)
    PATTERN = r"^Unable to load specified config file: (?P<conf_file>.+)$"
    TEMPLATE = _("Unable to load configuration file: %(conf_file)s")

# Will most probably not be used since we're always launching HellaNZB
# with the -c argument. Just for the sake of completeness. *g*
class ConfigNotFindableError(LogRecord): # findAndLoadConfig(...)
    PATTERN = r"^Could not find configuration file in the following dirs: \[(?P<conf_dirs>.+)\]$"
    TEMPLATE = _("Could not find the configuration file in the following directories: %(conf_dirs)s")

class ConfigNotReadableWarning(LogRecord): # loadConfig(...)
    PATTERN = r"^Unable to read config file: (?P<conf_file>.+)$"
    TEMPLATE = _("Unable to read configuration file: %(conf_file)s")

class UnhandledConfigError(_ExceptionHandler, LogRecord): # loadConfig(...)
    PATTERN = r"^An unexpected error occurred while reading the config file"
    TEMPLATE = _("An unexpected error occurred while reading the configuration file")

class ConfigError(_FatalErrorHandler, LogRecord): # loadConfig(...)
    PATTERN = r"^A problem occurred while reading the config file"

class MainError(_FatalErrorHandler, LogRecord): # main()
    PATTERN = r"^Exiting"

class UnhandledMainError(_ExceptionHandler, LogRecord): # main() and processArgs()
    PATTERN = r"^An unexpected problem occurred, exiting"
    TEMPLATE = _("An unexpected problem occurred")

class InvalidUmaskError(LogRecord): # init()
    PATTERN = r"^Config file option: Hellanzb\.UMASK is not a valid integer$"
    TEMPLATE = _("Hellanzb.UMASK is not a valid integer")

class InstallLibnotifyError(LogRecord): # init()
    PATTERN = r"^Please install notify-python or disable Hellanzb\.LIBNOTIFY_NOTIFY$"
    TEMPLATE = _("Please install notify-python or disable Hellanzb.LIBNOTIFY_NOTIFY")

class LibnotifyInitError(LogRecord): # init()
    PATTERN = r"^Cannot initialize libnotify$"
    TEMPLATE = _("Unable to initialize libnotify")

# prefy.py already takes care of this and defaults GROWL_NOTIFY to False.
class GrowlUndefinedError(LogRecord): # init()
    PATTERN = r"^Required option not defined in config file: Hellanzb\.GROWL_NOTIFY$"
    TEMPLATE = _("Hellanzb.GROWL_NOTIFY is not defined in the configuration file")

class GrowlInfoMissingError(LogRecord): # init()
    PATTERN = r"^Hellanzb.GROWL_NOTIFY enabled\. Required option not defined in config file: Hellanzb\.(?P<option>.+)$"
    TEMPLATE = _("Hellanzb.GROWL_NOTIFY enabled. Required option not defined in configuration file: Hellanzb.%(option)s")

class LaunchInfo(LogRecord):
    PATTERN = r"^hellanzb v(?P<version>0\.\d+) \(config = (?P<options>.+)\)$"
    TEMPLATE = _("HellaNZB %(version)s (configuration: %(options)s)")
    
    def set_data(self, data):
        options = data["options"].split(", ")
        
        data.update({
            "conf_file": options[0],
            "daemonized": "daemonized" in options,
            "c_yenc": "C yenc module" in options,
            "mac_binary": "MacBinary" in options
        })
        
        translations = {
            "daemonized": _("daemonized"),
            "C yenc module": _("C yenc module")
        }
        
        def translate_options(option):
            if option in translations:
                return translations[option]
            else:
                return option
        
        data["options"] = ", ".join(map(translate_options, options))
        
        LogRecord.set_data(self, data)
