# $Id$
# Implementation of XEP-0080 "User Location"

namespace eval geoloc {
    variable node http://jabber.org/protocol/geoloc
    variable substatus
    variable geoloc

    custom::defvar options(auto-subscribe) 0 \
	[::msgcat::mc "Auto-subscribe to other's user location notifications."] \
	-command [namespace current]::register_in_disco \
	-group PEP -type boolean

    pubsub::register_event_notification_handler $node \
	    [namespace current]::process_geoloc_notification
    hook::add user_geoloc_notification_hook \
	    [namespace current]::notify_via_status_message

    hook::add finload_hook \
	    [namespace current]::on_init 60
    hook::add connected_hook \
	    [namespace current]::on_connect_disconnect
    hook::add disconnected_hook \
	    [namespace current]::on_connect_disconnect
    hook::add roster_jid_popup_menu_hook \
	    [namespace current]::add_roster_pep_menu_item
    hook::add roster_user_popup_info_hook \
	    [namespace current]::provide_roster_popup_info
    hook::add userinfo_hook \
	    [namespace current]::provide_userinfo

    disco::register_feature $node

    variable fields [list alt area bearing building country datum \
			  description error floor lat locality lon \
			  postalcode region room speed street text \
			  timestamp uri]

    array set labels [list alt         [::msgcat::mc "Altitude:"] \
			   area        [::msgcat::mc "Area:"] \
			   bearing     [::msgcat::mc "Bearing:"] \
			   building    [::msgcat::mc "Building:"] \
			   country     [::msgcat::mc "Country:"] \
			   datum       [::msgcat::mc "GPS datum:"] \
			   description [::msgcat::mc "Description:"] \
			   error       [::msgcat::mc "Horizontal GPS error:"] \
			   floor       [::msgcat::mc "Floor:"] \
			   lat         [::msgcat::mc "Latitude:"] \
			   locality    [::msgcat::mc "Locality:"] \
			   lon         [::msgcat::mc "Longitude:"] \
			   postalcode  [::msgcat::mc "Postal code:"] \
			   region      [::msgcat::mc "Region:"] \
			   room        [::msgcat::mc "Room:"] \
			   speed       [::msgcat::mc "Speed:"] \
			   street      [::msgcat::mc "Street:"] \
			   text        [::msgcat::mc "Text:"] \
			   timestamp   [::msgcat::mc "Timestamp:"] \
			   uri         [::msgcat::mc "URI:"]]
}

proc geoloc::register_in_disco {args} {
    variable options
    variable node

    if {$options(auto-subscribe)} {
       disco::register_feature $node+notify
    } else {
       disco::unregister_feature $node+notify
    }
}

proc geoloc::add_roster_pep_menu_item {m connid jid} {
    set rjid [roster::find_jid $connid $jid]

    if {$rjid == ""} {
 	set rjid [node_and_server_from_jid $jid]
    }

    set pm [pep::get_roster_menu_pep_submenu $m $connid $rjid]

    set mm [menu $pm.geoloc -tearoff no]
    $pm add cascade -menu $mm \
	    -label [::msgcat::mc "User location"]

    $mm add command \
	    -label [::msgcat::mc "Subscribe"] \
	    -command [list [namespace current]::subscribe $connid $rjid]
    $mm add command \
	    -label [::msgcat::mc "Unsubscribe"] \
	    -command [list [namespace current]::unsubscribe $connid $rjid]

    hook::run roster_pep_user_geoloc_menu_hook $mm $connid $rjid
}

proc geoloc::subscribe {connid jid args} {
    variable node
    variable substatus

    set to [node_and_server_from_jid $jid]
    set cmd [linsert $args 0 [namespace current]::subscribe_result $connid $to]
    pep::subscribe $to $node \
	    -connection $connid \
	    -command $cmd
    set substatus($connid,$to) sent-subscribe
}

proc geoloc::unsubscribe {connid jid args} {
    variable node
    variable substatus

    set to [node_and_server_from_jid $jid]
    set cmd [linsert $args 0 [namespace current]::unsubscribe_result $connid $to]
    pep::unsubscribe $to $node \
	    -connection $connid \
	    -command $cmd
    set substatus($connid,$to) sent-unsubscribe
}

# Err may be one of: OK, ERR and DISCONNECT
proc geoloc::subscribe_result {connid jid res child args} {
    variable substatus

    set cmd ""
    foreach {opt val} $args {
	switch -- $opt {
	    -command {
		set cmd $val
	    }
	    default {
		return -code error "unknown option: $opt"
	    }
	}
    }

    switch -- $res {
	OK {
	    set substatus($connid,$jid) from
	}
	ERR {
	    set substatus($connid,$jid) error
	}
	default {
	    return
	}
    }

    if {$cmd != ""} {
	lappend cmd $jid $res $child
	eval $cmd
    }
}

proc geoloc::unsubscribe_result {connid jid res child args} {
    variable substatus
    variable geoloc
    variable fields

    set cmd ""
    foreach {opt val} $args {
	switch -- $opt {
	    -command {
		set cmd $val
	    }
	    default {
		return -code error "unknown option: $opt"
	    }
	}
    }

    if {[string equal $res OK]} {
	set substatus($connid,$jid) none
	foreach f $fields {
	    catch {unset geoloc($f,$connid,$jid)}
	}
    }

    if {$cmd != ""} {
	lappend cmd $jid $res $child
	eval $cmd
    }
}

proc geoloc::provide_roster_popup_info {var connid user} {
    variable substatus
    variable geoloc

    upvar 0 $var info

    set jid [node_and_server_from_jid $user]

    if {[info exists geoloc(title,$connid,$jid)]} {
	append info [::msgcat::mc "\n\tLocation: %s : %s" \
				  $geoloc(lat,$connid,$jid) \
				  $geoloc(lon,$connid,$jid)]
    } elseif {[info exists substatus($connid,$jid)]} {
	append info [::msgcat::mc "\n\tUser location subscription: %s" \
				  $substatus($connid,$jid)]
    } else {
	return
    }

}

proc geoloc::process_geoloc_notification {connid jid items} {
    variable node
    variable geoloc
    variable fields

    foreach f $fields {
	set $f ""
    }
    set retract false
    set parsed  false

    foreach item $items {
	jlib::wrapper:splitxml $item tag vars isempty chdata children

	switch -- $tag {
	    retract {
		set retract true
	    }
	    default {
		foreach igeoloc $children {
		    jlib::wrapper:splitxml $igeoloc tag1 vars1 isempty1 chdata1 children1

		    if {![string equal $tag1 geoloc]} continue
		    set xmlns [jlib::wrapper:getattr $vars1 xmlns]
		    if {![string equal $xmlns $node]} continue

		    set parsed true

		    foreach i $children1 {
			jlib::wrapper:splitxml $i tag2 vars2 isempty2 chdata2 children2

			if {[lsearch -exact $fields $tag2] >= 0} {
			    set $tag2 $chdata2
			}
		    }
		}
	    }
	}
    }

    if {$parsed} {
	foreach f $fields {
	    set geoloc($f,$connid,$jid) [set $f]
	}
	hook::run user_geoloc_notification_hook $connid $jid $lat $lon
    } elseif {$retract} {
	foreach f $fields {
	    catch {unset geoloc($f,$connid,$jid)}
	}
	hook::run user_geoloc_notification_hook $connid $jid "" ""
    }
}

proc geoloc::notify_via_status_message {connid jid lat lon} {
    set contact [::roster::itemconfig $connid $jid -name]
    if {$contact == ""} {
	set contact $jid
    }

    if {$lat == "" && $lon == ""} {
	set msg [::msgcat::mc "%s's location is unset" $contact]
    } else {
	set msg [::msgcat::mc "%s's location changed to %s : %s" \
			      $contact $lat $lon]
    }

    set_status $msg
}

proc geoloc::publish {connid args} {
    variable node
    variable fields

    foreach f $fields {
	set $f ""
    }
    set callback ""
    foreach {opt val} $args {
	switch -- $opt {
	    -command { set callback $val }
	    default {
		set opt [string trimleft $opt -]
		if {[lsearch -exact $fields $opt] >= 0} {
		    set $opt $val
		}
	    }
	}
    }

    set content {}
    foreach f $fields {
	if {[set $f] != ""} {
	    lappend content [jlib::wrapper:createtag $f -chdata [set $f]]
	}
    }

    set cmd [list pep::publish_item $node geoloc \
		  -connection $connid \
		  -payload [list [jlib::wrapper:createtag geoloc \
				      -vars [list xmlns $node] \
				      -subtags $content]]]

    if {$callback != ""} {
	lappend cmd -command $callback
    }

    eval $cmd
}

proc geoloc::unpublish {connid args} {
    variable node

    set callback ""
    foreach {opt val} $args {
	switch -- $opt {
	    -command { set callback $val }
	}
    }

    set cmd [list pep::delete_item $node geoloc \
		  -notify true \
		  -connection $connid]

    if {$callback != ""} {
	lappend cmd -command $callback
    }

    eval $cmd
}

proc geoloc::on_init {} {
    set m [pep::get_main_menu_pep_submenu]
    set mm [menu $m.geoloc -tearoff $::ifacetk::options(show_tearoffs)]
    $m add cascade -menu $mm \
	   -label [::msgcat::mc "User location"]
    $mm add command -label [::msgcat::mc "Publish user location..."] \
	    -state disabled \
	    -command [namespace current]::show_publish_dialog
    $mm add command -label [::msgcat::mc "Unpublish user location"] \
	    -state disabled \
	    -command [namespace current]::show_unpublish_dialog
    $mm add checkbutton -label [::msgcat::mc "Auto-subscribe to other's user location"] \
	    -variable [namespace current]::options(auto-subscribe)
}

proc geoloc::on_connect_disconnect {args} {
    set mm [pep::get_main_menu_pep_submenu].geoloc
    set idx [expr {$::ifacetk::options(show_tearoffs) ? 1 : 0}]

    switch -- [llength [jlib::connections]] {
	0 {
	    $mm entryconfigure $idx -state disabled
	    $mm entryconfigure [incr idx] \
		-label [::msgcat::mc "Unpublish user location"] \
		-state disabled
	}
	1 {
	    $mm entryconfigure $idx -state normal
	    $mm entryconfigure [incr idx] \
		-label [::msgcat::mc "Unpublish user location"] \
		-state normal
	}
	default {
	    $mm entryconfigure $idx -state normal
	    $mm entryconfigure [incr idx] \
		-label [::msgcat::mc "Unpublish user location..."] \
		-state normal
	}
    }
}

proc geoloc::show_publish_dialog {} {
    variable fields
    variable labels
    variable myjid
    foreach ff $fields {
	variable geoloc$ff
    }

    set w .user_geoloc
    if {[winfo exists $w]} {
	destroy $w
    }

    set connids [jlib::connections]
    if {[llength $connids] == 0} {
	NonmodalMessageDlg [epath] \
		-aspect 50000 \
		-icon error \
		-title [::msgcat::mc "Error"] \
		-message [::msgcat::mc "Publishing is only possible\
					while being online"]
	return
    }

    Dialog $w -title [::msgcat::mc "User location"] \
	    -modal none -separator 1 -anchor e -default 0 -cancel 1 -parent .
    $w add -text [::msgcat::mc "Publish"] \
	   -command [list [namespace current]::do_publish $w]
    $w add -text [::msgcat::mc "Cancel"] -command [list destroy $w]

    set f [$w getframe]

    set connjids [list [::msgcat::mc "All"]]
    foreach connid $connids {
	lappend connjids [jlib::connection_jid $connid]
    }
    set myjid [lindex $connjids 0]

    label $f.ccap -text [::msgcat::mc "Use connection:"]
    ComboBox $f.conn -editable false \
	    -values $connjids \
	    -textvariable [namespace current]::myjid
    if {[llength $connjids] > 1} {
	grid $f.ccap   -row 0 -column 0 -sticky e
	grid $f.conn   -row 0 -column 1 -sticky ew
    }

    set row 1
    foreach ff $fields {
	label $f.l$ff -text $labels($ff)
	entry $f.$ff -textvariable [namespace current]::geoloc$ff
	grid $f.l$ff  -row $row -column 0 -sticky e
	grid $f.$ff   -row $row -column 1 -sticky ew
	incr row
    }

    grid columnconfigure $f 1 -weight 1

    $w draw
}

proc geoloc::do_publish {w} {
    variable fields
    variable myjid
    foreach ff $fields {
	variable geoloc$ff
    }

    set args {}
    foreach ff $fields {
	lappend args -$ff [set geoloc$ff]
    }

    foreach connid [jlib::connections] {
	if {[string equal $myjid [jlib::connection_jid $connid]] || \
		[string equal $myjid [::msgcat::mc "All"]]} {
	    eval [list publish $connid \
		       -command [namespace current]::publish_result] \
		       $args
	    break
	}
    }

    foreach ff $fields {
	unset geoloc$ff
    }
    unset myjid
    destroy $w
}

# $res is one of: OK, ERR, DISCONNECT
proc geoloc::publish_result {res child} {
    switch -- $res {
	ERR {
	    set error [error_to_string $child]
	}
	default {
	    return
	}
    }

    NonmodalMessageDlg [epath] \
	    -aspect 50000 \
	    -icon error \
	    -title [::msgcat::mc "Error"] \
	    -message [::msgcat::mc "User location publishing failed: %s" $error]
}

proc geoloc::show_unpublish_dialog {} {
    variable myjid

    set w .user_geoloc
    if {[winfo exists $w]} {
	destroy $w
    }

    set connids [jlib::connections]
    if {[llength $connids] == 0} {
	NonmodalMessageDlg [epath] \
		-aspect 50000 \
		-icon error \
		-title [::msgcat::mc "Error"] \
		-message [::msgcat::mc "Unpublishing is only possible\
					while being online"]
	return
    }

    Dialog $w -title [::msgcat::mc "User location"] \
	    -modal none -separator 1 -anchor e -default 0 -cancel 1 -parent .
    $w add -text [::msgcat::mc "Unpublish"] \
	   -command [list [namespace current]::do_unpublish $w]
    $w add -text [::msgcat::mc "Cancel"] -command [list destroy $w]

    set f [$w getframe]

    set connjids [list [::msgcat::mc "All"]]
    foreach connid $connids {
	lappend connjids [jlib::connection_jid $connid]
    }
    set myjid [lindex $connjids 0]

    label $f.ccap -text [::msgcat::mc "Use connection:"]
    ComboBox $f.conn -editable false \
	    -values $connjids \
	    -textvariable [namespace current]::myjid

    if {[llength $connjids] > 1} {
	grid $f.ccap   -row 0 -column 0 -sticky e
	grid $f.conn   -row 0 -column 1 -sticky ew
    }

    grid columnconfigure $f 1 -weight 1

    if {[llength $connids] == 1} {
	do_unpublish $w
    } else {
	$w draw
    }
}

proc geoloc::do_unpublish {w} {
    variable myjid

    foreach connid [jlib::connections] {
	if {[string equal $myjid [jlib::connection_jid $connid]] || \
		[string equal $myjid [::msgcat::mc "All"]]} {
	    unpublish $connid \
		    -command [namespace current]::unpublish_result
	    break
	}
    }

    unset myjid
    destroy $w
}

# $res is one of: OK, ERR, DISCONNECT
proc geoloc::unpublish_result {res child} {
    switch -- $res {
	ERR {
	    if {[lindex [error_type_condition $child] 1] == "item-not-found"} {
		return
	    }
	    set error [error_to_string $child]
	}
	default {
	    return
	}
    }

    NonmodalMessageDlg [epath] \
	    -aspect 50000 \
	    -icon error \
	    -title [::msgcat::mc "Error"] \
	    -message [::msgcat::mc "User location unpublishing failed: %s" $error]
}

proc geoloc::provide_userinfo {notebook connid jid editable} {
    variable geoloc
    variable m2d
    variable ::userinfo::userinfo
    variable fields
    variable labels

    if {$editable} return

    set barejid [node_and_server_from_jid $jid]
    if {![info exists geoloc(alt,$connid,$barejid)]} return

    foreach ff $fields {
	set userinfo(geoloc$ff,$jid) $geoloc($ff,$connid,$barejid)
    }

    set f [pep::get_userinfo_dialog_pep_frame $notebook]
    set mf [userinfo::pack_frame $f.geoloc [::msgcat::mc "User location"]]

    set row 0
    foreach ff $fields {
	userinfo::pack_entry $jid $mf $row geoloc$ff $labels($ff)
	incr row
    }
}

# vim:ts=8:sw=4:sts=4:noet
