#!/usr/pkg/bin/perl
# -*- indent-tabs-mode: nil; -*-
# vim:ft=perl:et:sw=4
# $Id$

# Sympa - SYsteme de Multi-Postage Automatique
#
# Copyright (c) 1997, 1998, 1999 Institut Pasteur & Christophe Wolfhugel
# Copyright (c) 1997, 1998, 1999, 2000, 2001, 2002, 2003, 2004, 2005,
# 2006, 2007, 2008, 2009, 2010, 2011 Comite Reseau des Universites
# Copyright (c) 2011, 2012, 2013, 2014, 2015, 2016, 2017 GIP RENATER
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

use lib split(/:/, $ENV{SYMPALIB} || ''), '/usr/pkg/sympa/bin';
use strict;
use warnings;
use English qw(-no_match_vars);
use Getopt::Long;
use Pod::Usage;
use Data::Dumper;

use Sympa::Constants;
use Sympa::Database;
use Sympa::DatabaseDriver::LDAP;
use Sympa::Log;    # Show err logs on STDERR.

my %options;
unless (
    GetOptions(
        \%options,
        (   map {"$_=s"} @{Sympa::DatabaseDriver::LDAP->required_parameters},
            @{Sympa::DatabaseDriver::LDAP->optional_parameters},
            qw(use_ssl use_start_tls),    # Deprecated as of 6.2.15
            qw(filter scope)
        ),
        qw(suffix:s attrs:s),
        qw(help version)
    )
) {
    pod2usage(-exitval => 1, -output => \*STDERR);
}
if ($options{'help'}) {
    pod2usage(0);
} elsif ($options{'version'}) {
    printf "Sympa %s\n", Sympa::Constants::VERSION;
    exit 0;
}

# Parameters deprecated as of 6.2.15.
if ($options{use_start_tls}) {
    $options{use_tls} = 'starttls';
} elsif ($options{use_ssl}) {
    $options{use_tls} = 'ldaps';
}
delete $options{use_start_tls};
delete $options{use_ssl};

if ($options{'bind_dn'} and not $options{'bind_password'}) {
    local $SIG{TERM} = sub { system qw(stty echo) };
    system qw(stty -echo);
    print 'Bind password:';
    my $password = <STDIN>;
    chomp $password;
    print "\n";
    $SIG{TERM}->();

    $options{'bind_password'} = $password;
}

my $db = Sympa::Database->new('LDAP', %options);
unless ($db
    and defined $options{'suffix'}
    and defined $options{'filter'}) {
    pod2usage(-exitval => 1, -output => \*STDERR);
}

print join ' ',
    map { sprintf '%s=%s', $_, $options{$_} } qw(host suffix filter);
print "\n";

my ($mesg, $res);

$db->connect or die sprintf "Connect impossible: %s\n", ($db->error || '');
$mesg = $db->do_operation(
    'search',
    base   => $options{'suffix'},
    filter => $options{'filter'},
    scope  => ($options{'scope'} || 'sub'),
    attrs =>
        ($options{'attrs'} ? [split /\s*,\s*/, $options{'attrs'}] : ['']),
) or die sprintf "Search  impossible: %s\n", $db->error;
$res = $mesg->as_struct;

my $cpt = 0;
foreach my $dn (keys %$res) {

    my $hash = $res->{$dn};
    print "#$dn\n";

    foreach my $k (keys %$hash) {
        my $array = $hash->{$k};
        if ((ref($array) eq 'ARRAY') and ($k ne 'jpegphoto')) {
            printf "\t%s => %s\n", $k, join(',', @$array);
        } else {
            printf "\t%s => %s\n", $k, $array;
        }
    }
    $cpt++;
}

print "Total : $cpt\n";

$db->disconnect or printf "disconnect impossible: %s\n", $db->error;

__END__

=encoding utf-8

=head1 NAME

sympa_test_ldap, sympa_test_ldap.pl - Testing LDAP connection for Sympa

=head1 SYNOPSIS

  sympa_test_ldap.pl --filter=string --host=string --suffix=string
  [ --attrs=[ string,...|* ] ]
  [ --bind_dn=string [ --bind_password=string ] ]
  [ --port=string ] [ --scope=base|one|sub ]
  [ --use_tls=starttls|ldaps|none
    [ --ca_file=string ] [ --ca_path=string ]
    [ --ca_verify=none|optional|require ]
    [ --ssl_cert=string ] [ --ssl_ciphers=string ] [ --ssl_key=string ]
    [ --ssl_version=sslv2|sslv3|tlsv1|tlsv1_1|tlsv1_2|tlsv1_3 ] ]

  sympa_test_ldap.pl --help

  sympa_test_ldap.pl --version

=head1 DESCRIPTION

sympa_test_ldap.pl tests LDAP connection and search operation using LDAP
driver of Sympa.

=head1 SEE ALSO

L<Sympa::DatabaseDriver::LDAP>.

=head1 HISTORY

testldap.pl was renamed to sympa_test_ldap.pl on Sympa 6.2.

C<--use_ssl> and C<--use_start_tls> options were obsoleted by Sympa 6.2.15.
C<--use_tls> option would be used instead.

=cut
