# $Id: exceptions.rb 231 2003-09-19 13:06:01Z bolzer $
# Author::  Oliver M. Bolzer (mailto:oliver@fakeroot.net)
# Copyright:: (c) Oliver M. Bolzer, 2002
# License:: Distributes under the same terms as Ruby
# Exceptions defined for Vapor

module Vapor
  
# module for exceptions defined and raised by Vapor
module Exceptions

  # arent class for all Exception from Vapor
  class VaporException < StandardError 
  end
 
  # source of HIGH part for OID Generation not set
  class HighSourceNotSetException < VaporException #:nodoc:
  end

  # Could not contact Repository
  class RepositoryOfflineError < VaporException
  end

  # some kind of inconsistency in the backend
  class BackendInconsistentError < VaporException
  end

  # Persistable is readonly and no modifying operations can be
  # done on it.
  class PersistableReadOnlyError < VaporException
  end

  # class' superclass is not known to the Repository
  class UnknownSuperclassError < BackendInconsistentError
  end

  # class is already known to Repository
  class DuplicateClassError < BackendInconsistentError
  end

  # the metadata specified has some kind of error
  class InvalidMetadataError < BackendInconsistentError
  end

  # marking an already persistent object again as persistent
  class ObjectAlreadyPersistentError < VaporException
  end

  # class is a Persistable but not registered with the Repository 
  class ClassNotKnownError < VaporException
    def initialize( klass = "unknown" )
      super( "Persistable class #{klass} not known to Repository" )
    end
  end

  # class is not deletable because it has child classes
  class ClassNotDeletableError < VaporException
    def initialize( message, children = [] )
      super( message )
      @children = children.dup
      @children.freeze
    end
    attr_reader :children
  end

  # some type is not *yet* implemented
  class TypeNotImplementedError < VaporException #:nodoc:
  end

  # inconsistency between the type specified bymetadata and
  # it the actual value's type
  class VaporTypeError < VaporException
  end

  # some kind of Error occured during Qury processing
  class QueryError < VaporException
  end

  # Query does not parse correclty
  class InvalidQueryError < QueryError
  end

  # some kind of Error occured during Transaction processing
  class TransactionError < VaporException
  end

  # some kind of Error occured during Transaction Commit
  class CommitError < TransactionError
    def initialize( cause = nil )
      @causing_object = cause
    end
    attr_accessor :causing_object
  end
  
  # uniqueness constraint violated
  class UniquenessError < CommitError
  end

  # Object has been changed in the Datastore since it has
  # been loaded.
  class StaleObjectError < CommitError
  end

  # Object has been deleted by another PersistenceManager
  # since it has been loaded.
  class DeletedObjectError < CommitError
  end

  # Transaction is already finished, all operations on it raise
  # this exception
  class StaleTransactionError < TransactionError
  end

  # Transaction started while another is already
  # in progress
  class NestedTransactionError < TransactionError
  end

  # abort() or rollback() has been called on a Transaction.
  class TransactionAbortedError < TransactionError
  end 
end # module Exceptions
end # module Vaoir
