/*--------------------------------------------------------------------*//*:Ignore this sentence.
Copyright (C) 1999, 2001, 2005 SIL International. All rights reserved.

Distributable under the terms of either the Common Public License or the
GNU Lesser General Public License, as specified in the LICENSING.txt file.

File: XftFont.h
Author: Orignal by Keith Stribley heavily modified by Tim Eves
Responsibility: Keith Stribley, Tim Eves
Last reviewed: Not yet.

Description:
    A Font is an object that represents a font-family + bold + italic setting, that contains
	Graphite tables.
----------------------------------------------------------------------------------------------*/
#ifndef GR_FREETYPEFONT_INCLUDED
#define GR_FREETYPEFONT_INCLUDED
#include <string>
#include <graphite/GrClient.h>
#include <graphite/Font.h>

#include <map>
#include <ft2build.h>
#include FT_TRUETYPE_TABLES_H
//:End Ignore


namespace gr
{


/*-----------------------------------------------------------------------------
	Stock implementation of an Xft font class. 
------------------------------------------------------------------------------*/
/** 
* An implementation of the Graphite gr::Font class using Freetype to retrieve font 
* information.
*/
class FreetypeFont : public Font
{
public:
  /**
  * Alternative user constructor. Constructs the font from the supplied FT_Face.
  * @param ftFace the FT_Face handle to initialise from (must not be null)
  */
  FreetypeFont(FT_Face ftFace, int dpiX=72, int dpiY=72, 
  	FT_Int32 load_flags=FT_LOAD_DEFAULT);
  
  
  /**
  * Destructor - cleans up the tables that it has allocated if all other copies
  * of this font have been deleted.
  */
  virtual ~FreetypeFont();
  
  /**
  * Returns a copy of the recipient. Specifically needed to store the Font 
  * in a segment. 
  * @internal
  */
  virtual Font * copyThis();
  
  /**
  * Copy constructor - note the tables are shared between the copy and the
  * original font for efficiency reasons. 
  * The last copy to be deleted will delete the tables.
  * @param font the FreetypeFont object to copy from.
  * @internal
  */
  FreetypeFont(const FreetypeFont & font);

  //virtual FontErrorCode isValidForGraphite(int * pnVersion = NULL, int * pnSubVersion = NULL);
  /**
  * Returns a pointer to the start of a table in the font. The ftblid32 type 
  * is a 32-bit unsigned integer.
  * If the Font class cannot easily determine the length of the table, it may 
  * set 0 as the length (while returning a non-NULL pointer to the table). This
  * means that certain kinds of error checking cannot be done by the Graphite
  * engine.
  * Throws an exception if there is some other error in reading the table, or if
  * the table asked for is not one of the expected ones (cmap, head, name, Sile,
  * Silf, Feat, Gloc, Glat). [If we want to be able to read “non-standard” 
  * tables, then the caller needs to be responsible for freeing the memory.]
  */
  virtual const void * getTable(fontTableId32 tableID, size_t * pcbSize);
  
  /**
  * Returns the basic metrics of the font. It corresponds to the current
  * GrGraphics::get_FontAscent, get_FontDescent and GetFontEmSquare methods.
  * @param pAscent pointer to hold font ascent
  * @param pDescent pointer to hold font descent 
  * @param pEmSquare pointer to hold font EM square
  */
  virtual void getFontMetrics(float * pAscent = NULL, float * pDescent = NULL,
          float * pEmSquare = NULL);
          
  /**  Converts the point number of a glyph’s on-curve point to a pair of x/y
  * coordinates in pixels. The default implementation will read the curve
  * information directly from the font and perform a simple transformation to
  * pixels. Some subclasses (e.g., WinFont) will use a system-level API call to
  * return hinted metrics.
  * Note that the coordinates returned are floating point values, using a 
  * special floating-point point class.
  * @param gid glyph id
  * @param pointNum within glyph
  * @param xyReturn reference to a Point object to hold the x,y result
  */
  virtual void getGlyphPoint(gid16 gid, unsigned int pointNum, gr::Point & xyReturn);
  
  /**
  * Returns the metrics of a glyph in the font as given by Freetype. 
  * Note that the coordinates returned are floating point values, using special 
  * floating-point rectangle and point classes.
  * @param glyphID
  * @param boundingBox reference to gr::Rect to hold bounding box of glyph
  * @param advances refererence to gr::Point to hold the horizontal / vertical
  *                 advances
  */
  virtual void getGlyphMetrics(gid16 glyphID, gr::Rect & boundingBox, gr::Point & advances);
  
  /**
  * Returns true if the given arguments specify a font that contains Graphite
  * tables—specfically an “Silf” table.
  * @param face FT_Face handle
  * @return true if this font has the silf table
  */
  static bool FontHasGraphiteTables(FT_Face face);

  // Temporary, until interface gets thoroughly reworked:
  //GrEngine * GraphiteEngine();
        
  /**
  * Returns the font ascent in pixels. Using floating point allows us to 
  * avoid rounding errors. 
  * Value is the same as that returned by getFontMetrics()
  * @return the font ascent
  */
  virtual float ascent();

  /**
  * Returns the font descent in pixels with a positive sign.
  * Value is the same as that returned by getFontMetrics()
  * @return the font descent in pixels.
  */
  virtual float descent();

  /**
  * Returns the total height of the font. 
  * Equivalent to ascent() + descent().
  * @return font height
  */
  virtual float height();

  /**
  * @return true if the font is a bold styled one. 
  */
  virtual bool bold();
  
  /**
  * @return true if the font is italic
  */
  virtual bool italic();

  /** @return the DPI for the x-axis.
  */
  virtual unsigned int getDPIx();

  /** @return Returns the DPI for the y-axis.
  */
  virtual unsigned int getDPIy();
 
protected:
  virtual void UniqueCacheInfo(std::wstring &, bool &, bool &);
  
  // Specific to FreetypeFont:
  //
  /** @return the current Freetype GLyph load flag settings
  */
  FT_Int32 glyphLoadFlags() const throw();
  
  /** @return ste the Freetype GLyph load flag settings
  */
  void setGlyphLoadFlags(FT_Int32 load_flags) throw();
  
  /** @return set the DPI for the x-axis.
  */
  void setDPIx(unsigned int dpi);

  /** @return set the DPI for the y-axis.
  */
  void setDPIy(unsigned int dpi);

private:
  typedef std::map<gid16, std::pair<gr::Rect, gr::Point> >	GlyphMetricMap;
  typedef std::map<fontTableId32, std::pair<FT_Byte *, size_t> >	TableMap;

  /** Default constructor is not used */
  FreetypeFont();
  
  // Member variables:
  FT_Face   m_ftFace;
  FT_Int32  m_ft_load_flags;

  unsigned long	m_clrFore, m_clrBack;
  bool         	m_fBold, m_fItalic;
  float        	m_pixHeight, m_ascent, m_descent, m_emSquare;
  unsigned int 	m_dpiX, m_dpiY;
  std::wstring 	m_faceName;
  GlyphMetricMap m_glyphMetrics;
  TableMap     	m_tables;
};

/** @return set the DPI for the x-axis.
*/
inline void FreetypeFont::setDPIx(unsigned int dpi)	{ m_dpiX = dpi; }

/** @return set the DPI for the y-axis.
*/
inline void FreetypeFont::setDPIy(unsigned int dpi)	{ m_dpiY = dpi; }

inline FT_Int32 FreetypeFont::glyphLoadFlags() const throw() { 
	return m_ft_load_flags;
}
  
inline void FreetypeFont::setGlyphLoadFlags(FT_Int32 load_flags) throw () {
	m_ft_load_flags = load_flags;
}


} // namespace gr

#endif


