package PAR::StrippedPARL::Dynamic;
use 5.008001;
use strict;
use warnings;
our $VERSION = '0.958';

use base 'PAR::StrippedPARL::Base';

our $Data_Pos = tell DATA;

=head1 NAME

PAR::StrippedPARL::Dynamic - Data package containing a dynamic PARL

=head1 SYNOPSIS

  # For details, see PAR::StrippedPARL::Base.
  PAR::StrippedPARL::Dynamic->write_parl($file) or die "Some error...";

=head1 DESCRIPTION

This class is internal to PAR. Do not use it outside of PAR.

This class is basically just a container for a dynamic binary PAR loader
which doesn't include the PAR code like the F<parl> or F<parl.exe>
you are used to. If you're really curious, I'll tell you it is
just a copy of the F<myldr/par> (or F<myldr/par.exe>) file.

The data is appended during the C<make> phase of the PAR build process,
but only if applicable: If you perl is static, you won't get the dynamic
loader.

If the binary data isn't appended during the build process, the class
methods will return the empty list.

=head1 CLASS METHODS

Inherits the methods from L<PAR::StrippedPARL::Base>.

=cut

sub _data_pos {
    my $class = shift;
    return $Data_Pos;
}

=head1 AUTHORS

Steffen Mueller E<lt>smueller@cpan.orgE<gt>,
Audrey Tang E<lt>cpan@audreyt.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright 2006-2009 by Steffen Mueller E<lt>smueller@cpan.orgE<gt>.

This program is free software; you can redistribute it and/or 
modify it under the same terms as Perl itself.

See F<LICENSE>.

=cut

$^V eq 5.38.2 
    or die sprintf("Perl (%s) version (%vd) doesn't match the version (5.38.2) ".
                   "that PAR::Packer was built with; please rebuild PAR::Packer", 
                   $^X, $^V);

1;

__DATA__
M?T5,1@("`0`````````````#`"L````!````````(P``````````0```````
M`+T``````@!``#@`!P!``!L`&@````8````$`````````$``````````0```
M``````!``````````8@````````!B``````````(`````P````0````````!
MR`````````'(`````````<@`````````%P`````````7``````````$````!
M````!0```````````````````````````````````````*QT````````K'0`
M`````!````````$````'````````K'@``````!"L>```````$*QX````````
M#?`````````.D```````$````````@````8```````"LH```````$*R@````
M```0K*`````````!X`````````'@``````````@````$````!`````````'@
M`````````>`````````!X``````````L`````````"P`````````!&1TY5``
M```$````````J&P```````"H;````````*AL`````````00````````!!```
M```````$+W5S<B]L:6)E>&5C+VQD+F5L9E]S;P``````!P````0````!3F5T
M0E-$```[FLH`````!`````0````#4&%8````````````````80```&<`````
M`````````#`````.```````````````W````#P```#\`````````+P``````
M```%````2@```%X`````````'P```$D````(````'0```!0````:````00``
M``````!:`````````!<````[````50```%\`````````8P```&4````N````
M`````%0```!"````8@```%D```!#`````````$8````B````-````%@````F
M````*@`````````)````3``````````,`````````&8```!7````2````"``
M```&````9`````````!<`````````!@````M````2P``````````````````
M`!(````R``````````````!=````*P`````````+`````````"0````X````
M`````$T````'````1``````````Z`````````````````````````$<`````
M````.0```%````!@`````````&$`````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M```````````3````````````````````````````````````````````````
M````(P```"$`````````*0`````````G````*````"P```````````````T`
M````````'````!L````9```````````````1`````````#,````Q````-@``
M`!X```````````````````````````````````!`````)0``````````````
M`````!8``````````````#X````*````````````````````/0````````!/
M`````````%,``````````````#P```!2`````````$X`````````10```!``
M`````````````%$````5````5@```#4```!;````````````````````````
M```````````````````#```)````````(N`````````````````#```3````
M```0KH`````````````````=``````````````(````````````````=````
M``````````,``````````````)@2``````````````````````````````P2
M`````````````````````````````!02````````````````````````````
M`TL2```*````````,"``````````#````!PB````````````````````````
M`````"L2`````````````````````````````Q02````````````````````
M`````````#$2`````````````````````````````UP2```*````````.\``
M````````*````#@2`````````````````````````````R$2````````````
M`````````````````$`2`````````````````````````````W<2```*````
M````1V``````````@````(,2`````````````````````````````$@2````
M`````````````````````````X,0```5```````0NF@``````````````%@2
M`````````````````````````````&<2````````````````````````````
M`&T2`````````````````````````````'02````````````````````````
M`````'X2`````````````````````````````(@0```+````````9P``````
M`````````(X2`````````````````````````````)<2````````````````
M`````````````@D2`````````````````````````````)T2````````````
M`````````````````*02`````````````````````````````+(2````````
M`````````````````````,,1`````````````````````````````I`B````
M`````````````````````````-`2`````````````````````````````XH2
M```*````````,J``````````(````Y<2```*````````,L``````````9```
M`Z,2```*````````/0`````````'3````-X6````````````````````````
M`````/$2`````````````````````````````S,2````````````````````
M`````````/P2`````````````````````````````041```6```````0NP``
M````````"````0T2`````````````````````````````PH2````````````
M`````````````````1D2`````````````````````````````2`2````````
M`````````````````````[`2```*````````7^`````````!$````3,1````
M`````````````````````````442`````````````````````````````542
M`````````````````````````````6,2````````````````````````````
M`6@2`````````````````````````````[L1```5```````0NF``````````
M"````7<2`````````````````````````````8,2````````````````````
M`````````9@2`````````````````````````````:,2````````````````
M`````````````T80```6```````0NP@``````````````LL2````````````
M`````````````````;82`````````````````````````````_82```*````
M````(P```````````````<02`````````````````````````````<P2````
M`````````````````````````=@2`````````````````````````````\42
M```*````````,T`````````%I````=X2````````````````````````````
M`>D2`````````````````````````````](2```*````````80`````````!
MJ````?H2`````````````````````````````]P0```6```````0NF@`````
M`````````^@2```*````````+V``````````4````_`2```*````````9$``
M```````"C````@$2`````````````````````````````G@B````````````
M`````````````````_42```*````````(P```````````````_T2```*````
M````,$`````````"5```!`H2```*````````/```````````\````@@2````
M````````````````````````!!P2```*````````.0``````````F````A$2
M`````````````````````````````B(2````````````````````````````
M`K02````````````````````````````!"H2```*````````.:``````````
M:````BD2`````````````````````````````C(2````````````````````
M`````````C@2`````````````````````````````OT2````````````````
M`````````````CT2`````````````````````````````O`1```6```````0
MNH``````````"````D42`````````````````````````````R@2````````
M`````````````````````DP2````````````````````````````!#(2```*
M````````7X``````````7````E,2`````````````````````````````F(1
M`````````````````````````````N41```5```````0ND``````````"```
M!#40```)````````(N``````````````!#L2```*````````1&`````````"
M]````2P2`````````````````````````````F<2````````````````````
M`````````SH2``````````````````````````!L:6)P97)L+G-O`')E86QL
M;V,`;65M;6]V90!?7V-X85]F:6YA;&EZ90!R;61I<@!S=')C:'(`<W1R;F-M
M<`!S=')R8VAR`&)O;W1?1'EN84QO861E<@!P97)L7V-O;G-T<G5C=`!A8F]R
M=`!U;FQI;FL`7U]L<W1A=#4P`'!E<FQ?9G)E90!?9FEN:0!?7W-T870U,`!?
M97AI=`!M96UC<'D`<&5R;%]D97-T<G5C=`!?7W-T86-K7V-H:U]F86EL`%!,
M7V1O7W5N9'5M<`!097)L7W-V7W-E='!V`%!,7V-U<G)E;G1?8V]N=&5X=`!P
M97)L7V%L;&]C`&-L;W-E9&ER`&5N=FER;VX`7U]O<&5N9&ER,S``;65M<V5T
M`%!E<FQ?<V%F97-Y<VUA;&QO8P!?7W-T86-K7V-H:U]G=6%R9`!097)L7V=V
M7V9E=&-H<'8`7U]S<')I;G1F7V-H:P!R96%D`%!E<FQ?<WES7VEN:70S`%!E
M<FQ?9V5T7W-V`%!E<FQ?8W)O86M?;F]C;VYT97AT`%!E<FQ?;F5W6%,`4&5R
M;%]A=&9O<FM?=6YL;V-K`%!E<FQ?<WES7W1E<FT`7U]E<G)N;P!097)L7VUG
M7W-E=`!M:V1I<@!P97)L7W!A<G-E`%!E<FQ?;F5W6%-?9FQA9W,`86-C97-S
M`&=E='!I9`!V9G!R:6YT9@!097)L7V%T9F]R:U]L;V-K`&UE;6-M<`!P97)L
M7W)U;@!L<V5E:P!O<&5N`'-P<FEN=&8`9V5T=6ED`'-T<FQE;@!?7W1L<U]G
M971?861D<@!?7W-&`'-T<F-M<`!L:6)M+G-O+C``7U]D97)E9VES=&5R7V9R
M86UE7VEN9F\`7U]R96=I<W1E<E]F<F%M95]I;F9O`&QI8F-R>7!T+G-O+C$`
M<W1R9'5P`&QI8G!T:')E860N<V\N,0!P=&AR96%D7V%T9F]R:P!L:6)C+G-O
M+C$R`%]?<')O9VYA;64`7U]P<U]S=')I;F=S`%]?9V5T<'=U:60U,`!?7W-Y
M<V-A;&P`7U]S='!C<'E?8VAK`&%T97AI=`!?;&EB8U]I;FET`'-T<G1O:P!?
M7W)E861D:7(S,`!?96YD`'!A<E]C=7)R96YT7V5X96,`9V5T7W5S97)N86UE
M7V9R;VU?9V5T<'=U:60`<&%R7V-L96%N=7``7V5D871A`'!A<E]B87-E;F%M
M90!P87)?9&ER;F%M90!P87)?;6MT;7!D:7(`<VAA7W5P9&%T90!M>5]P87)?
M<&P`<&%R7VEN:71?96YV`'-H85]F:6YA;`!?7V)S<U]S=&%R=`!X<U]I;FET
M`&UA:6X`7U]S=&%R=`!P87)?9FEN9'!R;V<`<&%R7W-E='5P7VQI8G!A=&@`
M<&%R7V5N=E]C;&5A;@!P87)?9&EE`'-H85]I;FET`%A37TEN=&5R;F%L<U]0
M05)?0D]/5``O=7-R+W!K9R]L:6(O<&5R;#4O-2XS."XP+W-P87)C-C0M;F5T
M8G-D+71H<F5A9"UM=6QT:2]#3U)%.B]U<W(O<&MG+VQI8@``````````````
M```0KH@`````````%@```````",@```````0KI``````````%@``````$+J(
M```````0KI@`````````%@```````&<8```````0KJ``````````%@``````
M$+J1```````0KJ@`````````%@``````$+J8```````0KK``````````%@``
M````$+J0```````0KK@`````````%@``````$+K8```````0KL``````````
M%@```````*A@```````0KL@`````````%@```````"_````````0KM``````
M````%@``````$+I0```````0KM@`````````%@``````$+K@```````0KN``
M````````%@``````$+KH```````0KN@`````````%@``````$+KP```````0
MKO``````````%@```````&<P```````0KO@`````````%@```````&<X````
M```0KP``````````%@```````&=````````0KP@`````````%@```````&=@
M```````0KQ``````````%@```````&K0```````0KQ@`````````%@``````
M`&=X```````0KR``````````%@```````&>8```````0KR@`````````%@``
M`````&>(```````0KS``````````%@```````&>@```````0KS@`````````
M%@```````&>H```````0KT``````````%@```````&>P```````0KT@`````
M````%@```````&?````````0KU``````````%@```````&?(```````0KU@`
M````````%@```````&?8```````0KV``````````%@```````&?@```````0
MKV@`````````%@```````&?P```````0KW``````````%@```````&@`````
M```0KW@`````````%@```````&@0```````0KX``````````%@```````&@@
M```````0KX@`````````%@```````&@P```````0KY``````````%@``````
M`&A(```````0KY@`````````%@```````&A@```````0KZ``````````%@``
M`````&AP```````0KZ@`````````%@```````&B8```````0K[``````````
M%@```````&B````````0K[@`````````%@```````&B@```````0K\``````
M````%@```````&BP```````0K\@`````````%@```````&C````````0K]``
M````````%@```````&C@```````0K]@`````````%@```````&CH```````0
MK^``````````%@```````&CP```````0K^@`````````%@```````&CX````
M```0K_``````````%@```````&D````````0K_@`````````%@```````&D(
M```````0L```````````%@```````&D8```````0L`@`````````%@``````
M`&D@```````0L!``````````%@```````&DH```````0L!@`````````%@``
M`````&EX```````0L"``````````%@```````&G0```````0L"@`````````
M%@```````&G8```````0L#``````````%@```````&H(```````0L#@`````
M````%@```````&H8```````0L$``````````%@```````&DX```````0L$@`
M````````%@```````&C0```````0L%``````````%@```````&C8```````0
ML%@`````````%@```````&GH```````0L&``````````%@```````&GX````
M```0L&@`````````%@```````&G@```````0L'``````````%@```````&HP
M```````0L'@`````````%@```````&HX```````0L(``````````%@``````
M`&I````````0L(@`````````%@```````&I(```````0L)``````````%@``
M`````&I@```````0L)@`````````%@```````&IH```````0L*``````````
M%@```````&J0```````0L*@`````````%@```````&K8```````0L+``````
M````%@```````&K@```````0L+@`````````%@``````$*R(```````0L,``
M````````%@``````$*QT```````0L,@`````````%@``````$*QT```````0
ML.``````````%@``````$*QT```````0L.@`````````%@``````$*R0````
M```0L/``````````%@```````&<````````0L1``````````%@``````$+L`
M```````0L2``````````%@```````*EP```````0L2@`````````%@``````
M$*QT```````0L3``````````%@``````$+I(```````0L3@`````````%@``
M````$+I@```````0L4@`````````%@``````$*QX```````0L5@`````````
M%@``````$*QT```````0L6``````````%@``````$*R````````0L7``````
M````%@``````$*R@```````0L8@`````````%@``````$*QT```````0L9``
M````````%@``````$+J````````0L:``````````%@``````$+I````````0
ML:@`````````%@```````$1@```````0ND``````````%@``````$+J(````
M```0ND@`````````%@``````$+I(```````0NF``````````%@```````&KH
M```````0L-`````)````%``````````````````0L-@````3````%```````
M```````````0L/@````A````%``````````````````0L0`````B````%```
M```````````````0L0@````G````3P`````````````````0L1@````Q````
M%``````````````````0L4`````Z````%``````````````````0L6@```!+
M````%``````````````````0L8````!1````%``````````````````0L9@`
M``!@````%``````````````````0LH`````%````%0`````````````````0
MLJ`````&````%0`````````````````0LL`````'````%0``````````````
M```0LN`````)````%0`````````````````0LP`````*````%0``````````
M```````0LR`````+````%0`````````````````0LT`````,````%0``````
M```````````0LV`````.````%0`````````````````0LX`````/````%0``
M```````````````0LZ`````0````%0`````````````````0L\`````2````
M%0`````````````````0L^`````5````%0`````````````````0M``````6
M````%0`````````````````0M"`````7````%0`````````````````0M$``
M```8````%0`````````````````0M&`````9````%0`````````````````0
MM(`````;````%0`````````````````0M*`````<````%0``````````````
M```0M,`````=````%0`````````````````0M.`````>````%0``````````
M```````0M0`````?````%0`````````````````0M2`````@````%0``````
M```````````0M4`````B````%0`````````````````0M6`````C````%0``
M```````````````0M8`````H````%0`````````````````0M:`````I````
M%0`````````````````0M<`````J````%0`````````````````0M>`````L
M````%0`````````````````0M@`````M````%0`````````````````0MB``
M```N````%0`````````````````0MD`````O````%0`````````````````0
MMF`````R````%0`````````````````0MH`````S````%0``````````````
M```0MJ`````T````%0`````````````````0ML`````U````%0``````````
M```````0MN`````W````%0`````````````````0MP`````X````%0``````
M```````````0MR`````Y````%0`````````````````0MT`````\````%0``
M```````````````0MV`````]````%0`````````````````0MX`````_````
M%0`````````````````0MZ````!`````%0`````````````````0M\````!!
M````%0`````````````````0M^````!#````%0`````````````````0N```
M``!$````%0`````````````````0N"````!&````%0`````````````````0
MN$````!*````%0`````````````````0N&````!+````%0``````````````
M```0N(````!/````%0`````````````````0N*````!2````%0``````````
M```````0N,````!3````%0`````````````````0N.````!5````%0``````
M```````````0N0````!6````%0`````````````````0N2````!7````%0``
M```````````````0N4````!8````%0`````````````````0N6````!9````
M%0`````````````````0N8````!;````%0`````````````````0N:````!<
M````%0`````````````````0N<````!=````%0`````````````````0N>``
M``!D````%0`````````````````0N@````!E````%0`````````````````0
MNB````!F````%0``````````G>._4$``$"<!````@<?@"('H````````````
M``````"\$```B!```)`0``,02`#=DA```0$````!`````0```)WCOU`[````
M`P```"\`!")``!#LK@7C4+H78MB"$&)`^EW`'>Y=P`&`IT`7&F``"`$```#"
M7T``G\!``+H'8`B`IT`7*F___<)?0`"!S^`(`0```#!H``,!`````0```)WC
MOU`#````.0```"\`!")``!#4K@7B\((08M"X%R+X^%W`'.Y=P`&`I<`<&F``
M4K00(`&R$#__M2ZP(#$@```W'___L!8@!+LV<""@.``:MS;P"!!H`#"S-G`4
M"&``2("B`!`88`!2@@!`&("@0!H(8`!9@*(`&0A@`&8#/__\@RAP(((08`&`
MH@`!&F``=X<R(`H)`L``AA#`!,8D8`2$"B/_@S(P(`<@A!B""&/_@A!``\(D
M8`@#('`8@A"``04@RAS")&`4A!"@(,0D8`P%((00A!"@!<0D8!"!W&`$@=Q@
M"('<8`R!W&`0@=Q@%)$R,"H#`,``D!(``=`D0`"!W&``K@7@&("EP!P:8``:
M`0```,)=X`B""$`=@*!@^!)@`&X!````PEW@$)_`0`#B7<``@B(`$04`(`"$
M`$`"@*"`&QAO_\:`H@`=@RA@"`4,(`"#,&`*@A!``L(D0`"!W&``K@7@&("E
MP!PJ;__KPEW@"('/X`@!````@@HC_P4@<!B"$$`"PB1@!('<8`21,C`*`P#`
M`)`2``'0)$``@=Q@`!!O_]FN!>`8@@HC_P4@AA^"$$`"!2!P&,(D8`3$)&`(
M@=Q@!('<8`@0;__QD#@`"(($?_R0($`(`R>$&,(D8`B1,B`"`Q```)`2``'0
M)&`$@=Q@!('<8`@#((3XPB1``('<8``0;__`K@7@&(,J(`<'((08@S!@$X0*
M(_^"$$`#PB1@!`,@<!B"$(`!!2#*',(D8`R$$*`,Q"1@"('<8`2!W&`(@=Q@
M#)$R,!8#`,``D!(``=`D0`"!W&``$&__JJX%X!B$(``(!R"$&(,PL`R0"B__
M@@AC_X(`8`&"$$`#PB1@!`,@B`""$&`!PB1@"`,@<!B0$@`!`R#*'-`D8!""
M$&`,PB1@#('<8`2!W&`(@=Q@#('<8!"%,+`6`P#``(00@`'$)$``@=Q@`!!O
M_XZN!>`80`0C:@$````!````,&@`!P$````!`````0````$````!`````0``
M`)WCOU`O``0A0``0%JX%X_@"QD!C`P```#<```#$7D``@A!C$+86XI#&7F`0
MPEW``?9=P!O(6(``\G!```+!`$[&=L``!P```(80XR#&7<`#R'#``,18@`#"
M"(``@RA@&("@8``"0``+@SA@&("@8"\23__ZA`"@`<1PP`#""(``@RA@&("@
M8``23__Y@SA@&`+&``0!````0`0C&I`0`!A`!"2@`0````,```""$&+PPEW`
M`0+`0$,!````.P```#D```"Z%V)(N!<B8/A=P!SZ7<`=@*=`'!I@``D1````
MPE]``)_`0`"Z!V`(@*=`'"IO__W"7T``$0```#L```"0$B`(.0```-!=P`BZ
M%V,(N!<BJ$`$(OOX7<`<^EW`'8"G0!P:8``)$0```,)?0`"?P$``N@=@"("G
M0!PJ;__]PE]``!$```"0$B)P0`0B[=!=P`A___[#`0```-!&8`C47L``0``/
M%])>0`!`!"*E`0````,````%````@A!C((00H!#"7<`!Q%W``@K^/\/$<$``
M$&__PP$````5````EA`@$Y02H!B2$"`"U%W`"D`$(W20$"`$0`0C&I`0(`%_
M__[0.P```!!O_[\Y````,&@`!`$````!`````0```)WCOU`O``0A0``/GJX%
MXA@"Q@`V`P```((08I#N7<`!]%W```+&@#*Z$"``PDX``(888#V`H``!A$`@
M`("@``."0"``@(B``1)``!:V$"``^EZ```+'0!&W/N``N!``&I`0`!V4$``;
M0`0BII(0`!B`HB``,D@`!_I?(`C"3T`;@*!@/2)``!.V!N`!^E\@"`K_?_2X
M!R`(@<_@")`0``VV$``8PD[@`8888#V`H``!A$`@`("@``."0"``@(B``1)'
M__FV!N`!$&__XK8FP!BX)P`:N3\P`[H'0!OX)D``@<_@")`0``VZ$"``@<_@
M")`0``TP:``#`0````$```"=X[ZP.P```"\`!"%```]=K@7A%+H78ICZ7<`=
MPE]``,)WI_>"$"```L8`!@$```#"3@``@*!@`!)```H!````PE^G]\1?0`""
M&$`"A!`@``K`0%$!````@<_@"`$```!`!",(D!``&`+R/_:X$``(L@>G7R,`
M`#PA```00`0D$9`0`!P"P@`Q`0```,(*(`V`H&`N$D@`!K8"(`W"#N`!@*!@
M``)/__8!````P@H@#8"@8"XB2``KP@[@`4`$(^F0$``8M!``"$`$(^:0$``;
MD`:`")`"(`)`!"/JD3H@`!,```"6$``;DA)@<)00`!C27<`)0`0CP[80``B2
M$``90`0BB)`0`!N`HC__`D``!Y`0`!O"!Z=G@@A`$8"@0!`"0``9`0```$`$
M(F8!````0`0B3)`0`!M`!"/BD!``'"KZ/]3""B`-0`0BQI`0`!Q`!"(4D!``
M&!!O_[H!````@*!@+A)/_]8!````P@[@`H"@8``"3__"`0```!!O_]`!````
M?___GP$````0;__I`0```$`$(HL!`````0```)WCOK`#````D@>G7X(08IB0
M$``8+P`$($``#O*N!>-H[EW``<)=P`#"=Z?W@A`@`$`$(D0!````@I(@`#)(
M`!""$"``Q`>G9P<``#P)```HA`B``P<``!"&&(`#A!B`!("@``.&8#__@*``
M`H1@/_^`D,`"$D``"I(0(`+$7Z?WQEW``(08@`.&$"``"L"`";$X8`"!S^`(
M`0```$`$(R&0$``8@*``"!!O__6"8#__0`0B7`$````!`````0```)WCNT`Y
M````N@>C]R\`!"!```[$K@7BL+@7(IB4$"0`^%W`')(0`!B0$``=PE\``,)W
MI_>"$"``0`0AR@$```"0`C__@*(`'3AH``G"2@``$&@`"Y(0("^0`C__@*(`
M'0)@``>2$"`OPDH``("@8"\B3__ZP"H``)(0("]`!"'9D!``'0+"``Z`H@`=
M.&@``L`J``!`!",;D!``'<)?I_?$7P``@AA``H00(``*P$`.L!``"('/X`@!
M````PE^G]\1?``""&$`"A!`@``K`0`8Q````L!8@>/!=P!@0A",)@>@``$`$
M(A\!`````0```#!H``0!`````0````$```"=X[]`PDY``#L````O``0@0``.
MA*X%X;"Z%V*8@AA@/?I=P!V`H``!LF9__\)?0`#"=Z?W@A`@`$`$(S*0$``9
MMA``")('I_-___[6D!``&`+"`%ZX$``(0`0C*@$```"`IL`("&``4.!'I_,U
M````M!:BD/1=P!JA+#`#PDX``(888#V`H``!A$`@`("@``."0"``@(B``2)`
M``Z"$"``@A``&,1(8`&(&*`]@*```H9`(`"`H``$A$`@`("(P`(23__Y@@!@
M`8(@0!B#.&``D`;@`OA>@`!`!",2D`(``0+"`"G0=P`0PD>G\X8&(`'(#@``
M^%Z``(4H<`.+*2`8@SE@&("@``7$7P`"@AA@/8I`(`"`H``!@D`@`(")0`$"
M0``/R"B``,((P`"$`*`!B2A@&,(H@`"#.2`8@*``!((88#V(0"``@*```8)`
M(`"`B0`!$D__]88`X`&&`*`!@A`@/<(H@`"$$"``P@Y``L(HP`*#*&`8@*!@
M`!)/__R$`*`!PE^G]\1?0`""&$`"A!`@``K`0$`!````@<_@"`$```"$$"``
MP@Y``L(O``*#*&`8@*!@`!)/__R$`*`!$&__\0$````U````M!:BD/1=P!KD
M7H``PER```+`0"N"$``2N!`@`,188`B&$``<@@!@"`KXO_VX!R`!D@#@`Y,Z
M8`"3*G`#`P```((08#CN7<`!P@7``("@8``22``3@A`@`9`0``E`!"*^PB7`
M``+R/]6B$``(H3\@`)(0`!*A+#`#0`0@[Y00`!#B=H``@@<@`8,X8`"#*'`#
MP'1``1!O_X?X)Z?S0`0@WI`0`!+0=H```O(_Q*(0``BA/R``$&__]*$L,`.2
M$"`0$&__W[@0(`!`!"%S`0````$```"=X[]0$0```#<````O``0?0``-W*X%
MXQ"0`B*(-0```-!=P`@S````D`'`"+86X$"T%J)8\%H``+(68(#V7<`;]%W`
M&O)=P!D0A"'>@>@``#!H``0!`````0````$```"=X[]0$0```#4````O``0?
M0``-Q*X%XK"0`B*(,P```-!=P`BT%J,HD`'`"/1=P!JX$``(T%H``+(68(CR
M7<`9?___V+H0(`#P7P``.0```+@7()#X7<`<$(0A^I?H`!R=X[]0@<_@")`0
M(``P:``%`0````$````!`````0```)WCNI`A````$0```"\`!!]```VDK@7B
M,*`4(IB0$B"8X%W`$)('HUO07<`(PEP``,)WI_>"$"``?__]^@$```"V$``(
MDA`@+T`$(*Z0$``8`L(`#S4````1````DA``&)`2(*A___\'T%W`",)?I_?$
M7```@AA``H00(``*P$!S`0```('/X`@!````D!``&;06H*"R!Z/W]%W`&J('
MHU]`!"'X)P``/)(0`!I`!"$M)0``(`+R/^FZ$``(PD]``("@8``"2`!#`0``
M``+&P`>2$``=0`0B.Y`0`!N`HB```D``-)(0`!I`!"(FD!``'8("/_^"!T`!
M@*=``0IH``BX$``($&@`#0$```""`'__@*=``0)@``8!````Q$A``("@H"\B
M3__ZP"A``$`$(A60$``=N!``"$`$(A*0$``8N`(`'+@'(`&`IR/_&&?_PQL`
M```7````\'.HKYH38+"6$N"XVEW`#9@0`!V4$"0`UEW`"Y(0(`!`!"$RD!``
M&9(0`!%`!""OD!``&8"B(``22``(DA``&L('HV>""$`3@*!`$@)``!N2$"`!
MDA``&D`$(.R0$"```O(_J+H0``C"3T``@*!@`!)/_\$!````(L;`!SL```#"
M#L``@*!@+B)(``?"#N`!.P```+@0(`&Z%V!X$&__T?I=P!V`H&```D__[)(0
M`!H0;__Y.P```$`$(720$``9@*(@`!)/_^62$``:$0```)(0`!F0$B"H?__^
ME-!=P`A`!"&2D!``&1!O_XNP$``(0`0@I@$````!````,&@``P$````!````
MG>._4)`0`!A`!"`^DA`@+X("(`&Q>A0!$(0A@H'H``"=X[]0+P`$'D``#0:N
M!>.X`L8`#I`0`!C"3@``@*!@`")```LQ````0`0AM@$```"0`B`!@*(D`!A@
M``D!````$&_^,('H```Q````L!8@>/!=P!@0A"%K@>@``('/X`B0$"``,&@`
M!P$````!`````0````$````!`````0```)WCOT`Y````-P```"\`!!Y```SD
MK@7C,+@7(IBV%N#`^%W`'+H'I_/V7<`;PE\``,)WI_>"$"``DA``'7___3F0
M$``;(L(`%C<````#````Q$>G\X(08I#"7<`!A2BP`\A80`""`0`"QEA@"`+P
M__/&<0`"Q%A@$((`8`@*^+_^Q'!``)(0`!U___TFD!``&PKZ/_`#````-P``
M`+86X,CV7<`;DA``'7___1Z0$``;(L(`%C<````#````Q$>G\X(08I#"7<`!
MA2BP`\A80`""`0`"QEA@"`+P__/&<0`"Q%A@$((`8`@*^+_^Q'!``)(0`!U_
M__T+D!``&PKZ/_`#````-P```+86X-#V7<`;DA``'7___0.0$``;(L(`%C<`
M```#````Q$>G\X(08I#"7<`!A2BP`\A80`""`0`"QEA@"`+P__/&<0`"Q%A@
M$((`8`@*^+_^Q'!``)(0`!U___SPD!``&PKZ/_`#````-P```+86X-CV7<`;
MDA``'7___.B0$``;`L(`%C$````#````Q$>G\X(08I#"7<`!A2BP`\A80`""
M`0`"QEA@"`+P__/&<0`"Q%A@$((`8`@*^+_^Q'!``)(0`!U___S5D!``&PKZ
M/_`#````,0```+`6(.#P7<`8DA``'7___,V0$``8(L(`%C<````#````Q$>G
M\X(08I#"7<`!A2BP`\A80`""`0`"QEA@"`+P__/&<0`"Q%A@$((`8`@*^+_^
MQ'!``)(0`!U___RZD!``&`KZ/_`#````-P```+86X.CV7<`;DA``'7___+*0
M$``;(L(`%C<````#````Q$>G\X(08I#"7<`!A2BP`\A80`""`0`"QEA@"`+P
M__/&<0`"Q%A@$((`8`@*^+_^Q'!``)(0`!U___R?D!``&PKZ/_`#````-P``
M`+86X)CV7<`;DA``'7___)>0$``;`L(`%C4````#````Q$>G\X(08I#"7<`!
MA2BP`\A80`""`0`"QEA@"`+P__/&<0`"Q%A@$((`8`@*^+_^Q'!``)(0`!U_
M__R$D!``&PKZ/_`#````-0```+06H/#T7<`:DA``'7___'R0$``:`L(`%C,`
M```#````Q$>G\X(08I#"7<`!A2BP`\A80`""`0`"QEA@"`+P__/&<0`"Q%A@
M$((`8`@*^+_^Q'!``)(0`!U___QID!``&@KZ/_`#````,P```+(68/CR7<`9
MDA``'7___&&0$``9`L(`%B$````#````Q$>G\X(08I#"7<`!A2BP`\A80`""
M`0`"QEA@"`+P__/&<0`"Q%A@$((`8`@*^+_^Q'!``)(0`!U___Q.D!``&0KZ
M/_`#````(0```*`4(0#@7<`0DA``'7___$:0$``0(L(`%B$````#````Q$>G
M\X(08I#"7<`!A2BP`\A80`""`0`"QEA@"`+P__/&<0`"Q%A@$((`8`@*^+_^
MQ'!``)(0`!U___PSD!``$`KZ/_`#````(0```*`4(*C@7<`0DA``'7___"N0
M$``0`L(`%A$````#````Q$>G\X(08I#"7<`!A2BP`\A80`""`0`"QEA@"`+P
M__/&<0`"Q%A@$((`8`@*^+_^Q'!``)(0`!U___P8D!``$`KZ/_`#````$0``
M`)(0`!V0$B$(?__\$=!=P`@"P@`$DA``"'___260$``9$0```)(0`!V0$B$0
M?__\"-!=P`@"P@`&DA``"!$```"0$B$8?__]&M!=P`@1````DA``'9`2(2!_
M__O]T%W`"`+"``Z2$``(?__]$9`0`!O"7Z?WQ%\``((80`*$$"``"L!`$3,`
M``"R%F$H\EW`&1!O_0>!Z```$0```)(0`!V0$B$P?__[ZM!=P`@"\C_QDA``
M"'___/Z0$``:$&__[0$```!`!!\2`0````$````P:``'`0````$````!````
M`0````$````!````G>._0#L````1````+P`$'4``"W2N!>%PNA=BF)`2(/#Z
M7<`=D@>G\[`0(`#07<`(PE]``,)WI_>"$"``?__[R0$````"P@`*`0```,)*
M``"$&&`P@*```K!@/_^`H``!@F`__[`6``&P'B`!PE^G]\1?0`""&$`"A!`@
M``K`0`2Q/B``@<_@"`$```!`!![E`0````$````!`````0```)WCOT`1````
M`P```/)WJ(<O``0=0``+2ZX%X,R"$&*8D!(C&/1WJ(_07<`(E`>HAY(0`!CN
M7<`!]G>HEY`"(3#X=ZB?^G>HI\)=P`#"=Z?W@A`@`$`$'Z/4=Z?O0`0>(9`0
M(/\!````,&@`!@$````!`````0````$````!`````P``0((8?T"=XX`!(0``
M`)00(`(O``0=0``+*JX%X$B@%"*8DA`@`.!=P!"0$``8LA`__\)<``#"=Z?W
M@A`@`$`$'Z@!````AS9P,+0"/_^#/K`_A3!P,((&@`*""$`#@B!``K0F@`$F
MQH`RL!`__S<````Y``!`N!\_\+86X%""!R?_(P`@`?9=P!NC-'`'N`!`'K,N
M<!"4$"``DA``&D`$'Y"0$``8@*(__P)@`#,1````E!``$9(0`!Q`!![QD!``
M&("B/_\"0``BNSH@`("G8`<(8``CN@=_^+H'`!V`IT`<&F@`!Y00(`@0:``>
MM`:`&8"G0!P*8``:E!`@")(0`!M`!!]?D!``'8"B(``R3__YN@=__S$``$"P
M'C_P@@8G_[``0!ZZ)T`8L`=`&L)?I_?$7```@AA``H00(``*P$`3`0```('/
MX`@!````$0```)`2(4!___^*T%W`"+0&@!F`IH`9$F__S900(``0;__PL!`_
M_Y`2(3A___^!T%W`"!!O_\V4$``10`0>70$````!`````0````$```"=X[]0
M0`0?;P$```!`!!]=`0````+"``.P$``(\%H``('/X`@!````,&@`!@$````!
M`````0````$````!````G>._0#L````Y````+P`$'$``"K2N!>)PNA=A2+@7
M(ICZ7<`=D@>G\_A=P!R0$``=PE\``,)WI_>"$"``?__["0$````"P@`&MA``
M",)*``"`H&``$D``"@$```#"7Z?WQ%\``((80`*$$"``"L!`'K(0`!@0;_P3
MD>@`'4`$'U&0$``8M!``"$`$'TZ0$``;D`:`"$`$'U.0`B`"%P```!,```"6
M$N"@DA)@N-9=P`N8$``;E!``&-)=P`E`!!\ILA``",)?I_?$7P``@AA``H00
M(``*P$`$`0```!!O^_F1Z``=0`0>#P$````!````,&@`!`$````!`````0``
M`)WCO@`S````.0```,!WIM<O``0<0``*<ZX%X6RR%F"8.P```,!WIQ\+````
M"0```/)=P!D'````!0```,!WIN\#````-0```+06HIBX%R%0NA=@>(H185CT
M7<`:B!$A8(80X6C*7<`%A!"A<((087C(7<`$QEW``[8'IL.0$``9Q%W``I(0
M`!N@$``8PEW``?A=P!SZ7<`=\%Z``/!WI_>P$"``^'>FQ_IWIL_*=Z;_R'>G
M!\9WIP_$=Z<7?__ZJ<)WIN<"P@`&L!``",)*``"`H&``$D`!00$```!___]Y
M`0````+"`1:Z$``(0`0>]9`0`!V1*C`!0`0>^I`"(`'"3T``L!``"("@8``"
M0``0U`]``",```"X$``8HA1A@.)=P!&4"J#_D!``')(0`!%`!![,N@=@`<)/
M0`"X!R`"@*!@`!)/__C4#T```P```+@'IO>"$&$8$&@`!=!=P`'07R`(`L(`
M2;@'(`A___I]DA``&P+R/_NZ$``(PDH``("@8``B3__XT%\@"'__^QT!````
M@*(@`")/__/07R`(0`0>B)`0`!T"P@`XNA``")`0`!U`!![#(P```*00``A`
M!![`D!``&*0$@`BB%&"PI`2D!.)=P!&E/*``0`0>P9`0`!(9````$P```)@3
M(8B2$F&0V%W`#)H0`!B6$``1TEW`"900`!U`!!Z6N!``")(0(<!`!!XKD!``
M'("B/_\"0`"QD@>G)T`$'4:0$``<NI(@``)(`#7"!Z<O$0```!,```#47```
MD!(C&)(289C07<`(EA``'+`0(`#27<`)0`0=6)`"(3#"7Z?WQ%Z``((80`*$
M$"``"L!!,@$```"!S^`(`0```/I?IL<0:``(N!`@`(,H<`."`&?_@@!`'OI8
M?L@B]W_#NA`@`,)/0`"`H&``(D?_O[H0(`"0$``=?__ZT+@'(`&`HB```D__
M\H,_(`!`!!X[D!``'0KR/[6Z$``(@S\@`(,H<`."`&?_@@!`'OI8?L@J_W_N
MPD]``!!O_ZRZ$"``!0``/(((0`(%```0@*!``A)/_\H1````0`0>6/`'IT.`
MI@`(,D__Q1$```#"!Z<O@@AA_X"@8<`R3__`$0```$`$'F:0$``2L!``"!$`
M``"0$B&@DA``&W__^@#07<`(`L(`II(0``A___OTT%P```+"`**D$``(0`0>
M4)`0`!*`HB`#!$@`$[80``B1.B``$P```)(28:B0`C_\TEW`"4`$'E:0!(`(
M@*(@`!)(``F`IN`$`D``T8(&__N#.&``PDR``8"@8"\B0`#-]EP@"'___@H!
M````@*(@`!)(``D3````DA`@`$`$'@R0$``2@*(@`!)``(.V$``($P```!$`
M``"2$F$HD!(@\-)=P`DW````T%W`"+86X)!___KG]EW`&T`$'<4!````$P``
M`)@0``B2$F&PEA``$=)=P`F4$``<FA``&T`$'@20$``8$&@`%9(0(<!`!!V(
M`0```,("``"`H&`1$D``%`$```!`!!VRN@=@`?9SJ*\3````DA)AN)@0``C2
M7<`)FS]@`)80`!&4$``<0`0=\)`0`!B2$"'`0`0=A9`0`!B`HC__`D__Z@$`
M``!`!!R`D!``')`0`!E___J]DA``&'___HN0$``8$&__8P$```!`!!UG`0``
M`,("``"`H&`1`D__39('IR?Z7```0`0=8+`0(``#````$P```-A"``""$&,8
MDA)AP,)=P`&6$``<E!``'=)=P`E`!!RED`!A,!!O_TT!````$0```+@0(`"0
M$B'($&@`"-!=P`B#*'`#@@!G_X(`0![06'[@(L(`%Y`0(`V2$``;?__Y?;@'
M(`$"\C_W@S\@`,)*``"`H&```D__\X,_(`!`!!V-`0````KR/M:Z$``(@S\@
M`(,H<`."`&?_@@!`'M!8?N`*^C_ODA``&Y`0(`U`!!W).P```+H78="P$``(
M^EW`'1!O_M&4$"!3?__^2@$```#"7Z?WQ%Z``((80`*$$"``"L!`5`$````0
MA!UR@>@``!!O_V#D7```?__]Q@$````&^C]]H!``")("/_:4$"``D!``&T`$
M'7>D!Z>_E!`@!I(0`!)`!!S;D!``&X"B(`8":``'Q`>GOQ$```"0$B'8?__]
ME-!=P`C$!Z>_`P`0T((084.`H(`!$D__:!,```#$%*`$`P``$H(08$6`H(`!
M,D__8Q$```"4$"``D@0_SI`0`!M`!!U;N@>GQY00("B2$``=0`0<OY`0`!N`
MHB`H`F@`!QL````1````D!(AV'___7C07<`(&P```!,```#`+Z?OFA-@D)(2
M8>#:7<`-F!``'980`!'27<`)E!``'$`$'5R0$``8$&__<@$```#V7"`(`O;_
M-0$```!`!!UMD!``&X"B(`,(;_\P$P```)`"/_R2$F'HD`;`"$`$'7727<`)
M@*(@`!!O_RBE9$`;0`0<.`$````!````,&@`!0$````!`````0````$```"=
MX[]`.0```#L````O``0:0``(G*X%XA"X%R*8D`=BB/A=P!S07<`(L@>G\\)?
M``#"=Z?W@A`@``$```"0`<`(Q%H```,```"2$``9@A!@X-!=P`'""*"P?__X
MZL(HH+$"P@`&`0```,)*``"`H&`Q(D``:,)*(`%___N:`0```)`'8HC07<`(
MD`'`"!,```"T$``(T%H``)(28?"6$"`#TEW`"4`$'%>4$"`!`L(`$+80``C"
M6B`0%0```)02H?C07H``TEA``$`$'`[47<`*PE[@$-)80``#`!``Q`)@#("(
M@`$20`!H`0```)`'8HC07<`(D`'`"!,```"T$``(T%H``)(28@"6$"`#TEW`
M"4`$'#R4$"`!`L(`#;80``@1````DA``&9`2(*A___BUT%W`"`+"`$>4$``(
MPE[@$-!>@`!`!!OOTEA``)`'8HC07<`(D`'`"!,```#V6@``NA``")(28@B4
M$"`!TEW`"4`$'$20$``;%0```)(0``B4$J(0D!``&T`$&][47<`*$0```,)?
M0``[````D!(@F+H78%C07<`(DA``&?I=P!U___B4P"A@L0+"`!O0=T``0`0<
M:)(0(<"&`B`!A!H@$8(0(`"`H.`!@V<@`8"@``*$0"``@(A``A)``#@#````
MPE^G]\1?``""&$`"A!`@``K`0#`!````@<_@"`$```"`H&``(D__G)`'8H@0
M;_^7`0````,```""$&!@PEW``7___9S06$``"OH_XM!W0``1````D!(B&$`$
M'!;07<`(?__Z7`$````"P@`/L!``",)>X!"4$``(T%Z``$`$&Z726$``0`0;
M.Y`0`!@0;_^UD`=BB$`$'"_07H``$&__F9`'8H@#````Q%[@$((08&#07H``
MPEW``=)8@`#"6$``0`0;E-180``0;_^FD`=BB$`$&X`!````!0```-1?0`""
M$&!@A!"B(,)=P`&6$``(T%W``L)80`!`!!ONTEA```$````P:``#`0````$`
M``"=X[]0?__\9P$```""$"``@*``"(-^-`&&0"``+P`$&4``!]BN!>,`@(A`
M`S)```3"3@``@<_@"`$```"`H&```D___0$```!___K&D!``&'__^KP!````
M$P```)00(`22$F&(0`0:Y])=P`F`HB``$D__\0$````0;_AB@>@``)WCNZ`#
M````A@>E=X(08IBZ$#__+P`$&4``![JN!>*(XEW``8@&(#BX!B!XNS=P((00
M``/"7$``PG>G]X(0(`#"60``A`"@$(@!(`B`IP`$BS!P((((0!W*<+_P$F__
M^<)PO_B"$#__N`#B`(,P<"#$6.!HA@#@"("@P!SZ6.`XREC@",A8__B$&(`=
MA!B`!808@`2),+`?A`"``H00@`2$"(`!$F__\\1PX'CT7@``"1:@GCD;MGK*
M7Z5WB!$AF;@7(Z'R7B`0A3:P&SLCQN_V7B`(ARZP!8H!0`3D7B`8AA#``KH7
M8-S>7B`@FR[P'H8(P`&5+K`>A!Y`$M9?I8>*`4`/A`B`&^Q?I<^?-O`"A@#`
M!<I?I7^>$T`/A!B`$NA?I=>:"\`!A@#``H0>0`V*`4`$A`B`&H8(P`&$&(`9
MB@%`$IDH\`6$`(`%BS#P&Y8"P`28$P`%E@+`&8LVL`*$`(`,BA*`!=1?I8^$
M"(`!GAO`!9@)0`&>"\`#E`*`!)X;P`V4`H`-DRBP!9X#P`N1*/`>K`6`!)<P
ML!NH!0`$FS#P`I820`N:$@`-A@/`"]!?I9^>&4`-RE^EEX8(P`&>"\`"E@M`
M`8H!0`2>&\`,D@%`#)X#P`J+*/`%D`(`!)4P\!N9*+`>E!%`"HLPL`*$`\`*
MF!,`!80(@`&*&T`,E`L``=I?I:>*"4`#G@(`"XH90`N:`T`$B@%`"9H#0`J3
M*/`>AS#P`I$PL!N2$D`#ERBP!88;``F&",`"EA+`"(88P`J*`4`+U%^EMX8`
MP`^*"4`!UE^EKY\HL!Z8"D`!A3"P`I8"P`2>$\`"E`*`!*$Q<!N$&D`/D2EP
M!80(@`60$@`0A!B`#.!?I;^&`,`(A`"`#88(P`&2"\`!FREP'I8"P`S87Z7'
MBS%P`J`$``2G,/`;FA-`!9$H\`6*&\`-D!(`$XH)0`.$`(`(E`*`"80(@`&0
M"T`!DAE`"9X$``B*`D`+F`,`!*$HL`63,+`;ERCP'I(4``G@7Z7?AS#P`HH!
M0`F6$L`#B@E``9H;0`N&"L`!F@M``I@#``.:&T`(T%^EYZ`$``2G,7`;E`-`
M"I,I<`60`@`$FRBP'I(20!.%,+`"E`*`"=)?I>^:$T`"A`J``90:P`V6"T`!
ME`J`!:P%@`N&&H`#U%^E]Y("0`2K,+`;A@#`#Y\I<!Z4`H`$BS%P`J<HL`6>
M$\`%IA3`%8H;0`^&`,`3B@E``JLHL!Z&",`!A3"P`HH90`O67Z7_JA5``IH+
MP`&O*/`%A!O`%=Y?IA>G,/`;B@%`#-A?I@>F%<`3A`B``Z@%``V$&(`-VE^F
M#XH!0!.6`L`$B@E``9@#``2O*7`%F@-`!(@`@!:F#4`!K3%P&Z`$`!.L%<`6
MG@/`'(4H\!Z(`0`6[%^F)X<P\`*("0`!A!"``ZP%@!SL=Z5GK2EP'H8=0`*&
M",`%J@B``8LQ<`*0`@`5K!6`!888P!/F7Z8?A!B`%H8`P!2$"(`$B@V``:H8
M@!62`D`%H`5`$.I?IC>F!,`<KRDP!:DQ,!NJ!4`<ZG>E7ZLI,!ZH%<`4[E^F
M+XDQ,`*&`,`4A@C``:H50`2I*/`%K!V`%84P\!N(#8`#[%^F/X05``+H7Z9'
MB!D`!80$``*@#4`!J`4`'.AWI5>$"(`!J2CP'I0"@!"',/`"D`$`"*@5``.N
M!<`<JAU`%*P%@!R+*+`%A@U``HDPL!N&&,`0X%^F5X@10`22`,`)H`0`'(@"
M``3@=Z5/T%^F3Z$HL!Z("0`!A3"P`HH-``&@%``"J@(`'*@=`!"6`L`%D2DP
M!80-``2',3`;A!B`!<I?IE^&$@`#T%^F9Y0`@`J&`D`#J`%`'(8(P`&0`@`<
MT'>E1XLH\`62#``!A3#P&Y@#``F1*3`>A!%``HDQ,`*$`H`"U%^F;Y`2``2(
M"(`!H!P`"(H*``&$#``#H`*`'(08@`F:`T`%TE^F=Y4I,`66`(`+A3$P&Y("
M0!S2=Z4_A!*``M1?IG^6`L`"E`*`'(0*P`'4=Z4WUE^FAY4H\!Z',/`"AA*`
M`Y8"P!S6=Z4OD!H``Y((P`&6"@`$G@/`"94HL`6*&L`%D3"P&XH!0`S87Z:/
MEA*`")`#`!S87Z:7E@K``8H!0`N8`P`<V'>E)Y8)0`&9*3`>B3$P`I4R\!N(
M$P`$V%^FGX88P`28`P`<A@C``MAWI1^8"0`!AAC`":8$P`R&`,`-F2KP!9H3
M``K87Z:GF@M``9(#`!S87Z:OA@#`#9H(P`&X`P`<^'>E%[DHL!Z%,+`"F3-P
M&X07``+X7Z:WB!D``HH9``7(7Z5GE`<`'8H)0`&X"(`!U'>E#YX!0`_*7Z:_
MN`$`'(DK<`6*`4`=RG>E9X@1``S*7Z;'B`D``9X#P`28`4`=V'>E!X@+P`&+
M*O`>V%^FSY<R\`**$4`+E@,`'=9WI/^$&(`%F`E``888@`.8!<`,ES$P&X8(
MP`&%*3`%I@3``\9?IM>$$(`+K@#`'<9?IM^$"(`!I@3``H8`P!W&=Z3WA`S`
M`8<K<!Z;,W`"ES"P&X80P`W:7Z;GBAE``YX90`^:`T`=RE^E7YX+P`':=Z3O
MF@C``;@'``_>7Z;OF@%`#8LHL`6>`\`=WG>E7XH10`O>7Z;WB@E``;@'``66
M`\`=B@\``=9WI.>?*3`>B3$P`I4Q<!N>$\`$R%^F_X88P`^6"\`!B`$`':88
MP!/(=Z3?B`S``98%@`N'*7`%F`,`!,A?IP>&$,`*I@$`'<A?IP^&",`!F`,`
M`ZP!`!V&"P`!B2BP'H4PL`*5,/`;B!$``L1?IQ>>&\`$A`"`';@;P!S>7Z57
MQ'>DUX0)``&X#P`!G@/``IH#0!R%*/`%N!"`"L1?IQ^X#P`!A`"`'<1WI5>X
M`T`<Q%^G)YH/``&$`(`=Q'>DSX4I<!Z+,7`"A!"`!<I?IR^(&0`"F!D`#)0!
M0!W4=Z3'B@L``90(@`&6`L`%RE^G-Y0%0`J9,W`;B2MP!:H!0!W*7Z<_B!$`
M#)@!0!V("0`!V'>DOXLH\!Z6`L`$AS#P`H@*P`&9,3`;BA%``\9?IT>$&(`%
MA!B`'(8`P!WX7Z5/A`B``<9WI+>&"4`!A`/``MY?IT^X!P`#ARDP!;H#P!WZ
M=Z5/AA#`#+LK<!Z&",`!FS-P`IX`@`.Z%T`-!3*8L(H90!V$$*'6EAE`"Y@/
M0`&:"L`!F`4`#)0"@`W:7Z=7A@O``9<P\!N+*/`%J`-``MI?IU^*$4`+F@-`
M`MIWI*^*"4`!FRDP'I0"@`6),3`"B@J``9<Q<!N:$T`$R%^G9[H?0`VZ'T`/
MB`$``MY?I4>Z#T`!R'>DIX@+0`&Z!P`=^%^G;YX#P`2)*7`%N`<``OAWI4>(
M$0`+^%^G=X@)``&Z!T`$E@<``H@/0`'6=Z2?N2CP'H<P\`*7,3`;N!<``\9?
MIW^:&T`<E!M`"H8`P`+&=Z27F@J``88/``&8`P`-VE^GAZ`$``.'*3`%F@-`
M`MIWI(^&$,`+VE^GCX8(P`&8`P`#E`-``H8+``'4=Z2'FREP'M1?I3>+,7`"
MFA-`!<I?IY>X'P`-NA\`'?A?I3^6`4`"N@]``8H+0`'6=Z1_N@/`'=Y?IY^X
M!P`%ES#P&XLH\`6>`\`"WG>E/]Y?IZ>*$4`+B@E``=9?I2^>`\`"WG>D=[H'
M0`6?*3`>B@]``8DQ,`*>$\`$R%^GKYH;0`^8&T`,B`$``LAWI&^:"P`!B`O`
M`:`$``W:7Z>WE`*`!)DQ<!N)*7`%F@-``MIWI3?:7Z>_B!$`#(@)``&:`T`"
MVG>D9X@$``2;*/`>F`D``8<P\`*&$T`#VE^GQYX;P`.>&\`=H`-``IX+P`&:
M",`!N`<`#]Y?I\^:`L`-NRLP!9<S,!NZ%T`+E@/``M9WI2_>7Z?7N@]``;H'
M`!V>`\`"WG>D7[@/0`&?*7`>BS%P`HH3P`7>7Z??AAC`!98#P`*(&,`$UG>D
M5X<O,`6("0`!ES<P&X@"@`2&$,`+UE^GYYX)0`&&",`!G@(`#Y0"P`+67Z?O
MA@$``X@(P`&6`L`"A2LP'I`"P!J9,S`"ES$P&Y@0@`R$&4`,RE^E)[0+``&$
M&(`=A`B``;H!0!JU*3`%F@-``HLO,!Z$%H`+N3<P`H0(@`&*$4`<F@-``H0;
M``6X"T`!AAB``[0)0`&9+S`%A`C``8<W,!N>`\`"A2DP'H83``/87Z4?B3$P
M`H8(P`&($(`$A@/``YX(P`&*&4`$ES/P&YH90`V%+S`>F@M``8LK\`6T`P`:
MN3<P`HH10`N$$(`<F`D``8H)0`&(&0`"N@=`#889``.Z!T`%A@C``8H/0`&T
M!H`#QE^E%YLI<`6X"(`!B3%P&[@`P!R($T`$F`)`#(<K\!Z("0`!GS/P`K0&
M@`2($,`/A@Z``808@`26$#__FS#P&X08@!V?*/`%A`B``;LI<!Z>$\`-BS%P
M`H0#``*Z%T`%A`"`#Y@9`!V$"(`!F!L`&O1?I6>>#T`!FRBP!9@+``&+,+`;
MN`<`#(H30`6T!H`/F2CP'HH'``6',/`"@@E``<I?I0^7,O`@AA,``]A?I0>:
M",`+NA=``YX+P`VZ#T`"B`D`"[H70`^Y*+`>B`%`!)\H<`6(!T`$A3"P`HH#
M``V[,'`;N!<``KH3P!V&$,`<N@$`'8@/``N$#T`+F@M`!/I?I/^?*'`>A@C`
M`880P`V8!T`$VE^D[[LHL`6T`,`:AS"P&X,P<`*&%T`#GA/``88&@`."",`+
MN!<`#X8+P`NZ#P`"B`D``ZX%P`.Y*+`>B!=`!+LH<`6*`0`%B3!P&X4PL`*(
M%T`$N!<``H@!0`2$"0`+GA/`'(@/``N*"\`!WE^E7[LHL`6&",`$M2AP'H81
M0`/*7Z3W@S!P`I@`P`R',+`;M!:``8870`.X%P`:A@,``[H/``+87Z2_@@C`
M"X8.@`N*`4`$B`D``X@70`2:`T`#NRAP!:X!`!>),'`;N2BP'H@70`2%,+`"
MB`7`!+@7``*$"0`+M!:`'(@/``NZ#H`!A@C`!(870`.>`\`$NRBP!8H`P`6'
M,+`;M2AP'H870`.#,'`"A@%``[06@`&"",`+N!<`&H8.@`N*#P`"B`D``[LH
M<`6($4`$RE^DY[DHL!Z:`0`-B3!P&XH!0`.%,+`"B!=`!+@7``*(`T`$A`D`
M"[06@!R(#P`+N@Z``;4HL`6&",`$FRAP'H870`/Z7Z3?@S!P`IX`P`^',+`;
MFA-``886@`.X%P`-A@/``[@/``*"",`+A@M`"[H'0`2("0`#B!<`!*8$P`.Y
M*'`%B@$`!8DP<!NU*+`>B!<`!(4PL`*(`4`$N2AP'K06@`*:$T`:A`D`"X@.
M@`N*"T`!A@C`!*P%@`2#,'`"AA%``[H`P!VX%P`!BRBP!;06@!R',+`;AA%`
M`XH.@`*U*+`>A@=``X4PL`*"",`+A@\`"[06@`*("0`#N!<`&KLH<`6($4`$
MRE^DUY\H<!ZF`0`3B3!P&XH!0`.(%T`$N@\``8,P<`*(!,`$A`D`"X@.@`N&
M",`$GA/``8870`/Z7Z57M!:`#[DHL`6L`,`6AS"P&[H'0`2;*+`>AA<``X8%
M@`.X#H`"A3"P`H((P`N&"\`+FA-``H@)``.>$\`-M2AP!8@7``3X7Z3/B@$`
M!8DP<!NX!P`#B!:`!(@!0`2*"\`!GRAP'H0)``NU*+`%B`M`"X,P<`*&",`$
MAA%``YX3P`'*7Z3'N@#`'9H30`^',+`;B@%`!(86@`.U*+`>A@=``X((P`N&
M"\`+N@M``H@)``.(%T`$J@5``[LH<`6X`0`<B3!P&X4PL`*(%T`$M!:``H@'
M``2$"0`+GA/`&H@.@`NZ"\`!N2BP!88(P`2?*'`>AA=``XH`P`6Z`P`$AS"P
M&X,P<`*&%P`#GA/``88!0`."",`+M!:`#X8+P`N*#H`"M2BP'H@)``.%,+`"
MB!%`!,I?I+>T%H`"J@$`%80.@`N>$\`:N2AP!8H!0`.),'`;A@C``ILH<!Z(
M%P`$N`O``8@%0`2&%P`#^%^E3X@)``N#,'`"N@#`'9X'``*:$T`!N2DP!8<Q
M,!N&%P`#N!:`#88'0`.X#P`$@@C`"X8+0`N$"(`#J`4``[LH<`6X%P`"A3!P
M&XH'``6Z%T`"A2DP'KH/0`N),3`"B@%`'800@`2("4`+N@B`"YH30`*X"T`!
MA@C`'=I?I*^U*3`%AA<``X8`P`^Z`T`=GRAP'H,P<`*Y,3`;GA/``808@`^X
M%H`<A!B`!;@/``O*7Z2GA`B`"X8`P!RH`(`4@@C`"X4I,!ZX"\`+B3$P`HH!
M0!RU,'`;A!"`!+DH<`6>&\`"AAO``[@7`!J&",`+N`\`"[H`P!W&7Z5'M`B`
M"Z@%`!RX#0`+M`#`&H<H<!Z#,'`"GR\P!880P`&)-S`;A!B``X@3P`2H&(`4
MQ%^DGX@)``NH#0`+B`=`!+H(P`NZ`(`=@@D`"X4O,!Z*`4`4N3<P`ILP<!N$
M$(`<GRAP!888P`*(&,`$GA/`#88)``N>"\`+M`:``\9?I)>*`4`/G@B`"[@)
M0`N;+S`%A@#`#Y\H<!Z#,'`"B3<P&YX3P`&$&(`/B!-`!(@)``N*&(`%Q%^D
MCX@&@`2:"\`+@@D`"YH`@`V%+S`>B@E`"[DW,`*Z!T`%M2AP!800@!R9,'`;
MGAO``HH6@`R(&\`$B@E`"X@)``NZ!T`%A@#`!,A?I(>X#T`+M`B`"YDO,`6T
M`0`:BS<P&XDH<!Z*$P`%@S!P`HH)0`N($0`!A@#`!808@`2"",`+NAB`'<1?
MI'^*"0`+GRAP!;H/0`N9,'`;B@"`!84O,!Z:`T`=N3<P`KH3P`RZ#T`+A!"`
M'(@9``*:`T`=N@M`"X89``.?+W`%A@C`"XDW<!NT!H`#B!/`!+@(@`O>7Z4_
MARAP'H@)``N#,'`"B`:`!((0P`&T"0`+A!B``;@#P!R:&(`-Q%^D=YX(0`N9
M-K`;F@M`"X<NL`6>`(`/AA#`#(H!0`V%+W`>A@C`"[LW<`**`4`#A!"`';H)
M0`N'+W`%@AA``ILW<!N"&$`$@@A`"X@(@`NX!P`!@A#`#<9?I&^;+K`>@@A`
M"[4VL`*X!P`!@A-`&H@`P`2$&(`!A@\`"X08@`7*7Z4WF@A`"[4H\`6$"(`+
MF@%`#9X#P`*+,/`;A2]P'K06@`6+-W`"N@Z`"X00@`6Z`\`=@AA``HH/0`NU
M*7`%@AA`')\Q<!N""$`+@@$``8@6@`_T7Z1GN`B`"X@)``NX!H`<B`!`!+4H
M\!Z',/`"@A:``X8)``N$&(`!M`A`"X08@!V@!``:NRCP!80(@`NU,/`;F@-`
M`H4I<!ZZ%T`:BS%P`KH/0`N$$(`%N@-`'8(80`**#T`+@AA`!)X(@`NU,7`;
M@@A`"XDI<`6X!P`!PE^E+X@1`!J>`$`/B`D`"X,H\!ZT!P`$AS#P`K@.@`N;
M-S`;@A!``\9?I%^$&(`!B`A`"X08@!V(`,`$NR\P!80(@`N'*7`>H`0``HLQ
M<`*$%T`-AA#`!80(@`N8!``"@AA``[H+``N"&$`:BR]P!8((0`N%-W`;M`/`
M`8(10`+$7Z17F@C`"X((0`N:`(`-M`:``84O,!Z*#H`+N3<P`I\Q<!N$$(`<
MAAC``H((@`N&&,`,E`*``;DI<`6&",`+@R]P'H@!``.[-W`"AA<`#X8(P`N"
M$$`=G@$``X08@`&X"\`+A!B`&H<O,`6$"(`+M3<P&[H#0`*&$,`:B`A`"X8(
MP`NT`@`$A2EP'KH'0`/&7B`@BS%P`H00@`6*#T`+@AA``H8`@`."&$`/A@C`
M"\9V("")*7`%@@A`"Y\Q<!N"`H`!AR\P'H@1``^Y-S`"@@!`!(80P!R""$`+
MA!B``X@`0!N$&(`=B`D`"\AV(`B$"(`+A@#`$KLH<`6(!H`"@S!P&X0(P`O$
M=B`8A2EP'H870`&#,7`"A@$``X(0@`&$",`+Q'8``+(`0!F6#D`+UG8@$,)?
MI_?$7$``@AA``H00(``*P$`$`0```('/X`@!````0`05;P$````!````,&@`
M!`$````!`````0```)WCOU!`!!:7D!`@@`,9T4C`<B`H@A!C`<)R```#._-J
MP'(@,((08XG"<B`(`R8NM\`B('B"$&#^PG(@$`,$#)6"$&!VPG(@&`,P]+B"
M$&'PPG(@((''X`B1Z``(`0```)WCOU#*7B`H@RZP`X@0(`#&7B`PA3:P';@&
M(#B"`$`%^@8@>(,P8`#"=B`H@*%``8EG,`&`IV``A@#`!(0`@`,"2``7Q'8@
M,((0($"V$``:D3]@`+H@0!V2$``9@*=`&K=DP!V0!P`(NS[@`$`$%264$``=
MP@8@>((&P`&`H&!`$D@`(L(F('BT)H`;D!``&+4^H`!___G;L@9`'8"FH#\$
M0``5NA``&;0&O\"W-J`&N@;@`;LW8`"[+W`&N@9`'9(0`!F4$"!`D!``'$`$
M%0VR!F!`?__YRY`0`!B`ID`=$F__^9(0`!FW+N`&M":`&[4^H`"4$``:DA``
M'4`$%0&0$``<]"8@>('/X`@!````,&@`!`$````!`````0```)WCOU#V7F`H
MM`9@.(00/X#X7F`PNA``&)(0(`"5-O`#E`J@/X,ZH`"4`J`!D3J@`,0N@`&`
MHJ`X!$@`5I`&@`B"$"!`E"!`"D`$%3:5.J``?__YI)`0`!F4$"`XDA`@`$`$
M%3"0$``:@S<P&,(N8'"0$``9@S<P$,(N8'&P$``9@S;P&/@N8'/"+F!TA3;P
M$(,V\`C$+F!UPBY@=KDW,`CV+F!W?__YC_@N8'+"7D``@S!P&,(O0`#"7D``
M@S!P$,(O8`'"7D``@S!P",(O8`+"7D``PB]@`\)>8`B#,'`8PB]@!,)>8`B#
M,'`0PB]@!<)>8`B#,'`(PB]@!L)>8`C"+V`'PEY@$(,P<!C"+V`(PEY@$(,P
M<!#"+V`)PEY@$(,P<`C"+V`*PEY@$,(O8`O"7F`8@S!P&,(O8`S"7F`8@S!P
M$,(O8`W"7F`8@S!P",(O8`["7F`8PB]@#\)>8""#,'`8PB]@$,)>8""#,'`0
MPB]@$<)>8""#,'`(PB]@$L)>8"#"+V`3$(043H'H``""$"`XE"!`"D`$%.*5
M.J``$&__LX,W,!@P:``&`0````$````!`````0````$```"=X[]0`P```"\`
M!!)```$%K@7CM((08##"7<`!Q`A``("@H``20``E!0```(80(`&$$*)0Q%W`
M`@+`@`?&*$```P```((08K#"7<`!0`03]=!80``[````.0```+H78CBX%R)H
M^EW`'?A=P!RZ!V`(@*=`'!I@``D#````PE]``)_`0`"Z!V`(@*=`'"IO__W"
M7T```P```((08NC"7<`!`L!`!C$```"P%B*@\%W`&!"$%3R!Z```@<_@"`$`
M``"=X[]0`P```"\`!!)```#5K@7B]((08"#"7<`!Q`A``("@H``20``>!0``
M`(80(`&$$**`Q%W``@+`@`G&*$``$P```!$```"2$F`HD!(BH-)=P`E`!!1;
MT%W`"#L````#````NA=BR((08N#Z7<`=[EW``;H'8`B`I<`="&``"`$```#"
M7?_XG\!``*X%__B`I<`=.&___<)=__B!S^`(`0```#!H``<!`````0````$`
M```!`````0````$```"=X[\P.P```)0'I]_R=Z?G+P`$$D```*.N!>(LNA=B
MF)('I^?T=Z??^EW`'9`'J'_P)ZA_PE]``,)WI_>"$"``0`04D`$````5````
M$0```)02HL"0$B,`U%W`"M!=P`A`!!2HDA``"@,```""$&)XPEW``<((0`"`
MH&```D@`63D````W````N!<@:/A=P!R0!N*(T%W`")`!P`CV!ZA_Q%H``)`&
MX`2V!N`#P@BC\I$Z(`"1*C`#@A!@`T`$%%'"**/R`P```,9?I^<)````@A!B
MN`4```#*!ZA_]%W``0,```"($2!@A!"B*((08C#R6,``R%W`!)80``B`H6`!
MQ%W``O1>@`#"7<`!T'$``/)R``#$<B`(]'(@$`1``%7"<B`8A@#@"(H!8`.$
M$"`$R%C``(,XH`"$`*`!@RAP`X8`X`B`H(`%$D__^LARP`$3````T%\``)@0
M(`"%.*``DA)@2(4HL`/27<`)E3[@`$`$%(S`<L`"@*(@``)``"8!````T%\`
M`$`$$\ZV!Z?SL!``"!$```"0$B#HDA``&W__\*#07<`(`L(`'P,```!`!!.<
MT%\``$`$%%H!````PE^G]\1?0`""&$`"A!`@``K`0"P!````@<_@"`$```!`
M!!/8N!<@:/A=P!P"P@`CT'<``$`$$VLW````D`;BB-!=P`B0`<`(PEH``!!O
M_Z'`*&AV0`04H]!?```0;__;T%\``((08%CT7<`!T%Z``"+"``81````?__X
M.@$````0;__<`0```)(0`!N0$B"8?__P=-!=P`@"^C_6T':``'__^#`!````
M$&__T@$````0;_^XA!`@!$`$$O*0$"`!0`03F`$````!````,&@`!0$````!
M`````0````$```"!P^`(K@/`%S!H``8!`````0````$````!`````0```)WC
MOU!___[O`0```(''X`B!Z````````'!S7W-T<FEN9W,@;6ES<VEN9PH`````
M`"5S+R5S````+@````````!$>6YA3&]A9&5R.CIB;V]T7T1Y;F%,;V%D97(`
M`````$EN=&5R;F%L<SHZ4$%2.CI"3T]4`````%!!4E]414U0``````````!0
M05)?4%)/1TY!344`````.@`````````O`````````"5S)7,E<P``4$523#5,
M24(``````````%!%4DQ,24(`4$523#5/4%0``````````%!%4DQ)3P``4$%2
M7TE.251)04Q)6D5$`%!!4E]34$%73D5$``````!005)?0TQ%04X`````````
M4$%27T1%0E5'`````````%!!4E]#04-(10````````!005)?1TQ/0D%,7T1%
M0E5'``````````!005)?1TQ/0D%,7U1-4$1)4@````````!005)?5$U01$E2
M````````4$%27T=,3T)!3%]414U0`%!!4E]'3$]"04Q?0TQ%04X`````````
M`#$`````````;'-E96L@9F%I;&5D`````')E860@9F%I;&5D``````!,1%],
M24)205)97U!!5$@`55-%4@````!365-414T``"]T;7`O````5$U01$E2``!4
M14U01$E2`%1%35``````5$U0``````!54T523D%-10``````````)3`R>```
M``!P87(M`````"5S)7,E<R5S```````````E<SH@8W)E871I;VX@;V8@<')I
M=F%T92!S=6)D:7)E8W1O<GD@)7,@9F%I;&5D("AE<G)N;STE:2D*````````
M)7,Z('!R:79A=&4@<W5B9&ER96-T;W)Y("5S(&ES('5N<V%F92`H<&QE87-E
M(')E;6]V92!I="!A;F0@<F5T<GD@>6]U<B!O<&5R871I;VXI"@```````%!!
M5$@`````<&%R;``````N<&%R`````'-H;W)T(')E860````````E<R5S8V%C
M:&4M)7,E<P``)7,E<W1E;7`M)74E<P```"5S)7-T96UP+25U+25U)7,`````
M`````!@`````````<&5R;``````P`````````%!!4CHZ4&%C:V5R.CI615)3
M24].`````#$N,#8S````56YA8FQE('1O(&-R96%T92!C86-H92!D:7)E8W1O
M<GD``````````"5S.B!C<F5A=&EO;B!O9B!P<FEV871E(&-A8VAE('-U8F1I
M<F5C=&]R>2`E<R!F86EL960@*&5R<FYO/25I*0H``````````"UE````````
M+2T```````!E=F%L("=E>&5C("]U<W(O8FEN+W!E<FP@("U3("0P("1[,2LB
M)$`B?2<*:68@,#L@(R!N;W0@<G5N;FEN9R!U;F1E<B!S;VUE('-H96QL"G!A
M8VMA9V4@7U]P87)?<&P["FUY("@D4$%27TU!1TE#+"`D1DE,15]O9F9S971?
M<VEZ92P@)&-A8VAE7VYA;65?<VEZ92D["D)%1TE.('L*)%!!4E]-04=)0R`]
M(")<;E!!4BYP;5QN(CL*)$9)3$5?;V9F<V5T7W-I>F4@/2`T.R`@(",@<&%C
M:R@B3B(I"B1C86-H95]N86UE7W-I>F4@/2`T,#L*)%!+6DE07TU!1TE#(#T@
M(E!+7#`P,UPP,#0B.PI]"G-U8B!F:6YD7W!A<E]M86=I8PI["FUY("@D9F@I
M(#T@0%\["FUY("1C:'5N:U]S:7IE(#T@-C0@*B`Q,#(T.PIM>2`D8G5F.PIM
M>2`D<VEZ92`]("US("1F:#L*;7D@)'!O<R`]("@D<VEZ92TQ*2`M("@D<VEZ
M92TQ*2`E("1C:'5N:U]S:7IE.R`@("`@(",@3D]413H@)'!O<R!I<R!A(&UU
M;'1I<&QE(&]F("1C:'5N:U]S:7IE"G=H:6QE("@D<&]S(#X](#`I('L*<V5E
M:R`D9F@L("1P;W,L(#`["G)E860@)&9H+"`D8G5F+"`D8VAU;FM?<VEZ92`K
M(&QE;F=T:"@D4$%27TU!1TE#*3L*:68@*"AM>2`D:2`](')I;F1E>"@D8G5F
M+"`D4$%27TU!1TE#*2D@/CT@,"D@>PIR971U<FX@)'!O<R`K("1I.PI]"B1P
M;W,@+3T@)&-H=6YK7W-I>F4["GT*<F5T=7)N("TQ.PI]"FUY("@D<&%R7W1E
M;7`L("1P<F]G;F%M92P@0'1M<&9I;&4L("5-;V1U;&5#86-H92D["D5.1"![
M(&EF("@D14Y6>U!!4E]#3$5!3GTI('L*<F5Q=6ER92!&:6QE.CI496UP.PIR
M97%U:7)E($9I;&4Z.D)A<V5N86UE.PIR97%U:7)E($9I;&4Z.E-P96,["FUY
M("1T;W!D:7(@/2!&:6QE.CI"87-E;F%M93HZ9&ER;F%M92@D<&%R7W1E;7`I
M.PIO=71S*'%Q6U)E;6]V:6YG(&9I;&5S(&EN("(D<&%R7W1E;7`B72D["D9I
M;&4Z.D9I;F0Z.F9I;F1D97!T:"AS=6(@>R`H("UD("D@/R!R;61I<B`Z('5N
M;&EN:R!]+"`D<&%R7W1E;7`I.PIR;61I<B`D<&%R7W1E;7`["FEF("@M9"`D
M<&%R7W1E;7`@)B8@)%Y/(&YE("=-4U=I;C,R)RD@>PIM>2`D=&UP(#T@;F5W
M($9I;&4Z.E1E;7`H"E1%35!,051%(#T^("=T;7!86%A86"<L"D1)4B`]/B!&
M:6QE.CI"87-E;F%M93HZ9&ER;F%M92@D=&]P9&ER*2P*4U5&1DE8(#T^("<N
M8VUD)RP*54Y,24Y+(#T^(#`L"BD["FUY("1F:6QE;F%M92`]("1T;7`M/F9I
M;&5N86UE.PIP<FEN="`D=&UP(#P\(BXN+B(["G@],3L@=VAI;&4@6R!<)'@@
M+6QT(#$P(%T[(&1O"G)M("UR9B`G)'!A<E]T96UP)PII9B!;(%PA("UD("<D
M<&%R7W1E;7`G(%T[('1H96X*8G)E86L*9FD*<VQE97`@,0IX/6!E>'!R(%PD
M>"`K(#%@"F1O;F4*<FT@)R1F:6QE;F%M92<*+BXN"F-L;W-E("1T;7`["F-H
M;6]D(#`W,#`L("1F:6QE;F%M93L*;7D@)&-M9"`]("(D9FEL96YA;64@/B]D
M978O;G5L;"`R/B8Q("8B.PIS>7-T96TH)&-M9"D["F]U=',H<7%;4W!A=VYE
M9"!B86-K9W)O=6YD('!R;V-E<W,@=&\@<&5R9F]R;2!C;&5A;G5P.B`D9FEL
M96YA;65=*3L*?0I]('T*0D5'24X@>PI);G1E<FYA;',Z.E!!4CHZ0D]/5"@I
M(&EF(&1E9FEN960@)DEN=&5R;F%L<SHZ4$%2.CI"3T]4.PIE=F%L('L*7W!A
M<E]I;FET7V5N=B@I.PIM>2`D<75I970@/2`A)$5.5GM005)?1$5"54=].PIM
M>2`E<WES(#T@*`IP871H7W-E<"`@("`]/B`H)%Y/(#U^("]>35-7:6XO(#\@
M)SLG(#H@)SHG*2P*7V5X92`@("`@("`@/3X@*"1>3R`]?B`O7B@_.DU35VEN
M?$]3,GQC>6=W:6XI+R`_("<N97AE)R`Z("<G*2P*7V1E;&EM("`@("`@/3X@
M*"1>3R`]?B`O7DU35VEN?$]3,B\@/R`G7%PG(#H@)R\G*2P**3L*7W-E=%]P
M<F]G;F%M92@I.PIO=71S*'%Q6UPD<')O9VYA;64@/2`B)'!R;V=N86UE(ETI
M.PI?<V5T7W!A<E]T96UP*"D["F]U=',H<7%;7"1%3E9[4$%27U1%35!](#T@
M(B1%3E9[4$%27U1%35!](ETI.PIM>2`H)'-T87)T7W!O<RP@)'-T87)T7V]F
M7T9)3$5?<V5C=&EO;BD["DU!1TE#.B!["FQO8V%L("1324=[7U]705).7U]]
M(#T@<W5B('M].PIU;FQE<W,@*&]P96X@7T9(+"`G/#IR87<G+"`D<')O9VYA
M;64I('L*;W5T<RAQ<5M#86XG="!R96%D(&9R;VT@9FEL92`B)'!R;V=N86UE
M(ETI.R`@(R!D;VXG="!U<V4@)"$@:&5R92!A<R!I="!R97%U:7)E<R!%<G)N
M;RYP;0IL87-T($U!1TE#.PI]"FUY("1M86=I8U]P;W,@/2!F:6YD7W!A<E]M
M86=I8R@J7T9(*3L*:68@*"1M86=I8U]P;W,@/"`P*2!["F]U=',H<7%;0V%N
M)W0@9FEN9"!M86=I8R!S=')I;F<@(B1005)?34%'24,B(&EN(&9I;&4@(B1P
M<F]G;F%M92)=*3L*;&%S="!-04=)0SL*?0IO=71S*")&;W5N9"!005(@;6%G
M:6,@870@<&]S:71I;VX@)&UA9VEC7W!O<R(I.PIS965K(%]&2"P@)&UA9VEC
M7W!O<R`M("1&24Q%7V]F9G-E=%]S:7IE("T@;&5N9W1H*")<,$-!0TA%(BDL
M(#`["G)E860@7T9(+"`D8G5F+"!L96YG=&@H(EPP0T%#2$4B*3L*:68@*"1B
M=68@;F4@(EPP0T%#2$4B("8F("1B=68@;F4@(EPP0TQ%04XB*2!["F]U=',H
M(DYO(&-A8VAE(&UA<FME<B!F;W5N9"(I.PIL87-T($U!1TE#.PI]"F]U=',H
M<7%;0V%C:&4@;6%R:V5R("(D8G5F(B!F;W5N9%TI.PIS965K(%]&2"P@)&UA
M9VEC7W!O<R`M("1&24Q%7V]F9G-E=%]S:7IE+"`P.PIR96%D(%]&2"P@)&)U
M9BP@)$9)3$5?;V9F<V5T7W-I>F4["FUY("1O9F9S970@/2!U;G!A8VLH(DXB
M+"`D8G5F*3L*;W5T<R@B3V9F<V5T(&9R;VT@<W1A<G0@;V8@1DE,17,@:7,@
M)&]F9G-E="(I.PIS965K(%]&2"P@)&UA9VEC7W!O<R`M("1&24Q%7V]F9G-E
M=%]S:7IE("T@)&]F9G-E="P@,#L*)'-T87)T7V]F7T9)3$5?<V5C=&EO;B`]
M('1E;&P@7T9(.PIM>2`E<F5Q=6ER95]L:7-T.PIR96%D(%]&2"P@)&)U9BP@
M-#L@("`@("`@("`@("`@("`@("`@("`@("`@("`C(')E860@=&AE(&9I<G-T
M(")&24Q%(@IW:&EL92`H)&)U9B!E<2`B1DE,12(I('L*<F5A9"!?1D@L("1B
M=68L(#0["G)E860@7T9(+"`D8G5F+"!U;G!A8VLH(DXB+"`D8G5F*3L*;7D@
M)&9U;&QN86UE(#T@)&)U9CL*;W5T<RAQ<5M5;G!A8VMI;F<@1DE,12`B)&9U
M;&QN86UE(BXN+ETI.PIM>2`D8W)C(#T@*"`D9G5L;&YA;64@/7X@<WQ>*%MA
M+69<9%U[.'TI+WQ\("D@/R`D,2`Z('5N9&5F.PIM>2`H)&)A<V5N86UE+"`D
M97AT*2`]("@D8G5F(#U^(&U\*#\Z+BHO*3\H+BHI*%PN+BHI?"D["G)E860@
M7T9(+"`D8G5F+"`T.PIR96%D(%]&2"P@)&)U9BP@=6YP86-K*").(BP@)&)U
M9BD["FEF("AD969I;F5D*"1E>'0I(&%N9"`D97AT("%^("]<+B@_.G!M?'!L
M?&EX?&%L*20O:2D@>PIM>2`D9FEL96YA;64@/2!?<V%V95]A<R@B)&-R8R1E
M>'0B+"`D8G5F+"`P-S4U*3L*)%!!4CHZ2&5A=GDZ.D9U;&Q#86-H97LD9G5L
M;&YA;65](#T@)&9I;&5N86UE.PHD4$%2.CI(96%V>3HZ1G5L;$-A8VAE>R1F
M:6QE;F%M97T@/2`D9G5L;&YA;64["GT*96QS92!["B1-;V1U;&5#86-H97LD
M9G5L;&YA;65](#T@>PIB=68@/3X@)&)U9BP*8W)C(#T^("1C<F,L"FYA;64@
M/3X@)&9U;&QN86UE+`I].PHD<F5Q=6ER95]L:7-T>R1F=6QL;F%M97TK*SL*
M?0IR96%D(%]&2"P@)&)U9BP@-#L*?0IL;V-A;"!`24Y#(#T@*'-U8B!["FUY
M("@D<V5L9BP@)&UO9'5L92D@/2!`7SL*<F5T=7)N(&EF(')E9B`D;6]D=6QE
M(&]R("$D;6]D=6QE.PIM>2`D:6YF;R`]("1-;V1U;&5#86-H97LD;6]D=6QE
M?3L*<F5T=7)N('5N;&5S<R`D:6YF;SL*9&5L971E("1R97%U:7)E7VQI<W1[
M)&UO9'5L97T["B1)3D-[)&UO9'5L97T@/2`B+VQO861E<B\D:6YF;R\D;6]D
M=6QE(CL*:68@*"1%3E9[4$%27T-,14%.?2!A;F0@9&5F:6YE9"@F24\Z.D9I
M;&4Z.FYE=RDI('L*;7D@)&9H(#T@24\Z.D9I;&4M/FYE=U]T;7!F:6QE(&]R
M(&1I92`B0V%N)W0@8W)E871E('1E;7`@9FEL93H@)"$B.PHD9F@M/F)I;FUO
M9&4H*3L*)&9H+3YP<FEN="@D:6YF;RT^>V)U9GTI.PHD9F@M/G-E96LH,"P@
M,"D["G)E='5R;B`D9F@["GT*96QS92!["FUY("1F:6QE;F%M92`](%]S879E
M7V%S*"(D:6YF;RT^>V-R8WTN<&TB+"`D:6YF;RT^>V)U9GTI.PHD:6YF;RT^
M>V9I;&5](#T@)&9I;&5N86UE.PIO<&5N(&UY("1F:"P@)SPZ<F%W)RP@)&9I
M;&5N86UE(&]R(&1I92!Q<5M#86XG="!R96%D("(D9FEL96YA;64B.B`D(5T[
M"G)E='5R;B`D9F@["GT*9&EE(")";V]T<W1R87!P:6YG(&9A:6QE9#H@8V%N
M)W0@9FEN9"!M;V1U;&4@)&UO9'5L92$B.PI]+"!`24Y#*3L*<F5Q=6ER92!8
M4TQO861E<CL*<F5Q=6ER92!005(Z.DAE879Y.PIR97%U:7)E($-A<G`Z.DAE
M879Y.PIR97%U:7)E($5X<&]R=&5R.CI(96%V>3L*4$%2.CI(96%V>3HZ7VEN
M:71?9'EN86QO861E<B@I.PIR97%U:7)E($E/.CI&:6QE.PIW:&EL92`H;7D@
M)&9I;&5N86UE(#T@*'-O<G0@:V5Y<R`E<F5Q=6ER95]L:7-T*5LP72D@>PIU
M;FQE<W,@*"1)3D-[)&9I;&5N86UE?2!O<B`D9FEL96YA;64@/7X@+T)31%!!
M3B\I('L*:68@*"1F:6QE;F%M92`]?B`O7"YP;6,_)"]I*2!["G)E<75I<F4@
M)&9I;&5N86UE.PI]"F5L<V4@>PID;R`D9FEL96YA;64@=6YL97-S("1F:6QE
M;F%M92`]?B`O<VET96-U<W1O;6EZ95PN<&PD+SL*?0I]"F1E;&5T92`D<F5Q
M=6ER95]L:7-T>R1F:6QE;F%M97T["GT*:68@*"1B=68@;F4@)%!+6DE07TU!
M1TE#*2!["F]U=',H<7%;3F\@>FEP(&9O=6YD(&%F=&5R($9)3$4@<V5C=&EO
M;B!I;B!F:6QE("(D<')O9VYA;64B72D["FQA<W0@34%'24,@.PI]"B1S=&%R
M=%]P;W,@/2`H=&5L;"!?1D@I("T@-#L@("`@("`@("`@("`@("`@(R!S=&%R
M="!O9B!Z:7`*?0IM>2!`<&%R7V%R9W,["FUY("@D;W5T+"`D8G5N9&QE+"`D
M;&]G9F@L("1C86-H95]N86UE*3L*9&5L971E("1%3E9[4$%27T%04%]21553
M17T[(",@<V%N:71I>F4@*%)%55-%(&UA>2!B92!A('-E8W5R:71Y('!R;V)L
M96TI"FEF("@A)'-T87)T7W!O<R!O<B`H)$%21U9;,%T@97$@)RTM<&%R+6]P
M=&EO;G,G("8F('-H:69T*2D@>PIM>2`E9&ES=%]C;60@/2!Q=R@*<"`@(&)L
M:6)?=&]?<&%R"FD@("!I;G-T86QL7W!A<@IU("`@=6YI;G-T86QL7W!A<@IS
M("`@<VEG;E]P87(*=B`@('9E<FEF>5]P87(**3L*:68@*$!!4D=6(&%N9"`D
M05)'5ELP72!E<2`G+2UR975S92<I('L*<VAI9G0@0$%21U8["B1%3E9[4$%2
M7T%04%]2155317T@/2!S:&EF="!`05)'5CL*?0IE;'-E('L@(R!N;W)M86P@
M<&%R;"!B96AA=FEO=7(*;7D@0&%D9%]T;U]I;F,["G=H:6QE("A`05)'5BD@
M>PHD05)'5ELP72`]?B`O7BTH6T%)34]"3&)Q<&EU<U1V72DH+BHI+R!O<B!L
M87-T.PII9B`H)#$@97$@)TDG*2!["G!U<V@@0&%D9%]T;U]I;F,L("0R.PI]
M"F5L<VEF("@D,2!E<2`G32<I('L*979A;"`B=7-E("0R(CL*?0IE;'-I9B`H
M)#$@97$@)T$G*2!["G5N<VAI9G0@0'!A<E]A<F=S+"`D,CL*?0IE;'-I9B`H
M)#$@97$@)T\G*2!["B1O=70@/2`D,CL*?0IE;'-I9B`H)#$@97$@)V(G*2![
M"B1B=6YD;&4@/2`G<VET92<["GT*96QS:68@*"0Q(&5Q("=")RD@>PHD8G5N
M9&QE(#T@)V%L;"<["GT*96QS:68@*"0Q(&5Q("=Q)RD@>PHD<75I970@/2`Q
M.PI]"F5L<VEF("@D,2!E<2`G3"<I('L*;W!E;B`D;&]G9F@L("(^/B(L("0R
M(&]R(&1I92!Q<5M#86XG="!O<&5N(&QO9R!F:6QE("(D,B(Z("0A73L*?0IE
M;'-I9B`H)#$@97$@)U0G*2!["B1C86-H95]N86UE(#T@)#(["GT*<VAI9G0H
M0$%21U8I.PII9B`H;7D@)&-M9"`]("1D:7-T7V-M9'LD,7TI('L*9&5L971E
M("1%3E9[)U!!4E]414U0)WT["FEN:71?:6YC*"D["G)E<75I<F4@4$%2.CI$
M:7-T.PHF>R)005(Z.D1I<W0Z.B1C;60B?2@I('5N;&5S<R!`05)'5CL*)GLB
M4$%2.CI$:7-T.CHD8VUD(GTH)%\I(&9O<B!`05)'5CL*97AI=#L*?0I]"G5N
M<VAI9G0@0$E.0RP@0&%D9%]T;U]I;F,["GT*?0II9B`H)&]U="D@>PI["G)E
M<75I<F4@24\Z.D9I;&4["G)E<75I<F4@07)C:&EV93HZ6FEP.PIR97%U:7)E
M($1I9V5S=#HZ4TA!.PI]"FUY("1P87(@/2!S:&EF="A`05)'5BD["FUY("1Z
M:7`["FEF("AD969I;F5D("1P87(I('L*;W!E;B!M>2`D9F@L("<\.G)A=R<L
M("1P87(@;W(@9&EE('%Q6T-A;B=T(&9I;F0@<&%R(&9I;&4@(B1P87(B.B`D
M(5T["F)L97-S*"1F:"P@)TE/.CI&:6QE)RD["B1Z:7`@/2!!<F-H:79E.CI:
M:7`M/FYE=SL**"`D>FEP+3YR96%D1G)O;49I;&5(86YD;&4H)&9H+"`D<&%R
M*2`]/2!!<F-H:79E.CI::7`Z.D%:7T]+*"D@*0IO<B!D:64@<7%;17)R;W(@
M<F5A9&EN9R!Z:7`@87)C:&EV92`B)'!A<B)=.PI]"FUY("5M971A7W!A<B`]
M(&1O('L*:68@*"1Z:7`@86YD(&UY("1M971A(#T@)'II<"T^8V]N=&5N=',H
M)TU%5$$N>6UL)RDI('L*)&UE=&$@/7X@<R\N*EYP87(Z)"\O;7,["B1M971A
M(#U^(',O7EQ3+BHO+VUS.PHD;65T82`]?B`O7B`@*%M>.ETK*3H@*"XK*20O
M;6<["GT*?3L*:68@*&1E9FEN960@)'!A<BD@>PIO<&5N(&UY("1P:"P@)SPZ
M<F%W)RP@)'!A<B!O<B!D:64@<7%;0V%N)W0@<F5A9"!P87(@9FEL92`B)'!A
M<B(Z("0A73L*;7D@)&)U9CL*<F5A9"`D<&@L("1B=68L(#0["F1I92!Q<5LB
M)'!A<B(@:7,@;F]T(&$@<&%R(&9I;&5=('5N;&5S<R`D8G5F(&5Q("102UI)
M4%]-04=)0SL*8VQO<V4@)'!H.PI]"D-R96%T95!A=&@H)&]U="D@.PIM>2`D
M9F@@/2!)3SHZ1FEL92T^;F5W*`HD;W5T+`I)3SHZ1FEL93HZ3U]#4D5!5"@I
M('P@24\Z.D9I;&4Z.D]?4D174B@I('P@24\Z.D9I;&4Z.D]?5%)53D,H*2P*
M,#<W-RP**2!O<B!D:64@<7%;0V%N)W0@8W)E871E(&9I;&4@(B1O=70B.B`D
M(5T["B1F:"T^8FEN;6]D92@I.PIS965K(%]&2"P@,"P@,#L*;7D@)&QO861E
M<CL*:68@*&1E9FEN960@)'-T87)T7V]F7T9)3$5?<V5C=&EO;BD@>PIR96%D
M(%]&2"P@)&QO861E<BP@)'-T87)T7V]F7T9)3$5?<V5C=&EO;CL*?2!E;'-E
M('L*;&]C86P@)"\@/2!U;F1E9CL*)&QO861E<B`](#Q?1D@^.PI]"FEF("@A
M)$5.5GM005)?5D520D%424U](&%N9"`D;&]A9&5R(#U^("]>*#\Z(R%\7$!R
M96TI+RD@>PIR97%U:7)E(%!!4CHZ1FEL=&5R.CI0;V13=')I<#L*4$%2.CI&
M:6QT97(Z.E!O9%-T<FEP+3YA<'!L>2A<)&QO861E<BP@)#`I.PI]"B1F:"T^
M<')I;G0H)&QO861E<BD*;W(@9&EE('%Q6T5R<F]R('=R:71I;F<@;&]A9&5R
M('1O("(D;W5T(CH@)"%=.PII9B`H)&)U;F1L92D@>PIR97%U:7)E(%!!4CHZ
M2&5A=GD["E!!4CHZ2&5A=GDZ.E]I;FET7V1Y;F%L;V%D97(H*3L*:6YI=%]I
M;F,H*3L*<F5Q=6ER95]M;V1U;&5S*"D["FUY($!I;F,@/2!G<F5P('L@(2]"
M4T1004XO('T*9W)E<"!["B@D8G5N9&QE(&YE("=S:71E)RD@;W(**"1?(&YE
M("1#;VYF:6<Z.D-O;F9I9WMA<F-H;&EB97AP?2!A;F0*)%\@;F4@)$-O;F9I
M9SHZ0V]N9FEG>W!R:79L:6)E>'!]*3L*?2!`24Y#.PIS?"\K?"]\9RP@<WPO
M)'Q\(&9O<F5A8V@@0&EN8SL*;7D@)69I;&5S.PHD9FEL97-[)%]]*RL@9F]R
M($!$>6YA3&]A9&5R.CID;%]S:&%R961?;V)J96-T<RP@=F%L=65S("5)3D,[
M"FUY("1L:6)?97AT(#T@)$-O;F9I9SHZ0V]N9FEG>VQI8E]E>'1].R`@("`@
M("`@(",@6%A8(&QI8E]E>'0@=G,@9&QE>'0@/PIM>2`E=W)I='1E;CL*9F]R
M96%C:"!M>2`D:V5Y("AS;W)T(&ME>7,@)69I;&5S*2!["FUY("@D9FEL92P@
M)&YA;64I.PII9B`H9&5F:6YE9"AM>2`D9F,@/2`D4$%2.CI(96%V>3HZ1G5L
M;$-A8VAE>R1K97E]*2D@>PHH)&9I;&4L("1N86UE*2`]("@D:V5Y+"`D9F,I
M.PI]"F5L<V4@>PIF;W)E86-H(&UY("1D:7(@*$!I;F,I('L*:68@*"1K97D@
M/7X@;7Q>7%$D9&ER7$4O*"XJ*21\:2D@>PHH)&9I;&4L("1N86UE*2`]("@D
M:V5Y+"`D,2D["FQA<W0["GT*:68@*"1K97D@/7X@;7Q>+VQO861E<B];7B]=
M*R\H+BHI)'PI('L*:68@*&UY("1R968@/2`D36]D=6QE0V%C:&5[)#%]*2![
M"B@D9FEL92P@)&YA;64I(#T@*"1R968L("0Q*3L*;&%S=#L*?0II9B`H+68@
M(B1D:7(O)#$B*2!["B@D9FEL92P@)&YA;64I(#T@*"(D9&ER+R0Q(BP@)#$I
M.PIL87-T.PI]"GT*?0I]"FYE>'0@=6YL97-S(&1E9FEN960@)&YA;64["FYE
M>'0@:68@)'=R:71T96Y[)&YA;65]*RL["FYE>'0@:68@(7)E9B@D9FEL92D@
M86YD("1F:6QE(#U^("]<+EQ1)&QI8E]E>'1<120O:3L*;W5T<RAS<')I;G1F
M*'%Q6U!A8VMI;F<@1DE,12`B)7,B+BXN72P@<F5F("1F:6QE(#\@)&9I;&4M
M/GMN86UE?2`Z("1F:6QE*2D["FUY("1C;VYT96YT.PII9B`H<F5F*"1F:6QE
M*2D@>PHD8V]N=&5N="`]("1F:6QE+3Y[8G5F?3L*?0IE;'-E('L*;&]C86P@
M)"\@/2!U;F1E9CL*;W!E;B!M>2`D=&@L("<\.G)A=R<L("1F:6QE(&]R(&1I
M92!Q<5M#86XG="!R96%D("(D9FEL92(Z("0A73L*)&-O;G1E;G0@/2`\)'1H
M/CL*8VQO<V4@)'1H.PI005(Z.D9I;'1E<CHZ4&]D4W1R:7`M/F%P<&QY*%PD
M8V]N=&5N="P@(CQE;6)E9&1E9#XO)&YA;64B*0II9B`A)$5.5GM005)?5D52
M0D%424U](&%N9"`D;F%M92`]?B`O7"XH/SIP;7QI>'QA;"DD+VD["E!!4CHZ
M1FEL=&5R.CI0871C:$-O;G1E;G0M/FYE=RT^87!P;'DH7"1C;VYT96YT+"`D
M9FEL92P@)&YA;64I.PI]"B1F:"T^<')I;G0H(D9)3$4B+`IP86-K*"=.)RP@
M;&5N9W1H*"1N86UE*2`K(#DI+`IS<')I;G1F*"(E,#AX+R5S(BP@07)C:&EV
M93HZ6FEP.CIC;VUP=71E0U)#,S(H)&-O;G1E;G0I+"`D;F%M92DL"G!A8VLH
M)TXG+"!L96YG=&@H)&-O;G1E;G0I*2P*)&-O;G1E;G0I"F]R(&1I92!Q<5M%
M<G)O<B!W<FET:6YG(&5M8F5D9&5D($9)3$4@=&\@(B1O=70B.B`D(5T["F]U
M=',H<7%;5W)I='1E;B!A<R`B)&YA;64B72D["GT*?0II9B`H)'II<"D@>PHD
M>FEP+3YW<FET951O1FEL94AA;F1L92@D9F@I(#T]($%R8VAI=F4Z.EII<#HZ
M05I?3TL*;W(@9&EE('%Q6T5R<F]R('=R:71I;F<@>FEP('!A<G0@;V8@(B1O
M=70B73L*?0II9B`H)&UE=&%?<&%R>V-L96%N?2D@>PHD9F@M/G!R:6YT*")<
M,$-,14%.(BD["GT*96QS92!["FEF("@A9&5F:6YE9"`D8V%C:&5?;F%M92D@
M>PIM>2`D8W1X(#T@1&EG97-T.CI32$$M/FYE=R@Q*3L*<V5E:R`D9F@L(#`L
M(#`["B1C='@M/F%D9&9I;&4H)&9H*3L*<V5E:R`D9F@L(#`L(#(["B1C86-H
M95]N86UE(#T@)&-T>"T^:&5X9&EG97-T.PI]"B1C86-H95]N86UE(#T@<W5B
M<W1R("1C86-H95]N86UE+"`P+"`D8V%C:&5?;F%M95]S:7IE.PHD8V%C:&5?
M;F%M92`N/2`B7#`B('@@*"1C86-H95]N86UE7W-I>F4@+2!L96YG=&@@)&-A
M8VAE7VYA;64I.PHD9F@M/G!R:6YT*"1C86-H95]N86UE+`HB7#!#04-(12(I
M.PI]"FUY("1O9F9S970@/2`D9F@M/G1E;&P@+2!L96YG=&@H)&QO861E<BD[
M"B1F:"T^<')I;G0H<&%C:R@G3B<L("1O9F9S970I+`HD4$%27TU!1TE#*3L*
M)&9H+3YC;&]S92!O<B!D:64@<7%;17)R;W(@=W)I=&EN9R!T<F%I;&5R(&]F
M("(D;W5T(CH@)"%=.PIC:&UO9"`P-S4U+"`D;W5T.PIE>&ET.PI]"GL*;&%S
M="!U;FQE<W,@9&5F:6YE9"`D<W1A<G1?<&]S.PI?9FEX7W!R;V=N86UE*"D[
M"G)E<75I<F4@4$%2.PI005(Z.DAE879Y.CI?:6YI=%]D>6YA;&]A9&5R*"D[
M"GL*<F5Q=6ER92!&:6QE.CI&:6YD.PIR97%U:7)E($%R8VAI=F4Z.EII<#L*
M?0IM>2`D9F@@/2!)3SHZ1FEL92T^;F5W.R`@("`@("`@("`@("`@("`@("`@
M("`@("`@("`@(R!!<F-H:79E.CI::7`@;W!E<F%T97,@;VX@86X@24\Z.DAA
M;F1L90HD9F@M/F9D;W!E;BAF:6QE;F\H7T9(*2P@)W(G*2!O<B!D:64@<7%;
M9F1O<&5N*"D@9F%I;&5D.B`D(5T["D%R8VAI=F4Z.EII<#HZ<V5T0VAU;FM3
M:7IE*"US(%]&2"D["FUY("1Z:7`@/2!!<F-H:79E.CI::7`M/FYE=SL**"1Z
M:7`M/G)E861&<F]M1FEL94AA;F1L92@D9F@L("1P<F]G;F%M92D@/3T@07)C
M:&EV93HZ6FEP.CI!6E]/2R@I*0IO<B!D:64@<7%;17)R;W(@<F5A9&EN9R!Z
M:7`@87)C:&EV92`B)'!R;V=N86UE(ET["D%R8VAI=F4Z.EII<#HZ<V5T0VAU
M;FM3:7IE*#8T("H@,3`R-"D["G!U<V@@0%!!4CHZ3&EB0V%C:&4L("1Z:7`[
M"B1005(Z.DQI8D-A8VAE>R1P<F]G;F%M97T@/2`D>FEP.PIO=71S*")%>'1R
M86-T:6YG('II<"XN+B(I.PII9B`H9&5F:6YE9"`D14Y6>U!!4E]414U0?2D@
M>R`C('-H;W5L9"!B92!S970@870@=&AI<R!P;VEN="$*9F]R96%C:"!M>2`D
M;65M8F5R("@@)'II<"T^;65M8F5R<R`I('L*;F5X="!I9B`D;65M8F5R+3YI
M<T1I<F5C=&]R>3L*;7D@)&UE;6)E<E]N86UE(#T@)&UE;6)E<BT^9FEL94YA
M;64["FYE>'0@=6YL97-S("1M96UB97)?;F%M92`]?B!M>PI>"B\_<VAL:6(O
M"B@_.B1#;VYF:6<Z.D-O;F9I9WMA<F-H;F%M97TO*3\**%M>+UTK*0HD"GUX
M.PIM>2`D97AT<F%C=%]N86UE(#T@)#$["FUY("1D97-T7VYA;64@/2!&:6QE
M.CI3<&5C+3YC871F:6QE*"1%3E9[4$%27U1%35!]+"`D97AT<F%C=%]N86UE
M*3L*:68@*"UF("1D97-T7VYA;64@)B8@+7,@7R`]/2`D;65M8F5R+3YU;F-O
M;7!R97-S9613:7IE*"DI('L*;W5T<RAQ<5M3:VEP<&EN9R`B)&UE;6)E<E]N
M86UE(B!S:6YC92!I="!A;')E861Y(&5X:7-T<R!A="`B)&1E<W1?;F%M92)=
M*3L*?2!E;'-E('L*;W5T<RAQ<5M%>'1R86-T:6YG("(D;65M8F5R7VYA;64B
M('1O("(D9&5S=%]N86UE(BXN+ETI.PHH)&UE;6)E<BT^97AT<F%C=%1O1FEL
M94YA;65D*"1D97-T7VYA;64I(#T]($%R8VAI=F4Z.EII<#HZ05I?3TLH*2D*
M;W(@9&EE('%Q6T5R<F]R(&5X=')A8W1I;F<@>FEP(&UE;6)E<B!T;R`B)&1E
M<W1?;F%M92)=.PIC:&UO9"@P-34U+"`D9&5S=%]N86UE*2!I9B`D7D\@97$@
M(FAP=7@B.PI]"GT*?0I]"G5N;&5S<R`H)%!!4CHZ3&EB0V%C:&5[)'!R;V=N
M86UE?2D@>PID:64@/#P@(BXB('5N;&5S<R!`05)'5CL*57-A9V4Z("0P(%L@
M+4%L:6(N<&%R(%T@6R`M261I<B!=(%L@+4UM;V1U;&4@72!;('-R8RYP87(@
M72!;('!R;V=R86TN<&P@70HD,"!;("U"?"UB(%T@6RU/;W5T9FEL95T@<W)C
M+G!A<@HN"B1%3E9[4$%27U!23T=.04U%?2`]("1P<F]G;F%M92`]("0P(#T@
M<VAI9G0H0$%21U8I.PI]"G-U8B!#<F5A=&50871H('L*;7D@*"1N86UE*2`]
M($!?.PIR97%U:7)E($9I;&4Z.D)A<V5N86UE.PIM>2`H)&)A<V5N86UE+"`D
M<&%T:"P@)&5X="D@/2!&:6QE.CI"87-E;F%M93HZ9FEL97!A<G-E*"1N86UE
M+"`H)UPN+BHG*2D["G)E<75I<F4@1FEL93HZ4&%T:#L*1FEL93HZ4&%T:#HZ
M;6MP871H*"1P871H*2!U;FQE<W,H+64@)'!A=&@I.R`C(&UK<&%T:"!D:65S
M('=I=&@@97)R;W(*?0IS=6(@<F5Q=6ER95]M;V1U;&5S('L*<F5Q=6ER92!L
M:6(["G)E<75I<F4@1'EN84QO861E<CL*<F5Q=6ER92!I;G1E9V5R.PIR97%U
M:7)E('-T<FEC=#L*<F5Q=6ER92!W87)N:6YG<SL*<F5Q=6ER92!V87)S.PIR
M97%U:7)E($-A<G`["G)E<75I<F4@0V%R<#HZ2&5A=GD["G)E<75I<F4@17)R
M;F\["G)E<75I<F4@17AP;W)T97(Z.DAE879Y.PIR97%U:7)E($5X<&]R=&5R
M.PIR97%U:7)E($9C;G1L.PIR97%U:7)E($9I;&4Z.E1E;7`["G)E<75I<F4@
M1FEL93HZ4W!E8SL*<F5Q=6ER92!84TQO861E<CL*<F5Q=6ER92!#;VYF:6<[
M"G)E<75I<F4@24\Z.DAA;F1L93L*<F5Q=6ER92!)3SHZ1FEL93L*<F5Q=6ER
M92!#;VUP<F5S<SHZ6FQI8CL*<F5Q=6ER92!!<F-H:79E.CI::7`["G)E<75I
M<F4@1&EG97-T.CI32$$["G)E<75I<F4@4$%2.PIR97%U:7)E(%!!4CHZ2&5A
M=GD["G)E<75I<F4@4$%2.CI$:7-T.PIR97%U:7)E(%!!4CHZ1FEL=&5R.CI0
M;V13=')I<#L*<F5Q=6ER92!005(Z.D9I;'1E<CHZ4&%T8VA#;VYT96YT.PIR
M97%U:7)E(&%T=')I8G5T97,["F5V86P@>R!R97%U:7)E($-W9"!].PIE=F%L
M('L@<F5Q=6ER92!7:6XS,B!].PIE=F%L('L@<F5Q=6ER92!38V%L87(Z.E5T
M:6P@?3L*979A;"![(')E<75I<F4@07)C:&EV93HZ56YZ:7`Z.D)U<G-T('T[
M"F5V86P@>R!R97%U:7)E(%1I93HZ2&%S:#HZ3F%M961#87!T=7)E('T["F5V
M86P@>R!R97%U:7)E(%!E<FQ)3SL@<F5Q=6ER92!097)L24\Z.G-C86QA<B!]
M.PIE=F%L('L@<F5Q=6ER92!U=&8X('T["GT*<W5B(%]S971?<&%R7W1E;7`@
M>PII9B`H9&5F:6YE9"`D14Y6>U!!4E]414U0?2!A;F0@)$5.5GM005)?5$5-
M4'T@/7X@+R@N*RDO*2!["B1P87)?=&5M<"`]("0Q.PIR971U<FX["GT*9F]R
M96%C:"!M>2`D<&%T:"`H"BAM87`@)$5.5GLD7WTL('%W*"!005)?5$U01$E2
M(%1-4$1)4B!414U01$E2(%1%35`@5$U0("DI+`IQ=R@@0SI<7%1%35`@+W1M
M<"`N("D**2!["FYE>'0@=6YL97-S(&1E9FEN960@)'!A=&@@86YD("UD("1P
M871H(&%N9"`M=R`D<&%T:#L*;7D@)'5S97)N86UE.PIM>2`D<'=U:60["F5V
M86P@>R@D<'=U:60I(#T@9V5T<'=U:60H)#XI(&EF(&1E9FEN960@)#X[?3L*
M:68@*"!D969I;F5D*"97:6XS,CHZ3&]G:6Y.86UE*2`I('L*)'5S97)N86UE
M(#T@)E=I;C,R.CI,;V=I;DYA;64["GT*96QS:68@*&1E9FEN960@)'!W=6ED
M*2!["B1U<V5R;F%M92`]("1P=W5I9#L*?0IE;'-E('L*)'5S97)N86UE(#T@
M)$5.5GM54T523D%-17T@?'P@)$5.5GM54T52?2!\?"`G4UE35$5-)SL*?0IM
M>2`D<W1M<&1I<B`]("(D<&%T:"1S>7-[7V1E;&EM?7!A<BTB+G5N<&%C:R@B
M2"HB+"`D=7-E<FYA;64I.PIM:V1I<B`D<W1M<&1I<BP@,#<U-3L*;7D@)&-A
M8VAE7V1I<CL*:68@*"1%3E9[4$%27T-,14%.?2D@>PHD8V%C:&5?9&ER(#T@
M(G1E;7`M)"0B.PI]"F5L<V4@>PIO<&5N(&UY("1F:"P@(CPZ<F%W(BP@)'!R
M;V=N86UE(&]R(&1I92!Q<5M#86XG="!R96%D("(D<')O9VYA;64B.B`D(5T[
M"FEF("@H;7D@)&UA9VEC7W!O<R`](&9I;F1?<&%R7VUA9VEC*"1F:"DI(#X]
M(#`I('L*<V5E:R`D9F@L("1M86=I8U]P;W,*+2`D1DE,15]O9F9S971?<VEZ
M90HM(&QE;F=T:"@B7#!#04-(12(I+"`P.PIM>2`D8G5F.PIR96%D("1F:"P@
M)&)U9BP@;&5N9W1H*")<,$-!0TA%(BD["FEF("@D8G5F(&5Q(")<,$-!0TA%
M(BD@>PIS965K("1F:"P@)&UA9VEC7W!O<PHM("1&24Q%7V]F9G-E=%]S:7IE
M"BT@;&5N9W1H*")<,$-!0TA%(BD*+2`D8V%C:&5?;F%M95]S:7IE+"`P.PIR
M96%D("1F:"P@)&)U9BP@)&-A8VAE7VYA;65?<VEZ93L*)&)U9B`]?B!S+UPP
M+R]G.PHD8V%C:&5?9&ER(#T@(F-A8VAE+21B=68B.PI]"GT*8VQO<V4@)&9H
M.PI]"FEF("@A)&-A8VAE7V1I<BD@>PHD8V%C:&5?9&ER(#T@(G1E;7`M)"0B
M.PHD14Y6>U!!4E]#3$5!3GT@/2`Q.PI]"B1S=&UP9&ER("X]("(D<WES>U]D
M96QI;7TD8V%C:&5?9&ER(CL*;6MD:7(@)'-T;7!D:7(L(#`W-34["B1%3E9[
M4$%27U1%35!](#T@)'-T;7!D:7(["FQA<W0["GT*)'!A<E]T96UP(#T@)#$@
M:68@)$5.5GM005)?5$5-4'T@86YD("1%3E9[4$%27U1%35!](#U^("\H+BLI
M+SL*?0IS=6(@7W-A=F5?87,@>PIM>2`H)&YA;64L("1C;VYT96YT<RP@)&UO
M9&4I(#T@0%\["FUY("1F=6QL;F%M92`]("(D<&%R7W1E;7`O)&YA;64B.PIU
M;FQE<W,@*"UE("1F=6QL;F%M92D@>PIM>2`D=&5M<&YA;64@/2`B)&9U;&QN
M86UE+B0D(CL*;7D@)&9H.PHH;W!E;B`D9F@L("<^.G)A=R<L("1T96UP;F%M
M92D*)B8@*'!R:6YT("1F:"`D8V]N=&5N=',I"B8F("AC;&]S92`D9F@I"F]R
M(&1I92!Q<5M%<G)O<B!W<FET:6YG("(D=&5M<&YA;64B.B`D(5T["F-H;6]D
M("1M;V1E+"`D=&5M<&YA;64@:68@9&5F:6YE9"`D;6]D93L*<F5N86UE*"1T
M96UP;F%M92P@)&9U;&QN86UE*2!O<B!U;FQI;FLH)'1E;7!N86UE*3L*?0IR
M971U<FX@)&9U;&QN86UE.PI]"G-U8B!?<V5T7W!R;V=N86UE('L*:68@*&1E
M9FEN960@)$5.5GM005)?4%)/1TY!345](&%N9"`D14Y6>U!!4E]04D]'3D%-
M17T@/7X@+R@N*RDO*2!["B1P<F]G;F%M92`]("0Q.PI]"B1P<F]G;F%M92!\
M?#T@)#`["FEF("@D14Y6>U!!4E]414U0?2!A;F0@:6YD97@H)'!R;V=N86UE
M+"`D14Y6>U!!4E]414U0?2D@/CT@,"D@>PHD<')O9VYA;64@/2!S=6)S='(H
M)'!R;V=N86UE+"!R:6YD97@H)'!R;V=N86UE+"`D<WES>U]D96QI;7TI("L@
M,2D["GT*:68@*"$D14Y6>U!!4E]04D]'3D%-17T@;W(@:6YD97@H)'!R;V=N
M86UE+"`D<WES>U]D96QI;7TI(#X](#`I('L*:68@*&]P96X@;7D@)&9H+"`G
M/"<L("1P<F]G;F%M92D@>PIR971U<FX@:68@+7,@)&9H.PI]"FEF("@M<R`B
M)'!R;V=N86UE)'-Y<WM?97AE?2(I('L*)'!R;V=N86UE("X]("1S>7-[7V5X
M97T["G)E='5R;CL*?0I]"F9O<F5A8V@@;7D@)&1I<B`H<W!L:70@+UQ1)'-Y
M<WMP871H7W-E<'U<12\L("1%3E9[4$%42'TI('L*;F5X="!I9B!E>&ES=',@
M)$5.5GM005)?5$5-4'T@86YD("1D:7(@97$@)$5.5GM005)?5$5-4'T["B1D
M:7(@/7X@<R]<421S>7-[7V1E;&EM?5Q%)"\O.PHH*"1P<F]G;F%M92`]("(D
M9&ER)'-Y<WM?9&5L:6U])'!R;V=N86UE)'-Y<WM?97AE?2(I+"!L87-T*0II
M9B`M<R`B)&1I<B1S>7-[7V1E;&EM?21P<F]G;F%M921S>7-[7V5X97TB.PHH
M*"1P<F]G;F%M92`]("(D9&ER)'-Y<WM?9&5L:6U])'!R;V=N86UE(BDL(&QA
M<W0I"FEF("US("(D9&ER)'-Y<WM?9&5L:6U])'!R;V=N86UE(CL*?0I]"G-U
M8B!?9FEX7W!R;V=N86UE('L*)#`@/2`D<')O9VYA;64@?'P]("1%3E9[4$%2
M7U!23T=.04U%?3L*:68@*&EN9&5X*"1P<F]G;F%M92P@)'-Y<WM?9&5L:6U]
M*2`\(#`I('L*)'!R;V=N86UE(#T@(BXD<WES>U]D96QI;7TD<')O9VYA;64B
M.PI]"FUY("1P=V0@/2`H9&5F:6YE9"`F0W=D.CIG971C=V0I(#\@0W=D.CIG
M971C=V0H*0HZ("@H9&5F:6YE9"`F5VEN,S(Z.D=E=$-W9"D@/R!7:6XS,CHZ
M1V5T0W=D*"D@.B!@<'=D8"D["F-H;VUP*"1P=V0I.PHD<')O9VYA;64@/7X@
M<R]>*#\]7"Y<+C]<421S>7-[7V1E;&EM?5Q%*2\D<'=D)'-Y<WM?9&5L:6U]
M+SL*)$5.5GM005)?4%)/1TY!345](#T@)'!R;V=N86UE.PI]"G-U8B!?<&%R
M7VEN:71?96YV('L*:68@*"`D14Y6>U!!4E])3DE424%,25I%1'TK*R`]/2`Q
M("D@>PIR971U<FX["GT@96QS92!["B1%3E9[4$%27TE.251)04Q)6D5$?2`]
M(#(["GT*9F]R("AQ=R@@4U!!5TY%1"!414U0($-,14%.($1%0E5'($-!0TA%
M(%!23T=.04U%("D@*2!["F1E;&5T92`D14Y6>R)005)?)%\B?3L*?0IF;W(@
M*'%W+R!435!$25(@5$5-4"!#3$5!3B!$14)51R`O*2!["B1%3E9[(E!!4E\D
M7R)](#T@)$5.5GLB4$%27T=,3T)!3%\D7R)](&EF(&5X:7-T<R`D14Y6>R)0
M05)?1TQ/0D%,7R1?(GT["GT*:68@*"1%3E9[4$%27U1%35!]*2!["F1E;&5T
M92`D14Y6>U!!4E]#3$5!3GT["GT*?0IS=6(@;W5T<R!["G)E='5R;B!I9B`D
M<75I970["FEF("@D;&]G9F@I('L*<')I;G0@)&QO9V9H(")`7UQN(CL*?0IE
M;'-E('L*<')I;G0@(D!?7&XB.PI]"GT*<W5B(&EN:71?:6YC('L*<F5Q=6ER
M92!#;VYF:6<["G!U<V@@0$E.0RP@9W)E<"!D969I;F5D+"!M87`@)$-O;F9I
M9SHZ0V]N9FEG>R1??2P@<7<H"F%R8VAL:6)E>'`@<')I=FQI8F5X<"!S:71E
M87)C:&5X<"!S:71E;&EB97AP"G9E;F1O<F%R8VAE>'`@=F5N9&]R;&EB97AP
M"BD["GT*<&%C:V%G92!M86EN.PIR97%U:7)E(%!!4CL*=6YS:&EF="!`24Y#
M+"!<)E!!4CHZ9FEN9%]P87(["E!!4BT^:6UP;W)T*$!P87)?87)G<RD["F1I
M92!Q<5MP87(N<&PZ($-A;B=T(&]P96X@<&5R;"!S8W)I<'0@(B1P<F]G;F%M
M92(Z($YO('-U8V@@9FEL92!O<B!D:7)E8W1O<GE<;ET*=6YL97-S("UE("1P
M<F]G;F%M93L*9&\@)'!R;V=N86UE.PI#3U)%.CIE>&ET*"0Q*2!I9B`H)$`@
M/7XO7E]42U]%6$E47UPH*%QD*RE<*2\I.PID:64@)$`@:68@)$`["GT["B0Z
M.E]?15)23U(@/2`D0"!I9B`D0#L*?0I#3U)%.CIE>&ET*"0Q*2!I9B`H)#HZ
M7U]%4E)/4B`]?B]>7U1+7T58251?7"@H7&0K*5PI+RD["F1I92`D.CI?7T52
M4D]2(&EF("0Z.E]?15)23U(["C$["E]?14Y$7U\*``!#04-(10``<&5R;'AS
M:2YC`````1L#.P```0`````?__]ZM````1S__WL4```!-/__?A0```%,__]_
M]````:S__X#T```!Q/__@I0```'<__^#5````?3__X14```"#/__AO0```(D
M__^'5````CS__X>T```"5/__A]0```)L__^*-````H3__XI4```"G/__BM0`
M``*T__^0E````LS__Y$T```"Y/__D;0```+\__^35````Q3__Y.4```#+/__
ME)0```-$__^;]````US__Y[T```#=/__GW0```.D__^W%````[S__[=T```#
MU/__N)0```/L__^Z5````93__[L4```!?/__N]0```.,__^^=````60````4
M``````%Z4@`$>`\!&PP._P\````````4````'/__>9````!4`$$M"0\?#1X`
M```4````-/__>=@```+D`$$M"0\?#1X````4````3/__?,````'0`$$M"0\?
M#1X````4````9/__O0@````(```````````````4````?/__N9````"D`$$M
M"0\?#1X````4````E/__N+@```#``$$M"0\?#1X````4````K/__?D````#T
M`$$M"0\?#1X````4````Q/__?R@```&@`$$M"0\?#1X````4````W/__@+``
M``"\`$$M"0\?#1X````4````]/__@5@```#P`$$M"0\?#1X````4```!#/__
M@D````*@`$$M"0\?#1X````4```!)/__A,@```!0`$$M"0\?#1X````4```!
M//__A1````!@`$$M"0\?#1X````4```!5/__A5@````,`$$M"0\?#1X````4
M```!;/__A6````)4`$$M"0\?#1X````4```!A/__AZ@````@`$$M"0\?#1X`
M```4```!G/__A[````!D`$$M"0\?#1X````4```!M/__B!@```6D`$$M"0\?
M#1X````4```!S/__C<````"8`$$M"0\?#1X````4```!Y/__CD@```!H`$$M
M"0\?#1X````4```!_/__CK````&8`$,M"0\?#1X````4```"%/__D#@````H
M`$$M"0\?#1X````4```"+/__D&````#P`$$M"0\?#1X````4```"1/__D4@`
M``=,`$$M"0\?#1X````4```"7/__F)````+T`$$M"0\?#1X````4```"=/__
MFW@```"``$$M"0\?#1X````4```"C/__N$````*,`$$M"0\?#1X````4```"
MI/__F\@``!>0`$$M"0\?#1X````4```"O/__LU````!<`$$M"0\?#1X````4
M```"U/__LY@```$0`$$M"0\?#1X````4```"[/__M*````&H`$$M"0\?#1X`
M`````````/__________``````````#__________P``````````````````
M`````````````0`````````!``````````$````````";@`````````!````
M`````J8``````````0````````*[``````````$````````"V@`````````/
M````````!%$`````````#````````"+@``````````T```````!G````````
M```$`````````A``````````!0````````[@``````````8````````%.```
M```````*````````!)D`````````"P`````````8`````````!4`````````
M```````````#```````0L@```````````@````````70`````````!0`````
M````!P`````````7````````'1``````````!P```````!.```````````@`
M```````/8``````````)`````````!@`````<````0`````````#`````'``
M``$`````````!`````!O___[``````@`````````;___^0````````!<````
M````````````````````````````````````````````````````````````
M`````````````````````````````````````````````````!"LH```````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M`````````````````````````````P``@#!O_^<!`````0````$````!````
M`0````$````#``"@,&__WP$````!`````0````$````!`````0````,``,`P
M;__7`0````$````!`````0````$````!`````P``X#!O_\\!`````0````$`
M```!`````0````$````#``$`,&__QP$````!`````0````$````!`````0``
M``,``2`P;_^_`0````$````!`````0````$````!`````P`!0#!O_[<!````
M`0````$````!`````0````$````#``%@,&__KP$````!`````0````$````!
M`````0````,``8`P;_^G`0````$````!`````0````$````!`````P`!H#!O
M_Y\!`````0````$````!`````0````$````#``'`,&__EP$````!`````0``
M``$````!`````0````,``>`P;_^/`0````$````!`````0````$````!````
M`P`"`#!O_X<!`````0````$````!`````0````$````#``(@,&__?P$````!
M`````0````$````!`````0````,``D`P;_]W`0````$````!`````0````$`
M```!`````P`"8#!O_V\!`````0````$````!`````0````$````#``*`,&__
M9P$````!`````0````$````!`````0````,``J`P;_]?`0````$````!````
M`0````$````!`````P`"P#!O_U<!`````0````$````!`````0````$````#
M``+@,&__3P$````!`````0````$````!`````0````,``P`P;_]'`0````$`
M```!`````0````$````!`````P`#(#!O_S\!`````0````$````!`````0``
M``$````#``-`,&__-P$````!`````0````$````!`````0````,``V`P;_\O
M`0````$````!`````0````$````!`````P`#@#!O_R<!`````0````$````!
M`````0````$````#``.@,&__'P$````!`````0````$````!`````0````,`
M`\`P;_\7`0````$````!`````0````$````!`````P`#X#!O_P\!`````0``
M``$````!`````0````$````#``0`,&__!P$````!`````0````$````!````
M`0````,`!"`P;_[_`0````$````!`````0````$````!`````P`$0#!O_O<!
M`````0````$````!`````0````$````#``1@,&_^[P$````!`````0````$`
M```!`````0````,`!(`P;_[G`0````$````!`````0````$````!`````P`$
MH#!O_M\!`````0````$````!`````0````$````#``3`,&_^UP$````!````
M`0````$````!`````0````,`!.`P;_[/`0````$````!`````0````$````!
M`````P`%`#!O_L<!`````0````$````!`````0````$````#``4@,&_^OP$`
M```!`````0````$````!`````0````,`!4`P;_ZW`0````$````!`````0``
M``$````!`````P`%8#!O_J\!`````0````$````!`````0````$````#``6`
M,&_^IP$````!`````0````$````!`````0````,`!:`P;_Z?`0````$````!
M`````0````$````!`````P`%P#!O_I<!`````0````$````!`````0````$`
M```#``7@,&_^CP$````!`````0````$````!`````0````,`!@`P;_Z'`0``
M``$````!`````0````$````!`````P`&(#!O_G\!`````0````$````!````
M`0````$````#``9`,&_^=P$````!`````0````$````!`````0````,`!F`P
M;_YO`0````$````!`````0````$````!`````P`&@#!O_F<!`````0````$`
M```!`````0````$````#``:@,&_^7P$````!`````0````$````!`````0``
M``,`!L`P;_Y7`0````$````!`````0````$````!`````P`&X#!O_D\!````
M`0````$````!`````0````$````#``<`,&_^1P$````!`````0````$````!
M`````0````,`!R`P;_X_`0````$````!`````0````$````!`````P`'0#!O
M_C<!`````0````$````!`````0````$````#``=@,&_^+P$````!`````0``
M``$````!`````0````,`!X`P;_XG`0````$````!`````0````$````!````
M`P`'H#!O_A\!`````0````$````!`````0````$````#``?`,&_^%P$````!
M`````0````$````!`````0````,`!^`P;_X/`0````$````!`````0````$`
M```!`````P`(`#!O_@<!`````0````$````!`````0````$````#``@@,&_]
M_P$````!`````0````$````!`````0`````````````````````````*4$%2
M+G!M"@````````````````````!'0T,Z("A.971"4T0@;F(S(#(P,C,Q,#`X
M*2`Q,"XU+C``1T-#.B`H;F(S(#(P,C,Q,#`X*2`Q,"XU+C``````````!P``
M``@````&3F5T0E-$``!M961M:60``"1.971"4T0Z(&-R=#`N4RQV(#$N,B`R
M,#$X+S$Q+S(V(#$W.C,W.C0V(&IO97)G($5X<"`D`"1.971"4T0Z(&-R=#`M
M8V]M;6]N+F,L=B`Q+C(W(#(P,C(O,#8O,C$@,#8Z-3(Z,3<@<VMR;&P@17AP
M("0`)$YE=$)31#H@8W)T:2Y3+'8@,2XQ(#(P,3,O,#<O,3$@,#8Z-3<Z,34@
M;6%R=&EN($5X<"`D`"1.971"4T0Z(&-R=&)E9VEN+F,L=B`Q+C$W(#(P,3@O
M,3(O,C@@,3@Z,3<Z,3$@8VAR:7-T;W,@17AP("0`)$YE=$)31#H@8W)T96YD
M+E,L=B`Q+C$@,C`Q,R\P-R\Q,2`P-CHU-SHQ-2!M87)T:6X@17AP("0`)$YE
M=$)31#H@8W)T;BY3+'8@,2XQ(#(P,3,O,#<O,3$@,#8Z-3<Z,34@;6%R=&EN
M($5X<"`D```N<VAS=')T86(`+FEN=&5R<``N;F]T92YN971B<V0N:61E;G0`
M+FYO=&4N;F5T8G-D+G!A>``N:&%S:``N9'EN<WEM`"YD>6YS='(`+G)E;&$N
M9'EN`"YR96QA+G!L=``N:6YI=``N=&5X=``N9FEN:0`N<F]D871A`"YE:%]F
M<F%M95]H9'(`+F5H7V9R86UE`"YC=&]R<P`N9'1O<G,`+FIC<@`N9'EN86UI
M8P`N9V]T`"YD871A`"YB<W,`+F-O;6UE;G0`+FYO=&4N;F5T8G-D+FUC;6]D
M96P`````````````````````````````````````````````````````````
M``````````````````````````````````````````L````!``````````(`
M```````!R`````````'(`````````!<````````````````````!````````
M```````3````!P`````````"`````````>`````````!X``````````8````
M````````````````!```````````````)@````<``````````@````````'X
M`````````?@`````````%`````````````````````0``````````````#<`
M```%``````````(````````"$`````````(0`````````R@````%````````
M```````(``````````0````]````"P`````````"````````!3@````````%
M.`````````FH````!@````,`````````"``````````8````10````,`````
M`````@````````[@````````#N`````````$F0````````````````````$`
M`````````````$T````$``````````(````````3@````````!.`````````
M"9`````%```````````````(`````````!@```!7````!`````````!"````
M````'1`````````=$`````````70````!0```!0`````````"``````````8
M````80````$`````````!@```````"+@````````(N``````````%```````
M``````````````0``````````````&<````!``````````8````````C````
M`````",`````````1``````````````````````@``````````````!M````
M`0`````````&````````9P````````!G```````````4````````````````
M````!```````````````<P````$``````````@```````&<8````````9Q@`
M``````!!4@````````````````````@``````````````'L````!````````
M``(```````"H;````````*AL`````````00````````````````````$````
M``````````")`````0`````````"````````J7````````"I<`````````,$
M````````````````````"```````````````DP````$``````````P``````
M$*QX````````K'@`````````$`````````````````````@`````````````
M`)H````!``````````,``````!"LB````````*R(`````````!``````````
M```````````(``````````````"A`````0`````````#```````0K)@`````
M``"LF``````````(````````````````````"```````````````I@````8`
M`````````P``````$*R@````````K*`````````!X`````8`````````````
M``@`````````$````*\````!``````````,``````!"N@````````*Z`````
M`````S`````````````````````(``````````@```!<`````0`````````'
M```````0L@````````"R``````````A````````````````````!````````
M```@````M`````$``````````P``````$+I`````````ND``````````*```
M``````````````````@``````````````+H````(``````````,``````!"Z
M@````````+IH`````````(@```````````````````!```````````````"_
M`````0`````````P``````````````````"Z:``````````]````````````
M`````````0`````````!````R`````<`````````$```````````````````
MNJ@`````````'`````````````````````0`````````'````!\````!````
M`````#```````````````````+K$`````````5D````````````````````!
M``````````$````!`````P````````````````````````````"\'0``````
;``#=`````````````````````0``````````
