"""
RTCM Protocol Standard payload definitions

Created on 14 Feb 2022

Information sourced from RTCM STANDARD 10403.3 © 2016 RTCM

Message definitions 1240-1264 sourced from from:
- Proposal of new RTCM SSR Messages SSR Stage 1: Galileo, QZSS, SBAS, BDS
- Proposal of new RTCM SSR Messages SSR Stage 2: Vertical TEC (VTEC)

Payload definitions are contained in a series of dictionaries.
Repeating and conditional elements are defined as a tuple of
(element size/presence designator, element dictionary). The element
size/presence designator can take one of the following forms:

*Repeating elements:*
 - an integer representing the fixed size of the repeating element N.
 - a string representing the name of a preceding attribute containing
   the size of the repeating element N e.g.

.. code-block:: python

    "groupsat": (
        "IDF010",
        {
            "IDF011": "GNSS Satellite ID",
            etc ...
        },
    ),

*Conditional elements:*
 - a tuple containing a string and a value, representing the name of a
   preceding attribute and the value it must take in order for the
   optional element to be present e.g.

.. code-block:: python

    "optL1CA": (
        ("DF422_1", 1),  # if DF422_1 = 1
        {
            "DF423": "GLONASS L1 C/A Code-Phase Bias",
        },
    ),

A '+1' or '+2' suffix indicates that the attribute name must be suffixed
with the specified number of nested element indices e.g. '"IDF023+1"' -> '"IDF023_01"'

In some instances, the size of the repeating element is derived from multiple
attributes. In these cases the element size is denoted by a composite attribute
name which is calculated within `rtcmmessage.py` e.g. 'NHARMCOEFFC'

:author: semuadmin
:copyright: SEMU Consulting © 2022
:license: BSD 3-Clause
"""

# pylint: disable=too-many-lines

from pyrtcm.rtcmtypes_core import NHARMCOEFFC, NHARMCOEFFS, NRES

# *************************************************************
# RTCM3 MESSAGE GROUP HEADER DEFINITIONS
# *************************************************************
HDR_1001_1004 = {
    "DF002": "Message Number",
    "DF003": "Reference Station ID",
    "DF004": "GPS Epoch Time (TOW)",
    "DF005": "Synchronous GNSS Flag",
    "DF006": "No. of GPS Satellite Signals Processed",
    "DF007": "GPS Divergence-free Smoothing Indicator",
    "DF008": "GPS Smoothing Interval",
}

HDR_1005_1006 = {
    "DF002": "Message Number",
    "DF003": "Reference Station ID",
    "DF021": "Reserved for ITRF Realization Year",
    "DF022": "GPS Indicator",
    "DF023": "GLONASS Indicator",
    "DF024": "Reserved for Galileo Indicator",
    "DF141": "Reference-Station Indicator",
    "DF025": "Antenna Reference Point ECEF-X",
    "DF142": "Single Receiver Oscillator Indicator",
    "DF001_1": "Reserved",
    "DF026": "Antenna Reference Point ECEF-Y",
    "DF364": "Quarter Cycle Indicator",
    "DF027": "Antenna Reference Point ECEF-Z",
}

HDR_1009_1012 = {
    "DF002": "Message Number",
    "DF003": "Reference Station ID",
    "DF034": "GLONASS Epoch Time (tk)",
    "DF005": "Synchronous GNSS Flag",
    "DF035": "No. of GLONASS Satellite Signals Processed",
    "DF036": "GLONASS Divergence-free Smoothing Indicator",
    "DF037": "GLONASS Smoothing Interval",
}

HDR_1015_1017 = {
    "DF002": "Message Number",
    "DF059": "Network ID",
    "DF072": "Subnetwork ID",
    "DF065": "GPS Epoch Time (GPS TOW)",
    "DF066": "GPS Multiple Message Indicator",
    "DF060": "Master Reference Station ID",
    "DF061": "Auxiliary Reference Station ID",
    "DF067": "# of GPS Sats",
}

HDR_1037_1039 = {
    "DF002": "Message Number",
    "DF059": "Network ID",
    "DF072": "Subnetwork ID",
    "DF233": "GLONASS Network Epoch Time",
    "DF066": "Multiple Message Indicator",
    "DF060": "Master Reference Station ID",
    "DF061": "Auxiliary Reference Station ID",
    "DF234": "# of GLONASS Data Entries",
}

# *************************************************************
# RTCM3 MESSAGE PAYLOAD DEFINITIONS
# *************************************************************
RTCM_PAYLOADS_GET = {
    "1001": {
        **HDR_1001_1004,
        "group": (
            "DF006",
            {
                "DF009": "GPS Satellite ID",
                "DF010": "GPS L1 Code Indicator",
                "DF011": "GPS L1 Pseudorange",
                "DF012": "GPS L1 PhaseRange - L1 Pseudorange",
                "DF013": "GPS L1 Lock time Indicator",
            },
        ),
    },
    "1002": {
        **HDR_1001_1004,
        "group": (
            "DF006",
            {
                "DF009": "GPS Satellite ID",
                "DF010": "GPS L1 Code Indicator",
                "DF011": "GPS L1 Pseudorange",
                "DF012": "GPS L1 PhaseRange - L1 Pseudorange",
                "DF013": "GPS L1 Lock time Indicator",
                "DF014": "GPS Integer L1 Pseudorange Modulus Ambiguity",
                "DF015": "GPS L1 CNR",
            },
        ),
    },
    "1003": {
        **HDR_1001_1004,
        "group": (
            "DF006",
            {
                "DF009": "GPS Satellite ID",
                "DF010": "GPS L1 Code Indicator",
                "DF011": "GPS L1 Pseudorange",
                "DF012": "GPS L1 PhaseRange - L1 Pseudorange",
                "DF013": "GPS L1 Lock time Indicator",
                "DF016": "GPS L2 Code Indicator",
                "DF017": "GPS L2-L1 Pseudorange Difference",
                "DF018": "GPS L2 PhaseRange - L1 Pseudorange",
                "DF019": "GPS L2 Lock time Indicator",
            },
        ),
    },
    "1004": {
        **HDR_1001_1004,
        "group": (
            "DF006",
            {
                "DF009": "GPS Satellite ID",
                "DF010": "GPS L1 Code Indicator",
                "DF011": "GPS L1 Pseudorange",
                "DF012": "GPS L1 PhaseRange - L1 Pseudorange",
                "DF013": "GPS L1 Lock time Indicator",
                "DF014": "GPS Integer L1 Pseudorange Modulus Ambiguity",
                "DF015": "GPS L1 CNR",
                "DF016": "GPS L2 Code Indicator",
                "DF017": "GPS L2-L1 Pseudorange Difference",
                "DF018": "GPS L2 PhaseRange - L1 Pseudorange",
                "DF019": "GPS L2 Lock time Indicator",
                "DF020": "GPS L2 CNR",
            },
        ),
    },
    "1005": HDR_1005_1006,
    "1006": {**HDR_1005_1006, "DF028": "Antenna Height"},
    "1007": {
        "DF002": "Message Number",
        "DF003": "Reference Station ID",
        "DF029": "Descriptor Counter N",
        "group": (
            "DF029",
            {
                "DF030": "Antenna Descriptor",
            },
        ),
        "DF031": "Antenna Setup ID",
    },
    "1008": {
        "DF002": "Message Number",
        "DF003": "Reference Station ID",
        "DF029": "Descriptor Counter N",
        "group1": (
            "DF029",
            {
                "DF030": "Antenna Descriptor",
            },
        ),
        "DF031": "Antenna Setup ID",
        "DF032": "Serial Number Counter M",
        "group2": (
            "DF032",
            {
                "DF033": "Antenna Serial Number",
            },
        ),
    },
    "1009": {
        **HDR_1009_1012,
        "group": (
            "DF035",
            {
                "DF038": "GLONASS Satellite ID (Satellite Slot Number)",
                "DF039": "GLONASS Code Indicator",
                "DF040": "GLONASS Satellite Frequency Channel Number",
                "DF041": "GLONASS L1 Pseudorange",
                "DF042": "GLONASS L1 PhaseRange - L1 Pseudorange",
                "DF043": "GLONASS L1 Lock time Indicator",
            },
        ),
    },
    "1010": {
        **HDR_1009_1012,
        "group": (
            "DF035",
            {
                "DF038": "GLONASS Satellite ID (Satellite Slot Number)",
                "DF039": "GLONASS L1 Code Indicator",
                "DF040": "GLONASS Satellite Frequency Channel Number",
                "DF041": "GLONASS L1 Pseudorange",
                "DF042": "GLONASS L1 PhaseRange - L1 Pseudorange",
                "DF043": "GLONASS L1 Lock time Indicator",
                "DF044": "GLONASS Integer L1 Pseudorange Modulus Ambiguity",
                "DF045": "GLONASS L1 CNR",
            },
        ),
    },
    "1011": {
        **HDR_1009_1012,
        "group": (
            "DF035",
            {
                "DF038": "GLONASS Satellite ID (Satellite Slot Number)",
                "DF039": "GLONASS L1 Code Indicator",
                "DF040": "GLONASS Satellite Frequency Channel Number",
                "DF041": "GLONASS L1 Pseudorange",
                "DF042": "GLONASS L1 PhaseRange - L1 Pseudorange",
                "DF043": "GLONASS L1 Lock time Indicator",
                "DF046": "GLONASS L2 Code Indicator",
                "DF047": "GLONASS L2-L1 Pseudorange Difference",
                "DF048": "GLONASS L2 PhaseRange - L1 Pseudorange",
                "DF049": "GLONASS L2 Lock time Indicator",
            },
        ),
    },
    "1012": {
        **HDR_1009_1012,
        "group": (
            "DF035",
            {
                "DF038": "GLONASS Satellite ID (Satellite Slot Number)",
                "DF039": "GLONASS L1 Code Indicator",
                "DF040": "GLONASS Satellite Frequency Channel Number",
                "DF041": "GLONASS L1 Pseudorange",
                "DF042": "GLONASS L1 PhaseRange - L1 Pseudorange",
                "DF043": "GLONASS L1 Lock time Indicator",
                "DF044": "GLONASS Integer L1 Pseudorange Modulus Ambiguity",
                "DF045": "GLONASS L1 CNR",
                "DF046": "GLONASS L2 Code Indicator",
                "DF047": "GLONASS L2-L1 Pseudorange Difference",
                "DF048": "GLONASS L2 PhaseRange - L1 Pseudorange",
                "DF049": "GLONASS L2 Lock time Indicator",
                "DF050": "GLONASS L2 CNR",
            },
        ),
    },
    "1013": {
        "DF002": "Message Number",
        "DF003": "Reference Station ID",
        "DF051": "Modified Julian Day (MJD) Number",
        "DF052": "Seconds of Day (UTC)",
        "DF053": "No. of Message ID Announcements to Follow (Nm)",
        "DF054": "Leap Seconds, GPS-UTC",
        "group": (
            "DF053",
            {
                "DF055": "Message ID",
                "DF056": "Message Sync Flag",
                "DF057": "Message Transmission Interval",
            },
        ),
    },
    "1014": {
        "DF002": "Message Number",
        "DF059": "Network ID",
        "DF072": "Subnetwork ID",
        "DF058": "Number of Auxiliary Stations Transmitted",
        "DF060": "Master Reference Station ID",
        "DF061": "Auxiliary Reference Station ID",
        "DF062": "Aux-Master Delta Latitude",
        "DF063": "Aux-Master Delta Longitude",
        "DF064": "Aux-Master Delta Height",
    },
    "1015": {
        **HDR_1015_1017,
        "group": (
            "DF067",
            {
                "DF068": "GPS Satellite ID",
                "DF074": "GPS Ambiguity Status Flag",
                "DF075": "GPS Non Sync Count",
                "DF069": "GPS Ionospheric Carrier Phase Correction Difference",
            },
        ),
    },
    "1016": {
        **HDR_1015_1017,
        "group": (
            "DF067",
            {
                "DF068": "GPS Satellite ID",
                "DF074": "GPS Ambiguity Status Flag",
                "DF075": "GPS Non Sync Count",
                "DF070": "GPS Geometric Carrier Phase Correction Difference",
                "DF071": "GPS IODE",
            },
        ),
    },
    "1017": {
        **HDR_1015_1017,
        "group": (
            "DF067",
            {
                "DF068": "GPS Satellite ID",
                "DF074": "GPS Ambiguity Status Flag",
                "DF075": "GPS Non Sync Count",
                "DF070": "GPS Geometric Carrier Phase Correction Difference",
                "DF071": "GPS IODE",
                "DF069": "GPS Ionospheric Carrier Phase Correction Difference",
            },
        ),
    },
    # "1018": {RESERVED for Alt Ionospheric Correction Diff Msg},
    "1019": {
        "DF002": "Message Number",
        "DF009": "GPS Satellite ID",
        "DF076": "GPS Week Number",
        "DF077": "GPS SV ACCURACY",
        "DF078": "GPS CODE ON L2",
        "DF079": "GPS IDOT Rate of Inclination Angle",
        "DF071": "GPS IODE Issue of Data (Ephemeris)",
        "DF081": "GPS toc Reference Time, Clock",
        "DF082": "GPS af2 Clock correction drift rate",
        "DF083": "GPS af1 Clock correction drift",
        "DF084": "GPS af0 Clock correction bias",
        "DF085": "GPS IODC Issue of Data (Clock)",
        "DF086": "GPS Crs Amplitude of the Sine Harmonic Correction Term to the Orbit Radius",
        "DF087": "GPS ∆n Mean Motion Difference from Computed Value",
        "DF088": "GPS M0 Mean Anomaly at Reference Time",
        "DF089": "GPS Cuc Amplitude of the Cosine Harmonic Correction Term to the Argument of Latitude",
        "DF090": "GPS e Eccentricity",
        "DF091": "GPS Cus Amplitude of the Sine Harmonic Correction Term to the Argument of Latitude",
        "DF092": "GPS A½ Square Root of the Semi-Major Axis",
        "DF093": "GPS toe Reference Time Ephemeris",
        "DF094": "GPS Cic Amplitude of the Cosine Harmonic Correction Term to the Angle of Inclination",
        "DF095": "GPS Ω0 Longitude of Ascending Node of Orbit Plane at Weekly Epoch",
        "DF096": "GPS Cis Amplitude of the Sine Harmonic Correction Term to the Angle of Inclination",
        "DF097": "GPS i0 Inclination Angle at Reference Time",
        "DF098": "GPS Crc Amplitude of the Cosine Harmonic Correction Term to the Orbit Radius",
        "DF099": "GPS ω Argument of Perigee",
        "DF100": "GPS ΩDOT Rate of Right Ascension",
        "DF101": "GPS tGD",
        "DF102": "GPS SV HEALTH",
        "DF103": "GPS L2 P data flag",
        "DF137": "GPS Fit Interval",
    },
    "1020": {
        "DF002": "Message Number",
        "DF038": "GLONASS Satellite ID (Satellite Slot Number)",
        "DF040": "GLONASS Satellite Frequency Channel Number",
        "DF104": "GLONASS almanac health (Cn word)",
        "DF105": "GLONASS almanac health availability indicator",
        "DF106": "GLONASS P1",
        "DF107": "GLONASS tk",
        "DF108": "GLONASS MSB of Bn word",
        "DF109": "GLONASS P2",
        "DF110": "GLONASS tb",
        "DF111": "GLONASS xn(tb), first derivative",
        "DF112": "GLONASS xn(tb)",
        "DF113": "GLONASS xn(tb), second derivative",
        "DF114": "GLONASS yn(tb), first derivative",
        "DF115": "GLONASS yn(tb)",
        "DF116": "GLONASS yn(tb), second derivative",
        "DF117": "GLONASS zn(tb), first derivative",
        "DF118": "GLONASS zn(tb)",
        "DF119": "GLONASS zn(tb), second derivative",
        "DF120": "GLONASS P3",
        "DF121": "GLONASS γn(tb)",
        "DF122": "GLONASS-M P",
        "DF123": "GLONASS-M ln (third string)",
        "DF124": "GLONASS τn(tb)",
        "DF125": "GLONASS-M Δτn",
        "DF126": "GLONASS En",
        "DF127": "GLONASS-M P4",
        "DF128": "GLONASS-M FT",
        "DF129": "GLONASS-M NT",
        "DF130": "GLONASS-M M",
        "DF131": "GLONASS The Availability of Additional Data",
        "DF132": "GLONASS NA",
        "DF133": "GLONASS τc",
        "DF134": "GLONASS-M N4",
        "DF135": "GLONASS-M τGPS",
        "DF136": "GLONASS-M ln (fifth string)",
        "DF001_7": "Reserved",
    },
    "1021": {
        "DF002": "Message Number",
        "DF143": "Source-Name Counter",
        "group1": (
            "DF143",
            {
                "DF144": "Source-Name",
            },
        ),
        "DF145": "Target-Name Counter",
        "group2": (
            "DF145",
            {
                "DF146": "Target-Name",
            },
        ),
        "DF147": "System Identification Number",
        "DF148": "Utilized Transformation Message Indicator",
        "DF149": "Plate Number",
        "DF150": "Computation Indicator",
        "DF151": "Height Indicator",
        "DF152": "ΦV - Latitude of Origin, Area of Validity",
        "DF153": "ΛV - Longitude of Origin, Area of Validity",
        "DF154": "∆φV - N/S Extension, Area of Validity",
        "DF155": "∆λV - E/W Extension, Area of Validity",
        "DF156": "dX - Translation in X-direction",
        "DF157": "dY - Translation in Y-direction",
        "DF158": "dZ - Translation in Z-direction",
        "DF159": "R1 - Rotation Around the X-axis",
        "DF160": "R2 - Rotation Around the Y-axis",
        "DF161": "R3 - Rotation Around the Z-axis",
        "DF162": "dS - Scale Correction",
        "DF166": "add aS - Semi-major Axis of Source System Ellipsoid",
        "DF167": "add bS - Semi-minor Axis of Source System Ellipsoid",
        "DF168": "add aT - Semi-major Axis of Target System Ellipsoid",
        "DF169": "add bT - Semi-minor Axis of Target System Ellipsoid",
        "DF214": "Horizontal Helmert/Molodenski Quality Indicator",
        "DF215": "Vertical Helmert/Molodenski Quality Indicator",
    },
    "1022": {
        "DF002": "Message Number",
        "DF143": "Source-Name Counter",
        "group1": (
            "DF143",
            {
                "DF144": "Source-Name",
            },
        ),
        "DF145": "Target-Name Counter",
        "group2": (
            "DF145",
            {
                "DF146": "Target-Name",
            },
        ),
        "DF147": "System Identification Number",
        "DF148": "Utilized Transformation Message Indicator",
        "DF149": "Plate Number",
        "DF150": "Computation Indicator",
        "DF151": "Height Indicator",
        "DF152": "ΦV - Latitude of Origin, Area of Validity",
        "DF153": "ΛV - Longitude of Origin, Area of Validity",
        "DF154": "∆φV - N/S Extension, Area of Validity",
        "DF155": "∆λV - E/W Extension, Area of Validity",
        "DF156": "dX - Translation in X-direction",
        "DF157": "dY - Translation in Y-direction",
        "DF158": "dZ - Translation in Z-direction",
        "DF159": "R1 - Rotation Around the X-axis",
        "DF160": "R2 - Rotation Around the Y-axis",
        "DF161": "R3 - Rotation Around the Z-axis",
        "DF162": "dS - Scale Correction",
        "DF163": "XP  - X Coordinate for M-B Rotation Point",
        "DF164": "YP  - Y Coordinate for M-B Rotation Point",
        "DF165": "ZP  - Z Coordinate for M-B Rotation Point",
        "DF166": "add aS - Semi-major Axis of Source System Ellipsoid",
        "DF167": "add bS - Semi-minor Axis of Source System Ellipsoid",
        "DF168": "add aT - Semi-major Axis of Target System Ellipsoid",
        "DF169": "add bT - Semi-minor Axis of Target System Ellipsoid",
        "DF214": "Horizontal Helmert/Molodenski Quality Indicator",
        "DF215": "Vertical Helmert/Molodenski Quality Indicator",
    },
    "1023": {
        "DF002": "Message Number",
        "DF147": "System Identification Number",
        "DF190": "Horizontal Shift Indicator",
        "DF191": "Vertical Shift Indicator",
        "DF192": "φ0 - Latitude of Origin of Grids",
        "DF193": "λ0 - Longitude of Origin of Grids",
        "DF194": "∆φ - N/S Grid Area Extension",
        "DF195": "∆λ - E/W Grid Area Extension",
        "DF196": "Mean ∆φ - Mean Latitude Offset",
        "DF197": "Mean ∆λ - Mean Longitude Offset",
        "DF198": "Mean ∆H - Mean Height Offset",
        "group1": (
            NRES,
            {
                "DF199": "δφi - Latitude Residual",
                "DF200": "δλi - Longitude Residual",
                "DF201": "δhi - Height Residual",
            },
        ),
        "DF212": "Horizontal Interpolation Method Indicator",
        "DF213": "Vertical Interpolation Method Indicator",
        "DF216": "Horizontal Grid Quality Indicator",
        "DF217": "Vertical Grid Quality Indicator",
        "DF051": "Modified Julian Day (MJD) Number",
    },
    "1024": {
        "DF002": "Message Number",
        "DF147": "System Identification Number",
        "DF190": "Horizontal Shift Indicator",
        "DF191": "Vertical Shift Indicator",
        "DF202": "N0 - Northing of Origin",
        "DF203": "E0 - Easting of Origin",
        "DF204": "∆N - N/S Grid Area Extension",
        "DF205": "∆E - E/W Grid Area Extension",
        "DF206": "Mean ∆N - Mean Local Northing Offset",
        "DF207": "Mean ∆E - Mean Local Easting Offset",
        "DF208": "Mean ∆h - Mean Local Height Offset",
        "group1": (
            NRES,
            {
                "DF209": "δNi - Residual in Local Northing",
                "DF210": "δEi - Residual in Local Easting",
                "DF211": "δhi - Residual in Local Height",
            },
        ),
        "DF212": "Horizontal Interpolation Method Indicator",
        "DF213": "Vertical Interpolation Method Indicator",
        "DF216": "Horizontal Grid Quality Indicator",
        "DF217": "Vertical Grid Quality Indicator",
        "DF051": "Modified Julian Day (MJD) Number",
    },
    "1025": {
        "DF002": "Message Number",
        "DF147": "System Identification Number",
        "DF170": "Projection Type",
        "DF171": "LaNO - Latitude of Natural Origin",
        "DF172": "LoNO - Longitude of Natural Origin",
        "DF173": "add SNO - Scale Factor at Natural Origin",
        "DF174": "FE - False Easting",
        "DF175": "FN - False Northing",
    },
    "1026": {
        "DF002": "Message Number",
        "DF147": "System Identification Number",
        "DF170": "Projection Type",
        "DF176": "LaFO - Latitude of False Origin",
        "DF177": "LoFO - Longitude of False Origin",
        "DF178": "LaSP1 - Latitude of Standard Parallel No. 1",
        "DF179": "LaSP2 - Latitude of Standard Parallel No. 2",
        "DF180": "EFO - Easting of False Origin",
        "DF181": "NFO - Northing of False Origin",
    },
    "1027": {
        "DF002": "Message Number",
        "DF147": "System Identification Number",
        "DF170": "Projection Type",
        "DF182": "Rectification Flag",
        "DF183": "LaPC - Latitude of Projection Center",
        "DF184": "LoPC - Longitude of Projection Center",
        "DF185": "AzIL - Azimuth of Initial Line",
        "DF186": "Diff ARSG - Difference, Angle from Rectified to Skew Grid",
        "DF187": "Add SIL - Scale factor on Initial Line",
        "DF188": "EPC - Easting at Projection Center",
        "DF189": "NPC - Northing at Projection Center",
    },
    # "1028": {RESERVED for Global Plate-Fixed XFormation},
    "1029": {
        "DF002": "Message Number",
        "DF003": "Reference Station ID",
        "DF051": "Modified Julian Day (MJD) Number",
        "DF052": "Seconds of Day (UTC)",
        "DF138": "Number of Characters to Follow",
        "DF139": "Number of UTF-8 Code Units (N)",
        "group": (
            "DF139",
            {
                "DF140": "UTF-8 Character Code Units",
            },
        ),
    },
    "1030": {
        "DF002": "Message Number",
        "DF224": "GPS Residuals Epoch Time (TOW)",
        "DF003": "Reference Station ID",
        "DF223": "N-Refs",
        "DF006": "GPS Number of Satellite Signals Processed",
        "group": (
            "DF006",
            {
                "DF009": "GPS Satellite ID",
                "DF218": "soc",
                "DF219": "sod",
                "DF220": "soh",
                "DF221": "sIc",
                "DF222": "sId",
            },
        ),
    },
    "1031": {
        "DF002": "Message Number",
        "DF225": "GLONASS Residuals Epoch Time (tk)",
        "DF003": "Reference Station ID",
        "DF223": "N-Refs",
        "DF035": "GLONASS Number of Satellite Signals Processed",
        "group": (
            "DF035",
            {
                "DF038": "GLONASS Satellite ID",
                "DF218": "soc",
                "DF219": "sod",
                "DF220": "soh",
                "DF221": "sIc",
                "DF222": "sId",
            },
        ),
    },
    "1032": {
        "DF002": "Message Number",
        "DF003": "Non-Physical Reference Station ID",
        "DF226": "Physical Reference Station ID",
        "DF021": "ITRF Epoch Year",
        "DF025": "Physical reference station ARP ECEF-X",
        "DF026": "Physical reference station ARP ECEF-Y",
        "DF027": "Physical reference station ARP ECEF-Z",
    },
    "1033": {
        "DF002": "Message Number",
        "DF003": "Reference Station ID",
        "DF029": "Antenna Descriptor Counter N",
        "group1": (
            "DF029",
            {"DF030": "Antenna Descriptor"},
        ),
        "DF031": "Antenna Setup ID",
        "DF032": "Antenna Serial Number Counter M",
        "group2": (
            "DF032",
            {
                "DF033": "Antenna Serial Number",
            },
        ),
        "DF227": "Receiver Type Descriptor Counter I",
        "group3": (
            "DF227",
            {
                "DF228": "Receiver Type Descriptor",
            },
        ),
        "DF229": "Receiver Firmware Version Counter J",
        "group4": (
            "DF229",
            {
                "DF230": "Receiver Firmware Version",
            },
        ),
        "DF231": "Receiver Serial Number Counter K",
        "group5": (
            "DF231",
            {
                "DF232": "Receiver Serial Number",
            },
        ),
    },
    "1034": {
        "DF002": "Message Number",
        "DF003": "Reference Station ID",
        "DF240": "GPS FKP Epoch Time (TOW)",
        "DF006": "No. of GPS Satellite Signals Processed",
        "group": (
            "DF006",
            {
                "DF009": "GPS Satellite ID",
                "DF071": "GPS Issue of data ephemeris (IODE)",
                "DF242": "N0: Geometric gradient (North)",
                "DF243": "E0: Geometric gradient (East)",
                "DF244": "NI: Ionospheric gradient (North)",
                "DF245": "EI: Ionospheric gradient (East)",
            },
        ),
    },
    "1035": {
        "DF002": "Message Number",
        "DF003": "Reference Station ID",
        "DF241": "GLONASS FKP Epoch Time",
        "DF035": "No. of GLONASS Satellite Signals Processed",
        "group": (
            "DF035",
            {
                "DF038": "GLONASS Satellite ID",
                "DF392": "GLONASS Issue Of Data (IOD)",
                "DF242": "N0: Geometric gradient (North)",
                "DF243": "E0: Geometric gradient (East)",
                "DF244": "NI: Ionospheric gradient (North)",
                "DF245": "EI: Ionospheric gradient (East)",
            },
        ),
    },
    # "1036": {Not Used},
    "1037": {
        **HDR_1037_1039,
        "group": (
            "DF234",
            {
                "DF038": "GLONASS Satellite ID (Satellite Slot Number)",
                "DF235": "GLONASS Ambiguity Status Flag",
                "DF236": "GLONASS Non Sync Count",
                "DF237": "GLONASS Ionospheric Carrier Phase Correction Difference",
            },
        ),
    },
    "1038": {
        **HDR_1037_1039,
        "group": (
            "DF234",
            {
                "DF038": "GLONASS Satellite ID (Satellite Slot Number)",
                "DF235": "GLONASS Ambiguity Status Flag",
                "DF236": "GLONASS Non Sync Count",
                "DF238": "GLONASS Geometric Carrier Phase Correction Difference",
                "DF239": "GLONASS IOD",
            },
        ),
    },
    "1039": {
        **HDR_1037_1039,
        "group": (
            "DF234",
            {
                "DF038": "GLONASS Satellite ID (Satellite Slot Number)",
                "DF235": "GLONASS Ambiguity Status Flag",
                "DF236": "GLONASS Non Sync Count",
                "DF238": "GLONASS Geometric Carrier Phase Correction Difference",
                "DF239": "GLONASS IOD",
                "DF237": "GLONASS Ionospheric Carrier Phase Correction Difference",
            },
        ),
    },
    "1041": {
        "DF002": "Message Number",
        "DF516": "NavIC/IRNSS Satellide ID",
        "DF517": "Week Number (WN)",
        "DF518": "Clock bias (af0)",
        "DF519": "Clock drift (af1",
        "DF520": "Clock drift rate (af2)",
        "DF521": "SV Accuracy (URA)",
        "DF522": "Time of clock (toc)",
        "DF523": "Total Group Delay (TGD)",
        "DF524": "Mean Motion Difference (∆n)",
        "DF525": "Issue of Data Ephemeris & Clock (IODEC)",
        "DF526": "Reserved bits after IODEC",
        "DF527": "L5 Flag",
        "DF528": "S Flag",
        "DF529": "Cuc",
        "DF530": "Cus",
        "DF531": "Cic",
        "DF532": "Cis",
        "DF533": "Crc",
        "DF534": "Crs",
        "DF535": "Rate of Inclination angle (IDOT)",
        "DF536": "Mean Anomaly (M0)",
        "DF537": "Time of ephemeris (tOE)",
        "DF538": "Eccentricity (e)",
        "DF539": "Square root of Semi major axis (√A)",
        "DF540": "Long of Ascending Node (Ω0)",
        "DF541": "Argument of perigee (ω)",
        "DF542": "Rate of RAAN (ΩDOT)",
        "DF543": "Inclination (i0)",
        "DF544": "2 spare bits after IDOT",
        "DF545": "2 spare bits after i0",
    },
    "1042": {
        "DF002": "Message Number",
        "DF488": "BDS Satellite ID",
        "DF489": "BDS Week Number",
        "DF490": "BDS URAI",
        "DF491": "BDS IDOT",
        "DF492": "BDS AODE",
        "DF493": "BDS Toc",
        "DF494": "BDS a2",
        "DF495": "BDS a1",
        "DF496": "BSD a0",
        "DF497": "BDS AODC",
        "DF498": "BDS Crs",
        "DF499": "BDS ∆n",
        "DF500": "BDS M0",
        "DF501": "BDS Cuc",
        "DF502": "BDS e",
        "DF503": "BDS Cus",
        "DF504": "BDS A½",
        "DF505": "BDS Toe",
        "DF506": "BDS Cic",
        "DF507": "BDS Ω0",
        "DF508": "BDS Cis",
        "DF509": "BDS i0",
        "DF510": "BDS Crc",
        "DF511": "BDS ω",
        "DF512": "BDS ΩDOT",
        "DF513": "BDS TGD1",
        "DF514": "BDS TGD2",
        "DF515": "BSD SV Health",
    },
    "1044": {
        "DF002": "Message Number",
        "DF429": "QZSS Satellite ID",
        "DF430": "QZSS toc",
        "DF431": "QZSS af2",
        "DF432": "QZSS af1",
        "DF433": "QZSS af0",
        "DF434": "QZSS IODE",
        "DF435": "QZSS Crs",
        "DF436": "QZSS ∆n",
        "DF437": "QZSS M0",
        "DF438": "QZSS Cuc",
        "DF439": "QZSS e",
        "DF440": "QZSS Cus",
        "DF441": "QZSS A½",
        "DF442": "QZSS toe",
        "DF443": "QZSS Cic",
        "DF444": "QZSS Ω0",
        "DF445": "QZSS Cis",
        "DF446": "QZSS i0",
        "DF447": "QZSS Crc",
        "DF448": "QZSS ω",
        "DF449": "QZSS Ω0n DOT",
        "DF450": "QZSS i0-DOT",
        "DF451": "QZSS Codes on L2 Channel",
        "DF452": "QZSS Week Number",
        "DF453": "QZSS URA",
        "DF454": "QZSS SV health",
        "DF455": "QZSS TGD",
        "DF456": "QZSS IODC",
        "DF457": "QZSS Fit Interval",
    },
    "1045": {
        "DF002": "Message Number",
        "DF252": "Galileo Satellite ID",
        "DF289": "Galileo Week Number",
        "DF290": "Galileo IODnav",
        "DF291": "Galileo SV SISA",
        "DF292": "Galileo Rate of Inclination (IDOT)",
        "DF293": "Galileo toc",
        "DF294": "Galileo af2",
        "DF295": "Galileo af1",
        "DF296": "Galileo af0",
        "DF297": "Galileo Crs",
        "DF298": "Galileo ∆n",
        "DF299": "Galileo M0",
        "DF300": "Galileo Cuc",
        "DF301": "Galileo Eccentricity (e)",
        "DF302": "Galileo Cus",
        "DF303": "Galileo A½",
        "DF304": "Galileo toe",
        "DF305": "Galileo Cic",
        "DF306": "Galileo Ω0",
        "DF307": "Galileo Cis",
        "DF308": "Galileo i0",
        "DF309": "Galileo Crc",
        "DF310": "Galileo ω",
        "DF311": "Galileo ΩDOT",
        "DF312": "Galileo BGD (E1/E5a)",
        "DF314": "Galileo E5a Signal Health Status (OSHS)",
        "DF315": "Galileo E5a Data Validity Status (OSDVS)",
        "DF001_7": "Reserved",
    },
    "1046": {
        "DF002": "Message Number",
        "DF252": "Galileo Satellite ID",
        "DF289": "Galileo Week Number",
        "DF290": "Galileo IODnav",
        "DF286": "Galileo SISA Index (E1,E5b)",
        "DF292": "Galileo Rate of Inclination (IDOT)",
        "DF293": "Galileo toc",
        "DF294": "Galileo af2",
        "DF295": "Galileo af1",
        "DF296": "Galileo af0",
        "DF297": "Galileo Crs",
        "DF298": "Galileo ∆n",
        "DF299": "Galileo M0",
        "DF300": "Galileo Cuc",
        "DF301": "Galileo Eccentricity (e)",
        "DF302": "Galileo Cus",
        "DF303": "Galileo A½",
        "DF304": "Galileo toe",
        "DF305": "Galileo Cic",
        "DF306": "Galileo Ω0",
        "DF307": "Galileo Cis",
        "DF308": "Galileo i0",
        "DF309": "Galileo Crc",
        "DF310": "Galileo ω",
        "DF311": "Galileo ΩDOT",
        "DF312": "Galileo BGD (E1/E5a)",
        "DF313": "Galileo BGD (E5b,E1)",
        "DF316": "Galileo E5b Signal Health Status",
        "DF317": "Galileo E5b Data Validity Status",
        "DF287": "Galileo E1b Signal Health Status",
        "DF288": "Galileo E1b Data Validity Status",
        "DF001_2": "Reserved",
    },
    "1057": {
        "DF002": "Message Number",
        "DF385": "GPS Epoch Time 1s",
        "DF391": "SSR Update Interval",
        "DF388": "Multiple Message Indicator",
        "DF375": "Satellite Reference Datum",
        "DF413": "IOD SSR",
        "DF414": "SSR Provider ID",
        "DF415": "SSR Solution ID",
        "DF387": "No. of Satellites",
        "group": (
            "DF387",
            {
                "DF068": "GPS Satellite ID",
                "DF071": "GPS IODE",
                "DF365": "Delta Radial",
                "DF366": "Delta Along-Track",
                "DF367": "Delta Cross-Track",
                "DF368": "Dot Delta Radial",
                "DF369": "Dot Delta Along-Track",
                "DF370": "Dot Delta Cross-Track",
            },
        ),
    },
    "1058": {
        "DF002": "Message Number",
        "DF385": "GPS Epoch Time 1s",
        "DF391": "SSR Update Interval",
        "DF388": "Multiple Message Indicator",
        "DF413": "IOD SSR",
        "DF414": "SSR Provider ID",
        "DF415": "SSR Solution ID",
        "DF387": "No. of Satellites",
        "group": (
            "DF387",
            {
                "DF068": "GPS Satellite ID",
                "DF376": "Delta Clock C0",
                "DF377": "Delta Clock C1",
                "DF378": "Delta Clock C2",
            },
        ),
    },
    "1059": {
        "DF002": "Message Number",
        "DF385": "GPS Epoch Time 1s",
        "DF391": "SSR Update Interval",
        "DF388": "Multiple Message Indicator",
        "DF413": "IOD SSR",
        "DF414": "SSR Provider ID",
        "DF415": "SSR Solution ID",
        "DF387": "No. of Satellites",
        "groupsat": (
            "DF387",
            {
                "DF068": "GPS Satellite ID",
                "DF379": "No. of Code Biases Processed",
                "groupbias": (  # nested group
                    "DF379+1",  # +1 signifies 1 nested group index must be added
                    {
                        "DF380": "GPS Signal and Tracking Mode Indicator",
                        "DF383": "Code Bias",
                    },
                ),
            },
        ),
    },
    "1060": {
        "DF002": "Message Number",
        "DF385": "GPS Epoch Time 1s",
        "DF391": "SSR Update Interval",
        "DF388": "Multiple Message Indicator",
        "DF375": "Satellite Reference Datum",
        "DF413": "IOD SSR",
        "DF414": "SSR Provider ID",
        "DF415": "SSR Solution ID",
        "DF387": "No. of Satellites",
        "group": (
            "DF387",
            {
                "DF068": "GPS Satellite ID",
                "DF071": "GPS IODE",
                "DF365": "Delta Radial",
                "DF366": "Delta Along-Track",
                "DF367": "Delta Cross-Track",
                "DF368": "Dot Delta Radial",
                "DF369": "Dot Delta Along-Track",
                "DF370": "Dot Delta Cross-Track",
                "DF376": "Delta Clock C0",
                "DF377": "Delta Clock C1",
                "DF378": "Delta Clock C2",
            },
        ),
    },
    "1061": {
        "DF002": "Message Number",
        "DF385": "GPS Epoch Time 1s",
        "DF391": "SSR Update Interval",
        "DF388": "Multiple Message Indicator",
        "DF413": "IOD SSR",
        "DF414": "SSR Provider ID",
        "DF415": "SSR Solution ID",
        "DF387": "No. of Satellites",
        "group": (
            "DF387",
            {
                "DF068": "GPS Satellite ID",
                "DF389": "SSR URA",
            },
        ),
    },
    "1062": {
        "DF002": "Message Number",
        "DF385": "GPS Epoch Time 1s",
        "DF391": "SSR Update Interval",
        "DF388": "Multiple Message Indicator",
        "DF413": "IOD SSR",
        "DF414": "SSR Provider ID",
        "DF415": "SSR Solution ID",
        "DF387": "No. of Satellites",
        "group": (
            "DF387",
            {
                "DF068": "GPS Satellite ID",
                "DF390": "High Rate Clock Correction",
            },
        ),
    },
    "1063": {
        "DF002": "Message Number",
        "DF386": "GLONASS Epoch Time 1s",
        "DF391": "SSR Update Interval",
        "DF388": "Multiple Message Indicator",
        "DF375": "Satellite Reference Datum",
        "DF413": "IOD SSR",
        "DF414": "SSR Provider ID",
        "DF415": "SSR Solution ID",
        "DF387": "No. of Satellites",
        "group": (
            "DF387",
            {
                "DF384": "GLONASS Satellite ID",
                "DF392": "GLONASS IOD",
                "DF365": "Delta Radial",
                "DF366": "Delta Along-Track",
                "DF367": "Delta Cross-Track",
                "DF368": "Dot Delta Radial",
                "DF369": "Dot Delta Along-Track",
                "DF370": "Dot Delta Cross-Track",
            },
        ),
    },
    "1064": {
        "DF002": "Message Number",
        "DF386": "GLONASS Epoch Time 1s",
        "DF391": "SSR Update Interval",
        "DF388": "Multiple Message Indicator",
        "DF413": "IOD SSR",
        "DF414": "SSR Provider ID",
        "DF415": "SSR Solution ID",
        "DF387": "No. of Satellites",
        "group": (
            "DF387",
            {
                "DF384": "GLONASS Satellite ID",
                "DF376": "Delta Clock C0",
                "DF377": "Delta Clock C1",
                "DF378": "Delta Clock C2",
            },
        ),
    },
    "1065": {
        "DF002": "Message Number",
        "DF386": "GLONASS Epoch Time 1s",
        "DF391": "SSR Update Interval",
        "DF388": "Multiple Message Indicator",
        "DF413": "IOD SSR",
        "DF414": "SSR Provider ID",
        "DF415": "SSR Solution ID",
        "DF387": "No. of Satellites",
        "groupsat": (
            "DF387",
            {
                "DF384": "GLONASS Satellite ID",
                "DF379": "No. of Code Biases Processed",
                "groupbias": (  # nested group
                    "DF379+1",  # +1 signifies 1 nested group index must be added
                    {
                        "DF381": "GLONASS Signal and Tracking Mode Indicator",
                        "DF383": "Code Bias",
                    },
                ),
            },
        ),
    },
    "1066": {
        "DF002": "Message Number",
        "DF386": "GLONASS Epoch Time 1s",
        "DF391": "SSR Update Interval",
        "DF388": "Multiple Message Indicator",
        "DF375": "Satellite Reference Datum",
        "DF413": "IOD SSR",
        "DF414": "SSR Provider ID",
        "DF415": "SSR Solution ID",
        "DF387": "No. of Satellites",
        "group": (
            "DF387",
            {
                "DF384": "GLONASS Satellite ID",
                "DF392": "GLONASS IOD",
                "DF365": "Delta Radial",
                "DF366": "Delta Along-Track",
                "DF367": "Delta Cross-Track",
                "DF368": "Dot Delta Radial",
                "DF369": "Dot Delta Along-Track",
                "DF370": "Dot Delta Cross-Track",
                "DF376": "Delta Clock C0",
                "DF377": "Delta Clock C1",
                "DF378": "Delta Clock C2",
            },
        ),
    },
    "1067": {
        "DF002": "Message Number",
        "DF386": "GLONASS Epoch Time 1s",
        "DF391": "SSR Update Interval",
        "DF388": "Multiple Message Indicator",
        "DF413": "IOD SSR",
        "DF414": "SSR Provider ID",
        "DF415": "SSR Solution ID",
        "DF387": "No. of Satellites",
        "group": (
            "DF387",
            {
                "DF384": "GLONASS Satellite ID",
                "DF389": "SSR URA",
            },
        ),
    },
    "1068": {
        "DF002": "Message Number",
        "DF386": "GLONASS Epoch Time 1s",
        "DF391": "SSR Update Interval",
        "DF388": "Multiple Message Indicator",
        "DF413": "IOD SSR",
        "DF414": "SSR Provider ID",
        "DF415": "SSR Solution ID",
        "DF387": "No. of Satellites",
        "group": (
            "DF387",
            {
                "DF384": "GLONASS Satellite ID",
                "DF390": "High Rate Clock Correction",
            },
        ),
    },
    "1230": {
        "DF002": "Message Number",
        "DF003": "Reference Station ID",
        "DF421": "GLONASS Code-Phase bias indicator",
        "DF001_3": "Reserved",
        "DF422_1": "GLONASS FDMA signals mask L1 C/A",
        "DF422_2": "GLONASS FDMA signals mask L1 P",
        "DF422_3": "GLONASS FDMA signals mask L2 C/A",
        "DF422_4": "GLONASS FDMA signals mask L2 P",
        "optL1CA": (
            ("DF422_1", 1),  # if DF422_1 = 1
            {
                "DF423": "GLONASS L1 C/A Code-Phase Bias",
            },
        ),
        "optL1P": (
            ("DF422_2", 1),  # if DF422_2 = 1
            {
                "DF424": "GLONASS L1 P Code-Phase Bias",
            },
        ),
        "optL2CA": (
            ("DF422_3", 1),  # if DF422_3 = 1
            {
                "DF425": "GLONASS L2 C/A Code-Phase Bias",
            },
        ),
        "optL2P": (
            ("DF422_4", 1),  # if DF422_4 = 1
            {
                "DF426": "GLONASS L2 P Code-Phase Bias",
            },
        ),
    },
    "1240": {
        "DF002": "Message Number",
        "DF458": "Galileo Epoch Time 1s",
        "DF391": "SSR Update Interval",
        "DF388": "Multiple Message Indicator",
        "DF375": "Satellite Reference Datum",
        "DF413": "IOD SSR",
        "DF414": "SSR Provider ID",
        "DF415": "SSR Solution ID",
        "DF387": "No. of Satellites",
        "group": (
            "DF387",
            {
                "DF252": "Galileo Satellite ID",
                "DF459": "Galileo IODnav I/NAV",
                "DF365": "Delta Radial",
                "DF366": "Delta Along-Track",
                "DF367": "Delta Cross-Track",
                "DF368": "Dot Delta Radial",
                "DF369": "Dot Delta Along-Track",
                "DF370": "Dot Delta Cross-Track",
            },
        ),
    },
    "1241": {
        "DF002": "Message Number",
        "DF458": "Galileo Epoch Time 1s",
        "DF391": "SSR Update Interval",
        "DF388": "Multiple Message Indicator",
        "DF413": "IOD SSR",
        "DF414": "SSR Provider ID",
        "DF415": "SSR Solution ID",
        "DF387": "No. of Satellites",
        "group": (
            "DF387",
            {
                "DF252": "Galileo Satellite ID",
                "DF376": "Delta Clock C0",
                "DF377": "Delta Clock C1",
                "DF378": "Delta Clock C2",
            },
        ),
    },
    "1242": {
        "DF002": "Message Number",
        "DF458": "Galileo Epoch Time 1s",
        "DF391": "SSR Update Interval",
        "DF388": "Multiple Message Indicator",
        "DF413": "IOD SSR",
        "DF414": "SSR Provider ID",
        "DF415": "SSR Solution ID",
        "DF387": "No. of Satellites",
        "groupsat": (
            "DF387",
            {
                "DF252": "Galileo Satellite ID",
                "DF379": "No. of Code Biases Processed",
                "groupbias": (  # nested group
                    "DF379+1",  # +1 signifies 1 nested group index must be added
                    {
                        "DF382": "Galileo Signal and Tracking Mode Indicator",
                        "DF383": "Code Bias",
                    },
                ),
            },
        ),
    },
    "1243": {
        "DF002": "Message Number",
        "DF458": "Galileo Epoch Time 1s",
        "DF391": "SSR Update Interval",
        "DF388": "Multiple Message Indicator",
        "DF375": "Satellite Reference Datum",
        "DF413": "IOD SSR",
        "DF414": "SSR Provider ID",
        "DF415": "SSR Solution ID",
        "DF387": "No. of Satellites",
        "group": (
            "DF387",
            {
                "DF252": "Galileo Satellite ID",
                "DF459": "Galileo IODnav I/NAV",
                "DF365": "Delta Radial",
                "DF366": "Delta Along-Track",
                "DF367": "Delta Cross-Track",
                "DF368": "Dot Delta Radial",
                "DF369": "Dot Delta Along-Track",
                "DF370": "Dot Delta Cross-Track",
                "DF376": "Delta Clock C0",
                "DF377": "Delta Clock C1",
                "DF378": "Delta Clock C2",
            },
        ),
    },
    "1244": {
        "DF002": "Message Number",
        "DF458": "Galileo Epoch Time 1s",
        "DF391": "SSR Update Interval",
        "DF388": "Multiple Message Indicator",
        "DF413": "IOD SSR",
        "DF414": "SSR Provider ID",
        "DF415": "SSR Solution ID",
        "DF387": "No. of Satellites",
        "group": (
            "DF387",
            {
                "DF252": "Galileo Satellite ID",
                "DF389": "SSR URA",
            },
        ),
    },
    "1245": {
        "DF002": "Message Number",
        "DF458": "Galileo Epoch Time 1s",
        "DF391": "SSR Update Interval",
        "DF388": "Multiple Message Indicator",
        "DF413": "IOD SSR",
        "DF414": "SSR Provider ID",
        "DF415": "SSR Solution ID",
        "DF387": "No. of Satellites",
        "group": (
            "DF387",
            {
                "DF252": "Galileo Satellite ID",
                "DF390": "High Rate Clock Correction",
            },
        ),
    },
    "1246": {
        "DF002": "Message Number",
        "DF460": "QZSS Epoch Time 1s",
        "DF391": "SSR Update Interval",
        "DF388": "Multiple Message Indicator",
        "DF375": "Satellite Reference Datum",
        "DF413": "IOD SSR",
        "DF414": "SSR Provider ID",
        "DF415": "SSR Solution ID",
        "DF387": "No. of Satellites",
        "group": (
            "DF387",
            {
                "DF429": "QZSS Satellite ID",
                "DF434": "QZSS IODE",
                "DF365": "Delta Radial",
                "DF366": "Delta Along-Track",
                "DF367": "Delta Cross-Track",
                "DF368": "Dot Delta Radial",
                "DF369": "Dot Delta Along-Track",
                "DF370": "Dot Delta Cross-Track",
            },
        ),
    },
    "1247": {
        "DF002": "Message Number",
        "DF460": "QZSS Epoch Time 1s",
        "DF391": "SSR Update Interval",
        "DF388": "Multiple Message Indicator",
        "DF413": "IOD SSR",
        "DF414": "SSR Provider ID",
        "DF415": "SSR Solution ID",
        "DF387": "No. of Satellites",
        "group": (
            "DF387",
            {
                "DF429": "QZSS Satellite ID",
                "DF376": "Delta Clock C0",
                "DF377": "Delta Clock C1",
                "DF378": "Delta Clock C2",
            },
        ),
    },
    "1248": {
        "DF002": "Message Number",
        "DF460": "QZSS Epoch Time 1s",
        "DF391": "SSR Update Interval",
        "DF388": "Multiple Message Indicator",
        "DF413": "IOD SSR",
        "DF414": "SSR Provider ID",
        "DF415": "SSR Solution ID",
        "DF387": "No. of Satellites",
        "groupsat": (
            "DF387",
            {
                "DF429": "QZSS Satellite ID",
                "DF379": "No. of Code Biases Processed",
                "groupbias": (  # nested group
                    "DF379+1",  # +1 signifies 1 nested group index must be added
                    {
                        "DF461": "QZSS Signal and Tracking Mode Identifier",
                        "DF383": "Code Bias",
                    },
                ),
            },
        ),
    },
    "1249": {
        "DF002": "Message Number",
        "DF460": "QZSS Epoch Time 1s",
        "DF391": "SSR Update Interval",
        "DF388": "Multiple Message Indicator",
        "DF375": "Satellite Reference Datum",
        "DF413": "IOD SSR",
        "DF414": "SSR Provider ID",
        "DF415": "SSR Solution ID",
        "DF387": "No. of Satellites",
        "group": (
            "DF387",
            {
                "DF429": "QZSS Satellite ID",
                "DF434": "QZSS IODE",
                "DF365": "Delta Radial",
                "DF366": "Delta Along-Track",
                "DF367": "Delta Cross-Track",
                "DF368": "Dot Delta Radial",
                "DF369": "Dot Delta Along-Track",
                "DF370": "Dot Delta Cross-Track",
                "DF376": "Delta Clock C0",
                "DF377": "Delta Clock C1",
                "DF378": "Delta Clock C2",
            },
        ),
    },
    "1250": {
        "DF002": "Message Number",
        "DF460": "QZSS Epoch Time 1s",
        "DF391": "SSR Update Interval",
        "DF388": "Multiple Message Indicator",
        "DF413": "IOD SSR",
        "DF414": "SSR Provider ID",
        "DF415": "SSR Solution ID",
        "DF387": "No. of Satellites",
        "group": (
            "DF387",
            {
                "DF429": "QZSS Satellite ID",
                "DF389": "SSR URA",
            },
        ),
    },
    "1251": {
        "DF002": "Message Number",
        "DF460": "QZSS Epoch Time 1s",
        "DF391": "SSR Update Interval",
        "DF388": "Multiple Message Indicator",
        "DF413": "IOD SSR",
        "DF414": "SSR Provider ID",
        "DF415": "SSR Solution ID",
        "DF387": "No. of Satellites",
        "group": (
            "DF387",
            {
                "DF429": "QZSS Satellite ID",
                "DF390": "High Rate Clock Correction",
            },
        ),
    },
    "1252": {
        "DF002": "Message Number",
        "DF462": "SBAS Epoch Time 1s",
        "DF391": "SSR Update Interval",
        "DF388": "Multiple Message Indicator",
        "DF375": "Satellite Reference Datum",
        "DF413": "IOD SSR",
        "DF414": "SSR Provider ID",
        "DF415": "SSR Solution ID",
        "DF387": "No. of Satellites",
        "group": (
            "DF387",
            {
                "DF463": "SBAS Satellite ID",
                "DF468": "SBAS t0 Modulo",
                "DF469": "SBAS IODE CRC",
                "DF365": "Delta Radial",
                "DF366": "Delta Along-Track",
                "DF367": "Delta Cross-Track",
                "DF368": "Dot Delta Radial",
                "DF369": "Dot Delta Along-Track",
                "DF370": "Dot Delta Cross-Track",
            },
        ),
    },
    "1253": {
        "DF002": "Message Number",
        "DF462": "SBAS Epoch Time 1s",
        "DF391": "SSR Update Interval",
        "DF388": "Multiple Message Indicator",
        "DF413": "IOD SSR",
        "DF414": "SSR Provider ID",
        "DF415": "SSR Solution ID",
        "DF387": "No. of Satellites",
        "group": (
            "DF387",
            {
                "DF463": "SBAS Satellite ID",
                "DF376": "Delta Clock C0",
                "DF377": "Delta Clock C1",
                "DF378": "Delta Clock C2",
            },
        ),
    },
    "1254": {
        "DF002": "Message Number",
        "DF462": "SBAS Epoch Time 1s",
        "DF391": "SSR Update Interval",
        "DF388": "Multiple Message Indicator",
        "DF413": "IOD SSR",
        "DF414": "SSR Provider ID",
        "DF415": "SSR Solution ID",
        "DF387": "No. of Satellites",
        "groupsat": (
            "DF387",
            {
                "DF463": "SBAS Satellite ID",
                "DF379": "No. of Code Biases Processed",
                "groupbias": (  # nested group
                    "DF379+1",  # +1 signifies 1 nested group index must be added
                    {
                        "DF464": "SBAS Signal and Tracking Mode Identifier",
                        "DF383": "Code Bias",
                    },
                ),
            },
        ),
    },
    "1255": {
        "DF002": "Message Number",
        "DF462": "SBAS Epoch Time 1s",
        "DF391": "SSR Update Interval",
        "DF388": "Multiple Message Indicator",
        "DF375": "Satellite Reference Datum",
        "DF413": "IOD SSR",
        "DF414": "SSR Provider ID",
        "DF415": "SSR Solution ID",
        "DF387": "No. of Satellites",
        "group": (
            "DF387",
            {
                "DF463": "SBAS Satellite ID",
                "DF468": "SBAS t0 Modulo",
                "DF469": "SBAS IODE CRC",
                "DF365": "Delta Radial",
                "DF366": "Delta Along-Track",
                "DF367": "Delta Cross-Track",
                "DF368": "Dot Delta Radial",
                "DF369": "Dot Delta Along-Track",
                "DF370": "Dot Delta Cross-Track",
                "DF376": "Delta Clock C0",
                "DF377": "Delta Clock C1",
                "DF378": "Delta Clock C2",
            },
        ),
    },
    "1256": {
        "DF002": "Message Number",
        "DF462": "SBAS Epoch Time 1s",
        "DF391": "SSR Update Interval",
        "DF388": "Multiple Message Indicator",
        "DF413": "IOD SSR",
        "DF414": "SSR Provider ID",
        "DF415": "SSR Solution ID",
        "DF387": "No. of Satellites",
        "group": (
            "DF387",
            {
                "DF463": "SBAS Satellite ID",
                "DF389": "SSR URA",
            },
        ),
    },
    "1257": {
        "DF002": "Message Number",
        "DF462": "SBAS Epoch Time 1s",
        "DF391": "SSR Update Interval",
        "DF388": "Multiple Message Indicator",
        "DF413": "IOD SSR",
        "DF414": "SSR Provider ID",
        "DF415": "SSR Solution ID",
        "DF387": "No. of Satellites",
        "group": (
            "DF387",
            {
                "DF463": "SBAS Satellite ID",
                "DF390": "High Rate Clock Correction",
            },
        ),
    },
    "1258": {
        "DF002": "Message Number",
        "DF465": "BDS Epoch Time 1s",
        "DF391": "SSR Update Interval",
        "DF388": "Multiple Message Indicator",
        "DF375": "Satellite Reference Datum",
        "DF413": "IOD SSR",
        "DF414": "SSR Provider ID",
        "DF415": "SSR Solution ID",
        "DF387": "No. of Satellites",
        "group": (
            "DF387",
            {
                "DF488": "BDS Satellite ID",
                "DF470": "BDS toe Modulo",
                "DF471": "BDS IOD",
                "DF365": "Delta Radial",
                "DF366": "Delta Along-Track",
                "DF367": "Delta Cross-Track",
                "DF368": "Dot Delta Radial",
                "DF369": "Dot Delta Along-Track",
                "DF370": "Dot Delta Cross-Track",
            },
        ),
    },
    "1259": {
        "DF002": "Message Number",
        "DF465": "BDS Epoch Time 1s",
        "DF391": "SSR Update Interval",
        "DF388": "Multiple Message Indicator",
        "DF413": "IOD SSR",
        "DF414": "SSR Provider ID",
        "DF415": "SSR Solution ID",
        "DF387": "No. of Satellites",
        "group": (
            "DF387",
            {
                "DF488": "BDS Satellite ID",
                "DF376": "Delta Clock C0",
                "DF377": "Delta Clock C1",
                "DF378": "Delta Clock C2",
            },
        ),
    },
    "1260": {
        "DF002": "Message Number",
        "DF465": "BDS Epoch Time 1s",
        "DF391": "SSR Update Interval",
        "DF388": "Multiple Message Indicator",
        "DF413": "IOD SSR",
        "DF414": "SSR Provider ID",
        "DF415": "SSR Solution ID",
        "DF387": "No. of Satellites",
        "groupsat": (
            "DF387",
            {
                "DF488": "BDS Satellite ID",
                "DF379": "No. of Code Biases Processed",
                "groupbias": (  # nested group
                    "DF379+1",  # +1 signifies 1 nested group index must be added
                    {
                        "DF467": "BDS Signal and Tracking Mode Identifier",
                        "DF383": "Code Bias",
                    },
                ),
            },
        ),
    },
    "1261": {
        "DF002": "Message Number",
        "DF465": "BDS Epoch Time 1s",
        "DF391": "SSR Update Interval",
        "DF388": "Multiple Message Indicator",
        "DF375": "Satellite Reference Datum",
        "DF413": "IOD SSR",
        "DF414": "SSR Provider ID",
        "DF415": "SSR Solution ID",
        "DF387": "No. of Satellites",
        "group": (
            "DF387",
            {
                "DF488": "BDS Satellite ID",
                "DF470": "BDS toe Modulo",
                "DF471": "BDS IOD",
                "DF365": "Delta Radial",
                "DF366": "Delta Along-Track",
                "DF367": "Delta Cross-Track",
                "DF368": "Dot Delta Radial",
                "DF369": "Dot Delta Along-Track",
                "DF370": "Dot Delta Cross-Track",
                "DF376": "Delta Clock C0",
                "DF377": "Delta Clock C1",
                "DF378": "Delta Clock C2",
            },
        ),
    },
    "1262": {
        "DF002": "Message Number",
        "DF465": "BDS Epoch Time 1s",
        "DF391": "SSR Update Interval",
        "DF388": "Multiple Message Indicator",
        "DF413": "IOD SSR",
        "DF414": "SSR Provider ID",
        "DF415": "SSR Solution ID",
        "DF387": "No. of Satellites",
        "group": (
            "DF387",
            {
                "DF488": "BDS Satellite ID",
                "DF389": "SSR URA",
            },
        ),
    },
    "1263": {
        "DF002": "Message Number",
        "DF465": "BDS Epoch Time 1s",
        "DF391": "SSR Update Interval",
        "DF388": "Multiple Message Indicator",
        "DF413": "IOD SSR",
        "DF414": "SSR Provider ID",
        "DF415": "SSR Solution ID",
        "DF387": "No. of Satellites",
        "group": (
            "DF387",
            {
                "DF488": "BDS Satellite ID",
                "DF390": "High Rate Clock Correction",
            },
        ),
    },
    "1264": {
        "DF002": "Message Number",
        "DF385": "GPS Epoch Time 1s",
        "DF391": "SSR Update Interval",
        "DF388": "Multiple Message Indicator",
        "DF413": "IOD SSR",
        "DF414": "SSR Provider ID",
        "DF415": "SSR Solution ID",
        "DF478": "VTEC Quality Indicator",
        "DF472": "Number of Ionospheric Layers",  # Nᴵᴸ - 1
        "groupion": (
            "DF472",
            {
                "DF473": "Height of Ionospheric Layer",
                "DF474": "Spherical Harmonics Degree",  # N - 1
                "DF475": "Spherical Harmonics Order",  # M - 1
                "groupcoeffC": (
                    NHARMCOEFFC,
                    {
                        "DF476": "Spherical Harmonic Coefficient C",
                    },
                ),
                "groupcoeffS": (
                    NHARMCOEFFS,
                    {
                        "DF477": "Spherical Harmonic Coefficient S",
                    },
                ),
            },
        ),
    },
    "1300": {
        "DF002": "Message Number",
        "DF562": "Service CRS Name Counter",
        "group-DF562": (
            "DF562",
            {
                "DF563": "Service CRS Name",
            },
        ),
        "DF564": "Coordinate Epoch CE",
    },
    "1301": {
        "DF002": "Message Number",
        "DF143": "",
        "group-DF143": (
            "DF143",
            {
                "DF144": "",
            },
        ),
        "DF145": "",
        "group-DF145": (
            "DF145",
            {
                "DF146": "",
            },
        ),
        "DF147": "System Identification Number",
        "DF148": "Utilized Tranformation Message Indicator",
        "DF547": "Reference Epoch t0",
        "DF548": "dX0 Translation in X for Reference Epoch",
        "DF549": "dY0 Translation in Y for Reference Epoch",
        "DF550": "dZ0 Translation in Z for Reference Epoch",
        "DF551": "R01 Rotation about X for Reference Epoch",
        "DF552": "R02 Rotation about Y for Reference Epoch",
        "DF553": "R03 Rotation about Z for Reference Epoch",
        "DF554": "dS0 Scale Correction for Reference Epoch",
        "DF555": "dot dX Rate of Change of Translation in X",
        "DF556": "dot dY Rate of Change of Translation in Y",
        "DF557": "dot dZ Rate of Change of Translation in Z",
        "DF558": "dot R1 Rate of Change of Rotation about X",
        "DF559": "dot R2 Rate of Change of Rotation about Y",
        "DF560": "dot R3 Rate of Change of Rotation about Z",
        "DF561": "dot dS Rate of Change of Scale Correction",
    },
    "1302": {
        "DF002": "Message Number",
        "DF565": "RTCM CRS Name Counter",
        "group-DF565": (
            "DF565",
            {
                "DF566": "RTCM CRS Name",
            },
        ),
        "DF567": "Anchor - Global/Plate Fixed Indicator",
        "DF149": "Plate Number",
        "DF568": "Number of Database Links",
        "group-DF568": (
            "DF568",
            {
                "DF569": "Database Link Counter",
                # TODO check correct payload definition
                # 1302 messages from EUREF01 appear to be truncated?
                "group-DF569": (
                    "DF569+1",
                    {
                        "DF570": "Database Link",
                    },
                ),
            },
        ),
    },
    "1303": {
        "DF002": "Message Number",
        "DF571": "Beidou Residuals Epoch Time TOW",
        "DF003": "Reference Station ID",
        "DF223": "N-Refs",
        "DF572": "Beidou Number of Satellite Signals Processed",
        "group-DF572": (
            "DF572",
            {
                "DF488": "BDS Satellite ID",
                "DF218": "soc",
                "DF219": "sod",
                "DF220": "soh",
                "DF221": "sIc",
                "DF222": "sId",
            },
        ),
    },
    "1304": {
        "DF002": "Message Number",
        "DF573": "Galileo Residuals Epoch Time TOW",
        "DF003": "Reference Station ID",
        "DF223": "N-Refs",
        "DF574": "Galileo Number of Satellite Signals Processed",
        "group-DF574": (
            "DF574",
            {
                "DF252": "Galileo Satellite ID",
                "DF218": "soc",
                "DF219": "sod",
                "DF220": "soh",
                "DF221": "sIc",
                "DF222": "sId",
            },
        ),
    },
    "1305": {
        "DF002": "Message Number",
        "DF575": "QZSS Residuals Epoch Time TOW",
        "DF003": "Reference Station ID",
        "DF223": "N-Refs",
        "DF576": "QZSS Number of Satellite Signals Processed",
        "group-DF576": (
            "DF576",
            {
                "DF429": "QZSS Satellite ID",
                "DF218": "soc",
                "DF219": "sod",
                "DF220": "soh",
                "DF221": "sIc",
                "DF222": "sId",
            },
        ),
    },
}
