/*
	config.h

	The Config class implements all the functionality related to manipulating
	the pgjobs configuration settings. This file is built in both the agent
	and the configuration tool.

	fstream is for single use! Instances must be recreated before use and
	deleted after.

	Project: pgjobs
	Author: Zlatko Michailov
	Created:  4-Jul-2003
	Updated: 19-Jul-2003
	Updated: 23-Jul-2003
	Updated: 24-Jul-2003
	Updated: 27-Jul-2003
	Updated:  2-Aug-2003
	Updated:  3-Aug-2003
	Updated:  2-Oct-2003
	Updated:  6-Oct-2003
	Updated: 13-Oct-2003
	Updated: 15-Oct-2003
	Updated:  7-May-2004
	Updated: 26-May-2004
	Updated: 18-Jun-2004

	This file is provided as is, with no warranty. Use at your own risk.

	Copyright (c) 2003-2004, Zlatko Michailov

*/



#ifndef __CONFIG_H__
#define __CONFIG_H__



//--------------------------------------------------------------------------------

#include <fstream>
#include <string>
#include "defs.h"

using namespace std;



//--------------------------------------------------------------------------------
// Database Reference:
// Must include a database name, server and port are optional

class DbRef
{
public:
	string	Database;
	string	Host;
	string	Port;


public:
	DbRef(	const string& database,
			const string& host	= Default,
			const string& port  = Default )
	{
		Database	= database;
		Host		= host;
		Port		= port;
	}


public:
	bool ToPsqlOptions( string& psqlOptions )
	{
		psqlOptions.clear();

		psqlOptions  = " -d \"";
		psqlOptions += Database;
		psqlOptions += "\" ";

		if ( Host != Default )
		{
			psqlOptions += " -h \"";
			psqlOptions += Host;
			psqlOptions += "\" ";
		}

		if ( Port != Default )
		{
			psqlOptions += " -p \"";
			psqlOptions += Port;
			psqlOptions += "\" ";
		}

		return true;
	}
};



//--------------------------------------------------------------------------------
// Db row info

struct DbInfo
{
	string	ID;
	string	Host;
	string	Port;
	string	Database;
	string	LogLevel;
	string	User;
	string	Notification;
	string	Recipient;
};



//--------------------------------------------------------------------------------

typedef istream::pos_type StreamPosition;



//--------------------------------------------------------------------------------
// Complete Config API

class Config
{
private:
	static const string	__empty;
	static string		__trash;


// Data members
protected:
	string		FileSpec;
	ifstream*	In;
	ofstream*	Out;


// Construction
public:
	Config( const string& fileSpec );
	~Config();


// Public API
public:
	bool EnableDb( const DbRef& dbref, string& err = __trash );
	bool DisableDb( const DbRef& dbref, string& err = __trash );
	bool GetMaster( string& row, string& err = __trash );
	bool SetMaster( const string row, string& err = __trash );
	bool GetSystem( string& row, string& err = __trash );
	bool SetSystem( const string row, string& err = __trash );
	bool GetDb( const DbRef& dbref, string& row, string& err = __trash );
	bool SetDb( const DbRef& dbref, const string row, string& err = __trash );
	bool SeqGetDb( StreamPosition& pos, DbInfo& dbInfo, string& err = __trash );

	bool GetCol( const string& row, int i, string& col, string& err = __trash );
	bool SetCol( string& row, int i, const string& col, string& err = __trash );


// Internal support - File level
protected:
	bool NextDbId( string& dbid, string& err );
	bool FindDbId( const DbRef& dbref, string& dbid, string& err = __trash );
	bool GetDbId( bool exists, const DbRef& dbref, string& dbid, string& err = __trash );
	bool GetRow( const string& rowid, string& row, string& err = __trash );
	bool SetRow( const string& rowid, const string& row, string& err = __trash );
	bool DelRow( const string& rowid, string& err = __trash );
	bool GetSetRow( bool mustSet, const string& rowid, const string& rowIn, string& rowOut, string& err = __trash );


// Internal support - Row level
protected:
	bool GetSetCol( bool mustSet, const string& row, int i, const string& colIn, string& out, string& err = __trash );
	bool FindChar( string& remain, char ch, bool mustFind, string& extract, string& err = __trash );


// Internal support - I/O
protected:
	bool Open( bool withWrite = false, string& err = __trash );
	bool Backup( string& err = __trash );
	bool ReadLine( string& line, string& err = __trash );
	bool WriteLine( const string& line, string& err = __trash );
	void Seek( StreamPosition pos );
	void Tell( StreamPosition& pos );
	void Close();


// Self test
public:
	void SelfTest( ostream& out );

protected:
	void SelfTestRead( ostream& out );
	void SelfTestReadWrite( ostream& out );
	void SelfTestColumn( ostream& out );
	void SelfTestRow( ostream& out );
	void SelfTestId( ostream& out );
	void SelfTestPublic( ostream& out );
	void SelfTestSeq( ostream& out );
};



#endif	// __CONFIG_H__


