/*
	log.cpp

	The Log class implements asynchronous message logging using the process
	security credentials. The clients of the class may have impersonated less-
	privileged users and thus may not have sufficient access to the log files.

	Project: pgjobs
	Author: Zlatko Michailov
	Created:  2-Oct-2003
	Updated:  6-Oct-2003
	Updated:  5-May-2004
	Updated:  6-May-2004

	This file is provided as is, with no warranty. Use at your own risk.

	Copyright (c) 2003-2004, Zlatko Michailov

*/



//--------------------------------------------------------------------------------

#include "log.h"
#include <sys/time.h>
#include <string>
#include <fstream>
#include "defs.h"


using namespace std;
using namespace msdk;



//--------------------------------------------------------------------------------
// Log Message:
// This class is used internally by class Log. It is defined separately for
// clarity of the source code.

static const int agentOffset		= 0;
static const int statusOffset		= sizeof( char* );
static const int timestampOffset	= statusOffset + sizeof( char* );
static const int textOffset			= timestampOffset + sizeof( tm );


// Construct from parameters
LogMessage::LogMessage( int			command,
						const char*	agent,
						const char*	status,
						const char*	text ) :
	Message( command )
{
	int dataLength = textOffset + 1; // The 1 is for the trailing '\0'
	int textLength = 0;

	if ( text )
	{
		textLength = strlen( text );
		dataLength += textLength;
	}

	// Manually fill up the buffer of the underlying Message
	if ( AllocBuffer( dataLength ) )
	{
		// Copy data: Agent and Status must point to static strings.
		// Threfore copy the pointers, not the entire strings.
		// text may point to a buffer in the stack. Therefore
		// copy its buffer with the trailing '\0'.
		*( const char** )( DataBuffer + agentOffset )	= agent;
		*( const char** )( DataBuffer + statusOffset )	= status;

		time_t t = time( 0 );
		localtime_r( &t, ( tm* )( DataBuffer + timestampOffset ) );

		if ( text )
		{
			memmove( DataBuffer + textOffset, text, textLength );
		}
		DataBuffer[ textOffset + textLength ] = '\0';

		// Initialize "this" members
		Agent		= agent;
		Status		= status;
		Timestamp	= ( tm* )( DataBuffer + timestampOffset );
		Text		= DataBuffer + textOffset;
	}
}



// Construct from Message
LogMessage::LogMessage( const Message& message ) :
	Message( message )
{
	// Only initialize "this" members
	Agent		= *( char** )( DataBuffer + agentOffset );
	Status		= *( char** )( DataBuffer + statusOffset );
	Timestamp	= ( tm* )( DataBuffer + timestampOffset );
	Text		= DataBuffer + textOffset;
}



//--------------------------------------------------------------------------------
// Class Log exposes a public synchronous method to generate and post a LogMessage
// whilst the processing of the message is asynchronous.

// Clients must initialize their sessions first
bool Log::InitAgent( const char* agent )
{
	LogMessage logMsg( CmdLogInitAgent, agent );
	return PostMessage( logMsg );
}



// Clients log messages through this method
bool Log::Record( const char* agent, const char* status, const char* text )
{
	LogMessage logMsg( CmdLogMessage, agent, status, text );
	return PostMessage( logMsg );
}



//--------------------------------------------------------------------------------
// Overridables

void Log::ProcessMessage( const Message& message )
{
	LogMessage	logMsg( message );
	string		path;

	switch ( logMsg.Command )
	{
	case CmdLogInitAgent:
		// Create directory hierarchy
		path  = LogRoot;
		path += logMsg.Agent;
		mkdirhier( path.c_str(), 0775 );
		break;

	case CmdLogMessage:
		RecordInternal( logMsg );
		break;
	}
}



//--------------------------------------------------------------------------------
// Internal support

// Write the message to the proper log file
bool Log::RecordInternal( const LogMessage& logMsg )
{
	string		logFilePath;
	string		line;
	bool		ok = false;

	// Construct log file path and output line
	if ( GetLogOutput( logMsg, logFilePath, line ) )
	{
		ofstream out;

		out.open( logFilePath.c_str(), ios_base::out | ios_base::app );

		// Write line and close output
		if ( out.is_open() )
		{
			out << line << endl;
			out.close();
			ok = true;
		}
	}

	return ok;
}



// Compose file path and output line
bool Log::GetLogOutput( const LogMessage& logMsg, string& logFilePath, string& line )
{
	bool ok = false;

	// Log file path pattern: /var/log/pgjobs/<agent>/<yyyy.mm.dd>.txt
	// Log line pattern: date|time|status|text

	// Compose date
	char		date[ 16 ];
	const tm*	timestamp = logMsg.Timestamp;
	sprintf( date, DatePattern,
					timestamp->tm_year + TMYearBase,
					timestamp->tm_mon + TMMonthBase,
					timestamp->tm_mday );

	// Compose time
	char	time[ 16 ];
	sprintf( time, TimePattern,
					timestamp->tm_hour,
					timestamp->tm_min,
					timestamp->tm_sec );

	// Compose log file path
	logFilePath.clear();
	if ( logMsg.Agent )
	{
		logFilePath  = LogRoot;
		logFilePath += logMsg.Agent;
		logFilePath += '/';
		logFilePath += date;
		logFilePath += LogExt;
	}

	// Compose output line
	line.clear();
	if ( !logFilePath.empty() )
	{
		line  = date;
		line += RowColChar;
		line += time;
		line += RowColChar;
		if ( logMsg.Status )
		{
			line += logMsg.Status;
		}
		line += RowColChar;
		if ( logMsg.Text )
		{
			line += logMsg.Text;
		}

		ok = true;
	}

	return ok;
}


