/*
	masteragent.cpp

	The MasterAgent class reads config entries, creates DbAgent instances,
	and runs them.

	Project: pgjobs
	Author: Zlatko Michailov
	Created: 13-Oct-2003
	Updated: 28-Oct-2003
	Updated:  5-May-2004
	Updated: 26-May-2004

	This file is provided as is, with no warranty. Use at your own risk.

	Copyright (c) 2003-2004, Zlatko Michailov

*/



//--------------------------------------------------------------------------------

#include <stdlib.h>
#include "masteragent.h"
#include "dbagent.h"


using namespace msdk;



//--------------------------------------------------------------------------------
// Construction

MasterAgent::MasterAgent() :
	Agent( Master, LogLevelDebug ),
	Thread()
{
	// We have set the log level to Debug temporarily
	// until we read the master config row. Then we'll
	// reset it with the proper value.
}



//--------------------------------------------------------------------------------
// Overridables

void MasterAgent::Startup()
{
	bool ok;

	// Start and init log
	ok = SharedLog.Start();
	if ( ok )
	{
		ok = InitLog();
	}

	// Read master config
	if ( ok )
	{
		ok = ReadMasterConfig();
	}

	// Sequentially read db configs and spawn DbAgents
	if ( ok )
	{
		ok = SpawnDbAgents();
	}

	// Are we ready to run?
	IsReadyToRun = ok;
}



void MasterAgent::Shutdown()
{
	int n = DbAgents.size();
	int i;

	Status = StatusShutdown;

	// Stop all DbAgents
	Record ( "Stopping DbAgents...", LogLevelSystemVerbose );
	for ( i = 0; i < n; i++ )
	{
		DbAgents[ i ]->Stop();
	}

	// Wait for all DbAgents to finish
	for ( i = 0; i < n; i++ )
	{
		DbAgents[ i ]->WaitUntilFinished();
	}
	Record ( "DbAgents down", LogLevelSystemInfo );

	// Release DbAgent instances
	while ( !DbAgents.empty() )
	{
		DbAgent* dbAgent = DbAgents.front();
		DbAgents.erase( DbAgents.begin() );

		delete dbAgent;
	}

	// Stop log
	SharedLog.Stop();

	// Wait for log to finish
	SharedLog.WaitUntilFinished();
}



void MasterAgent::Run()
{
	if ( IsReadyToRun )
	{
		Status = StatusRunning;
		Record( "Up and running", LogLevelSystemInfo );

		// Sleep infinitely until stopped by TERM signal handler
		for ( ;; )
		{
			// We must provide a positive number of seconds to sleep,
			// e.g. 1 hour (3,600 seconds)
			Sleep( 3600 );
		}
	}
	else
	{
		Record( "Not able to run! Shutting down...", LogLevelError );
	}
}



//--------------------------------------------------------------------------------
// Internal support

bool MasterAgent::ReadMasterConfig()
{
	Config	config( ConfigFilePath );
	string	masterRow;
	string	logLevelCol;
	string	error;
	int		logLevel;
	bool	ok;

	// Read master config row
	Record( "Trying to read master config row...", LogLevelSystemVerbose );
	ok = config.GetMaster( masterRow, error );
	if ( ok )
	{
		ok = config.GetCol( masterRow, 1, logLevelCol, error );
		if ( ok )
		{
			Record( "Master config row read", LogLevelSystemInfo );
			logLevel = atoi( logLevelCol.c_str() );
		}
	}

	// Try to repair the master row before giving up
	if ( !ok )
	{
		Record( error.c_str(), LogLevelError );
		Record( "Master config row missing or broken! Trying to repair...", LogLevelError );

		// Repair row: master=2
		masterRow  = Master;
		masterRow += RowMainChar;
		masterRow += '0' + LogLevelSystemVerbose;
		ok = config.SetMaster( masterRow, error );
		if ( ok )
		{
			Record( "Master config row repaired", LogLevelSystemInfo );
			logLevel = LogLevelSystemInfo;
		}
		else
		{
			Record( error.c_str(), LogLevelError );
			Record( "Master config row cannot be repaired!", LogLevelError );
		}
	}

	// Reset LogLevel to the proper value
	if ( ok )
	{
		LogLevel = logLevel;
	}

	return ok;
}



bool MasterAgent::SpawnDbAgents()
{
	Config			config( ConfigFilePath );
	StreamPosition	pos = 0;
	DbInfo			dbInfo;
	string			error;
	bool			ok;

	do
	{
		// Read DbConfigs sequentially
		error.clear();
		ok = config.SeqGetDb( pos, dbInfo, error );
		if ( ok )
		{
			string text;
			string dbaDescr;

			// ID ( Host, Port, Database )
			dbaDescr  = dbInfo.ID;
			dbaDescr += " ( ";
			dbaDescr += dbInfo.Host;
			dbaDescr += ", ";
			dbaDescr += dbInfo.Port;
			dbaDescr += ", ";
			dbaDescr += dbInfo.Database;
			dbaDescr += " )";

			text  = "Spawning DbAgnt: ";
			text += dbaDescr;
			text += "...";
			Record( text.c_str(), LogLevelSystemVerbose );

			// Create and start DbAgent
			DbAgent* dbAgent = new DbAgent( dbInfo );
			if ( dbAgent )
			{
				if ( dbAgent->Start() )
				{
					// Add instance to list
					DbAgents.push_back( dbAgent );

					text  = "DbAgent ";
					text += dbaDescr;
					text += " spawned";
					Record( text.c_str(), LogLevelSystemInfo );
				}
				else
				{
					text  = "Could not start DbAgent ";
					text += dbaDescr;
					text += "!";
					Record( text.c_str(), LogLevelError );

					// Instance created but could not start: release
					delete dbAgent;
				}
			} // instance allocated
		} // got config
	}
	while ( ok || !error.empty() );

	return true; !DbAgents.empty();
}

