/****************************************************************************
**
** Copyright (C) 2021 The Qt Company Ltd.
** Contact: https://www.qt.io/licensing/
**
** This file is part of the test suite of the Qt Toolkit.
**
** $QT_BEGIN_LICENSE:GPL-EXCEPT$
** Commercial License Usage
** Licensees holding valid commercial Qt licenses may use this file in
** accordance with the commercial license agreement provided with the
** Software or, alternatively, in accordance with the terms contained in
** a written agreement between you and The Qt Company. For licensing terms
** and conditions see https://www.qt.io/terms-conditions. For further
** information use the contact form at https://www.qt.io/contact-us.
**
** GNU General Public License Usage
** Alternatively, this file may be used under the terms of the GNU
** General Public License version 3 as published by the Free Software
** Foundation with exceptions as appearing in the file LICENSE.GPL3-EXCEPT
** included in the packaging of this file. Please review the following
** information to ensure the GNU General Public License requirements will
** be met: https://www.gnu.org/licenses/gpl-3.0.html.
**
** $QT_END_LICENSE$
**
****************************************************************************/

#include <QFile>
#include <QMap>
#include <QString>
#include <QTest>
#include <qdebug.h>


class tst_QMap : public QObject
{
    Q_OBJECT

private slots:
    void insertion_int_int();
    void insertion_int_string();
    void insertion_string_int();

    void lookup_int_int();
    void lookup_int_string();
    void lookup_string_int();

    void iteration();
    void toStdMap();
    void iterator_begin();

    void ctorStdMap();

    void insertion_int_intx();
    void insertion_int_int_with_hint1();
    void insertion_int_int2();
    void insertion_int_int_with_hint2();

    void insertion_string_int2();
    void insertion_string_int2_hint();

    void insertMap();

private:
    QStringList helloEachWorld(int count);
};

QStringList tst_QMap::helloEachWorld(int count)
{
    QStringList result;
    result.reserve(count);
    result << QStringLiteral("Hello World"); // at index 0, not used

    char16_t name[] = u"Hello    World";
    QStringView str(name);
    for (int i = 1; i < count; ++i) {
        auto p = name + 6; // In the gap between words.
        for (const auto ch : QChar::fromUcs4(i))
            p++[0] = ch;
        result << str.toString();
    }
    return result;
}

void tst_QMap::insertion_int_int()
{
    QMap<int, int> map;
    QBENCHMARK {
        for (int i = 0; i < 100000; ++i)
            map.insert(i, i);
    }
}

void tst_QMap::insertion_int_intx()
{
    // This is the same test - but executed later.
    // The results in the beginning of the test seems to be a somewhat inaccurate.
    QMap<int, int> map;
    QBENCHMARK {
        for (int i = 0; i < 100000; ++i)
            map.insert(i, i);
    }
}

void tst_QMap::insertion_int_int_with_hint1()
{
    QMap<int, int> map;
    QBENCHMARK {
        for (int i = 0; i < 100000; ++i)
            map.insert(map.constEnd(), i, i);
    }
}

void tst_QMap::insertion_int_int2()
{
    QMap<int, int> map;
    QBENCHMARK {
        for (int i = 100000; i >= 0; --i)
            map.insert(i, i);
    }
}

void tst_QMap::insertion_int_int_with_hint2()
{
    QMap<int, int> map;
    QBENCHMARK {
        for (int i = 100000; i >= 0; --i)
            map.insert(map.constBegin(), i, i);
    }
}

void tst_QMap::insertion_int_string()
{
    QMap<int, QString> map;
    QString str("Hello World");
    QBENCHMARK {
        for (int i = 0; i < 100000; ++i)
            map.insert(i, str);
    }
}

void tst_QMap::insertion_string_int()
{
    QMap<QString, int> map;
    const QStringList names = helloEachWorld(100000);
    QBENCHMARK {
        for (int i = 1; i < 100000; ++i)
            map.insert(names.at(i), i);
    }
}

void tst_QMap::lookup_int_int()
{
    QMap<int, int> map;
    for (int i = 0; i < 100000; ++i)
        map.insert(i, i);

    int sum = 0;
    QBENCHMARK {
        for (int i = 0; i < 100000; ++i)
             sum += map.value(i);
    }
}

void tst_QMap::lookup_int_string()
{
    QMap<int, QString> map;
    QString str("Hello World");
    for (int i = 0; i < 100000; ++i)
        map.insert(i, str);

    QBENCHMARK {
        for (int i = 0; i < 100000; ++i)
             str = map.value(i);
    }
}

void tst_QMap::lookup_string_int()
{
    QMap<QString, int> map;
    const QStringList names = helloEachWorld(100000);
    for (int i = 1; i < 100000; ++i)
        map.insert(names.at(i), i);

    int sum = 0;
    QBENCHMARK {
        for (int i = 1; i < 100000; ++i)
            sum += map.value(names.at(i));
    }
}

// iteration speed doesn't depend on the type of the map.
void tst_QMap::iteration()
{
    QMap<int, int> map;
    for (int i = 0; i < 100000; ++i)
        map.insert(i, i);

    int j = 0;
    QBENCHMARK {
        for (int i = 0; i < 100; ++i) {
            QMap<int, int>::const_iterator it = map.constBegin();
            QMap<int, int>::const_iterator end = map.constEnd();
            while (it != end) {
                j += *it;
                ++it;
            }
        }
    }
}

void tst_QMap::toStdMap()
{
    QMap<int, int> map;
    for (int i = 0; i < 100000; ++i)
        map.insert(i, i);

    QBENCHMARK {
        std::map<int, int> n = map.toStdMap();
        n.begin();
    }
}

void tst_QMap::iterator_begin()
{
    QMap<int, int> map;
    for (int i = 0; i < 100000; ++i)
        map.insert(i, i);

    QBENCHMARK {
        for (int i = 0; i < 100000; ++i) {
            QMap<int, int>::const_iterator it = map.constBegin();
            QMap<int, int>::const_iterator end = map.constEnd();
            if (it == end) // same as if (false)
                ++it;
        }
    }
}

void tst_QMap::ctorStdMap()
{
    std::map<int, int> map;
    for (int i = 0; i < 100000; ++i)
        map.insert(std::pair<int, int>(i, i));

    QBENCHMARK {
        QMap<int, int> qmap(map);
        qmap.constBegin();
    }
}

class XString : public QString
{
public:
    bool operator < (const XString& x) const // an expensive operator <
    {
        return toInt() < x.toInt();
    }
};

void tst_QMap::insertion_string_int2()
{
    QMap<XString, int> map;
    QBENCHMARK {
        for (int i = 1; i < 5000; ++i) {
            XString str;
            str.setNum(i);
            map.insert(str, i);
        }
    }
}

void tst_QMap::insertion_string_int2_hint()
{
    QMap<XString, int> map;
    QBENCHMARK {
        for (int i = 1; i < 5000; ++i) {
            XString str;
            str.setNum(i);
            map.insert(map.end(), str, i);
        }
    }
}

void tst_QMap::insertMap()
{
    QMap<int, int> map;
    for (int i = 0; i < 100000; ++i)
        map.insert(i * 4, 0);
    QMap<int, int> map2;
    for (int i = 0; i < 50000; ++i)
        map2.insert(i * 7, 0);
    QBENCHMARK_ONCE {
        map.insert(map2);
    }
}

QTEST_MAIN(tst_QMap)

#include "main.moc"
