#!/bin/bash
# Copyright (C) 2013 Tomasz Olszak <olszak.tomasz@gmail.com>
# All rights reserved.
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions
# are met:

# 1. Redistributions of source code must retain the copyright
#    notice, this list of conditions and the following disclaimer.
# 2. Redistributions in binary form must reproduce the copyright
#    notice, this list of conditions and the following disclaimer in the
#    documentation and/or other materials provided with the distribution.
# 3. The name of the author may not be used to endorse or promote products 
#    derived from this software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
# IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
# OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
# IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
# INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
# NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
# DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
# THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
# (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
# THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#

set -e

if [ -z "$TIZEN_VERSION" -o -z "$COMPILER_DIRNAME" -o -z "$COMPILER_PREFIX" \
     -o -z "$DEVICE" ]; then
      echo "You must set all obligatory input variables: TIZEN_VERSION, COMPILER_DIRNAME, COMPILER_PREFIX, DEVICE"
fi

set +e
isEmulator=$(echo $DEVICE | grep -c emulator)
set -e

export TIZEN_VERSION

if [ "$isEmulator" == "1" ]; then
    export CROSS_COMPILE_FOR=emulator
else 
    export CROSS_COMPILE_FOR=device
fi

function usage {
    echo -e "\n"
    echo "Script for Qt for tizen compilation. "
    echo "Set Parameters or environment variables with absolute paths"
    echo ""
    echo "Usage:"
    echo "sudo [MAKE_THREADS=numberOfCPUCores] STANDARD_USER=$USER [TIZENSDKCOMPILERDIR=tizen/compiler] [TIZENSYSROOT=tizenSysroot] [HOSTPREFIX=qtHostInstallDir] ./downloadAndBuildAll.sh"
    echo ""
    echo "STANDARD_USER - your standard user, without superuser privileges"
    echo "TIZENSDKCOMPILERDIR - should be set to tizen-sdk/tools/$COMPILER_DIRNAME "
    echo "QT5SRCDIR - directory with qt5 sources"
    echo "TIZENSYSROOT - sysroot created from emulator devel headers and libraries - by default qt will be installed in \$TIZENSYSROOT/opt/usr/qt5"
    echo "HOSTPREFIX - here qmake and other host binaries will be installed"
    echo "If you want to download qt source, tizen packages and autoomatically build just pass STANDARD_USER as first parameter(\$USER)"
    echo "If corresponding environment variables are set then script can be invoked without parameters"
    echo "set MAKE_THREADS environment variable to number of your CPU cores to fast up compilation step"    
    echo ""
    cd $CURRENT_DIR
    exit 0;
}

if [ "$1" == "-h" -o "$1" == "--help" ]; then
    usage
fi

CURRENT_USER=$(whoami)

if [ $CURRENT_USER != "root" ]; then
    echo "run this script with superuser privileges"
    usage
fi


if [ "$STANDARD_USER" == "" ]; then
    echo "STANDARD_USER variable was not set"
    usage
fi

set +e

type rpm2cpio &> /dev/null

if [ "$?" != "0" ]; then
    echo "rpm2cpio is needed for creating rootfs - install rpm2cpio"
    exit 1
fi

type cpio &> /dev/null

if [ "$?" != "0" ]; then
    echo "cpio is needed for creating rootfs - install cpio"
    exit 1
fi

id $STANDARD_USER &> /dev/null

if [ "$?" != "0" ]; then
    echo "there is no user $STANDARD_USER in the system"
    exit 1
fi

set -e

if [ "$TIZENSDKCOMPILERDIR" == "" ]; then
    if [ -f $HOME/tizen-sdk/tools/$COMPILER_DIRNAME/bin/$COMPILER_PREFIX-g++ ]; then
        TIZENSDKCOMPILERDIR=$HOME/tizen-sdk/tools/$COMPILER_DIRNAME
        echo "Setting TIZENSDKCOMPILERDIR to $TIZENSDKCOMPILERDIR"
    fi
fi

if [ "$TIZENSYSROOT" == "" ]; then
    TIZENSYSROOT=$PWD/rootfs
    echo "Setting TIZENSYSROOT to $TIZENSYSROOT"
fi

if [ "$QT5SRCDIR" == "" ]; then
    QT5SRCDIR=$PWD/qt5.gitorious
    echo "Setting QT5SRCDIR to $QT5SRCDIR"
fi

if [ "$HOSTPREFIX" == "" ]; then
    HOSTPREFIX=$PWD/qt5hostInstall
    echo "Setting HOSTPREFIX to $HOSTPREFIX"
fi


if [ -z "$STANDARD_USER" -o -z "$QT5SRCDIR" -o -z "$TIZENSYSROOT" \
     -o -z "$TIZENSDKCOMPILERDIR" -o -z "$HOSTPREFIX" ]; then
      usage
fi

if [ "$TIZENSYSROOT" == "$PWD/rootfs" ]; then

    if [ ! -d repos ]; then
        echo "getting tizen repositories"
        su -m -l $STANDARD_USER -c "../common/download_repos.sh"
    fi

    mkdir -p rootfs

    if [ ! -d rootfs/usr/lib ]; then
        cd rootfs
        for rpmFile in `find ../repos -name "*.rpm"`; 
        do 
            rpm2cpio $rpmFile | cpio -imdv -R $STANDARD_USER:$STANDARD_USER; 
        done
        cd ..
    else
        echo "Looks like rootfs is already prepared - rootfs/usr/lib exists"
    fi
fi

if [ "$QT5SRCDIR" == "$PWD/qt5.gitorious" ]; then
    if [ ! -d qt5.gitorious ]; then
        su $STANDARD_USER -c "git clone -b release https://git.gitorious.org/qt/qt5.git qt5.gitorious"
        cd qt5.gitorious
        su $STANDARD_USER -c "git clone -b wip/tizen https://git.gitorious.org/qt/qtquickcontrols-tizen.git"
	cd qtquickcontrols-tizen
	su $STANDARD_USER -c "git reset --hard fd66e35cc0b0f0d841d8aec46e85afd6ff89444e"
	cd ..
        su $STANDARD_USER -c "./init-repository --http --no-webkit --module-subset=qtactiveqt,qtbase,qtdeclarative,qtdoc,qtgraphicaleffects,qtimageformats,qtjsbackend,qtmultimedia,qtquickcontrols,qtsensors,qtserialport,qtsvg,qttools,qttranslations,qtxmlpatterns"
        cd qtquickcontrols
        su $STANDARD_USER -c "git checkout wip/tizen"
	su $STANDARD_USER -c "git reset --hard b3bff2c1d98c4a29d3ad249167409e64a836d2bf"
        cd ../
        cd qtbase
        su $STANDARD_USER -c "git checkout wip/tizen"
	su $STANDARD_USER -c "git reset --hard b2c3b3deed52cd1562f26c68eefac9ac76c24357"
	cd ../qtsensors
	su $STANDARD_USER -c "git checkout wip/tizen"
	su $STANDARD_USER -c "git reset --hard 7865efa71e8dba289c9b1ac19454f0a9311c8328"
        cd ../../
    else
        echo "Looks like qt5.gitorious directory exists"
    fi;
fi;

su $STANDARD_USER -c "mkdir -p buildQt5"
cd buildQt5

echo "================================================="
echo "Configuration:"
echo "STANDARD_USER=$STANDARD_USER"
echo "QT5SRCDIR=$QT5SRCDIR"
echo "TIZENSYSROOT=$TIZENSYSROOT"
echo "TIZENSDKCOMPILERDIR=$TIZENSDKCOMPILERDIR"
echo "HOSTPREFIX=$HOSTPREFIX"
echo "================================================="

export PKG_CONFIG_LIBDIR=$TIZENSYSROOT/usr/lib
export PKG_CONFIG_SYSROOT_DIR=$TIZENSYSROOT
export PKG_CONFIG_PATH=$TIZENSYSROOT/usr/lib/pkgconfig


su $STANDARD_USER -c "$QT5SRCDIR/configure -hostprefix $HOSTPREFIX -v \
                    -release -opengl es2 -opensource \
                    -confirm-license -device $DEVICE \
                    -device-option CROSS_COMPILE=$TIZENSDKCOMPILERDIR/bin/$COMPILER_PREFIX- \
                    -prefix /opt/usr/apps/Qt5Install -sysroot $TIZENSYSROOT -nomake examples \
                    -qpa xcb -system-libpng -qt-xcb -no-widgets -no-eglfs -no-pch -no-sql-mysql"

MAKE_ARGS=""

if [ "$MAKE_THREADS" != "" ]; then
    echo "setting make thread"
    MAKE_ARGS="-j $MAKE_THREADS"
fi
echo make $MAKE_ARGS

su $STANDARD_USER -c "make $MAKE_ARGS"
echo make install $MAKE_ARGS
su $STANDARD_USER -c "make install $MAKE_ARGS"

if [ -d $QT5SRCDIR/qtquickcontrols-tizen ]; then
    su $STANDARD_USER -c "mkdir -p qtquickcontrols-tizen"
    cd qtquickcontrols-tizen
    su $STANDARD_USER -c "$HOSTPREFIX/bin/qmake $QT5SRCDIR/qtquickcontrols-tizen"
    su $STANDARD_USER -c "make $MAKE_ARGS"
    su $STANDARD_USER -c "make install $MAKE_ARGS"
fi
